
"The following code installs class Bacillus, an animated simulation of a
 bacillus swimming as described by Julius Adler in his Scientific American
 article (April 1976).  It also installs some changes to class Animation and
 DemoClass needed by the simulation.

 Note: since there are no directory prefixes in the path names, the files
 LOADER.ST, RANDOM.ST, BACILLUS.ST, RUN.PXL, and TUMBLE.PXL must be in the
 current directory.

 If class Loader or Random isn't installed, evaluate the appropriate
 expression(s) with `do it.'

    (File pathName: 'loader.st') fileIn; close
    (File pathName: 'random.st') fileIn; close 

 To install the simulation code evaluate the following expression with
 `do it.'

    (File pathName: 'bacillus.st') fileIn; close

 Finally, evaluate the following expressions with `do it.'

    | stream icons |
    stream := Disk file: 'run.pxl'.
    icons :=  Loader new loadFrom: stream.
    Bacillus runIcons: icons.
    stream close.
    stream := Disk file: 'tumble.pxl'.
    icons :=  Loader new loadFrom: stream.
    Bacillus tumbleIcons: icons.
    Bacillus initialize

 To test the code, popup the demo menu and choose 'bacillus.'
"!

! Animation methods !
shiftRate

    "Answer the shiftRate of the receiver."

    ^shiftRate! !

! Animation methods !
tell: aName setIcon: anIndex

    "Tell the pen  with name aName to start with the icon corresponding
     to anIndex."

    self getCurrentPen: aName.
    curPen sourceX: anIndex! !

Object subclass: #Bacillus
  instanceVariableNames:
    'heading longRun maxTurns minTurns position shortRun threshold'
  classVariableNames:
    'IconOffset OffStage Rndm RunIcons Runners Stage StageCenter TumbleIcons'
  poolDictionaries: ''!

"Instance Variables
    heading -- Integer -- The angle between bacillus axis and the x-axis;
        measured clockwise from the x-axis.
    longRun -- Integer -- The distance a bacillus moves when performing a
        long run.
    maxTurns -- Integer -- The maximum number of full turns a bacillus makes
        when performing a tumble.
    minTurns -- Integer -- The minimum number of full turns a bacillus makes
        when performing a tumble; this is in addition to the turning required
        to make the computed change in heading.
    position -- Point -- The current position of the bacillus; measured in
        pixels from the upper left-hand corner of the display.
    shortRun -- Integer -- The distance a bacillus moves when performing
        a short run.
    threshold -- Float -- A control parameter determining how strongly the
        bacillus is attracted to StageCenter.  Acceptable values can be
        generated by evaluating `angle degreesToRadians cos' with `angle'
        strictly between 0 and 90; the value used here (see instance method
        `initialize') was generated by setting `angle' equal to 45.  This
        produces a strong attraction.
"!

"Class Variables
    IconOffset -- Point -- The offset from the origin of an icon to its visual
        center.  As no single value will work for all the icons, the value
        used here (see class method `initialize') is a compromise.
    OffStage -- Point -- A safe place to hide the icons when they should not
        be visible.
    Rndm -- Random -- A random number generator.
    RunIcons -- IdentityDictionary -- The icons displayed when the bacillus
        makes straight runs; indexed by heading.
    Runners -- IdentityDictionary -- A dictionary that associates headings
        with the corresponding symbolic key required by the animation
        controller when the bacillus makes straight runs.
    Stage -- Rectangle -- The area of the display screen in which the bacillus
        icons are visible.
    StageCenter -- Point -- The center of the Stage; this is the position of
        the attracting point.
    TumbleIcons -- IdentityDictionary -- The icons displayed when the bacillus
        tumbles; indexed by heading.
"!

"Global variables
    JSB -- Bacillus -- Julius Synthetic Bacillus, the star of the show.
    JSBmotor -- Animation -- the animation controller producing JSB's antics
        as seen on the display.
"!

! Bacillus class methods !
initialize

    " Initialize some of the class variables."

    OffStage := -100 @ -100.
    Rndm := Random new.
    IconOffset := 48 @ 16.
    Runners := IdentityDictionary new.
    Runners
        at: 0 put: #run0;
        at: 45 put: #run45;
        at: 90 put: #run90;
        at: 135 put: #run135;
        at: 180 put: #run180;
        at: 225 put: #run225;
        at: 270 put: #run270;
        at: 315 put: #run315! !

! Bacillus class methods !
newMotor: aRectangle

    "Make a new animation controller for the simulation.  The controller will
     have its display area delineated by aRectangle."

    | iconArray |

    Stage := aRectangle.
    StageCenter := Stage center.
    JSBmotor := Animation new
        initialize: aRectangle;
        speed: 2;
        shiftRate: 3.
    0 to: 315 by: 45 do:
        [
        :angle |
            iconArray := Array new: 1.
            iconArray
                at: 1 put: (RunIcons at: angle).
            JSBmotor
                add: iconArray
                    name: (Runners at: angle)
                    color: #black;
                tell: (Runners at: angle) place: OffStage
        ].
    iconArray := Array new: 8.
    iconArray
        at: 1 put: (TumbleIcons at: 0);
        at: 2 put: (TumbleIcons at: 45);
        at: 3 put: (TumbleIcons at: 90);
        at: 4 put: (TumbleIcons at: 135);
        at: 5 put: (TumbleIcons at: 180);
        at: 6 put: (TumbleIcons at: 225);
        at: 7 put: (TumbleIcons at: 270);
        at: 8 put: (TumbleIcons at: 315).
    JSBmotor
        add: iconArray name: #tumble color: #black;
        tell: #tumble place: OffStage! !

! Bacillus class methods !
runIcons: anIconDictionary

    "Store an object in the RunIcons class variable; the object should be
     an identity dictionary with entries of the form <angle, icon>, where
     angle is one of the sequence {0, 45, ... , 315} and icon is a form.
     Normally, the identity dictionary will be loaded from the file RUN.PXL."

    RunIcons := anIconDictionary! !

! Bacillus class methods !
tumbleIcons: anIconDictionary

    "Store an object in the TumbleIcons class variable; the object should be
     an identity dictionary with entries of the form <angle, icon>, where
     angle is one of the sequence {0, 45, ... , 315} and icon is a form.
     Normally, the identity dictionary will be loaded from the file
     TUMBLE.PXL."

    TumbleIcons := anIconDictionary! !

! Bacillus methods !
demo

    "Prompt for an iteration limit and run the bacillus demo for that many
     iterations."

    | pen runCount |

        "Draw a spot to mark the attracting point."
    pen := Pen new
        place: StageCenter;
        solidEllipse: 4 aspect: 1.
    JSBmotor setBackground.
        "Starting offstage means a cold start, so bring JSB onstage at
         stage center; otherwise, continue from where JSB was at the
         end of the last demo."
    position = OffStage ifTrue:
        [ position := StageCenter - IconOffset ].
    heading := (8 * Rndm next) truncated * 45.
    runCount := Prompter prompt: 'how many runs?' defaultExpression: '30'.
        "RunCount will be nil if the user want to cancel?  Exit if that's
         true."
    runCount isNil ifTrue:
        [ ^nil ].
    CursorManager normal hide.
    runCount timesRepeat:
        [ self swim ].
    JSBmotor
        tell: (Runners at: heading) place: OffStage.
    CursorManager normal display! !

! Bacillus methods !
fullTurns

    "Answer a integer sampled uniformly from the interval
     [minTurns, maxTurns]."

    ^minTurns + ((1 + maxTurns - minTurns) * Rndm next) truncated.! !

! Bacillus methods !
initialize

    "Initialize the instance variables of the receiver."

    heading := 0.
    longRun := 150.
    maxTurns := 2.
    minTurns := 1.
    position := OffStage.
    shortRun := 15.
    threshold := 7.0710678e-1! !

! Bacillus methods !
newHeading

    "Answer a random heading sampled uniformly from the sequence
     {0, 45, ... , 315} but excluding the current heading."

    ^(45 * (7 * Rndm next + 1) truncated + heading) \\ 360! !

! Bacillus methods !
runFor: aDistance at: anAngle

    "Perform the straight run part of the bacillus swimming action."

    | runner |

    runner := Runners at: anAngle.
    JSBmotor
        tell: runner place: position;
        tell: runner direction: heading;
        tell: runner go: aDistance;
        tell: runner place: OffStage.
    position :=
        (anAngle integerCos @ anAngle integerSin) * aDistance // 100
        + position! !

! Bacillus methods !
senseAttraction

    "Answer true if the receiver is moving toward the attracting point
     (StageCenter); otherwise, answer false."

    | rad len v1 v2 |

    rad := heading degreesToRadians.
    v1 := rad cos @ rad sin.
    v2 := StageCenter - position - IconOffset.
    len := (v2 dotProduct: v2) sqrt.
    len <= 1.0
        ifTrue: [ ^false ].
    ^(v1 dotProduct: v2 * len reciprocal) > threshold.! !

!Bacillus methods !
swim

    "Perform the bacillus swimming action: a random straight run, followed by
     a period of random tumbling.  The length of the straight run is determ-
     ined by the sign of receiver's component of motion toward the attracting
     point (StageCenter); a positive component produces a long run and
     negative one produces a short run."

    self senseAttraction
        ifTrue:
            [ self runFor: longRun at: heading ]
        ifFalse:
            [ self runFor: shortRun at: heading ].
    self tumbleTo: self newHeading! !

! Bacillus methods !
tumbleTo: newHeading

    "Perform the tumble part of the bacillus swimming action; i.e., rotate
     for a random number of full turns plus that fraction of a turn required
     to achieve a heading equal to newHeading."

    | changeInHeading toAndFro |

    JSBmotor
        "Select the icon corresponding to the heading."
        tell: #tumble setIcon: 1 + (heading // 45);
        tell: #tumble place: position.
    changeInHeading := (newHeading - heading) \\ 360.
    toAndFro := 0.
    (changeInHeading // 45 + (8 * self fullTurns)) * JSBmotor shiftRate
        timesRepeat:
            [
            JSBmotor tell: #tumble direction: toAndFro.
            JSBmotor tell: #tumble go: 1.
            toAndFro := (toAndFro + 180) \\ 360
            ].
    JSBmotor tell: #tumble place: OffStage.
    heading := newHeading! !

! DemoClass methods !
bacillus

    "Run the Julius Synthetic Bacillus demo after performing initialization
     on such global variables as require it."

    JSB isNil ifTrue:
        [ JSB := Bacillus new initialize ].
    JSBmotor isNil ifTrue:
        [ Bacillus newMotor: rectangle ].
    JSB demo! !

! DemoClass methods !
clear

    "Clear the graphics display rectangle."

    Display white: rectangle! !

! DemoClass methods !
demoMenu

    "Answer the menu for the receiver."

    | labelTmp |

    labelTmp := 'walking line\dragon\mandala\multi mandala\',
        'multi pentagon\multi spiral\bouncing ball\bacillus\clear\exit'.
    ^Menu
        labels: labelTmp withCrs
        lines: #(3 6 8)
        selectors: #(walkLine dragon mandala multiMandala multiPentagon
                     multiSpiral bounceBall bacillus clear exit)! !
