
(* \\\\\\\\\\\\\\\\\\\ Window Routines \\\\\\\\\\\\\\\\\\\\ *)

 PROCEDURE draw_frame(x1,y1,x2,y2 : counter ; title : string80 ;
                      frame_color : color) ;
  (* Draw a frame on the screen at absolute screen positions *)
  (* x1,y1 - upper left corner *)
  (* x2,y2 - lower right corner *)
  CONST
   bar = #196 ;
   vert_bar = #179 ;
   upper_lf = #218 ;
   upper_rt = #191 ;
   lower_lf = #192 ;
   lower_rt = #217 ;
  VAR
   i : 1 .. 25 ;
   border : string80 ;

  PROCEDURE get_frame_co_ords ;
   BEGIN
    x1 := min(max(1,x1),78) ;
    y1 := min(max(1,y1),23) ;
    x2 := min(max(3,x2),80) ;
    y2 := min(max(3,y2),25) ;
   END ; (* get_frame_co_ords *)

  PROCEDURE write_title ;
   BEGIN
    IF length(title) > (x2 - x1 - 1)
     THEN title := copy(title,1,x2 - x1 - 1) ;
    write(title) ;
    write(copy(border,1,length(border) - length(title))) ;
   END ; (* write_title *)

  BEGIN
   get_frame_co_ords ;
   window(1,1,80,25) ;
   border := '' ;
   makestr(border,x2 - x1 - 1) ;
   fillchar(border[1],x2 - x1 - 1,bar) ;
   gotoxy(x1,y1) ;
   textcolor(frame_color) ;
   textbackground(black) ;
   write(upper_lf) ;
   write_title ;
   write(upper_rt) ;
   FOR i := y1 + 1 TO y2 - 1  DO
    BEGIN
     gotoxy(x1,i) ;
     write(vert_bar) ;
     gotoxy(x2,i) ;
     write(vert_bar) ;
    END ;
   gotoxy(x1,y2) ;
   write(lower_lf) ;
   write(border) ;
   IF (wherex = 80) AND (wherey = 25)
    THEN
     BEGIN
      mem[seg(screen_base^) : 3998] := ord(lower_rt) ;
      mem[seg(screen_base^) : 3999] := (black SHL 4) + frame_color ;
     END
    ELSE write(lower_rt) ;
  END ; (* draw_frame *)


 PROCEDURE retrace_wait ;
  (* This routine is a delay to prevent snow on a CGA screen *)
  (* It is unecessary for mono and EGA. It watches the color status reg *)
  (* until the horizontal retrace is finished. On CGA clones it may not *)
  (* be needed, so try removing the calls to it and see if you get snow. *)
  CONST
   color_status_reg = $3DA ;
  BEGIN
   IF monitor_kind = color_monitor
    THEN WHILE (port[color_status_reg] AND $08) = 0 DO ;
  END ; (* retrace_wait *)


 PROCEDURE get_monitor_type ;
  (* find out what kind of display we are using *)
  (* A hercules card is a mono card *)
  VAR
   regs : dos_rec ;
  BEGIN
   WITH regs DO
    BEGIN
     ah := $12 ;
     bh := $03 ;
     bl := $10 ;
    END ;
   intr($10,regs) ;
   IF regs.bh < 2
    THEN
     BEGIN
      monitor_kind := ega_monitor ;
      screen_base := ptr(color_base,0) ;
     END
    ELSE
     BEGIN
      regs.ax := $0F00 ;
      intr($10,regs) ;
      IF regs.al < 7
       THEN
        BEGIN
         monitor_kind := color_monitor ;
         screen_base := ptr(color_base,0) ;
        END
       ELSE
        BEGIN
         monitor_kind := mono_monitor ;
         screen_base := ptr(mono_base,0) ;
        END
     END ;
  END ; (* get_monitor_type *)


 PROCEDURE move_from_scrn(save_org,save_size : window_pos ;
                          save_scrn : char_ptr) ;
  (* Move data from physical screen memory-map area to save_scrn *)
  (* i.e. reads the the screen *)
  (* It moves characters and attributes starting at location given by *)
  (* save_org. It copies save_size.x cols by save_size.y rows *)
  (* Copy is performed on row at a time *)
  (* This routine is extremely machine specific *)
  VAR
   physical_scrn : char_ptr ;
   i : row_pos ;
  BEGIN
   physical_scrn := ptr(seg(screen_base^),ofs(screen_base^) +
                        ((save_org.y - 1) * 80 + (save_org.x - 1)) * 2) ;
   FOR i := 1 TO save_size.y DO
    BEGIN
     retrace_wait ;
     move(physical_scrn^,save_scrn^,save_size.x * 2) ;
     physical_scrn := ptr(seg(physical_scrn^),ofs(physical_scrn^) + 160) ;
     save_scrn := ptr(seg(save_scrn^),ofs(save_scrn^) + save_size.x * 2) ;
    END ;
  END ; (* move_from_scrn *)


 PROCEDURE move_to_scrn(save_org,save_size : window_pos ;
                        save_scrn : char_ptr) ;
  (* Move data from save_scrn to physical screen memory-map area, *)
  (* i.e. displays data on the screen *)
  (* It moves characters and attributes starting at location given by *)
  (* save_org. It copies save_size.x cols by save_size.y rows *)
  (* Copy is performed on row at a time *)
  (* This routine is extremely machine specific *)
  VAR
   physical_scrn : char_ptr ;
   i : row_pos ;
  BEGIN
   physical_scrn := ptr(seg(screen_base^),ofs(screen_base^) +
                        ((save_org.y - 1) * 80 + (save_org.x - 1)) * 2) ;
   FOR i := 1 TO save_size.y DO
    BEGIN
     retrace_wait ;
     move(save_scrn^,physical_scrn^,save_size.x * 2) ;
     physical_scrn := ptr(seg(physical_scrn^),ofs(physical_scrn^) + 160) ;
     save_scrn := ptr(seg(save_scrn^),ofs(save_scrn^) + save_size.x * 2) ;
    END ;
  END ; (* move_to_scrn *)


 PROCEDURE window_reverse ;
  (* After this routine is called all text written to current window will be *)
  (* displayed in reverse video *)
  BEGIN
   WITH window_list^ DO
    BEGIN
     textcolor(back_color) ;
     textbackground(fore_color) ;
    END ;
  END ; (* window_reverse *)


 PROCEDURE window_normal ;
  (* returns to normal colors *)
  (* After this routine is called all text written to current window will be *)
  (* displayed in the colors declared when the window was opened *)
  BEGIN
   WITH window_list^ DO
    BEGIN
     textcolor(fore_color) ;
     textbackground(back_color) ;
    END ;
  END ; (* window_normal *)


 PROCEDURE window_write(s : string80) ;
  (* Write a string to the window at the current cursor position in the *)
  (* window described by the first item on the window list *)
  (* Strings too long for the window are truncated at the right edge of *)
  (* the window. All of the fooling around in last row is to prevent *)
  (* the window from scrollong when you write to the lower left corner. *)
  VAR
   y_pos : byte ;

  PROCEDURE last_row ;
   VAR
    x_pos,i : byte ;
    done : boolean ;

   PROCEDURE handle_last ;
    (* This routine makes sonme BIOS calls to get the current screen *)
    (* attribute and then pokes the character into the lower right hand *)
    (* corner. There's probably better ways to do this. *)
    VAR
     attrib : byte ;
     last_pos : counter ;
     regs : dos_rec ;
    BEGIN
     WITH window_list^ DO
      BEGIN
       regs.ax := $0F00 ;
       intr($10,regs) ;
       regs.ax := $0200 ;
       regs.dh := (abs_org.y - 1) + (y_pos - 1) ;
       regs.dl := (abs_org.x - 1) + (x_pos - 2) ;
       intr($10,regs) ;
       regs.ax := $0800 ;
       intr($10,regs) ;
       attrib := regs.ah ;
       last_pos := (((abs_org.y - 1) + (y_pos - 1)) * 80
                   + (abs_org.x - 1) + (x_pos - 1)) * 2 ;
       mem[seg(screen_base^) : last_pos] := ord(s[i]) ;
       mem[seg(screen_base^) : last_pos + 1] := attrib ;
       gotoxy(window_size.x,y_pos) ;
       done := true ;
      END ;
    END ; (* handle_last *)

   BEGIN
    WITH window_list^ DO
     BEGIN
      i := 1 ;
      done := false ;
      WHILE (i <= length(s)) AND (NOT done) DO
       BEGIN
        x_pos := wherex ;
        IF (x_pos = window_size.x) AND (y_pos = window_size.y)
         THEN handle_last
        ELSE IF x_pos = window_size.x
         THEN
          BEGIN
           write(s[i]) ;
           gotoxy(window_size.x,y_pos) ;
           done := true ;
          END
        ELSE write(s[i]) ;
        i := i + 1 ;
       END ;
     END ;
   END ; (* last_row *)

  BEGIN
   y_pos := wherey ;
   WITH window_list^ DO
    IF y_pos = window_size.y
     THEN last_row
     ELSE
      BEGIN
       write(copy(s,1,min(length(s),window_size.x - wherex + 1))) ;
       IF wherey <> y_pos
        THEN gotoxy(window_size.x,y_pos) ;
      END ;
  END ; (* window_write *)


 PROCEDURE window_writeln(s : string80) ;
  (* write a string to the current window and the move cursor to *)
  (* start of the next line *)
  BEGIN
   window_write(s) ;
   IF wherey < window_list^.window_size.y
    THEN gotoxy(1,wherey + 1) ;
  END ; (* window_writeln *)


 PROCEDURE get_window_co_ords(s_ptr : window_ptr ;
                            VAR act_org,act_size : window_pos) ;
  (* Get the actual origin and size of the window described by *)
  (* s_ptr. The physical size of the window includes the frame. The *)
  (* size and origin in the descriptor do not. *)
  BEGIN
   WITH s_ptr^ DO
    IF has_frame
     THEN
      BEGIN
       act_org.x := min(max(abs_org.x - 1,1),80) ;
       act_org.y := min(max(abs_org.y - 1,1),25) ;
       act_size.x := max(min(window_size.x + 2,81 - act_org.x),1) ;
       act_size.y := max(min(window_size.y + 2,26 - act_org.y),1) ;
      END
     ELSE
      BEGIN
       act_org := abs_org ;
       act_size.x := max(min(window_size.x,81 - act_org.x),1) ;
       act_size.y := max(min(window_size.y,26 - act_org.y),1) ;
      END ;
  END ; (* get_window_co_ords *)


 PROCEDURE save_window ;
  (* save the date from the current window in the windows save area *)
  (* If the window doesn't have a save area yet, allocate one for it *)
  (* We don't allocate any storage for data for the window until it *)
  (* is switched out *)
  (* move_from_screen does the actual move from the screen *)
  VAR
   save_size,save_org : window_pos ;
  BEGIN
   IF window_list <> NIL
    THEN
     WITH window_list^ DO
      BEGIN
       cursor_pos.x := wherex ;
       cursor_pos.y := wherey ;
       get_window_co_ords(window_list,save_org,save_size) ;
       IF scrn_area = NIL
        THEN getmem(scrn_area,2 * save_size.x * save_size.y) ;
       move_from_scrn(save_org,save_size,scrn_area) ;
      END ;
  END ; (* save_window *)


 PROCEDURE ins_desc(p : window_ptr) ;
  (* Insert a window descriptor at the front of the window list *)
  BEGIN
   p^.next_window :=window_list ;
   IF window_list = NIL
    THEN last_window := p
    ELSE window_list^.prev_window := p ;
   p^.prev_window := NIL ;
   window_list := p ;
  END ; (* ins_desc *)


 PROCEDURE del_desc(del_ptr : window_ptr) ;
  (* delete a descriptor from the window list *)
  BEGIN
   IF del_ptr = window_list
    THEN
     BEGIN
      window_list := del_ptr^.next_window ;
      window_list^.prev_window := NIL ;
     END
    ELSE
     BEGIN
      del_ptr^.prev_window^.next_window := del_ptr^.next_window ;
      IF del_ptr^.next_window <> NIL
       THEN del_ptr^.next_window^.prev_window := del_ptr^.prev_window ;
     END ;
   IF window_list = NIL
    THEN last_window := NIL
   ELSE IF del_ptr = last_window
    THEN last_window := del_ptr^.prev_window ;
  END ; (* scrn_del_desc *)


 FUNCTION open_window(org_x : col_pos ; org_y : row_pos ; size_x : col_pos ;
                      size_y : row_pos ; use_frame : boolean ; title : string80 ;
                      f_color,b_color,frame_color : color) : window_ptr ;
   (* Create a new window and place it at front of the window list *)
   (* This window becomes the current window and is displayed on the screen *)
   (* The old window is saved and can be restored *)
   (* Returns a pointer to the descriptor of the new window *)
   (* org_x,org_y - the upper left hand corner of the window on the PC *)
   (*               screen. Co-ordinates are measured from (1,1). The frame *)
   (*               is not part of the window, it is outside. *)
   (* size_x,size_y - the number of columns and rows in the window. The *)
   (*                 frame is not included *)
   (* use_frame - true if you want a frame around the window. If use_frame *)
   (*             is false, title and frame_color are ignored *)
   (* title - string printed on top line of frame *)
   (* f_color - the text color *)
   (* b_color - the background color *)
   (* frame_color - color of the frame, if present *)

  PROCEDURE create_descriptor ;
   (* create a window descriptor and insert it in the window list *)
   VAR
    p : window_ptr ;
   BEGIN
    getmem(p,sizeof(window_desc)) ;
    WITH p^ DO
     BEGIN
      abs_org.x := org_x ;
      abs_org.y := org_y ;
      window_size.x := min(size_x,81 - abs_org.x) ;
      window_size.y := min(max(2,size_y),26 - abs_org.y) ;
      cursor_pos.x := 1 ;
      cursor_pos.y := 1 ;
      has_frame := use_frame ;
      fore_color := f_color ;
      back_color := b_color ;
      scrn_area := NIL ;
      ins_desc(p) ;
     END ;
   END ; (* create_descriptor *)

  BEGIN
   IF window_list <> NIL
    THEN save_window ;
   create_descriptor ;
   WITH window_list^ DO
    BEGIN
     IF use_frame
      THEN draw_frame(abs_org.x - 1,abs_org.y - 1,abs_org.x + window_size.x,
                      abs_org.y + window_size.y,title,frame_color) ;
     window(abs_org.x,abs_org.y,abs_org.x + window_size.x - 1,
            abs_org.y + window_size.y - 1) ;
     textcolor(fore_color) ;
     textbackground(back_color) ;
     clrscr ;
    END ;
   open_window := window_list ;
  END ; (* open_window *)


 PROCEDURE display_window(win_ptr : window_ptr) ;
  (* display the window whose descriptor is win_ptr on the screen *)
  (* this routine is called by other routines and shouldn't be called *)
  (* directly. Use use_window instead *)
  VAR
   save_size,save_org : window_pos ;
  BEGIN
   WITH win_ptr^ DO
    BEGIN
     get_window_co_ords(win_ptr,save_org,save_size) ;
     move_to_scrn(save_org,save_size,scrn_area) ;
    END ;
  END ; (* display_window *)


 PROCEDURE use_window(win_ptr : window_ptr) ;
  (* make win_ptr the current window, display it and restore cursor *)
  (* to its original position. The old window is saved and becomes the *)
  (* second window on the list *)
  BEGIN
   IF win_ptr <> NIL
    THEN
     IF win_ptr <> window_list
      THEN
       BEGIN
        save_window ;
        del_desc(win_ptr) ;
        ins_desc(win_ptr) ;
        display_window(win_ptr) ;
        WITH window_list^ DO
         BEGIN
          window(abs_org.x,abs_org.y,abs_org.x + window_size.x - 1,
                 abs_org.y + window_size.y - 1) ;
          gotoxy(cursor_pos.x,cursor_pos.y) ;
          textcolor(fore_color) ;
          textbackground(back_color) ;
         END ;
       END ;
  END ; (* use_window *)


 PROCEDURE scrn_refresh ;
  (* Re-draw the entire screen. The screen is assembled in a memory *)
  (* buffer before being moved to physical screen. The screen is assembled *)
  (* from the last window forward. We assemble the screen in memory *)
  (* to prevent the annoying screen blank which occurs when you assemble *)
  (* dirctly in the screen area *)
  (* screen - 4000 byte memory region to assemeble the screen *)
  VAR
   physical_scrn,save_scrn,screen : char_ptr ;
   save_size,save_org : window_pos ;
   i : row_pos ;

  PROCEDURE scrn_fill(win_ptr : window_ptr) ;
   (* This routine is like move_to_scrn, except it moves the data to *)
   (* the buffer rather than the actual screen *)
   BEGIN
    IF win_ptr <> NIL
     THEN
      BEGIN
       WITH win_ptr^ DO
        BEGIN
         get_window_co_ords(win_ptr,save_org,save_size) ;
         physical_scrn := ptr(seg(screen^),ofs(screen^) +
                             ((save_org.y - 1) * 80 + (save_org.x - 1)) * 2) ;
         save_scrn := scrn_area ;
         FOR i := 1 TO save_size.y DO
          BEGIN
           move(save_scrn^,physical_scrn^,save_size.x * 2) ;
           physical_scrn := ptr(seg(physical_scrn^),ofs(physical_scrn^) + 160) ;
           save_scrn := ptr(seg(save_scrn^),ofs(save_scrn^) + save_size.x * 2 ) ;
          END ;
        END ;
       scrn_fill(win_ptr^.prev_window) ;
      END ;
   END ; (* scrn_fill *)

  BEGIN
   getmem(screen,4000) ;
   fillchar(screen^,4000,chr(0)) ;
   scrn_fill(last_window) ;
   save_org.x := 1 ;
   save_org.y := 1 ;
   save_size.x := 80 ;
   save_size.y := 25 ;
   move_to_scrn(save_org,save_size,screen) ;
   freemem(screen,4000) ;
   IF window_list <> NIL
    THEN
     WITH window_list^ DO
      BEGIN
       window(abs_org.x,abs_org.y,abs_org.x + window_size.x - 1,
              abs_org.y + window_size.y - 1) ;
       gotoxy(cursor_pos.x,cursor_pos.y) ;
       textcolor(fore_color) ;
       textbackground(back_color) ;
      END
    ELSE window(1,1,80,25) ;
  END ; (* scrn_refresh *)


 PROCEDURE close_window(win_ptr : window_ptr) ;
  (* remove the window from the window_list, and then call scrn_refesh *)
  (* update the screen. If win_ptr is the current window, the next window *)
  (* becomes the active window *)
  VAR
   save_org,save_size : window_pos ;

  FUNCTION found_window : boolean ;
   VAR
    p : window_ptr ;
    found : boolean ;
   BEGIN
    found := false ;
    p := window_list ;
    WHILE (p <> NIL) AND (NOT found) DO
     BEGIN
      found := (win_ptr = p) ;
      p := p^.next_window ;
     END ;
    found_window := found ;
   END ; (* found_window *)

  BEGIN
   IF found_window
    THEN
     BEGIN
      IF win_ptr <> window_list
       THEN save_window ;
      get_window_co_ords(win_ptr,save_org,save_size) ;
      del_desc(win_ptr) ;
      IF win_ptr^.scrn_area <> NIL
       THEN freemem(win_ptr^.scrn_area,2 * save_size.x * save_size.y) ;
      freemem(win_ptr,sizeof(window_desc)) ;
      scrn_refresh ;
     END ;
  END ; (* close_window *)

(* ///////////////////// Window routines for this program ////////// *)

 PROCEDURE wait ;
  (* Display a message at bottom of screen and and wait for user to *)
  (* press a key *)
  VAR
   ch : char ;
   old_window : window_ptr ;
  BEGIN
   old_window := window_list ;
   use_window(message_window) ;
   clrscr ;
   gotoxy(1,2) ;
   window_write('Press any key to continue ') ;
   read(kbd,ch) ;
   clrscr ;
   use_window(old_window) ;
  END ; (* wait *)


 PROCEDURE init_windows ;
  (* Initialize windows for this program *)
  BEGIN
   clrscr ;
   get_monitor_type ;
   IF monitor_kind = mono_monitor
    THEN button_fore := blue
    ELSE button_fore := yellow ;
   button_back := black ;
   window_list := NIL ;
   message_window := open_window(2,23,78,2,false,'',white,black,white) ;
   IF monitor_kind = mono_monitor
    THEN text_window := open_window(2,2,78,20,true,'Opt',white,black,white)
    ELSE text_window := open_window(2,2,78,20,true,'Opt',white,blue,white) ;
   gotoxy(32,5) ;
   window_writeln('OPT    ver 1.0') ;
   gotoxy(31,9) ;
   window_writeln('Copyright [c] 1987') ;
   gotoxy(31,11) ;
   window_writeln('Knowledge Garden Inc.') ;
   gotoxy(31,12) ;
   window_writeln('473A Malden Bridge Rd.') ;
   gotoxy(31,13) ;
   window_writeln('Nassau, NY 12123') ;
   wait ;
   clrscr ;
  END ; (* init_windows *)


 PROCEDURE finish_up ;
  (* Clean up screen before leaving *)
  BEGIN
   window(1,1,80,25) ;
   textcolor(white) ;
   textbackground(black) ;
   clrscr ;
  END ; (* finish_up *)


 PROCEDURE error(msg : string80) ;
  (* Display a message and wait for the user to read it *)
  VAR
   old_window,error_window : window_ptr ;
  BEGIN
   old_window := window_list ;
   error_window := open_window(10,10,60,3,true,'Error',white,red,white) ;
   window_writeln('') ;
   window_write(msg) ;
   wait ;
   close_window(error_window) ;
   use_window(old_window) ;
  END ; (* error *)

(* \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\ *)
