'From Objectworks\Smalltalk(R), Release 4.1 of 15 April 1992 on 5 April 1993 at 9:12:53 pm'!



Rectangle subclass: #CoolDrawRectangle
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Framework'!


!CoolDrawRectangle methodsFor: 'initialize'!

on: aGraph origin: originPoint corner: cornerPoint
	"Init the receivers'  SkyBlueVariables on aGraph"
	self obsoleteMessage.
"Use on:origin:corner:usingName:
	origin := SkyBlueVariable newNamed: 'origin', self asOop printString in: aGraph with: originPoint.
	corner := SkyBlueVariable newNamed: 'corner', self asOop printString in: aGraph with: cornerPoint.
	^self"!

on: aGraph origin: originPoint corner: cornerPoint usingName: aName owner: own 
	"Init the receivers' CoolDrawVariables on aGraph"

	origin := CoolDrawPoint
				newIn: aGraph
				with: originPoint
				owner: own.
	corner := CoolDrawPoint
				newIn: aGraph
				with: cornerPoint
				owner: own.
	^self! !

!CoolDrawRectangle methodsFor: 'private'!

computeBounds
	"Answer the Rectangle which minimally bounds the coordinate region used
	by the receiver, independent of considerations such as line width."

	^Rectangle origin: self origin corner: self corner! !

!CoolDrawRectangle methodsFor: 'printing'!

printOn: aStream 
	"Append to the argument aStream a sequence of characters that identifies the receiver.
	The general format is
		originPoint corner: cornerPoint."

	self origin printOn: aStream.
	aStream nextPutAll: ' corner: '.
	self corner printOn: aStream!

storeOn: aStream
	"Append to the argument aStream a sequence of characters that is an expression 
	whose evaluation creates a rectangle similar to the receiver.  The general format
	for rectangles is
		( class-name origin: aNumber corner: aNumber)"


	aStream nextPut: $(;
	nextPutAll: self species name;
	nextPutAll: ' origin: ';
	store: self origin;
	nextPutAll: ' corner: ';
	store: self corner;
	nextPut: $).! !

!CoolDrawRectangle methodsFor: 'copying'!

copyUsing: aCopyState
	origin := aCopyState copy: origin.
	corner := aCopyState copy: corner!

postCopy
	"Unlike a Rectangle we don't ask our super (Object) to postCopy"
	"super postCopy."
	self origin: (self origin copy).
	self corner: (self corner copy).! !

!CoolDrawRectangle methodsFor: 'transforming'!

moveBy: aPoint 
	"Change the corner positions of the receiver so that its area translates by
	the amount defined by the argument, aPoint."

	self origin: (self origin + aPoint).
	self corner: (self corner + aPoint)!

moveTo: aPoint 
	"Change the corners of the receiver so that its top left position is aPoint."

	self corner: (self corner + aPoint - self origin).
	self origin: aPoint!

scaledBy: scale 
	"Answer a new Rectangle scaled by the argument scale, a Point or a scalar."

	^ self species vertex: self origin * scale vertex: self corner * scale!

translatedBy: factor 
	"Answer a ourselves translated by factor, a Point or a scalar."

	^self  origin: self origin + factor corner: self corner + factor! !

!CoolDrawRectangle methodsFor: 'truncation and round off'!

rounded
	"Answer a Rectangle whose origin and corner are rounded."

	^self species origin: self origin rounded corner: self corner rounded!

truncated
	"Answer a Rectangle whose origin and corner are truncated."

	^self species origin: self origin truncated corner: self corner truncated! !

!CoolDrawRectangle methodsFor: 'testing'!

contains: aRectangle 
	"Answer whether the receiver is equal to aRectangle or whether aRectangle 
	is contained within the receiver."

	^aRectangle origin >= self origin and: [aRectangle corner <= self corner]!

containsPoint: aPoint 
	"Answer whether the argument aPoint is within the receiver."

	^self origin <= aPoint and: [aPoint < self corner]!

intersects: aRectangle 
	"Answer whether aRectangle intersects the receiver anywhere."

	^ self origin < aRectangle corner
		and: [aRectangle origin <  self corner
		and: [ origin < self corner
		and: [aRectangle origin < aRectangle corner]]]! !

!CoolDrawRectangle methodsFor: 'rectangle functions'!

areasOutside: aRectangle
	"Answer a Collection of Rectangles comprising the parts of the
	receiver that do not lie within aRectangle."

	| areas yOrigin yCorner |
		"Make sure the intersection is non-empty"
	(self intersects: aRectangle)
		ifFalse: [^Array with: self].
	areas := OrderedCollection new.
	aRectangle origin y > self origin y
		ifTrue: [areas add: (self origin corner: self corner x @ (yOrigin := aRectangle origin y))]
		ifFalse: [yOrigin := self origin y].
	aRectangle corner y < self corner y
		ifTrue: [areas add: (self origin x @ (yCorner := aRectangle  corner y) corner: self corner)]
		ifFalse: [yCorner := self corner y].
	aRectangle origin x > self origin x 
		ifTrue: [areas add: (self origin x @ yOrigin corner: aRectangle origin x @ yCorner)].
	aRectangle corner x < self corner x 
		ifTrue: [areas add: (aRectangle corner x @ yOrigin corner: self corner x @ yCorner)].
	^areas!

expandedBy: delta 
	"Answer a Rectangle that is outset from the receiver by delta.   
	 delta is a Rectangle, Point, or scalar."

	(delta isKindOf: Rectangle)
		ifTrue: [^self species 
					origin: self origin - delta origin 
					corner: self corner + delta corner]
		ifFalse: [^self species  
					origin: self origin - delta 
					corner: self corner + delta]!

insetBy: delta 
	"Answer a Rectangle that is inset from the receiver by delta.   
	 delta is a Rectangle, Point, or scalar."

	(delta isKindOf: Rectangle)
		ifTrue: [^self species 
					origin: self origin + delta origin 
					corner: self corner - delta corner]
		ifFalse: [^self species 
					origin: self origin + delta 
					corner: self corner - delta]!

insetOriginBy: originDeltaPoint cornerBy: cornerDeltaPoint 
	"Answer a Rectangle that is inset from the receiver by a given amount in the 
	origin and corner."

	^self species
		origin: self origin + originDeltaPoint
		corner: self corner - cornerDeltaPoint!

intersect: aRectangle 
	"Answer a Rectangle that is the area in which the receiver overlaps with 
	aRectangle. "

	^self species 
		origin: (self origin max: aRectangle origin)
		corner: (self corner min: aRectangle corner)!

merge: aRectangle 
	"Answer a Rectangle that contains both the receiver and  the
	argument aRectangle."

	^self species 
		origin: (self origin min: aRectangle origin)
		corner: (self corner max: aRectangle corner)! !

!CoolDrawRectangle methodsFor: 'comparing'!

= aRectangle 
	"Answer whether the receiver's species, origin and corner match those of
	the argument, aRectangle."

	(aRectangle species = Rectangle class) | (aRectangle species = CoolDrawRectangle)
		ifTrue: [^origin contents = aRectangle origin and: [corner contents = aRectangle corner]]
		ifFalse: [^false]!

hash
	"Answer a SmallInteger unique to the receiver."

	^origin contents hash bitXor: corner contents hash! !

!CoolDrawRectangle methodsFor: 'accessing'!

bottom
	"Answer the position of the receiver's bottom horizontal line."

	^self corner y!

bottom: anInteger 
	"Set the position of the bottom horizontal line of the receiver."

	corner contents: (self corner y: anInteger)!

bottomCenter
	"Answer the point at the center of the bottom horizontal line of the receiver."

	^((self origin x + self corner x) // 2) @ self corner y!

bottomLeft
	"Answer the point at the left edge of the bottom horizontal line of the receiver."

	^self origin x @ self corner y!

bottomRight
	"Answer the point at the right edge of the bottom horizontal line of the receiver."

	^self corner!

bottomRight: bottomRightPoint 
	"Set the position of the right corner of the bottom horizontal line of the receiver."

     self corner: bottomRightPoint!

center
	"Answer the point at the center of the receiver."

	^((self origin x + self corner x) // 2) @ ((self origin y + self corner y) // 2)!

corner
	"Answer the point at the bottom right corner of the receiver."

	^ corner contents!

corner: cornerPoint 
	"Set the point at the bottom right corner of the receiver."

	corner contents: cornerPoint!

extent
	"Answer a Point representing the extent of the receiver, that is
	one whose x coordinate is the width and whose y coordinate is the height."

	^self corner - self origin!

extent: extentPoint 
	"Set the extent (width and height) of the receiver to be the 
	argument extentPoint."

	self corner: (self origin + extentPoint)!

height
	"Answer the height of the receiver."

	^self corner y - self origin y!

height: heightInteger 
	"Change the receiver's bottom y to make its height 
	the argument heightInteger."

	self corner: (self corner y: (self origin y + heightInteger))!

left
	"Answer the position of the receiver's left vertical line."

	^self origin x!

left: anInteger 
	"Set the position of the receiver's left vertical line."

	self origin: (self origin x: anInteger)!

origin
	"Answer the point at the top left corner of the receiver."

	^origin contents!

origin: originPoint 
	"Set the point at the top left corner of the receiver."

	origin contents: originPoint!

origin: originPoint corner: cornerPoint
	"Set the points at the top left corner and the bottom right corner of the receiver."

	self origin: originPoint.
	self corner: cornerPoint!

origin: originPoint extent: extentPoint
	"Set the point at the top left corner of the receiver to be originPoint and
	set the width and height of the receiver to be extentPoint."

	self origin: originPoint.
	self corner: (origin contents + extentPoint)!

right
	"Answer the position of the receiver's right vertical line."

	^self corner x!

right: anInteger 
	"Set the position of the receiver's right vertical line."

	self corner: (self corner x: anInteger)!

top
	"Answer the position of the receiver's top horizontal line."

	^self origin y!

top: anInteger 
	"Set the position of the receiver's top horizontal line."

	self origin: (self origin y: anInteger)!

topCenter
	"Answer the point at the center of the receiver's top horizontal line."

	^((self origin x + self corner x) // 2) @ self origin y!

topLeft
	"Answer the point at the top left corner of the receiver's top horizontal line."

	^self origin!

topLeft: topLeftPoint 
	"Set the point at the top left corner of the receiver's top horizontal line."

	self origin: topLeftPoint!

topRight
	"Answer the point at the top right corner of the receiver's top horizontal line."

	^self corner x @ self origin y!

width
	"Answer the width of the receiver."

	^self corner x - self origin x!

width: widthInteger 
	"Change the receiver's right vertical line to make its width widthInteger."

	self corner: (self corner x: (self origin x + widthInteger))! !

!CoolDrawRectangle methodsFor: 'constrainable accessing'!

constrainableCorner
	^corner!

constrainableOrigin
	^origin! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

CoolDrawRectangle class
	instanceVariableNames: ''!


!CoolDrawRectangle class methodsFor: 'conversion'!

asRectangle
	"Answer a Rectangle which has the same dimensions as the receiver."
	^Rectangle origin: self origin corner: self corner! !

!CoolDrawRectangle class methodsFor: 'instance creation'!

newOn: aGraph with: aRectangle
	self obsoleteMessage.
	"^self new on: aGraph origin: aRectangle origin corner: aRectangle corner"!

newOn: aGraph with: aRectangle usingName: aName owner: own 
	"Answer an instance of the receiver with the dimensions of 
	aRectangle and constraint variables created in aGraph"

	^self new
		on: aGraph
		origin: aRectangle origin
		corner: aRectangle corner
		usingName: aName
		owner: own! !


Model subclass: #Drawing
	instanceVariableNames: 'figures damagedRegion pageResolution constraintGraph constraintClusters '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Framework'!
Drawing comment:
'A Drawing captures the essence of a graphical drawing.  It contains a number of Figures each of which have some visual representation.   A Drawing keeps a region of damage that reflects changes in the visual appearance of the Drawing.  This would normally be left to the underlying windowing system except this causes excessive flickering during redrawing.  When a Drawing redraws its damagedRegion, it draws it off-screen and then copies the region directly to the screen.

Instance Variables :
figures		<OrderedCollection>
	The figures contained in the Drawing.  The figures are ordered to preserve ''on-top-of-ness''.  The first figure in figures is the topmost one.
damagedRegion	<Rectangle | nil>	
	The area of the Drawing which has been damaged by the insertion, deletion, or motion of one or more figures.
pageResolution	<Point>
	The size of the printed page in pixels.

'!


!Drawing methodsFor: 'initialize'!

initialize
	figures := OrderedCollection new.
	constraintGraph := ColbaltBlueConstraintGraph new.
	constraintClusters := OrderedCollection new.
	self magnification: 1.!

installInEditor: anEditor
	"This can be overriden to change the tools, menu, etc. of the editor."!

release
	super release.
	figures do: [:each | each isFigure ifTrue: [each release]].
	figures := nil.
	constraintGraph release.
	constraintGraph := nil.
	constraintClusters := nil! !

!Drawing methodsFor: 'accessing'!

bottom
        ^figures last!

boundingBox
	figures isEmpty ifFalse: [^figures inject: figures first displayBox into: [:sum :each | sum merge: each displayBox]]
		ifTrue: [^0 @ 0 extent: 0 @ 0]!

figureAt: aPoint
        | aFigure | 
        figures do:
                [:each |
                aFigure := each figureAt: aPoint.
                aFigure notNil ifTrue: [^aFigure]].
        ^nil!

figures
        ^figures!

figuresIn: aRectangle
        ^figures select: [:each | each containedBy: aRectangle]!

figuresIntersecting: aRectangle
        ^figures select: [:each | each intersects: aRectangle]!

magnification
        ^72.0 / pageResolution x!

magnification: aNumber
        pageResolution := 72.0@72.0 / aNumber!

members
        | aSet | 
        aSet := IdentitySet new.
        figures do: [:each | each do: [:aFigure | aSet add: aFigure]].
        ^aSet!

tools
	"Answer the OrderedCollection of Tools that my class defines."

	^self class tools!

top
        ^figures first! !

!Drawing methodsFor: 'page accessing'!

pageExtent
        ^(self pageSize - (self pageMargin * 2) * pageResolution) rounded!

pageMargin
        ^0.5 @ 0.5!

pageSize
        ^8.5 @ 11! !

!Drawing methodsFor: 'testing'!

isDrawing
        ^true!

isFigure
        ^false! !

!Drawing methodsFor: 'adding'!

add: aFigure
        figures addFirst: aFigure.
        aFigure container: self!

add: newFigure behind: oldFigure
        figures add: newFigure after: oldFigure.
        newFigure container: self!

addAll: aCollection
        aCollection reverseDo: [:each | self add: each]!

addLast: newFigure 
	"Add newFigure behind all others"

	self add: newFigure behind: figures last! !

!Drawing methodsFor: 'removing'!

delete: aFigure 
	"Remove aFigure as a component of this Drawing."

	| vars clusters |
	figures remove: aFigure ifAbsent: [].
	vars := aFigure allConstrainedVariables.
	vars do: [:var | constraintGraph removeVariable: var].
	clusters := self clustersConstraining: vars.
	clusters do: [:clu | self removeCluster: clu].
	aFigure container: nil!

deleteAll: aCollection
        aCollection do:
                [:each | self delete: each]!

remove: aFigure 
	"Remove aFigure as a component of this Drawing.
	Not to be confused with delete!!"

	figures remove: aFigure ifAbsent: [].
	aFigure container: nil!

removeAll: aCollection
	self notYetImplemented.
        aCollection do:
                [:each | self remove: each]! !

!Drawing methodsFor: 'copying'!

copy
	| aDrawing |
	aDrawing := self class new initialize.
	aDrawing magnification: self magnification.
	Locator copyWhile: [figures reverseDo: [:each | aDrawing add: each copy]].
	^aDrawing! !

!Drawing methodsFor: 'comparing'!

= aDrawing
        ^(aDrawing isKindOf: Drawing) and: [figures = aDrawing figures]!

hash
        ^figures hash! !

!Drawing methodsFor: 'sorting'!

bringToFront: aFigure
        (figures includes: aFigure)
            ifTrue: [figures remove: aFigure.
                     figures addFirst: aFigure]!

sendToBack: aFigure
        (figures includes: aFigure)
            ifTrue: [figures remove: aFigure.
                     figures addLast: aFigure]! !

!Drawing methodsFor: 'displaying'!

displayOn: aGraphicsContext
	| aRect |
	aRect := aGraphicsContext clippingBounds.
	figures reverseDo: [:each | 
		(each displayBox intersects: aRect) ifTrue: [each displayOn: aGraphicsContext]].
	self displayPageBreaksOn: aGraphicsContext.!

displayPageBreaksOn: aGraphicsContext 
	"Draw a dashed line that indicates the edge of an 8.5x11 page. This doesnt work yet, but it will."

"	| aPattern oldPaint |
	aGraphicsContext tilePhase: 8 @ 8.
	aPattern := Pattern from: (Image
					extent: 8 @ 8
					depth: 1
					palette: (MappedPalette with: aGraphicsContext medium background with: ColorValue black)
					bits: #[225 225 225 225 225 225 225 225 225 225 225 225 225 225 225 225 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 ]).
	oldPaint := aGraphicsContext paint.
	aGraphicsContext paint: aPattern.
	aGraphicsContext displayRectangularBorder: (0 @ 0 extent: 50 @ 50)
		at: 0 @ 0.
	aGraphicsContext paint: oldPaint"! !

!Drawing methodsFor: 'damaging'!

damagedRegion
	^damagedRegion!

damageRegion: aRectangle 
	damagedRegion := damagedRegion isNil
				ifTrue: [aRectangle]
				ifFalse: [damagedRegion merge: aRectangle]!

isDamaged
        ^damagedRegion notNil!

undamage
        damagedRegion := nil!

update: aFigure 
	"damageRegion should be sent by component.  The only exception is
	if a figure is in many drawings, like in HyperCard."

	self damageRegion: aFigure displayBox! !

!Drawing methodsFor: 'editing'!

edit
	"Schedule a DrawingEditor on myself."

	DrawingEditor openOn: self! !

!Drawing methodsFor: 'animating'!

step!

wait! !

!Drawing methodsFor: 'cooldraw'!

addCluster: cluster
	constraintClusters addLast: cluster!

clustersConstraining: variables 
	^constraintClusters select: [:each | each constrainsAnyOf: variables]!

constraintGraph
	^constraintGraph!

copyClustersUsing: aCopyState
	^(constraintClusters select: [:each | each onlyConstrainsAnyOf: aCopyState restrictToVariables]) collect: [:clu | aCopyState copy: clu]!

currentPlan
	^constraintGraph extractEditPlanUsingClass: CoolDrawPlan!

damageAll
	figures do: [:each | each willChange]!

removeCluster: cluster
	constraintClusters remove: cluster! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Drawing class
	instanceVariableNames: ''!


!Drawing class methodsFor: 'instance creation'!

fromFile: aString 
	"Read a drawing from the file named aString."

	(Filename named: aString) exists
		ifTrue: [^BinaryStorage fromFileNamed: aString]
		ifFalse: [^self new]!

new
        ^super new initialize! !


VisualComponent subclass: #Handle
	instanceVariableNames: 'locator '
	classVariableNames: 'HandleBitmap HandleSize '
	poolDictionaries: ''
	category: 'HotDraw-Framework'!
Handle comment:
'A Handle is used to manipulate another Figure.  Handles operate through Locators to change a certain aspect of a Figure.  Handles use the invoke: and invokeStep: methods.  These are used to permit the handle to affect its change while it is being pressed.

Instance Variables :
locator		<Locator>	Intermediate link between a Handle and its Figure.

Class Variable :
HandleSize	<Point>	The extent of the Rectangle used to draw Handles.

'!


!Handle methodsFor: 'accessing'!

bounds
        ^self displayBox!

center
        ^self displayBox center!

displayBox
        ^Rectangle origin: self origin extent: self extent!

extent
        ^HandleSize!

handles
        ^Array new!

locator
        ^locator!

menu
	^PopUpMenu
		labels: 'no menu' withCRs
		lines: #( )
		values: #(#yourself )!

origin
        ^locator value - (HandleSize // 2) rounded!

owner
        ^locator object!

owner: aFigure
        locator := locator copyOn: aFigure! !

!Handle methodsFor: 'invoking'!

initializeInvoke: aView!

invoke: aView 
	| oldPoint newPoint sensor datum |
	sensor := aView controller sensor.
	oldPoint := sensor cursorPoint.
	datum := self initializeInvoke: aView.
	aView hideHandlesWhile: [[sensor redButtonPressed]
			whileTrue: 
				[newPoint := sensor cursorPoint.
				self invokeStep: newPoint - oldPoint with: datum.
				"oldPoint := newPoint."
				aView repairDamage]].
	self terminateInvoke: datum!

invokeStep: deltaPoint
	self obsoleteMessage!

invokeStep: deltaPoint with: aDatum!

terminateInvoke: aDatum! !

!Handle methodsFor: 'displaying'!

displayOn: aGraphicsContext at: aPoint 
	| roundedPoint handleRectangle |
	roundedPoint := aPoint rounded.	"+ aGraphicsContext translation"
	handleRectangle := roundedPoint extent: HandleSize.
	locator object majorColorValue = ColorValue black
		ifTrue: [aGraphicsContext paint: ColorValue gray]
		ifFalse: [aGraphicsContext paint: ColorValue black].
	aGraphicsContext displayRectangle: handleRectangle!

old.displayOn: aGraphicsContext at: aPoint 
	"This is a revised version of the Handle drawing method. It is about twice as fast per 
	Handle as the original. The combination rule should be RasterOp reverse but that doesn't 
	reverse the Image on PCs and Macs. Check to see if the Handle is inside the visible 
	region because contestOfArea: fails otherwise."

	| clipRectangle dwgImage roundedPoint |
	((aGraphicsContext medium displayBox translatedBy: aGraphicsContext medium globalOrigin negated)
		contains: (aPoint + aGraphicsContext translation extent: HandleSize))
		ifTrue: 
			[roundedPoint := aPoint  rounded.
			clipRectangle := (roundedPoint + aGraphicsContext translation) extent: HandleSize.
			dwgImage := (aGraphicsContext medium contentsOfArea: clipRectangle) first.
			dwgImage
				copy: (0 @ 0 extent: HandleSize)
				from: 0 @ 0
				in: dwgImage
				rule: RasterOp reverse.
					"Replace this with 10 when it doesn't work"
					"Replace this with RasterOp reverse when it works."
			dwgImage displayOn: aGraphicsContext at: roundedPoint]! !

!Handle methodsFor: 'testing'!

canBeConnected
	^false!

isActive
        ^true!

isFor: aFigure
        ^self owner == aFigure! !

!Handle methodsFor: 'private'!

setLocator: aLocator 
	locator := aLocator! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Handle class
	instanceVariableNames: ''!


!Handle class methodsFor: 'instance creation'!

initialize
	"Handle initialize"
	HandleSize := 6 @ 6!

on: aFigure at: aSymbol
        ^self new setLocator: (Locator on: aFigure at: aSymbol)!

on: aFigure at: aSymbol with: anArgument
        ^self new setLocator: (Locator on: aFigure at: aSymbol with: anArgument)! !


Handle subclass: #TrackHandle
	instanceVariableNames: 'gettor constrainor '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Handles'!
TrackHandle comment:
'TrackHandle has been significantly modified between HotDraw and CoolDraw.  In both systems, a TrackHandle is a Handle used to make changes to a Figure by direct manipulation of the TrackHandle.  In HotDraw, the TrackHandle, using its Locator, senses a certain aspect of the Figure and uses the information to change the Figure.  In CoolDraw, the TrackHandle uses constraints to do the manipulation.

The protocol is as follows: when the handle is invoked, the initializeInvoke method creates an equality constraint between the constrainable variable that the handle refers to (side bar: each variable in a figure has two aspects - itself as a normal Smalltalk value and itself as a constrainable variable) and the mouse/cursor position.  Then a constraint plan is created to encapsulate the solution to the entire constraint graph of the drawing.  As the handle is moved, the constraint plan is repeatedly re-executed.  When the handle is released, the temporary mouse constraints are removed.
'!


!TrackHandle methodsFor: 'invoking'!

getConstrainableVariables: aView
	^self owner perform: gettor!

initializeInvoke: aView 
	| datum collec  |
	"Build a new datum (used to pass information within the invokeStep iteration).
	Building a new datum creates a SkyBlueVariable in the drawing's constraint graph.
	This variable is the 'mouse' variable and represents the current mouse position."
	datum := CoolDrawHandleDatum newIn: aView drawing.

	"Get the constrainable variable(s) that this handle is attached to."
	collec := self getConstrainableVariables: aView.

	"Set up the mouse constraint, i.e., a medium one-way constraint between
	the mouse variable and the variable that the handle is constraining.  Offset
	this by the current value of the constrained variable.  In other words, the
	mouse is assumed to be at 0@0 when over this handle."
	collec do: [:var | 
		(self perform: constrainor with: datum mousePoint with: var)
			do: [:each | datum addConstraint: each]].

	"Get the plan and we're ready to go.  We can get the 'edit plan' because the
	SkyBlue algorithm has already solved the stay plan.  (If you don't understand
	this comment then ignore it.)"
	datum plan: aView currentPlan.

	^datum!

invokeStep: deltaPoint with: aDatum 
	aDatum mousePoint contents: deltaPoint.
	aDatum plan execute.!

terminateInvoke: aDatum 
	"Remove the constraints that were added."
	aDatum removeConstraints.

	"Then remove the mouse variable, etc."
	aDatum release.! !

!TrackHandle methodsFor: 'constraints'!

mouseConstraint: mouseVar xy: dataVar 
	"Constrains the v1 variable (a Point) with a one-way equality 
	constraint from the v0 variable (a Point).  Constraint both x and y."

	| offset aStrength him himmethod1 her hermethod1 |
	offset := dataVar contents.
	aStrength := SkyBlueStrength medium.
	him := SkyBlueConstraint new.
	him name: mouseVar constrainableX printString , ' =x=> ' , dataVar constrainableX printString.
	him strength: aStrength.
	him variables: (Array with: mouseVar constrainableX with: dataVar constrainableX).
	him errorfunction: [:vars | ((vars at: 1) contents + offset x - (vars at: 2) contents) abs].
	himmethod1 := SkyBlueMethod new.
	himmethod1 inputs: (Array with: mouseVar constrainableX).
	himmethod1 outputs: (Array with: dataVar constrainableX).
	himmethod1 external: true.
	himmethod1 block: [:in :out | (out at: 1) contents: (in at: 1) contents + offset x].
	him methods: (Array with: himmethod1).

	her := SkyBlueConstraint new.
	her name: mouseVar constrainableY printString , '=y=>' , dataVar constrainableY printString.
	her strength: aStrength.
	her variables: (Array with: mouseVar constrainableY with: dataVar constrainableY).
	her errorfunction: [:vars | ((vars at: 1) contents + offset y - (vars at: 2) contents) abs].
	hermethod1 := SkyBlueMethod new.
	hermethod1 inputs: (Array with: mouseVar constrainableY).
	hermethod1 outputs: (Array with: dataVar constrainableY).
	hermethod1 external: true.
	hermethod1 block: [:in :out | (out at: 1) contents: (in at: 1) contents + offset y].
	her methods: (Array with: hermethod1).
	^Array with: him with: her!

mouseConstraint: mouseVar yDiv10: dataVar 
	"Constrains the v1 variable (an Integer) with a one-way 
	equality constraint of the y coordinate of v0 divided by 10."

	| me memethod1 offset v0 v1 |
	v0 := mouseVar constrainableY.
	v1 := dataVar.
	me := SkyBlueConstraint new.
	me name: v0 printString , ' =/10=> ' , v1 printString.
	me strength: SkyBlueStrength medium.
	offset := v1 contents.
	me variables: (Array with: v0 with: v1).
	me errorfunction: [:vars | (((vars at: 1) contents // 10) + offset - (vars at: 2) contents) abs].
	memethod1 := SkyBlueMethod new.
	memethod1 inputs: (Array with: v0).
	memethod1 outputs: (Array with: v1).
	memethod1 external: true.
	memethod1 block: [:in :out | (out at: 1) contents: (in at: 1) contents // 10 + offset].
	me methods: (Array with: memethod1).
	^Array with: me!

mouseConstraint: mouseVar yDiv4: dataVar 
	"Constrains the v1 variable (an Integer) with a one-way 
	equality constraint of the y coordinate of v0 divided by 4."

	| me memethod1 offset v0 v1 |
	v0 := mouseVar constrainableY.
	v1 := dataVar.
	me := SkyBlueConstraint new.
	me name: v0 printString , ' =/4=> ' , v1 printString.
	me strength: SkyBlueStrength medium.
	offset := v1 contents.
	me variables: (Array with: v0 with: v1).
	me errorfunction: [:vars | (((vars at: 1) contents // 4) + offset - (vars at: 2) contents) abs].
	memethod1 := SkyBlueMethod new.
	memethod1 inputs: (Array with: v0).
	memethod1 outputs: (Array with: v1).
	memethod1 external: true.
	memethod1 block: [:in :out | (out at: 1) contents: (in at: 1) contents // 4 + offset].
	me methods: (Array with: memethod1).
	^Array with: me! !

!TrackHandle methodsFor: 'private'!

setSense: senseSelector change: changeSelector 
	self obsoleteMessage.
	"sense := senseSelector.
	change := changeSelector"!

setUsing: getSelector constraining: constrainSelector
	gettor := getSelector.
	constrainor := constrainSelector! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

TrackHandle class
	instanceVariableNames: ''!


!TrackHandle class methodsFor: 'instance creation'!

on: aFigure at: aSymbol using: usingSymbol constraining: constrainingSymbol
       ^(self on: aFigure at: aSymbol)
                setUsing: usingSymbol constraining: constrainingSymbol!

on: aFigure at: aSymbol with: anArgument change: changeSelector
self obsoleteMessage.
       "^(self on: aFigure at: aSymbol with: anArgument)
                setSense: nil change: changeSelector"!

on: aFigure at: aSymbol with: anArgument using: usingSymbol constraining: constrainingSymbol
       ^(self on: aFigure at: aSymbol with: anArgument)
                   setUsing: usingSymbol constraining: constrainingSymbol! !

!TrackHandle class methodsFor: 'corner handles'!

allCornersOf: aFigure
        ^OrderedCollection new
                add: (self topLeftOf: aFigure);
                add: (self topRightOf: aFigure);
                add: (self bottomLeftOf: aFigure);
                add: (self bottomRightOf: aFigure);
                yourself!

bottomLeftOf: aFigure 
	^self
		on: aFigure
		at: #bottomLeft
		using: #constrainableBottomLeft
		constraining: #mouseConstraint:xy:!

bottomRightOf: aFigure 
	^self
		on: aFigure
		at: #bottomRight
		using: #constrainableBottomRight
		constraining: #mouseConstraint:xy:!

topLeftOf: aFigure 
	^self
		on: aFigure
		at: #topLeft
		using: #constrainableTopLeft
		constraining: #mouseConstraint:xy:!

topRightOf: aFigure 
	^self
		on: aFigure
		at: #topRight
		using: #constrainableTopRight
		constraining: #mouseConstraint:xy:! !

!TrackHandle class methodsFor: 'other handles'!

borderColorOf: aFigure 
	^self
		on: aFigure
		at: #offCenter:
		with: 0 @ 10
		using: #constrainableBorderColor
		constraining: #mouseConstraint:yDiv10:!

colorOf: aFigure 
	^self
		on: aFigure
		at: #offCenter:
		with: 10 @ 0
		using: #constrainableFillColor
		constraining: #mouseConstraint:yDiv10:!

positionOf: aFigure 
	^self
		on: aFigure
		at: #center
		using: #constrainablePosition
		constraining: #mouseConstraint:xy:!

startPointOf: aFigure 
	^self
		on: aFigure
		at: #startPoint
		using: #constrainableStartPoint
		constraining: #mouseConstraint:xy:!

stopPointOf: aFigure 
	^self
		on: aFigure
		at: #stopPoint
		using: #constrainableStopPoint
		constraining: #mouseConstraint:xy:!

widthOf: aFigure
	^self
		on: aFigure
		at: #offCenter:
		with: -10 @ 0
		using: #constrainableBorderWidth
		constraining: #mouseConstraint:yDiv4:! !


TrackHandle subclass: #SelectionTrackHandle
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Handles'!
SelectionTrackHandle comment:
'A SelectionTrackHandle (again, massively modified vis a vis HotDraw) is a handle that applies to the entire current selection of the drawing rather than just the figure to which the handle is attached.
'!


!SelectionTrackHandle methodsFor: 'invoking'!

getConstrainableVariables: aView 
	| collec |
	collec := IdentitySet new.
	aView selections do: [:each | collec addAll: (each perform: gettor)].
	^collec! !


Handle subclass: #ConnectionHandle
	instanceVariableNames: 'className connectionAction '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Handles'!
ConnectionHandle comment:
'ConnectionHandle is a Handle that creates a connectiing Figure from the Figure to which it belongs and the nearest other figure.  After the ConnectionHandle is released, the connecting Figure is added to the Drawing.  It has an optional connectionAction, which is a block that gets evaluated with the two Figures.  This makes it easy to set up specialized constraints between objects that are connected.

Instance Variable :
className	<Symbol>		The name of the Class of the connecting Figure which will be created (usually ConnectionFigure).
connectionAction <Block of: Figure of: Figure>   The first figure is the source of the Connection.
'!


!ConnectionHandle methodsFor: 'accessing'!

action: aBlock
	connectionAction := aBlock!

connectionClass
        ^Smalltalk at: className! !

!ConnectionHandle methodsFor: 'invoking'!

invoke: aView 
	| aFigure |
	aFigure := self findTarget: aView.
	aFigure isNil ifTrue: [^self].
	self connectTo: aFigure in: aView.
	connectionAction isNil ifFalse: [connectionAction value: self owner value: aFigure]! !

!ConnectionHandle methodsFor: 'private'!

connectTo: aFigure in: aView 
	| aConnectionFigure |
	aConnectionFigure := self connectionClass startLocation: locator stopLocation: aFigure locator onView: aView. 
	aView hideHandlesWhile: [aView addFigure: aConnectionFigure].!

findTarget: aView 
	| aFigure aSensor aGC myCenter newPoint |
	aSensor := aView controller sensor.
	aGC := aView graphicsContext.
	newPoint := myCenter := self center.
	aView hideHandles.
	[aSensor redButtonPressed]
		whileTrue: 
			[aView drawing damageRegion: ((Rectangle vertex: myCenter vertex: newPoint)
					expandedBy: 2).
			aView repairDamage.
			newPoint := aSensor cursorPoint.
			aFigure := aView figureAt: newPoint.
			(aFigure = self or: [(self isFor: aFigure)
					or: [aFigure notNil and: [aFigure canBeConnected not]]])
				ifTrue: [aFigure := nil].
			newPoint := aFigure isNil
						ifTrue: [newPoint]
						ifFalse: [aFigure connectionPosition].
			aGC displayLineFrom: myCenter to: newPoint].
	aView drawing damageRegion: ((Rectangle vertex: myCenter vertex: newPoint)
			expandedBy: 2).
	aView repairDamage.
	aView showHandles.
	^aFigure!

setClassName: aSymbol 
	className := aSymbol! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ConnectionHandle class
	instanceVariableNames: ''!


!ConnectionHandle class methodsFor: 'instance creation'!

on: aFigure at: aSymbol
        ^self on: aFigure at: aSymbol class: DependentLineFigure!

on: aFigure at: aSymbol class: aClass
        ^(super on: aFigure at: aSymbol) setClassName: aClass name! !


Model subclass: #DrawingEditor
	instanceVariableNames: 'tools currentTool fileName drawing '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Framework'!
DrawingEditor comment:
'DrawingEditor is a subclass of Model that represents a graphical drawing editor.  It is the Model of the MVC triad that includes a DrawingView and a DrawingViewController.  It shares the DrawingView and DrawingController with a Drawing.

Instance Variables:
tools		<Collection>
	A collection of tools that can be used on the current Drawing.  These tools are determined by the Drawing that this DrawingEditor is editing.
currentTool	<Tool>
	One of the Tools from the tools Collection.
fileName		<String>
	The filename where the current Drawing is stored.  Initially left nil then set to NewDrawing as a default.
drawing		<Drawing>
	The Drawing which is currently being edited.
'!


!DrawingEditor methodsFor: 'accessing'!

currentTool
	^currentTool!

currentTool: aTool 
	"Set my currentTool to aTool, assuming aTool is a member of my tools."

	currentTool := aTool.
	self changed: #tool.!

drawing
	^drawing copy!

menu
	^PopUpMenu
		labels: 'inspect\save\load\copy\cut\paste\group\ungroup\compose\decompose' withCRs
		lines: #(1 3 6 8)
		values: #(#inspectDrawing #saveDrawing #loadDrawing #copy #cut #paste #group #ungroup #compose #decompose)!

tools 
	^tools!

tools: aToolCollection 
	"Set my tools to aToolCollection and reset the currentTool."

	tools := aToolCollection.
	tools isNil ifFalse: [self currentTool: tools first]! !

!DrawingEditor methodsFor: 'drawing operations'!

inspectDrawing
	drawing inspect!

loadDrawing
	"Read a Drawing from a file."

	| aDrawing myDrawingViews |
	fileName isNil ifTrue: [fileName := 'NewDrawing'].
	fileName := DialogView requestFileName: 'Load drawing from file' default: '/project/thinglabx/examples/*.cool' .
	(fileName isEmpty not and: [(Filename named: fileName) exists])
		ifTrue: [(Smalltalk classNames includes: #BinaryObjectStorage)
				ifFalse: 
					[DialogView notify: 'Binary Object Storage is not available.'.
					^self]
				ifTrue: [Cursor read
						showWhile: 
							[| t |
							t := BinaryObjectStorage onOld: (Filename named: fileName) readStream.
							aDrawing := t next.
							t close]]]
		ifFalse: 
			[Transcript show: 'New Drawing'; cr.
			aDrawing := Drawing new].
	self setDrawing: aDrawing.
	myDrawingViews := self myDependents select: [:each | each class == DrawingView].
	myDrawingViews do: [:each | each setDrawing: aDrawing].
	tools do: [:each | each controller: myDrawingViews first controller].
	drawing damageRegion: drawing boundingBox.
	drawing figures do: [:each | (each isKindOf: CachedFigure)
			ifTrue: [each fillCache]].
	myDrawingViews
		do: 
			[:each | 
			each invalidate.
			each repairDamage]!

saveDrawing
	"Write my Drawing to a file. Nil out the caches of CachedFigures because they don't 
	survive the saving operation (they use Masks and Pixmaps). Also break dependencies 
	before I save the drawing"

	| osFilter deps |
	fileName isNil ifTrue: [fileName := 'NewDrawing'].
	fileName := DialogView requestFileName: 'Save drawing in file' default: fileName.
	fileName isEmpty ifTrue: [^self].
	osFilter := Filename named: fileName.
	(osFilter canBeWritten and: [osFilter exists
			ifTrue: [DialogView confirm: 'That file exists, replace it']
			ifFalse: [true]])
		ifTrue: [(Smalltalk classNames includes: #BinaryObjectStorage)
				ifFalse: [DialogView notify: 'Binary Object Storage is not available.']
				ifTrue: 
					[drawing figures do: [:each | (each isKindOf: CachedFigure)
							ifTrue: [each damageCache]].
					deps := drawing myDependents.
					drawing myDependents: nil.
					Cursor write showWhile: [(BinaryObjectStorage onNew: osFilter writeStream) nextPut: drawing; close].
					drawing figures do: [:each | (each isKindOf: CachedFigure)
							ifTrue: [each fillCache]].
					drawing myDependents: deps]].! !

!DrawingEditor methodsFor: 'private'!

setDrawing: aDrawing 
	"Set my drawing to aDrawing."

	drawing := aDrawing.
	aDrawing installInEditor: self! !

!DrawingEditor methodsFor: 'cooldraw'!

currentPlan
	^drawing currentPlan! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

DrawingEditor class
	instanceVariableNames: ''!


!DrawingEditor class methodsFor: 'instance creation'!

new
	| editor |
	editor := super new.
	editor tools: self defaultTools.
	^editor!

open
	"DrawingEditor open"

	self openOn: self drawingClass new!

openOn: aDrawing 
	| aDrawingView aToolPaletteView aWindow container toolPaletteSize aDrawingEditor translatingView |
	aDrawingEditor := self new setDrawing: aDrawing.
	toolPaletteSize := aDrawingEditor tools size * (Tool toolImageSize + Tool toolIconSpacing + Tool toolHiLiteSize) x + Tool toolIconSpacing x + 4.
	aDrawingView := DrawingView on: aDrawingEditor. 
	aDrawingView setDrawing: aDrawing.
	aDrawingEditor currentTool controller: aDrawingView controller.	"for CoolDrawStartupTool"
	aToolPaletteView := ToolPaletteView on: aDrawingEditor.
	aWindow := ScheduledWindow new.
	aWindow label: 'Drawing'.
	aWindow minimumSize: 380 @ (toolPaletteSize max: 300).
	container := CompositePart new.
	container add: aToolPaletteView borderedIn: (0 @ 0 extent: 4 + Tool toolImageSize x + Tool toolHiLiteSize x + (2 * Tool toolIconSpacing) x @ toolPaletteSize).
	translatingView := TranslatingWrapper on: aDrawingView.
	container add: translatingView borderedIn: (4 + Tool toolImageSize x + Tool toolHiLiteSize x + (2 * Tool toolIconSpacing) x @ 0 corner: 600 @ 600).
	aWindow component: container.
	aWindow open.!

openOn: aDrawing withLabel: aLabel
	| aDrawingView aToolPaletteView aWindow container toolPaletteSize aDrawingEditor translatingView |
	aDrawingEditor := self new setDrawing: aDrawing.
	toolPaletteSize := aDrawingEditor tools size * (Tool toolImageSize + Tool toolIconSpacing + Tool toolHiLiteSize) x + Tool toolIconSpacing x + 4.
	aDrawingView := DrawingView on: aDrawingEditor. 
	aDrawingView setDrawing: aDrawing.
	aDrawingEditor currentTool controller: aDrawingView controller.	"for CoolDrawStartupTool"
	aToolPaletteView := ToolPaletteView on: aDrawingEditor.
	aWindow := ScheduledWindow new.
	aWindow label: aLabel.
	aWindow minimumSize: 360 @ (toolPaletteSize max: 300).
	container := CompositePart new.
	container add: aToolPaletteView borderedIn: (0 @ 0 extent: 4 + Tool toolImageSize x + Tool toolHiLiteSize x + (2 * Tool toolIconSpacing) x @ toolPaletteSize).
	translatingView := TranslatingWrapper on: aDrawingView.
	container add: translatingView borderedIn: (4 + Tool toolImageSize x + Tool toolHiLiteSize x + (2 * Tool toolIconSpacing) x @ 0 corner: 600 @ 600).
	aWindow component: container.
	aWindow open.! !

!DrawingEditor class methodsFor: 'defaults'!

defaultTools
	"Answer an OrderedCollection of the tools I can use."

	^(OrderedCollection new) add: SelectionTool new;
		add: ScrollingTool new; 
		add: FigureActionTool bringToFront;
		add: FigureActionTool sendToBack; 
		add: FigureActionTool delete;
		add: LineFigure creationTool;
		add: ArrowFigure creationTool;
		add: RectangleFigure creationTool;
		add: EllipseFigure creationTool;
		add: TextFigure creationTool;
		add: ImageFigure creationTool;
		yourself!

drawingClass
	^Drawing! !


DrawingEditor subclass: #CoolDrawingEditor
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Framework'!


!CoolDrawingEditor methodsFor: 'accessing'!

tools: aToolCollection 
	| t |
	super tools: aToolCollection.
	t := CoolDrawStartupTool newOrNil.
	t notNil ifTrue: [self currentTool: t]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

CoolDrawingEditor class
	instanceVariableNames: ''!


!CoolDrawingEditor class methodsFor: 'defaults'!

defaultTools
	"Answer an OrderedCollection of the tools I can use."

	^(OrderedCollection new) add: SelectionTool new;
		add: ScrollingTool new; 
		add: FigureActionTool bringToFront;
		add: FigureActionTool sendToBack; 
		add: FigureActionTool delete;
		add: LineFigure creationTool;
		add: ArrowFigure creationTool;
		add: RectangleFigure creationTool;
		add: EllipseFigure creationTool;
		"add: TextFigure creationTool;"
		"add: ImageFigure creationTool;"
		add: ConstraintCreationTool newOne;
		add: ConstraintCreationTool newTwo;
		add: ConstraintCreationTool newThree;
		add: ConstraintDeletionTool new;
		yourself! !


!ColorValue methodsFor: 'comparing'!

= aColor
	self == aColor ifTrue: [^true].
	^aColor class == self class and:
		[aColor scaledRed = red and:
			[aColor scaledGreen = green and:
				[aColor scaledBlue = blue]]]! !


!DialogView class methodsFor: 'other dialogs'!

notify: messageString 
	"Notify the user of messageString and do not dismiss until OK is pressed."
	"DialogView notify: 'This is an example notifier that is really long.'"

	| aModel composite yp ok bang |
	aModel := ValueHolder with: nil.
	bang := Image extent: (32 @ 32) depth: 1 palette: MonoMappedPalette blackWhite bits: #[
2r00000000 2r00000000 2r00000000 2r00000000  
2r01111111 2r11111111 2r11111111 2r11111110  
2r01111111 2r11111111 2r11111111 2r11111110  
2r01111111 2r11111111 2r11111111 2r11111110  
2r01111111 2r11111100 2r00111111 2r11111110  
2r01111111 2r11110000 2r00001111 2r11111110  
2r01111111 2r11110000 2r00000111 2r11111110  
2r01111111 2r11100000 2r00000111 2r11111110  
2r01111111 2r11100000 2r00000111 2r11111110  
2r01111111 2r11100000 2r00000111 2r11111110  
2r01111111 2r11100000 2r00000111 2r11111110  
2r01111111 2r11110000 2r00000111 2r11111110  
2r01111111 2r11110000 2r00001111 2r11111110  
2r01111111 2r11111000 2r00011111 2r11111110  
2r01111111 2r11111000 2r00011111 2r11111110  
2r01111111 2r11111100 2r00111111 2r11111110  
2r01111111 2r11111110 2r01111111 2r11111110  
2r01111111 2r11111110 2r01111111 2r11111110  
2r01111111 2r11111111 2r11111111 2r11111110  
2r01111111 2r11111111 2r11111111 2r11111110  
2r01111111 2r11111110 2r01111111 2r11111110  
2r01111111 2r11111000 2r00011111 2r11111110  
2r01111111 2r11110000 2r00001111 2r11111110  
2r01111111 2r11110000 2r00001111 2r11111110  
2r01111111 2r11110000 2r00001111 2r11111110  
2r01111111 2r11110000 2r00001111 2r11111110  
2r01111111 2r11111000 2r00011111 2r11111110  
2r01111111 2r11111110 2r01111111 2r11111110  
2r01111111 2r11111111 2r11111111 2r11111110  
2r01111111 2r11111111 2r11111111 2r11111110  
2r01111111 2r11111111 2r11111111 2r11111110  
2r00000000 2r00000000 2r00000000 2r00000000  
] pad: 8.
	ok := LabeledBooleanView new model: ((PluggableAdaptor on: aModel)
					selectValue: true).
	ok beVisual: ' OK ' asText allBold asComposedText.
	ok controller beTriggerOnUp.
	(composite := self model: aModel) addVerticalSpace: 5; addVisual: bang atX: 0.5; addVerticalSpace: 5; addVisual: (messageString asText allBold asComposedText) atX: 0.5 ; addVerticalSpace: 5.
	ok := (BorderedWrapper on: ok)
				borderWidth: 2.
	yp := composite yPosition.
	composite indent: 1;
		addWrapper: ok
		atX: 0.5; yPosition: yp; addVerticalSpace: 10; open! !


View subclass: #ImageView
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Interface'!
ImageView comment:
'Used for CoolDrawStartupPictureEditor'!


!ImageView methodsFor: 'displaying'!

displayOn: aGraphicsContext
	model image displayOn: aGraphicsContext! !


!Text methodsFor: 'emphasis'!

makeSelectorBoldIn: aClass 
	"For formatting Smalltalk source code, set the emphasis of that 
	portion of 
	the receiver's string that parses as a message selector to be bold."

	| parser state |
	string size = 0 ifTrue: [^self].
	(parser := aClass parserClass new) parseSelector: string.
	self
		emphasizeFrom: 1
		to: (parser endOfLastToken min: string size)
		with: #bold.
	"Set the emphasis of comments to italic (in a stupid way, I admit)."
	state := nil.
	1 to: string size do: [:i | (string at: i) = $"
			ifTrue: [state isNil
				ifTrue: [state := i]
				ifFalse: 
					[self
						emphasizeFrom: state
						to: i
						with: #italic.
					state := nil]]]! !


Model subclass: #SkyBlueDemoBrowser
	instanceVariableNames: 'graph '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SkyBlue-Examples'!


!SkyBlueDemoBrowser methodsFor: 'accessing'!

graph: g
	graph := g! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SkyBlueDemoBrowser class
	instanceVariableNames: ''!


!SkyBlueDemoBrowser class methodsFor: 'instance creation'!

example1
	"SkyBlueDemoBrowser example1"
	"a = b = c"	

	| g v1 v2 v3 c1 c2 c3 c4 c5  |
	g := SkyBlueMVCConstraintGraph new.
	v1 := SkyBlueMVCVariable newIn: g.
	v2 := SkyBlueMVCVariable newIn: g.
	v3 := SkyBlueMVCVariable newIn: g.
	c1 := SkyBlueConstraint equalityBetween: v1 and: v2.
	c2 := SkyBlueConstraint equalityBetween: v2 and: v3.
	c3 := SkyBlueConstraint weakStayOn: v1.
	c4 := SkyBlueConstraint weakStayOn: v2.
	c5 := SkyBlueConstraint weakStayOn: v3.
	g addConstraint: c1.
	g addConstraint: c2.
	g addConstraint: c3.
	g addConstraint: c4.
	g addConstraint: c5.
	self openOn: g!

example2
	"SkyBlueDemoBrowser example2"
	"a + b = c"

	| g v1 v2 v3 c1  c3 c4 c5  |
	g := SkyBlueMVCConstraintGraph new.
	v1 := SkyBlueMVCVariable newIn: g with: 0.
	v2 := SkyBlueMVCVariable newIn: g with: 0.
	v3 := SkyBlueMVCVariable newIn: g with: 0.
	c1 := SkyBlueConstraint plusBetween: v1 and: v2 and: v3.
	c3 := SkyBlueConstraint weakStayOn: v1.
	c4 := SkyBlueConstraint weakStayOn: v2.
	c5 := SkyBlueConstraint weakStayOn: v3.
	g addConstraint: c1.
	g addConstraint: c3.
	g addConstraint: c4.
	g addConstraint: c5.
	self openOn: g!

example3
	"SkyBlueDemoBrowser example3"
	"a >= b"

	| g v1 v2  c1  c3 c4   |
	g := ColbaltBlueMVCConstraintGraph new.
	v1 := SkyBlueMVCVariable newIn: g with: 0.
	v2 := SkyBlueMVCVariable newIn: g with: 0.
	c1 := ColbaltBlueNonuniqueConstraint greaterEqualBetween: v1 and: v2.
	c3 := SkyBlueConstraint weakStayOn: v1.
	c4 := SkyBlueConstraint weakStayOn: v2.
	g addConstraint: c1.
	g addConstraint: c3.
	g addConstraint: c4.
	self openOn: g!

example4
	"SkyBlueDemoBrowser example4"
	"a >= b >= c" "max >= y >= min"

	| g v1 v2  c1  c3 c4   v3 c2 |
	g := ColbaltBlueMVCConstraintGraph new.
	v1 := SkyBlueMVCVariable newIn: g with: 0.
	v2 := SkyBlueMVCVariable newIn: g with: 0.
	v3 := SkyBlueMVCVariable newIn: g with: 0.
	c1 := ColbaltBlueNonuniqueConstraint greaterEqualBetween: v1 and: v2.
	c2 := ColbaltBlueNonuniqueConstraint greaterEqualBetween: v2 and: v3.
	c3 := SkyBlueConstraint weakStayOn: v1.
	c4 := SkyBlueConstraint weakStayOn: v2.
	g addConstraint: c1.
	g addConstraint: c2.
	g addConstraint: c3.
	g addConstraint: c4.
	self openOn: g!

example5
	"SkyBlueDemoBrowser example5"
	"a >= b >= c with a & c fixed" "b bounded by a and c"

	| g v1 v2  c1  c3 c4   v3 c2 |
	g := ColbaltBlueMVCConstraintGraph new.
	v1 := SkyBlueMVCVariable newIn: g with: 50.
	v2 := SkyBlueMVCVariable newIn: g with: 20.
	v3 := SkyBlueMVCVariable newIn: g with: 10.
	c1 := ColbaltBlueNonuniqueConstraint greaterEqualBetween: v1 and: v2.
	c2 := ColbaltBlueNonuniqueConstraint greaterEqualBetween: v2 and: v3.
	c3 := SkyBlueConstraint stayOn: v1.
	c4 := SkyBlueConstraint stayOn: v3.
	g addConstraint: c1.
	g addConstraint: c2.
	g addConstraint: c3.
	g addConstraint: c4.
	self openOn: g!

openOn: g 
	| me varbs topWindow topView frac offset tv |
	me := self new.
	me graph: g.
	varbs := g variables.
	topWindow := ScheduledWindow
				model: me
				label: 'SkyBlue Browser'
				minimumSize: 100 @ (varbs size * 50).
	topView := DependentComposite new.
	frac := 1.0 / varbs size.
	offset := 0.0.
	varbs
		do: 
			[:each | 
			tv := TextView
						on: each
						aspect: #text
						change: #acceptText:from:
						menu: #textMenu
						initialSelection: nil.
			topView add: (LookPreferences edgeDecorator on: tv)
				in: (0 @ offset extent: 1 @ frac).
			offset := offset + frac].
	topWindow component: topView.
	topWindow open! !


!Object methodsFor: 'cooldraw'!

bjorn: aString 
	"Browser browseAllCallsOn: #bjorn:"

	InputState default shiftDown ifTrue: [self halt: 'bjorn: ' , aString]!

ian: aString 
	"Browser browseAllCallsOn: #ian:"

	InputState default shiftDown ifTrue: [self halt: 'ian: ' , aString]!

notYetImplemented
	"Browser browseAllCallsOn: #notYetImplemented"

	| t r s |
	t := thisContext sender.
	r := t receiver class name.
	s := t method who at: 2.
	Object haltSignal raiseRequestWith: t errorString: r , '>>' , s , ' is not yet implemented'!

notYetImplemented: aString
	"Browser browseAllCallsOn: #notYetImplemented:"

	| t r s |
	t := thisContext sender.
	r := t receiver class name.
	s := t method who at: 2.
	Object haltSignal raiseRequestWith: t errorString: aString, ' in ', r , '>>' , s , ' is not yet implemented'!

obsoleteMessage
	"Browser browseAllCallsOn: #obsoleteMessage"
	self error: 'obsolete message invoked'! !


Object subclass: #CoolDrawConstraintDialog
	instanceVariableNames: 'drawing figure1 param1 figure2 param2 multiplier offset relation strength name '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Interface'!
CoolDrawConstraintDialog comment:
'CoolDrawConstraintDialog is the model for the constraint creation dialog window (as well as the creator of that window).  The information from the dialog window is stored in an instance of this object and then returned to the ConstraintCreationTool.  The dialog window has various bits of consistency checking (such as automatic updating of the offset) whose logic is incorporated in this class.  Other than that, this class is just a data repository.
'!


!CoolDrawConstraintDialog methodsFor: 'initialize'!

initialize: f1 and: f2 on: d
	figure1 := f1.
	figure2 := f2.
	drawing := d.
	param1 := ValueHolder with: nil.
	param2 := ValueHolder with: nil.
	relation := ValueHolder with: #=.
	offset := ValueHolder with: ''.
	multiplier := ValueHolder with: ''.
	strength := ValueHolder with: #required.
	param1 addDependent: self.
	param2 addDependent: self.
	^self!

initialize: f1 on: d
	figure1 := f1.
	drawing := d.
	param1 := ValueHolder with: nil.
	relation := ValueHolder with: #stay.
	strength := ValueHolder with: #strong.
	param1 addDependent: self.
	^self!

release
	drawing := nil.
	figure1 := nil.
	figure2 := nil.
	param1 removeDependent: self.
	param1 := nil.
	param2 removeDependent: self.
	param2 := nil.
	relation := nil.
	offset := nil.
	multiplier := nil.
	strength := nil! !

!CoolDrawConstraintDialog methodsFor: 'accessing'!

constrainableVariable1
	^figure1 perform: (figure1 convertToConstrainable: param1 value)!

constrainableVariable2
	^figure2 perform: (figure2 convertToConstrainable: param2 value)!

constraintGraph
	^drawing constraintGraph!

drawing
	^drawing!

multiplier
	^nil class evaluatorClass new
		evaluate: multiplier value
		in: nil
		to: nil
		notifying: self
		ifFail: [0]!

name
	^name isNil
		ifTrue: 
			[| s |
			s := 'custom ' , param1 value , ' ' , relation value.
			param2 notNil ifTrue: [s := s , ' ' , param2 value].
			s]
		ifFalse: [name]!

name: n
	name := n!

offset
	^nil class evaluatorClass new
		evaluate: offset value
		in: nil
		to: nil
		notifying: self
		ifFail: [0]!

relation
	^relation value!

strength
	^SkyBlueStrength perform: strength value! !

!CoolDrawConstraintDialog methodsFor: 'testing'!

isCustom
	^relation value = #custom!

isOk
	^relation value notNil
	& (relation value ~= #custom)
	& param1 value notNil
	& (param2 isNil or: [param2 value notNil])! !

!CoolDrawConstraintDialog methodsFor: 'adaptor'!

cancel
	relation value: nil.
	self changed: #cancel!

custom
	relation value: #custom.
	self changed: #cancel!

ok
	(offset notNil and: [offset dependents isEmpty not])
		ifTrue: [offset value: (offset dependents at: 1) displayContents asString].
	(multiplier notNil and: [multiplier dependents isEmpty not])
		ifTrue: [multiplier value: (multiplier dependents at: 1) displayContents asString].
	self changed: #ok!

valueHolderForMultipler
	^multiplier!

valueHolderForOffset
	^offset!

valueHolderForParamOf: aFigure 
	^figure1 == aFigure
		ifTrue: [param1]
		ifFalse: [param2]!

valueHolderForRelation
	^relation!

valueHolderForStrength
	^strength! !

!CoolDrawConstraintDialog methodsFor: 'updating'!

recomputeOffset: change2 
	| v1 v2 |
	param2 isNil ifTrue: [^self].
	param1 value isNil ifTrue: [^self].
	param2 value isNil ifTrue: [^self].
	v1 := figure1 perform: param1 value.
	v2 := figure2 perform: param2 value.
	v1 class == v2 class
		ifTrue: 
			[offset value: (v2 - v1) printString.
			multiplier value: 1 printString]
		ifFalse: [change2
				ifTrue: [param2 value: nil]
				ifFalse: [param1 value: nil]]!

update: anAspectSymbol with: aParameter from: aSender 
	(aSender == param1 or: [aSender == param2])
		ifTrue: [self recomputeOffset: (aSender == param1)]
		ifFalse: [super
				update: anAspectSymbol
				with: aParameter
				from: aSender]! !

!CoolDrawConstraintDialog methodsFor: 'printing'!

printOn: aStream
	aStream nextPutAll: self class name.
	aStream nextPutAll: '-'.
	aStream cr.
	figure1 printOn: aStream. aStream tab.
	param1 value printOn: aStream. aStream cr.
	figure2 printOn: aStream. aStream tab.
	param2 value printOn: aStream. aStream cr.
	aStream nextPutAll: 'X * ', multiplier value, ' + ', offset value, ' ', relation value, ' Y'; cr.
	strength value printOn: aStream! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

CoolDrawConstraintDialog class
	instanceVariableNames: ''!


!CoolDrawConstraintDialog class methodsFor: 'instance creation'!

between: figure1 and: figure2 and: figure3 on: drawing 
	self notYetImplemented!

between: figure1 and: figure2 on: drawing 
	| aModel composite |
	aModel := self new
				initialize: figure1
				and: figure2
				on: drawing.
	composite := CoolDrawDialogView model: aModel.
	composite width: 300.
	self title: 'Add Constraint' on: composite.
	self standardSelections: composite from: figure1 and: figure2.
	self strengthSelectionRow: composite.
	self customCancelButtons: composite.
	composite addVerticalSpace: 10; open.
	^aModel isCustom
		ifTrue: [self customBetween: figure1 and: figure2 on: drawing]
		ifFalse: [aModel]!

customBetween: figure1 and: figure2 on: drawing
	| aModel composite yp zp |
	aModel := self new initialize: figure1 and: figure2 on: drawing.
	composite := CoolDrawDialogView model: aModel.
	composite width: 300.
	self title: 'Add Custom Constraint' on: composite.
	yp := composite yPosition.
	self variableSelectionColumn: composite atX: 0.1 atY: yp figure: figure1.
	zp := composite yPosition.
	self variableSelectionColumn: composite atX: 0.6 atY: yp figure: figure2.
	zp > composite yPosition ifTrue: [composite yPosition: zp].
	self twoRelationSelectionRow: composite.
	self multiplierOffsetRow: composite.
	self strengthSelectionRow: composite.
	self okCancelButtons: composite.
	composite addVerticalSpace: 10; open.
	^aModel!

customOn: figure1 on: drawing
	| aModel composite yp  |
	aModel := self new initialize: figure1 on: drawing.
	composite := CoolDrawDialogView model: aModel.
	composite width: 300.
	self title: 'Add Custom Constraint' on: composite.
	yp := composite yPosition.
	self variableSelectionColumn: composite atX: 0.3 atY: yp figure: figure1.
	self oneRelationSelectionRow: composite.
	self strengthSelectionRow: composite.
	self okCancelButtons: composite.
	composite addVerticalSpace: 10; open.
	^aModel!

on: figure1 on: drawing 
	| aModel composite |
	aModel := self new
				initialize: figure1
				on: drawing.
	composite := CoolDrawDialogView model: aModel.
	composite width: 300.
	self title: 'Add Constraint' on: composite.
	self standardSelections: composite from: figure1.
	self strengthSelectionRow: composite.
	self customCancelButtons: composite.
	composite addVerticalSpace: 10; open.
	^aModel isCustom
		ifTrue: [self customOn: figure1 on: drawing]
		ifFalse: [aModel]! !

!CoolDrawConstraintDialog class methodsFor: 'view creation'!

customCancelButtons: composite 
	| yp button  wrapper |
	composite addVerticalSpace: 5.
	yp := composite yPosition.
	button := (Button trigger) label: 'Custom'; model: ((PluggableAdaptor on: composite model)
					performAction: #custom).
	wrapper := BoundedWrapper on: button.
	composite yPosition: yp; addWrapper: wrapper atX: 0.3.
	button := (Button trigger) label: 'Cancel'; model: ((PluggableAdaptor on: composite model)
					performAction: #cancel).
	wrapper := BoundedWrapper on: button.
	composite yPosition: yp; addWrapper: wrapper atX: 0.6!

multiplierOffsetRow: composite
	|  holder  yp |
	yp := composite yPosition.
	composite yPosition: yp.
	composite addTextLabel: 'multipler' atX: 0.1.
	composite yPosition: yp.
	holder := composite model valueHolderForMultipler.
	composite addTextFieldOn: holder initially: '' fromX: 0.24 toX: 0.50.
	composite yPosition: yp.
	composite addTextLabel: 'offset' atX: 0.6.
	composite yPosition: yp.
	holder := composite model valueHolderForOffset.
	composite addTextFieldOn: holder initially: '' fromX: 0.63 toX: 0.90!

okCancelButtons: composite 
	| yp button  wrapper |
	composite addVerticalSpace: 5.
	yp := composite yPosition.
	button := (Button trigger) label: 'Ok'; model: ((PluggableAdaptor on: composite model)
					performAction: #ok).
	wrapper := BoundedWrapper on: button.
	composite yPosition: yp; addWrapper: wrapper atX: 0.3.
	button := (Button trigger) label: 'Cancel'; model: ((PluggableAdaptor on: composite model)
					performAction: #cancel).
	wrapper := BoundedWrapper on: button.
	composite yPosition: yp; addWrapper: wrapper atX: 0.6!

oneRelationSelectionRow: composite
	| collec holder button  |
	composite addVerticalSpace: 5.
	collec := #(('fixed' #stay)).
	holder := composite model valueHolderForRelation.
	composite
		addRow: collec
		fromX: 0.4
		toX: 0.7
		collect: 
			[:config | 
			button := (Button toggle) label: (config at: 1); model: ((PluggableAdaptor on: holder)
							selectValue: (config at: 2)).
			BoundedWrapper on: button].!

standardSelections: composite from: figure1 
	| collec param1 relation |
	collec := OrderedCollection new.
	self standardSelections: figure1 into: collec.
	composite addVerticalSpace: 5.
	param1 := composite model valueHolderForParamOf: figure1.
	relation := composite model valueHolderForRelation.
	composite
		addAll: collec
		inRows: collec size + 1 // 2
		fromX: 0.1
		toX: 0.9
		collect: 
			[:config | 
			"( name p1 relation )"
			| button |
			button := Button trigger label: (config at: 1).
			button model: ((PluggableAdaptor on: param1)
					getBlock: [:m | false]
					putBlock: 
						[:m :v | 
						composite model name: (config at: 1).
						param1 value: (config at: 2).
						relation value: (config at: 3).
						composite model ok]
					updateBlock: [:m :a :p | false]).
			BoundedWrapper on: button]!

standardSelections: composite from: figure1 and: figure2 
	| collec param1 param2 relation offset |
	collec := OrderedCollection new.
	self
		standardSelections: figure1
		and: figure2
		into: collec.
	composite addVerticalSpace: 5.
	param1 := composite model valueHolderForParamOf: figure1.
	param2 := composite model valueHolderForParamOf: figure2.
	relation := composite model valueHolderForRelation.
	offset := composite model valueHolderForOffset.
	composite model valueHolderForMultipler value: '1'.
	composite
		addAll: collec
		inRows: collec size + 1 // 2
		fromX: 0.1
		toX: 0.9
		collect: 
			[:config | 
			"( name p1 relation p2 offset? )"
			| button |
			button := Button trigger label: (config at: 1).
			button model: ((PluggableAdaptor on: param1)
					getBlock: [:m | false]
					putBlock: 
						[:m :v | 
						composite model name: (config at: 1).
						param1 value: (config at: 2).
						relation value: (config at: 3).
						param2 value: (config at: 4).
						(config at: 5)
							ifTrue: 
								[| v1 v2 |
								v1 := figure1 perform: param1 value.
								v2 := figure2 perform: param2 value.
								offset value: (v2 - v1) printString]
							ifFalse: [offset value: '0'].
						composite model ok]
					updateBlock: [:m :a :p | false]).
			BoundedWrapper on: button]!

strengthSelectionRow: composite
	| collec holder button |
	composite addVerticalSpace: 5.
	collec := #(('required' #required) ('strong' #strong ) ('medium' #medium) ('weak' #weak)).
	holder := composite model valueHolderForStrength.
	composite
		addRow: collec
		fromX: 0.05
		toX: 0.95
		collect: 
			[:config | 
			button := (Button toggle) label: (config at: 1); model: ((PluggableAdaptor on: holder)
							selectValue: (config at: 2)).
			BoundedWrapper on: button]!

title: aString on: composite
	composite addVerticalSpace: 5.
	composite addTextLabel: aString!

twoRelationSelectionRow: composite
	| collec holder button  |
	composite addVerticalSpace: 5.
	collec := #(('<=' #<=) ('==' #= ) ('>=' #>=)).
	holder := composite model valueHolderForRelation.
	composite
		addRow: collec
		fromX: 0.3
		toX: 0.7
		collect: 
			[:config | 
			button := (Button toggle) label: (config at: 1); model: ((PluggableAdaptor on: holder)
							selectValue: (config at: 2)).
			BoundedWrapper on: button].!

variableSelectionColumn: composite atX: xp atY: yp figure: figure 
	| button  holder collec |
	composite yPosition: yp.
	composite addVerticalSpace: 5.
	composite addVisual: figure name asComposedText fromX: xp-0.05 toX: xp+0.4.
	holder := composite model valueHolderForParamOf: figure.
	collec := figure constrainableFeatures.
	composite
		addColumn: collec
		fromX: xp
		toX: xp + 0.4
		collect: 
			[:config | 
			"( name selector )"
			button := (Button toggle) label: (config at: 1); model: ((PluggableAdaptor on: holder)
							selectValue: (config at: 2)).
			BoundedWrapper on: button].! !

!CoolDrawConstraintDialog class methodsFor: 'standard constraints'!

standardSelectionAbove: figure1 and: figure2 into: collec 
	(figure1 isConstrainable: #bottom) & (figure2 isConstrainable: #top)
		ifTrue:	[collec add: #('offset above' #bottom #= #top true ).
				collec add: #('above' #bottom #>= #top false )]!

standardSelectionAlign: figure1 and: figure2 into: collec 
	(figure1 isConstrainable: #left) & (figure2 isConstrainable: #left)
		ifTrue: [collec add: #('align on left' #left #= #left false )].
	(figure1 isConstrainable: #right) & (figure2 isConstrainable: #right)
		ifTrue: [collec add: #('align on right' #right #= #right false )].
	(figure1 isConstrainable: #top) & (figure2 isConstrainable: #top)
		ifTrue: [collec add: #('align on top' #top #= #top false )].
	(figure1 isConstrainable: #bottom) & (figure2 isConstrainable: #bottom)
		ifTrue: [collec add: #('align on bottom' #bottom #= #bottom false)].
	(figure1 isConstrainable: #centerx) & (figure2 isConstrainable: #centerx)
		ifTrue: [collec add: #('align center vert.' #centerx #= #centerx false )].
	(figure1 isConstrainable: #centery) & (figure2 isConstrainable: #centery)
		ifTrue: [collec add:#('align center horiz.' #centery #= #centery false )].!

standardSelectionExtent: figure1 and: figure2 into: collec 
	(figure1 isConstrainable: #width) & (figure2 isConstrainable: #width)
		ifTrue: [collec add: #('same width' #width #= #width false )].
	(figure1 isConstrainable: #height) & (figure2 isConstrainable: #height)
		ifTrue: [collec add: #('same height' #height #= #height false)]!

standardSelectionFixed: figure1 into: collec 
	(figure1 isConstrainable: #top)
		ifTrue: [collec add: #('anchor top' #top #stay )].
	(figure1 isConstrainable: #bottom)
		ifTrue: [collec add: #('anchor bottom' #bottom #stay )].
	(figure1 isConstrainable: #left)
		ifTrue: [collec add: #('anchor left' #left #stay )].
	(figure1 isConstrainable: #right)
		ifTrue: [collec add: #('anchor right' #right #stay )].
	(figure1 isConstrainable: #width)
		ifTrue: [collec add: #('fixed width' #width #stay )].
	(figure1 isConstrainable: #height)
		ifTrue: [collec add: #('fixed height' #height #stay )]!

standardSelectionLeftOf: figure1 and: figure2 into: collec 
	(figure1 isConstrainable: #right) & (figure2 isConstrainable: #left)
		ifTrue:	[collec add: #('offset left of' #right #= #left true ).
				collec add: #('left of' #right #<= #left false )]!

standardSelectionLinePlus: figure1 and: figure2 into: collec 
	(figure1 isConstrainable: #center) & (figure2 isConstrainable: #startPoint)
		ifTrue: [collec add: #('start from center' #center #= #startPoint false )].
	(figure1 isConstrainable: #startPoint) & (figure2 isConstrainable: #center)
		ifTrue: [collec add: #('start from center' #startPoint #= #center false )].
	(figure1 isConstrainable: #center) & (figure2 isConstrainable: #stopPoint)
		ifTrue: [collec add: #('stop at center' #center #= #stopPoint false )].
	(figure1 isConstrainable: #stopPoint) & (figure2 isConstrainable: #center)
		ifTrue: [collec add: #('stop at center' #stopPoint #= #center false )].

	(figure1 isConstrainable: #topLeft) & (figure2 isConstrainable: #startPoint)
		ifTrue: [collec add: #('start from top left' #topLeft #= #startPoint false )].
	(figure1 isConstrainable: #startPoint) & (figure2 isConstrainable: #topLeft)
		ifTrue: [collec add: #('start from top left' #startPoint #= #topLeft false )].
	(figure1 isConstrainable: #bottomRight) & (figure2 isConstrainable: #startPoint)
		ifTrue: [collec add: #('start bottom right' #bottomRight #= #startPoint false )].
	(figure1 isConstrainable: #startPoint) & (figure2 isConstrainable: #bottomRight)
		ifTrue: [collec add: #('start bottom right' #startPoint #= #bottomRight false )].

	(figure1 isConstrainable: #topLeft) & (figure2 isConstrainable: #stopPoint)
		ifTrue: [collec add: #('stop at top left' #topLeft #= #stopPoint false )].
	(figure1 isConstrainable: #stopPoint) & (figure2 isConstrainable: #topLeft)
		ifTrue: [collec add: #('stop at top left' #stopPoint #= #topLeft false )].
	(figure1 isConstrainable: #bottomRight) & (figure2 isConstrainable: #stopPoint)
		ifTrue: [collec add: #('stop bottom right' #bottomRight #= #stopPoint false )].
	(figure1 isConstrainable: #stopPoint) & (figure2 isConstrainable: #bottomRight)
		ifTrue: [collec add: #('stop bottom right' #stopPoint #= #bottomRight false )].!

standardSelectionLines: figure1 into: collec 
	(figure1 isConstrainable: #startPoint)
		ifTrue: [collec add: #('anchor start' #startPoint #stay )].
	(figure1 isConstrainable: #stopPoint)
		ifTrue: [collec add: #('anchor stop' #stopPoint #stay )].
	(figure1 isConstrainable: #length)
		ifTrue: [collec add: #('fixed length' #length #stay )]!

standardSelections: figure1 and: figure2 into: collec 
	self
		standardSelectionAbove: figure1
		and: figure2
		into: collec.
	self
		standardSelectionLeftOf: figure1
		and: figure2
		into: collec.
	self
		standardSelectionAlign: figure1
		and: figure2
		into: collec.
	self
		standardSelectionExtent: figure1
		and: figure2
		into: collec.
	self
		standardSelectionLinePlus: figure1
		and: figure2
		into: collec.!

standardSelections: figure1 into: collec 
	self
		standardSelectionFixed: figure1
		into: collec.
	self
		standardSelectionLines: figure1
		into: collec! !


Object subclass: #SkyBlueVariableState
	instanceVariableNames: 'determinedby walkstrength mark valid '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SkyBlue-Basics'!


!SkyBlueVariableState methodsFor: 'accessing'!

determinedBy
	^determinedby!

determinedBy: b
	determinedby := b!

mark
	^mark!

mark: b
	mark := b!

valid
	^valid!

valid: b
	valid := b!

walkStrength
	^walkstrength!

walkStrength: s
	walkstrength := s! !

!SkyBlueVariableState methodsFor: 'printing'!

printOn: aStream 
	aStream nextPutAll: '[: ' , determinedby printString , ' ' , walkstrength printString , ' ' , mark printString , ' ' , valid printString , ':]'! !


Object subclass: #SkyBluePlan
	instanceVariableNames: 'rootconstraints goodconstraints goodmethods badconstraints valid '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SkyBlue-Basics'!


!SkyBluePlan methodsFor: 'invoking'!

execute
	valid ifFalse: [^self error: 'Trying to execute an invalid plan'].
	goodmethods do: [:mt | mt executePropagateValid]! !

!SkyBluePlan methodsFor: 'private'!

from: roots good: goodcns bad: badcns
	valid := true. 
	rootconstraints := roots.
	goodconstraints := goodcns.
	goodmethods := goodcns collect: [:cn | cn selectedMethod].
	badconstraints := badcns.
	^self! !

!SkyBluePlan methodsFor: 'printing'!

printOn: aStream
	self printOn: aStream indent: 0!

printOn: aStream indent: sp 
	1 to: sp do: [:i | aStream space; space; space].
	aStream nextPutAll: '---- ' , self asOop printString , '-Methods ----'; cr.
	goodmethods
		do: 
			[:each | 
			1 to: sp do: [:i | aStream space; space; space].
			each printOn: aStream.
			aStream cr]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SkyBluePlan class
	instanceVariableNames: ''!


!SkyBluePlan class methodsFor: 'instance creation'!

newValidFrom: roots good: goodcns bad: badcns
	^self new from: roots good: goodcns bad: badcns! !


SkyBluePlan subclass: #ColbaltBluePlan
	instanceVariableNames: 'checkers graph state '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'ColbaltBlue-Basics'!


!ColbaltBluePlan methodsFor: 'initialize-release'!

check: checkcns graph: g
	graph := g. 
	state := graph getTheStateOfEverything.
	checkers := OrderedCollection new.
	checkcns reverseDo: [:cn | checkers add: (ColbaltBlueChecker new
				initializeWith: cn
				state: state
				graph: graph
				planclass: self class)].
	^self!

release
	super release.
	graph := nil.
	state := nil.! !

!ColbaltBluePlan methodsFor: 'invoking'!

execute
	self executeLowlevel ifTrue: [graph setTheStateOfEverythingCopy: state]!

executeLowlevel
	super execute.
	checkers do: [:each | each execute ifTrue: [^true]].
	^false! !

!ColbaltBluePlan methodsFor: 'testing'!

isThunk
	^false! !

!ColbaltBluePlan methodsFor: 'printing'!

printOn: aStream indent: sp 
	super printOn: aStream indent: sp.
	1 to: sp do: [:i | aStream space; space; space].
	aStream nextPutAll: '---- ' , self asOop printString , '-Checkers ----'; cr.
	checkers
		do: 
			[:each | 
			each printOn: aStream indent: sp.
			aStream cr]! !


ColbaltBluePlan subclass: #CoolDrawPlan
	instanceVariableNames: 'changedObjects '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Framework'!
CoolDrawPlan comment:
'CoolDrawPlan is a specialization of a standard constraint plan that deals with the need to damage regions in the drawing for each figure that is changed by the constraints.
'!


!CoolDrawPlan methodsFor: 'initialization'!

collectChangedFigures
	changedObjects := IdentitySet new.
	goodmethods do: [:m | m outputs do: [:v | v owner notNil ifTrue: [changedObjects add: v owner]]].
	changedObjects := changedObjects asArray! !

!CoolDrawPlan methodsFor: 'invoking'!

executeLowlevel
	| t |
	changedObjects do: [:each | each willChange].
	t := super executeLowlevel.
	changedObjects do: [:each | each changed].
	^t! !

!CoolDrawPlan methodsFor: 'printing'!

printOn: aStream indent: sp 
	super printOn: aStream indent: sp.
	1 to: sp do: [:i | aStream space; space; space].
	aStream nextPutAll: '---- ' , self asOop printString , '-Figures ----'; cr.
	changedObjects
		do: 
			[:each | 
			1 to: sp do: [:i | aStream space; space; space].
			each printOn: aStream.
			aStream cr]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

CoolDrawPlan class
	instanceVariableNames: ''!


!CoolDrawPlan class methodsFor: 'instance creation'!

newValidFrom: roots good: goodcns bad: badcns
	| me |
	me := super new from: roots good: goodcns bad: badcns.
	me collectChangedFigures.
	^me! !


Object subclass: #CoolDrawHandleDatum
	instanceVariableNames: 'drawing plan mousePoint constraints objects '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Support'!
CoolDrawHandleDatum comment:
'CoolDrawHandleDatum is used to pass information around within the invocation methods of a Tool.  The initializeInvoke: creates a datum which is then passed to the invoke: and then the invokeStep: methods and finally to the terminateInvoke method.  This provides the equivalent of static scoping with the initializeInvoke method enclosing the remaining ones, while still allowing refinement of all methods in subclasses.  This object is merely a data storage entity and thus has little relevant behavior.

The only special thing it does is to create a "mouse" variable, i.e., a CoolDraw variable (a constrainable variable) that represents the current mouse/cursor position.
'!


!CoolDrawHandleDatum methodsFor: 'initialize-release'!

initializeFor: aDrawing 
	constraints := OrderedCollection new.
	drawing := aDrawing.
	mousePoint := CoolDrawPoint
				newIn: self constraintGraph
				with: 0 @ 0
				owner: nil.
	^self!

release
	| v |
	constraints isNil ifFalse: [self halt: 'constraints should be empty'].
	self constraintGraph removeVariable: (v := mousePoint constrainableX).
	v release.
	self constraintGraph removeVariable: (v := mousePoint constrainableY).
	v release.
	drawing := nil.
	^self! !

!CoolDrawHandleDatum methodsFor: 'accessing'!

addConstraint: c
	constraints add: c.
	self constraintGraph addConstraint: c!

constraintGraph
	^drawing constraintGraph!

constraints
	^constraints!

drawing
	^drawing!

mousePoint
	^mousePoint!

objects
	^objects!

objects: o
	objects := o!

plan
	^plan!

plan: p
	plan := p!

removeConstraints
	constraints do: [:each | self constraintGraph removeConstraint: each].
	constraints := nil!

variable
	^variable! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

CoolDrawHandleDatum class
	instanceVariableNames: ''!


!CoolDrawHandleDatum class methodsFor: 'instance creation'!

newIn: aDrawing
	^super new initializeFor: aDrawing! !


Object subclass: #CoolDrawPoint
	instanceVariableNames: 'x y '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Framework'!
CoolDrawPoint comment:
'Allows constraint of both the x & y coordinates of a point. Not that the interface is kept
similar to that of a SkyBlueVariable to allow easy integration into the existing framework.

Instance variables:
	x <SkyBlueVariable>	The constrainable x coord.
	y <SkyBlueVariable>	The constrainable y coord.
	'!


!CoolDrawPoint methodsFor: 'initialize-release'!

fromExisting: constrainableX and: constrainableY 
	x := constrainableX.
	y := constrainableY.
	^self!

initializeIn: aGraph with: aPoint owner: own 
	x := CoolDrawVariable
				newNamed: own asOop printString, 'x.', self asOop printString
				in: aGraph
				with: aPoint x.
	y := CoolDrawVariable
				newNamed: own asOop printString, 'y.', self asOop printString
				in: aGraph
				with: aPoint y.
	x owner: own.
	y owner: own.
	^self! !

!CoolDrawPoint methodsFor: 'accessing'!

constrainableX
	"Answer the receiver's x coordinate as a SkyBlueVariable"

	^x!

constrainableY
	"Answer the receiver's y coordinate as a SkyBlueVariable"

	^y!

constraints
	"Answer an ordered collection of the receiver's SkyBlueVariable's constraints"
	
	| myConstraints |
	myConstraints := x constraints.
	myConstraints addAll: y constraints.
	^myConstraints.!

contents
	"Answer the receiver as a point"
	^self point.!

contents: aPoint
	"Set the contents to of the receiver's x & y SkyBlueVariables to be that of 
	aPoint's x & y."
	^self point: aPoint.! !

!CoolDrawPoint methodsFor: 'enumerating'!

do: aBlock
	aBlock value: self! !

!CoolDrawPoint methodsFor: 'printing'!

printOn: aStream
	aStream nextPutAll: 'a ', self class name, ' = ', self contents printString! !

!CoolDrawPoint methodsFor: 'private'!

point
	"Answer the receiver's SkyBlueVariables x&y as a point."

	| xVal yVal  |
	xVal := self x.
	yVal := self y.
	^(xVal @ yVal)!

point: aPoint
	"Set the contents to of the receiver's x & y SkyBlueVariables to be that of 
	aPoint's x & y."
	self x: aPoint x.
	self y: aPoint y.
	^self!

x
	"Answers the receivers SkyBlueVariable x's value"
	^x contents.!

x: val
	"Set the receivers SkyBlueVariable x's value"
	x contents: val.!

y
	"Answers the receivers SkyBlueVariable y's value"
	^y contents.!

y: val
	"Set the receivers SkyBlueVariable y's value"
	y contents: val.! !

!CoolDrawPoint methodsFor: 'copying'!

copyUsing: aCopyState
	x := aCopyState copy: x.
	y := aCopyState copy: y! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

CoolDrawPoint class
	instanceVariableNames: ''!


!CoolDrawPoint class methodsFor: 'instance creation'!

newFromExisting: constrainableX and: constrainableY
	^self new fromExisting: constrainableX and: constrainableY!

newIn: aGraph with: aPoint owner: own
	^self new initializeIn: aGraph with: aPoint owner: own! !


Object subclass: #CoolDrawConstraintCluster
	instanceVariableNames: 'constraints name '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Support'!


!CoolDrawConstraintCluster methodsFor: 'initialize-release'!

initialize
	constraints := Set new.
	^self! !

!CoolDrawConstraintCluster methodsFor: 'accessing'!

addConstraint: c
	constraints add: c!

constraints
	^constraints!

name
	^name!

name: n
	name := n! !

!CoolDrawConstraintCluster methodsFor: 'testing'!

constrainsAnyOf: variables 
	constraints do: [:cn | cn variables do: [:v | (variables includes: v)
				ifTrue: [^true]]].
	^false!

onlyConstrainsAnyOf: variables 
	constraints do: [:cn | cn variables do: [:v | (variables includes: v)
				ifFalse: [^false]]].
	^true! !

!CoolDrawConstraintCluster methodsFor: 'copying'!

copyUsing: aCopyState
	constraints := constraints collect: [:cn | aCopyState copy: cn].
	"name"! !


Object subclass: #SkyBlueMark
	instanceVariableNames: 'value '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SkyBlue-Basics'!


!SkyBlueMark methodsFor: 'printing'!

printOn: aStream
	aStream nextPutAll: 'mark!!', value printString! !

!SkyBlueMark methodsFor: 'private'!

value: a
	value := a! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SkyBlueMark class
	instanceVariableNames: 'lastmark '!


!SkyBlueMark class methodsFor: 'class initialization'!

initialize
	"SkyBlueMark initialize"
	lastmark := 1! !

!SkyBlueMark class methodsFor: 'instance creation'!

new
	| me |
	me := super new.
	lastmark := lastmark + 1.
	me value: lastmark.
	^me! !


Object subclass: #ColbaltBlueConstraintGraphState
	instanceVariableNames: 'constraintstates varstates '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'ColbaltBlue-Basics'!


!ColbaltBlueConstraintGraphState methodsFor: 'accessing'!

constraints
	^constraintstates!

constraints: c variables: v
	constraintstates := c.
	varstates := v.
	^self!

variables
	^varstates! !

!ColbaltBlueConstraintGraphState methodsFor: 'printing'!

dump
	constraintstates do: [:each | Transcript show: each isActive printString; space].
	Transcript cr.!

printOn: aStream 
	aStream nextPutAll: 'Constraints:'; cr.
	aStream tab; nextPutAll: '[* selected_method   mark   active *]'; cr.
	constraintstates
		do: 
			[:each | 
			aStream tab.
			each printOn: aStream.
			aStream cr].
	aStream cr; nextPutAll: 'Variables:'; cr.
	varstates
		do: 
			[:each | 
			aStream tab.
			each printOn: aStream.
			aStream cr]! !


Object subclass: #CoolDrawPasteState
	instanceVariableNames: 'toDrawing '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Support'!


!CoolDrawPasteState methodsFor: 'initialize-release'!

initialize: d
	toDrawing := d.
	^self!

release
	toDrawing := nil! !

!CoolDrawPasteState methodsFor: 'accessing'!

constraintGraph
	^toDrawing constraintGraph! !


Object subclass: #SkyBlueConstraintGraph
	instanceVariableNames: 'variables constraints '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SkyBlue-Basics'!
SkyBlueConstraintGraph comment:
'This is a fairly direct implementation of Michael Sannella''s SkyBlue algorithm as documented by his draft technical report dated November 10, 1992.  This implementation was coded by me, Bjorn N. Freeman-Benson on February 1, 1993.  Michael''s procedure names were converted to method names by translating "-" to capitalization (e.g., exec-from-roots to execFromRoots).  His variable names were converted by removing the "-" without capitalization.  I have attempted to introduce some object-orientedness by distributing the responsibility for various task throughout the classes, but I have not attempted to redesign the algorithm in any way (yet).  I did change the way Plans work, slightly.
'!


!SkyBlueConstraintGraph methodsFor: 'public accessing'!

addConstraint: aConstraint 
	self primAddConstraint: aConstraint.
	self doAddConstraint: aConstraint!

addVariable: v
	variables add: v!

extractEditPlanUsingClass: planClass
	| sources |
	sources := self editSourceConstraints.
	^self extractPlan: sources usingClass: planClass!

extractStayPlanUsingClass: planClass
	| sources |
	sources := constraints select: [:cn | cn isSource].
	^self extractPlan: sources usingClass: planClass!

release
	variables do: [:each | each release].
	constraints do: [:each | each release].
	super release!

removeConstraint: aConstraint 
	self primRemoveConstraint: aConstraint.
	aConstraint isEnforced ifTrue: [self doRemoveConstraint: aConstraint]!

removeVariable: aVar
        aVar constraints isEmpty
                ifFalse: [aVar constraints copy do: [:each | self removeConstraint: each]].
        variables remove: aVar! !

!SkyBlueConstraintGraph methodsFor: 'accessing'!

constraints
	^constraints!

editSourceConstraints
	^constraints select: [:cn | cn isExternal]!

variables
	^variables! !

!SkyBlueConstraintGraph methodsFor: 'sky blue'!

collectUnenforced: unenforced and: vars and: collectionstrength and: collectequal 
	| donemark |
	donemark := SkyBlueMark new.
	vars do: [:var | var
			collectUnenforcedMark: unenforced
			and: collectionstrength
			and: collectequal
			and: donemark]!

doAddConstraint: aConstraint 
	| unenforced execroots |
	aConstraint resetSkyBlueFields: self.
	unenforced := SortedCollection sortBlock: [:a :b | a strength > b strength].
	execroots := OrderedCollection new.
	unenforced add: aConstraint.
	self updateSolutionGraphFrom: unenforced and: execroots.
	self execFromRoots: execroots!

doRemoveConstraint: aConstraint
	^self doRemoveConstraint: aConstraint propagate: true!

doRemoveConstraint: aConstraint propagate: propFlag 
	| unenforced execroots oldoutputs |
	unenforced := SortedCollection sortBlock: [:a :b | a strength > b strength].
	execroots := OrderedCollection new.
	propFlag
		ifTrue: 
			[oldoutputs := aConstraint selectedMethod outputs.
			aConstraint selectedMethod: nil.
			oldoutputs
				do: 
					[:var | 
					var determinedBy: nil.
					var walkStrength: SkyBlueStrength absoluteWeakest].
			self propagateWalkStrength: oldoutputs.
			oldoutputs do: [:each | execroots addFirst: each]]
		ifFalse: [oldoutputs := #()].
	self
		collectUnenforced: unenforced
		and: oldoutputs
		and: aConstraint strength
		and: true.
	self updateSolutionGraphFrom: unenforced and: execroots.
	self execFromRoots: execroots!

execFromCycle: cn and: propmark 
	cn mark = propmark
		ifTrue: 
			[cn mark: nil.
			cn selectedMethod outputs
				do: 
					[:var | 
					var valid: false.
					var activeConstraintsDo: [:consumingcn | consumingcn ~~ cn & consumingcn isEnforced ifTrue: [self execFromCycle: consumingcn and: propmark]]]]!

execFromRoots: execroots 
	| propmark execpplan cn |
	propmark := SkyBlueMark new.
	execpplan := OrderedCollection new.
	execroots do: [:each | each isConstraint ifTrue: [each pplanAdd: execpplan and: propmark]].
	execroots do: [:each | each isVariable ifTrue: [each determinedBy isNil & each isValid not ifTrue: [each pplanAdd: execpplan and: propmark]]].
	[execpplan isEmpty]
		whileFalse: 
			[cn := execpplan removeFirst.
			cn mark ~= propmark ifFalse: [(cn areAnyImmediateUpstreamCnsMarked: propmark)
					ifTrue: [self execFromCycle: cn and: propmark]
					ifFalse: 
						[cn mark: nil.
						cn selectedMethod executePropagateValid]]]!

extractPlan: planroots usingClass: planClass
	| goodcns badcns propmark pplan cn |
	goodcns := OrderedCollection new.
	badcns := OrderedCollection new.
	propmark := SkyBlueMark new.
	pplan := OrderedCollection new.
	planroots do: [:each | each isEnforced ifTrue: [each pplanAdd: pplan and: propmark]].
	[pplan isEmpty]
		whileFalse: 
			[cn := pplan removeFirst.
			cn mark = propmark ifTrue: [(cn areAnyImmediateUpstreamCnsMarked: propmark)
					ifTrue: [badcns addFirst: cn]
					ifFalse: 
						[cn mark: nil.
						goodcns addFirst: cn]]].
	^planClass
		newValidFrom: planroots
		good: goodcns reverse
		bad: badcns!

propagateWalkStrength: roots 
	| propmark walkpplan cn |
	propmark := SkyBlueMark new.
	walkpplan := OrderedCollection new.
	roots 
		pplanAdd: walkpplan
		and: propmark.
	[walkpplan isEmpty]
		whileFalse: 
			[cn := walkpplan removeFirst.
			(cn areAnyImmediateUpstreamCnsMarked: propmark)
				ifTrue: [cn selectedMethod inputs do: [:var | var determinedBy notNil ifTrue: [var determinedBy mark = propmark ifTrue: [var walkStrength: SkyBlueStrength absoluteWeakest]]]].
			cn selectedMethod outputs do: [:var | var walkStrength: (cn computeWalkabout: var)].
			cn mark: nil]!

pvineGrow: rootstrength and: donemark and: pvinestack and: redetermined 
	| cn ok |
	pvinestack isEmpty
		ifTrue: [^true]
		ifFalse: 
			[cn := pvinestack removeFirst.
			cn mark = donemark
				ifTrue: [ok := self
								pvineGrow: rootstrength
								and: donemark
								and: pvinestack
								and: redetermined]
				ifFalse: [cn strength < rootstrength
						ifTrue: [ok := cn
										pvineLeaf: rootstrength
										and: donemark
										and: pvinestack
										and: redetermined]
						ifFalse: [ok := cn
										pvineBranch: rootstrength
										and: donemark
										and: pvinestack
										and: redetermined]].
			ok ifFalse: [pvinestack addFirst: cn].
			^ok]!

updateSolutionGraphFrom: unenforced and: execroots 
	| constraint redetermined ok t |
	[unenforced isEmpty]
		whileFalse: 
			[constraint := unenforced removeFirst.
			redetermined := OrderedCollection new.
			ok := constraint buildPvine: redetermined.
			ok
				ifTrue: 
					[t := OrderedCollection new.
					t addAll: redetermined.
					t add: constraint.
					self propagateWalkStrength: t.
					self
						collectUnenforced: unenforced
						and: redetermined
						and: constraint strength
						and: false.
					execroots addFirst: constraint.
					redetermined do: [:each | execroots addFirst: each]]]! !

!SkyBlueConstraintGraph methodsFor: 'printing'!

printOn: aStream 
	aStream nextPutAll: 'a ' , self class name; cr.
	aStream nextPutAll: 'Constraints'; cr.
	constraints
		do: 
			[:each | 
			each printOn: aStream.
			aStream cr; tab.
			each copyOfState printOn: aStream.
			aStream cr].
	aStream nextPutAll: 'Variables:'; cr.
	variables
		do: 
			[:each | 
			each printOn: aStream.
			aStream cr; tab.
			each copyOfState printOn: aStream.
			aStream cr].! !

!SkyBlueConstraintGraph methodsFor: 'private'!

constraints: c
	constraints := c!

primAddConstraint: aConstraint 
	constraints add: aConstraint.
	aConstraint variables do: [:each | each attachConstraint: aConstraint].!

primRemoveConstraint: aConstraint 
	aConstraint graph: nil.
	constraints remove: aConstraint.
	aConstraint variables do: [:each | each deattachConstraint: aConstraint]!

variables: v
	variables := v! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SkyBlueConstraintGraph class
	instanceVariableNames: ''!


!SkyBlueConstraintGraph class methodsFor: 'instance creation'!

new
	| me |
	me := super new.
	me constraints: (SortedCollection sortBlock: [:a :b | a strength >= b strength]).
	me variables: OrderedCollection new.
	^me! !


SkyBlueConstraintGraph subclass: #SkyBlueMVCConstraintGraph
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SkyBlue-Examples'!


!SkyBlueMVCConstraintGraph methodsFor: 'adaptor'!

mvcChange: aVariable to: aText from: aController 
	| value cn |
	value := nil class evaluatorClass new
				evaluate: aText readStream
				in: nil
				to: nil
				notifying: self
				ifFail: [^self error: 'evaluation failed'].
	cn := SkyBlueConstraint mediumAssign: value to: aVariable.
	self addConstraint: cn.
	self removeConstraint: cn.
	^true! !


SkyBlueConstraintGraph subclass: #ColbaltBlueConstraintGraph
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'ColbaltBlue-Basics'!


!ColbaltBlueConstraintGraph methodsFor: 'public accessing'!

addConstraint: aConstraint 
	| state |
	aConstraint isUnique
		ifTrue: 
			[super addConstraint: aConstraint]
		ifFalse: 
			[aConstraint setState: ColbaltBlueConstraintState new.
			aConstraint active: false.
			self primAddConstraint: aConstraint].
	state := self getTheStateOfEverything.
	self checkAndExecuteInactiveConstraints.
	self setTheStateOfEverything: state!

removeConstraint: aConstraint 
	| state |
	aConstraint isUnique
		ifTrue: [super removeConstraint: aConstraint]
		ifFalse: 
			[self primRemoveConstraint: aConstraint.
			self doRemoveConstraint: aConstraint propagate: false].
	state := self getTheStateOfEverything.
	self checkAndExecuteInactiveConstraints.
	self setTheStateOfEverything: state! !

!ColbaltBlueConstraintGraph methodsFor: 'colbalt blue'!

checkAndExecuteInactiveConstraints
	constraints reverseDo: [:cn | (cn isActive not and: [cn isSatisfied not])
			ifTrue: 
				[cn active: true.
				self doAddConstraint: cn.
				^self checkAndExecuteInactiveConstraints]]!

extractPlan: planroots usingClass: planClass
	| plan cns |
	plan := super extractPlan: planroots usingClass: planClass.
	cns := constraints select: [:cn | cn isActive not].
	plan check: cns graph: self.
	^plan!

getTheStateOfEverything
	^ColbaltBlueConstraintGraphState new constraints: (constraints collect: [:cn | cn copyOfState])
				variables: (variables collect: [:var | var copyOfState])!

setTheStateOfEverything: descriptor
	constraints with: descriptor constraints do: [:cn :st | cn setState: st].
	variables with: descriptor variables do: [:var :st | var setState: st]!

setTheStateOfEverythingCopy: descriptor
	constraints with: descriptor constraints do: [:cn :st | cn setState: st copy].
	variables with: descriptor variables do: [:var :st | var setState: st copy]! !


ColbaltBlueConstraintGraph subclass: #ColbaltBlueMVCConstraintGraph
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SkyBlue-Examples'!


!ColbaltBlueMVCConstraintGraph methodsFor: 'adaptor'!

mvcChange: aVariable to: aText from: aController 
	| value cn |
	value := nil class evaluatorClass new
				evaluate: aText readStream
				in: nil
				to: nil
				notifying: self
				ifFail: [^self error: 'evaluation failed'].
	cn := SkyBlueConstraint mediumAssign: value to: aVariable.
	self addConstraint: cn.
	self removeConstraint: cn.
	^true! !


!GraphicsContext methodsFor: 'accessing'!

paint: paintValue
	"Set the default paint that I use to render uncolored objects."

	paint == paintValue ifTrue: [^self].
	(medium canBePaintedWith: paintValue)
		ifFalse: [^self error: 'Invalid paint value.'].
	paint := paintValue.
	paintValue installOn: self! !

!GraphicsContext methodsFor: 'displaying'!

clear
	| oldPaint oldDevicePaint background |
	oldPaint := self paint.
	oldDevicePaint := devicePaint.
	background := medium background.
	oldPaint = background ifFalse: [self paint: medium background].
	self displayRectangle: self clippingBounds.
	paint := oldPaint.
	devicePaint := oldDevicePaint! !


!Collection methodsFor: 'sky blue'!

pplanAdd: pplan and: donemark
	self do: [:each | each pplanAdd: pplan and: donemark]! !


Object subclass: #Constraint
	instanceVariableNames: 'action observedObjects '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Constraints'!


!Constraint methodsFor: 'initializing'!

for: aBlock on: aCollection
	action := aBlock.
	observedObjects := aCollection.
	observedObjects do: [:each | each addDependent: self]! !

!Constraint methodsFor: 'updating'!

update: anObject
	action value! !


!BlockClosure methodsFor: 'converting'!

onChange: aCollection
	"Evaluate the block whenever any of the objects in aCollection changes.
	In other words, make the block a Constraint, and have it depend on the
	objects.
	Note that constraints automatically set up their dependencies."
	^ Constraint new for: self on: aCollection.! !


Object subclass: #CoolDrawStartupPicturesEditor
	instanceVariableNames: 'forms current '
	classVariableNames: 'TextMenu '
	poolDictionaries: ''
	category: 'CoolDraw-Support'!


!CoolDrawStartupPicturesEditor methodsFor: 'accessing'!

acceptText: txt from: cntrl
	| v |
	v := nil class evaluatorClass new
		evaluate: txt asString
		in: nil
		to: nil
		notifying: self
		ifFail: [0].
	(forms at: current) at: 2 put: v.
	^true!

addForm
	self notYetImplemented!

deleteCurrent
	self notYetImplemented!

image
	^(forms at: current) at: 1!

imageList
	^(1 to: forms size) collect: [:i | i printString]!

imageNumber
	^current printString!

imageNumber: s
	current := nil class evaluatorClass new
		evaluate: s
		in: nil
		to: nil
		notifying: self
		ifFail: [0].
	self changed: #image.
	self changed: #text!

read
	| b |
	b := BinaryObjectStorage onOld: (Filename named: 'cooldraw.startup.picture') readStream.
	forms := b next.
	b close.
	current := 1.!

redefineForm
	(forms at: current) at: 1 put: Image fromUser.
	self changed: #image!

save
	| b |
	b := BinaryObjectStorage onNew: (Filename named: 'cooldraw.startup.picture') writeStream.
	b nextPut: forms.
	b close!

text
	^((forms at: current) at: 2) printString!

textMenu
	"Answer a Menu of operations on the source code that is to 	be displayed when the operate menu button is pressed."
	"Browser flushMenus"

	TextMenu == nil ifTrue: [TextMenu := PopUpMenu
					labels: 'again\undo\copy\cut\paste\do it\print it\inspect\accept\cancel\format\spawn\explain\hardcopy' withCRs
					lines: #(2 5 8 10 13)
					values: #(#again #undo #copySelection #cut #paste #doIt #printIt #inspectIt #accept #cancel #format:from: #spawnEdits:from: #explain:fromController: #hardcopy )].
	^TextMenu! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

CoolDrawStartupPicturesEditor class
	instanceVariableNames: ''!


!CoolDrawStartupPicturesEditor class methodsFor: 'instance creation'!

open
	"CoolDrawStartupPicturesEditor open"

	| mod win comp lst txt t v button |
	mod := self new read.
	win := ScheduledWindow
				model: mod
				label: 'CoolDraw Startup Pictures'
				minimumSize: 450 @ 350.
	comp := DependentComposite new.
	comp add: (ImageView new model: mod)
		borderedIn: (0.2 @ 0 extent: 0.8 @ 0.9).
	lst := LookPreferences edgeDecorator on: (SelectionInListView
					on: mod
					printItems: false
					oneItem: false
					aspect: #imageNumber
					change: #imageNumber:
					list: #imageList
					menu: #imageMenu
					initialSelection: #imageNumber).
	comp add: lst in: (0 @ 0 extent: 0.2 @ 0.7).
	txt := TextView
				on: mod
				aspect: #text
				change: #acceptText:from:
				menu: #textMenu
				initialSelection: ''.
	comp add: (LookPreferences edgeDecorator on: txt)
		in: (0 @ 0.7 extent: 0.2 @ 0.2).
	t := #(#(#save 'Save' ) #(#load 'Load' ) #(#redefineForm 'Redefine' ) #(#addForm 'Add' ) #(#deleteForm 'Delete' ) ).
	1 to: t size
		do: 
			[:i | 
			v := t at: i.
			button := (Button trigger) model: ((PluggableAdaptor on: mod)
							performAction: (v at: 1)); label: (v at: 2).
			comp add: button in: (i - 1 * (1.0 / t size) @ 0.9 extent: 1.0 / t size @ 0.1)].
	win component: comp.
	win open! !


Object subclass: #ColbaltBluePlanThunk
	instanceVariableNames: 'constraint graph state planclass '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'ColbaltBlue-Basics'!


!ColbaltBluePlanThunk methodsFor: 'initialize-release'!

constraint: c state: s graph: g planclass: p
	constraint := c.
	state := s.
	graph := g.
	planclass := p.
	^self!

release
	constraint := nil.
	state := nil.
	graph := nil.
	planclass := nil.
	super release! !

!ColbaltBluePlanThunk methodsFor: 'testing'!

isThunk
	^true! !

!ColbaltBluePlanThunk methodsFor: 'evaluating'!

evaluate
	| plan sources |
	graph setTheStateOfEverythingCopy: state.
	constraint active: true.
	graph doAddConstraint: constraint.
	sources := graph editSourceConstraints.
	sources add: constraint.
	plan := graph extractPlan: sources usingClass: planclass.
	self release.
	^plan! !

!ColbaltBluePlanThunk methodsFor: 'printing'!

printOn: aStream
	aStream nextPutAll: '<<'.
	constraint printOn: aStream.
	aStream nextPutAll: '>>'! !


Object subclass: #ColbaltBlueChecker
	instanceVariableNames: 'constraint thunk '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'ColbaltBlue-Basics'!


!ColbaltBlueChecker methodsFor: 'initialize-release'!

initializeWith: c state: s graph: g planclass: p 
	constraint := c.
	thunk := ColbaltBluePlanThunk new
				constraint: c
				state: s
				graph: g
				planclass: p.
	^self!

release
	constraint := nil.
	thunk release.
	thunk := nil.
	super release! !

!ColbaltBlueChecker methodsFor: 'invoking'!

execute
	constraint isSatisfied ifFalse: [thunk isThunk ifFalse: [^thunk executeLowlevel]
			ifTrue: 
				[thunk := thunk evaluate.
				thunk executeLowlevel.
				^true]].
	^false! !

!ColbaltBlueChecker methodsFor: 'printing'!

printOn: aStream
	constraint printOn: aStream.
	aStream nextPutAll: ' ?? '.
	aStream cr.
	thunk printOn: aStream!

printOn: aStream indent: sp 
	1 to: sp do: [:i | aStream space; space; space].
	constraint printOn: aStream.
	aStream nextPutAll: ' ?? '.
	aStream cr.
	thunk isThunk
		ifTrue: 
			[1 to: sp + 1 do: [:i | aStream space; space; space].
			thunk printOn: aStream]
		ifFalse: [thunk printOn: aStream indent: sp + 1]! !


Object subclass: #SkyBlueConstraintState
	instanceVariableNames: 'selectedmethod mark '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SkyBlue-Basics'!


!SkyBlueConstraintState methodsFor: 'accessing'!

mark
	^mark!

mark: b
	mark := b!

resetSkyBlueFields
	mark := nil.
	selectedmethod := nil!

selectedMethod
	^selectedmethod!

selectedMethod: m
	selectedmethod := m! !

!SkyBlueConstraintState methodsFor: 'testing'!

isActive
	^true!

isEnforced
	^selectedmethod isNil not!

isPatched
	^false! !

!SkyBlueConstraintState methodsFor: 'printing'!

printOn: aStream
	aStream nextPutAll: '[* ', selectedmethod printString, ' ', mark printString, ' *]'! !


SkyBlueConstraintState subclass: #ColbaltBlueConstraintState
	instanceVariableNames: 'active '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'ColbaltBlue-Basics'!


!ColbaltBlueConstraintState methodsFor: 'accessing'!

active: v
	active := v! !

!ColbaltBlueConstraintState methodsFor: 'testing'!

isActive
	^active! !

!ColbaltBlueConstraintState methodsFor: 'printing'!

printOn: aStream
	aStream nextPutAll: '[* ', selectedmethod printString, ' ', mark printString, ' ', active printString, ' *]'! !


Object subclass: #PositionConstraint
	instanceVariableNames: 'aLocation receiver settingMessage '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Constraints'!


!PositionConstraint methodsFor: 'initializing'!

location: l receiver: r sending: m
	aLocation := l.
	receiver := r.
	settingMessage := m.! !

!PositionConstraint methodsFor: 'damage control'!

update: aFigure
   receiver willChange.
   receiver perform: settingMessage with: aLocation value.
   receiver changed! !


Object subclass: #CoolDrawCopyState
	instanceVariableNames: 'copyDictionary restrictVariables copiedConstraints '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Support'!


!CoolDrawCopyState methodsFor: 'initialize-release'!

initialize: g 
	copyDictionary := IdentityDictionary new.
	restrictVariables := IdentitySet new.
	g do: [:fig | restrictVariables addAll: fig allConstrainedVariables].
	copiedConstraints := IdentitySet new.
	^self!

release
	copyDictionary := nil.
	restrictVariables := nil.
	copiedConstraints := nil! !

!CoolDrawCopyState methodsFor: 'testing'!

containsAllVariables: vars 
	vars do: [:v | (restrictVariables includes: v)
			ifFalse: [^false]].
	^true! !

!CoolDrawCopyState methodsFor: 'accessing'!

addConstraint: c
	copiedConstraints add: c!

collectedConstraints
	| t |
	t := SortedCollection  sortBlock: [:a :b | a strength >= b strength].
	t addAll: copiedConstraints.
	^t!

restrictToVariables
	^restrictVariables! !

!CoolDrawCopyState methodsFor: 'copying'!

copy: anObject 
	| n   |
	anObject isNil ifTrue: [^nil].
	^copyDictionary at: anObject
		ifAbsent: 
			[n := anObject shallowCopy postCopy.
			copyDictionary at: anObject put: n.
			n copyUsing: self.
			n]!

primCopy: anObject 
	| n   |
	anObject isNil ifTrue: [^nil].
	^copyDictionary at: anObject
		ifAbsent: 
			[n := anObject shallowCopy postCopy.
			copyDictionary at: anObject put: n.
			n]! !


Object subclass: #Locator
	instanceVariableNames: 'receiver selector arguments '
	classVariableNames: 'CopiedFigures '
	poolDictionaries: ''
	category: 'HotDraw-Constraints'!


!Locator methodsFor: 'accessing'!

object
        ^receiver!

value
        ^arguments == nil
                ifTrue: [receiver perform: selector]
                ifFalse: [receiver perform: selector withArguments: arguments]! !

!Locator methodsFor: 'converting'!

asPoint
	^self value! !

!Locator methodsFor: 'copying'!

copy
        ^CopiedFigures notNil
                ifTrue: [self copyOn: (Locator copyAt: receiver)]
                ifFalse: [super copy]!

copyOn: anObject
        ^self species on: anObject at: selector withArguments: arguments! !

!Locator methodsFor: 'printing'!

printOn: aStream
	aStream nextPutAll: self class name, '('.
	receiver printOn: aStream.
	aStream space.
	selector printOn: aStream.
	aStream nextPutAll: ')'! !

!Locator methodsFor: 'private'!

setReceiver: anObject selector: aSymbol arguments: anArray 
	receiver := anObject.
	selector := aSymbol.
	arguments := anArray! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Locator class
	instanceVariableNames: ''!


!Locator class methodsFor: 'instance creation'!

on: anObject at: aSymbol
        ^self new setReceiver: anObject selector: aSymbol arguments: nil!

on: anObject at: aSymbol with: anElement
        ^self new setReceiver: anObject selector: aSymbol arguments: (Array with: anElement)!

on: anObject at: aSymbol withArguments: anArray
        ^self new setReceiver: anObject selector: aSymbol arguments: anArray! !

!Locator class methodsFor: 'copying'!

copyAt: aFigure
        ^self copyAt: aFigure ifAbsent: [aFigure copy]!

copyAt: aFigure ifAbsent: aBlock
        "^CopiedFigures at: aFigure ifAbsent:
                [CopiedFigures at: aFigure put: aBlock value]"  

^CopiedFigures == nil
                ifTrue:
                        [aBlock value]
                ifFalse:
                        [CopiedFigures at: aFigure ifAbsent:
                                [CopiedFigures at: aFigure put: aBlock value]]!

copyWhile: aBlock
        "Answer a copy of aFigure preserving the identity of shared Figures."
 
        | anObject | 
        CopiedFigures := IdentityDictionary new.
        anObject := aBlock value.
        CopiedFigures := nil.
        ^anObject! !


Object subclass: #MultiheadedConstraint
	instanceVariableNames: 'sources sink action '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Constraints'!
MultiheadedConstraint comment:
'A MultiheadedConstraint makes the state of one object be the function
of the states of many other objects.  The one object is the "sink" and the
many objects are the "sources".  For example, if the value of one cell
in a spreadsheet is the sum of five other cells, the five cells are the
sources and the cell with the sum is the sink.  A MultiheadedConstraint
also has an "action", which is a block with two arguments that is
evaluated when any of the sources is changed.  The first block is
the sources, the second is the sink.  A block to compute a sum would be
[:sources :sink | sink value: (sources inject: 0 into: [:sum :each: | sum + each value]

Adding a source to the constraint makes the constraint a dependent of
the source.'!


!MultiheadedConstraint methodsFor: 'initialization'!

for: aFigure
	sink := aFigure.
	sources := OrderedCollection new.!

for: aFigure action: actionBlock
	sink := aFigure.
	sources := OrderedCollection new.
	action := actionBlock! !

!MultiheadedConstraint methodsFor: 'accessing'!

addSource: aFigure
	sources add: aFigure.
	aFigure addDependent: self.
	self update: aFigure! !

!MultiheadedConstraint methodsFor: 'damage control'!

update: aFigure 
	sink number: (action value: sources value: sink)! !


Object subclass: #SkyBlueAbstraction
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SkyBlue-Basics'!


!SkyBlueAbstraction methodsFor: 'testing'!

isConstraint
	^false!

isVariable
	^false! !


SkyBlueAbstraction subclass: #SkyBlueMethod
	instanceVariableNames: 'inputs outputs external codeblock '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SkyBlue-Basics'!


!SkyBlueMethod methodsFor: 'accessing'!

execute
	codeblock value: inputs value: outputs!

inputs
	^inputs!

outputs
	^outputs! !

!SkyBlueMethod methodsFor: 'testing'!

isExternal
	^external!

isSource
	^inputs isEmpty!

isStay
	^inputs isEmpty & external not! !

!SkyBlueMethod methodsFor: 'printing'!

printOn: aStream 
	aStream nextPutAll: '('.
	inputs
		do: 
			[:each | 
			each printOn: aStream.
			aStream space].
	aStream nextPutAll: '-'.
	external
		ifTrue: [aStream nextPutAll: '>> ']
		ifFalse: [aStream nextPutAll: '> '].
	outputs
		do: 
			[:each | 
			each printOn: aStream.
			aStream space].
	aStream nextPutAll: ')'! !

!SkyBlueMethod methodsFor: 'sky blue'!

executePropagateValid
	| inputsvalid |
	inputsvalid := true.
	inputs do: [:var | var isValid ifFalse: [inputsvalid := false]].
	inputsvalid ifTrue: [self execute].
	outputs do: [:var | var valid: inputsvalid]!

possibleMethod: aConstraint and: rootstrength and: donemark 
	outputs
		do: 
			[:var | 
			var mark = donemark ifTrue: [^false].
			var walkStrength >= rootstrength
				ifTrue: 
					[aConstraint selectedMethod isNil ifTrue: [^false].
					(aConstraint selectedMethod outputs includes: var)
						ifFalse: [^false]]].
	^true! !

!SkyBlueMethod methodsFor: 'private'!

block: b
	codeblock := b!

external: e
	external := e!

inputs: a
	inputs := a.!

outputs: a
	outputs := a! !

!SkyBlueMethod methodsFor: 'copying'!

copyUsing: aCopyState
	inputs := inputs collect: [:v | aCopyState copy: v].
	outputs := outputs collect: [:v | aCopyState copy: v].
	"external"
	"codeblock"! !


SkyBlueAbstraction subclass: #SkyBlueVariable
	instanceVariableNames: 'name graph contents constraints state '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SkyBlue-Basics'!


!SkyBlueVariable methodsFor: 'initialize-release'!

initializeVariable: n in: g 
	^self initializeVariable: n in: g with: nil!

initializeVariable: n in: g with: val 
	name := n.
	self resetSkyBlueFields: g.
	contents := val.
	^self!

release
	graph := nil.
	contents := nil.
	constraints := nil.
	state := nil.
	super release!

resetSkyBlueFields: aGraph 
	"name"
	graph := aGraph.
	"contents"
	constraints := OrderedCollection new.
	state := SkyBlueVariableState new.
	state determinedBy: nil.
	state walkStrength: SkyBlueStrength absoluteWeakest.
	state mark: nil.
	state valid: true.
	^self! !

!SkyBlueVariable methodsFor: 'accessing'!

constraints
	^constraints!

contents
	^contents!

contents: v
	contents := v.
	self changed: #text! !

!SkyBlueVariable methodsFor: 'testing'!

isValid
	^state valid!

isVariable
	^true! !

!SkyBlueVariable methodsFor: 'printing'!

printOn: aStream 
	aStream nextPutAll: '{' , "self asOop printString, '*'," name.
	InputState default shiftDown
		ifTrue: 
			[aStream nextPut: $=.
			self contents printOn: aStream].
	aStream nextPutAll: '}'! !

!SkyBlueVariable methodsFor: 'sky blue'!

activeConstraintsDo: aBlock 
	^constraints do: [:each | each isActive ifTrue: [aBlock value: each]]!

attachConstraint: c
	constraints add: c!

collectUnenforcedMark: unenforced and: collectionstrength and: collectequal and: donemark 
	self activeConstraintsDo: [:cn | cn ~~ state determinedBy & (cn mark ~= donemark)
			ifTrue: 
				[cn mark: donemark.
				cn isEnforced
					ifTrue: [cn selectedMethod outputs do: [:outvar | outvar
								collectUnenforcedMark: unenforced
								and: collectionstrength
								and: collectequal
								and: donemark]]
					ifFalse: [cn strength < collectionstrength | (collectequal & (cn strength = collectionstrength)) ifTrue: [unenforced add: cn]]]]!

deattachConstraint: c
	constraints remove: c!

determinedBy
	^state determinedBy!

determinedBy: aConstraint
	state determinedBy: aConstraint!

mark
	^state mark!

mark: b
	state mark: b!

pplanAdd: pplan and: donemark 
	self activeConstraintsDo: [:cn | cn ~~ state determinedBy ifTrue: [cn pplanAdd: pplan and: donemark]]!

valid: b
	state valid: b!

walkStrength
	^state walkStrength!

walkStrength: aStrength
	state walkStrength: aStrength! !

!SkyBlueVariable methodsFor: 'colbalt blue'!

copyOfState
	^state copy!

setState: s
	state := s! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SkyBlueVariable class
	instanceVariableNames: ''!


!SkyBlueVariable class methodsFor: 'instance creation'!

newIn: aGraph 
	^self newNamed: nil in: aGraph!

newIn: aGraph with: val
	^self newNamed: nil in: aGraph with: val!

newNamed: aString in: aGraph 
	^self newNamed: aString in: aGraph with: nil!

newNamed: aString in: aGraph with: val
	| me n |
	me := self new.
	aString isNil
		ifTrue: [n := 'v' , me asOop printString]
		ifFalse: [n := aString].
	me initializeVariable: n in: aGraph with: val.
	aGraph addVariable: me.
	^me! !


SkyBlueVariable subclass: #CoolDrawVariable
	instanceVariableNames: 'owner '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Framework'!
CoolDrawVariable comment:
'CoolDrawVariable is a specialization of a standard constrainable variable (SkyBlueVariable) that remembers its owning figure and thus can help with damaging the figures region (for redrawing) whenever the value of the variable changes.
'!


!CoolDrawVariable methodsFor: 'initialize-release'!

release
	owner := nil.
	super release! !

!CoolDrawVariable methodsFor: 'accessing'!

owner
	^owner!

owner: o
	owner := o! !

!CoolDrawVariable methodsFor: 'enumerating'!

do: aBlock
	aBlock value: self! !

!CoolDrawVariable methodsFor: 'copying'!

copyUsing: aCopyState 
	| t |
	"name"
	graph := nil.
	"contents"
	t := OrderedCollection new.
	constraints do: [:cn | (aCopyState containsAllVariables: cn variables)
			ifTrue: [t add: (aCopyState copy: cn)]].
	constraints := t.
	state := nil.
	owner := aCopyState copy: owner!

pasteUsing: aPasteState owner: o 
	self resetSkyBlueFields: aPasteState constraintGraph.
	owner := o.
	aPasteState constraintGraph addVariable: self! !


CoolDrawVariable subclass: #CoolDrawActiveVariable
	instanceVariableNames: 'performBlock '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Framework'!
CoolDrawActiveVariable comment:
'A CoolDrawActiveVariable adds the ability to have a block of code executed every time its
contents are changed.

Instance Variables:
	performBlock  <Block Closure>		The code to be executed at contents updating.'!


!CoolDrawActiveVariable methodsFor: 'accessing'!

contents: v
	super contents: v.
	performBlock value!

performBlock
	^performBlock!

performBlock: aBlock
	performBlock := aBlock! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

CoolDrawActiveVariable class
	instanceVariableNames: ''!


!CoolDrawActiveVariable class methodsFor: 'instance creation'!

newIn: aGraph 
	^self newNamed: nil in: aGraph peformBlock: []!

newIn: aGraph with: val
	^self newNamed: nil in: aGraph with: val performBlock: []!

newIn: aGraph with: val perfomBlock: aBlock
	^self newNamed: nil in: aGraph with: val performBlock: aBlock!

newNamed: aString in: aGraph 
	^self newNamed: aString in: aGraph with: nil performBlock: []!

newNamed: aString in: aGraph peformBlock: aBlock
	^self newNamed: aString in: aGraph with: nil performBlock: aBlock!

newNamed: aString in: aGraph with: val 
	^self
		newNamed: aString
		in: aGraph
		with: val
		performBlock: []!

newNamed: aString in: aGraph with: val performBlock: aBlock
	|me|
	me := super newNamed: aString in: aGraph with: val
	me performBlock: aBlock.
	^me! !


SkyBlueVariable subclass: #SkyBlueMVCVariable
	instanceVariableNames: 'dependents '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SkyBlue-Examples'!


!SkyBlueMVCVariable methodsFor: 'adaptor'!

acceptText: aText from: aController 
	(graph
		mvcChange: self
		to: aText
		from: aController)
		ifTrue: 
			[self changed: #text.
			^true]
		ifFalse: [^false]!

inspectGraph
	^graph inspect!

text
	^contents printString!

textMenu
	^PopUpMenu
		labels: 'accept\cancel\inspect graph' withCRs
		lines: #()
		values: #(#accept #cancel #inspectGraph )! !

!SkyBlueMVCVariable methodsFor: 'private'!

myDependents
	"Answer the receiver's dependents or nil."

	^dependents!

myDependents: dependentsOrNil
	"Set the receivers dependents."

	dependents := dependentsOrNil!

postCopy
	"Do not copy the dependents list."

	super postCopy.
	self breakDependents! !


SkyBlueAbstraction subclass: #SkyBlueConstraint
	instanceVariableNames: 'name graph variables strength errorfunction methods state '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SkyBlue-Basics'!


!SkyBlueConstraint methodsFor: 'initialize-release'!

release
	graph := nil.
	variables := nil.
	errorfunction := nil.
	methods := nil.
	state := nil.
	super release!

resetSkyBlueFields: aGraph 
	"name"
	graph := aGraph.
	"variables"
	"strength"
	"errorfunction"
	"methods"
	state isNil ifTrue: [state := self stateClass new].
	state resetSkyBlueFields.
	state selectedMethod: nil.
	state mark: nil! !

!SkyBlueConstraint methodsFor: 'accessing'!

active: v
	state active: v!

graph: g
	graph := g!

mark
	^state mark!

mark: b
	state mark: b!

selectedMethod
	^state selectedMethod!

selectedMethod: m
	state selectedMethod: m!

stateClass
	^SkyBlueConstraintState!

strength
	^strength!

variables
	^variables! !

!SkyBlueConstraint methodsFor: 'testing'!

isActive
	^state isActive!

isConstraint
	^true!

isEnforced
	^state isEnforced!

isExternal
	methods do: [:mt | mt isExternal ifTrue: [^true]].
	^false!

isPatched
	^state isPatched!

isSatisfied
	^(errorfunction value: variables) = 0!

isSource
	methods do: [:mt | mt isSource ifTrue: [^true]].
	^false!

isUnique
	^true! !

!SkyBlueConstraint methodsFor: 'printing'!

printOn: aStream
	aStream nextPutAll: strength name, '(', name, ')'.
	"aStream nextPutAll: '('.
	variables do: [:v | v printOn: aStream. aStream space].
	aStream nextPutAll: ')'"! !

!SkyBlueConstraint methodsFor: 'sky blue'!

areAnyImmediateUpstreamCnsMarked: m 
	state selectedMethod inputs do: [:var | var determinedBy notNil ifTrue: [var determinedBy mark = m ifTrue: [^true]]].
	^false!

buildPvine: redetermined 
	| pvinestack donemark |
	pvinestack := OrderedCollection new.
	donemark := SkyBlueMark new.
	^self
		pvineBranch: self strength
		and: donemark
		and: pvinestack
		and: redetermined!

computeWalkabout: var 
	| minstrength maxstrength |
	minstrength := strength.
	methods do: [:mt | (mt outputs includes: var)
			ifFalse: 
				[maxstrength := SkyBlueStrength absoluteWeakest.
				mt outputs do: [:outvar | maxstrength < outvar walkStrength ifTrue: [(state selectedMethod outputs includes: outvar)
							ifFalse: [maxstrength := outvar walkStrength]]].
				maxstrength < minstrength ifTrue: [minstrength := maxstrength]]].
	^minstrength!

getPossibleMethods: rootstrength and: donemark 
	| good |
	good := OrderedCollection new.
	methods do: [:mt | (mt possibleMethod: self and: rootstrength and: donemark)
			ifTrue: [good add: mt]].
	^good!

pplanAdd: pplan and: donemark 
	self isEnforced & (state mark ~= donemark)
		ifTrue: 
			[state mark: donemark.
			state selectedMethod outputs do: [:var | var pplanAdd: pplan and: donemark].
			pplan addFirst: self]!

pvineBranch: rootstrength and: donemark and: pvinestack and: redetermined 
	| sortedmts nextcns ok |
	state mark: donemark.
	sortedmts := self sortMethods: (self
					getPossibleMethods: rootstrength
					and: donemark).
	sortedmts
		do: 
			[:mt | 
			nextcns := OrderedCollection new.
			mt outputs do: [:var | var determinedBy isNil ifFalse: [nextcns add: var determinedBy]].
			nextcns do: [:newcn | pvinestack addFirst: newcn].
			mt outputs do: [:var | var mark: donemark].
			ok := graph
						pvineGrow: rootstrength
						and: donemark
						and: pvinestack
						and: redetermined.
			ok
				ifTrue: 
					[state selectedMethod notNil ifTrue: [state selectedMethod outputs
							do: 
								[:var | 
								redetermined addFirst: var.
								var mark ~= donemark
									ifTrue: 
										[var determinedBy: nil.
										var walkStrength: SkyBlueStrength absoluteWeakest]]].
					state selectedMethod: mt.
					mt outputs do: [:var | var determinedBy: self].
					^true]
				ifFalse: 
					[mt outputs do: [:var | var mark: nil].
					nextcns do: [:newcn | pvinestack removeFirst]]].
	state mark: nil.
	^false!

pvineLeaf: rootstrength and: donemark and: pvinestack and: redetermined 
	| ok |
	state mark: donemark.
	ok := graph
				pvineGrow: rootstrength
				and: donemark
				and: pvinestack
				and: redetermined.
	ok
		ifTrue: 
			[state selectedMethod outputs
				do: 
					[:var | 
					redetermined addFirst: var.
					var mark ~= donemark
						ifTrue: 
							[var determinedBy: nil.
							var walkStrength: SkyBlueStrength absoluteWeakest]].
			state selectedMethod: nil.
			^true]
		ifFalse: 
			[state mark: nil.
			^false]!

sortMethods: meths
	^meths "so says Michael..."! !

!SkyBlueConstraint methodsFor: 'colbalt blue'!

copyOfState
	^state copy!

setState: s
	state := s! !

!SkyBlueConstraint methodsFor: 'private'!

errorfunction: aBlock
	errorfunction := aBlock!

methods: m
	methods := m!

name: n
	name := n!

strength: s
	strength := s!

variables: v 
	variables := v! !

!SkyBlueConstraint methodsFor: 'copying'!

copyUsing: aCopyState
	"name"
	graph := nil.
	variables := variables collect: [:v | aCopyState copy: v].
	"strength"
	"errorfunction"
	methods := methods collect: [:m | aCopyState copy: m].
	state := nil.
	aCopyState addConstraint: self!

pasteUsing: aPasteState 
	self resetSkyBlueFields: aPasteState constraintGraph.
	aPasteState constraintGraph addConstraint: self! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SkyBlueConstraint class
	instanceVariableNames: ''!


!SkyBlueConstraint class methodsFor: 'sky blue instances'!

assign: val to: v1 atStrength: s
	| me memethod1  |
	me := self new.
	me name: (v1 printString), ' := ', val printString.
	me strength: s.
	me variables: (Array with: v1).
	me errorfunction: [:vars | ((vars at: 1) contents - val) abs].
	memethod1 := SkyBlueMethod new.
	memethod1 inputs: (Array new).
	memethod1 outputs: (Array with: v1).
	memethod1 external: true.
	memethod1 block: [:in :out | (out at: 1) contents: val].
	me methods: (Array with: memethod1).
	^me!

editOn: v1
	^self editOn: v1 atStrength: (SkyBlueStrength required)!

editOn: v1 atStrength: s
	| me memethod1  |
	me := self new.
	me name: (v1 printString), ' edit'.
	me strength: s.
	me variables: (Array with: v1).
	me errorfunction: [:vars | 0].
	memethod1 := SkyBlueMethod new.
	memethod1 inputs: (Array new).
	memethod1 outputs: (Array with: v1).
	memethod1 external: true.
	memethod1 block: [:in :out | ].
	me methods: (Array with: memethod1).
	^me!

equalityBetween: v1 and: v2
	^self
		equalityBetween: v1
		and: v2
		atStrength: SkyBlueStrength required!

equalityBetween: v1 and: v2 atStrength: s
	| me memethod1 memethod2 |
	me := self new.
	me name: (v1 printString), ' = ', (v2 printString).
	me strength: s.
	me variables: (Array with: v1 with: v2).
	me errorfunction: [:vars | ((vars at: 1) contents - (vars at: 2) contents) abs].
	memethod1 := SkyBlueMethod new.
	memethod1 inputs: (Array with: v1).
	memethod1 outputs: (Array with: v2).
	memethod1 external: false.
	memethod1 block: [:in :out | (out at: 1) contents: (in at: 1) contents].
	memethod2 := SkyBlueMethod new.
	memethod2 inputs: (Array with: v2).
	memethod2 outputs: (Array with: v1).
	memethod2 external: false.
	memethod2 block: [:in :out | (out at: 1) contents: (in at: 1) contents].
	me methods: (Array with: memethod1 with: memethod2).
	^me!

mediumAssign: val to: v1
	^self assign: val to: v1 atStrength: (SkyBlueStrength medium)!

mediumEditOn: v1
	^self editOn: v1 atStrength: (SkyBlueStrength medium)!

plusBetween: v1 and: v2 and: v3
	^self plusBetween: v1 and: v2 and: v3 atStrength: (SkyBlueStrength required)!

plusBetween: v1 and: v2 and: v3 atStrength: s
	| me memethod1 memethod2 memethod3 |
	me := self new.
	me name: (v1 printString), ' = ', (v2 printString), ' + ', (v3 printString).
	me strength: s.
	me variables: (Array with: v1 with: v2 with: v3).
	me errorfunction: [:vars | ((vars at: 1) contents - ((vars at: 2) contents + (vars at: 3) contents)) abs].
	memethod1 := SkyBlueMethod new.
	memethod1 inputs: (Array with: v1 with: v2).
	memethod1 outputs: (Array with: v3).
	memethod1 external: false.
	memethod1 block: [:in :out | (out at: 1) contents: ((in at: 1) contents - (in at: 2) contents)].
	memethod2 := SkyBlueMethod new.
	memethod2 inputs: (Array with: v1 with: v3).
	memethod2 outputs: (Array with: v2).
	memethod2 external: false.
	memethod2 block: [:in :out | (out at: 1) contents: ((in at: 1) contents - (in at: 2) contents)].
	memethod3 := SkyBlueMethod new.
	memethod3 inputs: (Array with: v2 with: v3).
	memethod3 outputs: (Array with: v1).
	memethod3 external: false.
	memethod3 block: [:in :out | (out at: 1) contents: ((in at: 1) contents + (in at: 2) contents)].
	me methods: (Array with: memethod1 with: memethod2 with: memethod3).
	^me!

stayOn: v1
	^self stayOn: v1 atStrength: (SkyBlueStrength required)!

stayOn: v1 atStrength: s
	| me memethod1  |
	me := self new.
	me name: (v1 printString), ' stay'.
	me strength: s.
	me variables: (Array with: v1).
	me errorfunction: [:vars | 0].
	memethod1 := SkyBlueMethod new.
	memethod1 inputs: (Array new).
	memethod1 outputs: (Array with: v1).
	memethod1 external: false.
	memethod1 block: [:in :out | ].
	me methods: (Array with: memethod1).
	^me!

weakStayOn: v1
	^self stayOn: v1 atStrength: (SkyBlueStrength weak)! !


SkyBlueConstraint subclass: #ColbaltBlueNonuniqueConstraint
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'ColbaltBlue-Basics'!


!ColbaltBlueNonuniqueConstraint methodsFor: 'accessing'!

stateClass
	^ColbaltBlueConstraintState! !

!ColbaltBlueNonuniqueConstraint methodsFor: 'testing'!

isUnique
	^false! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ColbaltBlueNonuniqueConstraint class
	instanceVariableNames: ''!


!ColbaltBlueNonuniqueConstraint class methodsFor: 'colbalt blue instances'!

greaterEqualBetween: v1 and: v2
	^self
		greaterEqualBetween: v1
		and: v2
		atStrength: SkyBlueStrength required!

greaterEqualBetween: v1 and: v2 atStrength: s 
	| me memethod1 memethod2 |
	me := self new.
	me name: v1 printString , ' >= ' , v2 printString.
	me strength: s.
	me variables: (Array with: v1 with: v2).
	me errorfunction: [:vars | (vars at: 1) contents >= (vars at: 2) contents
			ifTrue: [0]
			ifFalse: [((vars at: 1) contents - (vars at: 2) contents) abs]].
	memethod1 := SkyBlueMethod new.
	memethod1 inputs: (Array with: v1).
	memethod1 outputs: (Array with: v2).
	memethod1 external: false.
	memethod1 block: [:in :out | (out at: 1) contents: (in at: 1) contents].
	memethod2 := SkyBlueMethod new.
	memethod2 inputs: (Array with: v2).
	memethod2 outputs: (Array with: v1).
	memethod2 external: false.
	memethod2 block: [:in :out | (out at: 1) contents: (in at: 1) contents].
	me methods: (Array with: memethod1 with: memethod2).
	^me! !


Object subclass: #CoolDrawDeleteDialog
	instanceVariableNames: 'drawing clusters clusterValues '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Interface'!


!CoolDrawDeleteDialog methodsFor: 'initialize'!

initialize: d clusters: c
	drawing := d.
	clusters := c asOrderedCollection.
	clusterValues := clusters collect: [:each | ValueHolder with: false].
	^self!

releaes
	drawing := nil.
	clusters := nil.
	clusterValues do: [:each | each release].
	clusterValues := nil.
	^self! !

!CoolDrawDeleteDialog methodsFor: 'accessing'!

valuesAndHolders
	| collec |
	collec := OrderedCollection new.
	clusters with: clusterValues do: [:v1 :v2 | collec add: (Array with: v1 name with: v2)].
	^collec! !

!CoolDrawDeleteDialog methodsFor: 'enumerating'!

selectedClustersDo: aBlock 
	clusters with: clusterValues do: [:v1 :v2 | v2 value ifTrue: [aBlock value: v1]]! !

!CoolDrawDeleteDialog methodsFor: 'testing'!

isOk
	drawing isNil ifTrue: [^false].
	clusterValues do: [:each | each value ifTrue: [^true]].
	^false! !

!CoolDrawDeleteDialog methodsFor: 'adaptor'!

cancel
	drawing := nil.
	self changed: #cancel!

ok
	self changed: #ok! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

CoolDrawDeleteDialog class
	instanceVariableNames: ''!


!CoolDrawDeleteDialog class methodsFor: 'instance creation'!

on: drawing clusters: clusters
	| aModel composite |
	aModel := self new
				initialize: drawing clusters: clusters.
	composite := CoolDrawDialogView model: aModel.
	composite width: 300.
	self title: 'Delete Constraints' on: composite.
	self clusterSelectionColumn: composite.
	self deleteCancelButtons: composite.
	composite addVerticalSpace: 10; open.
	^aModel! !

!CoolDrawDeleteDialog class methodsFor: 'view creation'!

clusterSelectionColumn: composite
	| button  collec |
	composite addVerticalSpace: 5.
	collec := composite model valuesAndHolders.
	composite
		addColumn: collec
		fromX: 0.1
		toX: 0.95
		collect: 
			[:config | 
			"( name holder )"
			button := (Button switch) label: (config at: 1); model: (config at: 2).
			BoundedWrapper on: button]!

deleteCancelButtons: composite 
	| yp button  wrapper |
	composite addVerticalSpace: 5.
	yp := composite yPosition.
	button := (Button trigger) label: 'Delete'; model: ((PluggableAdaptor on: composite model)
					performAction: #ok).
	wrapper := BoundedWrapper on: button.
	composite yPosition: yp; addWrapper: wrapper atX: 0.3.
	button := (Button trigger) label: 'Cancel'; model: ((PluggableAdaptor on: composite model)
					performAction: #cancel).
	wrapper := BoundedWrapper on: button.
	composite yPosition: yp; addWrapper: wrapper atX: 0.6!

title: aString on: composite
	composite addVerticalSpace: 5.
	composite addTextLabel: aString! !


!ChangeSet methodsFor: 'bjorn'!

getClassChanges
	^classChanges!

getClassRemoves
	^classRemoves!

getMethodChanges
	^methodChanges!

getReorganizeSystem
	^reorganizeSystem!

getSpecialDoits
	^specialDoIts!

integrate1: anotherChangeSet 
	reorganizeSystem := reorganizeSystem or: [anotherChangeSet getReorganizeSystem].
	specialDoIts addAll: anotherChangeSet getSpecialDoits!

integrate2: anotherChangeSet 
	|       old  |
	anotherChangeSet getClassChanges
		associationsDo: 
			[:assoc | 
			(assoc value includes: #rename)
				ifTrue: [old := nil.
					   assoc value inject: nil into: [:val :each | each class = ByteString ifTrue: [(each copyFrom: 1 to: 9) = 'oldName: '
								ifTrue: 
									[old := each copyFrom: 10 to: each size.
									old := old asSymbol.
									(classChanges includesKey: assoc key)
										ifTrue: [self halt: 'renamed to something that already exists'].
									(classChanges includesKey: old)
										ifFalse: [Transcript show: 'renaming from ' , old , ' to ' , assoc key , ' but no previous changes']
										ifTrue: 
											[classChanges at: assoc key put: (classChanges at: old).
											classChanges removeKey: old].
									(methodChanges includesKey: old)
										ifTrue: 
											[methodChanges at: assoc key put: (methodChanges at: old).
											methodChanges removeKey: old]]
								ifFalse: [self halt: 'This should not happen']]].
						old isNil ifTrue: [self halt: 'This should not happen either']].
			(assoc value includes: #add)
				ifTrue: 
					[(classChanges at: assoc key ifAbsent: [nil]) notNil ifTrue: [self halt: 'adding class ' , assoc key , ' that already exists'].
					(classChanges at: assoc key put: Set new)
						add: #add.
					(classRemoves includes: assoc key)
						ifTrue: 
							[Transcript show: 'adding class ' , assoc key , ' that was previously removed'.
							classRemoves remove: assoc key]].
			(assoc value includes: #change)
				ifTrue: [(self integrate2a: assoc key) ifFalse: [self integrate2b: assoc key and: #change]].
			(assoc value includes: #comment)
				ifTrue: [(self integrate2a: assoc key) ifFalse: [self integrate2b: assoc key and: #comment]].
			(assoc value includes: #reorganize)
				ifTrue: [(self integrate2a: assoc key) ifFalse: [self integrate2b: assoc key and: #reorganize]].
]!

integrate2a: cls 
	| s t |
	s := classChanges at: cls ifAbsent: [Set new].
	(s includes: #add)
		ifTrue: [^true].
	(cls includes: $ )
		ifTrue: 
			[t := cls copyFrom: 1 to: cls size - 6.
			s := classChanges at: t asSymbol ifAbsent: [Set new].
			(s includes: #add)
				ifTrue: [^true]].
	^false!

integrate2b: cls and: k
	| s |
	s := classChanges at: cls ifAbsent: [classChanges at: cls put: Set new].
	s add: k!

integrate3: anotherChangeSet 
	| cls meth sel act2 act1 act0 |
	anotherChangeSet getMethodChanges
		associationsDo: 
			[:assoc | 
			cls := assoc key.
			(classRemoves includes: cls)
				ifTrue: [self halt: 'method changes for a removed class'].
			meth := methodChanges at: cls ifAbsent: [methodChanges at: cls put: IdentityDictionary new].
			assoc value
				associationsDo: 
					[:a2 | 
					sel := a2 key.
					act2 := a2 value.
					act1 := meth at: sel ifAbsent: [nil].
					act1 == nil & (act2 == #add) ifTrue: [act0 := #add].
					act1 == nil & (act2 == #change) ifTrue: [act0 := #change].
					act1 == nil & (act2 == #remove) ifTrue: [act0 := #remove].
					act1 == #add & (act2 == #add)
						ifTrue: 
							[self halt: 'adding previously added method'.
							act0 := #add].
					act1 == #add & (act2 == #change) ifTrue: [act0 := #add].
					act1 == #add & (act2 == #remove) ifTrue: [act0 := nil].
					act1 == #change & (act2 == #add)
						ifTrue: 
							[self halt: 'adding previous existing method'.
							act0 := #add].
					act1 == #change & (act2 == #change) ifTrue: [act0 := #change].
					act1 == #change & (act2 == #remove) ifTrue: [act0 := #remove].
					act1 == #remove & (act2 == #add)
						ifTrue: 
							[self halt: 'adding previous existing method'.
							act0 := #add].
					act1 == #remove & (act2 == #change) ifTrue: [self halt: 'changing previously removed method'].
					act1 == #remove & (act2 == #remove)
						ifTrue: 
							[self halt: 'removing previously removed method'.
							act0 := #remove].
					act0 notNil ifTrue: [(self integrate2a: cls)
							ifTrue: [methodChanges removeKey: cls ifAbsent: []]
							ifFalse: [(methodChanges at: cls)
									at: sel put: act0]]]]!

integrate4: anotherChangeSet 
	anotherChangeSet getClassRemoves
		do: 
			[:each | 
			classRemoves add: each.
			(classChanges includesKey: each)
				ifTrue: [Transcript show: 'removing class ' , each , ' that has class changes'].
			classChanges removeKey: each ifAbsent: [].
			(methodChanges includesKey: each)
				ifTrue: [Transcript show: 'removing class ' , each , ' that has method changes'].
			methodChanges removeKey: each ifAbsent: []]!

integrate: anotherChangeSet 
	self integrate1: anotherChangeSet.
	self integrate2: anotherChangeSet.
	self integrate3: anotherChangeSet.
	self integrate4: anotherChangeSet.! !


Model subclass: #NumberHolder
	instanceVariableNames: 'number '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Constraints'!


!NumberHolder methodsFor: 'accessing'!

number
	^number!

number: aNumber
	number := aNumber.
	self changed! !

!NumberHolder methodsFor: 'damage control'!

changed
	self dependents do: [:dep | dep update: self]! !


VisualPart subclass: #Figure
	instanceVariableNames: 'dependents metaFigure graph '
	classVariableNames: 'ColorNameTable ColorTable '
	poolDictionaries: ''
	category: 'HotDraw-Framework'!
Figure comment:
'**
 	Change Date: March 1993
	Ian Perrigo
	Made Figures constrainable
**
Figure is an abstract class.

A Subclass must implement :
origin		The origin of the graphical image of the subclass.  This is used in computing the displayBox among other things.
extent		The further reaches of the graphical image of the subclass.  This is used in computing the displayBox among other things.
displayOn:	Display the graphical image of the Figure on the supplied GraphicsContext.
translateBy: 	Move a Figure.  
asImage		Answer an Image which is a visual representation of the Figure.

Instance Variables :
dependents	<Collection>
	A collection of objects that are dependent upon me.  This is the same implementation as Model.  It is copied here to override the default (less efficient) implementation of a global Dictionary.

Class Variables :
ColorTable	<Array>	An Array of the colors defined by the default Palette to be used in Figure coloring.
'!


!Figure methodsFor: 'initialize'!

initialize
	"Don't need to do anything. Just here to support it in my subclasses."! !

!Figure methodsFor: 'accessing'!

boundingBox
        ^0@0 extent: self extent!

connectionPosition
	"Return the position that a line should be connected to me."
        ^ self center!

displayBox
        ^self origin extent: self extent!

extent
	^self subclassResponsibility!

figure
	^self!

figureAt: aPoint
        ^(self containsPoint: aPoint)
                ifTrue: [self]
                ifFalse: [nil]!

figures
        ^Array with: self!

handles
        ^SelectionTrackHandle allCornersOf: self!

kindsOfFigures
        ^Array with: self class!

locator
        ^Locator on: self at: #center!

menu
	(metaFigure respondsTo: #menu)
		ifTrue: [^metaFigure menu].
	^PopUpMenu labels: 'cut\copy\paste' withCRs values: #(#cut #copy #paste )!

menuBindings
	"Return the collection of symbols in menu that represent messages that
	should be sent to me instead of to the controller."
        ^#()!

metaFigure
	"This is the object which holds behavior for the figure"

	^metaFigure!

metaFigure: aMetaFigureObject 
	"This is the object which holds behavior for the figure"

	metaFigure := aMetaFigureObject!

origin
        ^self subclassResponsibility!

owner
        ^self!

shape
	"Answer the Mask that is my shape.  Subclasses really should reimplement this."

	^Mask extent: self extent rounded! !

!Figure methodsFor: 'bounding box accessing'!

absolute: aRelativePoint 
	| aRectangle |
	aRectangle := self displayBox.
	^aRectangle extent * aRelativePoint + aRectangle origin!

bottom
        ^self displayBox bottom!

bottomCenter
        ^self displayBox bottomCenter!

bottomLeft
        ^self displayBox bottomLeft!

bottomRight
        ^self displayBox bottomRight!

center
        ^self displayBox center!

corner
        ^self displayBox corner!

left
        ^self displayBox left!

leftCenter
        ^self displayBox leftCenter!

offCenter: deltaPoint
        ^self center + deltaPoint!

offCorner: deltaPoint
        ^self displayBox corner + deltaPoint!

offOrigin: deltaPoint
        ^self displayBox origin + deltaPoint!

relative: anAbsolutePoint 
	| aRectangle |
	aRectangle := self displayBox.
	^anAbsolutePoint - aRectangle origin * 1.0 / aRectangle extent!

right
        ^self displayBox right!

rightCenter
        ^self displayBox rightCenter!

top
        ^self displayBox top!

topCenter
        ^self displayBox topCenter!

topLeft
        ^self displayBox topLeft!

topRight
        ^self displayBox topRight! !

!Figure methodsFor: 'copying'!

copy
self obsoleteMessage.
	"|  aCollection aFigure |
	^Locator copyAt: self
		ifAbsent: 
			[aCollection := dependents.
			dependents := nil.
			aFigure := super copy.
			dependents := aCollection.
			aFigure]"!

copyUsing: aCopyState
	"Copy my state for pasting.  Note that this means that
	I do not need a complete state (see Drawing|paste for
	more details).  Use the copy state to avoid cycles.
	Remember: I am the copy.  See EllipseFigure for an example."
	^self subclassResponsibility!

pasteUsing: aPasteState 
	graph := aPasteState constraintGraph.
	self allConstrainedVariables do: [:each | each pasteUsing: aPasteState owner: self]! !

!Figure methodsFor: 'comparing'!

= aFigure
        ^self class == aFigure class and: [self displayBox = aFigure displayBox]!

hash
        ^self displayBox hash! !

!Figure methodsFor: 'converting'!

asByteArray: aWordArray size: aSize 
	| aByteArray aCollection |
	aByteArray := ByteArray new: 2 * aWordArray size.
	1 to: aWordArray size do: [:eachIndex | aByteArray wordAt: eachIndex put: (aWordArray at: eachIndex)].
	aCollection := aByteArray asOrderedCollection.
	aCollection size == aSize
		ifTrue: [^aCollection asArray]
		ifFalse: [^(aCollection removeLast; yourself) asArray]!

asImage
	"Make the figure into an Image - sounds like a job for Subclass Man"

	self subclassResponsibility! !

!Figure methodsFor: 'damage control'!

changed
       container notNil ifTrue: [ container damageRegion: self displayBox].
        self dependents do: [:each | each update: self]!

dependentDrawing
 	"Answer the Drawing that is dependent upon the receiver. Note that the
	constrainable version supports only one dependent Drawing."
	
	^self container
	"self dependents do: [:each | each isDrawing ifTrue: [^each]].
	^nil"!

dependentDrawings
	
	^Array with: self dependentDrawing
       " ^self dependents select: [:each | each isDrawing]"!

dependentFigures
        ^self dependents select: [:each | each isFigure]!

updateDrawing
	self container damageAll!

willChange
	"Damage my region in my container.  At the moment it is
	hard to find my container, so I damage everything."
        container notNil ifTrue: [container damageRegion: self displayBox]! !

!Figure methodsFor: 'displaying'!

displayOn: aGraphicsContext
	^self subclassResponsibility!

displayShapeOn: aGraphicsContext at: aPoint 
	self subclassResponsibility! !

!Figure methodsFor: 'enumerating'!

do: aBlock
        aBlock value: self!

members
	| aSet |
	aSet := IdentitySet new.
	self do: [:each | aSet add: each].
	^aSet! !

!Figure methodsFor: 'sensing'!

senseBottomLeft: deltaPoint
	self obsoleteMessage.
	^Rectangle origin: (deltaPoint x)@0 extent: (deltaPoint x negated)@(deltaPoint y)!

senseBottomRight: deltaPoint 
	self obsoleteMessage.
	^Rectangle origin: 0 @ 0 extent: deltaPoint!

senseColor: deltaPoint
	self obsoleteMessage.
        ^deltaPoint y!

senseTopLeft: deltaPoint
	self obsoleteMessage.
	^Rectangle origin: deltaPoint extent: deltaPoint negated!

senseTopRight: deltaPoint
	self obsoleteMessage.
	^Rectangle origin: 0@(deltaPoint y) extent: (deltaPoint x)@(deltaPoint y negated)! !

!Figure methodsFor: 'testing'!

acceptsTyping
        ^false!

canBeConnected
        
        ^ true!

containedBy: aRectangle
        ^aRectangle contains: self displayBox!

containsPoint: aPoint
        ^self displayBox containsPoint: aPoint!

intersects: aRectangle
        ^self displayBox intersects: aRectangle!

isActive
        ^false!

isConnectionFigure
       
        ^false!

isDrawing
        ^false!

isFigure
        ^true! !

!Figure methodsFor: 'transforming'!

align: alignmentPoint with: relativePoint 
	self obsoleteMessage.
        "self translateBy: relativePoint - alignmentPoint"!

basicTranslateBy: aPoint
        self obsoleteMessage.
	"^self subclassResponsibility"!

scaleBy: aPoint
	self obsoleteMessage.
        "self
                align: self displayBox center
                with: (self displayBox center scaleBy: aPoint)"!

translateBy: aPoint
	"move me a distance of aPoint.
	Don't override this method, override basicTranslateBy:, instead."
	self obsoleteMessage.
     "self willChange.
	self basicTranslateBy: aPoint.
	self changed."!

translateTo: aPoint
	self obsoleteMessage.
       "self translateBy: aPoint - self displayBox corner"! !

!Figure methodsFor: 'private'!

myDependents
	"Answer the receiver's dependents or nil. Copied down from Model to make dependency checking faster."

	^dependents!

myDependents: dependentsOrNil
	"Set the receivers dependents. Copied down from Model to make dependency checking faster."

	dependents := dependentsOrNil! !

!Figure methodsFor: 'cooldraw'!

allConstrainedVariables
	"Return a collection of all the variables that this figure contributes
	to the constraint graph.  These are the variables that must be deleted
	when the figure is removed."

	self subclassResponsibility!

centerConstraint: tl and: c and: br in: graph 
	"Create a constraint of the form (left + right // 2 = center)."

	| cnst meth1 meth2 meth3 |
	cnst := SkyBlueConstraint new.
	cnst name: c printString , ' = (' , tl printString , ' + ' , br printString , ') / 2'.
	cnst strength: SkyBlueStrength required.
	cnst variables: (Array with: tl with: c with: br).
	cnst errorfunction: [:vars | ((((vars at: 1) contents + (vars at: 3) contents) // 2) - (vars at: 2) contents) abs].
	meth1 := SkyBlueMethod new.
	meth1 inputs: (Array with: tl with: br).
	meth1 outputs: (Array with: c).
	meth1 external: false.
	meth1 block: [:in :out | (out at: 1) contents: (in at: 1) contents + (in at: 2) contents // 2].
	meth2 := SkyBlueMethod new.
	meth2 inputs: (Array with: c with: tl).
	meth2 outputs: (Array with: br).
	meth2 external: false.
	meth2 block: [:in :out | (out at: 1) contents: (in at: 1) contents * 2 - (in at: 2) contents].
	meth3 := SkyBlueMethod new.
	meth3 inputs: (Array with: c with: br).
	meth3 outputs: (Array with: tl).
	meth3 external: false.
	meth3 block: [:in :out | (out at: 1) contents: (in at: 1) contents * 2 - (in at: 2) contents].
	cnst methods: (Array with: meth1 with: meth2 with: meth3).
	graph addConstraint: cnst!

constrainableFeatures
	"This is a list of pairs, each pair being the name of a feature and the 
	message to send to get the value of that feature. For an example, see 
	EllipseFigure."

	self subclassResponsibility!

constrainValueOf: v1 to: aValue
	|  cnst meth1  |
	cnst := SkyBlueConstraint new.
	cnst name: v1 printString.
	cnst strength:  SkyBlueStrength medium.
	cnst variables: (Array with: v1).
	cnst errorfunction: [:vars | self ian: 'implement this'].
	meth1 := SkyBlueMethod new.
	meth1 inputs: (Array new).
	meth1 outputs: (Array with: v1).
	meth1 external: false.
	meth1 block: [:in :out | (out at:1) contents: aValue ].
	cnst methods: (Array with: meth1).
	graph addConstraint: cnst.
	graph removeConstraint: cnst.
	self updateDrawing.!

convertToConstrainable: aSymbol
	"Return the symbol that references the constrainable version of what
	aSymbol references, e.g., #constrainableCenter for #center"

	^('constrainable',
		(String with: aSymbol first asUppercase),
		(aSymbol copyFrom: 2 to: aSymbol size)) asSymbol!

isConstrainable: aSelector
	"Return true if this selector is constrainable in this figure."

	self constrainableFeatures do: [:pair | (pair at: 2) = aSelector ifTrue: [^true]].
	^false!

majorColorValue
	"Return the ColorValue that is used for the majority of the figure,
	and thus the color that Handles must draw on top of."

	^nil!

name
	"Semi-friendly names for figures"

	^self class name, self asOop printString!

widthConstraint: l and: w and: r in: graph 
	"Create a contraint of the form (width = right - left)."

	| cnst meth1 meth2 meth3 |
	cnst := SkyBlueConstraint new.
	cnst name: w printString , ' = ' , r printString , ' - ' , l printString.
	cnst strength: SkyBlueStrength required.
	cnst variables: (Array with: l with: w with: r).
	cnst errorfunction: [:vars | ((vars at: 3) contents - (vars at: 1) contents - (vars at: 2) contents) abs].
	meth1 := SkyBlueMethod new.
	meth1 inputs: (Array with: r with: l).
	meth1 outputs: (Array with: w).
	meth1 external: false.
	meth1 block: [:in :out | (out at: 1) contents: (in at: 1) contents - (in at: 2) contents].
	meth2 := SkyBlueMethod new.
	meth2 inputs: (Array with: r with: w).
	meth2 outputs: (Array with: l).
	meth2 external: false.
	meth2 block: [:in :out | (out at: 1) contents: (in at: 1) contents - (in at: 2) contents].
	meth3 := SkyBlueMethod new.
	meth3 inputs: (Array with: l with: w).
	meth3 outputs: (Array with: r).
	meth3 external: false.
	meth3 block: [:in :out | (out at: 1) contents: (in at: 1) contents + (in at: 2) contents].
	cnst methods: (Array with: meth1 with: meth2 with: meth3).
	graph addConstraint: cnst! !

!Figure methodsFor: 'coloring'!

colorNamed: aString
	"Answer the color names aString from the class color table."

	^self class colorNamed: aString! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Figure class
	instanceVariableNames: ''!


!Figure class methodsFor: 'class initialization'!

initialize
	"Figure initialize"

	self initializeColorTable!

initializeColorTable
	"For colors use : Screen default colorPalette colors. Right now use a simple set of 
	grayscale colors.  Note that a 1:1 mapping exists between color names in the 
	ColorNameTable and the indices into the ColorTable. If new colors are added to
	the color table the indices in the ColorNameTable must be updated appropriately."

	ColorTable := (Array new: 7)
				at:1 put: ColorValue black;	
				at:2 put: ColorValue veryDarkGray;
				at:3 put: ColorValue darkGray;
				at:4 put: ColorValue gray;
				at:5 put: ColorValue lightGray;
				at:6 put: ColorValue veryLightGray;
				at:7 put: ColorValue white; yourself.

	ColorNameTable := (Dictionary new)
				at: 'black' put: 1;
				at: 'very dark gray' put: 2;
				at: 'dark gray' put: 3;
				at: 'gray' put: 4;
				at: 'light gray' put: 5;
				at: 'very light gray' put: 6;
				at: 'white' put: 7; yourself.
	^ColorTable.! !

!Figure class methodsFor: 'instance creation'!

allCreationTools
	| definingClasses |
	definingClasses := self allSubclasses select: [:each | each class includesSelector: #creationTool].
	^definingClasses collect: [:each | each creationTool]!

new
	^super new initialize.! !

!Figure class methodsFor: 'constants'!

black
	^1!

darkGray
	^2!

gray
	^3!

lightGray
	^4!

veryDarkGray
	^5!

veryLightGray
	^6!

white
	^ColorTable size! !

!Figure class methodsFor: 'colors'!

colorNamed: aString
	"Answer the index of the color from the color table with the key color named aString."
	^ColorNameTable at: aString!

colorValueAt: aString
	"Answer the color from the color table with the key color named aString."
	^ColorTable at: (ColorNameTable at: aString)! !


Figure subclass: #WrapperFigure
	instanceVariableNames: 'wrappedFigure '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
WrapperFigure comment:
'I am the superclass of wrappers of Figures.  Often a HotDraw application will want
to add some behavior to all Figures used in the drawing.  If Smalltalk had
multiple inheritance then we could define a class to specify the common
behavior and then each Figure used in the drawing would add it as a superclass.
Since Smalltalk does not have multiple inheritance, we will add behavior to
all Figures used in a drawing by making a subclass of WrapperFigure that
specifies the common behavior and then wrapping each Figure in the Drawing
with an instance of it.

The only instance variable is wrappedFigure, which holds the figure being
wrapped.  WrapperFigure delegates almost all operations to wrappedFigure.
'!


!WrapperFigure methodsFor: 'accessing'!

extent
	^wrappedFigure extent!

handles
        ^wrappedFigure handles!

menu
	^wrappedFigure menu!

origin
        ^wrappedFigure origin!

shape

	^wrappedFigure shape!

wrappedFigure
	^wrappedFigure! !

!WrapperFigure methodsFor: 'copying'!

copy
	|  aCollection aFigure |
	^Locator copyAt: self
		ifAbsent: 
			[aCollection := dependents.
			dependents := nil.
			aFigure := super copy.
			dependents := aCollection.
			aFigure wrappedFigure: wrappedFigure copy.
			aFigure]! !

!WrapperFigure methodsFor: 'initialize'!

wrappedFigure: aFigure
	wrappedFigure := aFigure.
	aFigure addDependent: self.! !

!WrapperFigure methodsFor: 'converting'!

asImage

	^wrappedFigure asImage! !

!WrapperFigure methodsFor: 'displaying'!

displayOn: aGraphicsContext
	^wrappedFigure displayOn: aGraphicsContext!

displayShapeOn: aGraphicsContext at: aPoint 
	wrappedFigure displayShapeOn: aGraphicsContext at: aPoint! !

!WrapperFigure methodsFor: 'testing'!

acceptsTyping
        ^wrappedFigure acceptsTyping!

canBeConnected
        
        ^ wrappedFigure canBeConnected!

isActive
        ^wrappedFigure isActive!

isConnectionFigure
       
        ^wrappedFigure isConnectionFigure! !

!WrapperFigure methodsFor: 'transforming'!

basicTranslateBy: aPoint
        self obsoleteMessage.
	"^wrappedFigure basicTranslateBy: aPoint"!

translateBy: aPoint
	self obsoleteMessage.
       "^wrappedFigure translateBy: aPoint"! !

!WrapperFigure methodsFor: 'damage control'!

update: aFigure
	"Is this really needed?"
        self changed! !


Figure subclass: #CachedFigure
	instanceVariableNames: 'cache origin '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
CachedFigure comment:
'CachedFigure is an abstract class that captures figures who wish to save their image to speed up motion.  It could be argued that all figures should do this but several use GraphicsContext primitives and should be fast enough not to need the cache.

Subclasses must implement fillCache to redraw their Images unto the cache.

Instance Varaibles :

cache	<Image | nil>
	The image to be displayed when I am moved or otherwise damaged.  If cache is nil, regenerate it with a call to fillCache.'!


!CachedFigure methodsFor: 'accessing'!

extent
	cache isNil ifTrue: [self fillCache].
	^cache extent!

origin
	^origin contents!

origin: aPoint
	origin contents: aPoint! !

!CachedFigure methodsFor: 'displaying'!

displayOn: aGC 
	"Fill my cache if it was empty or damaged. Then copy my cache directly to aGC."

	cache isNil ifTrue: [self fillCache].  
	cache displayOn: aGC at: self origin.! !

!CachedFigure methodsFor: 'damage control'!

damageCache
	"Mark my cache as damaged."

	cache := nil!

fillCache
	"Draw each of my component Figures onto my cache."

	^self subclassResponsibility! !

!CachedFigure methodsFor: 'transforming'!

basicTranslateBy: aPoint 
	self obsoleteMessage.
	"origin contents: (self origin translatedBy: aPoint).
	^origin contents"! !


CachedFigure subclass: #ImageFigure
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
ImageFigure comment:
'Class ImageFigure is a concrete class that represents arbitrary images in a Drawing.  The images are created by a call to Image class>fromUser.  The image depth is that of the underlying hardware.

Instance Variables :
origin	The origin of the image (which is held in the cache instance variable of my superclass).

'!


!ImageFigure methodsFor: 'displaying'!

displayShapeOn: aGraphicsContext at: aPoint 
	"Because I am a rectilinear figure, just color the area that is my bounding Rectangle."

	aGraphicsContext displayRectangle: (aPoint + self displayBox)! !

!ImageFigure methodsFor: 'damage control'!

damageCache
	"Don't damage my cache because I can't get it back"!

fillCache
	"This method should not be called as ImageFigures are created as Image fromUser."! !

!ImageFigure methodsFor: 'transforming'!

growBy: aPoint
	"Ignore this message.  ImageFigures don't scale yet."! !

!ImageFigure methodsFor: 'private'!

origin: aPoint image: anImage 
	origin contents: aPoint.
	cache := anImage! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ImageFigure class
	instanceVariableNames: ''!


!ImageFigure class methodsFor: 'instance creation'!

createNotifying: aView 
	| aFigure origin |
	origin := CoolDrawPoint newIn: (aView drawing graph)
							with: (aView controller sensor waitClickButton).
	aFigure := self new origin: origin image: (Image fromUser). 
	^aFigure!

creationTool
	| temp |
	temp := Image
				extent: 16 @ 16
				depth: 1
				palette: MappedPalette blackWhite
				bits: #[
2r11111111 2r11111111
2r10000001 2r11111111
2r10111111 2r11111111
2r10111111 2r11111111
2r10111111 2r11111111
2r10111111 2r11111111
2r10111111 2r11111111
2r11111111 2r11111111
2r11111111 2r11111111
2r11111111 2r11111101
2r11111111 2r11111101
2r11111111 2r11111101
2r11111111 2r11111101
2r11111111 2r11111101
2r11111111 2r10000001
2r11111111 2r11111111
] pad: 8.
	^CreationTool
		icon: temp
		cursor: Cursor crossHair
		class: self! !


CachedFigure subclass: #GroupFigure
	instanceVariableNames: 'figures '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
GroupFigure comment:
'A GroupFigure is a collection of Figures.  Scalings are not permitted on GroupFigures.

Instance Variables :

figures		<OrderedCollection>	
The Figures that make up the GroupFigure.  Each figure is translated so that its position is relative to the GroupFigure.

displayBox	<Rectangle>			
The area occupied by the intersections of the displayBoxes of each of the component Figures.  Cache it so we don''t have to recalculate it all the time.  Also used to translate the figures back to their real locations when ungrouped.

cache		<OpaqueImage>
The OpaqueImage of all the component Figures, precomputed so that motion is faster. An OpaqueImage is used so that the background objects will show through the gaps between component figures in the GroupFigure.  The OpaqueFigure is obtained by creating an Image of all of the component figures (using displayOn:at:) and a Mask of the shapes of all the figures (using displayShapeOn:at:).'!


!GroupFigure methodsFor: 'accessing'!

figures
        ^figures!

kindsOfFigures  
        ^self figures inject: (Set with: self class) into:
                [:sum :each |
                sum addAll: each kindsOfFigures; yourself]! !

!GroupFigure methodsFor: 'damage control'!

damageRegion: aFigure 
	"This message was sent to me because one of my components got damaged. Since I 
	already know about this message, just ignore it.

	It might be better to keep a damaged region and just invalidate part
	of the cache, like the drawing does."!

fillCache
	"Draw each of my component Figures onto my cache. My cache is an 
	OpaqueFigure so collect the images and their shapes from each of 
	my component figures."

	| aPixmap aMask myExtent |
	myExtent := (figures inject: figures first displayBox into: [:sum :each | sum merge: each displayBox]) extent.
	aPixmap := Pixmap extent: myExtent.
	aMask := Mask extent: myExtent.
	figures
		reverseDo: 
			[:each | 
			each displayOn: aPixmap graphicsContext at: Point zero.	"self origin negated"
			each displayShapeOn: aMask graphicsContext at: Point zero].
	cache := OpaqueImage figure: aPixmap asImage shape: aMask.!

update: aFigure 
	"This message was sent to me because one of my components got damaged. Since I 
	already know about this message, just ignore it."

	self error: 'not supposed to be called'! !

!GroupFigure methodsFor: 'copying'!

copy
	^self class figures: (figures collect: [:each | each copy])
		onGraph: self dependentDrawing graph!

copyWithPosition
	"This method was added by the HyperCard Project group.  When cards are copied, group figures must be translated to retain their current position."
	^self copy translateBy: (self origin).! !

!GroupFigure methodsFor: 'enumerating'!

do: aBlock 
	aBlock value: self.
	figures do: [:each | each do: aBlock]! !

!GroupFigure methodsFor: 'transforming'!

growBy: aPoint 
	"GroupFigures can't grow"!

scaleBy: aPoint 
	"GroupFigures cant be scaled."!

transformBy: aTransformation

      "No transformation permitted on GroupFigures"! !

!GroupFigure methodsFor: 'private'!

release
	figures do: 
		[:each | 
		each removeDependent: self.
		each translateBy: self origin].
	super release!

setFigures: aCollection 
	self obsoleteMessage.
	figures := aCollection.
	figures do: [:each | each container: self]. 
	self origin contents: (figures inject: figures first displayBox into: [:sum :each | sum merge: each displayBox]) origin.
	figures do: [:each | each translateBy: self origin negated].
	self fillCache!

setFigures: aCollection onGraph: aConstraintGraph 
	figures := aCollection.
	"Add the origin into the constraint graph before we set it's value"
	self setOrigin: 0@0 onGraph: aConstraintGraph.
	figures do: [:each | each container: self]. 
	self origin: (figures inject: figures first displayBox into: [:sum :each | sum merge: each displayBox]) origin.
	figures do: [:each | each translateBy: self origin negated].
	self fillCache!

setOrigin: aPoint onGraph: aConstraintGraph

	origin := CoolDrawPoint newIn: aConstraintGraph with: aPoint owner: self.! !

!GroupFigure methodsFor: 'sensing'!

senseBottomLeft: deltaPoint

"No scaling of GroupFigures"!

senseBottomRight: deltaPoint

"No scaling of GroupFigures"!

senseTopLeft: deltaPoint

"No scaling of GroupFigures"!

senseTopRight: deltaPoint

"No scaling of GroupFigures"! !

!GroupFigure methodsFor: 'displaying'!

displayShapeOn: aGraphicsContext at: aPoint 
	"Display the shape of my image on aGraphicsContext. Do this by having each of my 
	component figures display their shapes on aGraphicsContext."

	figures do: [:each | each displayShapeOn: aGraphicsContext at: aPoint + self origin]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GroupFigure class
	instanceVariableNames: ''!


!GroupFigure class methodsFor: 'instance creation'!

figures: aCollection 
	| aGroup |
	self obsoleteMessage.
	aGroup := self new setFigures: aCollection.
	aGroup origin: 0 @ 0.
	^aGroup!

figures: aCollection onGraph: aConstraintGraph 
	^self new setFigures: aCollection onGraph: aConstraintGraph!

figures: aCollection onView: aView 
	^self new setFigures: aCollection onGraph: aView drawing graph! !


Figure subclass: #PolylineFigure
	instanceVariableNames: 'points origin extent borderWidth color closed fillColor '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
PolylineFigure comment:
'***
	Modified March 1993
	By Ian Perrigo
	A Polyline Figure is now SkyBlueConstrainable. 
	Note that the extent and origin instance vars are not constrained
***
PolylineFigure is an abstract class designed to capture figures drawn with one or more lines.  This includes straight lines, rectangles, triangles, etc.  A PolylineFigure can be closed or open.  If it is closed, there is a fillColor that is used to color the area inside the polygon.

My subclasses must implement the following methods :
handles

In addition, it is often necessary to implement several handle-manipulation methods like growBy:, etc.

Instance Variables :
points	<OrderedCollection>
	The points that make up the figure.  They are drawn in the order they appear in the OrderedCollection.  If the figure is a closed PolylineFigure, the last point must be equal to the first point.

origin   <Integer>
extent  <Integer>
	These are recalcuated any time a point changes so that boundingBoxes
	can be calculated quickly.

width	<Integer>
	The pixel width at which the lines are drawn.

color		<ColorValue>
	The color of the lines (or the border of a closed, filled PolylineFigure).

closed	<Boolean>
	If closed is true, the figure is a closed figure and can have a fillColor.  It is assumed that the first member of points is the same as the last member of points.

fillColor	<ColorValue | nil>
	If the figure is closed, this color is used to fill the area inside the border.  If the figure is not closed, this variable is not used at all.'!


!PolylineFigure methodsFor: 'testing'!

isClosedFigure
	^closed! !

!PolylineFigure methodsFor: 'accessing'!

borderWidth
	^borderWidth contents!

borderWidth: anInteger 
	borderWidth contents: anInteger!

closed: aBoolean 
	"Set my instance variable closed to aBoolean *only* if the last point in my points collection 
	is the same as the first point, i.e. I'm a closed figure."

	self points first = self points last ifTrue: [closed := aBoolean]!

color
	^color contents!

color: aColorValue 
	color contents: aColorValue!

extent 
	^extent contents!

extent: aPoint
	^extent contents: aPoint!

fillColor
	^fillColor contents!

fillColor: aColorValue
	fillColor contents: aColorValue!

menu
	(metaFigure respondsTo: #menu)
		ifTrue: [^metaFigure menu].
	^PopUpMenu 
		labels: 'Line Width\Line Color\other' withCRs asText allBold 
		values: (Array
			with: (PopUpMenu 
				labels: '1 pixel\2 pixels\3 pixels\4 pixels\5 pixels' withCRs asText allBold
				 values: #(#border1 #border2 #border3 #border4 #border5 ))
			with: (PopUpMenu 
				labels: 'black\very dark gray\dark gray\gray\light gray\very light gray\white' withCRs asText allBold 
				values: #(#lblack #lveryDarkGray #ldarkGray #lgray #llightGray #lveryLightGray #lwhite ))
			with: super menu)!

origin
	^origin contents!

origin: aPoint
	^origin contents: aPoint!

points
	^points collect: [:each | each contents]!

points: aCollection 
	graph isNil ifFalse: [points
			do: 
				[:each | 
				graph removeVariable: each constrainableX.
				graph removeVariable: each constrainableY]].
	points := aCollection collect: [:aPoint | CoolDrawPoint
					newIn: graph
					with: aPoint
					owner: self].
	self recalculateBoundingBox! !

!PolylineFigure methodsFor: 'displaying'!

displayOn: aGraphicsContext 
	aGraphicsContext lineWidth: self borderWidth.
	closed
		ifFalse: 
			[aGraphicsContext paint: self theBorderColor.
			aGraphicsContext displayPolyline: self points]
		ifTrue: 
			[aGraphicsContext paint: self theFillColor.
			aGraphicsContext displayPolygon: self points.
			aGraphicsContext paint: self theBorderColor.
			aGraphicsContext displayPolyline: self points].
	"aGraphicsContext flush.
	aGraphicsContext medium screen sync"!

displayShapeOn: aGraphicsContext at: aPoint 
	"Display a shape on aGraphicsContext that corresponds to my image, a set of lines. If I am 
	a closed figure, then just display a Polygonal region so my inside color shows through."

	| oldWidth pointCollection |
	oldWidth := aGraphicsContext lineWidth. 
	pointCollection := self points collect: [:each | each asPoint.].
	aGraphicsContext lineWidth: self width.
	aGraphicsContext displayPolyline: pointCollection at: aPoint.
	closed ifTrue: [aGraphicsContext displayPolygon: pointCollection at: aPoint].
	aGraphicsContext lineWidth: oldWidth!

majorColorValue
	^closed ifFalse: [self theBorderColor]
		ifTrue: [self theFillColor]!

theBorderColor 
	^ColorTable at: ((color contents max: 1) min: ColorTable size)!

theFillColor
	fillColor contents isNil ifTrue: [^nil].
	^ColorTable at: ((fillColor contents max: 1) min: ColorTable size)! !

!PolylineFigure methodsFor: 'damaging'!

changed
	self recalculateBoundingBox.
	super changed! !

!PolylineFigure methodsFor: 'transforming'!

basicTranslateBy: aPoint 
	"Origin must move also."

	self obsoleteMessage.
	"points do: [:each | each contents: (each contents + aPoint)].
	self origin: (self origin translatedBy: aPoint)"!

rotateBy: anAngle 
	"Rotate the PolylineFigure about its center by the given angle in radians."

	| newX newY center |
	center := self center.
	self willChange.
	points do: [:each |  each contents: [newX := center x + (each contents x - center x * anAngle cos) - 
                                					(each contents y - center y * anAngle sin).
										newY := center y + (each contents y - center y * anAngle cos) + 
										(each contents x - center x * anAngle sin).
										newX @ newY] value].
	self changed!

rounded
	points do: [:each | each contents: (each contents rounded)]!

scaleBy: aPoint 
	self obsoleteMessage.
	"self willChange.
	points do: [:each | each contents: (each contents scaleBy: aPoint)].
	self changed"! !

!PolylineFigure methodsFor: 'private'!

points: aPointCollection width: anInteger color: aColorValue closed: aBoolean fillColor: anotherColorValue graph: aGraph 
	graph := aGraph.
	points := aPointCollection collect: [:aPoint | CoolDrawPoint
					newIn: aGraph with: aPoint owner: self].
	extent := CoolDrawPoint newIn: graph with: 0 @ 0 owner: self.
	origin := CoolDrawPoint newIn: graph with: 0 @ 0 owner: self.
	borderWidth := CoolDrawVariable
				newNamed: 'borderwidth'
				in: aGraph
				with: anInteger.
	color := CoolDrawVariable
				newNamed: 'bordercolor'
				in: aGraph
				with: aColorValue.
	closed := aBoolean.
	fillColor := CoolDrawVariable
				newNamed: 'fillcolor'
				in: aGraph
				with: anotherColorValue.
	borderWidth owner: self.
	color owner: self.
	fillColor owner: self.
	self recalculateBoundingBox!

recalculateBoundingBox 
	"Whenever any of my points change, I need to recalculate
	my origin and extent."

	"Origin is the smallest x and y values I have in my points collection. 
	Don't forget about the line width."

	self origin: (self points inject: self points first into: [:aPoint :minPoint | minPoint min: aPoint])
		- (self borderWidth  + 4.0) rounded.	

	"Extent is the maximum extent as a Point. 
	Don't forget to add some for the width of the line."

	self extent: (self points inject: self points first into: [:aPoint :maxPoint | maxPoint max: aPoint])
		+ (self borderWidth + 4.0) rounded - self origin.! !

!PolylineFigure methodsFor: 'coloring'!

border1

	self constrainValueOf: borderWidth to: 1.!

border2

	self constrainValueOf: borderWidth to: 2.!

border3

	self constrainValueOf: borderWidth to: 3.!

border4

	self constrainValueOf: borderWidth to: 4.!

border5

	self constrainValueOf: borderWidth to: 5.!

lblack
	"Set my border color to black"
	
	self constrainValueOf: color to: (self colorNamed: 'black').
"	self willChange.
	self color: self class black.
	self changed."!

ldarkGray
	"Set my border color to dark gray"

	self constrainValueOf: color to: (self colorNamed: 'dark gray').!

lgray
	"Set my border color to gray"

	self constrainValueOf: color to: (self colorNamed: 'gray').!

llightGray
	"Set my border color to light gray"

	self constrainValueOf: color to: (self colorNamed: 'light gray').!

lveryDarkGray
	"Set my border color to very dark gray" 

	self constrainValueOf: color to: (self colorNamed: 'very dark gray').!

lveryLightGray
	"Set my border color to very light gray"

	self constrainValueOf: color to: (self colorNamed: 'very light gray').!

lwhite
	"Set my border color to white"

	self constrainValueOf: color to: (self colorNamed: 'white').! !

!PolylineFigure methodsFor: 'cooldraw'!

allConstrainedVariables
	| c |
	c := OrderedCollection new.
	points
		do: 
			[:each | 
			c add: each constrainableX.
			c add: each constrainableY].
	c add: origin constrainableX.
	c add: origin constrainableY.
	c add: extent constrainableX.
	c add: extent constrainableY.
	c add: borderWidth.
	c add: fillColor.
	c add: color.
	^c! !

!PolylineFigure methodsFor: 'copying'!

copyUsing: aCopyState
	graph := nil.
	points := points collect: [:pt | aCopyState copy: pt].
	origin := aCopyState copy: origin.
	extent := aCopyState copy: extent.
	borderWidth := aCopyState copy: borderWidth.
	color := aCopyState copy: color.
	"closed"
	fillColor := aCopyState copy: fillColor! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

PolylineFigure class
	instanceVariableNames: ''!


!PolylineFigure class methodsFor: 'instance creation'!

withPoints: aCollection onView: aView
	| me |
	me := self new
		points: aCollection
		width: 1
		color: (self colorNamed: 'black')
		closed: false
		fillColor: nil 
		graph: aView drawing constraintGraph.
	me initialize.
	^me! !


PolylineFigure subclass: #RectangleFigure
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
RectangleFigure comment:
'RectangleFigure is the concrete subclass which represents a Rectangular region.  The four points that make up the RectangleFigure are kept in the points instance variable of my superclass PolylineFigure.  The order of these points is topLeft, topRight, bottomRight, and bottomLeft.  This order is used on occasion in the methods of RectangleFigure.

'!


!RectangleFigure methodsFor: 'accessing'!

handles
        ^(super handles) 
                                add: (SelectionTrackHandle colorOf: self); 
                                add: (SelectionTrackHandle borderColorOf: self); 
                                add: (SelectionTrackHandle widthOf: self); 
                                add: (ConnectionHandle on: self at: #center); 
                                yourself!

introduceTo: aTextFigure at: aPoint
        aTextFigure print: aPoint!

points: aCollection 
	"The rectangle only needs points 1 & 3 (topLeft & bottomRight) to 
	define it, so its points share constrained x & y variables as such: 
	shared x: 	1 & 4		2 & 3 
	shared y: 	1 & 2		3 & 4"
	
	|count| 
	graph isNil ifFalse: [
		count := 1.
		points do: [:each |
			(count = 1) | (count = 3) ifTrue: [ graph removeVariable: each constrainableX; 
											removeVariable: each constrainableY] .
			count := count + 1.]].
	points := Array new: 4.
	points at: 1 put: (CoolDrawPoint
			newIn: graph
			with: (aCollection at: 1)
			owner: self).
	points at: 3 put: (CoolDrawPoint
			newIn: graph
			with: (aCollection at: 3)
			owner: self).
	points at: 2 put: (CoolDrawPoint 
					newFromExisting: (points at: 3) constrainableX 
					and: (points at: 1) constrainableY).
	points at: 4 put: (CoolDrawPoint 
					newFromExisting: (points at: 1) constrainableX 
					and: (points at: 3) constrainableY).
	self recalculateBoundingBox! !

!RectangleFigure methodsFor: 'transforming'!

growBy: aRectangle 
self obsoleteMessage.
	"self willChange.
	self setRectangle: (aRectangle origin + self points first corner: aRectangle corner + (self points at: 3)).
	self changed"!

growNotifying: aView 
	| aHandle |
	aHandle := TrackHandle bottomRightOf: self.
	aView addHandle: aHandle.
	Cursor corner show.
	aHandle invoke: aView.
	Cursor origin show.
	aView removeHandle: aHandle! !

!RectangleFigure methodsFor: 'coloring'!

borderDarkenBy: anInteger 
	| index |
	self willChange.
	index := ColorTable indexOf: color.
	index := (index + (anInteger / 5) truncated min: ColorTable size)
				max: 1.
	self color: (ColorTable at: index)!

borderWidthBy: anInteger 
	self obsoleteMessage.
	"self willChange.
	self width: ((self width + (anInteger // 4) min: 25)
			max: 1)"!

darkenBy: anInteger
	self obsoleteMessage.
"
       | index | 
        index := ColorTable indexOf: fillColor.
        index := (index + ((anInteger / 5) truncated) min: ColorTable size) max: 1. 
        self fillColor: (ColorTable at: index). 
	Transcript show: 'index = ', index printString, 'anInt = ', anInteger printString.
	Transcript cr.
         self changed ."! !

!RectangleFigure methodsFor: 'private'!

rectangle: aRectangle width: anInteger color: aColorValue closed: aBoolean fillColor: anotherColorValue 
	self points: aRectangle asPolyline asOrderedCollection.

	self width: anInteger.
	self color: aColorValue.
	self closed: aBoolean.
	self fillColor: anotherColorValue.
	self recalculateBoundingBox!

setRectangle: aRectangle 
	"Set up my points to hold the proper values for aRectangle in the following order: origin, 
	topRight, corner, bottomLeft, origin."

	self points: aRectangle asPointArray asOrderedCollection! !

!RectangleFigure methodsFor: 'cooldraw'!

allConstrainedVariables
	| c |
	c := OrderedCollection new.
	"Points 2 & 4 share constrainable X and Y with 1 & 3, hence we only return points 1 & 3."
	c add: (points at:1) constrainableX.
	c add: (points at:1) constrainableY.
	c add: (points at:3) constrainableX.
	c add: (points at:3) constrainableY.
	c add: origin constrainableX.
	c add: origin constrainableY.
	c add: extent constrainableX.
	c add: extent constrainableY.
	c add: borderWidth.
	c add: fillColor.
	c add: color.
	^c!

borderColor
	^color contents!

colortable
	^ColorTable!

constrainableBorderColor
	^color!

constrainableBorderWidth
	^borderWidth!

constrainableBottom
	^(points at: 4) constrainableY!

constrainableBottomLeft
	^(points at: 4)!

constrainableBottomRight
	^(points at: 3)!

constrainableFeatures
	^#(#('top' #top ) #('left' #left ) #('right' #right ) #('bottom' #bottom ) #('top left' #topLeft ) #('top right' #topRight ) #('bottom left' #bottomLeft ) #('bottom right' #bottomRight ) #('border width' #borderWidth ) #('border color' #borderColor ) #('fill color' #fillColor ) )!

constrainableFillColor
	^fillColor!

constrainableLeft
	^(points at: 1) constrainableX!

constrainablePosition 
	^points!

constrainableRight
	^(points at: 2) constrainableX!

constrainableTop
	^(points at: 1) constrainableY!

constrainableTopLeft
	^(points at: 1)!

constrainableTopRight
	^(points at: 2)!

fillColor
	^fillColor contents! !

!RectangleFigure methodsFor: 'displaying'!

displayOn: aGraphicsContext 
	"Display the receiver on aGraphicsContext as a PolyLine. Five points are 
	needed to display as a closed rectangle."

	|aPointCollection|
	aPointCollection := self points asOrderedCollection.
	aPointCollection add: (self points at: 1).
	aGraphicsContext lineWidth: self borderWidth.
	"Paint the inside of the rectangle."
	aGraphicsContext paint: self theFillColor.
	aGraphicsContext displayPolygon: aPointCollection.
	"Paint the outline."
	aGraphicsContext paint: self theBorderColor.
	aGraphicsContext displayPolyline: aPointCollection.! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

RectangleFigure class
	instanceVariableNames: ''!


!RectangleFigure class methodsFor: 'instance creation'!

createNotifying: aView 
	| aFigure |
	aFigure := self rectangle: (aView controller sensor cursorPoint extent: 0 @ 0)
					onView: aView.
	"This is a hack to keep the drawing from sending any messages
	to the RectangleFigure when it is added.  This makes it easy to wrap it later.
	HyperCard is a good example of why this is necessary."
	aView drawing add: aFigure.
	aFigure growNotifying: aView.
	"This is a hack.  If we remove the figure by going
	through the view then the figure blinks.  This way
	we fool the view into thinking that the figure has
	not been removed."
	aView drawing remove: aFigure.
	 ^aFigure!

creationTool
	| temp |
	temp := Image
				extent: 16 @ 16
				depth: 1
				palette: MappedPalette blackWhite
				bits: #[255 255 255 255 255 255 255 255 192 3 255 255 192 3 255 255 207 243 255 255 207 243 255 255 207 243 255 255 207 243 255 255 207 243 255 255 207 243 255 255 207 243 255 255 207 243 255 255 192 3 255 255 192 3 255 255 255 255 255 255 255 255 255 255 ].
	^CreationTool
		icon: temp
		cursor: Cursor origin
		class: self!

rectangle: aRectangle onView: aView
	^(self new
		points: (Array with: Point zero)
		width: 1
		color: (self colorNamed: 'black')
		closed: true
		fillColor: (self colorNamed: 'white')
		graph: aView drawing constraintGraph)
		setRectangle: aRectangle! !


PolylineFigure subclass: #LineFigure
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
LineFigure comment:
'A LineFigure is the concrete class which represents a simple, straight line.  The endpoints are kept in the points instance variable defined in the superclass PolylineFigure.  The line width and color are also kept in instance variables.  It is not assumed that the points collection contain Points.  It is assumed that whatever is contained therein responds to the asPoint message with a Point.  This was partly done to generalize a LineFigure into a ConnectionFigure where the points are really Locators.'!


!LineFigure methodsFor: 'accessing'!

handles
	^Array with: (SelectionTrackHandle startPointOf: self)
		with: (SelectionTrackHandle stopPointOf: self)!

oldhandles
	self obsoleteMessage.
	"^Array with: (TrackHandle
			on: self
			at: #startPoint
			change: #moveStartBy:)
		with: (TrackHandle
				on: self
				at: #stopPoint
				change: #moveStopBy:)"!

shiftHandles
        ^self handles!

startPoint

        ^self points first asPoint!

startPoint: aPoint 	
	| pointVals |
	pointVals := self points.
	pointVals at: 1 put: aPoint.
	self points: pointVals!

startY
	^(points at: 1) y!

stopPoint

        ^(self points at: 2) asPoint!

stopPoint: aPoint 
	| pointVals |
	pointVals := self points.
	pointVals at: 2 put: aPoint.
	self points: pointVals!

stopY
	^(points at: 2) y! !

!LineFigure methodsFor: 'testing'!

containsPoint2: aPoint 
	^(self displayBox containsPoint: aPoint)
		and: [self isHorizontalOrVertical or: [(self distanceTo: aPoint)
					< 3]]!

containsPoint: aPoint 
	| temp displayBox |
	temp := (self displayBox containsPoint: aPoint)
		and: [self isHorizontalOrVertical or: [(self distanceTo: aPoint)
					< 3]].
	displayBox := self displayBox.
	^temp.!

distanceTo: aPoint 
	| p q d a b c |
	p := self startPoint.
	q := self stopPoint.
	d := p - q.
	a := 1.
	b := (d x / d y) negated.
	c := (p x + (p x - q x / (q y - p y) * p y)) negated.
	^(aPoint x * a + (aPoint y * b) + c) abs / (a squared + b squared) sqrt!

isHorizontalOrVertical
        | p q |
        p := self startPoint.
        q := self stopPoint.
        ^p x = q x | p y = q y! !

!LineFigure methodsFor: 'transforming'!

moveStartBy: deltaPoint 
	self obsoleteMessage.
	"self willChange.
	self startPoint: self startPoint + deltaPoint.
	self changed"!

moveStopBy: deltaPoint 
	self obsoleteMessage.
	"self willChange.
	self stopPoint: self stopPoint + deltaPoint.
	self changed"! !

!LineFigure methodsFor: 'cooldraw'!

constrainableBorderWidth
	^borderWidth!

constrainableColor
	^color!

constrainableFeatures
	^#(#('start y' #startY ) #('stop y' #stopY ) #('start x' #startX ) #('stop x' #stopX ) #('start' #startPoint ) #('stop' #stopPoint ) #('line width' #borderWidth) #('line color' #color))!

constrainablePosition
	^Array with: (points at: 1) with: (points at: 2)!

constrainableStartPoint
	^points at: 1!

constrainableStartX
	^(points at: 1) constrainableX!

constrainableStartY
	^(points at: 1) constrainableY!

constrainableStopPoint
	^points at: 2!

constrainableStopX
	^(points at: 2) constrainableX!

constrainableStopY
	^(points at: 2) constrainableY! !

!LineFigure methodsFor: 'displaying'!

majorColorValue
	^self borderWidth < 3 ifTrue: [ColorValue white] ifFalse: [super majorColorValue]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

LineFigure class
	instanceVariableNames: ''!


!LineFigure class methodsFor: 'instance creation'!

createNotifying: aView 
	| startPoint stopPoint aSensor aLineFigure gc |
	aSensor := aView controller sensor.
	startPoint := aSensor cursorPoint.
	stopPoint := aSensor cursorPoint.
	gc := aView graphicsContext.
	gc displayLineFrom: startPoint to: stopPoint.
	[aSensor redButtonPressed]
		whileTrue: 
			[aView drawing damageRegion: ((Rectangle vertex: startPoint vertex: stopPoint)
					expandedBy: 2 @ 2).
			aView repairDamage.
			stopPoint := aSensor cursorPoint.
			gc displayLineFrom: startPoint to: stopPoint].
	startPoint = stopPoint
		ifFalse: 
			[aLineFigure := self start: startPoint stop: stopPoint onView: aView.
			^aLineFigure].
	^nil!

creationTool
	| temp |
	temp := Image
				extent: 16 @ 16
				depth: 1
				palette: MappedPalette blackWhite
				bits: #[255 255 255 255 255 249 255 255 255 243 255 255 255 231 255 255 255 207 255 255 255 159 255 255 255 63 255 255 254 127 255 255 252 255 255 255 249 255 255 255 243 255 255 255 231 255 255 255 207 255 255 255 159 255 255 255 255 255 255 255 255 255 255 255 ].
	^CreationTool
		icon: temp
		cursor: Cursor crossHair
		class: self!

start: start stop: stop onView: aView
	^(self withPoints: (Array with: start with: stop) onView: aView)
		borderWidth: 1!

startLocation: startLocation stopLocation: stopLocation onView: aView
	| aFigure |
	aFigure := self start: startLocation value stop: stopLocation value onView: aView.
	startLocation object addDependent: (PositionConstraint new location: startLocation receiver: aFigure sending: #startPoint:).
	stopLocation object addDependent: (PositionConstraint new location: stopLocation receiver: aFigure sending: #stopPoint:). 
	^aFigure! !


LineFigure subclass: #ArrowFigure
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
ArrowFigure comment:
'I''m like a LineFigure except that I have an arrow-head at my ending point.'!


!ArrowFigure methodsFor: 'accessing'!

connectionPosition
        ^ self startPoint!

locator
        ^Locator on: self at: #startPoint! !

!ArrowFigure methodsFor: 'private'!

changed
	"self updateArrow."
	super changed!

createArrowHeadConstraint
	"Set up a constraint to maintain the receivers arrowHead"

	self
		arrowHeadConstraint: (points at: 1)
		and: (points at: 2)
		and: borderWidth
		and: (points at: 3)
		and: (points at: 5)
		with: (points at: 4)
		in: graph!

updateArrow
	"One of the end points has changed, so recalculate the other points."
	self obsoleteMessage.

	"| vector u aPoint stopPoint pointVals |
	pointVals := self points.
	stopPoint := self stopPoint.
	vector := stopPoint - self startPoint.
	vector = 0
		ifTrue: 
			[u := 0.
			aPoint := stopPoint]
		ifFalse: 
			[u := vector normal unitVector.
			aPoint := stopPoint - (vector unitVector * self width * 8)].
	pointVals at: 3 put: aPoint + (u * (self width * 4)).
	pointVals at: 5 put: aPoint - (u * (self width * 4)).
	pointVals at: 4 put: stopPoint.
	self points: pointVals.
	self recalculateBoundingBox."! !

!ArrowFigure methodsFor: 'cooldraw'!

arrowHeadConstraint: start and: stop and: width and: arrowLeft and: arrowRight with: arrowCenter in: aGraph 
	| cnst meth tmpCollection|
	cnst := SkyBlueConstraint new.
	cnst name: 'arrowHead'.
	self ian: 'Rename constraint to something more meaningful'.
	cnst strength: SkyBlueStrength required.
	tmpCollection := OrderedCollection new. 
	tmpCollection	add: start constrainableX; add: start constrainableY;
					add: stop constrainableX; add: stop constrainableY;
					add: arrowLeft constrainableX; add: arrowLeft constrainableY;
					add: arrowRight constrainableX; add: arrowRight constrainableY;
					add: arrowCenter constrainableX; add: arrowCenter constrainableY;
					add: width.
	cnst variables: tmpCollection asArray.
	cnst errorfunction: [:vars | self ian: 'implement this'. 0].
	meth := SkyBlueMethod new.
	tmpCollection := OrderedCollection new.
	tmpCollection	add: start constrainableX; add: start constrainableY;
					add: stop constrainableX; add: stop constrainableY;
					add: width.
	meth inputs: tmpCollection asArray.
	tmpCollection := OrderedCollection new.
 	tmpCollection	add: arrowLeft constrainableX; add: arrowLeft constrainableY;
					add: arrowRight constrainableX; add: arrowRight constrainableY;
					add: arrowCenter constrainableX; add: arrowCenter constrainableY.
	meth outputs: tmpCollection asArray.
	meth external: false.
	meth block: [:in :out | | u v aPoint | 
		v := (in at: 3) contents - (in at: 1) contents @ 
							((in at: 4) contents - (in at: 2) contents).
		v = 0
			ifTrue: [ u := 0. aPoint := (in at: 3) contents @ (in at: 4) contents]
			ifFalse: [ u := v normal unitVector.
					aPoint := (in at: 3) contents @ (in at: 4) contents -
								(v unitVector * (in at: 5) contents * 8)].
		(out at: 1) contents: (aPoint + (u * (in at: 5) contents * 4)) x.
		(out at: 2) contents: (aPoint + (u * (in at: 5) contents * 4 )) y.
		(out at: 3) contents: (aPoint - (u * (in at: 5) contents * 4)) x.
		(out at: 4) contents: (aPoint - (u * (in at: 5) contents * 4)) y.
		(out at: 5) contents: (in at: 3) contents.
		(out at: 6) contents: (in at: 4) contents].
	cnst methods: (Array with: meth).
	aGraph addConstraint: cnst! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ArrowFigure class
	instanceVariableNames: ''!


!ArrowFigure class methodsFor: 'instance creation'!

creationTool
        ^ CreationTool
                icon: (Image
        extent: 16@16
	  depth: 1
	  palette: MappedPalette blackWhite
        bits: #(255 255 255 255 255 1 255 255 255 1 255 255 255 225 255 255 255 201 255 255 255 153 255 255 255 57 255 255 254 121 255 255 252 249 255 255 249 255 255 255 243 255 255 255 231 255 255 255 207 255 255 255 159 255 255 255 255 255 255 255 255 255 255 255 ))
                cursor: Cursor crossHair
                class: self!

start: start stop: stop 
	"| anArrow anArray |"
	self obsoleteMessage.
	"anArray := Array new: 5.
	anArray at: 1 put: start.
	anArray at: 2 put: stop.
	anArray at: 3 put: 0@0.
	anArray at: 4 put: stop.
	anArray at: 5 put: 0@0.
	anArrow := (self withPoints: anArray) width: 1.
	anArrow updateArrow.
	^anArrow"!

start: start stop: stop onView: aView
	| anArrow anArray |
	anArray := Array new: 5.
	anArray at: 1 put: start.
	anArray at: 2 put: stop.
	anArray at: 3 put: 0@0.
	anArray at: 4 put: stop.
	anArray at: 5 put: 0@0.
	anArrow := (self withPoints: anArray onView: aView) borderWidth: 1.
	anArrow createArrowHeadConstraint.
	^anArrow! !


LineFigure subclass: #DependentLineFigure
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
DependentLineFigure comment:
'A DependentLineFigure is a LineFigure that can only be moved by constraints.'!


!DependentLineFigure methodsFor: 'transforming'!

basicTranslateBy: aPoint 
	"Ignore this message - the only way to move me is to move my end points"! !


Figure subclass: #TextFigure
	instanceVariableNames: 'origin grid paragraph margin opaque readOnly '
	classVariableNames: 'InsertionPoint '
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
TextFigure comment:
'**
	Modified: March 1993
	By: Ian Perrigo
	Comments: Made the origin a SkyBluePoint
**
A TextFigure is a Figure that permits the displaying of Text in a Drawing.  TextFigures offer many of the standard Text operations.

Instance Variables :
origin		<Point>			The starting Point to display the Text in the Drawing coordinates.
paragraph	<ComposedText>	The actual Text holder.  ComposedText is used because it offers the display capabilities needed.
margin		<Integer | nil>		The position of the right margin.  Initially this is way out to the right but gets reset to something more reasonable when Text entry is complete.
grid			<???>			Soon to become fontSize.  Real soon now.
opaque		<Boolean>		If opaque is true, obscure Figures underneath me when displaying myself.

Class Variables :
InsertionPoint	<Image>			The Image that gets displayed as the insertion point during Text entry.  Cached to make things a bit faster.

'!


!TextFigure methodsFor: 'accessing'!

copyFrom: start to: stop 
	^paragraph asText copyFrom: start to: stop!

extent
	"Answer the length and width of my image."

	^paragraph width @ paragraph height + (4 @ 0)!

handles
	self obsoleteMessage.
	"^Array with: (TrackHandle
			on: self
			at: #marginHandleLocation
			using: #constrainableMargin
			constraining: #mouseConstraint:yDif4isColor:)
		with: (TrackHandle
				on: self
				at: #heightHandleLocation
				using: #constrainableFont
				constraining: #mouseConstraint:yDiv4isColor:)"!

heightHandleLocation

        ^self center x @ (self origin y + self preferedHeight)!

indexForPoint: aPoint 
	"Answer the index into my text that most closely corresponds to aPoint."

	^(paragraph characterBlockAtPoint: aPoint - self origin) stringIndex!

margin
        ^margin contents notNil
                ifTrue: [margin contents rounded]
                ifFalse: [self margin: paragraph composeAll]!

margin: aNumber 
	margin contents: aNumber.
	paragraph compositionWidth: margin contents rounded.
	^margin contents!

marginHandleLocation

        ^origin x + self margin + 5 @ self center y!

menu
	(metaFigure respondsTo: #menu)
		ifTrue: [^metaFigure menu].
	^PopUpMenu
		labels: 'justification\visibility\other' withCRs
		lines: #(2 )
		values: (Array
				with: (PopUpMenu labels: 'flush left\centered\flush right\justified' withCRs values: #(#leftFlush #centered #rightFlush #justified ))
				with: (PopUpMenu labels: 'transparent\opaque' withCRs values: #(#transparent #opaque ))
				with: super menu)!

oldhandles
self obsoleteMessage.
	^Array with: (TrackHandle
			on: self
			at: #marginHandleLocation
			sense: #senseMargin:
			change: #changeMarginBy:)
		with: (TrackHandle
				on: self
				at: #heightHandleLocation
				change: #changeFontBy:)!

origin

        ^origin contents!

paragraph

        ^paragraph!

print: anObject

        self string: anObject printString!

readOnly: aBoolean
	readOnly := aBoolean!

replaceFrom: start to: stop with: aText notifying: aView 
	aView clearObject: self while: [paragraph
			replaceFrom: start
			to: stop
			with: aText].
	self margin: (paragraph rightMarginForComposition).
	self changed!

string

        ^paragraph asText asString!

string: aString

        self text: aString asText!

text: aText 
	self willChange.
	paragraph
		replaceFrom: 1
		to: paragraph text size
		with: aText.
	self changed! !

!TextFigure methodsFor: 'testing'!

acceptsTyping

        ^readOnly not!

isEmpty
	^paragraph text isEmpty!

isOpaque

        ^opaque! !

!TextFigure methodsFor: 'sensing'!

senseMargin: deltaPoint

        ^deltaPoint x! !

!TextFigure methodsFor: 'transforming'!

basicTranslateBy: aPoint 
	self obsoleteMessage.
	"origin contents: origin contents + aPoint"!

changeFontBy: aPoint 
	"Change my font up if aPoint is positive or down if aPoint is negative. Scale aPoint so that 
	the control isn't too sensitive."

	| styles grid |
	grid := aPoint y / 2 + self preferedLineGrid.
	styles := TextAttributes styles asOrderedCollection select: [:each | (TextAttributes styleNamed: each) lineGrid <= grid].
	styles isEmpty ifTrue: [^self].
	self textStyle: (TextAttributes styleNamed: styles last)!

changeMarginBy: anInteger 
	self willChange.
	self margin: (self margin + anInteger < 12  "Keep it from getting too small"
			ifTrue: [self margin]
			ifFalse: [self margin + anInteger]).
	self changed!

growBy: aPoint 
        "TextFigures can't grow."!

opaque
	self isOpaque
		ifFalse: 
			[opaque := true.
			self changed]!

recompose
	"Recompose my paragraph, resetting the width and possibly the right margin."

	self margin: paragraph composeAll!

scaleBy: aPoint 
	self obsoleteMessage.
	"self willChange.
	self origin: (origin contents scaleBy: aPoint).
	self margin isNil
		ifFalse: 
			[self margin contents: self  margin * aPoint x.
			paragraph compositionWidth: self margin rounded].
	self changed"!

textStyle: aTextStyle 
	self willChange.
	aTextStyle alignment = paragraph textStyle alignment
		ifTrue: [paragraph textStyle: aTextStyle]
		ifFalse: [paragraph textStyle: ((aTextStyle copy) alignment: paragraph textStyle alignment; yourself)].
	self changed!

transparent
	self isOpaque
		ifTrue: 
			[opaque := false.
			self changed]! !

!TextFigure methodsFor: 'aligning'!

centered
	self willChange.
	self margin: self margin.
	paragraph textStyle: paragraph textStyle copy.
	paragraph centered.
	self changed!

justified
	self willChange.
	self margin: self margin.
	paragraph textStyle: paragraph textStyle copy.
	paragraph justified.
	self changed!

leftFlush
	self willChange.
	self margin: self margin.
	paragraph textStyle: paragraph textStyle copy.
	paragraph leftFlush.
	self changed!

rightFlush
	self willChange.
	self margin: self margin.
	paragraph textStyle: paragraph textStyle copy.
	paragraph rightFlush.
	self changed! !

!TextFigure methodsFor: 'displaying'!

displayCaretAtIndex: anInteger on: aGraphicsContext 
	"Show a caret inbetween characters anInteger and anInteger+1. Actually, reverse the 
	image of a caret at this point. Thus we can toggle the caret on and off by calling this method 
	repeatedly."

	| ipExtent aPoint aBlock |
	ipExtent := InsertionPoint extent.
	aBlock := self paragraph characterBlockForIndex: anInteger.
	aPoint := aBlock left - (ipExtent x // 2) @ (aBlock top + self paragraph textStyle baseline) extent: ipExtent.
	InsertionPoint displayOn: aGraphicsContext at: self origin + aPoint origin!

displayOn: aGC 
	"If I am opaque, draw a box in the background color obscuring those figures beneath me."

	self isOpaque ifTrue: [aGC paint: Window currentWindow background; displayRectangle: (self boundingBox translateBy: self origin)]. 
	aGC paint: ColorValue black; display: paragraph at: self origin!

displayShapeOn: aGC at: aPoint 
	"Display my shape on aGraphicsContext. If I am opaque, just display a rectangular region 
	so that my background shows through."

	self isOpaque
		ifTrue: [aGC displayRectangle: (aPoint + self origin extent: self extent)]
		ifFalse: [aGC display: paragraph at: aPoint + self origin]!

highlightFrom: index1 to: index2 on: gc foreground: fColor background: bColor 
	"Display the image of my text from index1 to index2. Note they may not be in the proper 
	order. Use bColor for the background and fColor for the foreground."

	| start stop lineRange first last |
	index1 = index2 ifTrue: [^self].
	index1 < index2
		ifTrue: 
			[start := paragraph characterBlockForIndex: index1.
			stop := paragraph characterBlockForIndex: index2]
		ifFalse: 
			[start := paragraph characterBlockForIndex: index2.
			stop := paragraph characterBlockForIndex: index1].
	gc paint: bColor.
	start top = stop top
		ifTrue: 
			[gc displayRectangle: (start origin + self origin corner: stop bottomLeft + self origin).
			gc paint: fColor.
			^paragraph
				displayFromCharacter: start stringIndex
				to: stop stringIndex - 1
				startX: start left + self origin x
				forTranslation: self origin
				on: gc].
	lineRange := paragraph lineRangeFor: (start origin corner: stop corner).
	first := (paragraph lineAt: lineRange first) first.
	last := (paragraph lineAt: lineRange last) last.
	start stringIndex > first ifTrue: [first := start stringIndex].
	stop stringIndex - 1 < last ifTrue: [last := stop stringIndex - 1].
	gc displayRectangle: (start origin + self origin corner: paragraph right @ start bottom + self origin).
	gc displayRectangle: (paragraph left @ start bottom + self origin corner: paragraph right @ stop top + origin).
	gc displayRectangle: (paragraph left @ stop top + self origin corner: stop bottomLeft + self origin).
	gc paint: fColor.
	paragraph
		displayFromCharacter: first
		to: last
		startX: start left + origin x
		forTranslation: self origin
		on: gc! !

!TextFigure methodsFor: 'copying'!

copy
	| aComposedText newFigure |
	aComposedText := paragraph text deepCopy asComposedText.
	aComposedText textStyle: paragraph textStyle.
	aComposedText setHeight: paragraph height.
	aComposedText compositionWidth: paragraph width.
	newFigure := self class paragraph: aComposedText at: self origin.
	newFigure readOnly: self acceptsTyping not.
	self margin notNil ifTrue: [newFigure margin: self margin].
	^newFigure! !

!TextFigure methodsFor: 'private'!

preferedHeight

        ^self preferedLineGrid * paragraph numberOfLines!

preferedLineGrid
        ^paragraph textStyle lineGrid!

setParagraph: aParagraph origin: aPoint 
	"Reciever initialization - doesn't require the sender to know the view since the receiver
	will get the SkyBlueConstraint graph from its dependent Drawing."
	| aGraph|
	aGraph := self dependentDrawing constraintGraph.
	origin := CoolDrawPoint newIn: aGraph with: aPoint owner: self.
	grid := CoolDrawVariable newIn: aGraph with: aPoint owner: self.
	paragraph := aParagraph.
	opaque := false.
	margin := CoolDrawVariable newIn: aGraph with: nil.
	margin owner: self.
	paragraph compositionWidth: ComposedText defaultCompositionWidth.
	readOnly := false.!

setParagraph: aParagraph origin: aPoint onView: aView 
	| aGraph|
	aGraph := aView drawing constraintGraph.
	origin := CoolDrawPoint newIn: aGraph with: aPoint owner: self.
	grid := CoolDrawActiveVariable newIn: aGraph with: aPoint 
			performBlock: self gridPerformBlock.
	grid owner: self.
	paragraph := aParagraph.
	opaque := false.
	margin := CoolDrawVariable newIn: aGraph with: nil.
	margin owner: self.
	paragraph compositionWidth: ComposedText defaultCompositionWidth.
	readOnly := false.! !

!TextFigure methodsFor: 'cooldraw'!

constrainableFont!

constrainableMargin! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

TextFigure class
	instanceVariableNames: ''!


!TextFigure class methodsFor: 'initialization'!

initialize
	"TextFigure initialize"
	InsertionPoint := Image extent:7@4 depth: 1 palette: CoveragePalette monoMaskPalette bits: #[
2r0001000  
2r0011100  
2r0111110  
2r0111110  
] pad: 8! !

!TextFigure class methodsFor: 'instance creation'!

createNotifying: aView 
	| aFigure aPoint aParagraph |
	aPoint := aView sensor waitButton.
	aParagraph := ComposedText withText: Text new.
	"adjust aPoint to correct for the hotspot of the cursor"
	aFigure := self paragraph: aParagraph at: aPoint - (0 @ 8) onView: aView.	
	^aFigure!

creationTool
	| aCursor temp |
	aCursor := Cursor
				fromArray: #(0 1728 256 256 256 256 256 256 256 256 256 256 256 256 1728 0 )
			hotSpot: 8 @ 8
			name: ''.
	temp := Image
				extent: 16 @ 16
				depth: 1
				palette: MappedPalette blackWhite
				bits: #[255 255 255 255 192 3 255 255 192 3 255 255 222 123 255 255 254 127 255 255 254 127 255 255 254 127 255 255 254 127 255 255 254 127 255 255 254 127 255 255 254 127 255 255 254 127 255 255 254 127 255 255 254 127 255 255 252 63 255 255 255 255 255 255 ].
	^TextTool
		icon: temp
		cursor: aCursor
		class: self!

paragraph: aParagraph at: aPoint onView: aView

        ^self new setParagraph: aParagraph origin: aPoint onView: aView!

string: aString

        ^self string: aString at: 0@0!

string: aString at: aPoint 
	^(self paragraph: (ComposedText withText: aString asText)
		at: aPoint) recompose! !


TextFigure subclass: #FixedTextFigure
	instanceVariableNames: 'length '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
FixedTextFigure comment:
'a FixedTextFigure is just like any other TextFigure except the right margin is limited to length pixels.  This prohibits the user from causing the text to wrap around to the next line.  '!


!FixedTextFigure methodsFor: 'accessing'!

maxLength
	^length!

maxLength: anInteger
	length := anInteger.
	self recompose.!

replaceFrom: start to: stop with: aText notifying: aView 
	aView clearObject: self while: [paragraph
			replaceFrom: start
			to: stop
			with: aText].! !

!FixedTextFigure methodsFor: 'transforming'!

changeMarginBy: anInteger 
	"SingleLineTextFigures can't change their margins."!

recompose
	self margin: length! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

FixedTextFigure class
	instanceVariableNames: ''!


!FixedTextFigure class methodsFor: 'instance creation'!

string: aString at: aPoint 
	| aFTF |
	aFTF := self paragraph: (ComposedText withText: aString asText)
				at: aPoint.
	aFTF paragraph wordWrap: false.
	^aFTF! !


FixedTextFigure subclass: #NumberFigure
	instanceVariableNames: 'number '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
NumberFigure comment:
'I''m a TextFigure that holds numbers, and that supports protocol
"number" and "number:" to fetch and set this number.  I can be
edited by the user just like any other TextFigure.'!


!NumberFigure methodsFor: 'accessing'!

number
	^number contents!

number: aNumber
	"Ian - note that the instance var number (now an SKBVar) is not set here, this seems
	strange, I think it's a bug"
	number contents == aNumber ifTrue: [^nil]. 
	self string: aNumber printString.!

replaceFrom: start to: stop with: aText notifying: aView 
	"number := aText asNumber."
	self number: aText asNumber.
	super replaceFrom: start to: stop with: aText notifying: aView.!

text: aText 
	"number := aText asNumber."
	self number: aText asNumber.
	"In addition to everything else, this will damage me."
	super text: aText.! !

!NumberFigure methodsFor: 'private'!

initializeOnView: aView
	"Initialize the receivers SkyBlueVariable"

	number := CoolDrawVariable newIn: aView drawing  constraintGraph.
	number owner: self.
	^self! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

NumberFigure class
	instanceVariableNames: ''!


!NumberFigure class methodsFor: 'instance creation'!

createNotifying: aView 
	| aFigure aPoint aParagraph |
	aPoint := aView sensor waitButton.
	aParagraph := ComposedText withText: '0' asText.
	"Adjust aPoint to correct for the hotspot of the cursor"
	aFigure := self paragraph: aParagraph at: aPoint - (0 @ 8) onView: aView.
	aFigure initializeOnView: aView.	
	aFigure number: 0.
	^aFigure! !


Figure subclass: #CompositeFigure
	instanceVariableNames: 'figures visibleArea visibleFigures showVisibleArea '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
CompositeFigure comment:
'CompositeFigure is an abstract subclass that permits composed figures.  A CompositeFigure is like a GroupFigure in that it contains component figures, but it is different in that the components are still accessible.  Furthermore, a CompositeFigure has a bounding region (called the visibleArea).  Only the components in this area are visible.  The components outside the visibleArea are not drawn.  This permits active components that are not visible (or accessible) to the casual user.

When selected, a CompositeFigure offers handles on the corners of its visibleArea.  If this area is contracted so that it touches or covers a component figure, the figure is no longer displayed.  This is how active component figures are masked from view.  The component figures can be edited (including moved) by shift-selecting them.  That is, while the CompositeFigure is not selected, hold the left shift key and select the component figure.  The handles for the figure will be offered and the figure can be moved or edited in the normal fashion.  Note that if you move the figure (actually its bounding box) outside of the visibleArea of the CompositeFigure, the component figure will disappear.  You may still move it further but once released, the figure becomes an invisible component of the CompositeFigure.

Several visibleArea operations are currently supported via the yellow button menu.  You can show or hide the visibleArea which appears as a solid black rectangular outline.  You can also resize the visibleArea to include all the component figures.  This is valuable in development if you accidently move a component figure outside the visibleArea and release the mouse button.  It is expected that subclasses of CompositeFigure not necessarily permit users to manipulate the visibleArea in such a casual way unless it is important to the operation of the figure.

Concrete subclasses of CompositeFigure need not implement any new methods.  However, it is expected that certain behaviour would want to be changed.  For example, it is likely that subclasses will inhibit the ability to change the visibleArea of a CompositeFigure.  Other behaviour changes are also likely.'!


!CompositeFigure methodsFor: 'private'!

resetVisibleFigures
	visibleFigures := figures select: [:each | visibleArea contains: each displayBox]!

setFigures: aCollection visibleArea: aRectangle
	figures := aCollection.
	visibleArea := aRectangle.
	self resetVisibleFigures.
	figures do: [:each | each container: self. each addDependent: self].
	showVisibleArea := false.! !

!CompositeFigure methodsFor: 'testing'!

containsPoint: aPoint 
	^WindowSensor new shiftDown not and: [visibleArea containsPoint: aPoint]!

showingVisibleArea
	"Answer whether I am currently showing my visibleArea."

	^showVisibleArea! !

!CompositeFigure methodsFor: 'accessing'!

extent
	^(visibleArea extent + 0.5) rounded!

figureAt: aPoint 
	| aFigure |
	^WindowSensor new shiftDown
		ifTrue: 
			[visibleFigures do: 
				[:each | 
				aFigure := each figureAt: aPoint.
				aFigure notNil ifTrue: [^aFigure]].
			nil]
		ifFalse: [super figureAt: aPoint]!

figures
	^figures!

menu
	(metaFigure respondsTo: #menu)
		ifTrue: [^metaFigure menu].
	^PopUpMenu
		labels: 'cut\copy\paste\show visible area\hide visible area\reset visible area' withCRs
		lines: #(3 5 )
		values: #(#cut #copy #paste #showVisibleAreaIndicator #hideVisibleAreaIndicator #resetVisibleArea )!

origin
	^(visibleArea origin - 0.5) rounded!

visibleArea
	^visibleArea!

visibleArea: aRectangle
	visibleArea := aRectangle.!

visibleFigures
	^visibleFigures! !

!CompositeFigure methodsFor: 'displaying'!

displayOn: aGraphicsContext 
	"Draw each of my figures that fall entirely within my visible region on aGraphicsContext."

	visibleFigures reverseDo: [:each | each displayOn: aGraphicsContext].
	showVisibleArea ifTrue: [aGraphicsContext displayRectangularBorder: visibleArea]! !

!CompositeFigure methodsFor: 'damage control'!

damageRegion: aRegion
	"Since I don't keep a cache, I can ignore this."
	^container notNil ifTrue: [container damageRegion: aRegion]!

update: aFigure 
	"aFigure is one of my figures and it has changed in some way (likely it moved). This 
	message does *not* get sent when I get changed, though"

	self resetVisibleFigures.
	self changed.! !

!CompositeFigure methodsFor: 'transforming'!

basicTranslateBy: aPoint 
	self obsoleteMessage.
	"visibleArea := visibleArea translatedBy: aPoint.
	figures do: [:each | each translateBy: aPoint]"!

growBy: aRectangle 
	"Change the size of my visibleArea (not less than a reasonable minimum) and reset the 
	visible figures within it."

	(visibleArea origin < aRectangle origin negated or: [visibleArea extent < aRectangle extent negated])
		ifTrue: 
			[^self].
	self willChange.
	visibleArea := visibleArea origin + aRectangle origin extent: visibleArea extent + aRectangle extent.
	self resetVisibleFigures.	"maybe we made more figures visible/invisible."
	self changed!

hideVisibleAreaIndicator
	showVisibleArea := false.
	self changed!

resetVisibleArea
	self willChange.
	self visibleArea: (figures inject: figures first displayBox into: [:rect :each | rect merge: each displayBox]).
	self resetVisibleFigures.
	self changed!

showVisibleAreaIndicator
	showVisibleArea := true.
	self changed! !

!CompositeFigure methodsFor: 'copying'!

copy
	| aFigure |
	aFigure := super copy.
	aFigure setFigures: (figures collect: [:each | each copy])
		visibleArea: visibleArea copy.
	showVisibleArea ifTrue: [aFigure showVisibleAreaIndicator].
	^aFigure! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

CompositeFigure class
	instanceVariableNames: ''!


!CompositeFigure class methodsFor: 'instance creation'!

figures: aCollection 
	"Answer a new ComposedFigure with each figure in aCollection as a visible figure of the 
	composition. Set the visibleArea to cover all the figures."

	| aRectangle |
	aRectangle := aCollection inject: aCollection first displayBox into: [:sum :each | sum merge: each displayBox].
	^self visibleArea: aRectangle figures: aCollection!

visibleArea: aRectangle figures: aCollection
	^self new setFigures: aCollection visibleArea: aRectangle! !


Figure subclass: #EllipseFigure
	instanceVariableNames: 'ellipse borderWidth color fillColor center extent '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Figures'!
EllipseFigure comment:
'**
	Modified: March 1993
	By: Ian Perrigo
	Comments: Made SkyBlue constrainable
**
An EllipseFigure represents an ellipse.  The instance variable "elipse"
is its bounding box, i.e. it is a rectangle whose hight is the height of the
ellipse and whose width is the width of the elipse.

Other instance variables:
	width <Integer> the width of the border
	color <ColorValue> the color of the border
	fillColor <ColorValue> the color of the interior'!


!EllipseFigure methodsFor: 'accessing'!

borderWidth
	^borderWidth contents!

center
	"^ellipse center"
	^center contents!

displayBox
	^Rectangle origin: self origin - (0.5 + self borderWidth // 2) extent: self extent + self borderWidth!

extent
	"^ellipse extent rounded"
	^extent contents!

handles
	^(super handles)
		add: (SelectionTrackHandle colorOf: self);
		add: (SelectionTrackHandle borderColorOf: self);
		add: (SelectionTrackHandle widthOf: self);
		add: (ConnectionHandle on: self at: #center);
		yourself!

introduceTo: aTextFigure at: aPoint
	aTextFigure print: aPoint!

origin
	^ellipse origin! !

!EllipseFigure methodsFor: 'copying'!

copyUsing: aCopyState 
	graph := nil.
	borderWidth := aCopyState copy: borderWidth.
	color := aCopyState copy: color.
	fillColor := aCopyState copy: fillColor.
	center := aCopyState copy: center.
	extent := aCopyState copy: extent.
	ellipse := aCopyState copy: ellipse! !

!EllipseFigure methodsFor: 'coloring'!

borderColor: aColor
	self obsoleteMessage.
	"color contents: aColor.
	self changed"!

borderDarkenBy: anInteger
	self obsoleteMessage.
	"| index |
	index := ColorTable indexOf: color contents.
	index := (index + (anInteger / 5) truncated min: ColorTable size) max: 1.
	color contents: (ColorTable at: index).
	self changed"!

borderWidth: anInteger
	self obsoleteMessage.
	"self willChange.
	borderWidth contents: anInteger.
	self changed"!

borderWidthBy: anInteger
	self obsoleteMessage.
	"self borderWidth: ((self width + (anInteger // 4) min: 25) max: 1)"!

darkenBy: anInteger
	| index | 
	self obsoleteMessage.
	"index := ColorTable indexOf: fillColor contents.
	index := (index + ((anInteger / 5) truncated) min: ColorTable size) max: 1.
	fillColor contents: (ColorTable at: index).
	self changed"! !

!EllipseFigure methodsFor: 'transforming'!

basicTranslateBy: aPoint
	self obsoleteMessage.
	"ellipse := ellipse translatedBy: aPoint"!

growBy: aRectangle 
	self willChange.
	ellipse origin: ellipse origin + aRectangle origin.
	ellipse extent: ellipse extent + aRectangle extent.
	self changed!

growNotifying: aView
	| aHandle |
	aHandle := TrackHandle bottomRightOf: self.
	aView addFigure: self.
	aView addHandle: aHandle.
	aHandle invoke: aView.
"Remove figure because it is going to be added later."
	aView drawing remove: self.
	aView removeHandle: aHandle!

scaleBy: aPoint
	self obsoleteMessage.
	"self willChange.
	ellipse := ellipse scaleBy: aPoint.
	self changed"! !

!EllipseFigure methodsFor: 'displaying'!

displayOn: aGraphicsContext 
	fillColor contents isNil
		ifFalse: 
			[aGraphicsContext paint: self theFillColor.
			aGraphicsContext
				displayWedgeBoundedBy: ellipse asRectangle
				startAngle: 0
				sweepAngle: 360].
	self borderWidth > 0
		ifTrue: 
			[aGraphicsContext lineWidth: self borderWidth.
			aGraphicsContext paint: self theBorderColor.
			aGraphicsContext
				displayArcBoundedBy: ellipse asRectangle
				startAngle: 0
				sweepAngle: 360]!

displayShapeOn: aGraphicsContext at: aPoint 
        "Display a shape on aGraphicsContext that corresponds 
        to my image. First draw the shape outline, then the wedges (to 
        allow the fillColor to show up)."

        | oldWidth |
        oldWidth := aGraphicsContext lineWidth.
        aGraphicsContext lineWidth: self borderWidth.
        aGraphicsContext
                displayArcBoundedBy: (ellipse asRectangle translateBy: aPoint)
                startAngle: 0
                sweepAngle: 360.
        aGraphicsContext
                displayWedgeBoundedBy: (ellipse asRectangle translateBy: aPoint)
                startAngle: 0
                sweepAngle: 360.
        aGraphicsContext lineWidth: oldWidth!

theBorderColor
	^ColorTable at: ((color contents max: 1) min: ColorTable size)!

theFillColor
	fillColor contents isNil ifTrue: [^nil].
	^ColorTable at: ((fillColor contents max: 1) min: ColorTable size)! !

!EllipseFigure methodsFor: 'cooldraw'!

allConstrainedVariables
	| c |
	c := OrderedCollection new.
	c add: ellipse constrainableOrigin constrainableX.
	c add: ellipse constrainableOrigin constrainableY.
	c add: ellipse constrainableCorner constrainableX.
	c add: ellipse constrainableCorner constrainableY.
	c add: center constrainableX.
	c add: center constrainableY.
	c add: extent constrainableX.
	c add: extent constrainableY.
	c add: borderWidth.
	c add: fillColor.
	c add: color.
	^c!

borderColor
	^color contents!

centerx
	^self center x!

centery
	^self center y!

colortable
	^ColorTable!

constrainableBorderColor
	^color!

constrainableBorderWidth
	^borderWidth!

constrainableBottom
	^ellipse constrainableCorner constrainableY!

constrainableBottomLeft
	| a b |
	a := ellipse constrainableOrigin.
	b := ellipse constrainableCorner.
	^CoolDrawPoint newFromExisting: a constrainableX and: b constrainableY!

constrainableBottomRight
	^ellipse constrainableCorner!

constrainableCenter
	^center!

constrainableCenterx
	^self constrainableCenter constrainableX!

constrainableCentery
	^self constrainableCenter constrainableY!

constrainableExtent
	^extent!

constrainableFeatures
	^#(#('top' #top ) #('left' #left ) #('right' #right ) #('bottom' #bottom ) #('center x' #centerx ) #('center y' #centery ) #('width' #width ) #('height' #height ) #('top left' #topLeft ) #('top right' #topRight ) #('bottom left' #bottomLeft ) #('bottom right' #bottomRight ) #('center' #center ) #('extent' #extent ) #('border width' #borderWidth ) #('border color' #borderColor ) #('fill color' #fillColor ) )!

constrainableFillColor
	^fillColor!

constrainableHeight
	^self constrainableExtent constrainableY!

constrainableLeft
	^ellipse constrainableOrigin constrainableX!

constrainablePosition
	^Array with: ellipse constrainableOrigin with: ellipse constrainableCorner!

constrainableRight
	^ellipse constrainableCorner constrainableX!

constrainableTop
	^ellipse constrainableOrigin constrainableY!

constrainableTopLeft
	^ellipse constrainableOrigin!

constrainableTopRight
	| a b |
	a := ellipse constrainableOrigin.
	b := ellipse constrainableCorner.
	^CoolDrawPoint newFromExisting: b constrainableX and: a constrainableY!

constrainableWidth
	^self constrainableExtent constrainableX!

fillColor
	^fillColor contents!

height
	^self extent y!

majorColorValue
	^self theFillColor!

width
	^self extent x! !

!EllipseFigure methodsFor: 'private'!

setEllipse: aRectangle onView: aView 
	| graph |
	graph := aView drawing constraintGraph.
	ellipse := CoolDrawRectangle
				newOn: graph
				with: aRectangle
				usingName: self asOop printString
				owner: self.
	center := CoolDrawPoint
				newIn: graph
				with: ellipse center
				owner: self.
	extent := CoolDrawPoint
				newIn: graph
				with: ellipse extent
				owner: self.
	self
		centerConstraint: self constrainableLeft
		and: center constrainableX
		and: self constrainableRight
		in: graph.
	self
		centerConstraint: self constrainableTop
		and: center constrainableY
		and: self constrainableBottom
		in: graph.
	self
		widthConstraint: self constrainableLeft
		and: extent constrainableX
		and: self constrainableRight
		in: graph.
	self
		widthConstraint: self constrainableTop
		and: extent constrainableY
		and: self constrainableBottom
		in: graph!

width: anInteger color: aColorValue fillColor: anotherColorValue 
self obsoleteMessage.
	"width := anInteger.
	color contents: aColorValue.
	fillColor := anotherColorValue"!

width: anInteger color: aColorValue fillColor: anotherColorValue onView: aView 
	| graph |
	graph := aView drawing constraintGraph.
	borderWidth := CoolDrawVariable
				newNamed: 'borderwidth' , self asOop printString
				in: graph
				with: anInteger.
	color := CoolDrawVariable
				newNamed: 'bordercolor' , self asOop printString
				in: graph
				with: aColorValue.
	fillColor := CoolDrawVariable
				newNamed: 'fillcolor' , self asOop printString
				in: graph
				with: anotherColorValue.
	borderWidth owner: self.
	color owner: self.
	fillColor owner: self! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

EllipseFigure class
	instanceVariableNames: ''!


!EllipseFigure class methodsFor: 'instance creation'!

createNotifying: aView
	| aFigure |
	aFigure := self ellipse: (aView controller sensor cursorPoint extent: 0@0)
				onView: aView.
	aFigure growNotifying: aView.
	^aFigure!

creationTool
	| temp |

	temp := Image
		extent: 16@16
		depth: 1
		palette: MappedPalette blackWhite
		bits: #[255 255 255 255 255 255 255 255 248 31 255 255 240 15 255 255 227 199 255 255 199 227 255 255 207 243 255 255 207 243 255 255 207 243 255 255 207 243 255 255 199 227 255 255 227 199 255 255 240 15 255 255 248 31 255 255 255 255 255 255 255 255 255 255].
	^CreationTool
		icon: temp
		cursor: Cursor crossHair
		class: self!

ellipse: aRectangle onView: aView
	^(self new
		width: 1
		color: (self colorNamed: 'black')
		fillColor: (self colorNamed: 'white')
		onView: aView)
		setEllipse: aRectangle onView: aView! !


VisualComponent subclass: #Tool
	instanceVariableNames: 'icon cursor controller '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Framework'!
Tool comment:
'Tool is an abstract class capturing the behaviour of a drawing tool.  Tools change the way the DrawingController acts while working within a Drawing.

Instance Variables :
icon		<Image>		The Image displayed in the ToolPaletteView.
cursor	<Cursor>		The Cursor displayed while the Tool is active.
controller	<Controller>	The DrawingController which hands off messages to the Tool. 
'!


!Tool methodsFor: 'accessing'!

controller
        ^controller!

controller: aController 
	controller := aController!

cursor
	^cursor!

drawing
        ^self view drawing!

icon
	^icon!

model
        ^controller model!

sensor
        ^controller sensor!

view
        ^controller view! !

!Tool methodsFor: 'model accessing'!

figureAtCursor
	^self view figureAt: self sensor cursorPoint!

figureAtPoint: aPoint
		^self view figureAt: aPoint! !

!Tool methodsFor: 'testing'!

canCutCopyPaste
        ^false! !

!Tool methodsFor: 'invoking'!

activate
	"This message is sent when a tool is selected. The default 
	behavior is that changing tools cancels any selections"

	self view noSelections!

backspace!

controlCharacter: aCharacter!

deactivate!

display
	"Tools are given the opportunity to modify the view after the drawing has been displayed"!

press
	| aFigure |
	aFigure := self figureAtCursor.
	aFigure isNil
		ifTrue: [self pressBackground]
		ifFalse: [self pressFigure: aFigure]!

release!

type: aString! !

!Tool methodsFor: 'displaying'!

displayOn: aGraphicsContext
	icon displayOn: aGraphicsContext!

show
        cursor show! !

!Tool methodsFor: 'private'!

computeBoundingBox
	^icon preferredBounds!

setIcon: anImage cursor: aCursor 
	icon := anImage.
	cursor := aCursor! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Tool class
	instanceVariableNames: ''!


!Tool class methodsFor: 'instance creation'!

icon: anImage cursor: aCursor 
	^super new setIcon: anImage cursor: aCursor! !

!Tool class methodsFor: 'class constants'!

toolHiLiteSize
	"Answer a Point which is the size of the border around my icon when selected"

	^4@4!

toolIconSpacing
	"Answer a Point which is the size between adjacent tool icons"

	^2@2!

toolImageSize
	"Answer a Point which is the size of my icon"

	^16@16! !


Tool subclass: #ConstraintCreationTool
	instanceVariableNames: 'count figure1 figure2 figure3 '
	classVariableNames: 'Empty EmptyEmpty EmptyEmptyEmpty Full FullEmpty FullEmptyEmpty FullFull FullFullEmpty FullFullFull '
	poolDictionaries: ''
	category: 'CoolDraw-Tools'!
ConstraintCreationTool comment:
'The ConstraintCreationTool is a special tool for creating and adding constraints to a CoolDraw drawing.  Using the cursor, it prompts the user to select the figures to be constrained.  After enough figures have been selected, it uses a dialog window to query for the details of the constraint.  Then it creates and adds the constraint to the constraint graph of the drawing.

This class and its collaborators are not very well structured and will change in future versions of CoolDraw.
'!


!ConstraintCreationTool methodsFor: 'initialization'!

count: c
	count := c.
	^self! !

!ConstraintCreationTool methodsFor: 'invoking'!

activate
	cursor := self emptyState.
	self cursor show.
	super activate!

deactivate
	cursor := self emptyState.
	super deactivate!

pressBackground
	cursor := self emptyState.
	self cursor show!

pressFigure: aFigure 
	count = 1
		ifTrue: [self pressFigureOne: aFigure]
		ifFalse: [count = 2
				ifTrue: [self pressFigureTwo: aFigure]
				ifFalse: [count = 3
						ifTrue: [self pressFigureThree: aFigure]
						ifFalse: [self halt: 'not yet implemented']]]!

pressFigureOne: aFigure 
	| action |
	cursor := Full.
	self cursor show.
	figure1 := aFigure.
	action := CoolDrawConstraintDialog on: figure1 on: self view drawing.
	action isOk
		ifTrue: 
			[self createOneConstraintFrom: action.
			self updateDrawing].
	action release.
	cursor := Empty.
	self cursor show!

pressFigureThree: aFigure 
	| action |
	cursor == EmptyEmptyEmpty
		ifTrue: 
			[cursor := FullEmptyEmpty.
			self cursor show.
			figure1 := aFigure]
		ifFalse: [cursor == FullEmptyEmpty
				ifTrue: 
					[cursor := FullFullEmpty.
					self cursor show.
					figure2 := aFigure]
				ifFalse: 
					[cursor := FullFullFull.
					self cursor show.
					figure3 := aFigure.
					action := CoolDrawConstraintDialog
								between: figure1
								and: figure2
								and: figure3
								on: self view drawing.
					action isOk
						ifTrue: 
							[self createTwoConstraintFrom: action.
							self updateDrawing].
					action release.
					cursor := EmptyEmptyEmpty.
					self cursor show]]!

pressFigureTwo: aFigure 
	| action |
	cursor == EmptyEmpty
		ifTrue: 
			[cursor := FullEmpty.
			self cursor show.
			figure1 := aFigure]
		ifFalse: 
			[cursor := FullFull.
			self cursor show.
			figure2 := aFigure.
			action := CoolDrawConstraintDialog
						between: figure1
						and: figure2
						on: self view drawing.
			action isOk
				ifTrue: 
					[self createTwoConstraintFrom: action.
					self updateDrawing].
			action release.
			cursor := EmptyEmpty.
			self cursor show]! !

!ConstraintCreationTool methodsFor: 'constraint creation'!

createConstraintFrom: dialog v1: v1
	dialog relation = #stay
		ifTrue: [^self createStayConstraintFrom: dialog v1: v1].
	self notYetImplemented: dialog relation!

createConstraintFrom: dialog v1: v1 v2: v2 offset: offset 
	dialog relation = #=
		ifTrue: [^self createEqualityConstraintFrom: dialog v1: v1 v2: v2 offset: offset].
	dialog relation = #>=
		ifTrue: [^self createGreaterThanConstraintFrom: dialog v1: v1 v2: v2 offset: offset].
	dialog relation = #<=
		ifTrue: [^self createGreaterThanConstraintFrom: dialog v1: v2 v2: v1 offset: offset negated].
	self notYetImplemented: dialog relation!

createEqualityConstraintFrom: dialog v1: v1 v2: v2 offset: offset 
	| multiplier cnst meth1 meth2 |
	multiplier := dialog multiplier.
	cnst := SkyBlueConstraint new.
	cnst name: v1 printString , ' = ' , v2 printString.
	cnst strength: dialog strength.
	cnst variables: (Array with: v1 with: v2).
	cnst errorfunction: [:vars | (vars at: 1) contents * multiplier - (vars at: 2) contents - offset].
	meth1 := SkyBlueMethod new.
	meth1 inputs: (Array with: v1).
	meth1 outputs: (Array with: v2).
	meth1 external: false.
	meth1 block: [:in :out | (out at: 1) contents: (((in at: 1) contents) * multiplier) asInteger + offset].
	meth2 := SkyBlueMethod new.
	meth2 inputs: (Array with: v2).
	meth2 outputs: (Array with: v1).
	meth2 external: false.
	meth2 block: [:in :out | (out at: 1) contents: ((in at: 1) contents - offset) // multiplier].
	cnst methods: (Array with: meth1 with: meth2).
	^cnst!

createGreaterThanConstraintFrom: dialog v1: v1 v2: v2 offset: offset 
	| multiplier cnst meth1 meth2 |
	multiplier := dialog multiplier.
	cnst := ColbaltBlueNonuniqueConstraint new.
	cnst name: v1 printString , ' >= ' , v2 printString.
	cnst strength: dialog strength.
	cnst variables: (Array with: v1 with: v2).
	cnst errorfunction: [:vars | (((vars at: 1) contents * multiplier) + offset) >= (vars at: 2) contents
			ifTrue: [0]
			ifFalse: [(((vars at: 1) contents * multiplier) - offset - (vars at: 2) contents) abs]].
	meth1 := SkyBlueMethod new.
	meth1 inputs: (Array with: v1).
	meth1 outputs: (Array with: v2).
	meth1 external: false.
	meth1 block: [:in :out | (out at: 1) contents: (((in at: 1) contents) * multiplier) asInteger + offset].
	meth2 := SkyBlueMethod new.
	meth2 inputs: (Array with: v2).
	meth2 outputs: (Array with: v1).
	meth2 external: false.
	meth2 block: [:in :out | (out at: 1) contents: ((in at: 1) contents - offset) // multiplier].
	cnst methods: (Array with: meth1 with: meth2).
	^cnst!

createOneConstraintFrom: dialog 
	| v1 g cluster c1 c2 |
	v1 := dialog constrainableVariable1.
	g := dialog constraintGraph.
	cluster := CoolDrawConstraintCluster new initialize.
	cluster name: dialog name.
	(v1 respondsTo: #constrainableX)
		ifTrue: 
			[c1 := self createConstraintFrom: dialog v1: v1 constrainableX.
			c2 := self createConstraintFrom: dialog v1: v1 constrainableY.
			cluster addConstraint: c1; addConstraint: c2.
			g addConstraint: c1; addConstraint: c2.
			dialog drawing addCluster: cluster]
		ifFalse: 
			[c1 := self createConstraintFrom: dialog v1: v1.
			cluster addConstraint: c1.
			g addConstraint: c1.
			dialog drawing addCluster: cluster]!

createStayConstraintFrom: dialog v1: v1
	|  cnst meth1  |
	cnst := SkyBlueConstraint new.
	cnst name: v1 printString , ' stay'.
	cnst strength: dialog strength.
	cnst variables: (Array with: v1).
	cnst errorfunction: [:vars | 0].
	meth1 := SkyBlueMethod new.
	meth1 inputs: (Array new).
	meth1 outputs: (Array with: v1).
	meth1 external: false.
	meth1 block: [:in :out | ].
	cnst methods: (Array with: meth1).
	^cnst!

createTwoConstraintFrom: dialog 
	| v1 v2 offset g cluster c1 c2 |
	v1 := dialog constrainableVariable1.
	v2 := dialog constrainableVariable2.
	g := dialog constraintGraph.
	offset := dialog offset.
	cluster := CoolDrawConstraintCluster new initialize.
	cluster name: dialog name.
	(v1 respondsTo: #constrainableX)
		ifTrue: 
			[(offset isMemberOf: Point) ifFalse: [offset := offset@offset].
			c1 := self
						createConstraintFrom: dialog
						v1: v1 constrainableX
						v2: v2 constrainableX
						offset: offset x.
			c2 := self
						createConstraintFrom: dialog
						v1: v1 constrainableY
						v2: v2 constrainableY
						offset: offset y.
			cluster addConstraint: c1; addConstraint: c2.
			g addConstraint: c1; addConstraint: c2.
			dialog drawing addCluster: cluster]
		ifFalse: 
			[c1 := self
						createConstraintFrom: dialog
						v1: v1
						v2: v2
						offset: offset.
			cluster addConstraint: c1.
			g addConstraint: c1.
			dialog drawing addCluster: cluster]!

updateDrawing
	"This next is a hack because the SkyBlue algorithm 
	does not tell the figures that they have changed.  It
	does cause a double screen redraw on X systems,
	but I can't see a good solution to that."

	self view drawing damageAll! !

!ConstraintCreationTool methodsFor: 'private'!

emptyState
	^count = 1
		ifTrue: [Empty]
		ifFalse: [count = 2
				ifTrue: [EmptyEmpty]
				ifFalse: [count = 3
						ifTrue: [EmptyEmptyEmpty]
						ifFalse: [[self halt: 'not yet implemented']]]]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ConstraintCreationTool class
	instanceVariableNames: ''!


!ConstraintCreationTool class methodsFor: 'class initialization'!

initEmpty
	Empty := (Cursor 
			imageArray: #(
		2r0000000000000000
		2r0000000000111110
		2r0000000000011110
		2r0000000000001110
		2r0000000000000110
		2r0000000000000010
		2r0000000000000000
		2r0000000000111110
		2r0000000000100010
		2r0000000000100010
		2r0000000000100010
		2r0000000000100010
		2r0000000000100010
		2r0000000000100010
		2r0000000000111110
		2r0000000000000000)
			maskArray: #(
		2r0000000011111111
		2r0000000001111111
		2r0000000000111111
		2r0000000000011111
		2r0000000000001111
		2r0000000000000111
		2r0000000001111111
		2r0000000001111111
		2r0000000001111111
		2r0000000001111111
		2r0000000001111111
		2r0000000001111111
		2r0000000001111111
		2r0000000001111111
		2r0000000001111111
		2r0000000001111111)
			hotSpot: 15@0 name: 'empty').!

initEmptyEmpty
	EmptyEmpty := (Cursor 
			imageArray: #(
		2r0000000000000000
		2r0000000000111110
		2r0000000000011110
		2r0000000000001110
		2r0000000000000110
		2r0000000000000010
		2r0000000000000000
		2r0000001111011110
		2r0000001001010010
		2r0000001001010010
		2r0000001001010010
		2r0000001001010010
		2r0000001001010010
		2r0000001001010010
		2r0000001111011110
		2r0000000000000000)
			maskArray: #(
		2r0000000011111111
		2r0000000001111111
		2r0000000000111111
		2r0000000000011111
		2r0000000000001111
		2r0000000000000111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111)
			hotSpot: 15@0 name: 'empty-empty').!

initEmptyEmptyEmpty
	EmptyEmptyEmpty := (Cursor 
			imageArray: #(
		2r0000000000000000
		2r0000000000111110
		2r0000000000011110
		2r0000000000001110
		2r0000000000000110
		2r0000000000000010
		2r0000000000000000
		2r0111101111011110
		2r0100101001010010
		2r0100101001010010
		2r0100101001010010
		2r0100101001010010
		2r0100101001010010
		2r0100101001010010
		2r0111101111011110
		2r0000000000000000)
			maskArray: #(
		2r0000000011111111
		2r0000000001111111
		2r0000000000111111
		2r0000000000011111
		2r0000000000001111
		2r0000000000000111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111)
			hotSpot: 15@0 name: 'empty-empty-empty').!

initFull
	Full := (Cursor 
			imageArray: #(
		2r0000000000000000
		2r0000000000111110
		2r0000000000011110
		2r0000000000001110
		2r0000000000000110
		2r0000000000000010
		2r0000000000000000
		2r0000000000111110
		2r0000000000111110
		2r0000000000111110
		2r0000000000111110
		2r0000000000111110
		2r0000000000111110
		2r0000000000111110
		2r0000000000111110
		2r0000000000000000)
			maskArray: #(
		2r0000000011111111
		2r0000000001111111
		2r0000000000111111
		2r0000000000011111
		2r0000000000001111
		2r0000000000000111
		2r0000000001111111
		2r0000000001111111
		2r0000000001111111
		2r0000000001111111
		2r0000000001111111
		2r0000000001111111
		2r0000000001111111
		2r0000000001111111
		2r0000000001111111
		2r0000000001111111)
			hotSpot: 15@0 name: 'full').!

initFullEmpty
	FullEmpty := (Cursor 
			imageArray: #(
		2r0000000000000000
		2r0000000000111110
		2r0000000000011110
		2r0000000000001110
		2r0000000000000110
		2r0000000000000010
		2r0000000000000000
		2r0000001111011110
		2r0000001111010010
		2r0000001111010010
		2r0000001111010010
		2r0000001111010010
		2r0000001111010010
		2r0000001111010010
		2r0000001111011110
		2r0000000000000000)
			maskArray: #(
		2r0000000011111111
		2r0000000001111111
		2r0000000000111111
		2r0000000000011111
		2r0000000000001111
		2r0000000000000111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111)
			hotSpot: 15@0 name: 'full-empty').!

initFullEmptyEmpty
	FullEmptyEmpty := (Cursor 
			imageArray: #(
		2r0000000000000000
		2r0000000000111110
		2r0000000000011110
		2r0000000000001110
		2r0000000000000110
		2r0000000000000010
		2r0000000000000000
		2r0111101111011110
		2r0111101001010010
		2r0111101001010010
		2r0111101001010010
		2r0111101001010010
		2r0111101001010010
		2r0111101001010010
		2r0111101111011110
		2r0000000000000000)
			maskArray: #(
		2r0000000011111111
		2r0000000001111111
		2r0000000000111111
		2r0000000000011111
		2r0000000000001111
		2r0000000000000111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111)
			hotSpot: 15@0 name: 'full-empty-empty').!

initFullFull
	FullFull := (Cursor 
			imageArray: #(
		2r0000000000000000
		2r0000000000111110
		2r0000000000011110
		2r0000000000001110
		2r0000000000000110
		2r0000000000000010
		2r0000000000000000
		2r0000001111011110
		2r0000001111011110
		2r0000001111011110
		2r0000001111011110
		2r0000001111011110
		2r0000001111011110
		2r0000001111011110
		2r0000001111011110
		2r0000000000000000)
			maskArray: #(
		2r0000000011111111
		2r0000000001111111
		2r0000000000111111
		2r0000000000011111
		2r0000000000001111
		2r0000000000000111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111
		2r0000011111111111)
			hotSpot: 15@0 name: 'full-full').!

initFullFullEmpty
	FullFullEmpty := (Cursor 
			imageArray: #(
		2r0000000000000000
		2r0000000000111110
		2r0000000000011110
		2r0000000000001110
		2r0000000000000110
		2r0000000000000010
		2r0000000000000000
		2r0111101111011110
		2r0111101111010010
		2r0111101111010010
		2r0111101111010010
		2r0111101111010010
		2r0111101111010010
		2r0111101111010010
		2r0111101111011110
		2r0000000000000000)
			maskArray: #(
		2r0000000011111111
		2r0000000001111111
		2r0000000000111111
		2r0000000000011111
		2r0000000000001111
		2r0000000000000111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111)
			hotSpot: 15@0 name: 'full-full-empty').!

initFullFullFull
	FullFullFull := (Cursor 
			imageArray: #(
		2r0000000000000000
		2r0000000000111110
		2r0000000000011110
		2r0000000000001110
		2r0000000000000110
		2r0000000000000010
		2r0000000000000000
		2r0111101111011110
		2r0111101111011110
		2r0111101111011110
		2r0111101111011110
		2r0111101111011110
		2r0111101111011110
		2r0111101111011110
		2r0111101111011110
		2r0000000000000000)
			maskArray: #(
		2r0000000011111111
		2r0000000001111111
		2r0000000000111111
		2r0000000000011111
		2r0000000000001111
		2r0000000000000111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111)
			hotSpot: 15@0 name: 'full-full-full').!

initialize
	"ConstraintCreationTool initialize"
	self initEmptyEmpty.
	self initFullEmpty.
	self initFullFull.
	self initFull.
	self initEmpty.
	self initEmptyEmptyEmpty.
	self initFullEmptyEmpty.
	self initFullFullEmpty.
	self initFullFullFull! !

!ConstraintCreationTool class methodsFor: 'instance creation'!

newOne
	| me |
	me := self icon: (Empty image) cursor: Empty.
	me count: 1.
	^me!

newThree
	| me |
	me := self icon: (EmptyEmptyEmpty image) cursor: EmptyEmptyEmpty.
	me count: 3.
	^me!

newTwo
	| me |
	me := self icon: (EmptyEmpty image) cursor: EmptyEmpty.
	me count: 2.
	^me! !


Tool subclass: #CreationTool
	instanceVariableNames: 'className '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Tools'!
CreationTool comment:
'CreationTool is an abstract subclass of Tool.  CreationTools are Tools that create new Figures in a Drawing.  A CreationTool knows the class of the Figure to be created when the Tool is press in the background of a Drawing.

Instance Variables : 
className	<Symbol>		The classname of the Figure created when the CreationTool in used.'!


!CreationTool methodsFor: 'invoking'!

pressBackground
        self createFigure!

pressFigure: aFigure 
	"Just create a Figure on top of the existing figure - a reasonable default behaviour."

	self createFigure! !

!CreationTool methodsFor: 'private'!

createFigure
	"Create a figure and add it to the drawing."
	| aFigure |
	aFigure :=self creationClass createNotifying: self view.
     aFigure notNil ifTrue: [self view addFigure: aFigure].
	^aFigure!

creationClass
        ^Smalltalk at: className!

setClassName: aSymbol
        className := aSymbol! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

CreationTool class
	instanceVariableNames: ''!


!CreationTool class methodsFor: 'instance creation'!

cursor: aCursor class: aClass
        ^(self cursor: aCursor) setClassName: aClass name!

icon: anImage cursor: aCursor class: aClass
        ^(self icon: anImage cursor: aCursor) setClassName: aClass name! !


CreationTool subclass: #TextTool
	instanceVariableNames: 'start stop typingTarget '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Tools'!
TextTool comment:
'A TextTool has the responsibility of accepting typed text and mouse movements and translating them into the standard Text manipulation tasks.  Many of the methods in TextTool are similar to methods in ComposedTextView and ComposedText.  This is in part to hide the use of a ComposedText as a part of TextFigure.

Instance Variables :
start		<Integer>		The beginning index into the typingTarget of the current selection.
stop		<Integer>		The ending index into the typingTarget of the current selection.
typingTarget	<TextFigure>	The object which receives the Text typed and cuts/copies/pastes.

'!


!TextTool methodsFor: 'invoking'!

backspace
	| startIndex |
	typingTarget isNil ifTrue: [^self].
	self reverseSelection.
	startIndex := start max: 2.
	typingTarget
		replaceFrom: startIndex - 1
		to: stop - 1
		with: Text new
		notifying: self view.
	start := startIndex - 1.
	stop := start copy.
	self reverseSelection!

copy
	typingTarget isNil ifTrue: [^self].
	ParagraphEditor currentSelection: (typingTarget copyFrom: start to: stop - 1)!

cut
	self copy.
	typingTarget isNil ifTrue: [^self].
	self reverseSelection.
	typingTarget
		replaceFrom: start 
		to: stop - 1
		with: Text new
		notifying: self view.
	stop := start copy.
	self reverseSelection!

deactivate
	self emptyTextCheck.
	self cleanUpOldTypingTarget!

display
	typingTarget isNil | start isNil | stop isNil ifTrue: [^self].
	self reverseSelection!

paste
	| aText |
	typingTarget isNil ifTrue: [^self].
	self reverseSelection.
	typingTarget
		replaceFrom: start
		to: stop - 1
		with: (aText := ParagraphEditor currentSelection)
		notifying: self view.
	stop := start + aText size.
	self reverseSelection!

pressBackground
	"I might be in the middle of editing something already."
	self cleanUpOldTypingTarget.
	super pressBackground!

pressFigure: aFigure 
	aFigure acceptsTyping
		ifTrue: [self editFigure: aFigure]
		ifFalse: [self pressBackground "Just like we pressed the background"]!

type: aString 
	typingTarget isNil ifTrue: [^self].
	self reverseSelection.
	typingTarget
		replaceFrom: start
		to: stop - 1
		with: (Text fromString: aString)
		notifying: self view.
	start := start + aString size.
	stop := start copy.
	self reverseSelection.! !

!TextTool methodsFor: 'testing'!

canCutCopyPaste
        ^true! !

!TextTool methodsFor: 'editing'!

editFigure: aFigure 
	| newIndices |
	self cleanUpOldTypingTarget.
	typingTarget := aFigure.
	start := stop := aFigure indexForPoint: self sensor cursorPoint.
	newIndices := self mouseSelect: start to: stop.
	start := newIndices first.
	stop := newIndices last.
	self reverseSelection!

mouseSelect: oldStart to: oldStop 
	"Highlight start to stop in the typingTarget and accept changes to the selected region. 
	Changes come as movement while the red button is pressed. Don't permit scrolling at this 
	time."

	| newStart newStop pivot selectBackground selectForeground background foreground |
	pivot := newStart := newStop := typingTarget indexForPoint: self sensor cursorPoint.
	selectBackground := LookPreferences defaultForWindows selectionBackgroundColor.
	selectForeground := LookPreferences defaultForWindows selectionForegroundColor.
	background := LookPreferences defaultForWindows backgroundColor.
	foreground := ColorValue black.
	[self sensor redButtonPressed]
		whileTrue: 
			[newStop := typingTarget indexForPoint: self sensor cursorPoint.
			(newStop > newStart and: [newStart < pivot])
				ifTrue: [typingTarget
						highlightFrom: newStart
						to: newStop
						on: self view graphicsContext
						foreground: foreground
						background: background].
			(newStop < newStart and: [newStart > pivot])
				ifTrue: [typingTarget
						highlightFrom: newStop
						to: newStart
						on: self view graphicsContext
						foreground: foreground
						background: background].
			newStop = newStart
				ifFalse: 
					[typingTarget
						highlightFrom: pivot
						to: newStop
						on: self view graphicsContext
						foreground: selectForeground
						background: selectBackground.
					newStart := newStop]].
	pivot = newStop ifTrue: [typingTarget displayCaretAtIndex: pivot on: self view graphicsContext].
	newStop < pivot
		ifTrue: [^Array with: newStop with: pivot]
		ifFalse: [^Array with: pivot with: newStop]! !

!TextTool methodsFor: 'private'!

cleanUpOldTypingTarget
	"If I am currently editing a TextFigure, deselect it."
	typingTarget isNil
		ifFalse:
			[typingTarget recompose.
			typingTarget changed.
			typingTarget := nil.
			self view repairDamage].!

createFigure
	| aFigure |
	self emptyTextCheck.
	aFigure := super createFigure.
	start := stop := 1.
	self editFigure: aFigure.
	^aFigure!

emptyTextCheck
	typingTarget isNil ifTrue: [^self].
	typingTarget isEmpty 
		ifTrue: ["damage caret before removing the empty TextFigure."
				self view drawing 
					damageRegion: 
						(typingTarget origin extent: 10@40).
				self view remove: typingTarget.
				typingTarget := nil]!

reverseSelection
	typingTarget isNil ifTrue: [^self].
	start = stop
		ifTrue: [typingTarget displayCaretAtIndex: start on: self view graphicsContext]
		ifFalse: [typingTarget
				highlightFrom: start
				to: stop
				on: self view graphicsContext
				foreground: LookPreferences defaultForWindows selectionForegroundColor
				background: LookPreferences defaultForWindows selectionBackgroundColor]! !


CreationTool subclass: #WrapperCreationTool
	instanceVariableNames: 'wrapperClassName '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Tools'!


!WrapperCreationTool methodsFor: 'private'!

createFigure
	"Create a figure and add it to the drawing."

	| aFigure |
	aFigure := self creationClass createNotifying: self view.
	aFigure notNil ifTrue: [self view addFigure: (self wrapperClass new wrappedFigure: aFigure)]!

setWrapperClassName: aSymbol
        wrapperClassName := aSymbol!

wrapperClass
        ^Smalltalk at: wrapperClassName! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

WrapperCreationTool class
	instanceVariableNames: ''!


!WrapperCreationTool class methodsFor: 'instance creation'!

cursor: aCursor class: aClass wrapperClass: wrapperClass
        ^(self cursor: aCursor) setClassName: aClass name; setWrapperClassName: wrapperClass name!

icon: anImage cursor: aCursor class: aClass wrapperClass: wrapperClass
        ^(self icon: anImage cursor: aCursor) setClassName: aClass name; setWrapperClassName: wrapperClass name! !


Tool subclass: #SelectionTool
	instanceVariableNames: ''
	classVariableNames: 'MarqueeImages '
	poolDictionaries: ''
	category: 'HotDraw-Tools'!
SelectionTool comment:
' A SelectionTool is a Tool which is used to make a selection of one or more Figures in a Drawing.  If pressed while over a Figure, the Figure is added to the current selection of the Drawing.  If the left-shift key is held during red button presses, the selection is toggled.  If the SelectionTool is pressed in the background, a travelling marque is drawn (ala Apple Macintosh) and is used to enclose the desired Figures.

Class Variables :
MarqueeImages	<Array of Images>	A pair of Images that are alternated between to simulate a travelling marquee.
'!


!SelectionTool methodsFor: 'private'!

marqueeFromUser
	"Answer a Rectangle of the size selected by the user."

	|  gc   startPoint stopPoint patIndex oldStopPoint |
	gc := self view graphicsContext.
	patIndex := 1.
	gc paint: MarqueeImages first.
	gc lineWidth: 1.
	stopPoint := startPoint := self sensor cursorPoint.
	[self sensor redButtonPressed]
		whileTrue: 
			[oldStopPoint := stopPoint.
			stopPoint := self sensor cursorPoint.
			startPoint := startPoint min: stopPoint - (5 @ 5).
			gc displayRectangularBorder: (startPoint corner: oldStopPoint)
				at: Point zero.
			patIndex == 1
				ifTrue: [patIndex := 2]
				ifFalse: [patIndex := 1].
			gc paint: (MarqueeImages at: patIndex).
			oldStopPoint = stopPoint
				ifFalse: 
					[self view drawing damageRegion: (startPoint corner: oldStopPoint).
					self view repairDamage]].
	self view drawing damageRegion: (startPoint corner: stopPoint).
	self view repairDamage.
	^startPoint corner: stopPoint!

pressHandle: aHandle 
	^aHandle invoke: self view! !

!SelectionTool methodsFor: 'selecting'!

selectFigure: aFigure
        self sensor shiftDown
                ifTrue:
                        [self view toggleSelection: aFigure]
                ifFalse:
                        [(self view isSelected: aFigure)
                                ifFalse: [self view selection: aFigure]]!

selectGroup
	| aRectangle aCollection |
	self view noSelections.
	aRectangle := self marqueeFromUser.
	aCollection := self view figuresIn: aRectangle.
	self sensor shiftDown ifFalse: [self view selections: aCollection]
		ifTrue: [self view toggleSelections: aCollection]! !

!SelectionTool methodsFor: 'invoking'!

backspace
        self view removeSelections!

pressBackground
        self selectGroup!

pressFigure: aFigure

        (aFigure isActive and: [(self view isSelected: aFigure) not])
                ifTrue:
                        [self pressHandle: aFigure]
                ifFalse:
                        [self selectFigure: aFigure.
                        self moveFigure: aFigure]! !

!SelectionTool methodsFor: 'moving'!

isMovement
	| startPoint aSensor |
	aSensor := self controller sensor.
	startPoint := aSensor cursorPoint.
	[aSensor anyButtonPressed & ((startPoint dist: aSensor cursorPoint)
			< 1.0)] whileTrue.
	^(startPoint dist: aSensor cursorPoint)
		>= 1.0!

moveFigure: aFigure 
	self isMovement ifFalse: [^self].
	self pressHandle: (SelectionTrackHandle positionOf: aFigure)! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SelectionTool class
	instanceVariableNames: ''!


!SelectionTool class methodsFor: 'instance creation'!

new
	^self icon: (Image extent: 16@16 depth:1 palette: MappedPalette blackWhite bits: #[
2r11111111 2r11111111 
2r11111101 2r11111111 
2r11111100 2r11111111 
2r11111100 2r01111111 
2r11111100 2r00111111 
2r11111100 2r00011111 
2r11111100 2r00001111 
2r11111100 2r00000111 
2r11111100 2r00011111 
2r11111100 2r10011111 
2r11111101 2r10011111 
2r11111111 2r11001111 
2r11111111 2r11001111 
2r11111111 2r11100111 
2r11111111 2r11100111 
2r11111111 2r11111111 
] pad: 8) cursor: Cursor normal! !

!SelectionTool class methodsFor: 'initialization'!

initialize
	"Setup my MarqueeImages."
	"SelectionTool initialize"

	| foreground background |
	foreground := LookPreferences defaultForWindows foregroundColor.
	background := LookPreferences defaultForWindows backgroundColor.
	MarqueeImages := OrderedCollection with: (Pattern from: (Image
						extent: 8 @ 8
						depth: 1
						palette: (MappedPalette with: background with: foreground)
						bits: #[225 225 225 225 225 225 225 225 225 225 225 225 225 225 225 225 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 ])) 
				with: (Pattern from: (Image
							extent: 8 @ 8
							depth: 1
							palette: (MappedPalette with: foreground with: background)
							bits: #[225 225 225 225 225 225 225 225 225 225 225 225 225 225 225 225 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 ]))! !


Tool subclass: #DrawingActionTool
	instanceVariableNames: 'activateBlock deactivateBlock '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Tools'!
DrawingActionTool comment:
'DrawingActionTools are tools which operate on the entire Drawing.  When pressed (or alternately when the tool is selected) the actionBlock of a DrawingActionTool is evaluated with the DrawingEditor as an argument.

Instance Variables :
activateBlock	<Block Closure>
	The block to be executed when the tool is pressed.  This block should expect a single argument, the DrawingEditor.

deactivateBlock	<BlockClosure>
	The block to be executed when the tool is released.  This block should expect a single argument, the DrawingEditor.'!


!DrawingActionTool methodsFor: 'invoking'!

activate
	"When DrawingActionTools are activated, their activateBlock gets evaluated with the 
	Drawing as an argument."

	activateBlock value: self model.!

deactivate
	"When DrawingActionTools are deactivated, their deactivateBlock gets evaluated with the 
	Drawing as an argument."

	deactivateBlock value: self model.!

pressBackground
	"Do the same thing as when I am activated."

	self activate!

pressFigure: aFigure
	"Do the same thing as when I am activated."

	self activate! !

!DrawingActionTool methodsFor: 'private'!

activateBlock: aBlock deactivateBlock: dBlock 
	activateBlock := aBlock.
	deactivateBlock := dBlock! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

DrawingActionTool class
	instanceVariableNames: ''!


!DrawingActionTool class methodsFor: 'constants'!

loadTool
	| aTool bitArray anImage aMask aCursor |
	bitArray :=  #[
2r00000000 2r00000000
2r01111110 2r00000000 
2r01000011 2r00000000
2r01011011 2r10000000 
2r01000011 2r11000000
2r01011100 2r11000000 
2r01000000 2r01000000
2r01011111 2r01000000 
2r01000000 2r01000000
2r01011101 2r11000000 
2r01000000 2r01100000
2r01111111 2r11010010 
2r00000000 2r00001010
2r00000000 2r00000110 
2r00000000 2r00011110
2r00000000 2r00000000 
].
	aTool := self icon: (Image
					extent: 16 @ 16
					depth: 1
					palette: MappedPalette blackWhite
					bits: bitArray pad: 8)
				cursor: Cursor read.
	anImage := Image
					extent: 16 @ 16
					depth: 1
					palette: MappedPalette whiteBlack
					bits: bitArray pad: 8.
	aMask := Image
				extent: 16@16
				depth: 1
				palette: CoveragePalette monoMaskPalette
				bits: bitArray
				pad: 8.
	aCursor := Cursor image: anImage mask: aMask hotSpot: 0 @ 0 name: ''.
	aTool := self icon: anImage cursor: aCursor.
	^aTool
		activateBlock: 
			[:drawingEditor | 
			drawingEditor loadDrawing.
			drawingEditor currentTool: drawingEditor tools first]
		deactivateBlock: [:drawingEditor | ]!

saveTool
	| aTool bitArray aCursor anImage aMask |
	bitArray  := #[
2r00000000 2r00000000
2r01111110 2r00000000 
2r01000011 2r00000000
2r01011011 2r10000000 
2r01000011 2r11000000
2r01011100 2r11000000 
2r01000000 2r01000000
2r01011111 2r01000000 
2r01000000 2r01000000
2r01011101 2r11000000 
2r01000000 2r01000000
2r01111111 2r11011110 
2r00000000 2r00011000
2r00000000 2r00010100 
2r00000000 2r00010010
2r00000000 2r00000001 
] .
	anImage := Image
					extent: 16 @ 16
					depth: 1
					palette: MappedPalette whiteBlack
					bits: bitArray pad: 8.
	aMask := Image
				extent: 16@16
				depth: 1
				palette: CoveragePalette monoMaskPalette
				bits: bitArray
				pad: 8.
	aCursor := Cursor image: anImage mask: aMask hotSpot: 0 @ 0 name: ''.
	aTool := self icon: anImage cursor: aCursor.
	^aTool
		activateBlock: 
			[:drawingEditor | 
			drawingEditor saveDrawing.
			drawingEditor currentTool: drawingEditor tools first]
		deactivateBlock: [:drawingEditor | ]! !


Tool subclass: #ConstraintDeletionTool
	instanceVariableNames: ''
	classVariableNames: 'DeleteCursor '
	poolDictionaries: ''
	category: 'CoolDraw-Tools'!


!ConstraintDeletionTool methodsFor: 'invoking'!

pressBackground!

pressFigure: aFigure 
	| constrainableVariables clusters action |
	constrainableVariables := IdentitySet new.
	aFigure constrainableFeatures
		do: 
			[:each | 
			| v |
			v := aFigure perform: (aFigure convertToConstrainable: (each at: 2)).
			(v respondsTo: #constrainableX)
				ifTrue: 
					[constrainableVariables add: v constrainableX.
					constrainableVariables add: v constrainableY]
				ifFalse: [constrainableVariables add: v]].
	clusters := self view drawing clustersConstraining: constrainableVariables.
	clusters isEmpty
		ifTrue: 
			[DialogView warn: 'No constraints'.
			^self].
	action := CoolDrawDeleteDialog on: self view drawing clusters: clusters.
	action isOk
		ifTrue: 
			[self deleteClusters: action.
			self updateDrawing].
	action release! !

!ConstraintDeletionTool methodsFor: 'constraint deletion'!

deleteClusters: action 
	action
		selectedClustersDo: 
			[:clu | 
			clu constraints do: [:cn | self view drawing constraintGraph removeConstraint: cn].
			self view drawing removeCluster: clu]! !

!ConstraintDeletionTool methodsFor: 'private'!

updateDrawing
	"This next is a hack because the SkyBlue algorithm 
	does not tell the figures that they have changed.  It
	does cause a double screen redraw on X systems,
	but I can't see a good solution to that."

	self view drawing damageAll! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ConstraintDeletionTool class
	instanceVariableNames: ''!


!ConstraintDeletionTool class methodsFor: 'class initialization'!

initDeleteCursor
	"ConstraintDeletionTool initDeleteCursor"
	DeleteCursor := (Cursor 
			imageArray: #(
		2r0000000000000000
		2r0000000000111110
		2r0000000000011110
		2r0000000000001110
		2r0000000000000110
		2r0000000000000010
		2r0000000000000000
		2r0111101111011111
		2r0100101001011111
		2r0100101001111110
		2r0100101011110010
		2r0100101111010010
		2r0100111101010010
		2r0111111001010010
		2r1111101111011110
		2r1100000000000000)
			maskArray: #(
		2r0000000011111111
		2r0000000001111111
		2r0000000000111111
		2r0000000000011111
		2r0000000000001111
		2r0000000000000111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111
		2r1111111111111111)
			hotSpot: 15@0 name: 'delete').!

initialize
	"ConstraintDeletionTool initialize"
	self initDeleteCursor! !

!ConstraintDeletionTool class methodsFor: 'instance creation'!

new
	^self icon: DeleteCursor image cursor: DeleteCursor! !


Tool subclass: #ScrollingTool
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Tools'!
ScrollingTool comment:
'A ScrollingTool is a Tool which permits the Drawing to be interactively scrolled left, right up or down.  When pressed on the Drawing, the Drawing is told to scroll by the appropriate amount.'!


!ScrollingTool methodsFor: 'invoking'!

old.press
	"Make my drawing follow the cursorPoint."

	| oldPoint newPoint |
	oldPoint := self sensor cursorPoint.
	self sensor waitButton.
	self class closedHandCursor beCursor.
	[self sensor redButtonPressed]
		whileTrue: 
			[newPoint := self sensor cursorPoint.
			newPoint ~= oldPoint
				ifTrue: 
					[self view scrollBy: newPoint - oldPoint.
					oldPoint := newPoint]].
	self cursor beCursor!

press
	"Make my drawing follow the cursorPoint."

	| oldPoint newPoint  transWrap drawingView |
	oldPoint := self sensor cursorPoint.
	self sensor waitButton.
	self class closedHandCursor beCursor.
	drawingView := self view.
	transWrap := self view container.
	[self sensor redButtonPressed]
		whileTrue: 
			[newPoint := self sensor cursorPoint.
			newPoint ~= oldPoint
				ifTrue: 
					[transWrap setOrigin: transWrap translation + newPoint - oldPoint.
					drawingView redisplayAll.
					oldPoint := newPoint]].
	transWrap flushCoordinateCaches.
	self cursor beCursor! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ScrollingTool class
	instanceVariableNames: ''!


!ScrollingTool class methodsFor: 'instance creation'!

new
	^self icon: self flatHand cursor: Cursor hand! !

!ScrollingTool class methodsFor: 'constants'!

closedHandCursor
^Cursor fromArray: #(
2r0000000000000000
2r1000000000000001
2r1000000000000001
2r1001111111111001
2r1010010010010101
2r0111001001010110
2r0011001001010100
2r1111100000000111
2r0011000000000100
2r0101000000001010
2r1001000000001001
2r1001000000001001
2r1000100000010001
2r1000100000010001
2r0000100000010000
2r0000000000000000
)
			hotSpot: 0 @ 0
			name: 'closedHand'!

flatHand
^Image extent: 16@16 depth:1 palette: MappedPalette blackWhite bits: #[
2r11111110 2r01111111 
2r11100101 2r10001111 
2r11011001 2r10110111 
2r11011001 2r10110101 
2r11101101 2r10110010 
2r11101101 2r10110110 
2r10010111 2r11110110 
2r01100111 2r11111110 
2r01110111 2r11111101 
2r10111111 2r11111101 
2r11011111 2r11111101 
2r11101111 2r11111011 
2r11101111 2r11111011 
2r11110111 2r11110111 
2r11111011 2r11110111 
2r11111011 2r11110111 
] pad: 8! !


Tool subclass: #CoolDrawStartupTool
	instanceVariableNames: 'forms index process '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Tools'!


!CoolDrawStartupTool methodsFor: 'initialize-release'!

readFilesOrNil
	| f b |
	InputState default shiftDown ifTrue: [^nil].
	f := (Filename named: 'cooldraw.startup.picture').
	f exists ifFalse: [^nil].
	f := f readStream.
	b := BinaryObjectStorage onOld: f.
	forms := b next.
	b close.
	index := 1.
	^self! !

!CoolDrawStartupTool methodsFor: 'invoking'!

deactivate
	forms := nil.
	process terminate.
	process := nil.
	self view model currentTool: nil.
	self view topComponent display!

display
	| time |
	process isNil
		ifTrue: 
			[index := 0.
			process := [[forms notNil & self view notNil]
						whileTrue: [self sensor shiftDown
								ifFalse: 
									[index := index + 1.
									index > forms size ifTrue: [index := 1].
									self realDisplay.
									time := (forms at: index)
												at: 2.
									(Delay forSeconds: time) wait]]] newProcess.
			process resume]
		ifFalse: [self realDisplay]!

press
	self deactivate.
	self view model currentTool: self view model tools first!

realDisplay
	| gc center image pt |
	gc := self view graphicsContext.
	center := self view topComponent extent.
	gc clear.
	image := (forms at: index) at: 1.
	pt := center - image extent // 2.
	pt := pt - (20 @ 0).
	image displayOn: gc at: pt! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

CoolDrawStartupTool class
	instanceVariableNames: ''!


!CoolDrawStartupTool class methodsFor: 'instance creation'!

newOrNil
	| t |
	t := self icon: Cursor normal image cursor: Cursor normal.
	t := t readFilesOrNil.
	^t! !


Tool subclass: #FigureActionTool
	instanceVariableNames: 'actionBlock '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Tools'!
FigureActionTool comment:
'FigureActionTools are tool which operate on a figure when they are pressed.  The FigureActionTool has an actionBlock that gets evaluated when the tool is pressed on a figure.  The arguments to the block are the figure on which the tool was pressed and the view in which the cursor is located.  The view is necessary because any changes to the figure must be forwarded to the view.

Instance Variables :
actionBlock 	<BlockClosure>
	The block that gets evaluated when the tool is pressed.  This block should expect two arguments, a Figure and a View (in that order).'!


!FigureActionTool methodsFor: 'pressing'!

pressBackground
	"Since FigureActionTools need a Figure, don't do anything."!

pressFigure: aFigure 
	"Evaluate my actionBlock with aFigure as the argument to the block. Also need my View 
	so that the block can make the changes known to the View."

	^actionBlock value: aFigure value: self view! !

!FigureActionTool methodsFor: 'private'!

setActionBlock: aBlock 
	"Set my actionBlock to aBlock"

	actionBlock := aBlock! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

FigureActionTool class
	instanceVariableNames: ''!


!FigureActionTool class methodsFor: 'constants'!

bringToFront
	| temp aFAT |
	temp := Image extent: 16 @ 16 depth: 1 palette: MappedPalette blackWhite bits: #[
2r11111111 2r11111111
2r10000000 2r00111111 
2r10111011 2r10111111 
2r10110011 2r10111111
2r10100000 2r00011111 
2r10000000 2r00001111
2r10100000 2r00000111 
2r10110011 2r10000011
2r10111011 2r10100011 
2r10111111 2r10110011
2r10111111 2r10110011 
2r10111111 2r10110011
2r10111111 2r10100111 
2r10111111 2r10011111
2r10000000 2r00111111
2r11111111 2r11111111 
] pad: 8.
	aFAT := self icon: temp cursor: Cursor normal.
	^aFAT setActionBlock: [:aFigure :aView | aView bringToFront: aFigure].!

delete
	| temp aFAT aCursor |
	temp := Image extent: 16 @ 16 depth: 1 palette: MappedPalette blackWhite bits: #[
2r11111111 2r11011111  
2r11111111 2r10101111  
2r11111111 2r01110111  
2r11111110 2r11111011  
2r11111101 2r11111101  
2r11111011 2r11111001  
2r11110111 2r11110101  
2r11101111 2r11101101  
2r11011111 2r11011011  
2r11001111 2r10110111  
2r11010111 2r01101111  
2r11011010 2r11011111  
2r11101101 2r10111111  
2r11110101 2r01111111  
2r11111000 2r11111111  
2r11111101 2r11111111  
] pad: 8.
	aCursor := Cursor imageArray: #(
		2r0000000000000000
		2r0000000000000000
		2r0000000000000000
		2r0000000000000000
		2r0000111111110000
		2r0000100000010000
		2r0000100000010000
		2r0000100000010000
		2r0000100000010000
		2r0000100000010000
		2r0000100000010000
		2r0000111111110000
		2r0000000000000000
		2r0000000000000000
		2r0000000000000000
		2r0000000000000000
)
			maskArray: #(
		2r0000000000000000
		2r0000000000000000
		2r0000000000000000
		2r0001111111111000
		2r0001111111111000
		2r0001111111111000
		2r0001110000111000
		2r0001110000111000
		2r0001110000111000
		2r0001110000111000
		2r0001111111111000
		2r0001111111111000
		2r0001111111111000
		2r0000000000000000
		2r0000000000000000
		2r0000000000000000
)
			hotSpot: 8@8name: 'Delete'.
	aFAT := self icon: temp cursor: aCursor. 
	^aFAT setActionBlock: [:aFigure :aView | aView delete: aFigure].!

sendToBack
	| temp aFAT |
	temp := Image extent: 16 @ 16 depth: 1 palette: MappedPalette blackWhite bits: #[
2r11111111 2r11111111
2r10000000 2r00111111 
2r10111111 2r10111111
2r10111000 2r00011111 
2r10111000 2r00000111
2r10111000 2r00000011 
2r10111111 2r10110001
2r10111111 2r10111001 
2r10111111 2r10111101
2r10111111 2r10110101 
2r10111111 2r10100101
2r10111111 2r10000001 
2r10111111 2r10000011
2r10111111 2r10100111 
2r10000000 2r00110111
2r11111111 2r11111111 
] pad: 8.
	aFAT := self icon: temp cursor: Cursor normal. 
	^aFAT setActionBlock: [:aFigure :aView | aView sendToBack: aFigure].! !


ControllerWithMenu subclass: #DrawingController
	instanceVariableNames: 'polling '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Support'!
DrawingController comment:
'A DrawingController is the Controller component of a pair of MVC triads.  A Drawing and a DrawingView is one pair and a DrawingEditor and the same DrawingView is the other pair.  A DrawingController''s primary task to to delegate mouse and keyboard activity to the current Tool of the Drawing.'!


!DrawingController methodsFor: 'initialization'!

animateWithoutPausing
	polling := false!

initialize
	super initialize.
	polling := true! !

!DrawingController methodsFor: 'keyboard control'!

processKeyboard
	sensor keyboardPressed ifTrue: [self readKeyboard]!

readKeyboard
	| aStream aKeyboardEvent |
	aStream := WriteStream on: String new.
	[sensor keyboardPressed]
		whileTrue: 
			[aKeyboardEvent := sensor keyboardEvent.
			aKeyboardEvent hasCtrl
				ifTrue: 
					[self streamToTool: aStream.
					model currentTool controlCharacter: aKeyboardEvent key]
				ifFalse: [aKeyboardEvent keyCharacter = Character backspace
						ifTrue: [model currentTool backspace]
						ifFalse: [aStream nextPut: aKeyboardEvent keyCharacter]]].
	self streamToTool: aStream!

streamToTool: aStream 
	aStream isEmpty ifTrue: [^self].
	model currentTool type: aStream contents.
	aStream reset! !

!DrawingController methodsFor: 'control activity'!

controlActivity
	self viewHasCursor ifTrue: [self processKeyboard].
	super controlActivity.
	view step.!

controlInitialize
	"Make my cursor be the appropriate cursor for the current tool."

	model currentTool isNil ifFalse: [model currentTool cursor show]!

controlTerminate
	"Return the cursor to the normal cursor."

	Cursor normal show!

isControlWanted
	^view drawing notNil and: [super isControlWanted]!

localMenuItem: aSymbol 
	"Answer an Array of Symbols that represent the menu messages that should be sent to my 
	View as opposed to my Model."

	^#(#cut #paste #copy #group #ungroup #compose #decompose ) includes: aSymbol!

redButtonActivity
	model currentTool press.
	self sensor waitNoButton.
	model currentTool release!

yellowButtonActivity
	| aSelector aFigure |
	aFigure := view figureAtCursor.
	aFigure isNil ifTrue: [^self yellowButtonActivityDefault].
	aSelector := aFigure menu startUp.
	aSelector = 0
		ifFalse: 
			[(self localMenuItem: aSelector)
				ifTrue:  "local selectors to view, rest to figure"
					[view selection: aFigure.
					view perform: aSelector]
				ifFalse: 
					[self controlTerminate. 
					view noSelections.
					"This lets you put metaFigure messages on a menu, too."
					(aFigure respondsTo: aSelector) 
						ifTrue: [ aFigure perform: aSelector]
						ifFalse: [aFigure metaFigure perform: aSelector].
					view repairDamage].
			self controlInitialize]!

yellowButtonActivityDefault
	| menu selector |
	menu := model menu.
	menu == nil
		ifTrue: 
			[view flash.
			^super controlActivity].
	Cursor normal showWhile: [selector := menu startUp].
	selector == 0 ifFalse: [(self localMenuItem: selector)
			ifTrue: ["editing to view, rest to model"
				view perform: selector]
			ifFalse: 
				[self controlTerminate.
				selector numArgs = 1
					ifTrue: [model perform: selector with: self]
					ifFalse: [model perform: selector].
				self controlInitialize]]! !

!DrawingController methodsFor: 'basic control sequence'!

poll
	"If 'polling' is false then keep animating even 
      though the mouse is not moving or the user is not typing.

	 Otherwise, if there has been no input for a significant time, wait on a semaphore"

	ScheduledControllers checkForEvents.	
	self view == nil ifTrue: ["if the top view was closed"
		ScheduledControllers class closedWindowSignal raiseRequest].
     polling ifTrue: [self sensor pollForActivity]! !


!Cursor methodsFor: 'accessing'!

image
	^image! !


ControllerWithMenu subclass: #ToolPaletteController
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Support'!
ToolPaletteController comment:
'ToolPaletteController is the Controller responsible for detecting the change of Tools when a new Tool is selected for the ToolPalette.'!


!ToolPaletteController methodsFor: 'tool accessing'!

currentTool
	^self drawing currentTool!

currentTool: aTool 
	self drawing currentTool deactivate.
	self drawing currentTool: aTool.
	aTool activate.!

displayTools
	"Display my tools in my view."

	| position aGC oldPaint |
	position := Tool toolIconSpacing + (Tool toolHiLiteSize / 2).
	aGC := self view graphicsContext.
	oldPaint := aGC paint.
	self tools do: 
		[:each | 
		each = self currentTool
			ifFalse: [aGC paint: aGC medium background]
			ifTrue: [aGC paint: ColorValue black].
		aGC displayRectangle: (0 @ 0 extent: (Tool toolImageSize + Tool toolHiLiteSize)) at: (position - Tool toolIconSpacing).
		aGC paint: oldPaint.
		each displayOn: aGC at: position.
		position := position + (0 @ ((Tool toolImageSize + Tool toolHiLiteSize) y + Tool toolIconSpacing y))]!

toolAt: aPoint 
	"Answer the tool icon located at aPoint or nil if there is none."

	| position area |
	position := Tool toolIconSpacing + (Tool toolHiLiteSize / 2).
	self tools do: 
		[:each | 
		area := Rectangle origin: position extent: Tool toolImageSize.
		(area containsPoint: aPoint)
			ifTrue: [^each].
		position := position + (0 @ ((Tool toolImageSize + Tool toolHiLiteSize) y + Tool toolIconSpacing y))].
	^nil!

tools 
	^model tools!

tools: aCollection 
	model tools: aCollection.! !

!ToolPaletteController methodsFor: 'drawing accessing'!

drawing
	^model! !

!ToolPaletteController methodsFor: 'control defaults'!

controlTerminate
	Cursor normal show.!

redButtonActivity
	"Select the tool under the mouse (if any) and highlight it."

	| aTool |
	aTool := self toolAt: sensor cursorPoint.
	(aTool isNil not and: [(aTool = self currentTool) not])
		ifTrue: 
			[self currentTool: aTool]! !


!Project methodsFor: 'controlling'!

enter
	"The user has chosen to change the context of the workspace to be 
	that of  the receiver. Change the ChangeSet, Transcript, and collection of 
	scheduled views accordingly."

	InputState default shiftDown
		ifTrue: [self inspect]
		ifFalse: 
			[ChangeSet newChanges: projectChangeSet.
			CurrentProject := self.
			TextCollector newTranscript: projectTranscript.
			ControlManager newScheduler: projectWindows]! !


Magnitude subclass: #SkyBlueStrength
	instanceVariableNames: 'name numericvalue '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'SkyBlue-Basics'!


!SkyBlueStrength methodsFor: 'accessing'!

name
	^name! !

!SkyBlueStrength methodsFor: 'testing'!

< aStrength
	^numericvalue < aStrength numericvalue!

= aStrength
	^numericvalue = aStrength numericvalue!

hash
	^numericvalue! !

!SkyBlueStrength methodsFor: 'private'!

name: n numeric: v
	name := n.
	numericvalue := v.
	^self!

numericvalue
	^numericvalue! !

!SkyBlueStrength methodsFor: 'printing'!

printOn: aStream
	aStream nextPutAll: '%', name! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

SkyBlueStrength class
	instanceVariableNames: 'strengths absoluteweakest '!


!SkyBlueStrength class methodsFor: 'class initialization'!

initialize
	"SkyBlueStrength initialize"
	strengths := OrderedCollection new.
	strengths add: (self new name: 'required' numeric: 0).
	strengths add: (self new name: 'strong' numeric: -10).
	strengths add: (self new name: 'medium' numeric: -20).
	strengths add: (self new name: 'weak' numeric: -30).
	strengths add: (self new name: 'veryweak' numeric: -40).
	strengths add: (absoluteweakest := (self new name: 'absolute_weakest' numeric: -8888)).
	strengths add: (self new name: 'absolute_strongest' numeric: 9999).! !

!SkyBlueStrength class methodsFor: 'instance creation'!

absoluteWeakest
	^absoluteweakest!

medium
	strengths do: [:each | each name = 'medium' ifTrue: [^each]].
	self error.!

required
	strengths do: [:each | each name = 'required' ifTrue: [^each]].
	self error.!

strong
	strengths do: [:each | each name = 'strong' ifTrue: [^each]].
	self error.!

weak
	strengths do: [:each | each name = 'weak' ifTrue: [^each]].
	self error.! !


View subclass: #ToolPaletteView
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Support'!
ToolPaletteView comment:
'ToolPaletteView is the View responsible for the displaying and highlighting of Tools in the Palette Toolbar.'!


!ToolPaletteView methodsFor: 'controller accessing'!

defaultControllerClass
	^ToolPaletteController! !

!ToolPaletteView methodsFor: 'displaying'!

displayOn: aGraphicsContext 
	"Repair any damage done to my tool images"

	self controller displayTools! !

!ToolPaletteView methodsFor: 'updating'!

update: aSymbol 
	"My model has changed something, aSymbol, and I should react. Just redraw my tools."

	controller displayTools! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ToolPaletteView class
	instanceVariableNames: ''!


!ToolPaletteView class methodsFor: 'instance creation'!

on: aModel
	^self new setModel: aModel! !


View subclass: #DrawingView
	instanceVariableNames: 'handles drawing '
	classVariableNames: 'CopyBuffer '
	poolDictionaries: ''
	category: 'HotDraw-Support'!
DrawingView comment:
'A DrawingView is the View component of a MVC triad.  The Drawing and DrawingController are the other components. A DrawingEditor also uses the DrawingView in a MVC triad.
Instance Variables : 
handles		<OrderedCollection>
	A Collection of the Handles currently in use in the drawing.  
drawing		<Drawing>
	the Drawing of the DrawingEditor on which this View is dependent.

Class Variables :
CopyBuffer	<Array | nil>
	An Array of figures cut from the Drawing.
'!


!DrawingView methodsFor: 'accessing'!

drawing
        ^drawing!

sensor
	^controller sensor! !

!DrawingView methodsFor: 'initialize-release'!

initialize
	super initialize.
	handles := OrderedCollection new!

release
	drawing removeDependent: self.
	model removeDependent: self.
	super release! !

!DrawingView methodsFor: 'drawing displaying'!

displayModelClipped: aRectangle 
	"Display my Drawing on an offscreen Pixmap then blast the Pixmap to the Screen."

	| clipRectangle gc1 gc2 aPixmap |

	gc1 := self graphicsContext.
	clipRectangle := gc1 clippingBounds intersect: (aRectangle rounded expandedBy: 2).
	clipRectangle extent > (0 @ 0) ifFalse: [^self].
	aPixmap := Pixmap extent: clipRectangle extent.
	aPixmap background: self backgroundColor.
	gc2 := aPixmap graphicsContext. "Should have been cleared by extent:"
	gc2 translateBy: clipRectangle origin negated.
	drawing displayOn: gc2.
	gc1
		copyArea: clipRectangle
		from: gc2
		sourceOffset: Point zero "clipRectangle origin negated"
		destinationOffset: Point zero.
	gc1 flush.
	gc1 medium screen sync.
	model currentTool notNil ifTrue: [model currentTool display]	"should have been in HotDraw, I think"!

displayObject: aFigure 
	aFigure changed.
	self repairDamage!

redisplayAll
	(drawing isNil)
		ifTrue: [^self].
	self displayModelClipped: (self graphicsContext clippingBounds).
	drawing undamage!

repairDamage
	(drawing isNil or: [drawing isDamaged not])
		ifTrue: [^self].
	self displayModelClipped: (self graphicsContext clippingBounds intersect: drawing damagedRegion).
	drawing undamage! !

!DrawingView methodsFor: 'handle displaying'!

hideHandle: aHandle 
	"Hide aHandle on myself by redrawing it in the background color. Dont refresh here so that 
	when many handles are hidden, we only have to refresh once."

	drawing damageRegion: aHandle displayBox!

hideHandles
	
	handles do: [:each | self hideHandle: each].
	self repairDamage!

showHandle: aHandle 
	aHandle displayOn: self graphicsContext at: aHandle origin!

showHandles
	
	handles do: [:each | self showHandle: each].! !

!DrawingView methodsFor: 'drawing accessing'!

addAndSelectAll: aCollection 
	self noSelections.
	drawing addAll: aCollection.
	aCollection do: [:each | each changed].
	self repairDamage.
	self selections: aCollection!

addFigure: aFigure 
	drawing add: aFigure.
	aFigure changed.
	self repairDamage!

addFigureBehindAll: aFigure 
	"Add figure behind all other figures."

	drawing addLast: aFigure.
	aFigure changed.
	self repairDamage!

bringToFront: aFigure 
	drawing bringToFront: aFigure.
	self displayObject: aFigure!

delete: aFigure 
	aFigure changed.
	drawing delete: aFigure.
	self repairDamage.
	self bjorn: 'this will need to redraw the entire drawing'!

figureAt: selectionPoint 
	drawing == nil ifTrue: [^nil].
	^handles detect: [:each | each containsPoint: selectionPoint]
		ifNone: [drawing figureAt: selectionPoint]!

figureAtCursor
	^self figureAt: self sensor cursorPoint!

figuresIn: aRectangle 
	^drawing figuresIn: aRectangle!

remove: aFigure 
self obsoleteMessage.
	self removeNoUpdate: aFigure.
	self repairDamage.!

removeNoUpdate: aFigure 
	"This deletes aFigure, but will not show up until view gets repaired. 
	Useful when you wish to delete many figures and have it appear 
	like one quick delete stage."

self obsoleteMessage.
	aFigure changed.
	drawing remove: aFigure!

sendToBack: aFigure 
	drawing sendToBack: aFigure.
	self displayObject: aFigure! !

!DrawingView methodsFor: 'selection accessing'!

isSelected: aFigure 
	handles detect: [:each | each isFor: aFigure]
		ifNone: [^false].
	^true!

noSelections
	self hideHandles.
	handles := OrderedCollection new!

removeSelections
	| aCollection |
	aCollection := self selections.
	aCollection do: [:each | each changed].
	drawing deleteAll: aCollection.
	self noSelections.
	self repairDamage.
	self bjorn: 'this will need to redraw the entire drawing'!

selectAll
	self selections: drawing figures!

selection: aFigure
        self selections: (Array with: aFigure)!

selectionContainsPoint: aPoint 
	self selections detect: [:each | each containsPoint: aPoint]
		ifNone: [^false].
	^true!

selectionRegion
	| aCollection |
	aCollection := self selections.
	aCollection isEmpty ifTrue: [^0 @ 0 extent: 0 @ 0].
	^aCollection inject: aCollection first displayBox into: [:sum :each | sum merge: each displayBox]!

selections
        ^IdentitySet new addAll: (handles collect: [:each | each owner]); yourself!

selections: aCollection 
	self noSelections.
	self addHandlesForAll: aCollection!

sortedSelections
	| selections |
	selections := self selections.
	^drawing figures select: [:each | selections includes: each]!

toggleSelection: aFigure 
	(self isSelected: aFigure)
		ifTrue: [self removeHandlesFor: aFigure]
		ifFalse: [self addHandlesFor: aFigure]!

toggleSelections: aCollection 
	aCollection do: [:each | self toggleSelection: each]! !

!DrawingView methodsFor: 'handle accessing'!

addHandle: aHandle 
	handles add: aHandle.
	self showHandle: aHandle!

addHandlesFor: aFigure 
	aFigure handles do: [:each | self addHandle: each]!

addHandlesForAll: aCollection
        aCollection do: [:each | self addHandlesFor: each]!

handlesFor: aFigure
        ^handles select: [:each | each isFor: aFigure]!

removeHandle: aHandle 
	handles remove: aHandle.
	self hideHandle: aHandle.
	self repairDamage!

removeHandlesFor: aFigure
        (self handlesFor: aFigure) do:
                [:each | self removeHandle: each]! !

!DrawingView methodsFor: 'editing'!

compose
	"Compose all the selected figures into a ComposedFigure."

	| aCollection aCF | 
self notYetImplemented.
	aCollection := self sortedSelections.
	aCollection isEmpty ifTrue: [^self].
	self noSelections.
	drawing removeAll: aCollection.
	aCF := CompositeFigure figures: aCollection.
	drawing figures isEmpty
		ifTrue: [drawing add: aCF]
		ifFalse: [drawing add: aCF behind: drawing figures last].
	aCF changed.
	self repairDamage.
	self selection: aCF!

copy
	| anArray cs |
	model currentTool canCutCopyPaste ifTrue: [^model currentTool copy].
	anArray := self sortedSelections asArray.
	anArray isEmpty
		ifFalse: 
			[CopyBuffer := Array new: 3.
			CopyBuffer at: 1 put: OrderedCollection new.
			cs := CoolDrawCopyState new initialize: anArray.
			anArray do: [:fig | (CopyBuffer at: 1)
					add: (cs copy: fig)].
			CopyBuffer at: 2 put: cs collectedConstraints.
			CopyBuffer at: 3 put: (drawing copyClustersUsing: cs).
			cs release]!

cut
	model currentTool canCutCopyPaste ifTrue: [^model currentTool cut].
	self copy.
	self removeSelections!

decompose
	| aCollection |
self notYetImplemented.
	aCollection := self selections.
	aCollection isEmpty ifTrue: [^self].
	self noSelections.
	self selections: (aCollection inject: Set new
			into: 
				[:sum :each | 
				each hideVisibleAreaIndicator.
				each changed.
				each release.
				drawing remove: each.
				each figures
					reverseDo: 
						[:aFigure | 
						drawing add: aFigure.
						aFigure removeDependent: each].
				sum addAll: each figures; yourself]).!

group
	| aCollection aGroup |
self notYetImplemented.
	aCollection := self sortedSelections.
	aCollection isEmpty ifTrue: [^self].
	self noSelections.
	drawing removeAll: aCollection.
	aGroup := GroupFigure figures: aCollection asArray onView: self.
	drawing add: aGroup. 
	aGroup changed.
	self repairDamage.
	self selection: aGroup!

paste
	|      tmp cs ps |
	model currentTool canCutCopyPaste ifTrue: [^model currentTool paste].
	CopyBuffer isNil
		ifFalse: 
			["Make a copy of the paste buffer."
			tmp := Array new: 3.
			tmp at: 1 put: OrderedCollection new.
			cs := CoolDrawCopyState new initialize: (CopyBuffer at: 1).
			(CopyBuffer at: 1) do: [:fig | (tmp at: 1)
					add: (cs copy: fig)].
			tmp at: 2 put: cs collectedConstraints.
			tmp at: 3 put: ((CopyBuffer at: 3) collect: [:clu | cs copy: clu]).
			cs release.

			"Insert all of the figures variables into the constraint graph."
			ps := CoolDrawPasteState new initialize: drawing.
			(tmp at: 1) do: [:fig | fig pasteUsing: ps].

			"Insert all of the constraints into the constraint graph."
			(tmp at: 2) do: [:cn | cn pasteUsing: ps].

			"Add all the clusters to the drawing"
			(tmp at: 3) do: [:clu | drawing addCluster: clu].

			ps release.
			self noSelections.

			"Insert all of the figures into the drawing."
			self addAndSelectAll: (tmp at: 1)]!

ungroup
	| aCollection |
self notYetImplemented.
	aCollection := self selections.
	aCollection isEmpty ifTrue: [^self].
	self noSelections.
	self selections: (aCollection inject: Set new
			into: 
				[:sum :each | 
				each release.
				drawing remove: each.
				each figures reverseDo: [:aFigure | drawing add: aFigure].
				sum addAll: each figures; yourself])! !

!DrawingView methodsFor: 'scrolling'!

scrollBy: aPoint 
	self obsoleteMessage.
	"drawing figures do: [:each | each translateBy: aPoint].
	self repairDamage.
	self graphicsContext flush.
	self graphicsContext medium screen sync"! !

!DrawingView methodsFor: 'animating'!

clearObject: aFigure while: aBlock
        "Handles must be hidden"
        aFigure willChange.
        aBlock value.
        aFigure changed.
        self repairDamage!

hideHandlesWhile: aBlock 
	self hideHandles.
	aBlock value.
	self showHandles!

step
	drawing notNil
		ifTrue: 
			[drawing step.
			self repairDamage]! !

!DrawingView methodsFor: 'displaying'!

displayOn: aGraphicsContext 
	| clip |
	clip := aGraphicsContext clippingBounds.
	self displayModelClipped: clip.
	self showHandles! !

!DrawingView methodsFor: 'controller accessing'!

defaultControllerClass
	"Answer the class of my default controller."

	^DrawingController! !

!DrawingView methodsFor: 'updating'!

update: aSymbol
	"My model has changed aSymbol.  Right now I don't care about it.  This is being implemented."! !

!DrawingView methodsFor: 'private'!

setDrawing: aDrawing 
	"Set my drawing to aDrawing and add me to the list of dependents of aDrawing."

	drawing := aDrawing.
	aDrawing addDependent: self! !

!DrawingView methodsFor: 'cooldraw'!

currentPlan
	^model currentPlan! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

DrawingView class
	instanceVariableNames: ''!


!DrawingView class methodsFor: 'instance creation'!

new
	^super new initialize!

on: aDrawingEditor 
	"Create a DrawingView (and its Controller) for aDrawingEditor.  Set the controller of each Tool to my controller."

	| aDrawingView |
	aDrawingView := self new initialize.
	aDrawingEditor tools do: [:each | each controller: aDrawingView controller].
	^aDrawingView model: aDrawingEditor! !

!DrawingView class methodsFor: 'buffer accessing'!

copyBuffer: figureArray
        CopyBuffer := figureArray! !


DialogView subclass: #CoolDrawDialogView
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'CoolDraw-Interface'!


!CoolDrawDialogView methodsFor: 'updating'!

update: anAspect with: aParameter from: anObject
	(anObject == model and: [(anAspect == #ok) or: [anAspect == #cancel]])
		ifTrue: [self closeChannel value: true]! !


!Rectangle class methodsFor: 'instance creation'!

fromUser: gridPoint phase: phasePoint 
	"Answer an instance of the receiver that is determined by having the 
	user designate the top left and bottom right corners. The gridding for 
	user selection is represented by gridPoint and phasePoint, where 
	gridPoint specifies the grid spacing and phasePoint specifies the 
	grid alignment."
	"Rectangle fromUser: 100@100 phase: 7@7."

	| originPoint |
	^Cursor crossHair
		showWhile: 
			[originPoint := Screen default
						dragShape: (0 @ 0 extent: gridPoint) asPointArray
						offset: 0 @ 0
						gridPhase: phasePoint
						gridSpacing: gridPoint
						boundedBy: nil
						whileButton: 0
						isDown: false.
			Screen default
				resizeRectangle: (Rectangle origin: originPoint extent: gridPoint)
				minimumExtent: gridPoint
				resizeIncrement: gridPoint
				boundedBy: nil
				whileButton: 0
				isDown: true]! !

Smalltalk organization changeFromString: '(''Magnitude-General'' #Character #Date #Magnitude #Time #TimeZone)
(''Magnitude-Numbers'' #ArithmeticValue #Double #Float #Fraction #Integer #LargeNegativeInteger #LargePositiveInteger #LimitedPrecisionReal #Number #Random #SmallInteger)
(''Collections-Abstract'' #ArrayedCollection #Collection #SequenceableCollection)
(''Collections-Unordered'' #Bag #Dictionary #IdentityDictionary #IdentitySet #Set #WeakDictionary)
(''Collections-Sequenceable'' #Interval #LinkedList #OrderedCollection #SortedCollection)
(''Collections-String Support'' #ByteEncodedString #ByteString #ByteSymbol #GapString #ISO8859L1String #MacString #TwoByteString #TwoByteSymbol)
(''Collections-Text'' #CharacterArray #String #Symbol #Text)
(''Collections-Arrayed'' #Array #ByteArray #IntegerArray #RunArray #WeakArray #WordArray)
(''Collections-Streams'' #InternalStream #PeekableStream #PositionableStream #ReadStream #ReadWriteStream #Stream #TextStream #WriteStream)
(''Collections-Support'' #Association #Link #LookupKey)
(''Graphics-Geometry'' #Bezier #Circle #EllipticalArc #Geometric #LineSegment #Point #Polyline #Rectangle #Spline)
(''Graphics-Visual Objects'' #BorderedWrapper #BoundedWrapper #CompositePart #DependentComposite #DependentPart #LayoutWrapper #TranslatingWrapper #VisualComponent #VisualPart #Wrapper)
(''Graphics-Geometrical Objects'' #FillingWrapper #GeometricWrapper #GraphicsAttributes #GraphicsAttributesWrapper #StrokingWrapper)
(''Graphics-Images'' #CachedImage #Depth16Image #Depth1Image #Depth24Image #Depth2Image #Depth32Image #Depth4Image #Depth8Image #Image #OpaqueImage)
(''Graphics-Palettes'' #ColorPalette #CoveragePalette #FixedPalette #InverseColorMap #InverseColorMapInitializer #MappedPalette #MonoMappedPalette #Palette)
(''Graphics-Color Rendering'' #DitherUpTo4 #ErrorDiffusion #ImageRenderer #LuminanceBasedColorPolicy #NearestPaint #OrderedDither #PaintPolicy #PaintRenderer)
(''Graphics-Fonts'' #ByteCharacterEncoder #CharacterAttributes #CharacterComposer #CharacterEncoder #DeviceFont #FontDescription #FontDescriptionBundle #FontPolicy #ImplementationFont #LargeCharacterEncoder #MacFont #MSWindowsFont #SyntheticFont #TextAttributes #VariableSizeTextAttributes #XFont)
(''Graphics-Text Scanning'' #CharacterBlockScanner #CharacterScanner #CompositionScanner #DisplayScanner #TextMeasurer)
(''Graphics-Text Support'' #CharacterBlock #ComposedText #DispatchTable #LineInformationTable #OptimizedLineInformationTable #TextLineInterval #TextLines #TextList)
(''Graphics-Support'' #ColorValue #CoverageValue #GraphicsContext #Paint #Pattern #RasterOp #SimplePaint #SPActiveLines #SPFillLine #SPSortedLines)
(''Kernel-Objects'' #Boolean #False #Model #Object #True #UndefinedObject #UninterpretedBytes)
(''Kernel-Classes'' #Behavior #Class #ClassBuilder #ClassDescription #Metaclass)
(''Kernel-Methods'' #BlockClosure #BlockContext #CompiledBlock #CompiledCode #CompiledMethod #Context #InstructionClient #InstructionPrinter #InstructionStream #MarkedMethod #Message #MessageSend #MethodContext)
(''Kernel-Processes'' #Delay #Process #ProcessorScheduler #RecursionLock #Semaphore #SharedQueue)
(''Kernel-Exception Handling'' #Exception #HandlerCollection #Signal #SignalCollection)
(''Kernel-Support'' #ClassCategoryReader #ClassOrganizer #MethodDictionary #RemoteString #SourceCodeStream #SourceFileManager #SystemOrganizer)
(''Interface-Framework'' #Controller #ControlManager #ScheduledWindow #StandardSystemController #TranslatingSensor #View)
(''Interface-Widgets'' #ActionButton #BorderDecorationPolicy #BorderDecorator #Button #FixedThumbScrollbar #Scrollbar #ScrollbarController #SmalltalkBorderDecorationPolicy #SmalltalkWidgetPolicy #WidgetPolicy #WidgetSpecification)
(''Interface-Dialogs'' #BooleanWidgetView #CompositeView #DialogCompositeController #DialogController #DialogView #FractionalWidgetView #LabeledBooleanView #TextItemEditor #WidgetController)
(''Interface-Lists'' #ListController #ListView #SelectionInListController #SelectionInListView #SelectionSetInListController #SelectionSetInListView)
(''Interface-Text'' #ComposedTextView #DebuggerController #DebuggerTextView #ParagraphEditor #TextCollector #TextCollectorView #TextController #TextView)
(''Interface-Menus'' #MenuTracker #PopUpMenu)
(''Interface-Support'' #AutoScrollingView #Border #ControllerWithMenu #Icon #InputSensor #Layout #LayoutFrame #LayoutOrigin #LookPreferences #NoController #PassivityWrapper #PluggableAdaptor #ScrollValueHolder #ScrollWrapper #ValueHolder #ValueModel)
(''Tools-Programming'' #Browser #Debugger #Explainer #HierarchyBrowser #MethodListBrowser #NotifierController #NotifierView #ProcessHandle #SyntaxError)
(''Tools-Inspector'' #ChangeSetInspector #ContextInspector #DictionaryInspector #Inspector #OrderedCollectionInspector #SequenceableCollectionInspector)
(''Tools-Changes'' #ChangeList #ChangeListController #ChangeListView)
(''Tools-Misc'' #FileBrowser #LauncherController #LauncherView #Project)
(''System-Changes'' #Change #ChangeScanner #ChangeSet #ClassChange #ClassCommentChange #ClassDefinitionChange #ClassOtherChange #ClassRelatedChange #MethodChange #MethodDefinitionChange #MethodOtherChange #OtherChange)
(''System-Compiler-Program Objects'' #ArithmeticLoopNode #AssignmentNode #BlockNode #CascadeNode #ConditionalNode #LeafNode #LiteralNode #LoopNode #MessageNode #MethodNode #ParameterNode #ProgramNode #ProgramNodeBuilder #ReturnNode #SequenceNode #SimpleMessageNode #StatementNode #ValueNode #VariableNode)
(''System-Compiler-Names and Scopes'' #ArgumentVariable #InstanceVariable #LocalScope #LocalVariable #NameScope #NullScope #PseudoVariable #ReceiverVariable #RemoteVariable #StaticScope #StaticVariable #TemporaryVariable #UndeclaredVariable #VariableDefinition)
(''System-Compiler-Support'' #ByteCodeReadWriteStream #ByteCodeStream #CodeLabel #CodeStream #CompilerErrorHandler #DeferredBlock #DefineOpcodePool #InteractiveCompilerErrorHandler #LoggingCompilerErrorHandler #MethodNodeHolder #NonInteractiveCompilerErrorHandler #ProgramNodeEnumerator #ReadBeforeWrittenTester #RecodeStream #ScannerTable #SilentCompilerErrorHandler)
(''System-Compiler-Public Access'' #CCompatibleParser #CodeRegenerator #Compiler #Decompiler #Parser #Scanner #SmalltalkCompiler)
(''System-Support'' #DependentsCollection #KeyboardEvent #MemoryPolicy #ObjectMemory #SystemDictionary #WeakKeyAssociation)
(''System-Binary Storage'' #BinaryObjectStorage #BOSSBytes #BOSSCompiledCodeHolder #BOSSContents #BOSSDebugReader #BOSSReader #BOSSReaderMap #BOSSRegisteredObject #BOSSTransporter #BOSSWriter)
(''OS-Window System'' #Cursor #DisplaySurface #GraphicsHandle #InputState #Mask #Pixmap #Screen #UnmappableSurface #Window #WindowSensor)
(''OS-Streaming'' #BufferedExternalStream #ExternalConnection #ExternalReadAppendStream #ExternalReadStream #ExternalReadWriteStream #ExternalStream #ExternalWriteStream #FileConnection #IOBuffer #PositionalIOBuffer)
(''OS-Support'' #Filename #HandleRegistry #IOAccessor #OSErrorHolder #OSHandle)
(''OS-Unix'' #UnixDiskFileAccessor #UnixFilename #UnixIOAccessor)
(''OS-Dos'' #DosDiskFileAccessor #DosFilename #DosIOAccessor)
(''OS-Mac'' #MacDiskFileAccessor #MacFilename #MacIOAccessor)
(''HotDraw-Framework'' #Drawing #DrawingEditor #Figure #Handle #Tool)
(''HotDraw-Constraints'' #Constraint #Locator #MultiheadedConstraint #NumberHolder #PositionConstraint)
(''HotDraw-Figures'' #ArrowFigure #CachedFigure #CompositeFigure #DependentLineFigure #EllipseFigure #FixedTextFigure #GroupFigure #ImageFigure #LineFigure #NumberFigure #PolylineFigure #RectangleFigure #TextFigure #WrapperFigure)
(''HotDraw-Support'' #DrawingController #DrawingView #ToolPaletteController #ToolPaletteView)
(''HotDraw-Handles'' #ConnectionHandle)
(''HotDraw-Tools'' #CreationTool #DrawingActionTool #FigureActionTool #ScrollingTool #SelectionTool #TextTool #WrapperCreationTool)
(''CoolDraw-Framework'' #CoolDrawActiveVariable #CoolDrawingEditor #CoolDrawPlan #CoolDrawPoint #CoolDrawRectangle #CoolDrawVariable)
(''CoolDraw-Handles'' #SelectionTrackHandle #TrackHandle)
(''CoolDraw-Tools'' #ConstraintCreationTool #ConstraintDeletionTool #CoolDrawStartupTool)
(''CoolDraw-Support'' #CoolDrawConstraintCluster #CoolDrawCopyState #CoolDrawHandleDatum #CoolDrawPasteState #CoolDrawStartupPicturesEditor)
(''CoolDraw-Interface'' #CoolDrawConstraintDialog #CoolDrawDeleteDialog #CoolDrawDialogView #ImageView)
(''SkyBlue-Basics'' #SkyBlueAbstraction #SkyBlueConstraint #SkyBlueConstraintGraph #SkyBlueConstraintState #SkyBlueMark #SkyBlueMethod #SkyBluePlan #SkyBlueStrength #SkyBlueVariable #SkyBlueVariableState)
(''SkyBlue-Examples'' #ColbaltBlueMVCConstraintGraph #SkyBlueDemoBrowser #SkyBlueMVCConstraintGraph #SkyBlueMVCVariable)
(''ColbaltBlue-Basics'' #ColbaltBlueChecker #ColbaltBlueConstraintGraph #ColbaltBlueConstraintGraphState #ColbaltBlueConstraintState #ColbaltBlueNonuniqueConstraint #ColbaltBluePlan #ColbaltBluePlanThunk)
'!


ConstraintCreationTool initialize!

SkyBlueMark initialize!

Handle initialize!

TextFigure initialize!

SkyBlueStrength initialize!

SelectionTool initialize!

ConstraintDeletionTool initialize!

Figure initialize!

