// simpleGLmain.cpp (Rob Farber)
// http://www.drdobbs.com/architecture-and-design/222600097

#include "window.hh"

// open gl
#include "open_gl.hh"

#include <assert.h>

// cuda utility libraries
#include <cuda_runtime.h>
#include <cutil.h>
#include <cutil_inline.h>
#include <cutil_gl_inline.h>
#include <cutil_gl_error.h>
#include <cuda_gl_interop.h>
// #include <rendercheck_gl.h>

// The user must create the following routines:
// CUDA methods
extern void initCuda(int argc, char** argv);
extern void renderCuda(int);

// callbacks
extern void display();
extern void keyboard(unsigned char key, int x, int y);
extern void mouse(int button, int state, int x, int y);
extern void motion(int x, int y);
extern void reshape(int width, int height);

unsigned int timer = 0; // a timer for FPS calculations

/// First initialize OpenGL context, so we can properly set the GL
/// for CUDA. NVIDIA notes this is necessary in order to achieve
/// optimal performance with OpenGL/CUDA interop.	use command-line
/// specified CUDA device, otherwise use device with highest Gflops/s
void initCuda(int argc, char** argv)
{	
	if( cutCheckCmdLineFlag(argc, (const char**)argv, "device") ) {
		cutilGLDeviceInit(argc, argv);
	} else {
		cudaGLSetGLDevice( cutGetMaxGflopsDeviceId() );
	}
}

/// Set OpenGL state as appropriate after the window has been created.
void initGL() {
	// check for necessary OpenGL extensions
	glewInit();
	assert(glewIsSupported( "GL_VERSION_2_0 " ));	

	// a little more opengl initialization
	glClearColor(0.0, 0.0, 0.0, 1.0);
	glDisable(GL_DEPTH_TEST);

	// Enable Texturing - VERY IMPORTANT
	glEnable(GL_TEXTURE_2D);
}

// Simple method to display the Frames Per Second in the window title
void computeFPS()
{
	static int fpsCount=0;
	static int fpsLimit=100;

	fpsCount++;
	
	if (fpsCount == fpsLimit) {
		char fps[256];
		float ifps = 1.f / (cutGetAverageTimerValue(timer) / 1000.f);
		sprintf(fps, "15-668 - Project 2: %3.1f fps ", ifps);	
		
		glutSetWindowTitle(fps);
		fpsCount = 0; 
		
		cutilCheckError(cutResetTimer(timer));	
	}
}

void fpsDisplay()
{
	cutilCheckError(cutStartTimer(timer));	
	
	display();
	
	cutilCheckError(cutStopTimer(timer));
	computeFPS();
}

void createWindow(int argc, char ** argv)
{
	// Create the CUTIL timer
	cutilCheckError( cutCreateTimer( &timer));
	
	//Steps 1-2: create a window and GL context (also register callbacks)
	glutInit(&argc, argv);
	glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE);
	glutInitWindowSize(512, 512); // default dimensions
	glutCreateWindow("15-668 - Project 2");

	// register callbacks
	glutDisplayFunc(fpsDisplay);
	glutKeyboardFunc(keyboard);
	glutMotionFunc(motion);
	glutMouseFunc(mouse);
	glutReshapeFunc(reshape);
	
	// init opengl and cuda
	initGL();
	initCuda(argc, argv);
	CUT_CHECK_ERROR_GL();
}

void reshapeWindow(unsigned int width, unsigned int height)
{
	glutReshapeWindow(width, height);
}

void startApplication(int argc, char ** argv)
{
	// start rendering mainloop
	glutMainLoop();
	
	// clean up
	cudaThreadExit();
	cutilExit(argc, argv);
}
