/// \file Holds the data structure associated with an OpenGL pixel buffer object.
/// See http://www.songho.ca/opengl/gl_pbo.html.

#ifndef PIXEL_BUFFER_OBJECT_HH
#define PIXEL_BUFFER_OBJECT_HH

#include "open_gl.hh"
#include <cuda_runtime.h>

/// Combines an OpenGL pixel buffer object with a texture. The pixel buffer supports two access
/// modes: mapping to host memory (for transfer), or to device memory, for cuda write. The data
/// in this pixel buffer can be bound to the associated OpenGL texture for display.
class PixelBufferObject
{
public:
	/// Creates a pixel buffer object from the specified .png image
	/// \param file_name [in] the name of the .png image to load
	/// \return a new pixel buffer object representing the image
	static PixelBufferObject *
	loadImage(const char * file_name);
	
	/// Constructor
	PixelBufferObject(
		const unsigned int & width,   ///< width
		const unsigned int & height   ///< height
		);
		
	/// Destructor
	~PixelBufferObject();
	
	/// Gets the width of the Pixel Buffer
	const unsigned int & getWidth() const;

	/// Gets the width of the Pixel Buffer
	const unsigned int & getHeight() const;
	
	/// Gets the dimensions of the pixel buffer.
	const uint2 & getSize() const;
	
	/// Gets the OpenGL identifier for this pixel buffer object
	const GLuint & getID() const;
	
	/// When creating a memory map, it can either be on the device, or it can be on cuda
	enum MemoryMapType
	{
		HOST_MEMORY = 0, ///< Map the pbo to host memory for CPU operations.
		DEVICE_MEMORY    ///< Map the pbo to device memory for CUDA operations.
	};
	
	/// A memory map to this pixel buffer object. This class uses the
	/// RAII paradigm: when it is intantiated, the memory map opens,
	/// and you can start to access the mapped memory for this pixel
	/// buffer object. When it is deleted, the memory map is closed and
	/// OpenGL can begin asynchronous operations on this pixel buffer
	/// object.
	class MemoryMap
	{
		friend class PixelBufferObject;
		
	private:
		/// Starts the memory map.
		/// \param parent    The pixel buffer object to map.
		/// \param map_type  Are we mapping to host or device memory?
		MemoryMap(PixelBufferObject & parent, MemoryMapType map_type);
		
		/// The pixel buffer object to map.
		PixelBufferObject & m_parent;
		
		/// Stores whether we're mapping to host or device memory.
		const MemoryMapType m_map_type;
		
		/// Pointer to the mapped memory on the host or device.
		uchar4 * m_pointer;
		
	public:
		/// Ends the memory map.
		~MemoryMap();
		
		/// Returns the device pointer which can be mapped by CUDA.
		uchar4 * getPointer();		
	};
		
	/// Resource allocator for CPU write through host memory mapping.
	MemoryMap mapHostMemory();

	/// Resource allocator for CUDA write through device memory mapping.
	MemoryMap mapDeviceMemory();
		
	/// Binds the texture for OpenGL rendering. If necessary,
	/// DMAs data from the pixel buffer object to a texture.
	void bindTexture();
			
private:
	
	const uint2 m_size;             ///< dimensions of the buffer
	GLuint m_pbo_id;               ///< the id of this pixel buffer object
	GLuint m_texture_id;           ///< the id of the associated texture
	bool m_dirty;                  ///< true when pbo and texture are out of synch
};

#endif /* end of include guard: PIXEL_BUFFER_OBJECT_HH */
