/* $Id$ */

/*
 *  Papadimitriou Spiros
 *  spapadim+@cs.cmu.edu
 *
 *  CS213 - Lab assignment 3
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>
#include <string.h>

#include "ftime.h"
#include "memlib.h"
#include "malloc.h"

extern char *tracefiles[];
#include "tracefiles.h"

typedef struct range_t {
    struct range_t *next;
    char *lo, *hi;
} range_t;

typedef struct {
    enum {ALLOC, FREE} type;
    long index;
    long size;
} memop_t;

typedef struct {
    long num_blocks;
    long num_ops;
    long suggested_heap_size;
    memop_t *ops;
    char **blocks;
} trace_t;


/* Command-line options */
int verbose = 1;

/*****************
 * Range routines
 *****************/

static range_t *ranges = NULL;

/* Check if a pointer is 8-byte aligned */
#define IS_ALIGNED(p)  ((((unsigned long)(p))%8) == 0)

int add_range (char *lo, long size)
{
    char *hi = lo + size - 1;
    range_t *p;

    assert(size > 0);

    /* Check alignment */
    if (!IS_ALIGNED(lo)) {
        fprintf(stderr, "Misaligned region returned\n");
        if (verbose)
            fprintf(stderr, "Address: %p\n", lo);
        return 0;
    }

    /* Region lies within heap */
    if (lo < dseg_lo || lo > dseg_hi || hi < dseg_lo || hi > dseg_hi) {
        fprintf(stderr, "Region lies outside heap area\n");
        if (verbose) {
            fprintf(stderr, "Region: %p - %p\n", lo, hi);
            fprintf(stderr, "Heap: %p - %p\n", dseg_lo, dseg_hi);
        }
        return 0;
    }

    /* Region does not overlap any other region */
    for (p = ranges;  p != NULL;  p = p->next) {
        if ((lo >= p->lo && lo <= p-> hi) ||
            (hi >= p->lo && hi <= p->hi)) {
            fprintf(stderr, "Region overlap detected\n");
            if (verbose) {
                fprintf(stderr, "Region 1: %p - %p\n", lo, hi);
                fprintf(stderr, "Region 2: %p - %p\n", p->lo, p->hi);
            }
            return 0;
        }
    }

    /* Clobber region (zero miswritten control records) */
    bzero(lo, size);

    p = (range_t *)malloc(sizeof(range_t));
    p-> next = ranges;
    p->lo = lo;
    p->hi = hi;
    ranges = p;
    return 1;
}

void remove_range (char *lo)
{
    range_t *p, **prevpp = &ranges;

    for (p = ranges;  p != NULL; p = p->next) {
        if (p->lo == lo) {
            int size = p->hi - p->lo + 1;

            *prevpp = p->next;
            
            /* Clobber region again (zero 
             * miswritten control records) */
            bzero(p->lo, size);

            free(p);
            break;
        }
        prevpp = &(p->next);
    }
}

void clear_ranges (void)
{
    range_t *p, *pnext;

    for (p = ranges;  p != NULL;  p = pnext) {
        pnext = p->next;
        free(p);
    }

    ranges = NULL;
}


/*********************
 * Tracefile routines
 *********************/

trace_t *read_trace (char *filename)
{
    FILE *tracefile;
    trace_t *trace = (trace_t *)malloc(sizeof(trace_t));
    long num_blocks, num_ops;

    tracefile = fopen(filename, "r");
    if (tracefile != NULL) {
        char type[10];
        unsigned index, size;
        unsigned max_index = 0;
        unsigned op_index;

        fscanf(tracefile, "%ld", &(trace->suggested_heap_size));
        fscanf(tracefile, "%ld", &num_blocks);
        fscanf(tracefile, "%ld", &num_ops);
        trace->num_blocks = num_blocks;
        trace->num_ops = num_ops;

        trace->ops = (memop_t *)malloc(num_ops * sizeof(memop_t));
        trace->blocks = (char **)malloc(num_blocks * sizeof(char *));
        index = 0;
        op_index = 0;
        while (fscanf(tracefile, "%s", type) != EOF) {
            switch(type[0]) {
            case 'a':
                fscanf(tracefile, "%u %u", &index, &size);
                trace->ops[op_index].type = ALLOC;
                trace->ops[op_index].index = index;
                trace->ops[op_index].size = size;
                max_index = (index > max_index) ? index : max_index;
                break;
            case 'f':
                fscanf(tracefile, "%ud", &index);
                trace->ops[op_index].type = FREE;
                trace->ops[op_index].index = index;
                break;
            default:
                fprintf(stderr, "Bogus type character\n");
                exit(-1);
            }
            op_index++;
        }
        fclose(tracefile);

        assert(max_index == num_blocks);
        assert(num_ops == op_index);

        if (verbose) {
            printf("Read tracefile: %s\n", filename);
            printf("Blocks: %ld\n", num_blocks);
            printf("Operations: %ld\n\n", num_ops);
        }

        return trace;
    }

    return NULL;
}

void free_trace (trace_t *trace)
{
    free(trace->ops);
    free(trace->blocks);
    free(trace);
}

/********************
 * Testing functions
 ********************/

/* Test correctness */

void correctness (void)
{
    char **fname;

    for (fname = tracefiles;  **fname;  fname++) {
        trace_t *trace;
        long i;

        trace = read_trace(*fname);

        mem_reinit(0);
        clear_ranges();
        mm_init();

        for (i = 0;  i < trace->num_ops;  i++)
            switch (trace->ops[i].type) {
            case ALLOC:
              {
                int index = trace->ops[i].index;
                int size = trace->ops[i].size;

                char *p = mm_malloc(size);

                /* Test returned region (misalignment,
                 * overlap, miswritten control data) */
                if (!add_range(p, size)) {
                    fprintf(stderr, "Failed correctness test!\n");
                    if (verbose)
                        fprintf(stderr, "Operation: %ld (out of %ld)\n", i, trace->num_ops);
                    fprintf(stderr, "Please use gdb to inspect the coredump.\n\n");
                    abort();
                }

                /* Remember region */
                trace->blocks[index] = p;
                break;
              }
            case FREE:
              {
                int index = trace->ops[i].index;
                char *block = trace->blocks[index];

                mm_free(block);

                /* Remove region from list */
                remove_range(block);
                break;
              }
            }

        free_trace(trace);
    }

    printf("Correctness test passed!\n");
}

/* Test efficiency */

void efficiency (void)
{
}

/* Time speed */

static trace_t *trace;

void speed_funct (void)
{
    long i;

    mem_reinit(0);
    mm_init();

    for (i = 0;  i < trace->num_ops;  i++)
        switch (trace->ops[i].type) {
        case ALLOC:
          {
            int index = trace->ops[i].index;
            int size = trace->ops[i].size;

            char *p = mm_malloc(size);
            trace->blocks[index] = p;
            break;
          }
        case FREE:
          {
            int index = trace->ops[i].index;
            char *block = trace->blocks[index];

            mm_free(block);
            break;
          }
        }
}

double speed (void)
{
    char **fname;
    double total_time = 0.0;

    for (fname = tracefiles;  **fname;  fname++) {
        double time;

        trace = read_trace(*fname);
        time = ftime(speed_funct, 0.0);
        total_time += time;
        free_trace(trace);
    }

    return total_time;
}

/*******
 * Main
 *******/

int main (int argc, char *argv[])
{
    char c;

    double total_time;

    while ((c = getopt(argc, argv, "f:hvq")) != EOF)
        switch (c) {
        case 'f':
            break;
        case 'v':
            verbose = 1;
            break;
        case 'q':
            verbose = 0;
            break;
        case 'h':
            fprintf(stderr, "Usage: malloc [-ffile] [-h] [-v|-q]\n");
            exit(0);
        default:
            fprintf(stderr, "Unknown option; use -h for help\n");
            exit(1);
        }

    /* Print team info */
    printf("Teamname: %s\n", team.team);
    printf("Member 1: %s <%s>\n", team.name1, team.email1);
    if (*team.name2 || *team.email2)
        printf("Member 2: %s <%s>\n", team.name2, team.email2);
    printf("\n");

    /* Initialize memlib */
    mem_init();

    /* Perform tests */
    correctness();
    efficiency();
    total_time = speed();
    printf("Total time: %f\n", total_time);

    exit(0);
}

