/* $Id$ */

/*
 *
 *  CS213 - Lab assignment 3
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

#define DEBUG 0
#define NO_OF_LAST_BINS 4

team_t team = {
    /* Team name to be displayed on webpage */
    "<font size=1>Sweet Little Girl</font>",
    /* First member full name */
    "Ming Fai Wong",
    /* First member email address */
    "mingfai",
    /* Second member full name (leave blank if none) */
    "Rhys L Wong",
    /* Second member email address (blank if none) */
    "rlw"
};


/*********** define structs needed to hold control data for allocator ****************/

/*	+----------------------+
	| prevsize     | inuse |
	+--------------+-------+
	| size         | inuse |
	+--------------+-------+
	|//////////////////////|
	|///USER DATA//////////|
	+----------------------+--
	| prevsize     | inuse |
	+--------------+-------+
	| size         | inuse |
	+--------------+-------+
	|     prev pointer     |
	+----------------------+
	|     next pointer     |
	+----------------------+

	We use a segregated-list best-fit algorithm. Memory blocks are divided
into chunks with boundary tags to facilitate traversal and coalescing. Coalescing
is immediate. There are 128 size classes, approximately logarithmically spaced to
favour small chunks. Each size class strings together chunks of approximately the
same size, in a circular doubly-linked list.

	Each size class is called a bin, and the chunks linked to each bin are
SORTED such that smallest chunks are at the front and larger chunks at the back.
This facilitates search because the first satifying chunk will be the best fit.

	While search starts from the front of the list, freed chunks are added to
the back of the list. The effect of this is that if two chunks have the same size,
the most recently split chunk will be at the back. This approximates a least-
recently-used algorithm, giving all chunks equal opportunity to be coalesced with
neighbours before being reallocated.

	The topmost chunk is treated as a special case, and it contains the slack
between the highest allocated chunk and dseg_hi. If searching through the free
lists fails to find a suitably sized chunk, only then do we consider the topmost
chunk. This helps to constrain heap growth and achieve high utilization.

	Another method for fragmentation preventation and enhancing memory
utilization is predictive preallocation. If a certain small size class is very
popular and there are many requests for chunks of this size, then to avoid 
fragmentation, chunks of this size are contiguously preallocated from the
topmost chunk. This is so that when they are freed they can be coalesced into a
larger chunk instead of leaving small holes all over the memory arena.

*/

struct memchunk;
typedef struct memchunk *memchunkptr;

struct binptr {
	memchunkptr next;
};	

struct memchunk {
	int prevsize;		/* size of previous chunk */
	int size;		/* size of this chunk */
	memchunkptr prev;
	memchunkptr next;	 /* pointers to previous and next chunks in free list */
};

struct globalInfo {
	memchunkptr top;		/* pointer to topmost chunk */
	struct binptr bin[128];	/* the bins */
        short lastBin[NO_OF_LAST_BINS];
        short lastBinIndex;        
};
typedef struct globalInfo *globalsptr;


/* routine which given the request size, returns which bin it belongs to */

#define bin_num(sz) (                           \
	((sz>>9) ==    0) ?       (sz>> 3) :    \
	((sz>>9) <=    4) ?  56 + (sz>> 6) :    \
	((sz>>9) <=   20) ?  91 + (sz>> 9) :    \
	((sz>>9) <=   84) ? 110 + (sz>>12) :    \
	((sz>>9) <=  340) ? 119 + (sz>>15) :    \
	((sz>>9) <= 1364) ? 124 + (sz>>18) : 126	)



/* print out a memory map of the heap */
void printMemoryMap(memchunkptr curBlock) {

  // debug
  if ( DEBUG < 2 ) return;

  printf( "\nMemory Map\n==========\n  dseg_lo: %p  dseg_hi:  %p", dseg_lo, dseg_hi );

  while ( (char *) curBlock <= dseg_hi ) {

    printf( "\nBlock from %p to %p\n", curBlock, 
	    (void *) ((unsigned) curBlock + (curBlock->size & ~1) - 1) );
    printf( "  previous block size: %d (%s)", curBlock->prevsize,  
	    (((curBlock->size & 1)==0) ? "free" : "allocated") );
    printf( "  current block size: %d\n",  curBlock->size & ~1 );
    printf( "  prev: %p\n  next: %p\n", curBlock->prev, curBlock->next );
    curBlock = (memchunkptr) ((unsigned) curBlock + (curBlock->size & ~1));

  }

}


int mm_init (void)
{

	globalsptr globals;
	memchunkptr top;
	int controlsize;	/* amt. space taken up by control data */
	int i;

	/* declare one page to hold control data initially */
	if (!mem_sbrk( mem_pagesize() )) return -1;

	controlsize = (sizeof(*globals) + 7) & (~7);

	globals = (globalsptr)dseg_lo;	/* declare the global control data struct */

	for ( i = 0; i < 128; ++i ) {
		globals->bin[i].next = (memchunkptr)(((char *)&(globals->bin[i])) - 12);
	}
	/* the bins can now be referenced like memchunks -- pointers start
		 from 3rd word within structure */

	/* initialize the topmost chunk with the remaining slack */
	top = globals->top = (memchunkptr)(dseg_lo + controlsize);	
	top->size = 1;			/* always set previous block used */
	top->size |= mem_pagesize() - sizeof(*globals);	

	for ( i = 0; i < NO_OF_LAST_BINS; i++ ) globals->lastBin[i] = -1;
	globals->lastBinIndex = 0;

	// debug
	if ( DEBUG > 0 ) {
	  printf("** init **\n");
	  printMemoryMap(top);
	}

	return 0;
}

/* mask out the inuse bit to return the size fields of a chunk (including overhead) */
#define chunksize(chunkptr) ((chunkptr->size)&(~1))
#define prevchunksize(chunkptr) ((chunkptr->prevsize)&(~1))

/* return pointers to next and previous chunks in memory */
#define nextchunk(chunkptr) ((memchunkptr)(((char *)chunkptr)+chunksize(chunkptr)))
#define prevchunk(chunkptr) ((memchunkptr)(((char *)chunkptr)-prevchunksize(chunkptr)))

/* given a chunk, return pointer to start of user memory (ie, skip past overhead) */
#define usrmemptr(chunkptr) (((void *)chunkptr) + 8)

/* given a bin number, returns a pointer to the bin, cast into memchunkptr format */
#define bin2chunk(n) ( (memchunkptr)(((char *)&(globals->bin[n])) - 12) )

/* given a number, returns the value of the least significant bit */
#define lsb(n) ( n & 1 )

/* given a chunk, mark the next block's size bit to indicate that this block 
   is allocated */
#define mark_chunk_allocated(curr_chunk) ((nextchunk(curr_chunk))->size |= 1)

/* given a chunk, mark the next block's size bit to indicate that this block 
   is free */
#define mark_chunk_free(curr_chunk) ((nextchunk(curr_chunk))->size &= ~1)

/* debugging routines

void printchunk (memchunkptr curr_chunk) {
	printf ("[chunk addr=%d prevsize=%d size=%d prevptr=%d nextptr=%d]\n", curr_chunk, curr_chunk->prevsize, curr_chunk->size, curr_chunk->prev, curr_chunk->next);
}

void printbin (memchunkptr binptr) {

	memchunkptr curr_chunk;

	for ( curr_chunk = binptr->next; curr_chunk != binptr;
			curr_chunk = curr_chunk->next ) {
		printf ("\t : ");
		printchunk(curr_chunk);
	}
	printf("--end of binlist--\n");
}
*/

/* remove a chunk from the free list */
void chunk_unlink(memchunkptr chunkptr, memchunkptr binptr)
{
	memchunkptr prevnode = chunkptr->prev;
	memchunkptr nextnode = chunkptr->next;
	prevnode->next = chunkptr->next;
	if ( nextnode != binptr ) nextnode->prev = chunkptr->prev;
}

/* add a chunk to the free list */
void chunk_link(memchunkptr binptr,memchunkptr ptr,int prevsize,int size)
{
	/* we add from the back of the free list */
	memchunkptr curr_chunk;	
	memchunkptr lastchunkinbin;

	/* find the last chunk in this bin */
	for ( lastchunkinbin = binptr; lastchunkinbin->next != binptr; 
	lastchunkinbin = lastchunkinbin->next ) {}

	/* traverse until we find a chunk smaller than this one */
	for (curr_chunk = lastchunkinbin;	
			curr_chunk != binptr && chunksize(curr_chunk) > size;
			curr_chunk = curr_chunk->prev ) { }

	/* and we add -behind- the chunk we found */
	ptr->next = curr_chunk->next;
	ptr->prev = curr_chunk;	
	curr_chunk->next = ptr;
	if (ptr->next != binptr) ptr->next->prev = ptr;

	ptr->prevsize = prevsize;
	ptr->size = (size|1);	/*our predecessor is ALWAYS allocated */

	/* set next chunk's prevsize to reflect NOT in use!!!!! */
	(nextchunk(ptr))->prevsize = size;
	mark_chunk_free(ptr);
}

void *mm_malloc (size_t size)
{
	globalsptr globals;
	int curr_bin;
	memchunkptr curr_chunk;
	memchunkptr head;
	memchunkptr splitbin;
	int slack;
	int temp;
	int requestedSize = size;
	int count;

	if (!size) return 0;

	size += 4;	/* expand the request size to account for 8-byte header */
	size = (size + 7) & (~7);	/* pad up to multiple of 8 bytes */

	// debug
	if ( DEBUG > 0 ) {
	  printf("\n** malloc (modified size: %d) **\n", size);
	}

	globals = (globalsptr)dseg_lo;

	/* keep track of the bins that were accessed */
	globals->lastBin[globals->lastBinIndex++] = bin_num(size);
	globals->lastBinIndex %= NO_OF_LAST_BINS;

	if ( size < 512 ) {	/* optimize for small bins - no search necessary */

		curr_bin = size >> 3;
		head = bin2chunk(curr_bin);
		curr_chunk = head->next;

		/* since the next bin contains chunks only 8 bytes larger, no need
			to consider split even if we spill over into the next bin */
		if (curr_chunk == head) {
			head = (memchunkptr)(((char *)head) + 4);
			curr_chunk = head->next;
		}

		if (curr_chunk != head) { /* allocate, no need to split */
			chunk_unlink(curr_chunk, head);
			mark_chunk_allocated(curr_chunk);

			// debug
			if ( DEBUG > 0 ) {
			  printf( ">> allocate, no need to splite, returning: %p\n",
				  usrmemptr(curr_chunk) );
			  printMemoryMap((memchunkptr) 0x400b5210);
			}

			return usrmemptr(curr_chunk);

		}
		curr_bin += 2;	/* we've searched 2 bins. fallback to normal search */

	} else {

		curr_bin = bin_num( size );
	}

	head = bin2chunk(curr_bin);
	for ( ; curr_bin < 128; curr_bin++ ) {

		for ( curr_chunk = head->next;
			curr_chunk != head; curr_chunk = curr_chunk->next ) {

			slack = chunksize(curr_chunk) - size;
			if ( slack >= 16 ) { 		/* must split */
				chunk_unlink(curr_chunk, head);
				curr_chunk->size = size | lsb(curr_chunk->size);
				mark_chunk_allocated(curr_chunk);

				splitbin = bin2chunk(bin_num(slack));
				chunk_link(splitbin,(memchunkptr)(((char *)curr_chunk) + size),size,slack);
				
				// debug
				if ( DEBUG > 0 ) {
				  printf( ">> must split, returning: %p\n",
					  usrmemptr(curr_chunk) );
				  printMemoryMap((memchunkptr) 0x400b5210);
				}
				
				return usrmemptr(curr_chunk);
				
			} else if ( slack >= 0 ) {	 /* allocate, no need to split */
				chunk_unlink(curr_chunk, head);
				mark_chunk_allocated(curr_chunk);

				// debug
				if ( DEBUG > 0 ) {
				  printf( ">> allocate, no need to split, returning: %p\n",
					  usrmemptr(curr_chunk) );
				  printMemoryMap((memchunkptr) 0x400b5210);
				}

				return usrmemptr(curr_chunk);
			}
			/* chunk is too small, continue searching */
		}
		head = (memchunkptr)(((char *)head) + 4);	/*look in next bucket*/
	}

	/* no free chunk fits -- do predictive allocation from top */
	/* we do this only for small size requests so as not to waste too much
		memory if we predict wrongly. The idea is that if there are a lot
		of requests for this size class, and no free chunks of this size,
		then preallocate 16 chunks of exactly this size in anticipation of
		future requests. */

	if ( size < 512 ) {
		curr_bin = size >> 3;
		count = 0;
		for ( temp = 0; temp < NO_OF_LAST_BINS; temp++ )
		  count += (globals->lastBin[temp] == curr_bin);

		if ( count << 1 >= NO_OF_LAST_BINS ) {
			head = bin2chunk(curr_bin);
			if ( head->next == head ) {	/*this bin is empty*/

				curr_chunk = globals->top;
				slack = chunksize(curr_chunk) - (16*size);

				if ( slack < 0 ) {
					/* must expand heap area */
					if (!mem_sbrk( -slack )) return 0;
					slack = 0;
				}

				/* remember this chunk to release to user */
				splitbin = curr_chunk;
				curr_chunk->size = size + lsb(curr_chunk->size);
				(nextchunk(curr_chunk))->prevsize = size;
				mark_chunk_allocated(curr_chunk);
				curr_chunk = (memchunkptr)(((char *)curr_chunk) + size);
				curr_chunk->prevsize = size;

				for ( temp = 1; temp < 16; temp++ ) {
					chunk_link(head,curr_chunk,curr_chunk->prevsize,size);
					curr_chunk = (memchunkptr)(((char *)curr_chunk) + size);
					curr_chunk->prevsize = size;
				}

				globals->top = curr_chunk;
				globals->top->prevsize = size;
				globals->top->size = (slack | 1);
	
				// debug
				if ( DEBUG > 0 ) {
				  printf( ">> optimize, returning: %p\n",
					  usrmemptr(curr_chunk) );
				  printMemoryMap((memchunkptr) 0x400b5210);
				}

				return usrmemptr(splitbin);
			}
		}
	}

	/* no free chunk fits, and didn't qualify for predictive allocation, so grab from top */

	curr_chunk = globals->top;
	slack = chunksize(curr_chunk) - size;

	if ( slack < 0 ) {
	  /* get 8 more bytes than necessary */
	  if ( size - 8 < requestedSize ) {
	    size = requestedSize;
	    size += 8;	
	    size = (size + 7) & (~7);	/* pad up to multiple of 8 bytes */	    
	    slack = chunksize(curr_chunk) - size;
	  }

	  /* must expand heap area */
	  if (!mem_sbrk( -slack )) return 0;
	  slack = 0;
	}

	curr_chunk->size = (size | lsb(curr_chunk->size));
	mark_chunk_allocated(curr_chunk);
	globals->top = (memchunkptr)(((char *)curr_chunk) + size);
	globals->top->prevsize = size;
	globals->top->size = (slack | 1);

	// debug
	if ( DEBUG > 0 ) {
	  printf( ">> got new page or used top, returning: %p\n",
		  usrmemptr(curr_chunk) );
	  printMemoryMap((memchunkptr) 0x400b5210);
	}

	return usrmemptr(curr_chunk);
}

void mm_free (void *ptr)
{
	globalsptr globals = (globalsptr)dseg_lo;
	memchunkptr curr_chunk = (memchunkptr)(ptr - 8);
	memchunkptr startpos;
	memchunkptr binptr;
	int newchunksize;

	if (ptr == 0) return;	/* do nothing on free(NULL) */

	// debug
	if ( DEBUG > 0 ) {
	  printf( "\n** free (ptr: %p) **\n", ptr );
	}

	startpos = curr_chunk;
	newchunksize = chunksize(curr_chunk);

	/* immediate coalescing with neighbouring blocks. note that coalescing with
		topmost chunk is treated as a special case */

	if ( ((curr_chunk->size) & 1) == 0 ) {	/* if previous block free... */
		newchunksize += prevchunksize(curr_chunk);
		startpos = prevchunk(curr_chunk);	/* unlink previous block */
		chunk_unlink(startpos, bin2chunk(bin_num(chunksize(startpos))));
	}

	if (nextchunk(curr_chunk) == globals->top) {
		/* update chunksize and consolidate with top and return */
		startpos->size = (newchunksize + globals->top->size) | 
		                 lsb(startpos->size);
		mark_chunk_free(startpos);
		globals->top = startpos;

		return;
	}

	if ( ((nextchunk(nextchunk(curr_chunk))->size) & 1) == 0 ) { /* next block is free */
		chunk_unlink(nextchunk(curr_chunk),
			     bin2chunk(bin_num(chunksize(nextchunk(curr_chunk)))));		/* unlink next block */
		newchunksize += chunksize(nextchunk(curr_chunk));/* update chunksize */
	}

	/* link block with startpos and chunksize */
	binptr = bin2chunk(bin_num(newchunksize));
	chunk_link(binptr,startpos,startpos->prevsize,newchunksize);

	// debug
	if ( DEBUG > 0 ) {
	  printf( ">> freed a chunk of size %d at %p\n", 
		  chunksize(curr_chunk), curr_chunk );

	  printMemoryMap((memchunkptr) 0x400b5210);
	}

}

