
/*
memory block format:

	allocated block:
		word 0: size and status
		word 1: empty (8 byte alignment required)
		word 2: start of user data
		...
		word size: size
	free block:
		word 0: size and status
		word 1: pointer to next block
		word 2: pointer to previous block
		...	
		word size: size
	end block (refered to as wilderness):
		word 0: size and status 
		...
		word size: size

	implications:
		minimum block is 4 words (16 bytes)
		size info before and after block allows for:
			1. constant time coalescing
			2. bidirectional traversal

binning:
	we use linked lists of free blocks.  
	each list of free blocks is called a bin 

bin layout:
	. 128 bins logarithmically spaced 
		. bins for sizes less than and equal to 512 are 8 bytes apart
		
		index 2   3   4  .... 64  65  ..... 
		size  16  24  32      512 576 640   

	implications:
		. for allocation smallest free bin should be used (first match)
		. if nothing is free in the bins then wilderness is used

Note:
	We have left the bin code in.  However, our grade is actually higher
without the binnig code.  This anomoly is due to the miscalculated throughput
numbers of the standard malloc as compared with our malloc (LIBC malloc uses
sbrk(), while we got to use mem_sbrk() which made speed a non-issue).  
	For practical applications we would advise using the binning code.

Our score on the default trace files with code as is:
	Average overall utilization: 94.974%
	Time:  0.125, total ops: 68057
	Average throughput: 543.098 tops
	>Performance index:  96.733
*/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

team_t team = {
    "ether bingers",
    "jacob richman",   
    "jsr",            
    "pawel opalinski", 
    "pjo"             
};

/* debugging flags */
#define TEST_BIN_INDEX (0)
#define TEST_ALLOC_MAP (0)
#define TEST_SBRK      (0)
#define TEST_BS_TO_WS  (0)
#define TEST_ALIGN     (0)
#define TESTING        (TEST_BIN_INDEX | TEST_ALLOC_MAP | TEST_SBRK | TEST_BS_TO_WS | TEST_ALIGN )
#define BREAK_EARLY    (0)
#define FEW_BINS       (0)

#define ASSERT         (0)
#define MEM_SWEEP      (0)
#define ALLOC_DRAW_MEM (0)
#define FREE_DRAW_MEM  (0)
#define DRAW_MEM       (ALLOC_DRAW_MEM | FREE_DRAW_MEM)

/* functionality flags */
#define COALESE_PREV   (1)
#define COALESE_NEXT   (1)
#define COALESE_WILD   (1)
#define CHOP_BLOCKS    (1)
#define BIN_SEARCH     (1)
#define GRAB_SMALL     (1)
#define GRAB_FACTOR    (8)
#if GRAB_SMALL
	#define isSmall(size) ((size) <= (512/BYTES_PER_WORD))
#endif
#define SORT_BINS      (0)

/* system dependant constants */
typedef int word;
#define ALIGN          (2)
#define WASTE          (3)
#define MIN_ALLOC      (4)
#define BYTES_PER_WORD (4)
#define SBRK(x)        mem_sbrk( (x) * BYTES_PER_WORD )

/* alignment macros
   assumption: ALIGN and BYTES_PER_WORD are powers of 2 */
#define sizeAlign(x)            (((x)+(ALIGN-1)) & ~(ALIGN-1))
#define byteSizeToWordSize(x)   ((x>>2) + (!!(x & (BYTES_PER_WORD - 1))))


#if FEW_BINS
	#define NBINS     (2)
#else
	#define NBINS     (128)
#endif

/* defines which let heap mimic global variables */
// #define HEAP_VARS (0)
#define HEAP_VARS (2+NBINS)
#if HEAP_VARS
	#define SEG_START    ((word *)   dseg_lo            )
	#define bin          ((word **)  SEG_START          )
	#define wilderness   ((word *)   SEG_START[NBINS]   )
	#define heapStart    ((word *)   SEG_START[NBINS+1] )
#else
	word *bin[NBINS];
	word *wilderness;
	word *heapStart;
#endif

/* bin setup constants */
#define NBINCLASSES (4)
#if FEW_BINS
	int binSize[ NBINCLASSES ] = { 2, 4, 6, 8 };
	int binNum[ NBINCLASSES ] = { 0, 0, 0, 0 };
#else
	int binSize[ NBINCLASSES ] = { 2,  16, 128, 512  };
	int binNum[ NBINCLASSES ]  = { 64, 32, 16,  14   };
#endif

/* given a memory block size returns the index to the bin for that size */
int sizeToBinIndex(int size){
	int i,temp,binIndex = 0;	

	for ( i=0; i<NBINCLASSES ; i++ ){
		temp  = size / binSize[ i ];	
		if (temp>binNum[i]){
			temp = binNum[i];
		}
		size -= temp*binSize[i];
		binIndex += temp;
	}
	
	binIndex = binIndex + (!!size);
	#if ASSERT
		assert( binIndex >= 0 && binIndex < NBINS );
	#endif
	return binIndex;
}

/* testing code */
#if TEST_BIN_INDEX
#define TEST_SIZE (6)
int testSize[] = { 4, 16, 128, 130 , 43000/8, 50000};
void testBinSizeCode(){
	int i,temp;
	for(i=0; i < TEST_SIZE ; i++){
		temp = testSize[i];
		printf("sizeToBinIndex(%d)=%d\n",temp,sizeToBinIndex(temp));
	}
	for(i=1; i<testSize[TEST_SIZE-1] ; i++) {
		temp = sizeToBinIndex( i );
		assert( temp >= 0 && temp < NBINS );
	}
	printf("all sizeToBinIndex(%d -> %d) are within bounds\n",1,testSize[TEST_SIZE-1]);
}
#endif

/* helper macros for common memory block manipulation functions */
#define ALLOC_MAP (1)
#define isAllocated(x)     ((*x) & ALLOC_MAP)
#define markAsAllocated(x) ((*x) = ((*x) | ALLOC_MAP))
#define markAsFree(x)      ((*x) = ((*x) & (~ALLOC_MAP)))
#define sizeOf(x)          ((*x) & (~ALLOC_MAP))
#define setSize(x,s)       (*x) = (s); (*((x)+(s)-1)) = (s)
#define toEnd(x)           ((x)+sizeOf(x)-1)
#define toWordAddress(x)   ((word *)(((word)(x)) & (~(BYTES_PER_WORD-1))))

#define PREV_OFFSET   (1)
#define NEXT_OFFSET   (2)
#define nextFreeBlock(x)   ((word *)(*((x)+NEXT_OFFSET)))
#define prevFreeBlock(x)   ((word *)(*((x)+PREV_OFFSET)))

/* testing code */
#if TEST_ALLOC_MAP
void testAllocMapCode(){
	int i;

	markAsAllocated(wilderness);

	printf("isAllocated(wilderness)=%d\n",isAllocated(wilderness));
	printf("sizeOf(wilderness)=%d\n",sizeOf(wilderness));
	for(i=0;i<sizeOf(wilderness);i++){
		printf("word %d = %d\n",i,*(wilderness+i));
	}

	markAsFree(wilderness);

	printf("isAllocated(wilderness)=%d\n",isAllocated(wilderness));
	printf("sizeOf(wilderness)=%d\n",sizeOf(wilderness));
	for(i=0;i<sizeOf(wilderness);i++){
		printf("word %d = %d\n",i,*(wilderness+i));
	}

}
#endif

#if TEST_SBRK
void testSBRKCode(){
	printf("dseg_lo=%d\n",(word)dseg_lo);
	printf("dseg_hi=%d\n",(word)dseg_hi);
	printf("wilderness=%d\n",(word)wilderness);
	printf("wilderness+sizeOf(wilderness)=%d\n",((word)(wilderness+sizeOf(wilderness))));
	printf("toEnd(wilderness)=%d\n",(word)(toEnd(wilderness)));
	printf("toWordAddress(dseg_hi)=%d\n",(word)(toWordAddress(dseg_hi)));
}
#endif

#if TEST_BS_TO_WS
void testBSToWS(){
	printf("byteSizeToWordSize(%d)=%d\n",1,byteSizeToWordSize(1));
	printf("byteSizeToWordSize(%d)=%d\n",4,byteSizeToWordSize(4));
	printf("byteSizeToWordSize(%d)=%d\n",5,byteSizeToWordSize(5));
	printf("byteSizeToWordSize(%d)=%d\n",6,byteSizeToWordSize(6));
	printf("byteSizeToWordSize(%d)=%d\n",7,byteSizeToWordSize(7));
	printf("byteSizeToWordSize(%d)=%d\n",8,byteSizeToWordSize(8));
	printf("byteSizeToWordSize(%d)=%d\n",9,byteSizeToWordSize(9));
}
#endif

#if TEST_ALIGN
void testSizeAlignCode(){
	int i;
	for (i=0;i<17;i++) {
		printf("sizeAlign(%d)=%d\n",i,sizeAlign(i));
	}
}
#endif

#if DRAW_MEM
void drawMemory(){
	word *p;

	printf("heapStart=%d\n",(word)heapStart);
	printf("wildernes=%d\n",(word)wilderness);

	for (p=heapStart; p <= (toEnd(wilderness)) ; p++ ){
		printf("mem[%d]=%d\n",(word)(p-heapStart),*p);
	}

}

void drawBin(int i){
	word *p;
	for (p=bin[i]; p; p=nextFreeBlock(p)){
		printf("%d -> ",sizeOf(p));
	} 
}
void drawBins(){
	int i;
	for (i=0;i<NBINS;i++){
		printf("bin %d:\t",i);
		drawBin(i);
		printf("\n");
	}
}
#endif

#if ASSERT
int isInBin(int index , word *lookingFor ){
	word *p;
	for (p=bin[index];p;p=nextFreeBlock(p)){
		if (lookingFor == p){
			return 1;
		}
	}
	return 0;
}
int isInBins(word *lookingFor){
	return isInBin( sizeToBinIndex(sizeOf(lookingFor)) , lookingFor);
}
#endif

/* removes the given memory block from the specified bin */
void removeFromBin(int index, word *p ){
	word *prev,*next;

	#if ASSERT
		assert( isInBin( index , p ) );
	#endif
	
	prev = prevFreeBlock(p);
	next = nextFreeBlock(p);

	if (p==bin[index]){
                bin[index] = next;
        }

        if (prev) {
                nextFreeBlock(prev) = next;
        }
        if (next) {
                prevFreeBlock(next) = prev;
        }

	#if ASSERT
		assert( !isInBin( index, p ) );
	#endif

}

#if SORT_BINS
/* places the given memory blcok in the specified bin using insertion sort by size */
void putInBin(int index, word *p ){
	word *q = bin[index];

	prevFreeBlock(p) = NULL;
	while ( q && sizeOf(q) < sizeOf(p) ){
		prevFreeBlock(p) = q;
		q = nextFreeBlock(q);
	}

	if (prevFreeBlock(p)) {
		nextFreeBlock(prevFreeBlock(p)) = p;
	} else {
		bin[index]=p;
	}

	nextFreeBlock(p) = q;
	if (q){
		prevFreeBlock(q) = p;	
	}
	
}
#else
/* places the given memory block at the head of the specified bin */
void putInBin(int index, word *p ){
	
	#if ASSERT
		assert( !isInBin( index, p ) );
	#endif

	prevFreeBlock(p) = NULL;	
	nextFreeBlock(p) = bin[index];
	if (bin[index]){
		prevFreeBlock(bin[index]) = p;
	}
	bin[index] = p;

	#if ASSERT
		assert( isInBin( index, p ) );
	#endif
}
#endif

#if BREAK_EARLY
	int initCount=0;
#endif
/* initializes bins, wilderness, and heapStart */
int mm_init (void) {
	int i;
	word *temp;
	#if BREAK_EARLY
		if (initCount++) exit(0);
	#endif
	
	if (!(temp = SBRK( HEAP_VARS + MIN_ALLOC ) )) {
		return -1;
	}

	heapStart = wilderness = temp + HEAP_VARS;	
	setSize(wilderness, MIN_ALLOC);
	markAsFree(wilderness);

	for (i=0 ; i < NBINS ; i++) {
		bin[i] = NULL;
	}

	#if TEST_ALLOC_MAP
		testAllocMapCode();
	#endif

	#if TEST_SBRK
		testSBRKCode();
	#endif
	
	#if TEST_BS_TO_WS
		testBSToWS();
	#endif

	#if TEST_ALIGN
		testSizeAlignCode();
	#endif

	#if TEST_BIN_INDEX
		testBinSizeCode();
	#endif
	
	
	#if TESTING
		exit(0);
	#endif

	return 0;
}

/* gets the required size block from wilderness */
word *getFromWilderness(int size){
	word *freeLocation;
	int wildSize;

	wildSize = sizeOf(wilderness);
	if (wildSize < (size+MIN_ALLOC) ){
		#if GRAB_SMALL
			if (isSmall(size)){
				if (!(SBRK( GRAB_FACTOR*size + MIN_ALLOC - wildSize ))){
					return NULL;
				}
				freeLocation = wilderness;
				wilderness += (GRAB_FACTOR-1)*size;
				wildSize = size + MIN_ALLOC;
				while(freeLocation != wilderness){
					setSize(freeLocation,size);
					markAsFree(freeLocation);
					putInBin(sizeToBinIndex(size),freeLocation);
					freeLocation += size;	
				}
			} else {
		#endif
				if (!(SBRK( size + MIN_ALLOC - wildSize ))){
					return NULL;
				}
				wildSize = size + MIN_ALLOC;
		#if GRAB_SMALL
			}
		#endif

	}

	freeLocation = wilderness;
	wilderness = wilderness + size;

	wildSize -= size;
	setSize(wilderness, wildSize);
	markAsFree(wilderness);

	setSize(freeLocation,size);
	markAsFree(freeLocation);

	return freeLocation;	
}

#if CHOP_BLOCKS
/* if the given memory block is larger then the specified size the remainder is freed */
word *chop(word *p, int size){
	word *freeBlock;
	int totalSize = sizeOf(p);
	
	if ( totalSize >= (size+MIN_ALLOC) ){
		freeBlock = p+size;

		setSize(freeBlock,totalSize-size);
		markAsFree(freeBlock);
		putInBin(sizeToBinIndex(sizeOf(freeBlock)),freeBlock);

		setSize(p,size);
		markAsFree(p);
	}
	return p;
}
#endif

#if BIN_SEARCH
/* gets the first block in bins at or passed startIndex
   assumption: all elements in bin[startIndex] > size */
word *getFromLargerBins(int startIndex,int size){
	int i;
	word *p;
	for (i=startIndex;i<NBINS;i++){
		if (bin[i]){
			p = bin[i];
			removeFromBin(i,p);
			#if CHOP_BLOCKS
				p = chop(p,size);
			#endif
			return p;
		}
	}
	return getFromWilderness(size);
}
#endif

/* gets the required size block.
   attempts to do so from smallest bins first */
word *getFreeBlock(int size){
	int binIndex = sizeToBinIndex( size );
	word *p;

	for ( p=bin[binIndex]; p ; p=nextFreeBlock(p) ) {
		if (size <= sizeOf(p)){
			removeFromBin(binIndex,p);
			#if CHOP_BLOCKS
				p = chop(p,size);
			#endif
			return p;
		}
	}

	binIndex++;
	if (binIndex<=binNum[0]) binIndex++;	
	#if BIN_SEARCH
		return getFromLargerBins( binIndex, size ); 
	#else
		return getFromWilderness(size);
	#endif
}

/* testing code */
#if MEM_SWEEP
void memSweep(){
	word *p = heapStart;
	
	MEM_SWEEP_LOOP:
		if (p==wilderness){
			assert( ((char *)(p+sizeOf(wilderness))) == (dseg_hi+1) );
			return;
		} else {
			assert( p < wilderness );
			assert( p >= heapStart );
			if (isAllocated(p)){
				assert(!isInBins(p));
			} else {
				assert(isInBins(p));
			}
			p += sizeOf(p);
		}	
	goto MEM_SWEEP_LOOP;
}
#endif

/* allocates memory block of at least size.
   returns: pointer to beginning of block (NULL if not succesful) */
void *mm_malloc (size_t size) {
	word *freeLocation;
	#if MEM_SWEEP
		memSweep();
	#endif

	size = byteSizeToWordSize(size);
	size += WASTE;
	size = sizeAlign(size);
	
	#if ALLOC_DRAW_MEM
		printf("\nmm_malloc(%d)\n",size);
		drawMemory();
		drawBins();
	#endif

	if ( (freeLocation = getFreeBlock( size )) ) {
		markAsAllocated( freeLocation );
		freeLocation += ALIGN;
	}

	#if ALLOC_DRAW_MEM
		drawMemory();
		drawBins();
	#endif
	#if MEM_SWEEP
		memSweep();
	#endif
	return freeLocation;	
}


#if COALESE_WILD
/* 	if p borders the wildernes region we define wilderness to include p.
	assumption: p is a free block 
	returns: 1 if coalesed, 0 otherwise */
int coaleseWithWilderness(word *p){
	int totalSize;

	if ( (p + sizeOf(p)) != wilderness ){
		return 0;
	}

	totalSize = sizeOf(p) + sizeOf(wilderness);
	wilderness = p;
	setSize(wilderness,totalSize);
	
	return 1;	
}
#endif

#if (COALESE_PREV || COALESE_NEXT)
/* merges to blocks into one 
   assumption: both blocks are free, block are continous, first < second */
void merge(word *first, word *second){
	int totalSize = sizeOf(first) + sizeOf(second);
	setSize(first,totalSize);
	markAsFree(first);
}
#endif

#if COALESE_PREV
/* if p borders a free block on the left we combine the blocks 
   assumption: p is a free block 
   returns: pointer to combined region */
word *coaleseWithPrev(word *p){
	word *prev;
	if (p<=heapStart){
		return p;
	} else {
		prev = p - *(p-1);
		if (isAllocated(prev)){
			return p;
		} else {
			removeFromBin(sizeToBinIndex(sizeOf(prev)),prev);
			merge(prev,p);
			return prev;
		}
	}
		
}
#endif

#if COALESE_NEXT
/* if p borders a free block on the right we combine the blocks
   assumption: p is a free block
   returns: pointer to combined region */
word *coaleseWithNext(word *p){
	word *next;
	next = p + sizeOf(p);
	if ( next == wilderness ) {
		return p;
	} else {
		if (isAllocated(next)){
			return p;
		} else {
			removeFromBin(sizeToBinIndex(sizeOf(next)),next);
			merge(p,next);
			return p;
		}
	}
}
#endif
/* reclaims previously allocated memory block
   assumption: ptr was allocated by mm_malloc and has not been freed */
void mm_free (void *ptr) {
	word *p =  (word *) ptr;
	      p -= ALIGN;
	#if MEM_SWEEP
		memSweep();
	#endif
	
	markAsFree(p);
	#if FREE_DRAW_MEM
		printf("\nfree(%d)\n",(word)(p-heapStart));
		drawMemory();
		drawBins();
	#endif
	
	#if COALESE_PREV
		p = coaleseWithPrev(p);
	#endif
	#if COALESE_NEXT
		p = coaleseWithNext(p);	
	#endif
	#if COALESE_WILD
		if (!coaleseWithWilderness(p)){
	#endif
			putInBin( sizeToBinIndex( sizeOf(p) ), p );
	#if COALESE_WILD
		}
	#endif

	#if FREE_DRAW_MEM
		drawMemory();
		drawBins();
	#endif
	#if MEM_SWEEP
		memSweep();
	#endif
}
