#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

team_t team = {
    "ether bingers",   /* Team name to be displayed on webpage */
    "jacob richman",   /* First member full name */
    "jsr",             /* First member email address */
    "pawel opalinski", /* Second member full name */
    "pjo"              /* Second member email address  */
};


#define DEBUG          (0)
#define ASSERT         (0)
#define TEST_BIN_INDEX (0)
#define COALESE        (1)
#define ALIGN          (2)
#define WASTE          (3)
#define MIN_ALLOC      (4)
#define BYTES_PER_WORD (4)
#define SBRK(x)        mem_sbrk( (x) * BYTES_PER_WORD )

#define sizeAlign(x) (((x)+(ALIGN-1)) & ~(ALIGN-1))

typedef int word;
word *heapStart;

#define NBINS     (128)
word *bin[NBINS];
word *wilderness;


#define ALLOC_MAP (1)
#define isAllocated(x)     ((*x) & ALLOC_MAP)
#define markAsAllocated(x) ((*x) = ((*x) | ALLOC_MAP))
#define markAsFree(x)      ((*x) = ((*x) & (~ALLOC_MAP)))
#define sizeOf(x)          ((*x) & (~ALLOC_MAP))
#define setSize(x,s)       ((*x) = (s)); ((*((x)+(s)-1)) = (s))

#define PREV_OFFSET   (1)
#define NEXT_OFFSET   (2)
#define nextFreeBlock(x)   ((word *)(*(x+NEXT_OFFSET)))
#define prevFreeBlock(x)   ((word *)(*(x+PREV_OFFSET)))

#define NBINCLASSES (4)
int binSize[ NBINCLASSES ] = { 2,  16, 128, 512 };
int binNum[ NBINCLASSES ]  = { 64, 32, 16,  14   };

int isInBin( int binIndex, word *add ){
	word *p;
	for (p=bin[ binIndex ];p;p=nextFreeBlock(p)){
		if (p==add) {
			return 1;
		}
	}	
	return 0;
}
int isInBins( word *add ){
	int i;
	for(i=0;i<NBINS;i++){
		if (isInBin(i,add)){
			return 1;
		}
	}
	return 0;
}
void describe(word *p){
	int alloc,size1,size2,inBins;
	word *address;

	alloc = isAllocated(p);
	size1 = sizeOf(p);
	size2 = *(p+size1-1);
	address = (word *)(((int)p)/BYTES_PER_WORD);
	inBins = isInBins(p);

	printf("alloc=%d size=%d=%d address=%d::%d inBins=%d\n",alloc,size1,size2,(int)address,(int)address+size1,inBins);
}

void putInBin(int binIndex, word* p){
	#if ASSERT
		assert(!isAllocated(p));
		assert(!isInBins(p));
	#endif

	prevFreeBlock(p) = NULL;
	nextFreeBlock(p) = bin[ binIndex ];
	if ( bin[ binIndex ] ) {
		prevFreeBlock( bin[ binIndex ] ) = p;
	}
	bin[ binIndex ] = p;	

	#if ASSERT
		assert(isInBins(p));
		assert(isInBins(bin[ binIndex ]));
	#endif
}

int sizeToBinIndex(int size){
	/* replace ops with faster ops */
	/* unroll loop */
	int i,temp,binIndex = 0;	

	for ( i=0; i<NBINCLASSES ; i++ ){
		temp  = size / binSize[ i ];	
		if (temp>binNum[i]){
			temp = binNum[i];
		}
		size -= temp*binSize[i];
		binIndex += temp;
	}
	#if ASSERT
		assert( binIndex >= 0 );
		assert( binIndex < NBINS );
	#endif
	return binIndex + (!!size);
}

word* chop(word* block, int size){

	word *newFreeBlock=NULL;
	int blockSize = sizeOf(block);
	#if ASSERT
		assert(!isInBins(block));
	#endif
	if ( blockSize >= ( size + MIN_ALLOC ) ){
		newFreeBlock = block + size;
		setSize(newFreeBlock,blockSize-size);
		setSize(block,size);
	}

	#if ASSERT
		assert(size == (sizeOf(block) + sizeOf(newFreeBlock)));
	#endif

	return newFreeBlock;

}

void removeFromBin(int binIndex, word *p){
	word *prev,*next;

	#if ASSERT
		assert(isInBins(p));
	#endif

	prev = prevFreeBlock(p);
	next = nextFreeBlock(p);

	if (p==bin[ binIndex ]){
		bin[ binIndex ] = next;
	}

	if (prev) {
		nextFreeBlock(prev) = next;
	}
	if (next) {
		prevFreeBlock(next) = prev;
	}

	next=NULL;
	prev=NULL;
	#if ASSERT
		assert(!isInBins(p));
	#endif
}
void removeFromBins(word *p){
	#if ASSERT
		assert(isInBins(p));
	#endif
	removeFromBin( sizeToBinIndex(sizeOf(p)), p );
	#if ASSERT
		assert(!isInBins(p));
	#endif
}


#if TEST_BIN_INDEX
#define TEST_SIZE (10)
int testSize[] = { 4, 16, 128, 130, 32000/4, 33000/4, 38000/4, 39000/4, 43108/4, 256*1024*19 };


void testBinSizeCode(){
	int i,temp;
	for(i=0; i < TEST_SIZE ; i++){
		temp = testSize[i];
		printf("sizeToBinIndex(%d)=%d\n",temp,sizeToBinIndex(temp));
	}
}
#endif

word* allocateFromWilderness(int size){
	/* look for constant time speed ups */
	word *freeLocation;
	int wildSize = sizeOf(wilderness);

	#if ASSERT
		word *oldWilderness = wilderness;
		assert(size >= MIN_ALLOC);
	#endif

	if (wildSize < (size+ALIGN) ){
		if ( !SBRK( size+ALIGN - wildSize ) ){
			return NULL;
		}
		wildSize = size+ALIGN;
	} 

	freeLocation = wilderness;
	wilderness = wilderness + size;

	wildSize -= size;
	setSize(wilderness, wildSize);
	markAsFree(wilderness);

	setSize(freeLocation,size);
	markAsFree(freeLocation);
	#if ASSERT
		assert( freeLocation == oldWilderness );
	#endif

	return freeLocation;
}

word *binSearch(int binIndex, int size){
	int i;
	word *p,*temp;
	for (i=binIndex; i<NBINS ; i++){
		if ( bin[ i ] ){
			
			p = bin[ i ];
			removeFromBin( i, p );
				
			if ((temp = chop(p,size))) {
				#if ASSERT
					assert( size == (sizeOf(temp) + sizeOf(p)) );
				#endif
				markAsFree(temp);
				putInBin( sizeToBinIndex( sizeOf(temp) ), temp);
			}

			return p;
		}
	}
	return allocateFromWilderness( size );
}

/* maybe instead of linked list we should use trees */
word* getFreeBlock(int size){
	int binIndex = sizeToBinIndex( size );
	word *p,*temp;

	for ( p=bin[ binIndex ]; p ; p = nextFreeBlock(p) ){
		if ( size <= sizeOf(p) ) {
			#if ASSERT
				assert( isInBin( binIndex, p ) );
				assert( !isAllocated(p) );
			#endif

			removeFromBin(binIndex,p);

			if ((temp = chop(p,size))) {
				#if ASSERT
					assert( size == (sizeOf(temp) + sizeOf(p)) );
				#endif
				markAsFree(temp);
				#if ASSERT
					assert(!isInBins(temp));
				#endif
				putInBin( sizeToBinIndex( sizeOf(temp) ), temp);
				#if ASSERT
					assert(isInBins(temp));
				#endif
			}

			return p;
		}
	}
	
	return binSearch(binIndex+1+(binIndex<binNum[0]), size);		
}

int initCount;
int mm_init (void) {
	int i;

	for (i=0; i < NBINS ; i++){
		bin[i] = NULL;
	}

	if ( (wilderness = SBRK( ALIGN )) ){
		heapStart = wilderness;
		setSize(wilderness,ALIGN);
		markAsFree(wilderness);
	} else {
		return -1;
	}

	#if TEST_BIN_INDEX
		testBinSizeCode();
		exit(0);
	#endif

	return 0;
}

void *mm_malloc (size_t size) {
	word *freeLocation;
	#if DEBUG
		printf("mm_malloc(%d)\n",size);
	#endif
	size = (size+BYTES_PER_WORD-1)/BYTES_PER_WORD;
	#if DEBUG
		printf("size in words = %d\n",size);
	#endif

	size += WASTE;	
	size = sizeAlign(size);
	#if DEBUG
		printf("aligned size = %d\n",size);
	#endif

	#if ASSERT

	#endif

	if ( (freeLocation = getFreeBlock( size )) ){
		markAsAllocated( freeLocation );
		freeLocation += ALIGN;		/* adjust for user */
	} 

	#if DEBUG
		printf("returning from malloc %d %p\n",sizeOf(freeLocation),freeLocation+ALIGN);
	#endif
	
	return freeLocation;
}

void merge(word *first, word *second){
	int totalSize;
	#if ASSERT
		assert(!isAllocated(first));
		assert(!isAllocated(second));
		assert(!isInBins(first));
		assert(!isInBins(second));
		assert( ( first + sizeOf(first) ) == second );
	#endif

	totalSize=sizeOf(first)+sizeOf(second);
	setSize(first,totalSize);
}
#if COALESE

word *coaleseWithPrev(word *p){
	word *prev;
	int size_prev;

	if (p<=heapStart){
		return p;
	} else {
		size_prev = *(p-1);
		prev = p - size_prev;

		if (isAllocated(prev)) {
			#if ASSERT
				assert(!isInBins(prev));
			#endif
			return p;
		} else {
			/* coalese */
/*
			#if ASSERT
				assert(isInBins(prev));
			#endif
			if (isInBins(prev))
			removeFromBins(prev);

			#if ASSERT
				assert(!isInBins(prev));
			#endif
			merge(prev,p);
*/
			return p;
		}
	}
}
				
word *coaleseWithNext(word *p){
	word *next;
	int totalSize;
	
	next = p+sizeOf(p);
	if (isAllocated(next)){
		#if ASSERT
			assert(!isInBins(next));
		#endif
		return p;
	} else if (next >= wilderness){
		totalSize = sizeOf(p) + sizeOf(wilderness);	
		wilderness = p;
		setSize(wilderness,totalSize);
		return NULL;
	} else {
		/* coalese */
/*
		#if ASSERT
			assert(isInBins(next));
		#endif
		if (isInBins(next))
		removeFromBins(next);
		#if ASSERT
			assert(!isInBins(next));
		#endif
		merge(p,next);
*/
		return p;
	}
}
#endif

void mm_free(void *ptr) {
	word* p = ptr;
	      p -= ALIGN;

	/* set deallocated flag */	
	markAsFree(p);

	/* coalese */
		/* prev */
			/* make sure that previous block exists */
			/* if so then */
				/* if free */
					/* remove previous block from its list*/
					/* merge prev with this */

		/* next */
			/* if wilderness */
				/* attach to wilderness */
			/* otherwise */
				/* if free */
					/* remove next block from its list */
					/* merge next with this*/
	#if COALESE
		p = coaleseWithPrev(p);
		p = coaleseWithNext(p);	/* might return NULL if merged with wilderness */
	#endif

	/* put in its bin */
	if (p){
		putInBin(sizeToBinIndex(sizeOf(p)),p);
	}
}

