/* $Id: malloc.c,v 3.5.1 1999/10/21 13:46:44 kaustuv Exp $ */

/*
 *
 *  CS213 - Lab assignment 3
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

team_t team = {
    /* Team name to be displayed on webpage */
    "Allocate This!",
    /* First member full name */
    "Kaustuv Chaudhuri",
    /* First member email address */
    "kaustuv",
    /* Second member full name (leave blank if none) */
    "",
    /* Second member email address (blank if none) */
    ""
};

/*
 * Inspiration from the Knuth and Standish paper, and a survey by
 * Paul Wilson: ftp://ftp.cs.utexas.edu/pub/garbage/allocsrv.ps
 */

/*
 * Type declarations
 */

typedef struct _block block, * blockptr;

struct _block
{
  size_t prev_size;             /* Size of previous block (if free). */
  size_t size;                  /* Size in bytes, including overhead. */
  blockptr frwd;                  /* double links -- used only if free. */
  blockptr back;
};


/*  sizes, alignments */
/*  These are all declared as const so they are not all that different
 *  from #defines */
const unsigned int  SIZE_T =          (sizeof(size_t));
const unsigned int  ALIGNMENT       = (2 * sizeof (size_t));
const unsigned int  ALIGN_MASK      = (2 * sizeof(size_t) - 1);
const unsigned int  MINSIZE         = (sizeof (block));

/* convert blocks to user pointers */
static inline void * block2mem(void *p)
{
  return (void*)((char*)(p) + 2*SIZE_T);
}

/* convert user pointers to blocks */
static inline blockptr mem2block(mem)
{
  return (blockptr)((char*)(mem) - 2*SIZE_T);
}

/* pad request bytes into a usable size */
static inline long request2size(req)
{
  return (((long)((req) + (SIZE_T + ALIGN_MASK)) < \
           (long)(MINSIZE + ALIGN_MASK)) ? MINSIZE : \
          (((req) + (SIZE_T + ALIGN_MASK)) & ~(ALIGN_MASK)));
}

/* pointer to the next block of memory */
#define next_block(p) ((blockptr)( ((char*)(p)) + ((p)->size & ~0x1) ))

/* pointer to the previous block */
#define prev_block(p)\
   ((blockptr)( ((char*)(p)) - ((p)->prev_size) ))


/* convert a pointer + size to a block */
#define make_block(p, s)  ((blockptr)(((char*)(p)) + (s)))

/* check if the current pointer is in use */
#define inuse(p)\
    ((((blockptr)(((char*)(p))+((p)->size & ~0x1)))->size) & 0x1)

/* extract inuse bit of previous block */
#define prev_inuse(p)  ((p)->size & 0x1)

/* set inuse bit cleanly (i.e., without disturbing size) */
#define set_inuse(p)\
((blockptr)(((char*)(p)) + ((p)->size & ~0x1)))->size |= 0x1

/* clear inuse bit cleanly */
#define clear_inuse(p)\
((blockptr)(((char*)(p)) + ((p)->size & ~0x1)))->size &= ~(0x1)

/* get size, ignoring use bits */
#define blocksize(p)          ((p)->size & ~(0x1))

/* set size at head, without disturbing its use bit */
#define set_head_size(p, s)   ((p)->size = (((p)->size & 0x1) | (s)))

/* set header overwriting previous bits in header */
#define set_head(p, s)        ((p)->size = (s))

/* set size at 'footer' if it is not in use */
#define set_next_head(p, s)   (((blockptr)((char*)(p) + (s)))->prev_size = (s))

/* check inuse at offset */
#define inuse_at(p, s)\
 (((blockptr)(((char*)(p)) + (s)))->size & 0x1)

/* set inuse at offset */
#define set_inuse_at(p, s)\
 (((blockptr)(((char*)(p)) + (s)))->size |= 0x1)

/* clear inuse at offset */
#define clear_inuse_at(p, s)\
 (((blockptr)(((char*)(p)) + (s)))->size &= ~(0x1))


#define  nbins (128)

/* Bins
 *
 * There are
 *
 *     64 bins of size       8
 *     32 bins of size      64
 *     16 bins of size     512
 *      8 bins of size    4096
 *      4 bins of size   32768
 *      2 bins of size  262144
 *      1 bin  of size (what's left)
 *      and a partridge in a pear tree
 *
 * The lower 112 bins have blocks the same size, and are allocated much faster
 * than the others.
 */

#define av             ((blockptr *)((char *)dseg_lo + 2 * SIZE_T))

/* access macros */
#define bin_at(i)      ((blockptr)((char*)&(av[2*(i) + 2]) - 2*SIZE_T))
#define next_bin(b)    ((blockptr)((char*)(b) + 2 * sizeof(blockptr)))
#define prev_bin(b)    ((blockptr)((char*)(b) - 2 * sizeof(blockptr)))

/*
 * This uses Doug Lea's suggestions for improving the book-keeping of
 * bins at very insignificant overhead. The first two bins are used
 * for indexing, maintaining the 'top of memory' block and the last
 * remaindered block of memory. For a better explanation of the bin
 * structure, please see the following document:
 *
 *         http://g.oswego.edu/dl/html/malloc.html
 *
 * That document even talks about the various bin-packing algorithms,
 * specifically the details of the logarithmic sized bins, their numbers
 * etc.
 */

#define top            (bin_at(0)->frwd) /* The topmost block */
#define last_remainder (bin_at(1)) /* remainder from last split */

/* 'top' initially points to 'initial_top' */
#define initial_top    ((blockptr)(bin_at(0)))

/* Indexing into bins */

#define bin_idx(sz)                                                          \
(((((unsigned long)(sz)) >> 9) ==    0) ?       (((unsigned long)(sz)) >>  3): \
 ((((unsigned long)(sz)) >> 9) <=    4) ?  56 + (((unsigned long)(sz)) >>  6): \
 ((((unsigned long)(sz)) >> 9) <=   20) ?  91 + (((unsigned long)(sz)) >>  9): \
 ((((unsigned long)(sz)) >> 9) <=   84) ? 110 + (((unsigned long)(sz)) >> 12): \
 ((((unsigned long)(sz)) >> 9) <=  340) ? 119 + (((unsigned long)(sz)) >> 15): \
 ((((unsigned long)(sz)) >> 9) <= 1364) ? 124 + (((unsigned long)(sz)) >> 18): \
                                          126)
/* 'small' block bins */
#define MAX_SMALLBIN         63
#define MAX_SMALLBIN_SIZE   512
#define SMALLBIN_WIDTH        8

#define smallbin_idx(sz)  (((unsigned long)(sz)) >> 3)

/* Requests are `small' if both the corresponding and the next bin are small */
#define is_small_request(nb) (nb < MAX_SMALLBIN_SIZE - SMALLBIN_WIDTH)

/* This is the standard use-bit-vectors-to-speed-up-bin-searching thing */
#define BLOCKWIDTH     4   /* bins per block */

#define binblocks      (bin_at(0)->size) /* bitvector of nonempty blocks */
                                /* I know, it's a hack. But it that size
                                 * field was lying around uselessly anyway... */

/* bin<->block macros */
#define idx2binblock(ix)    ((unsigned)1 << (ix / BLOCKWIDTH))
#define mark_block(ii)      (binblocks |= idx2binblock(ii))
#define clear_block(ii)     (binblocks &= ~(idx2binblock(ii)))


/*  Other static bookkeeping data */

/* The first value returned from sbrk */
#define sbrk_base (* (char **)(dseg_lo))

/* The total memory obtained from system via sbrk */
#define sbrked_mem   (* (unsigned long *)((char *)dseg_lo + SIZE_T))

/* Linking blocks in bin lists.
 * Call these only with variables, not arbitrary expressions, as arguments. */

/* Place block p of size s in its bin, in size order,
 * putting it ahead of others of same size. */
#define block_link(P, S, IDX, BACK, FRWD)                                          \
{                                                                             \
  if (S < MAX_SMALLBIN_SIZE)                                                  \
  {                                                                           \
    IDX = smallbin_idx(S);                                                  \
    mark_block(IDX);                                                       \
    BACK = bin_at(IDX);                                                         \
    FRWD = BACK->frwd;                                                              \
    P->back = BACK;                                                               \
    P->frwd = FRWD;                                                               \
    FRWD->back = BACK->frwd = P;                                                      \
  }                                                                           \
  else                                                                        \
  {                                                                           \
    IDX = bin_idx(S);                                                       \
    BACK = bin_at(IDX);                                                         \
    FRWD = BACK->frwd;                                                              \
    if (FRWD == BACK) mark_block(IDX);                                         \
    else                                                                      \
    {                                                                         \
      while (FRWD != BACK && S < blocksize(FRWD)) FRWD = FRWD->frwd;                      \
      BACK = FRWD->back;                                                            \
    }                                                                         \
    P->back = BACK;                                                               \
    P->frwd = FRWD;                                                               \
    FRWD->back = BACK->frwd = P;                                                      \
  }                                                                           \
}


/* take a block off a bin */
#define block_unlink(P, BACK, FRWD)                                                     \
{                                                                             \
  BACK = P->back;                                                                 \
  FRWD = P->frwd;                                                                 \
  FRWD->back = BACK;                                                                \
  BACK->frwd = FRWD;                                                                \
}                                                                             \

/* Place p as the last remainder */
#define last_remainder_link(P)                                                \
{                                                                             \
  last_remainder->frwd = last_remainder->back =  P;                               \
  P->frwd = P->back = last_remainder;                                             \
}

/* Clear the last_remainder bin */
#define last_remainder_clear \
  (last_remainder->frwd = last_remainder->back = last_remainder)


/* extend the top of the memory as needed */
static void grow_top(size_t nb)
{
  char* brk;                    /* return value from sbrk */
  size_t front_misalign;        /* unusable bytes at front of sbrked space */
  size_t correction;            /* bytes for 2nd sbrk call */
  char* new_brk;                /* return of 2nd sbrk call */
  size_t top_size;              /* new size of top block */

  blockptr old_top = top;       /* Record state of old top */
  size_t old_top_size = blocksize(old_top);
  char* old_end = (char*)(make_block(old_top, old_top_size));

  /* Pad request with plus minimal overhead */
  size_t sbrk_size = nb + MINSIZE;

  /* get the pagesize *ONCE* */
  unsigned long pagesz = mem_pagesize ();

  /* If not the first time through, round to preserve page boundary */
  /* Otherwise, we need to correct to a page size below anyway. */
  if (sbrk_base != (char*)(-1))
    sbrk_size = (sbrk_size + (pagesz - 1)) & ~(pagesz - 1);

  brk = (char*)(mem_sbrk (sbrk_size));

  /* Fail if sbrk failed or if a foreign sbrk call killed our space, */
  /* or we were hit by cosmic rays. */
  if (brk == (char*)(NULL) || (brk < old_end && old_top != initial_top))
    return;     

  sbrked_mem += sbrk_size;

  if (brk == old_end) {         /* can just add bytes to current top */
    top_size = sbrk_size + old_top_size;
    set_head(top, top_size | 0x1);
  } else {
    if (sbrk_base == (char*)(-1))  /* First time through. Record base */
      sbrk_base = brk;
    else /* Someone else called mem_sbrk().  Count those bytes as sbrked_mem. */
      sbrked_mem += brk - (char*)old_end;

    /* Guarantee alignment of first new block made from this space */
    front_misalign = (unsigned long)block2mem(brk) & ALIGN_MASK;
    if (front_misalign > 0) {
      correction = (ALIGNMENT) - front_misalign;
      brk += correction;
    } else correction = 0;

    /* Guarantee the next mem_brk will be at a page boundary */
    correction += pagesz - ((unsigned long)(brk + sbrk_size) & (pagesz - 1));

    /* Allocate correction */
    new_brk = (char*)(mem_sbrk (correction));
    if (new_brk == (char*)(NULL)) return; 

    sbrked_mem += correction;

    top = (blockptr)brk;
    top_size = new_brk - brk + correction;
    set_head(top, top_size | 0x1);

    if (old_top != initial_top) {
      /* There must have been an intervening foreign sbrk call. */
      /* Add a useless blocks on both sides to prevent consolidation */

      /* If not enough space to do this, then user did something very wrong */
      if (old_top_size < MINSIZE) {
        set_head(top, 0x1); /* will force NULL return from mm_malloc */
        return;
      }

      /* Also keep size a multiple of ALIGNMENT */
      old_top_size = (old_top_size - 3*SIZE_T) & ~ALIGN_MASK;
      set_head_size(old_top, old_top_size);
      make_block(old_top, old_top_size          )->size =
        SIZE_T|0x1;
      make_block(old_top, old_top_size + SIZE_T)->size =
        SIZE_T|0x1;
      /* If possible, release the rest. */
      if (old_top_size >= MINSIZE) 
        mm_free(block2mem(old_top));
    }
  }
}

/* The Malloc Algorithm
 *
 * The requested size is first converted into a useable form, by
 * adding a 4 byte overhead, and possibly some more to get eight byte
 * alignment, and to obtain a size of at least 'minsize.' All fits are
 * considered exact if they are within 'minsize' bytes.
 *
 * From this, the first successful of the following steps is taken:
 *
 * 1. The bin corresponding to the request size is scanned, and if
 *    a chunk of exactly the right size is found, it is taken.
 *
 * 2. The most recently remaindered chunk is used if it is big enough.
 *    This goes in line with the Knuth and Standish 'roving' first-fit
 *    algorithm used in the absence of exact fits. Runs of consecutive
 *    requests use the remainder of the chunk used for the previous
 *    such request whenever possible.
 *
 * 3. The bins are scanned in increasing size order, using a chunk big
 *    enough to fulfil the request, and splitting off any remainder.
 *    This search is strictly by best fit; i.e., the smallest chunk that
 *    fits is selected.
 *
 * 4. If large enough, the chunk bordering the end of memory ('top') is
 *    split off.
 *
 * 5. If Otherwise the top of the memory is extended by obtaining more
 *    space from the system. Memory is gathered from the sustem in a way
 *    that allows chunks across differenc sbrk calls to be consolidated.
 *
 * All allocations are made from the 'lowest' part of any found chunk.
 */

void* mm_malloc(size_t bytes)
{
  blockptr victim;              /* inspected/selected block */
  size_t victim_size;           /* its size */

  int idx;                      /* index for bin traversal */
  blockptr bin;                 /* associated bin */

  blockptr remainder;           /* remainder from a split */
  long remainder_size;          /* its size */
  int remainder_index;          /* its bin index */
  unsigned long block;          /* block traverser bit */

  int startidx;                 /* first bin of a traversed block */

  blockptr fwd;                 /* misc temp for linking */
  blockptr bck;                 /* misc temp for linking */
  blockptr q;                   /* misc temp */

  size_t nb = request2size(bytes); /* padded request size; */

  /* STEP 1: Look for exact size */

  /* Check for exact match in a bin */
  if (is_small_request(nb)) {   /* *fast* alg for small bins */
    idx = smallbin_idx(nb); 

    /* No traversal or size check necessary for small bins.  */
    q = bin_at(idx);
    victim = q->back;

    /* Also scan the next one, since it would have a remainder < MINSIZE */
    if (victim == q) {
      q = next_bin(q);
      victim = q->back;
    }

    if (victim != q) {
      victim_size = blocksize(victim);
      block_unlink(victim, bck, fwd);
      set_inuse_at(victim, victim_size);
      return block2mem(victim);
    }

    /* otherwise we have seen two bins, so set index up by 2 */
    idx += 2;
  } else {                      /* general algorithm for large bins */
    idx = bin_idx(nb);
    bin = bin_at(idx);

    for (victim = bin->back; victim != bin; victim = victim->back) {
      victim_size = blocksize(victim);
      remainder_size = victim_size - nb;
      
      if (remainder_size >= (long)MINSIZE){ /* too big */
        idx --;                 /* adjust to rescan below after checking */
                                /* last remainder */
        break;   
      } else if (remainder_size >= 0) { /* exact fit */
        block_unlink(victim, bck, fwd);
        set_inuse_at(victim, victim_size);
        return block2mem(victim);
      }
    }

    idx++; 
  }

  /* STEP 2: Try to use the last split-off remainder */

  if ( (victim = last_remainder->frwd) != last_remainder) {
    victim_size = blocksize(victim);
    remainder_size = victim_size - nb;

    if (remainder_size >= (long)MINSIZE) { /* re-split */
      remainder = make_block(victim, nb);
      set_head(victim, nb | 0x1);
      last_remainder_link(remainder);
      set_head(remainder, remainder_size | 0x1);
      set_next_head(remainder, remainder_size);
      return block2mem(victim);
    }

    last_remainder_clear;

    if (remainder_size >= 0) {  /* exhaust */
      set_inuse_at(victim, victim_size);
      return block2mem(victim);
    }

    /* Else place in bin */
    block_link(victim, victim_size, remainder_index, bck, fwd);
  }

  /* STEP 3: Scan bins for big enough block*/

  if ( (block = idx2binblock(idx)) <= binblocks) {
    /* Get to the first marked block */
    if ( (block & binblocks) == 0) {
      /* force to an even block boundary */
      idx = (idx & ~(BLOCKWIDTH - 1)) + BLOCKWIDTH;
      block <<= 1;

      while ((block & binblocks) == 0) {
        idx += BLOCKWIDTH;
        block <<= 1;
      }
    }
      
    /* Go through each (possibly) non-empty block */
    while (1) {
      startidx = idx;          /* (track incomplete blocks) */
      q = bin = bin_at(idx);

      /* For each bin in this block ... */
      do {
        /* Find and use first big enough block ... */
        for (victim = bin->back; victim != bin; victim = victim->back) {
          victim_size = blocksize(victim);
          remainder_size = victim_size - nb;

          if (remainder_size >= (long)MINSIZE) { /* split */
            remainder = make_block(victim, nb);
            set_head(victim, nb | 0x1);
            block_unlink(victim, bck, fwd);
            last_remainder_link(remainder);
            set_head(remainder, remainder_size | 0x1);
            set_next_head(remainder, remainder_size);
            return block2mem(victim);
          } else if (remainder_size >= 0) {  /* take it */
            set_inuse_at(victim, victim_size);
            block_unlink(victim, bck, fwd);
            return block2mem(victim);
          }
        }
       bin = next_bin(bin);     /* failed; continue with next bin */
      } while ((++idx & (BLOCKWIDTH - 1)) != 0);

      /* Clear out the block bit. */
      do {   /* Possibly backtrack to try to clear a partial block */
        if ((startidx & (BLOCKWIDTH - 1)) == 0) {
          binblocks &= ~block;
          break;
        }

        startidx --;
        q = prev_bin(q);
      } while (q->frwd == q);

      /* Get to the next possibly nonempty block */
      if ( (block <<= 1) <= binblocks && (block != 0) ) {
        while ((block & binblocks) == 0) {
          idx += BLOCKWIDTH;
          block <<= 1;
        }
      } else break;
    }
  }


  /* STEP 5/6: Try to use top block */

  /* Require that there be a remainder, ensuring top always exists  */
  if ( (remainder_size = blocksize(top) - nb) < (long)MINSIZE) {
    grow_top(nb);               /* try to extend the top mem */
    if ( (remainder_size = blocksize(top) - nb) < (long)MINSIZE)
      return 0; /* propagate failure */
  }

  victim = top;
  set_head(victim, nb | 0x1);
  top = make_block(victim, nb);
  set_head(top, remainder_size | 0x1);
  return block2mem(victim);
} /* mm_malloc */

/* Phew */

/*
  Free Algorithm

  Cases:

    1. free(0) has no effect.  

    2. If a returned block borders the current high end of memory,
       it is consolidated into the top.

    3. Other blocks are consolidated as they arrive, and
       placed in corresponding bins. (This includes the case of
       consolidating with the current `last_remainder').

*/

void mm_free(void* mem)
{
  blockptr p;                   /* block corresponding to mem */
  size_t hd;                    /* its head field */
  size_t sz;                    /* its size */
  int       idx;                /* its bin index */
  blockptr next;                /* next contiguous block */
  size_t nextsz;                /* its size */
  size_t prevsz;                /* size of previous contiguous block */
  blockptr bck;                 /* misc temp for linking */
  blockptr fwd;                 /* misc temp for linking */
  int       islr;               /* track whether merging with last_remainder */

  if (mem == 0)                 /* free(0) has no effect */
    return;

  p = mem2block(mem);
  hd = p->size;

  sz = hd & ~0x1;
  next = make_block(p, sz);
  nextsz = blocksize(next);
  
  if (next == top) {            /* merge with top */
    sz += nextsz;

    if (!(hd & 0x1)) {          /* consolidate backward */
      prevsz = p->prev_size;
      p = make_block(p, -prevsz);
      sz += prevsz;
      block_unlink(p, bck, fwd);
    }

    set_head(p, sz | 0x1);
    top = p;
    return;
  }

  set_head(next, nextsz);       /* effectively, clear inuse bit */

  islr = 0;
  if (!(hd & 0x1)) {            /* consolidate backward */
    prevsz = p->prev_size;
    p = make_block(p, -prevsz);
    sz += prevsz;
    
    if (p->frwd == last_remainder) islr = 1; /* get last rem */
    else block_unlink(p, bck, fwd);
  }
  
  if (!(inuse_at(next, nextsz))) { /* consolidate forward */
    sz += nextsz;
    
    if (!islr && next->frwd == last_remainder) {  /* re-insert last_remainder */
      islr = 1; last_remainder_link(p);   
    } else block_unlink(next, bck, fwd);
  }

  set_head(p, sz | 0x1);
  set_next_head(p, sz);
  if (!islr)
    block_link(p, sz, idx, bck, fwd);  

  /* all done; implicit return */
}


/*
 * Initialise the bins and the sbrk variables
 */

int mm_init (void)
{
  int i;
  
  if(mem_sbrk (mem_pagesize ()) == NULL) /* cannot init (allocate first page)? */
    return -1;

  sbrk_base = (char *)(-1);     /* force the first mem_sbrk to load sbrk_base */
  sbrked_mem = 0;               /* even though we have one page, that is outside */
                                /* the jurisdiction of this mm */

  av[0] = av[1] = 0;            /* bins for random book-keeping */

  /* initialise the other bins. */
  for (i = 1; i <= nbins; i ++)
    av[2 * i] = av[2 * i + 1] = bin_at (i - 1);

  return 0;
}

/*
 * Wow, this reminds me of a couple of years back when I used to contribute
 * to gcc-devel. Mmm... compiler hacking...
 */
