/* $Id$ */

/*
 *
 *  CS213 - Lab assignment 3
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

team_t team = {
    /* Team name to be displayed on webpage */
    "renegade bunny squad",
    /* First member full name */
    "Charles K Smart",
    /* First member email address */
    "cks@andrew.cmu.edu",
    /* Second member full name (leave blank if none) */
    "Daniel Volkovich",
    /* Second member email address (blank if none) */
    "danielv@andrew.cmu.edu"
};


/*
	the algorithm: segregated fit
	(128 bins of different block sizes)

	The blocks are maintained using headers which prefix each
	block. The size and address of the previos block are stored
	in the header. The size is always a multiple of 8, so the
	least significant bit is used as an allocation flag (1 for
	allocated blocks).
	
	In free blocks, the header is extended 8 bytes further into
	the block, to link the free block into one of the 128
	circular bin lists.


	Blocks look like this:
	start->	+----------------------------------+
		| size of block                  |a| <- allocation flag
		+----------------------------------+
		| pointed to previous block        |
	mem->	+----------------------------------+
		| pointer to next in list          | <-+
		+----------------------------------+   | free blocks only
		| pointer to previous in list      | <-+
		+----------------------------------+
		|                                  |
		.                                  .
		.                                  .
	next->	+----------------------------------+


	mm_init() initializes the bins.

	mm_malloc(size) looks through the free bins, scanning upwards
	starting from bin_of(size). If no free blocks are found, it
	allocates more heap space. Blocks which are too large are split.

	mm_free(ptr) coalesces and then places the free memory into the
	free lists.
*/


/*
	constants
*/

#define NUM_BINS	128
#define PAGE_SIZE	0x1000
#define PAGE_MASK	0x0FFF
#define MIN_SIZE	16


/*
	data structures
*/

struct m_block {
	size_t		size;
	struct m_block	*prev;

	/* for free blocks only */
	struct m_block	*fd;
	struct m_block	*bk;
};

typedef struct m_block mblock;

struct m_header {
	mblock		*top;
	mblock		free[NUM_BINS];
};

typedef struct m_header mheader;


/*
	utilities
*/

/* get the user mem of a block */
#define mem_of(p)	((void *)&((p)->fd))

/* for dealing with allocation flags */
#define ALLOC		1
#define isalloc(p)	((p)->size & ALLOC)
#define mark_alloc(p)	(p)->size |= ALLOC
#define mark_free(p)	(p)->size &= ~ALLOC

#define prev_block(p)	((p)->prev)
#define next_block(p)	(mblock *)((char *)(p) + ((size_t)(p)->size & ~ALLOC))


/*
	bin_of(s) returns the bin index for blocks of size s.
	The partitioning numbers were taken verbatim from
	malloc-2.5.1.c written by Doug Lea in 1992.
*/

size_t bin_of(size_t s) {
	if (s < 512)			return (s >> 3);
	else if (s < 2560)		return 56 + (s >> 6);
	else if (s < 10752)		return 91 + (s >> 9);
	else if (s < 43520)		return 110 + (s >> 12);
	else if (s < 174592)		return 119 + (s >> 15);
	else if (s < 698880)		return 124 + (s >> 18);
	else if (s < 2796032)		return 126;
	else				return 127;
}


/*
	unlink a block from the doubly-linked list
	containing it.
*/

#define unlink_block(p)		\
{				\
	mblock *_bk = (p)->bk;	\
	mblock *_fd = (p)->fd;	\
	_fd->bk = _bk;		\
	_bk->fd = _fd;		\
}


/*
	push a freed, coalesced block into the lists of
	free blocks (in the correct bin).
*/

#define push_free(p)					\
{							\
	mblock *_bk = h->free + bin_of((p)->size);	\
	mblock *_fd = _bk->fd;				\
	_bk->fd = (p);					\
	(p)->bk = _bk;					\
	(p)->fd = _fd;					\
	_fd->bk = (p);					\
}


/*
	split a block which is too big into 2 chunks; place
	the remaineder back into the free lists.
*/

#define split_block(p, q)					\
{								\
	size_t	_dif = (p)->size - (q);				\
	if (_dif >= MIN_SIZE) {					\
		mblock *_nxt = next_block(p);			\
		mblock *_rem = (mblock *)((char *)_nxt - _dif);	\
		_nxt->prev = _rem;				\
		_rem->size = _dif;				\
		_rem->prev = (p);				\
		(p)->size = (q);				\
		push_free(_rem);				\
	}							\
}


/*
	coalesce a block with free neighbors. be sure to
	remove those neighbors from their respective lists.
*/

#define coalesce_block(p)				\
{							\
	mblock	*_nxt = next_block(p);			\
	mblock	*_prv = prev_block(p);			\
	if (!isalloc(_prv)) {				\
		unlink_block(_prv);			\
		_nxt->prev = _prv;			\
		_prv->size += (p)->size;		\
		(p) = _prv;				\
	}						\
	if (!isalloc(_nxt)) {				\
		unlink_block(_nxt);			\
		(p)->size += _nxt->size;		\
		_nxt = next_block(_nxt);		\
		_nxt->prev = (p);			\
	}						\
}


/*
	initialize the heap space (put a header down that will
	track the free blocks.)
*/

int mm_init (void)
{
	mheader *h;
	mblock *p;
	size_t i;


	/* make room for the header.  */

	mem_sbrk(2 * PAGE_SIZE);
	/*if (dseg_hi - dseg_lo + 1 < 2 * PAGE_SIZE) return -1;*/


	/* initialize the bins - have them contain themselves.  */

	h = (mheader *)dseg_lo;
	for (i = NUM_BINS, p = h->free; i; i--, p++) {
		p->size = 0;
		p->fd = p;
		p->bk = p;
	}


	/*
		initialize the bottom and top book-end blocks.
		mark them allocated so they won't be coalesced.
	*/

	p = (mblock *)(((size_t)(h + 1) + 7) & ~0x7);
	p->size = MIN_SIZE;
	p->prev = p;
	mark_alloc(p);

	p++;
	p->size = MIN_SIZE;
	p->prev = p - 1;
	mark_alloc(p);
	h->top = p;


	/* it all worked! */

	return 0;
}


mblock *more_heap (size_t size) {
	mblock *top, *p;
	mheader *h;
	size_t i;


	h = (mheader *)dseg_lo;


	/* how much space do we have after the top book-end? */

	top = h->top;
	p = (mblock *)(((size_t)dseg_hi - 8) & ~0x7);
	i = (size_t)p - (size_t)top;


	/* allocate more heap space if not enough */

	if (size > i) {
		/*
			it would seem page-alignment isn't a good thing.
			mem_sbrk((size - i + PAGE_MASK) & ~PAGE_MASK);
		*/
		mem_sbrk(size - i);
		p = (mblock *)(((size_t)dseg_hi - 8) & ~0x7);
		if ((size_t)p - (size_t)top < size) return NULL;
	}


	/* new block is old top */

	p = top;
	p->size = size;


	/* new top is end of new block */

	top = (mblock *)((char *)p + size);
	top->size = 8;
	top->prev = p;
	mark_alloc(top);
	h->top = top;

	return p;
}


void *mm_malloc (size_t size)
{
	mblock *p, *q;
	mheader *h;
	size_t bin;


	/* init vars */

	h = (mheader *)dseg_lo;
	size = (size + 15) & ~0x7;
	if (size < MIN_SIZE) size = MIN_SIZE;
	bin = bin_of(size);


	/* first - check free bin of matching index */

	p = h->free + bin;
	for (q = p->fd; q != p; q = q->fd)
		if (q->size >= size) {
			unlink_block(q);
			split_block(q, size);
			mark_alloc(q);
			return mem_of(q);
		}


	/* check greater free bins */

	q = h->free + bin + 1;
	p = h->free + NUM_BINS;
	while (q < p)
		if (q->fd != q) {	/* we know that q->size > size */
			q = q->fd;
			unlink_block(q);
			split_block(q, size);
			mark_alloc(q);
			return mem_of(q);
		} else
			q++;


	/* try to allocate another block */

	q = more_heap(size);
	if (q) {
		mark_alloc(q);
		return mem_of(q);
	}


	/* no more space */

	return NULL;
}


void mm_free (void *ptr)
{
	mheader *h = (mheader *)dseg_lo;
	mblock *p = (mblock *)((char *)ptr - 8);


	/* very straight-forward... */

	mark_free(p);
	coalesce_block(p);
	push_free(p);
}
