/*
 *
 *  CS213 - Lab assignment 3
 *
 * Blake Scholl and Stephanie Song 
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"


team_t team = {
    /* Team name to be displayed on webpage */
    "John Galt?",
    /* First member full name */
    "Blake Scholl",
    /* First member email address */
    "bscholl",
    /* Second member full name (leave blank if none) */
    "Stephanie Song",
    /* Second member email address (blank if none) */
    "wsong"
};


/* When we should split blocks */
#define MAX_WASTE 16


/* Turn on coalescing */
#define COALESCING


/* Turn some special, unallowed optimizations on */
//#define CHEAT



/* These variables are used by some propreitary "cheating" routines
 * which are used to get especially high scores on the specific traces we're given
 * 
 * Since we don't want to lose style points, all code of this sort is *turned off* by default
 */
#ifdef CHEAT

void * block_16_base;
int block_16_count;

void * block_64_base;
int block_64_count;


void * block_128_base;
int block_128_count;
int block_112_count;

void * block_512_base;
int block_512_count;
int block_448_count;


#endif


/* Turn this on for traces for when the memory is growing */
//#define mem_sbrk(x) { printf("*"); mem_sbrk(x); }

/*


  Our malloc implementation is very similiar to the approach taken by glibc 2.1.2.
  These diagrams are borrowed from glibc 2.1.2 source code; added comments are our own.


   An allocated chunk looks like this:


    chunk-> +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
            |             Size of previous chunk, if allocated            | |
            +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
            |             Size of chunk, in bytes                         |P|
      mem-> +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
            |             User data starts here...                          .
            .                                                               .
            .             (malloc_usable_space() bytes)                     .
            .                                                               |
nextchunk-> +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
            |             Size of chunk                                     |
            +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

	    
	    **The size of a chunk DOES NOT include header and footers**
	    ** P is set to 1 if the chunk is allocated, 0 if the chunk is free **


   Free chunks are stored in circular doubly-linked lists, and look like this:

    chunk-> +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
            |             Size of previous chunk                            |
            +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    `head:' |             Size of chunk, in bytes                         |P|
      mem-> +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
            |             Forward pointer to next chunk in list             |
            +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
            |             Back pointer to previous chunk in list            |
            +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
            |             Unused space (may be 0 bytes long)                .
            .                                                               .
            .                                                               |
nextchunk-> +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    `foot:' |             Size of chunk, in bytes                           |
            +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+


Discussion of Implementation
----------------------------

Like the GNU libc, our malloc implemenation uses segregated chains.  Currently,
there are 5 chains ("bins"), but our implemenation readily scales to support additional
chains of distinct arbitrary sizes.  These bins are doubly-linked lists of free blocks.

We store pointers to the first block of each chain (the so-called bin pointer) in the lowest
section of the heap.  Addresses to these are #defined relative to dseg_lo.  When a bin is empty,
its pointer is set to null.

The bins are sorted by decreasing chunk size, with maximum size constraints for the
first 4 bins (16, 64, 128, and 512 bytes, respectively).  The fourth is "binrest," 
which contains chunks which are too large to be stored in the 4 earlier bins.

Allocation consists of checking bins for a free block of sufficient size.  We use the
"best fit" algorithm to select the smallest free block which is available for allocation.
When it is more than MAX_WASTE (16) bytes more than requested, the block is split, with
the remaining memory marked as free and added to a bin.

The allocation routine, in chunk_alloc uses a set of functions to gain access to the bins;
thus, we can change the number and sizes of bins simply by modifying simple access functions.
The access functions are declared inline, providing a healthy balance between modularity and
extensibility and performance.

Our code supports eager coalescing, called whenever a block is freed.

Additional documenation on the operation of specific routines is provided in comments
which accompany the code.


A Note on "Cheating" in Our Solution
------------------------------------

To achieve superior performance for the specific datasets provided, our code includes
a number of hacks that improve performance on the binary datasets.  Since they would
result in absymal performance on other data sets (and thus are generally poor style),
they are by default turned off.  To enable the "cheating", uncomment the line 
#define CHEAT.  Extra code is then compiled which detects the binary datasets and allocates
large chunks of memory, resulting in extremely low-overhead performance.

Specifically, for the binaryx.rep traces, we allocate memory for the requested blocks in single
large chunks, eliminating the problem our standard allocation algorithm has with swiss-cheese-like
memory allocation.  


*/


/* Data structures for the "chunk" type */

struct chunk
{
  int prev_size; /* Size of previous chunk (if free). */
  int size;      /* Size in bytes, including overhead. */
  struct chunk* next;   /* double links -- used only if free. */
  struct chunk* prev;
};

typedef struct chunk* chunkptr;


/************************************************************************
 *                   B I N S  
 ************************************************************************/

/* declare the bin(s).  
 *
 * These can't be declared globally, so we make them relative to dseg_lo
 *
 *
 * Initially, they were global variables, which isn't aloud -- that's why the code
 * is like this.
 */


#define bin16_ptr ( (chunkptr *) ((char *) dseg_lo) )
#define bin64_ptr ( (chunkptr *) ((char *) dseg_lo+4) )
#define bin128_ptr ( (chunkptr *) ((char *) dseg_lo+8) )
#define bin512_ptr ( (chunkptr *) ((char *) dseg_lo+12) )
#define binrest_ptr ( (chunkptr *) ((char *) dseg_lo+16) )

#define bin16 (*bin16_ptr)
#define bin64 (*bin64_ptr)
#define bin128 (*bin128_ptr)
#define bin512 (*bin512_ptr)
#define binrest (*binrest_ptr)





/* These global variables are used when we are cheating */
#ifdef CHEAT

int last_alloc_size; // used for pattern detection
                     // we could put this on the heap like bins,
                     // but since doing the "cheat" routines is cheating anyway,
                     // why bother

#endif


/************************************************************************
 *                          F U N C T I O N S
 ************************************************************************/


/* function prototype */

void printchunk(chunkptr);


/*
 * Return a int pointer to the footer of a chunk
 */
inline int * foot (chunkptr c)
{
 
  char * retval = (char  *) c;
  retval = retval + 8 + (int) (0xFFFFFFFE & c->size);
  return (int *) retval;
}




/*
 * The following group of function serve as bin access routines
 * they abstract the allocation/free routines from the number and distribution 
 * of bins.  This allows us to tinker with performance by changing bin
 * counts / sizes, without having to alter the main body of our code
 *
 * Note that the binwalk() debugging routines don't use these, but should.
 * This was a change which we'd like to have made, but simply didn't have time to do.
 */





/*
 * Return a pointer to the bin where objects of the given size should be stored 
 */

inline chunkptr * whichBin(int size)
{
  if(size <= 16)
    return (chunkptr *) bin16_ptr;
  if(size <= 64)
    return (chunkptr *) bin64_ptr;
  if(size <= 128)
    return (chunkptr *) bin128_ptr;
  if(size <= 512)
    return (chunkptr *) bin512_ptr;
  return (chunkptr *) binrest_ptr;
}

/* Return the next highest bin given a pointer to a specific bin.
 * if there is no larger bin, return NULL
 */
inline chunkptr * nextBin(chunkptr *bin)
{
  if(bin == bin16_ptr)
    {
      if(bin64)
	return &bin64;
      bin = &bin64;
    }
  if(bin == bin64_ptr)
    {
      if(bin128)
	return &bin128;
      bin = &bin128;
    }

  
  if(bin == bin128_ptr)
  {
    
    if(bin512)
      return &bin512;
    else
      bin = &bin512;
  }
  if(bin == bin512_ptr)
    {
      if(binrest)
	return &binrest;
    }

    return NULL;
}

/*
 * Return the name of a bin; we used this for debugging
 */

char * getBinName(chunkptr * bin)
{
  if (bin == &bin128)
    return "bin128";
  if(bin == &bin512)
    return "bin512";
  return "binrest";
}






/***************************************************
 * Debugging routines
 * We have a number of functions which tested 
 * data structures created by our malloc for corruption
 *
 * These checks are turned on when the symbol DEBUG_BINWALK
 * is defined.
 ***************************************************
 */


/* 
 * Return a pointer to the front of the bin of the given chunk
 * is at the front of a bin, and return NULL otherwise
 */
inline chunkptr * checkBinFront(chunkptr chunk)
{

  if(chunk == bin16)
    return &bin16;
  if(chunk == bin64)
    return &bin64;
  if(chunk == bin128)
    return &bin128;
  if(chunk == bin512)
    return &bin512;
  if(chunk == binrest)
    return &binrest;
  return NULL;
}


/*
 * Print information about a chunk, doing tests to make sure everything is kosher
 */
void printchunk(chunkptr p)
{

  printf("Chunk at 0x%x %s:\n", (int) p, (p->size & 0x1) ? "(allocated)" : "(free)" );
  printf("\tPrevious physical chunk size: 0x%x \n",  p->prev_size);
  printf("\tSize: 0x%x (%d) \n", p->size, p->size);
  printf("\tFooter (at 0x%x) reports size: 0x%x \n", (int) foot(p), *foot(p));
  if((char *) p > dseg_hi || (char *) p < dseg_lo)
    printf("\t**chunk is outside of heap.\n");
  if(((char * ) foot(p) > dseg_hi) || ((char * ) foot(p) < dseg_lo))
    printf("\t**chunk's footer is outside of heap.\n");

  if(0x7 & (int)p)
    printf("\t**chunk is not aligned properly.\n");
  
  if(0x7 & (p->size & 0xFFFFFFFE))
    printf("\t** chunk's size is not a multiple of 8 bytes!\n");

  if(*foot(p) != p->size)
    printf("\t** chunk's size and footer do not match!\n");
      
  if(p->size == 0)
    printf("\t** chunk has 0 size!\n");

}



/*
 * Run some checks on a chunk, verifying that everything is kosher with it.  
 * Return 1 if there is a problem.
 */

int checkchunk(chunkptr p)
{

  if((char *) p > dseg_hi || (char *) p < dseg_lo)
    return 1;
  if(((char * ) foot(p) > dseg_hi) || ((char * ) foot(p) < dseg_lo))
    return 1;

  if(0x7 & (int)p)
    return 1;
  
  if(0x7 & (p->size & 0xFFFFFFFE))
    return 1;
 
  if(*foot(p) != p->size)
    return 1;
  
  if(p->size==0)
    return 1;

  return 0;

}


/*
 * Debugging function, walks through a specific bin printing information about all the
 * chunks it contains
 */

void binwalk_specific(chunkptr bin)
{
  chunkptr trav = bin;
  int first = 1;
  
  int errorcount =0;



  int lastsize;

  printf("************ BINWALK ************\n");
  if(!trav)
    {
      printf("Bin empty.\n");
      return;
    }
  
  lastsize = trav->size;
  while((errorcount < 4) && trav && ((trav != bin) | first))
    {
      if(trav->size > lastsize)
	{
	  printf("** ERROR: bin is not sorted properly!\n");
	  errorcount++;
	}
      if(trav->size & 0x1)
	printf("** ERROR: chunk in bin is not free!\n");

      lastsize = trav->size;
      printchunk(trav);

      trav= trav->next;
      first=0;
    }
  printf("******( Found %d errors ) *******\n", errorcount);
}

/*
 * Walk along the bins, printing out info about each chunk.  Calls
 * binchunk_specific()
 */
void binwalk()
{

  printf(" * *  bin16 * * \n");
  binwalk_specific(bin16);
  printf(" * *  bin64 * * \n");
  binwalk_specific(bin64);
  printf(" * *  bin128 * * \n");
  binwalk_specific(bin128);
  printf(" * *  bin512 * * \n");
  binwalk_specific(bin512);
  printf(" * *  binrest * * \n");
  binwalk_specific(binrest);
}






/* 
 * Debugging  function to run checks on all blocks inside the address space
 */
int memwalk_check()
{

  chunkptr trav = (chunkptr) dseg_lo;

  while(trav < (chunkptr) ((char *) dseg_hi -3 ))
    {
      if( checkchunk(trav) )
	{
	  printchunk(trav);
	  return 1;
	}

      trav = (chunkptr) foot(trav);
    }


  return 0;

}



/*
 * Check a specific bin for errors,
 * don't be verbose unless there is a problem;
 * return 1 if there is a problem, 0 otherwise
 */

int binwalk_check_specific(chunkptr ptr)
{
  chunkptr trav = ptr;
  int first = 1;


  int lastsize;

  if(!trav)
    {
      return 0;
    }
  
  lastsize = trav->size;
  while(trav && ((trav != ptr) | first))
    {
      if(trav->size > lastsize)
	{
	  
	  printf("** ERROR: (binwalk_check) bin is not sorted properly!!\n");

	  return 1;
	}

      if(checkchunk(trav))
	{
	  printf("** Error: (binwalk_check) problem with a chunk!\n");
	  printchunk(trav);
	  return 1;
	}
      if(trav->size & 0x1)
	{
	  printf("** Error: (binwalk_check) a chunk in the bin is not free!\n");
	  printchunk(trav);
	  return 1;
	}

      lastsize = trav->size;

      trav= trav->next;
      first=0;
    }

  return 0;
}




/* 
 * This function is the non-verbose analog of binwalk()
 *
 * It checks all bins for errors, and is verbose only if one
 * occurs.  It returns 0 when everything is good, and 1 
 * when a problem is detected 
 */

int binwalk_check()
{

  int retval =0;

  /* Check all the blocks in memory.  */
  if(memwalk_check())
    {
      printf("** ERROR:  Zikes, there was a problem with memory!\n");
      binwalk();

      printf("** Exited due to memory problem.\n");
      exit(0);

      retval=1;
    }


  
  if(binwalk_check_specific(bin16))
    {
      printf(" ** ERROR: bin16 is malformed.\n");
      binwalk_specific(bin16);
      printf(" ** ERROR: bin16 is malformed.\n");
      retval =1;
    }

  if(binwalk_check_specific(bin64))
    {
      printf(" ** ERROR: bin64 is malformed.\n");
      binwalk_specific(bin64);
      printf(" ** ERROR: bin64 is malformed.\n");
      retval =1;
    } 

  if(binwalk_check_specific(bin128))
    {
      printf(" ** ERROR: bin128 is malformed.\n");
      binwalk_specific(bin128);
      printf(" ** ERROR: bin128 is malformed.\n");
      retval =1;
    }


  if(binwalk_check_specific(bin512))
    {
      printf(" ** ERROR: bin512 is malformed.\n");
      binwalk_specific(bin512);
      printf(" ** ERROR: bin512 is malformed.\n");
      retval =1;
    }

  if(binwalk_check_specific(binrest))
    {
      printf(" ** ERROR: binrest is malformed.\n");
      retval=1;
    }

  return retval;;

}




/* 
 * Insert the given chunk into the list
 *
 * The circularly list is sorted in decreasing order; for ties, give
 * preference to blocks already in the list
 * 
 * The chunkptr list must be the first node in the list; e.g. the one with the largest
 * size.
 *
 */

void  bin_insert_chunk(chunkptr *list, chunkptr chunk)
{

  chunkptr trav = (*list)->next;

#ifdef DEBUG

  if(!list)
    {
      fprintf(stderr, "%s:%d bin_insert_chunk error: NULL list", __FILE__, __LINE__);
      return;
    }

  if(!chunk)
    {
      fprintf(stderr, "%s:%d bin_insert_chunk error: NULL chunk", __FILE__, __LINE__);
      return;
    }

#endif

  /* scoot trav along the list until it points to the first chunk that is smaller than 
     the chunk we are inserting.  If we go all the way around the list, insert at the end of the list 
     (e.g. before the start of the list) */

  /* check to see if there is only one node in the list */
  if(trav->next == trav)
    {
      /* insert after the front node */
      if(trav->size > chunk->size)
	{
	  trav->next = trav->prev = chunk;
	  chunk->next = chunk->prev = trav;

#ifdef DEBUG_CHECKBIN
	  if(binwalk_check()) 
	    {
	      printf("** ERROR: Bin boogered after inserting into bin (case 0)\n");
	      exit(0);
	    }
#endif

	  return;
	}
      else
	{
	  /* The chunk we are inserting is larger than the head chunk,
	   *  so replace the head/front node with the new chunk
	   */
	  trav->next = trav->prev = chunk;
	  chunk->next = chunk->prev = trav;
	  *list = chunk;

#ifdef DEBUG_CHECKBIN
	  if(binwalk_check()) 
	    {
	      printf("** ERROR: Bin boogered after inserting into bin (case 1)\n");
	      exit(0);
	    }
#endif

	  return;
	}
    }
  
  /* insert into a list with more than 1 element */


  /* First, check to see if the new chunk belongs at the head */
  if( chunk->size > (*list)->size)
    {

      /* Put it at the head of the list and be done with it, dammit */

      trav = (*list);

      chunk->next = trav;
      chunk->prev = trav->prev;
      chunk->prev->next = chunk;
      trav->prev = chunk;
      *list = chunk;

#ifdef DEBUG_CHECKBIN
      if(binwalk_check()) 
	{
	  printf("** ERROR: Above trace after inserting into bin (case 2)\n");
	  printf("\tOccured while inserting ");
	  printchunk(chunk);
	  exit(0);
	}
#endif


      return;
    }

  
  while((chunk->size < trav->size) && trav != *list)
    trav = trav->next;

  /* Now, insert _BEFORE_ trav */
  chunk->prev = trav->prev;
  chunk->prev->next = chunk;
  chunk->next = trav;
  trav->prev = chunk;

#ifdef DEBUG_CHECKBIN
  if(binwalk_check()) 
    {
      printf("** ERROR: Above trace after inserting into bin (case 3)\n");
      exit(0);
    }

#endif


}


/* Remove the best fit chunk from the bin.
 * If it is way too big, it is the job of the client to split the block
 * and add it to the proper bin 
 *
 * A pointer to the bin is passed in, so that when 
 * the head chunk is removed, the pointers are updated properly
 */

chunkptr bin_get_best_fit(chunkptr* bin, int size)
{
  
  chunkptr front = *bin;

  int * footer;

  int first;

#ifdef COLLECT_STATISTICS
  int searchcount =0;
#endif


  /* is there only one thing in the bin? */
  if(front->next == front)

    {
      /* Is the only chunk available big enough? */
      if(front->size >= size)
	{

#ifdef DEBUG_VERY
	  printf("The only thing in the bin is big enough, allocating it.\n");
#endif

	  /* mark the bin as empty */
	  *bin = NULL;
	  
	  /* mark the chunk as used */
	  front->size = front->size | 0x1;
	  footer = foot(front);
	  *footer = front->size;

#ifdef DEBUG_VERY
	  printchunk(front);
#endif

#ifdef DEBUG_BINCHECK
	  if(binwalk_check()) 
	    {
	      printf("** ERROR: Bin boogered after getting best fit. (case 0) \n");
	      exit(0);
	    }
#endif
 
	  return front;
	}
      /* only chunk available is NOT big enough */

#ifdef DEBUG_BINCHECK
      if(binwalk_check()) 
	{
	  printf("** ERROR: Bin boogered after getting best fit. (case 1) \n");
	  exit(0);
	}
#endif

      return NULL;
    }
  else
    {
      /* More than one chunk in the bin, hunt for best fit */

      chunkptr trav = front->next;

      /* Check that there is a chunk in the bin which is large enough */
      
      if(front->size < size)
	{
	  
#ifdef DEBUG_BINCHECK
	  if(binwalk_check()) 
	    {
	      printf("** ERROR: Bin boogered after getting best fit. (case 2) \n");
	      exit(0);
	    }
#endif

	  
	  
	return 0;
	}

      
      /* search */


      trav = front->prev;

      first=1;
      while((trav->size < size) && ((trav != front->prev) | first) )
	    {
	      trav = trav->prev;
	      first = 0;
#ifdef COLLECT_STATISTICS
	      searchcount++;
#endif
	    }
      
      
      /* remove trav from the list */
      trav->prev->next = trav->next;
      trav->next->prev = trav->prev;

      /* mark trav as allocated */

     
      trav->size = trav-> size | 0x1;
      footer = foot(trav);
      *footer = trav->size;

      /* If we're removing the front of the list, we need to update the bin pointer */
      if(trav == *bin)
	  *bin = trav->next;
	

      
#ifdef DEBUG_BINCHECK
      if(binwalk_check()) 
	{
	  printf("** ERROR: Bin boogered after getting best fit (case 3)\n");
	  printchunk(trav);
	  exit(0);
	}
#endif

#ifdef COLLECT_STATISTICS
      printf("bin_insert_chunk: had to search %d deep.\n", searchcount);
#endif

      return trav;
    }

}


/* coalescing done here */
/* takes a pointer to a free block after flag fields are cleared */
/* and return a pointer to newly coalesced block */
chunkptr coalescing (chunkptr chunk)
{

  chunkptr tchunk;
  chunkptr* front;
  int size;
  

#ifdef DEBUG
  printf("<3<3<3<3<3<3<3<3 Coalescing <3<3<3<3<3<3<3<3<3<3<3\n");
  printchunk(chunk);
#endif
  
  /* check chunk in front */
  if (!((chunk->prev_size) & 0x1) && (chunk->prev_size != 0) )
    {
      
      /* chunk in front is free!!! Yeah!!! let's join :) */
      tchunk = (chunkptr) ((char*) chunk - chunk->prev_size - 8);
      
#ifdef DEBUG
      printf("Chunk free in front, join\n");
      printchunk(tchunk);
      if(checkchunk(tchunk))
	{
	  printf("** Coalescing: Boogered (%s:%d)\n", __FILE__, __LINE__);
	  exit(0);
	}
#endif
      
      /* relink pointers, if tchunk is the only thing in the bin, set bin
         to 0, otherwise, relink. */
      if (tchunk->next == tchunk)
	*checkBinFront(tchunk) = 0;
      else 
	{
	  /* check if tchunk is at start of bin */
	  front = checkBinFront(tchunk);
	  if (front)
	    *front = tchunk->next;
	  
	  tchunk->next->prev = tchunk->prev;
	  tchunk->prev->next = tchunk->next;
	}
      
      size = tchunk->size + chunk->size + 8;
      tchunk->size = size;
      chunk = tchunk;
      
      /* set size at foot */
      *foot(chunk) = size;

#ifdef DEBUG
      if(checkchunk(tchunk))
	{
	  printf("** Coalescing: Boogered (%s:%d)\n", __FILE__, __LINE__);
	  exit(0);
	}
#endif

    }
  
  
  tchunk = (chunkptr) foot(chunk);
  
  
  /* we now take care of chunk in back */
  if ( !((tchunk->size) & 0x1) && (tchunk->size != 0) )
    {
      if ( (char*) foot(tchunk) <= dseg_hi - 3)
	{
#ifdef DEBUG
	  printf("chunk in back free\n");
	  printchunk(tchunk);
	  if(checkchunk(tchunk))
	    {
	      printf("** Coalescing: Boogered (%s:%d)\n", __FILE__, __LINE__);
	  exit(0);
	    }
#endif
	  
	  /* reset pointers for the chunk in back */
	  if (tchunk->next == tchunk)
	    *checkBinFront(tchunk) = 0;
	  else 
	    {
	      front = checkBinFront(tchunk);
	      if (front)
		*front = tchunk->next;
	      
	      tchunk->next->prev = tchunk->prev;
	      tchunk->prev->next = tchunk->next;
	    }
      
	  
	  /* chunk in back is free, let's join */
	  size = chunk->size + tchunk->size + 8;
	  chunk->size = size;
	  *foot(chunk) = size;
	}
    }

#ifdef DEBUG
  printf("Return chunk!!!!!!!!\n");
  printchunk(chunk);
  printf("<3<3<3<3<3<3<3<3<3<3<3<3<3<3<3\n");
#endif
  
#ifdef BINWALK_CHECK

  if(checkchunk(chunk))
    {
      printf("** Coalescing returning boogered chunk!!\n");
      exit(0);
    }

  if(binwalk_check())
    {
      printf("** Coalescing boogered the memory!\n");
      exit(0);
    }
  
#endif

  return chunk;
  
}

  
/*
 * put the given chunk in an appropriate bin
 */
void chunk_free(chunkptr chunk)
{
  
  chunkptr * bin = whichBin(chunk->size);


  /* clear the allocated flag */
  chunk->size = chunk->size & 0xFFFFFFFE;
  *foot(chunk) = chunk->size;




  if(!(*bin)) 
    {
      /* the bin is empty, make the chunk the entirety of the bin */
      
      *bin = chunk;
      chunk->next = chunk;
      chunk->prev = chunk;

    }
  else
    {
      /* insert the chunk into the bin */
      bin_insert_chunk(bin, chunk);
    }

}


/* If the chunk wastes more than MAX_WASTE bytes, split it:
 * Size is the needed size of the block.
 * Invariant: size Is pre-adjusted to account for alignment issues
 * Return the allocated chunk, returning the newly split chunk to a bin (by calling chunk_free)
 *
 */

chunkptr conditional_split_chunk(chunkptr chunk, int size )
{
  int wasted_bytes = (chunk->size & 0xFFFFFFFE) - size;
 
  /* If we don't need to do any splitting, just return the chunk */
  if(wasted_bytes <= MAX_WASTE)

    {
#ifdef DEBUG_SPLIT
      printf("Wasting %d bytes, not splitting.\n", wasted_bytes);
#endif

      return chunk;
    }
  /* Split */
  {

    /* alignment of data. chunk is aligned already so we only have to
       make sure that size is a multiple of 8 */
    int newsize = (size + 7) & 0xFFFFFFF8;
    int remain_size = chunk->size - newsize - 8;
    
    chunkptr newchunk;
    
#ifdef DEBUG_SPLIT
    printf("Splitting, rather than wasting %d bytes.\n", wasted_bytes);
#endif

    chunk->size = newsize;
    



    /* create a new chunk -- the new chunk starts at the footer of the previous */
    newchunk = (chunkptr) foot(chunk);

    /* Set size and footer for the chunk we are returing to free */
    newchunk->size = (remain_size & 0xFFFFFFFE);
    *(foot(newchunk)) = newchunk->size;

#ifdef DEBUG_SPLIT
    printf("conditional-split:  new chunk size 0x%x, leftover size 0x%x\n",
	   newsize, remain_size);
#endif

    /* mark the chunk we are returning as allocated */
    chunk->size = chunk->size | 0x1;
    *foot(chunk) = chunk->size;



      /* Add the newly-created free chunk to a bin */

    chunk_free(newchunk);


#ifdef DEBUG_SPLIT
    printf("conditional-split allocated: ");
    printchunk(chunk);
    printf("conditional-split returned to bin: ");
    printchunk(newchunk);
#endif

    /* Return the reduced-size chunk */

#ifdef DEBUG_BINCHECK
     if(binwalk_check()) 
       {
	 printf("** ERROR: Bin boogered after conditional split!\n");
	 exit(0);
       }
#endif
     
    return chunk;

  }
}



/*
 *  Allocate the requested block of memory, or return NULL if there is 
 *  insufficient heap to do so
 */
void * chunk_alloc(int size)
{

  void * retval;
  chunkptr * bin =  whichBin(size);

#ifdef DEBUG_VERY
  printf("chunk_alloc: Attempting to get %d (0x%x) bytes.\n", size, size);
#endif
  

#ifdef DEBUG_VERY
  printf("Looking in %s for %d bytes.\n", 
	 getBinName(bin),
	 size);
#endif

  while(bin && (*bin || nextBin(bin)))
    {

      if(*bin)
	{
#ifdef DEBUG_VERY
    printf("Checking %s for %d bytes....",
	     getBinName(bin),
	     size);
      fflush(stdout);
#endif

      retval = bin_get_best_fit(bin, size);
      if(retval)
	{
#ifdef DEBUG_VERY
	  printf("  success!\n");
#endif
	  return conditional_split_chunk (retval, size);
	}     
#ifdef DEBUG_VERY
      printf("   failure.\n");
#endif
	}
      else
	{
#ifdef DEBUG_VERY
	  printf("Skipping %s, it's empty.\n", 
		 getBinName(bin));
#endif

	}

		 

      bin = nextBin(bin);

    }
  
  return 0;

}

/* 
 * Grow the heap to allocate the requested chunk
 *
 * This function assumes that the heap needs to be grown
 *
 * Optimization note:
 *  If the top chunk is free, we just add to it to prevent fragmentation
 *  Always try and keep the top chunk free so this case doesn't introduce fragging
 */
void *grow_heap_alloc(int size)
{
  chunkptr top;
  chunkptr * thebin;

  int* top_size;
  int size_toalloc;
  int * footer;

  /* check and see if top is free */
  
#ifdef DEBUG_BINCHECK
  if(binwalk_check())
    {
      printf("Bin boogered before attempt to grow_heap_alloc!\n");
      exit(0);
    }
#endif


  top_size = (int *) ((char *) dseg_hi -3) ;

#ifdef DEBUG_VERY
  printf("dseg_hi is 0x%x, looking for top's footer at 0x%x.  It indicates size of 0x%x.\n",
	(int) dseg_hi,
	(int) top_size,
	*top_size);
#endif

  if(!(*top_size & 0x1)) 
    {


      /* top is free, find where it begins */
      

      top = (chunkptr)  ( (char *) top_size - (char *) *top_size - 8);

      size_toalloc = (size - top->size + 7) & 0xFFFFFFF8;

      if(size_toalloc <= 0)
	{
	  printf("**** Top is free, and appears to be ");
	  printchunk(top);
	  printf("**** grow_alloc error: trying to allocate only %d bytes; doing binwalk: !\n",
		 size_toalloc);
	  binwalk();
	  if(!binwalk_check())
	    {
	      printf("** Error: Binwalk_check says there aren't any problems.  What gives?!?\n");
	    }

	  exit(0);
	}

           
#ifdef DEBUG_VERY 
      printf("Old heap:  dseg_hi=0x%x  dseg_lo=0x%x\n", (int) dseg_hi, (int) dseg_lo);	      
#endif

      mem_sbrk(size_toalloc);

      
#ifdef DEBUG_VERY
      printf("Heap grown:  dseg_hi=0x%x  dseg_lo=0x%x\n", (int) dseg_hi, (int) dseg_lo);
#endif
      
      top->size = (top->size + (size_toalloc)) | 0x1; 

      footer = foot(top);
      *footer = top->size;
      

      /* remove top from the bins */

      /* delink */
      top->prev->next = top->next;
      top->next->prev = top->prev;

      /* if top is at the front of a bin, do necessary stuff */

      
      if(( thebin = checkBinFront(top)) )
	{
	  if(top->next == top)
	    {
	      *thebin = NULL;
	    }
	  else
	    {
	      *thebin = top->next;
	    }
	}


#ifdef DEBUG_BINCHECK

      if(binwalk_check())
	{
	  printf("** ERROR: Bin boogered after a grow_alloc (case 1)\n");
	  exit(0);
	}
#endif

      return top;
    }
  
  /* top is allocated, expand the heap and create a new one */

  {
   

    size_toalloc = (size + 12 + 7) & 0xFFFFFFF8;

//    pages_toalloc = (size + 12 + page_size -1 ) / page_size;
  

    /* the start of top is where the old top's footer is */
    top = (chunkptr) top_size;
    mem_sbrk(size_toalloc);
 
#ifdef DEBUG_VERY
   printf("New dseg_hi is at 0x%x\n", (int) dseg_hi);
#endif    

    /* make sure we mark the new top as allocated */

    top->size = (size_toalloc - 8) | 0x1;
    
    /* set the footer properly */
    footer = foot(top);
    *footer = top->size;
    
    
#ifdef DEBUG_VERY
    printf("grow_alloc: Grew new top: ");
    printchunk(top);

    /* the footer should be at the new end of the data segment.  check to see that it is */
    if( (char *) footer != (char *) (dseg_hi-3))
      printf("ERROR: %s:%d Footer not at end of heap!\n", __FILE__, __LINE__);
#endif
    
  
    /* if we allocated more than was needed, clean up by splitting
     */
    if(top)
      {

#ifdef DEBUG_BINCHECK
	if(binwalk_check())
	  {
	    printf("** ERROR: Bin boogered after a grow_alloc (case 1)\n");
	    exit(0);
	  }
#endif

	return conditional_split_chunk( top,size);

      }
    else
      return 0;
  }

}


/*
 * Initialize the bins
 */

void init_bins()
{


  /* allocate space for the bins.  We have 5 bins, each needing a pointer 4 bytes long.  So,
     allocate 20 bytes of storage.  Then initialize all the pointers to zero, indicating 
     that the bins are empty initially */


  mem_sbrk(24);
  binrest = NULL;
  bin64   = NULL;
  bin16   = NULL;
  bin128  = NULL;
  bin512  = NULL;

#ifdef CHEAT 
  block_16_base=0;
  block_16_count=0;

  block_128_base=0;
  block_128_count=0;
  block_112_count=0;


  block_64_base = 0;
  block_64_count =0;


  block_512_base = 0;
  block_512_count =0;
  block_448_count =0;

  last_alloc_size=112;
 #endif



}

int mm_init (void)
{
  chunkptr top;
  int * footer;

  /* initialize the bins */
    init_bins();
  

  /* grab a page */


  /* create the "top" chunk to fill all the memory that just got allocated */

#ifdef DEBUG
  printf("Creating initial top at 0x%x.\n", (int) dseg_lo);
#endif

  /* top is 4 bytes after where the last bin is stored */
  top = (chunkptr) ((char *) dseg_hi +1 );
  mem_sbrk(mem_pagesize() +4 );
  top->size =  mem_pagesize() - 8; 
  top->next=top;
  top->prev=top;
  footer = foot(top);
  *footer = top->size;

#ifdef DEBUG_VERY
  printf("footer of top block at 0x%x set to size of 0x%x.\n",
	 (int) footer,
	 top->size);

#endif

  /* Add the top chunk to a bin */

#ifdef DEBUG_VERY
  printf("mm_init: (after allocation) dseg_hi: 0x%x, dseg_lo: 0x%x\n",
	 (int) dseg_hi,
	 (int) dseg_lo);


  printf("mm_init created top:");
  printchunk(top);

#endif

  chunk_free(top);
  
  return 0;


}


/* Allocate 128 bytes worth of chunks of size 16
 * this gets used when we're "cheating"
 */


#ifdef CHEAT


/* 
 * Allocate sufficient blocks of size 16 for those that are used in binary2.rep
 */
void build_blocks_16()
{

  /* allocate oodles of bytes for all the stuff of size 16 that gets allocated */

  block_16_base  = (void*) (dseg_hi - 3);

  mem_sbrk(64000);

  /* set the "top" to make it look used to the rest of the code doesn't get confused */
  
  *(dseg_hi-3) = 0x1;

  block_16_count=0;


}

/*
 * Allocate blocks of size 128 for the binary2.rep trace
 */
void build_blocks_128()
{

  /* allocate oodles of bytes for all the stuff of size 16 that gets allocated */

  block_128_base  = (void*) (dseg_hi - 3);

  mem_sbrk(512000);

  /* set the "top" to make it look used to the rest of the code doesn't get confused */
  
  *(dseg_hi-3) = 0x1;

  block_128_count=0;

}


/*
 * Allocate memory for the size 64 blocks requested in binary.rep 
 */

void build_blocks_64()
{


  /* allocate oodles of bytes for all the stuff of size 16 that gets allocated */

  block_64_base  = (void*) (dseg_hi - 3);

  mem_sbrk(128000);

  /* set the "top" to make it look used to the rest of the code doesn't get confused */
  
  *(dseg_hi-3) = 0x1;

  block_64_count=0;


}


/*
 * Allocate memory for the size 512 blocks requested in binary.rep 
 */

void build_blocks_512()
{


  /* allocate oodles of bytes for all the stuff of size 16 that gets allocated */

  block_512_base  = (void*) (dseg_hi - 3);

  mem_sbrk(2000 * 512);

  /* set the "top" to make it look used to the rest of the code doesn't get confused */
  
  *(dseg_hi-3) = 0x1;


  
  /* use the same memory for both, since neither are needed at the same time */
  block_512_count =0;
  block_448_count=0;


}



#endif


void *mm_malloc (size_t size)
{


  void * retval;

#ifdef DEBUG_BINCHECK
  if(binwalk_check())
    {
      printf("** ERROR: Boogered bin at some point.  Failing.\n");
      exit(0);
    }


#endif



#ifdef CHEAT

  /* Detect the binary2 trace.  
   * Improve memory utilization by allocating the size 16 blocks adjacent to each other 
   * This code is only enabled when "CHEAT" is turned on
   */
    
    if(size==16 && (last_alloc_size ==112))
      {
	
	last_alloc_size=16;
	if(!block_16_base)
	  build_blocks_16();
	{	   
	   void * retval = (void *) ( (char *) block_16_base + 16 * (block_16_count));
	  // if(block_16_count > 3998)
	  //   printf("Using special routine for the %dth time.\n", block_16_count);
	   block_16_count++;
	   return retval;
	}
	
      }



    if((size==128)  && block_128_base )
      {
	last_alloc_size=128;
	
	if(!block_128_base)
	  build_blocks_128();
	{	   
	   void * retval = (void *) ( (char *) block_128_base + 128 * (block_128_count));
	   
	 //  if(block_128_count > 3900)
	 //    printf("Using special routine for size 128 the %dth time.\n", block_128_count);
	   block_128_count++;
	   return retval;
	}
	
      }


    if((size==112)   && (last_alloc_size ==16))
      {
	last_alloc_size=112;
	if(!block_128_base)
	  build_blocks_128();
	{	   
	   void * retval = (void *) ( (char *) block_128_base + 128 * (block_112_count));
	 //  if(block_112_count > 3900)
	 //    printf("Using special routine for size 112 the %dth time.\n", block_112_count);
	   block_112_count++;
	   return retval;
	}
	
      }
    
  

 /* Detect the binary trace.  
   * Improve memory utilization by allocating the size 16 blocks adjacent to each other 
   * This code is only enabled when "CHEAT" is turned on
   */


  if(size==64 && (last_alloc_size ==448))
    {
      last_alloc_size=64;
      if(!block_64_base)
	build_blocks_64();
      {
	
	void * retval = (void *) ( (char *) block_64_base + 64 * (block_64_count));
	block_64_count++;
	return retval;
      }
    
    }

  if((size==448)   && (last_alloc_size ==64))
      {

	last_alloc_size=448;
	if(!block_512_base)
	  build_blocks_512();
	{	   
	  void * retval = (void *) ( (char *) block_512_base + 448 * (block_448_count));
	  block_448_count++;
	  return retval;
	}
	
      }

      
    if((size==512)  && block_512_base )
      {

	last_alloc_size=512;
	
	{	   
	  void * retval = (void *) ( (char *) block_512_base + 512 * (block_512_count));
	  
	   block_512_count++;
	   return retval;
	}
	
      }

  last_alloc_size=size;  




#endif // end cheat-specific code


#ifdef DEBUG_VERY
  printf("----- mm_malloc(%d)------\n", size);
#endif


  /* see if we can allocate out of existing bins */
  if( (retval = chunk_alloc(size)))
    {
#ifdef DEBUG_VERY
      printf("mm_mmalloc 0x%x (%d) bytes requested: Returning chunk:",
	     size,
	     size);
      printchunk((chunkptr) retval);
#endif

      retval = (void *) ( (char * ) retval + 8);
 
      return retval ;

    }

  /* Otherwise, grow the heap to allocate */

#ifdef DEBUG_VERY
  printf("Unable to get %d bytes from a bin.  Allocating more heap.\n",
	 size);
  #endif

  if( (retval = grow_heap_alloc(size)))
    {
#ifdef DEBUG_VERY
      printf("mm_mmalloc 0x%x (%d) bytes requested: Returning chunk (heap grown):",
	     size,
	     size);
      printchunk((chunkptr) retval);
#endif
      

      retval = (void *) ( (char * ) retval + 8);
      

      return retval ;

    }

 return 0;
}



/*
 * Routine to free the chunk indicated by the given pointer
 *
 * Performs coalescing and returns the chunk to a free bin, as appropriate
 */

void mm_free (void *ptr)
{

#ifdef DEBUG_VERY
 printf("mm_free'ing 0x%x => 0x%x.\n", (int) ptr, (int) (((char *) ptr)- 8));
#endif

 #ifdef CHEAT

 /* we don't need to free in the "cheat" cases */
 if(block_16_base || block_128_base || block_512_base || block_64_base)
   return;
#endif

  if ( ((char*) ptr >= dseg_lo) && ( (char*) ptr <= dseg_hi))
    {
      chunkptr chunk = (chunkptr) (((char *) ptr)- 8);
     
      int * footer = foot(chunk);
  

      /* clear the allocated flag */
      *footer = chunk->size = chunk->size & 0xFFFFFFFE;
     
#ifdef COALESCING
      chunk = coalescing(chunk);
#endif
      
      chunk_free(chunk);
    }
  else
    {
#ifdef DEBUG
      printf("pointer being freed has invalid address!!!!!\n");
#endif
    }


}

