/* $Id: malloc.c,v 1.4 99/10/19 19:26:12 bechang Exp Locker: bechang $ */

/*
 *
 *  CS213 - Lab assignment 3
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"


/* Word Size = 4 bytes */
#define WORD_SIZE 4

/* Page Size = 4096 bytes */
#define PAGE_SIZE 4096

/* Determine the offset from dseg_lo of an free list */
#define EQCLASS_OFFSET(x) ((x)*WORD_SIZE)

#define NUM_EQCLASSES 10

/* Free list offsets. */
#define FORWARD_FREE_OFFSET WORD_SIZE
#define BACKWARD_FREE_OFFSET (2*WORD_SIZE)

/* Allocated block offsets. */
#define USER_DATA_OFFSET WORD_SIZE

/* Bookkeeping structures */
#define START_USE (NUM_EQCLASSES*WORD_SIZE+WORD_SIZE)

/* Max */
#define MAX(x,y) ((x)>(y)?(x):(y)) 

/* Decode Size */
#define GET_SIZE(x) ((x)>>1)

team_t team = {
    /* Team name to be displayed on webpage */
    "<FONT color=#005555><B>Distractions</B></FONT>",
    /* First member full name */
    "Bor-Yuh Evan Chang",
    /* First member email address */
    "bechang",
    /* Second member full name (leave blank if none) */
    "Ana Ramirez",
    /* Second member email address (blank if none) */
    "anar"
};

/* Private Helper Functions */
void make_hf(void* pBlock, int iBSize, int iAlloc);
void insert_free(int iClass, void* pBlock);
void remove_free(int iClass, void* pBlock);
void* find_free(int iClass, int iBSize);
void allocate(void* pBlock, int iNewBSize);
void* coalesce(int iReqBSize);
int determ_eqclass(int iBSize);

int mm_init (void)
{
    int i;
    void* pFreeBlk;

    /* Allocate 1 page. */
    if (!mem_sbrk(PAGE_SIZE))
        return -1;

    /* Initialize all the head pointers to NULL. */
    for (i = 0; i < NUM_EQCLASSES; i++)
    {
        *((int*) (dseg_lo + EQCLASS_OFFSET(i))) = (int) NULL;
    }

    /* Create the free block. */
    pFreeBlk = dseg_lo + START_USE;
    make_hf(pFreeBlk, PAGE_SIZE - START_USE - WORD_SIZE, 0);

    /* Insert free block. */
    insert_free(8, pFreeBlk);

    return 0;
}

void *mm_malloc (size_t size)
{
    int iClass;
    int iBSize;
    int iCollSize;
    int iCollField;
    int iNewPageSize;

    void* pFreeHead;
    void* pFreeFoot;

    /* Compute block size. */
    iBSize = size + 2*WORD_SIZE;    /* Reserve for header/footer. */
    iBSize = MAX(iBSize + (8*!!(iBSize % 8) - (iBSize % 8)), 4*WORD_SIZE);

    /* Find first free block. */
    iClass = determ_eqclass(iBSize);
    pFreeHead = find_free(iClass, iBSize);

    while (!pFreeHead && iClass < NUM_EQCLASSES)
        pFreeHead = find_free(++iClass, iBSize);

    /* If found a free block, then allocate the block. */
    if (pFreeHead)
    {
        remove_free(iClass, pFreeHead);
        allocate(pFreeHead, iBSize);
    }

    /* Otherwise, try coalescing, then finally get new page(s). */
    else
    {
        /* Coalesce returns a pointer to the first free block (after
           coalescing) greater than or equal to iBSize.  The block
           is not in any free list but has the correct header/footer
           information.  */
        pFreeHead = coalesce(iBSize);

        if (pFreeHead)
            allocate(pFreeHead, iBSize);
        else
        {
            /* We have to get new pages. */

#ifdef DEBUG
            fprintf(stderr, "Coalesing End for NEW PAGES (%d)...\n", iBSize);
#endif

            /* "Coalesce" End */
            pFreeFoot = dseg_hi - 7;
            pFreeHead = dseg_hi - 3;
            iCollField = *((int*) pFreeFoot);
            iCollSize = 0;

            while ( !(iCollField & 0x01) )  /* While Free (look at footers) */
            {
                /* Increment free size of the end. */
                iCollSize += GET_SIZE(iCollField);

                /* Get Head. */
                pFreeHead = pFreeFoot - GET_SIZE(iCollField) + WORD_SIZE;

                /* Remove the block from free lists. */
                remove_free(determ_eqclass(GET_SIZE(iCollField)), pFreeHead);

                /* Go to next footer if not at beginning. */
                if ((int) pFreeHead <= (int) (dseg_lo + START_USE))
                    break;
                else
                {
                    pFreeFoot = pFreeHead - WORD_SIZE;
                    iCollField = *((int*) pFreeFoot);
                }
            }

            /* iBSize > iCollSize */
            iNewPageSize = iBSize - iCollSize - WORD_SIZE;
            iNewPageSize = iNewPageSize +
                           (PAGE_SIZE*!!(iNewPageSize % PAGE_SIZE) -
                           (iNewPageSize % PAGE_SIZE));

            if(!mem_sbrk(iNewPageSize))
                return NULL;

            /* Make header/footer for pFreeHead.  We don't add in
               WORD_SIZE again because (iCollSize + iNewPageSize) is
               8-byte aligned. */
            make_hf(pFreeHead, iCollSize + iNewPageSize, 0);

            allocate(pFreeHead, iBSize);
        }
    }

#ifdef DEBUG
    if (!(*((int*) pFreeHead) & 0x01))
    {
        fprintf(stderr, "ERROR:  Returning block marked free.\n");
    }
#endif

    /* Return memory for user */
    return (pFreeHead + USER_DATA_OFFSET);
}

void mm_free (void *ptr)
{
    void* pFree;
    int iBSize;

    /* Get info about block to free. */
    pFree = ptr - USER_DATA_OFFSET;
    iBSize = GET_SIZE(*((int*) pFree));


    /* Change the allocated flag to free. */

    /* Header */
    *((int*) pFree) &= 0xfffffffe;

    /* Footer */
    *((int*) (pFree + iBSize - WORD_SIZE)) &= 0xfffffffe;


    /* Insert into free list. */
    insert_free(determ_eqclass(iBSize), pFree);
}

/**********************************************************************
* void make_hf(void* pBlock, int iSize, int iAlloc)
*
* Description:
*    Make the header and footer information for a block.
*
* Parameters:
*    pBlock - pointer to the beginning of the block.
*    iBSize - the size of the block (in bytes).
*    iAlloc - flag if the block is allocated (0 - free, 1 - used)
**********************************************************************/
void make_hf(void* pBlock, int iBSize, int iAlloc)
{
    /* Header/Footer field (31 bits - size, 1 bit - allocated) */
    int hf = (iBSize << 1) | (iAlloc & 0x01);

    /* Header */
    *((int*) pBlock) = hf;

    /* Footer */
    *((int*) (pBlock + iBSize - WORD_SIZE)) = hf;
}


/**********************************************************************
* void insert_free(int iClass, void* pBlock)
* 
* Description:
*   Insert a free block into the appropriate free list give the size.
*   Insert into a list of descending order by size.
*
* Parameters:
*   iClass - the equivalence class of free blocks to place this one into.
*   pBlock - the block to insert.
**********************************************************************/
void insert_free(int iClass, void* pBlock)
{
    int iBSize;
    void* pCurr;
    void* pPrev;

    iBSize = GET_SIZE(*((int*) pBlock));

    /* Pointer to head of the free list. */
    pCurr = (void*) *((int*) (dseg_lo + EQCLASS_OFFSET(iClass)));
    pPrev = NULL;

    /* Keep going while bigger than the current block. */
    while (pCurr && GET_SIZE(*((int*) pCurr)) < iBSize)
    {
        /* Keep Prev */
        pPrev = pCurr;

        /* Go the next free block. */
        pCurr = (void*) *((int*) (pCurr + FORWARD_FREE_OFFSET));
    }

    if (pPrev)
    {
        /* Connect the forward pointer of the prev node to the new node. */
        *((int*) (pPrev + FORWARD_FREE_OFFSET)) = (int) pBlock;
    }
    else
    {
        /* Set this block to be the head. */
        *((int*) (dseg_lo + EQCLASS_OFFSET(iClass))) = (int) pBlock;
    }

    if (pCurr)
    {
        /* Connect the back pointer of the current node to node to
           be inserted. */
        *((int*) (pCurr + BACKWARD_FREE_OFFSET)) = (int) pBlock;
    }

    /* Connect the forward pointer of the new node to the current node. */
    *((int*) (pBlock + FORWARD_FREE_OFFSET)) = (int) pCurr;

    /* Make the back pointer of this new node point to previous node.*/
    *((int*) (pBlock + BACKWARD_FREE_OFFSET)) = (int) pPrev;
}


/**********************************************************************
* void remove_free(int iClass, void* pBlock)
* 
* Description:
*   + Remove a block from the free list given by iClass.
*   + Defer coalescing 'til later.
*
* Parameters:
*   iClass - the equivalence class that pBlock belongs to.
*   pBlock - the block.
**********************************************************************/
void remove_free(int iClass, void* pBlock)
{
    void* pPrev = (void*) *((int*) (pBlock + BACKWARD_FREE_OFFSET));
    void* pNext = (void*) *((int*) (pBlock + FORWARD_FREE_OFFSET));

    /* If pBlock is the head node, then reset head to pBlock's forward
       node. */
    if (!pPrev)
        *((int*) (dseg_lo + EQCLASS_OFFSET(iClass))) = (int) pNext;    
    else
    {
        /* Make the previous block point to pBlock's forward node. */
        *((int*) (pPrev + FORWARD_FREE_OFFSET)) = (int) pNext;
    }

    if (pNext)
    {
        /* Make the next block point to pBlock's backward node. */
        *((int*) (pNext + BACKWARD_FREE_OFFSET)) = (int) pPrev;
    }
}


/**********************************************************************
* void* find_free(int iClass, int iBSize)
*
* Description:
*   Find the first free block with a size greater than or equal to
*   iBSize using the given equiv. class.
**********************************************************************/
void* find_free(int iClass, int iBSize)
{
    void* pCurr;

    /* Get head of the list. */
    pCurr = (void*) *((int*) (dseg_lo + EQCLASS_OFFSET(iClass)));

#ifdef DEBUG
    if (pCurr && *((int*) pCurr) & 0x01)
    {
        fprintf(stderr, "ERROR:  Allocated block on free list in find_free.\n");
    }
#endif
    
    while (pCurr && GET_SIZE(*((int*) pCurr)) < iBSize)
    {
        pCurr = (void*) *((int*) (pCurr + FORWARD_FREE_OFFSET));

#ifdef DEBUG
        if (pCurr && *((int*) pCurr) & 0x01)
        {
            fprintf(stderr, "ERROR:  Allocated block free in find_free.\n");
        }
#endif

    }

    return pCurr;
}

/**********************************************************************
* void allocate(void* pBlock, int iNewBSize)
* 
* Description:
*   Figure out if we want to split pBlock and do so.  Then mark
*   the block allocated.
*
* Invariants:
*   + Size of pBlock is >= iNewBSize.
*   + iNewBSize is rounded to the nearest 8-bytes with space
*     for header/footer.
*   + Assume the pBlock points to a free block with all the appropriate
*     header/footer information.
**********************************************************************/
void allocate(void* pBlock, int iNewBSize)
{
    int iOrigBSize;
    int iRemainBSize;
    void* pRemainBlock;

#ifdef DEBUG
    if (pBlock && *((int*) pBlock) & 0x01)
    {
        fprintf(stderr, "ERROR:  Block already allocated (allocate).\n");
    }
#endif

    iOrigBSize = GET_SIZE(*((int*) pBlock));


    /* Get Remainder to split */
    iRemainBSize = iOrigBSize - iNewBSize;
    iRemainBSize = iRemainBSize - (iRemainBSize % 8);
    iNewBSize = iOrigBSize - iRemainBSize;

    /* If the block size is not more than 16 bytes bigger, then don't
       split */
    if (iRemainBSize < 16)
        iNewBSize = iOrigBSize;

    /* Split */
    else
    {
        /* Find new block. */
        pRemainBlock = pBlock + iNewBSize;

        /* Make header/footer and put it on the right list. */
        make_hf(pRemainBlock, iRemainBSize, 0);
        insert_free(determ_eqclass(iRemainBSize), pRemainBlock);
    }

    /* Allocate block. */
    make_hf(pBlock, iNewBSize, 1);
}


/**********************************************************************
* void* coalesce(int iReqBSize)
*
* Description:
*   Coalesce starting with the smallest equivalence class (the block
*   ahead and the block behind).  Do this for each free block and each
*   equivalence class until we find a free block large enough for the
*   requested size.
**********************************************************************/
void* coalesce(int iReqBSize)
{
    void* pFree;
    void* pPrev;
    void* pNext;

    int iNewBSize, iCurrBSize, iPrevBSize, iNextBSize;
    int iPrevAlloc, iNextAlloc;
    int i;


#ifdef DEBUG
    fprintf(stderr, "COALESCING for block size: %d\n", iReqBSize);
#endif

    /* There are no free chunks (a block that could be coalesced) that
       belong to a smaller equivalence class than the current equiv
       class. */

    for (i = 0; i < NUM_EQCLASSES; i++)
    {
        pFree = (void*) *((int*) (dseg_lo + EQCLASS_OFFSET(i)));

        /* Look through this equiv class. */
        while (pFree)
        {
            /* Coalesce with neighboring blocks. */

            iCurrBSize = GET_SIZE(*((int*) pFree));
            iNewBSize = iCurrBSize;

            pPrev = pFree - WORD_SIZE;
            pNext = pFree + iCurrBSize;

            /* If we're at the top or bottom of the heap, mark
               the appropriate block as "allocated." */
            if ( ((int) pFree <= (int) (dseg_lo + START_USE)) &&
                 ((int) (pFree + iCurrBSize) >= (int) (dseg_hi - WORD_SIZE)) )
            {
                iPrevAlloc = 1;
                iNextAlloc = 1;
            }
            else if ((int) pFree <= (int) (dseg_lo + START_USE))
            {
                iPrevAlloc = 1;
                iNextAlloc = *((int*) pNext) & 0x01;
            }
            else if ((int) (pFree + iCurrBSize) >= (int) (dseg_hi - WORD_SIZE))
            {
                iPrevAlloc = *((int*) pPrev) & 0x01;
                iNextAlloc = 1;
            }
            else
            {
                iPrevAlloc = *((int*) pPrev) & 0x01;
                iNextAlloc = *((int*) pNext) & 0x01;
            }

            /* See if both the previous and next block are free. */
            if (!iPrevAlloc && !iNextAlloc)
            {
                iPrevBSize = GET_SIZE(*((int*) pPrev));         
                iNextBSize = GET_SIZE(*((int*) pNext));

                iNewBSize += iPrevBSize + iNextBSize;

                /* Move prev pointer to head of the previous node. */
                pPrev = pFree - iPrevBSize;

                /* Remove blocks from free lists. */
                remove_free(determ_eqclass(iPrevBSize), pPrev);
                remove_free(determ_eqclass(iCurrBSize), pFree);
                remove_free(determ_eqclass(iNextBSize), pNext);

                /* Make pFree point to this new larger free block. */
                pFree = pPrev;

                /* Make the header for the new free block. */
                make_hf(pFree, iNewBSize, 0);

                /* If large enough return it. */
                if (iNewBSize >= iReqBSize) 
                    return pFree;

                /* Insert new block into the appropriate free list. */
                else
                    insert_free(determ_eqclass(iNewBSize), pFree);
            }

            /* Try previous */
            else if (!iPrevAlloc)
            {
                iPrevBSize = GET_SIZE(*((int*) pPrev));         

                iNewBSize += iPrevBSize;

                /* Move prev pointer to head of the previous node. */
                pPrev = pFree - iPrevBSize;

                /* Remove blocks from free lists. */
                remove_free(determ_eqclass(iPrevBSize), pPrev);
                remove_free(determ_eqclass(iCurrBSize), pFree);

                /* Make pFree point to this new larger free block. */
                pFree = pPrev;

                /* Make the header for the new free block. */
                make_hf(pFree, iNewBSize, 0);

                /* If large enough return it. */
                if (iNewBSize >= iReqBSize) 
                    return pFree;

                /* Insert new block into the appropriate free list. */
                else
                    insert_free(determ_eqclass(iNewBSize), pFree);
            }

            /* Try next. */
            else if (!iNextAlloc)
            {
                iNextBSize = GET_SIZE(*((int*) pNext));

                iNewBSize += iNextBSize;

                /* Remove blocks from free lists. */
                remove_free(determ_eqclass(iCurrBSize), pFree);
                remove_free(determ_eqclass(iNextBSize), pNext);

                /* Make the header for the new free block. */
                make_hf(pFree, iNewBSize, 0);

                /* If large enough return it. */
                if (iNewBSize >= iReqBSize) 
                    return pFree;

                /* Insert new block into the appropriate free list. */
                else
                    insert_free(determ_eqclass(iNewBSize), pFree);
            }

            /* Otherwise, go the next one in the equiv class. */
            else
                pFree = (void*) *((int*) (pFree + FORWARD_FREE_OFFSET));
        }
    }       

    return NULL;
}

/**********************************************************************
* int determ_eqclass(int iBSize)
*
* Description:
*   Determine the appropriate free list equivalence class given
*   a block size.
**********************************************************************/
int determ_eqclass(int iBSize)
{
    if (iBSize <= 16)           /* 16 bytes */
        return 0;
    else if (iBSize <= 32)      /* 17-32 bytes */
        return 1;
    else if (iBSize <= 64)      /* 33-64 bytes */
        return 2;
    else if (iBSize <= 128)     /* 65-128 bytes */
        return 3;
    else if (iBSize <= 256)     /* 129-256 bytes */
        return 4;
    else if (iBSize <= 512)     /* 257-512 bytes */
        return 5;
    else if (iBSize <= 1024)    /* 513-1024 bytes */
        return 6;
    else if (iBSize <= 2048)    /* 1025-2048 bytes */
        return 7;
    else if (iBSize <= 4096)    /* 2049-4096 bytes */
        return 8;
    else                        /* 4096+ bytes */
        return 9;
}
