/* $Id$ */

/*
 *  Papadimitriou Spiros
 *  spapadim+@cs.cmu.edu
 *
 *  CS213 - Lab assignment 3
 *
 */


/*   OUR ALGORITHM

Basically, we're using a buddy system. for each block of memory, we use
the first 4 bytes to store information about the block:

the first three are used to store the position of the block in a "binary
tree". each bit tells whether you go right or left (1 or 0
respectively). the 4th bit stores the depth in the tree. the bottom of
the tree has depth 0, and each leaf at that level represents a block of
8 characters. (4 of which are used for our information, so, a block of 4
bytes allocatable memory.)  Only leaves represent blocks of memory.
thus, all nodes on a down-left diagonal path have the same 3 byte
positional code. (differentiated by the depth byte.)

one exception to that layout is that we use the first bit in the depth
byte to be a "used flag" that equals 0 when the block is free. (since a
depth value of 10 equals 2^10 bytes, * 3 (since 0 depth means 8 bytes),
2^7 = a maximum depth of 128, which would mean 2^128 bytes, which is
much much larger than we'll ever need. 

the pointers we return for memory are on multiples of 8, thus the 4
bytes of information we need are not. therefore, the beginning of our
allocatable memory ignores the first 4 bytes.  thus, the last block on a
page will run 4 bytes over the end. we start out with 3 pages of memory
allocated, 1 for our management purposes, one for allocation, and one to
handle the 4 byte overflow. each time we need more, we allocate twice as
much as we currently have to use. (thus, the first time we need more,
we'll as for 2 pages, since we currently have 1 we're allocating, the
next time we'll asks for 4).

as for the management of free lists, in the page we allocated to
ourselves, we first have 24 free list "pointers", each 4 bytes long
(although we only use 3) which refer to the 3 byte (24 bit) position in
the tree of a block. if there is no block pointed to, we use the value
-1 (1111).  In blocks that are in a particular free list, we keep a
pointer to the next item in the first 4 bytes of the memory we would
normally be returning a pointer to when the memory is allocated. (thus,
before we allocate it, we grab the pointer, and fix up the list).  

Also in the management page are values of the current number of
allocatable pages (1, 2, 4, 8, etc..), and the current depth of the
tree. (Starts out at 10, when we only have 1 page. (there are 1024 =
2^10 8 byte blocks in 1 page). when we ask for more space, we simply add
one level to the top.

 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

team_t team = {
    /* Team name to be displayed on webpage */
    "Goo",
    /* First member full name */
    "Timothy Alper",
    /* First member email address */
    "taa@andrew.cmu.edu",
    /* Second member full name (leave blank if none) */
    "Dave Fry",
    /* Second member email address (blank if none) */
    "fry@andrew.cmu.edu"
};

#define PAGE_SIZE 8192
/* points to beginning of allocatable memory */
#define MY_HEAP ((long int)(dseg_lo + PAGE_SIZE))
/* Storage of Total depth of tree */
#define TREE_DEPTH (*((int *)(dseg_lo + 12*8))) 
/* Storage of Number of pages of allocatable memory */
#define NUM_PAGES (*((long int *)(dseg_lo + 13*8)))
/* Returns Position in tree of first item in freelist at height */
#define FREE_LIST(height) ((*((int *)(dseg_lo + 4*(height)))))
/* Returns Reference bytes. */
#define REF_BYTES(depth) (*((int *)(MY_HEAP + ((long int)FREE_LIST(depth) * 8L) + 4L)))
/* Returns Pointer to Allocatable memory in a block */
#define MEM_POINTER(levels) (MY_HEAP + ((long int)FREE_LIST(levels) * 8L) + 8L)

int mm_init (void)
{
  long int mem_used = mem_usage();
  int i;

  for (i = 0; i < 12; i++)
    {
      /* set all of our freelist pointers to NULL (-1) */
      *((long int *)(dseg_lo + 8*i)) = -1L; 
    }


  /* first time through, nothing's been allocated before, set up heap stuff */
  if (mem_used == -1)
    {
      if (!mem_sbrk(3L * PAGE_SIZE))  /* 3 pages, 1 for our use, 1 for 4 byte overflow */
	return -1;
      TREE_DEPTH = 10;
      NUM_PAGES = 1;
    }

  /* we've been here before, already have heap stuff set up */
  else
    {
      int num_pages_used = ((mem_used + 1) / PAGE_SIZE) - 2;
      int num_divides = 0;
      NUM_PAGES = (long int)num_pages_used;

      while (num_pages_used > 1)
	{
	  num_pages_used /= 2;
	  num_divides++;
	}
      
      TREE_DEPTH = num_divides + 10;    
     
    }

  /* set up block with position 0 and proper height */
 
  *((int *)(MY_HEAP + 4)) = TREE_DEPTH;
  *((int *)(MY_HEAP + 8)) = -1;  /* set pointer to NULL */
  FREE_LIST(TREE_DEPTH) = 0;  /* set freelist to point to block */
  

  return 0;
}

void *mm_malloc (size_t size)
{

  long int factor = 8L;  /* smallest chunk size available, + 4 */
  int chunk_depth = 0;
  int i;
  int insert_into = -1;
  int buddy_info;
  int next_item;
  int temp_item;

  while (size > (factor - 4))
    {
      factor *= 2L;
      chunk_depth++;
    }

  for ( i = chunk_depth; i <= TREE_DEPTH; i++ )
    {
      if ((int) FREE_LIST(i) != -1 )
	{
	  insert_into = i;
	  i = TREE_DEPTH+1;
	}
    }

  if ( insert_into == -1 )
    /* WE NEED MORE SPACE */
    {
      do
	{ /* ask for more memory, update reference variables */
	  if (!mem_sbrk(NUM_PAGES * PAGE_SIZE)) {
	    return NULL;   /* no memory left */
	  }
	  /* if anything is on the freelist, grab it. */
	  next_item = FREE_LIST(TREE_DEPTH);
	  /* Add new buddy to freelist */
	  FREE_LIST(TREE_DEPTH) = (int)(1 << TREE_DEPTH);
	  /* set header of new buddy */
	  REF_BYTES(TREE_DEPTH) = (int)((1 << (TREE_DEPTH + 8)) + TREE_DEPTH);
	  /* set pointer in buddy to whatever was on free list */
	  (*((int *)(MEM_POINTER(TREE_DEPTH)))) = next_item;

	  NUM_PAGES *= 2;
	  TREE_DEPTH++;
	  
	}
      while ( TREE_DEPTH <= chunk_depth );

      insert_into = TREE_DEPTH-1;
    }

  /* Now we cut the node in the freelist down to the right size */
  while (insert_into > chunk_depth)
    {
      /* decrease the depth, since we're cutting it in half. */
      REF_BYTES(insert_into)--;

      /* set the header information for our new buddy */
      buddy_info = REF_BYTES(insert_into);
      buddy_info = buddy_info ^ ((int)(1 << (insert_into - 1 + 8)));
      *((int *)(MY_HEAP + ((long int)(buddy_info >> 8) * 8L) + 4L)) = buddy_info;

      /* save the pointer to the next item in the current free list */
      next_item = *((int *)(MEM_POINTER(insert_into)));

      /* set the first buddy to point to the second buddy */
      *((int *)(MEM_POINTER(insert_into))) = (buddy_info >> 8);
 
      /* set the second buddy to point to whatever the freelist being added to is pointed at */
      *((int *)(MY_HEAP + ((long int)(buddy_info >> 8) * 8L) + 8L)) = (int)FREE_LIST(insert_into - 1);

      /* set the freelist being added to to point to first buddy */
      FREE_LIST(insert_into - 1) = FREE_LIST(insert_into);

      /* set the old freelist to point to the next item */
      FREE_LIST(insert_into) = next_item;

      insert_into--;  /* decrement current block size */

    }


  /* Now we allocate the memory we need */

  /* Set the Allocated Bit in the Block being Allocated */
  
  REF_BYTES(chunk_depth) = REF_BYTES(chunk_depth) | 128;

  next_item = *((int *)(MEM_POINTER(chunk_depth)));
  temp_item = FREE_LIST(chunk_depth);
  FREE_LIST(chunk_depth) = next_item;
  
  return (void *)( MY_HEAP + ((long int)temp_item * 8L) + 8L);
  
}

void mm_free (void *ptr)
{
  int depth;
  int size;
  int i;
  int j;
  int buddy_pos;
  int is_used;
  int same_size;

  /* Remove the "Allocated Bit" */
  (*((int *)(ptr - 4L))) = (*((int *)(ptr - 4L))) & (~((int)128));

  /* Get the depth of this pointer */
  depth = (*((int *)(ptr - 4L))) % 128;

  /* Calculate the actual size of this buddy */
  size = (1 << (depth)); /*size of buddy in #pointers*/

  /* Calculate the Buddy's position in the tree */
  buddy_pos = ((*((int *)(ptr - 4L))) >> 8) ^ size;

  /* Is the buddy already allocated? */
  same_size = (((*((int *)(8L * (long int)buddy_pos + MY_HEAP + 4L))) % 128) == depth);

  is_used = ((*((int*)((long int)buddy_pos * 8L + MY_HEAP + 4L)))) & 128;

  if((!is_used) && same_size && (depth < TREE_DEPTH))
    {
      while((!is_used) && same_size && (depth < TREE_DEPTH))
	{
	  
	  i = FREE_LIST(depth);
	  j = -1;

	  while(i != buddy_pos)
	    {
	      j = i;
	      i = *((int*)((long int)i * 8L + MY_HEAP + 8L)); 
	    }
	  
	  if(j != -1)
	    {
	      /* the parent is not the FREE_LIST */
	      (*((int*)((long int)j * 8L + MY_HEAP + 8L))) = (*((int*)((long int)buddy_pos * 8L + MY_HEAP + 8L))); 
	      
	    }
	  else
	    {
	      /* the parent is the FREE_LIST */
	      FREE_LIST(depth) = *((int*)((long int)buddy_pos * 8L + MY_HEAP + 8L));
	      
	    }
	  /* Time to place the buddy back on the FREE_LIST with the new depth */
	  buddy_pos = buddy_pos & (~size);
	  
	  (*((int*)((long int)buddy_pos * 8L + MY_HEAP + 4L)))++;  /* update depth */
	  depth++;
	  (*((int*)((long int)buddy_pos * 8L + MY_HEAP + 8L))) = FREE_LIST(depth);
	  FREE_LIST(depth) = buddy_pos;
	  
	  size *= 2;

	  buddy_pos = buddy_pos ^ size;

	  same_size = (((*((int *)(8L * (long int)buddy_pos + MY_HEAP + 4L))) % 128) == depth);
	   
	 
	  is_used = (((*((int*)((long int)buddy_pos * 8L + MY_HEAP + 4L)))) & 128);
	}
    }
  else
    {
      /* No need to coalesce */
      
      (*((int*)(ptr))) = FREE_LIST(depth);
      FREE_LIST(depth) = ((*((int *)(ptr - 4L))) >> 8);
      
    }
  
  return;

}
