/* $Id$ */

/*
 *  Papadimitriou Spiros
 *  spapadim+@cs.cmu.edu
 *
 *  CS213 - Lab assignment 3
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>
#include <string.h>
#include <limits.h>

#include "memlib.h"
#include "malloc.h"

team_t team = {
	/* Team name to be displayed on webpage */
	"DC Enterprises",
	/* First member full name */
	"Cuong Do",
	/* First member email address */
	"coondog@cmu.edu",
	/* Second member full name (leave blank if none) */
	"",
	/* Second member email address (blank if none) */
	""
};



/*****************************************************************************
 * data structures
 */


/* segregated block list structure */
typedef struct chunk_list_tag {
	size_t                   size;        /* last bit is used bit */
	size_t                   prev_size;
	struct chunk_list_tag*   prev;
	struct chunk_list_tag*   next;
} chunk_list_t;


/* structure for beginning of list */
typedef struct {
	size_t         min_size;
	chunk_list_t*  begin;
} chunk_list_begin_t;


/*****************************************************************************
 * macros 
 */

#define DEBUG                         0
#define INITIAL_CHUNK_SIZE            4096L
#define NUM_LISTS                     111L
#define CHUNK_LIST_SIZE               32L
#define CHUNK_LIST_BEGIN_SIZE         16L
#define MIN_CHUNK_SIZE                CHUNK_LIST_SIZE
#define ACTIVE_HDR_SIZE               16L
#define INACTIVE_HDR_SIZE             CHUNK_LIST_SIZE
#define METADATA_SIZE                 (CHUNK_LIST_BEGIN_SIZE * NUM_LISTS)


#if DEBUG
#define DBG(x)  x
#else
#define DBG(X) ;
#endif

#define setup_first_list(num, sz) \
	for (i = 0; i <= (num); i++) {\
		/*DBG( fprintf(stderr, "i = %d\n", i) );*/ \
		g_chunk_lists[i].min_size = MIN_CHUNK_SIZE + (sz) * i;\
	}

#define setup_list(num, sz)  \
	for (old_i = i; i <= old_i + (num); i++)  {\
		/*DBG( fprintf(stderr, "i = %d\n", i) );*/ \
		g_chunk_lists[i].min_size = g_chunk_lists[i - 1].min_size + (sz);\
	}

#define setup_last_list(num, sz)  \
	for (old_i = i; i < old_i + (num); i++)  {\
		/*DBG( fprintf(stderr, "i = %d\n", i) );*/ \
		g_chunk_lists[i].min_size = g_chunk_lists[i - 1].min_size + (sz);\
	}

#define round8(num) ((((num) + 7L) / 8L) * 8L)

#define in_use(chunk) ((chunk)->size & 1L)
	
/*****************************************************************************
 * global variables (pointers to places in heap)
 */

chunk_list_begin_t*   g_chunk_lists;
chunk_list_t*         g_last_chunk;


/*****************************************************************************
 * function prototypes
 */

int find_list(size_t size);
void delete_entry(void *ptr, int list_num);
void insert_entry2(void *ptr, int list_num);
void insert_entry(void *ptr);
void move_entry(void *ptr, size_t old_size, size_t new_size);
void print_list(int list_num);
char *range_str(void *ptr);

/*****************************************************************************
 * function implementations
 */

int mm_init (void)
{
	chunk_list_t *chunk_list;
	int i, old_i;
	void *first_chunk;
	
	/* avoid stupid mistakes with constants */
	assert(sizeof(chunk_list_t) == CHUNK_LIST_SIZE);
	assert(sizeof(void*) * 2 == ACTIVE_HDR_SIZE);
	assert(sizeof(chunk_list_begin_t) == CHUNK_LIST_BEGIN_SIZE);
	
	/* account for room to store lists */
	if (NULL == mem_sbrk(INITIAL_CHUNK_SIZE + METADATA_SIZE)) {
		printf("ERROR: unable to allocate initial %ld bytes of memory\n",
				INITIAL_CHUNK_SIZE + METADATA_SIZE);
		return -1;
	}

	DBG( printf("allocated %ld bytes of memory for metadata\n", 
				METADATA_SIZE) );

	/* initialize and place header and boundary tag on the initial block */
	first_chunk = dseg_lo + METADATA_SIZE;
	chunk_list = (chunk_list_t *)first_chunk;
	chunk_list->size = INITIAL_CHUNK_SIZE << 1L;
	chunk_list->next = chunk_list->prev = NULL;
	chunk_list->prev_size = 0;
	
	/* the last chunk is stored separately */
	g_last_chunk = first_chunk;
	DBG( printf("Initialized header for initial block.\n") );

	/* initialize segregated lists */
	g_chunk_lists = (chunk_list_begin_t *)(void *)dseg_lo;
	memset(g_chunk_lists, 0, CHUNK_LIST_BEGIN_SIZE * NUM_LISTS);

	setup_first_list(44, 8);        /* make a fibonacci-like distribution of lists */
	setup_list(21, 16);             
	setup_list(13, 32);             /* there are probably going to be fewer large  */ 
	setup_list(8, 64);              /* chunks than small chunks, so have more      */
	setup_list(5, 128);             /* lists whose minimum chunk size is small     */
	setup_list(3, 256);
	setup_list(2, 512);
	setup_list(1, 1024);
	setup_list(1, 2048);
	setup_list(1, 4096);
	setup_last_list(1, 8192);

	/* setup last list differently... this will always contain any chunks
	 * that don't fit into the other lists */ 
	g_chunk_lists[i].min_size = LONG_MAX;
	g_chunk_lists[i].begin = NULL;

	DBG( printf("*** mm_init finished ***\n\n") );

	return 0;
}


void *mm_malloc(size_t size)
{
	int i;
	long diff, rounded_size = round8(size);
	size_t req_size;
	void *old_chunk, *new_chunk, *next_chunk;
	chunk_list_t *old_chunk_list, *new_chunk_list;

	req_size = rounded_size + ACTIVE_HDR_SIZE;
	if (req_size < MIN_CHUNK_SIZE)
		req_size = MIN_CHUNK_SIZE;
	
	for (i = find_list(req_size); i < NUM_LISTS - 1; i++) {
		/* iterate through the free chunk lists, starting from the first one
		 * that could hold a chunk large enough for the requested size */
		
		/* find where the chunk begins and calculate the difference
		 * between its size and the requested size */
		old_chunk_list = g_chunk_lists[i].begin;
		while ( old_chunk_list != NULL && ((old_chunk_list->size >> 1L) < req_size) )
		{
			old_chunk_list = old_chunk_list->next;
		}

		/* go to next list if the current list doesn't contain any free
		 * chunks */
		if (!old_chunk_list)
			continue;

		old_chunk = old_chunk_list;

		diff = ( old_chunk_list->size >> 1L) - req_size; 

		/* if the fit is exact, or the new chunk would be smaller than the
		 * minimum chunk size, just set the used bit in the chunk and
		 * return it */
		if (diff < MIN_CHUNK_SIZE) {
			delete_entry(old_chunk, find_list(old_chunk_list->size >> 1L));
			old_chunk_list->size = 
				(((old_chunk_list->size >> 1L) - ACTIVE_HDR_SIZE) << 1L) | 1L;

			return (void *)((size_t)old_chunk + ACTIVE_HDR_SIZE);
		}

		/* split chunk if the new chunk would be greater than or equal to
		 * the minimum chunk size */
		else {
			/* set size of new chunk */
			new_chunk = (char *)old_chunk + req_size;

			new_chunk_list = (chunk_list_t *)new_chunk;
			new_chunk_list->size = diff << 1L;
			new_chunk_list->prev_size = (size_t)new_chunk_list - (size_t)old_chunk;
			new_chunk_list->prev = old_chunk_list->prev;
			new_chunk_list->next = old_chunk_list->next;

			next_chunk = (void *)((char *)new_chunk_list + diff);
			((chunk_list_t *)next_chunk)->prev_size = diff;

			delete_entry(old_chunk, find_list(old_chunk_list->size >> 1L));

			/* adjust size of original chunk and move it to proper list*/
			old_chunk_list->size = ((req_size - ACTIVE_HDR_SIZE) << 1L) | 1L;

			/* insert new chunk into correct list */
			insert_entry(new_chunk);

			return (char *)old_chunk + ACTIVE_HDR_SIZE;
		}
	}

	/*************************************************/
	/* if we got here, we have to use the last block */
	/*************************************************/
	diff = (g_last_chunk->size >> 1L) - req_size;

	/* not enough memory in last chunk to accomodate request, so allocate more
	 * memory */
	if (diff < 0) {
		/* allocate just enough memory to accomodate the requested size and
		 * another chunk that's exactly large enough to hold a block list node
		 */
		if (mem_sbrk(-diff + MIN_CHUNK_SIZE) == NULL) {
			return NULL;
		}
		
		old_chunk = g_last_chunk;
		((chunk_list_t *)old_chunk)->size = ((req_size - ACTIVE_HDR_SIZE) << 1L) | 1L;
		
		/* calculate where last chunk is, copy the block list info to it, and
		 * update the size */
		g_last_chunk = (void *)(dseg_hi - MIN_CHUNK_SIZE + 1L);
		new_chunk_list = g_last_chunk;
		new_chunk_list->size = MIN_CHUNK_SIZE << 1L;
		new_chunk_list->prev_size = (size_t)g_last_chunk - (size_t)old_chunk;
		new_chunk_list->prev = new_chunk_list->next = NULL;

		return (char *)old_chunk + ACTIVE_HDR_SIZE;
	}
	
	/* if leftover chunk is smaller than the minimum size, allocate a new
	 * chunk of the minimum size, and return the whole (previous) last chunk */
	if (diff < MIN_CHUNK_SIZE) {
		old_chunk = g_last_chunk;
		old_chunk_list = old_chunk;
		old_chunk_list->size = ((old_chunk_list->size >> 1L) - ACTIVE_HDR_SIZE) << 1L | 1L;
		g_last_chunk = (void*)(dseg_hi + 1);

		/* grab enough memory for the new chunk */
		if (mem_sbrk(MIN_CHUNK_SIZE) == NULL) {
			return NULL;
		}

		/* copy old block list info to new chunk and update size */
		new_chunk_list = g_last_chunk;
		new_chunk_list->size = MIN_CHUNK_SIZE << 1L;
		new_chunk_list->prev_size = (size_t)new_chunk_list - (size_t)old_chunk_list;
		new_chunk_list->prev = new_chunk_list->next = NULL;

		return (char *)old_chunk + ACTIVE_HDR_SIZE;
	}

	/* if we get here, the leftover chunk is big enough to hold a block list
	 * node, so split the last chunk */
	old_chunk = g_last_chunk;
	g_last_chunk = (void*)((char *)old_chunk + req_size);

	/* update size on original last chunk */
	((chunk_list_t *)old_chunk)->size = ((req_size - ACTIVE_HDR_SIZE) << 1L) | 1L;

	/* calculate where last chunk is, copy the block list info to it, and
	 * update the size and previous size */
	new_chunk_list = g_last_chunk;
	new_chunk_list->size = diff << 1L;
	new_chunk_list->prev_size = (size_t)new_chunk_list - (size_t)old_chunk; 
	new_chunk_list->next = NULL;
	new_chunk_list->prev = NULL;

	return (char *)old_chunk + ACTIVE_HDR_SIZE;
}


void print_list(int list_num)
{
	chunk_list_t *curr = g_chunk_lists[list_num].begin;

	while (curr) {
		fprintf(stderr, "%p -> ", curr);
		curr = curr->next;
	}

	fprintf(stderr, "NULL\n");
}


char *range_str(void *ptr)
{
	size_t size = ((chunk_list_t*)ptr)->size >> 1L;
	int    in_use = ((chunk_list_t*)ptr)->size & 1L;
	static char buf[80];

	if (in_use)
		sprintf(buf, "%p - %p (%ld bytes)]", ptr, (void*)((size_t)ptr + size + 15L), size);
	else
		sprintf(buf, "%p - %p (%ld bytes)]", ptr, (void*)((size_t)ptr + size - 1L), size);

	return buf;
}


/* finds an entry in the segregated lists and returns the index to the list */
int find_list(size_t size) 
{
	int left = 0, right = NUM_LISTS - 2, pos;

	pos = (right - left) / 2;
	while (right >= left) {
		pos = (left + right) / 2;
		if ( (size >= g_chunk_lists[pos].min_size) 
				&& (size < g_chunk_lists[pos + 1].min_size) ) 
			return pos;

		if (size < g_chunk_lists[pos].min_size) 
			right = pos - 1;
		else
			left = pos + 1;
	}

	fprintf(stderr, "Could not find target!\n");
	exit(1);
}


/* deletes an entry from a chunk list */
void delete_entry(void *ptr, int list_num)
{	
	chunk_list_t *list = ptr;

	/* link previous node with next node */
	if (list->prev)
		list->prev->next = list->next;
	else
		g_chunk_lists[list_num].begin = list->next;

	/* link next node with previous node */
	if (list->next)
		list->next->prev = list->prev;
}


void move_entry(void *ptr, size_t old_size, size_t new_size)
{
	int i;

	/* don't move entry if it still belongs in its current list */
	i = find_list(old_size);
	if ( (new_size >= g_chunk_lists[i].min_size)
			&& (new_size < g_chunk_lists[i + 1].min_size))
		return;
		
	delete_entry(ptr, i);
	
	i = find_list(new_size);
	insert_entry2(ptr, i);
	((chunk_list_t *)ptr)->size = (new_size) << 1L;
}


/* inserts an entry directly into a list, given a list number */
void insert_entry2(void *ptr, int list_num)
{
	chunk_list_t *list = (chunk_list_t *)ptr;
	
	/* link the new element properly */
	list->prev = NULL;
	list->next = g_chunk_lists[list_num].begin;

	/* link original beginning of list to new entry */
	if (g_chunk_lists[list_num].begin)
		g_chunk_lists[list_num].begin->prev = list;

	g_chunk_lists[list_num].begin = ptr;
}


/* inserts an entry into the proper chunk list */
void insert_entry(void *ptr)
{	
	int i;
	size_t size = ((chunk_list_t *)ptr)->size >> 1L;

	i = find_list(size);

	insert_entry2(ptr, i);
}


void mm_free (void *ptr)
{
	chunk_list_t *chunk_list       = NULL;
	chunk_list_t *prev_chunk_list  = NULL;
	chunk_list_t *next_chunk_list  = NULL;
	int          coalesced         = FALSE;
    size_t orig_size = 0L, new_size = 0L, prev_size = 0L;

	if (ptr == NULL)
		return;

	ptr = (char *)ptr - ACTIVE_HDR_SIZE;
	chunk_list = (chunk_list_t *)ptr;

	/* find the previous chunk */
	if ((size_t)chunk_list > (size_t)dseg_lo + METADATA_SIZE) 
		prev_chunk_list = (void*)((char *)chunk_list - chunk_list->prev_size);

	/* find next chunk */
	if ((size_t)chunk_list + (chunk_list->size >> 1L) + ACTIVE_HDR_SIZE 
			< (size_t)dseg_hi)
		next_chunk_list = (void *)((size_t)chunk_list 
				+ (chunk_list->size >> 1L) + ACTIVE_HDR_SIZE);

	/* coalesce this chunk with the previous chunk, if possible */
	if (prev_chunk_list && !in_use(prev_chunk_list)) {
		/* at most one of "used" bits will be sit, so this is safe */
		prev_size = orig_size = prev_chunk_list->size >> 1L;
		new_size =  orig_size + (chunk_list->size >> 1L) + ACTIVE_HDR_SIZE;
	
//		move_entry(prev_chunk_list, orig_size, new_size);

		/* store new size of list */
		prev_chunk_list->size = new_size << 1L;
		chunk_list = prev_chunk_list; 

		if (next_chunk_list) 
			next_chunk_list->prev_size = new_size;
			
		/* move new coalesced block to proper list */
		coalesced = TRUE;
	}
	
	/* coalesce this chunk with the next chunk, if possible */
	if (next_chunk_list && !in_use(next_chunk_list) && next_chunk_list != g_last_chunk)  {
		/* if this chunk is adjacent to a chunk that's not the last chunk,
		 * coalesce the two chunks */
	
		orig_size = chunk_list->size >> 1L;
		if (!coalesced) {
			new_size = (chunk_list->size >> 1L) + (next_chunk_list->size >> 1L)
				+ ACTIVE_HDR_SIZE;
		}
		else {
			new_size = (chunk_list->size >> 1L) + (next_chunk_list->size >> 1L);
		}
			
		chunk_list->size = new_size << 1L;

		/* delete the entry for the next chunk and move the entry for the
		 * current, enlarged chunk to its new list */
		delete_entry(next_chunk_list, find_list(next_chunk_list->size >> 1L));
	
		next_chunk_list = (void *)((char *)chunk_list + new_size);

		if ((size_t)next_chunk_list < (size_t)dseg_hi)
			next_chunk_list->prev_size = new_size;
	
		if (!coalesced)
			insert_entry(chunk_list);
		else
			move_entry(prev_chunk_list, prev_size, new_size);
	
		return;
	}	
	
	if (coalesced) {
		move_entry(prev_chunk_list, prev_size, new_size);
		return;
	}

	/* if this chunk can't be coalesced with its neighbors, insert this chunk 
	 * in front of the list to which it belongs */
	chunk_list->size = ((chunk_list->size >> 1L) + ACTIVE_HDR_SIZE) << 1L;
	insert_entry(ptr);
}
