/* $Id$ */

/*
 *  Papadimitriou Spiros
 *  spapadim+@cs.cmu.edu
 *
 *  CS213 - Lab assignment 3
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

// choose one (or none, and it will be alpha :)
//#define DLINUX
//#define DALPHA

team_t team = {
    /* Team name to be displayed on webpage */
    "Stony House Bandits",
    /* First member full name */
    "Kipton Barros",
    /* First member email address */
    "kbarros",
    /* Second member full name (leave blank if none) */
    "Joshua Baer",
    /* Second member email address (blank if none) */
    "shaddar"
};


/*
 * The algorithm is this:
 * The heap is divided into blocks.  Each block is either occupied or free.
 * Each block has it's size stored on both it's ends.  Embedded in the least
 * significant bit of this size is a flag: 0 for free, 1 for occupied.  All
 * this is used for coalescing.
 * All the free blocks are stored in a binary tree.  A pointer to the root
 * of this tree is stored at the beginning of the heap, in the 'header'.
 * On calls to malloc, a suitable block is chosen from the tree.  If it's
 * much bigger than needed, the extra space is 'shaved' off and put back
 * into the tree as a new block.  If no block of suitable size is found, we
 * will increase the heap and add it as a free block to the tree.
 * On calls to free, first we will try to coalesce the block with it's
 * neighbors, (removing them from the tree, if necessary), and then add the
 * (possibly expanded) block into the tree.
 * 
 * Notes: This was written to be mostly platform independent.  However there
 * are a couple strange things.  First, for this assignment, all pointers
 * returned were requested to be 8 byte alligned.  However, on most machines,
 * the pointer size is 4 bytes.
 *
 * Things to change to get this to run on a 32 bit machine s.t. malloc returns
 * 8 byte aligned pointers:
 *    MIN_BLOCK_SIZE
 */



///////////////////////
// TYPES
///////////////////////

// boolean typdefs
enum boolean {true, false};
typedef enum boolean bool;

// header for unallocated block
typedef struct block {
    // total size of this free block (including boundry data).
    // LSB stores allocated bit.
    size_t size1;
    
    // these are used in unallocated blocks
    struct block *P; // parent
    struct block *L; // left child
    struct block *R; // right child
    
    // variable amount of data stored here.
    // ....

    // this is at the end of the structure; LSB stores allocated bit.
    // int size2;
} block;

///////////////////////
// GLOBAL VARIABLES
///////////////////////

// root of the free-tree
block *root;

// invariant: there is always size_t more memory allocated beyond
// boundry.
block *boundry;

///////////////////////
// GENERAL MACROS
///////////////////////

#define MAX(a,b) ((a)>(b)?(a):(b))


///////////////////////
// BLOCK MACROS
///////////////////////

// the minimum block size
#ifdef DLINUX
// Use this if you're on a 32 bit machine but want bytes returned to
// be 8 byte aligned.  (goofy, i'll admit).
#define MIN_BLOCK_SIZE (3L*sizeof(void*)+2L*sizeof(size_t) + 4L)
#else
// This is standard
#define MIN_BLOCK_SIZE (3L*sizeof(void*)+2L*sizeof(size_t))
#endif

// memory blocks have data on the ends; cells are what the user sees
#define BLOCK2CELL(b) (void*)((long)(b)+sizeof(size_t))
#define CELL2BLOCK(c) (block*)((long)(c)-sizeof(size_t))


// previous block's size2; this is fast.
#define PREV_SIZE2(b) ((size_t*)((long)(b)-sizeof(size_t)))

// previous block
#define PREV(b) ((block*)((long)b-(*PREV_SIZE2(b)&(~1L))))
// next block's size1; takes as param a block
#define NEXT(b) ((block *)((long)(b)+((b)->size1&(~1L))))

// gets this block's size2
#define SIZE2(b) ((size_t*)((long)NEXT(b)-sizeof(size_t)))



// rounds a number up to allign by 8.  on 32 bit machines, change this
// to allign by 4.
inline size_t sizeAlign(size_t s) {
    if ((s & 7L) == 0L)
	return s;

    return s + 8L - (s & 7L);
}

int mm_init (void)
{
    if (mem_usage() <= 0) {
	if (!mem_sbrk(2 * sizeof(size_t)))
	    return -1;
    }
    
    // the free-tree is currently empty.
    root = NULL;

    // set 2 fake blocks as always occupied.  'boundry' is between them.
    *(size_t *)(dseg_lo) = 1L;
    boundry = (block *) ((long) dseg_lo + sizeof(size_t));
    boundry->size1 = 1L;

    return 0;
}

inline block *treeMinimum(block *x) {
    while (x->L)
	x = x->L;
    return x;
}

inline block *treeSuccessor(block *x) {
    block *y;

    if (x->R)
	return treeMinimum(x->R);
    
    y = x->P;
    while (y && (x == y->R)) {
	x = y;
	y = y->P;
    }

    return y;
}

void treeInsert(block *z) {
    block *x, *y;

    y = NULL;
    x = root;
    while (x) {
	y = x;
	if (z->size1 < x->size1)
	    x = x->L;
	else if (z->size1 > x->size1)
	    x = x->R;
	// this condition is optional.  it means subtrees can't be
	// inserted, but speeds up certain cases (binary.rep).
	else {
	    z->L = y->L;
	    if (z->L)
		z->L->P = z;
	    break;
	}
    }
    z->P = y;
    if (!y)
	root = z;
    else if (z->size1 > y->size1)
	y->R = z;
    else
	y->L = z;
}

void treeRemove(block *z) {
    block *x, *y;

    if (!z->L || !z->R)
	y = z;
    else
	y = treeSuccessor(z);

    if (y->L)
	x = y->L;
    else
	x = y->R;

    // splice y out
    if (x)
	x->P = y->P;

    if (!y->P)
	root = x;
    else {
	if (y == y->P->L)
	    y->P->L = x;
	else
	    y->P->R = x;
    }
    
    // swap z with y
    if (y != z) {
	// link left and right children
	y->L = z->L;
	if (y->L) y->L->P = y;
	y->R = z->R;
	if (y->R) y->R->P = y;

	// link parent
	if (z->P) {
	    if (z == z->P->L)
		z->P->L = y;
	    else
		z->P->R = y;
	}
	else {
	    root = y;
	}
	y->P = z->P;
    }
}

// gets the smallest block which has size at least 'size', or NULL if
// none exist.
inline block *removeLeastFit(size_t size) {
    block *n = root;
    block *lg = NULL; // the least block greater than size
    
    // locate the smallest such block
    while (n) {
	if (size < n->size1) {
	    lg = n;
	    n = n->L;
	}
	else if (size > n->size1) {
	    n = n->R;
	}
	else if (size == n->size1) {
	    lg = n;
	    break;
	}
    }
    if (!lg) return NULL;

    // remove it from the tree
    treeRemove(lg);

    return lg;
}

// shave of whatever extra there is from the block, and return it to the
// free-tree.  ('size' is all we need).
inline void shaveBlock(block *b, size_t size) {
    // if the block is big enough, shave what we can off from 'b' and put
    // it back in the tree as block 'c'.
    int c_size = b->size1 - size;
    if (c_size >= MIN_BLOCK_SIZE) {
	// the new block to add
	block *c;

	// set the new size, and the occupied bit
	b->size1 = size | 1L;
	*SIZE2(b) = size | 1L;
	
	// create the extra block, and add it to the tree
	c = NEXT(b);
	c->size1 = c_size;
	c->P = c->L = c->R = NULL;
	*SIZE2(c) = c_size;
	treeInsert(c);
    }
    // otherwise, don't bother shaving.
    else {
	// just set the occupied bit
	b->size1 |= 1L;
	*SIZE2(b) |= 1L;
    }
}

// if it's deemed reasonable, will expand the heap by at least 'size', and
// return true.  otherwise, will return false.
inline block *increaseHeap(size_t size) {
    block *a, *b;
    size_t increase;

    b = boundry;

    // try to coalesce with previous block
    if (!(*PREV_SIZE2(boundry) & 1L)) {
	a = PREV(boundry);
	treeRemove(a);
	increase = size - a->size1;
    }
    else {
	a = boundry;
	increase = size;
    }
    
    // try to increase the heap
    if (!mem_sbrk(increase)) {
	return NULL;
    }
    
    // set a's size correctly
    a->size1 = size;
    *SIZE2(a) = a->size1;
    
    // reset boundry
    boundry = NEXT(a);
    boundry->size1 = 1L;

    return a;
}


void *mm_malloc (size_t size) {
    block *b;

    // since each block takes extra space at it's edges, and needs to be
    // 8 byte aligned, we'll increase size by a little.
    size += sizeof(size_t) * 2L;
    size = sizeAlign(size);
    size = MAX(size, MIN_BLOCK_SIZE);

    // find the best fit in our free-tree for this size.
    b = removeLeastFit(size);

    // if there was nothing suitable in the tree, expand the heap.
    if (!b) {
	b = increaseHeap(size);
    }

    // if the heap wouldn't expand, return NULL;
    if (!b)
	return NULL;

    // shave off what we don't need from the block. 
    shaveBlock(b, size);

    return BLOCK2CELL(b);
}


void mm_free (void *ptr)
{
    block *b = CELL2BLOCK(ptr);

    // remove the occupied (LSB) bit
    b->size1 &= (~1L);
    *SIZE2(b) &= (~1L);

    // try to coalesce with previous block
    if (!(*PREV_SIZE2(b) & 1L)) {
	block *a = PREV(b);
       
	treeRemove(a);

	a->size1 = a->size1 + b->size1;
	*SIZE2(a) = a->size1;
	
	b = a;
    }

    // try to coalesce with next block
    if (!(NEXT(b)->size1 & 1L)) {
	block *c = NEXT(b);

	treeRemove(c);

	b->size1 = b->size1 + c->size1;
	*SIZE2(b) = b->size1;
    }
    
    b->P = b->L = b->R = NULL;
    treeInsert(b);
}


