/* $Id$ */

/*
 *  Papadimitriou Spiros
 *  spapadim+@cs.cmu.edu
 *
 *  CS213 - Lab assignment 3
 *
 */

/*
 *  Comments on this solution?  This is quite a strange solution.  It uses a block
 *  bitmap to store which chunks are in use.  The granularity of the bitmap is currently
 *  defined at compile time, but it could quite easily be turned into a runtime variable.
 *  When a block is filled up, a new block is then created.  If the block required doesn't
 *  fit, then a bigger block than the default size is created.
 *  
 *  It's fairly obvious that this solution is not really an ideal solution for a
 *  general purpose multi-tasking memory allocation routine.  I think a system like this
 *  would be good for something like, say, an embedded system without much memory.  Or
 *  pretty much any system where you were given a block of memory that was entirely
 *  yours, which you had to manage properly.  Given a system like that, free block
 *  searching could be biased towards certain sections for large and small blocks
 *  to minimize fragmentation as well.
  *  
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"


team_t team = {
    /* Team name to be displayed on webpage */
    "\\0",
    /* First member full name */
    "Gopi Flaherty",
    /* First member email address */
    "gf2e",
    /* Second member full name (leave blank if none) */
    "John Tunison",
    /* Second member email address (blank if none) */
    "jtunison"
};

//Default maximum block size
#define block_size 262144
//How many bytes to allocate initially
#define initial_size 8192
//How many bytes per bit in the bitmap
#define granularity 256
//How many bytes in the bitmap?
#define bitmap_size (block_size/granularity/8)


typedef struct
{
	unsigned long current_size;				//Current used size, in bytes
	unsigned long max_size;
	unsigned long bitmap_length;
	unsigned long free_space;
	char *end;
	char *next;						//Next block structure
	char *next_to_use;				//Where to store the next data block
	unsigned long *bitmap;			//address of the bitmap structure
	char *storage;					//The actual data area
} large_head_t;

typedef struct
{
	large_head_t *last_block;
	large_head_t *large_block;
} global_head_t;

void
mark_used(unsigned long *bitmap, unsigned long location, unsigned long length)
{
	unsigned long count;
	
	
	for(count=location/granularity;count<((length+location)/granularity);count++)
	{
		bitmap[count/64] |= (1L<<(count%64));
	}

}

void
mark_free(unsigned long *bitmap, unsigned long location, unsigned long length)
{
	unsigned long count;
	
	for(count=location/granularity;count<((length+location)/granularity);count++)
	{
		bitmap[count/64] &= ~(1L<<(count%64));
	}

}


unsigned long find_block(unsigned long *bitmap, unsigned long bitmap_length, unsigned long length)
{
	unsigned long count, icount;

	for(count=0;count<(bitmap_length*8 - length/granularity);count++)
	{
		for(icount = 0; icount<(length/granularity); icount++)
		{
			if(bitmap[(count+icount)/64]&(1L<< ((count+icount)%64)))
			{
				count += icount;
				break;
			}
		}
		
		if(icount == length/granularity)
		{
			return count+1;
		}
	}
	return 0;
}

int mm_init (void)
{
	global_head_t *global_head;
	large_head_t *large_head;
	int count;
	
	if(!mem_sbrk(sizeof(global_head_t) + sizeof(large_head_t) + initial_size + bitmap_size + 500)) return -1;
	
	global_head = (global_head_t *)dseg_lo;
	global_head->large_block = large_head = global_head + sizeof(global_head_t);
	global_head->last_block = large_head;
	
	large_head->current_size = initial_size;
	large_head->next = NULL;
	large_head->bitmap = (unsigned long)large_head + sizeof(large_head_t);
	large_head->storage = (unsigned long)large_head->bitmap + bitmap_size;
	large_head->free_space = block_size;
	large_head->max_size = block_size;
	large_head->bitmap_length = bitmap_size;
	
	for(count = 0; count<(bitmap_size/8); count++)
	{
		large_head->bitmap[count] = 0L;
	}
	
	if((unsigned long)large_head->storage%8)
		large_head->storage += 8 - (unsigned long)large_head->storage%8;
	
	large_head->next_to_use = large_head->storage;
	
	large_head->end = large_head->storage + (unsigned long)initial_size;
	
	return -1;
}

void mm_free (void *ptr)
{
	large_head_t *cur;
	
	cur = ((global_head_t *)dseg_lo)->large_block;
	
	ptr -= 8;

	do
	{
		if((ptr >= cur->storage)&&(ptr<=(cur->storage + cur->current_size)))
		{
			mark_free(cur->bitmap, (unsigned long)ptr - (unsigned long)cur->storage, *((unsigned long*)ptr));
			
			break;
		}
		cur = cur->next;
	}while(cur!=NULL);
	
	//If cur gets to be null, free was passed invalid data.
	assert(cur!=NULL);
	
}


large_head_t *make_block(large_head_t *last, unsigned long start_size, unsigned long max_size)
{
	//Create and allocate memory for a new structure...
	large_head_t *ret = last->max_size + last->storage;
	int count;
	
	if(!mem_sbrk(last->max_size - last->current_size + sizeof(large_head_t) + start_size + max_size/granularity/8 + 8192))
		return NULL;	//Whoops!  No more memory to allocate...
	
	last->next = ret;
	ret->next = NULL;
	ret->max_size = max_size;
	ret->current_size = start_size;
	ret->bitmap = ret + sizeof(large_head_t);
	ret->bitmap_length = max_size/granularity/8;
	ret->storage = ret->bitmap + ret->bitmap_length;
	ret->end = ret->storage + start_size;
	ret->next_to_use = ret->storage;
	ret->free_space = max_size;
	
	for(count=0; count < ret->bitmap_length/8; count++)
		ret->bitmap[count] = 0L;
	
	return ret;
}

char *get_block(large_head_t *block, size_t size)
{
	char *my_new;
	

	if((block->free_space>size)&&(my_new = find_block(block->bitmap, block->bitmap_length, size)))
	{
		my_new = ((unsigned long)my_new - 1) * granularity;
		if(my_new + size > block->current_size)
		{
			if(!mem_sbrk(my_new + size -block->current_size))
				return NULL;
			
			block->current_size = my_new + size;
		}
		block->end = my_new + 8 + (unsigned long)block->storage;
		
		mark_used(block->bitmap, my_new, size);
		
		*(unsigned long*)((unsigned long)block->storage + (unsigned long)my_new) = size;
		
		return my_new + (unsigned long)block->storage + 8;
	}
	
	return NULL;
}

void *mm_malloc(size_t size)
{
	large_head_t *cur = ((global_head_t *)dseg_lo)->large_block;
	char *my_new;
	
	size += 8;
	
	if(((unsigned long)size)%granularity) 
	{
		size = (void *)((unsigned long)size + granularity - size%granularity);
	}
	
	do
	{
		if(my_new = get_block(cur, size))
		{
			return my_new;
		}
		
		cur = cur->next;
	} while(cur!=NULL);
	
	//OK, couldn't find any space in the existing blocks; let's make a new one...
	
	cur = ((global_head_t *)dseg_lo)->last_block;
	
	if(size>block_size)
		cur = make_block(cur, size, size * 2);
	else
		cur = make_block(cur, size, block_size);
	
	if(cur==NULL)
		return NULL;
	
	((global_head_t *)dseg_lo)->last_block->next = cur;
	((global_head_t *)dseg_lo)->last_block = cur;
	
	return get_block(cur, size);
}


