
/*
 *  Papadimitriou Spiros
 *  spapadim+@cs.cmu.edu
 *
 *  CS213 - Lab assignment 3
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

typedef enum{ false=0, true=1 } bool;
//typedef unsigned long int ulong;

/*---------------------------------        INFO */
/*

we implemented a loose segregated list system.  There was a bit of a fuzzy
hueristic that tried to find a chunk of memory for malloc.  it would have
been much more efficient to handle free pages and keep them lying around so
any hash index can reach them, but it's 11:51 and my requested extention
was not granted.  Still, this is fairly clean code considering how well
it performs.

blocks are set up thusly:
    void* prevFree; --  ptr to the prev free block's 'next' pointer (8 bytes)
    unsigned long size; -- size of the whole block (in qwords)
    [data] -- passed to application (total size is: size-32 bytes)
    unsigned long size; -- size of the whole block (in qwords)
    void* nextFree; -- ptr to the next free block's 'prev' pointer (8 bytes)
    
if the sign bit of a size is set, the block is free.

reams are set up thusly:
    unsigned int stdSize; size type of the ream
    unsigned int numPages; number of pages in the ream.
    void* firstFree; -- points to lastFree if the whole ream is taken
    [data]
    void* lastFree; -- points to the previous nodes' next pointer
    void* nextReam; -- next page of this size type

*/
/*---------------------------------        DEFINES */

#define SIGN_BIT (1L<<63)
#define BLOCK_HEADER_FOOTER 32
#define BLOCK_HEADER_FOOTER_Q 4
//#define PAGE_MASK 0x1FFF
//#define PAGE_SIZE 8192
//#define PAGE_SIZEQ 1024
//#define PAGE_MASK 0xFFF
#define PAGE_MASK 0
#define PAGE_SIZE 5000
#define PAGE_SIZEQ 625
#define HASH_SIZE 12
#define FREE_PAGE_LIST 11
//#define printf
//#define //!

// these were originally functions and have been made defines for speed.
#define GetBlockPrevPtr( x ) (x)
#define GetBlockNextPtr( x ) (x + GetBlockSizeQ( x ) - 1 )
#define GetBlockSizeQ(x) ((ulong)(*(((ulong*)x)+1)) & ~SIGN_BIT )



/*---------------------------------        STRUCTURES */


/*---------------------------------        GLOBALS */

extern ulong* g_pageHashTable[ HASH_SIZE ];

/*---------------------------------        FUNCTION PROTOTYPES */

// Block functions
//ulong* GetBlockPrevPtr( ulong* block );
//ulong* GetBlockNextPtr( ulong* block );
//ulong GetBlockSizeQ( ulong* inBlock );
bool PrevBlockFree( ulong* currBlock );
void AllocateBlock( ulong* block, ulong* ream, int sizeInBytes );
void PrintBlockInfo( ulong* block, char* info );

// Ream functions
ulong* ReamHead( ulong* ream );
ulong* ReamTail( ulong* ream );
ulong* GetReamNextPtr( ulong* ream );
void AddBlockToReam( ulong* block );
void RemoveBlockFromReam( ulong* block, ulong* ream );
void AddReamToHash( ulong hashIndex, ulong* ream );
ulong GetReamSizeQ( ulong* ream );
void PrintReamInfo( ulong* ream, char* info );
bool ReamEmpty( ulong* ream );
void RemoveReamFromHash( ulong* ream );

// Other functions
ulong HashSize( ulong size );
void GenBounds( ulong hash, ulong *min, ulong *max );
void CreateAvailBlock( ulong* block, ulong sizeQ );
void InitNewReam( ulong* page, int nPages, int defSize );
ulong* FindMemoryInReam( ulong* page, int sizeInBytes );
void CoalesceBlock( ulong* block );
ulong* GetReam( int nPages, int sizeType );
ulong* FindMemoryInReamList( ulong* listPtr, int sizeInBytes, ulong** reamUsed );
void CreateUnAvailBlock( ulong* block, ulong* ream );
void SplitAvailBlock( ulong* block, ulong* ream, ulong splitAmtQ );


// Public functions
int mm_init (void);
void *mm_malloc (size_t size);
void mm_free (void *ptr);

team_t team = {
    /* Team name to be displayed on webpage */
    "2DX",
    /* First member full name */
    "Adrian Perez",
    /* First member email address */
    "amperez",
    /* Second member full name (leave blank if none) */
    "Hollis Blanchard",
    /* Second member email address (blank if none) */
    "hollis"
};

ulong* g_pageHashTable[ HASH_SIZE ];


// find our hash based on size, returns a ptr that points to a page.
ulong HashSize( ulong size )
{
    if( size <= 8L ) return 0L;
    if( size <= 16L ) return 1L;
    if( size <= 32L ) return 2L;
    if( size <= 64L ) return 3L;
    if( size <= 128L ) return 4L;
    if( size <= 256L ) return 5L;
    if( size <= 512L ) return 6L;
    if( size <= 1024L ) return 7L;
    if( size < PAGE_SIZE - 32 ) return 8L;
    return 9L;
}

void GenBounds( ulong hash, ulong *min, ulong *max )
{
    switch( hash )
    {
    case 0:
	*min = 0;
	*max = 4;
	break;
    case 1:
	*min = 0;
	*max = 4;
	break;
    case 2:
	*min = 3;
	*max = 5;
	break;
    case 3:
	*min = 2;
	*max = 5;
	break;
    case 4:
	*min = 2;
	*max = 6;
	break;
    case 5:
	*min = 4;
	*max = 7;
	break;
    case 6:
	*min = 5;
	*max = 8;
	break;
    case 7:
	*min = 5;
	*max = 9;
	break;
    case 8:
	*min = 5;
	*max = 9;
	break;
    case 9:
	*min = 7;
	*max = 9;
	break;

    }
}


// insert a ream to the beginning of the hash table
void AddReamToHash( ulong hashIndex, ulong* ream )
{
    ulong* nextPtr;

    *((uint*)ream) = (uint)hashIndex;

    nextPtr = GetReamNextPtr( ream );
    *nextPtr = (ulong)g_pageHashTable[ hashIndex ];
    g_pageHashTable[ hashIndex ] = ream;
}


// takes a free block, and grabs enough of it to make
// an allocated block and (if applicable)
// lops off the rest as free memory
void AllocateBlock( ulong* block, ulong* ream, int sizeInBytes )
{
    ulong blockSize;
    ulong sizeNeeded;
    ulong sizeInBytesQ;
    ulong leftover;

    if( sizeInBytes & 7 ) sizeInBytesQ = (sizeInBytes >> 3) + 1;
    else sizeInBytesQ = sizeInBytes>>3;

    // find out how much memory we need (in qwords)
    sizeNeeded = sizeInBytesQ + BLOCK_HEADER_FOOTER_Q;

    // and how much memory we actually have
    blockSize = GetBlockSizeQ( block );

    leftover = blockSize - sizeNeeded;

    // if the leftover is not enough to make another node
    if( leftover < BLOCK_HEADER_FOOTER_Q + 1 )
    {
        // waste the extra memory
        CreateUnAvailBlock( block, ream );
    }
    // otherwise, lop off part of it for our use, and keep the rest in the
    // free list
    else
    {
        SplitAvailBlock( block, ream, sizeNeeded );
    }

}


// add a block to a ream's free list
// we're assuming it was previously malloc'd,
// or has had its fields set to let us assume it.
void AddBlockToReam( ulong* block )
{
    ulong* prevPtr;
    ulong* nextPtr;
    ulong* ream;
    ulong* reamHead;
    ulong* reamHeadNext;

    // init everything
    ream = (ulong*)*GetBlockPrevPtr(block);
    prevPtr = GetBlockPrevPtr( block );
    nextPtr = GetBlockNextPtr( block );
    reamHead = ReamHead( ream );
    reamHeadNext = (ulong*)*reamHead;

    /* add ourselves to the page's free list.*/
    // this->next = head->next
    *nextPtr = (ulong)reamHeadNext;
    // head->next = this;
    *reamHead = (ulong)block;
    // this->next->prev = &(this->next)
    *((ulong*)*nextPtr) = (ulong)nextPtr;
    // this->prev = head;
    *prevPtr = (ulong)reamHead;

}


// remove a block from a ream's free list, taking
// care to set the prev field to the ream ptr.
void RemoveBlockFromReam( ulong* block, ulong* ream )
{
    ulong* prevPtr;
    ulong* nextPtr;
    ulong blockSize;

    //!printf("!!removefromream\n");

    // init everything
    blockSize =  GetBlockSizeQ( block );
    prevPtr =  GetBlockPrevPtr( block );
    nextPtr = GetBlockNextPtr( block );

    //-- remove the block from the list
    // this->pnext->pprev = this->pprev
    *((ulong*)*nextPtr ) = *prevPtr;

    // this->pprev = this->pnext
    *((ulong*)*prevPtr) = *nextPtr;

    //-- set the prev ptr to the ream
    *prevPtr = (ulong)ream; 
}

// trivial
//ulong* GetBlockPrevPtr( ulong* block )
//{
//    return block;
//}
//#define GetBlockPrevPtr( x ) (x)


// trivial
//ulong* GetBlockNextPtr( ulong* block )
//{
//    return block + GetBlockSizeQ( block ) - 1;
//}
//#define GetBlockNextPtr( x ) (x + GetBlockSizeQ( x ) - 1 )

// find the size of a block (in qwords)
//ulong GetBlockSizeQ( ulong* inBlock )
//{
//    //printf("### size was
//    return (ulong)(*(inBlock+1)) & ~SIGN_BIT;
//}
//#define GetBlockSizeQ( x ) ((ulong)(*(((ulong*)x)+1)) & ~SIGN_BIT )


// true if the previous contiguous block is free
bool PrevBlockFree( ulong* currBlock )
{
    ulong * prevSize = ((ulong*)currBlock)-2;
    if( *prevSize & SIGN_BIT )
        return true;
    return false;
}


// true if the next contiguous block is free
bool NextBlockFree( ulong* currBlock )
{
    ulong * nextSize = currBlock + GetBlockSizeQ( currBlock ) + 1;
    if( *nextSize & SIGN_BIT )
        return true;
    return false;
}


// takes a region of memory and declares it as 'free', adding it
// to it's appropriate ream's free list.
// this code assumes the block was previously malloced, so other
// code will have to make the block look like it was (specifically,
// set up the prev ptr to point to the beginning of a ream)
void CreateAvailBlock( ulong* block, ulong sizeQ )
{

    /* ### init the block structure ### */
    // 1st size = sizeInBytes
    *(block+1) = sizeQ | SIGN_BIT;
    // 2nd size = sizeInBytes
    *(block+sizeQ-2) = sizeQ | SIGN_BIT;

    /* add ourselves to the page's free list.*/
    /* nextFree and firstFree set here */
    AddBlockToReam( block );
}


// turn a free block into an allocated block
void CreateUnAvailBlock( ulong* block, ulong* ream )
{
    //!printf("!!!createunavailblock\n");
    // remove the block from the ream
    RemoveBlockFromReam( block, ream );

    // point prevFree to the beginning of the ream
    *(block) = (ulong)ream;

    // unset the 2 sign bits, and we're done.
    *(block+1) &= ~SIGN_BIT;
    *(block + GetBlockSizeQ( block ) - 2) &= ~SIGN_BIT;
}


// takes a free block, lops some memory off the front, and returns it
// as an alocated block
void SplitAvailBlock( ulong* block, ulong* ream, ulong splitAmtQ )
{

    ulong* availBlock = block + splitAmtQ;
    ulong* unAvailBlock = block;
    ulong origSize = GetBlockSizeQ(block);
    ulong* prevPtr = GetBlockPrevPtr( block );

    // the prevoius pointer lands on that block's 'next' node
    // make the *new* prev point to where this->prev pointed
    *availBlock = *prevPtr;

    // this->prev->next = this+splitAmt
    *((ulong*)*prevPtr) = (ulong)availBlock;

    // adjust the 2 sizes
    *(availBlock+1) = (origSize-splitAmtQ) | SIGN_BIT;
    *(availBlock + GetBlockSizeQ(availBlock)-2) = (origSize-splitAmtQ) | SIGN_BIT;

    // take the lopped off part, and create an allocated block.
    *unAvailBlock = (ulong)ream;
    *(unAvailBlock+1) = splitAmtQ & ~SIGN_BIT;
    *(unAvailBlock+GetBlockSizeQ(block)-2) = splitAmtQ & ~SIGN_BIT;

}


// called before we turn this into a 'free' block, so the ream ptr
// (stored in the prevFree ptr of malloc'ed blocks) is still valid
void CoalesceBlock( ulong* block )
{
    ulong prevFree, nextFree;
    ulong cCase;
    ulong* prevBlock;
    ulong* nextBlock;
    ulong newSize, thisSize, prevSize, nextSize;
    ulong* tempPtr;
    ulong* newNext;

    // check the previous and next blocks
    prevFree =  (ulong)PrevBlockFree( block );
    nextFree =  (ulong)NextBlockFree( block );
    thisSize = GetBlockSizeQ(block);

    // generate a case
    cCase = prevFree + (nextFree << 1);
    //printf("cCase = %lu\n", cCase );
    //DEBUG
    //cCase = 0;
    switch( cCase )
    {
    case 0: // neither neighbor was free
        CreateAvailBlock( block, GetBlockSizeQ(block) );
	break;

    case 1: // prev was free, next was not

        // prev block = block - length of prev block
     	prevBlock = block - (*(block-2)& ~SIGN_BIT);

	// debugging stuff
	newNext = GetBlockNextPtr( block );

        // update pointers
	tempPtr = (ulong*)*GetBlockNextPtr( prevBlock );
	*tempPtr = (ulong)newNext;

        // block's nextFree = prevBlock's nextFree
	*GetBlockNextPtr(block) = *(block-1);

        // update sizes
	prevSize = GetBlockSizeQ(prevBlock);
	newSize = thisSize + prevSize;

        // prev block's 1st size += block's 1st size
        *(prevBlock+1) = newSize | SIGN_BIT;

        // block's 2nd size = prev block's 1st size
        *(block+ thisSize -2) = newSize | SIGN_BIT;

	//PrintBlockInfo( prevBlock, "coal" );
	break;

    case 2: // next was free, prev was not

        // next block = block + block's length
        nextBlock = block + thisSize;

        // update sizes
	nextSize = GetBlockSizeQ(nextBlock);
	newSize = thisSize + nextSize;

        // next block's 2nd size += block's 1st size
        *(nextBlock + nextSize - 2) = newSize | SIGN_BIT;

        // block's 1st size += next block's 1st size
        *(block+1) = newSize | SIGN_BIT;

        // update pointers - adopt next block's ptrs
	// nextBlock->prev->next = thisBlock
	*((ulong*)*nextBlock) = (ulong)block;

        // block's prevFree = next block's prevFree
        *(block) = *(nextBlock);

	//PrintBlockInfo( block, "coal" );

	break;

    case 3: // both were free

	nextBlock = block + thisSize;
	prevBlock = block - (*(block-2)& ~SIGN_BIT);

	nextSize = GetBlockSizeQ(nextBlock);
	prevSize = GetBlockSizeQ(prevBlock);

	newSize = thisSize + nextSize + prevSize;


	RemoveBlockFromReam( prevBlock, (ulong*)*block );
	RemoveBlockFromReam( nextBlock, (ulong*)*block );

	// update sizes

	// next block's 2nd size += block 1st size + prev block's 1st size
	*(nextBlock + nextSize - 2) = newSize | SIGN_BIT;

	// prev block's 1st size += block's 1st size + next block's 1st size
	*(prevBlock+1) = newSize | SIGN_BIT;
	 
	*prevBlock = (ulong)*block;
	AddBlockToReam( prevBlock );
	 
	break;
    }

//     if (ReamEmpty(ream)){
// 	//printf("doing a Remove/add pair\n");
//     	RemoveReamFromHash(ream);
// 	// and add it now, to the last one
// 	AddReamToHash( FREE_PAGE_LIST, ream );
//     }
}
// returns head node of a ream
ulong* ReamHead( ulong* ream )
{
    return ream+1;
}


// return the length (in qwords) of a ream
ulong GetReamSizeQ( ulong* ream )
{
    uint* iReam = (uint*)ream;
    uint nPages = *(iReam+1);
    return (ulong)(nPages * PAGE_SIZEQ);
}


// returns tail node of a ream
ulong* ReamTail( ulong* ream )
{
    return ream + GetReamSizeQ( ream ) - 2;
}


// fairly trivial, returns the address of this reams 'next ream' pointer
ulong* GetReamNextPtr( ulong* ream )
{
    return ream + GetReamSizeQ( ream ) - 1;
}


// initializes a newly created ream (defSize is intended size type)
void InitNewReam( ulong* ream, int nPages, int defSize )
{
    uint* iReam = (uint*)ream;

    // set the first 16/last 16 bytes:

    // set nextPage == NULL;
    *(ream + PAGE_SIZEQ*nPages - 1) = NULL;

    // stdSize = intended size
    *(iReam ) = (uint)defSize;
    // numPages = nPages
    *(iReam + 1) = (uint)nPages;

    // ream->head = &ream->tail
    *ReamHead( ream ) = (ulong)ReamTail( ream );
    // ream->tail = &ream->head
    *ReamTail( ream ) = (ulong)ReamHead( ream );

    // set the soon-to-be avail blocks' 
    // prevFree to ream, as if already malloc'ed
    *(ream+2) = (ulong)ream;

    // take the rest of the memory, create a avail block
    CreateAvailBlock( ream+2, PAGE_SIZEQ*nPages-4 );

    assert( ReamEmpty( ream ) );
}


// given a page, this searches the free list for a block of memory
// NULL if nothing is found.
ulong* FindMemoryInReam( ulong* ream, int sizeInBytes )
{
    ulong* currBlock;
    ulong* reamHead = ReamHead( ream );
    ulong* reamTail = ReamTail( ream );
    ulong currSize;

    // if there are no free nodes, early out.
    if( (ulong*)*reamHead == reamTail ) return NULL;

    // otherwise, iterate through the list, searching for a block.
    currBlock = (ulong*)*reamHead;
    while( currBlock != reamTail )
    {
        currSize = GetBlockSizeQ( currBlock );

        // if the block is big enough...
        if( (long)((currSize << 3)-BLOCK_HEADER_FOOTER) >= (long)sizeInBytes ) {
            return currBlock;
        }

        // continue to the next node.
        currBlock = (ulong*)*GetBlockNextPtr( currBlock );
    }

    // if we get here, we didn't find a block of sufficient size
    return NULL;
}


// abstract away getting a region of memory from the driver
ulong* GetReam( int nPages, int sizeType )
{
    ulong* out = (ulong*)(((uchar*)dseg_hi)+1);

    mem_sbrk( PAGE_SIZE * nPages );

    InitNewReam(out, nPages, sizeType);
    return out;
}

// scans a ream list for memory
ulong* FindMemoryInReamList( ulong* listPtr, int sizeInBytes, ulong** reamUsed )
{
    ulong* currReam;
    ulong* foundMem;

    // early out if listPtr points to NULL
    if( listPtr == NULL ) return NULL;

    // otherwise iterate through the list of pages, checking each one.
    currReam = (ulong*)(listPtr);
    while( currReam != NULL )
    {
        foundMem = FindMemoryInReam( currReam, sizeInBytes );
        if( foundMem )
        {
            *reamUsed = currReam;
            return foundMem;
        }
        
        currReam =  (ulong*)*GetReamNextPtr( currReam );
    }

    return NULL;
}


// if true, the ream has exactly one (large) block in it and can be reclaimed
bool ReamEmpty( ulong* ream )
{
    ulong blockSize;
    ulong* firstBlock = (ulong*)*ReamHead( ream );
    uint* iReam = (uint*)ream;
    if( ReamTail(ream) == (ulong*)*GetBlockNextPtr( firstBlock ) )
    {
	blockSize = GetBlockSizeQ( firstBlock ) + 4L;
	if( blockSize == PAGE_SIZEQ * *(iReam+1) )
	    return true;
    }
    return false;
}


// iterate through the list of reams in a hash, find our prev, and remove ourselves
// (we also place ourselves in the free list)
void RemoveReamFromHash( ulong* ream )
{
    uint hashIndex = *((uint*)ream);
    ulong *currReam;

    //printf("removing ream %x from index: %i\n", ream, hashIndex );

    // initial test: see if we're the first one in the list.
    if( g_pageHashTable[ hashIndex ] == ream )
    {
	// then make the hash point to the next entry.
	g_pageHashTable[ hashIndex ] = (ulong*)*(GetReamNextPtr( ream ));
    }
    else
    {
	// otherwise, iterate and remove.
	currReam = g_pageHashTable[ hashIndex ];
	while( currReam && (ulong*)*(GetReamNextPtr( currReam )) != ream )
	    currReam = (ulong*)*(GetReamNextPtr( currReam ));

	// now currReam points to the ream we want to remove
	if( !currReam ) printf("ERROR!: removable page not found in list!\n");
//	assert( *(GetReamNextPtr( currReam )) == ream );

	// remove it.
	*(GetReamNextPtr(currReam)) = *(GetReamNextPtr(ream));

    }
    *((uint*)ream) = 69;
}

/*---------------------------------        FUNCTIONS */

int mm_init (void)
{
    int i;
     
    // init our hash table
    for( i=0; i<HASH_SIZE; i++ ) 
        g_pageHashTable[i] = NULL;

    return -1;
}

void *mm_malloc (size_t size)
{
    ulong* hashPtr, *hashPtr2;
    ulong* mlcMem = NULL;
    ulong* newReam;
    ulong hashIndex = HashSize( size ), hashInd2 = hashIndex;
    ulong* ream;
    ulong numPages;
    ulong min, max;

    //printf("## entered mm_malloc for %i bytes\n", (int)size);

    // get our hash index pointer
    hashPtr = g_pageHashTable[ hashIndex ];

    // see if we have memory
    mlcMem = NULL; hashInd2 = hashIndex+1;

    mlcMem =  FindMemoryInReamList( hashPtr, size, &ream );

    GenBounds( hashIndex, &min, &max );
    hashInd2 = min;
    while( !mlcMem && (hashInd2 <= max) )
    {// check the lists in our local neighborhood
	if( hashInd2 != hashIndex );
	hashPtr2 = g_pageHashTable[ hashInd2++ ];
	mlcMem =  FindMemoryInReamList( hashPtr2, size, &ream );
    }

    // if we didn't, add a page to the head of the list
    if( !mlcMem )
    {

// this code didn't work at the 11th hour so we had to drop it.
// 	// if there are any free pages, use them.
// 	hashPtr2 = g_pageHashTable[ FREE_PAGE_LIST ];
// 	mlcMem =  FindMemoryInReamList( hashPtr2, size, &ream );

// 	if( mlcMem )
// 	{
// 	    //printf("doing an add/remove in malloc\n");
// 	    // move the ream to the appropriate block.
// 	    RemoveReamFromHash( ream );
// 	    AddReamToHash( hashIndex, ream );
// 	}

 
	// if we're going to make a <1 page ream
	if( !mlcMem && hashIndex < 8L )
	{
	    newReam =  GetReam( 1, size );
	    ream = newReam;
	    mlcMem = FindMemoryInReam( newReam, size );
	}
	else if( hashIndex < 9L )
	{
	    newReam =  GetReam( 2, size );
	    ream = newReam;
	    mlcMem = FindMemoryInReam( newReam, size );
	}
	// otherwise, we're making a big one.
	else // (hashIndex == 9)
	{
	    numPages = (1 + size/(PAGE_SIZE-32));
	    newReam =  GetReam( numPages, size );
	    ream = newReam;
	    mlcMem = FindMemoryInReam( newReam, size );
	}
	
        // add the new ream to the hash
        AddReamToHash( hashIndex, newReam );
    }

    // so either way, we have our memory now.
    AllocateBlock( mlcMem, ream, size );

    //!printf("## mlcMem = '%#.8lx'\n", (ulong)mlcMem);
    //printf("   memory was %lx\n", (void*)(mlcMem+2));

    return (void*)(mlcMem+2);
}

void mm_free (void *ptr)
{
    ulong* block = ((ulong*)ptr)-2;
    //printf("### entered mm_free %lx\n",ptr);
    CoalesceBlock( block );
}
