/* $Id$ */

/*
 *  Joseph Hughes
 *  hughes@andrew.cmu.edu
 *
 *  CS213 - Lab assignment 3
 *
 *  This implementation does a first-fit over segregated free lists.
 *  While still slow, it has better performance than the previous
 *  buddy-system implementation.
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

team_t team = {
    /* Team name to be displayed on webpage */
    "///retrovirus///",
    /* First member full name */
    "Joseph Hughes",
    /* First member email address */
    "hughes@andrew.cmu.edu",
    /* Second member full name (leave blank if none) */
    "",
    /* Second member email address (blank if none) */
    ""
};

// useful constants

#define INFINITY (0L - 1L)
#define NUM_LISTS 50
#define INITIAL 2

#define B_FREE 1
#define B_USED 0

// unsigned integer types of specific sizes

typedef unsigned long uint64;
typedef unsigned int uint32;
typedef unsigned short uint16;
typedef unsigned char uint8;

// the format of the blocks' headers (and "toe" footers)

typedef struct _s_head
{
  struct _s_head* link;    // pointer to previous block in free list
  uint64 len:56;           // block length
  uint8 free:1;            // B_FREE if free, B_USED if used
} s_head;

// a free list identifier format

typedef struct
{
  uint64 max;              // maximum size of blocks in this list
  s_head* head;            // pointer to the head of this list
} s_list;

// structure of the main control block

typedef struct
{
  s_head* heap_start;      // pointer to first block in the heap
  s_list list[NUM_LISTS];  // the array of free lists
} s_control;

// pointer to the main control block

s_control* p_control;

// function prototypes

uint16 choose_list(uint64 len);
void b_add(s_head* p_head);
void b_remove(s_head* p_head);
void b_fragment(s_head* p_head, uint64 size);
s_head* b_congeal(s_head* p_head);

/*
 * mm_init
 *
 * This function completely (re)initializes the heap.
 *
 * Parameters:
 *     none
 * 
 * Returns:
 *     int - 0 if initialized OK, -1 if a problem was encountered
 */

int mm_init (void)
{
  uint32 initial;
  s_head* heap_start;
  s_head* heap_toe;
  uint8 count;

  // initial allocation

  initial = INITIAL*mem_pagesize();
  if (initial > mem_usage())
  {
    if(mem_sbrk(initial - mem_usage()) == NULL)
      return -1;
  }

  // configure control block

  p_control = (s_control*)(uint64)dseg_lo;
  p_control->heap_start = (s_head*)((uint64)dseg_lo + sizeof(s_control));
  for(count = 0; count < NUM_LISTS; count++)
    p_control->list[count].head = NULL;

  p_control->list[0].max = 40;
  p_control->list[1].max = 48;
  p_control->list[2].max = 56;
  for(count = 6; (count - 3) < (NUM_LISTS); count += 1)
    p_control->list[count - 3].max = (1L << count);
  p_control->list[NUM_LISTS - 1].max = INFINITY;

  // format initial block

  heap_start = p_control->heap_start;
  heap_start->len = (uint64)dseg_hi - (uint64)heap_start + 8L;
  heap_toe = (s_head*)((uint64)heap_start + heap_start->len - sizeof(s_head));
  heap_toe->len = heap_start->len;
  heap_start->free = B_FREE;
  heap_toe->free = B_FREE;
  heap_start->link = NULL;
  heap_toe->link = NULL;
  
  b_add(heap_start);

  return 0;
}

/*
 * mm_malloc
 *
 * This function allocates a block at least as large as the size specified.
 *
 * Parameters:
 *     size_t size - the size in bytes of the block to be allocated
 *
 * Returns:
 *     void* - if successful, a pointer to the newly-allocated block
 *             if unsuccessful, NULL
 */

void *mm_malloc (size_t size)
{
  s_head* p_head;
  s_head* p_toe;
  s_head* p_found = NULL;
  uint16 count;
  uint64 ask_size;
  uint64 act_size;

  // make sure allocation is 8-byte-aligned and includes space for headers

  if ((size % 8L) != 0)
    size = ((size / 8L) * 8L) + 8L;
  size += 2 * sizeof(s_head);

  // look for a sufficiently-large free block

  count = choose_list(size);
  while((count < NUM_LISTS) && (p_found == NULL))
  {
    p_head = p_control->list[count].head;
    while((p_head != NULL) && (p_found == NULL))
    { 
      p_toe = (s_head*)((uint64)p_head + p_head->len - sizeof(s_head));
      if(p_head->len >= size)
	p_found = p_head;
      else
        p_head = p_toe->link;
    }
    count++;
  }

  if (p_found == NULL)
  {
    // couldn't find a block large enough, so grow the heap

    // make sure new block is a multiple of the page size

    ask_size = size;
    if((ask_size % mem_pagesize()) != 0)
      ask_size = ((ask_size / mem_pagesize())*mem_pagesize()) + mem_pagesize();

    // allocate the new space (and return NULL on failure)

    p_found = (s_head*)((uint64)dseg_hi + 8L);
    if (mem_sbrk(ask_size) == NULL)
      return NULL;

    // format the new block

    p_toe = (s_head*)((uint64)dseg_hi - sizeof(s_head) + 8L);
    act_size = (uint64)dseg_hi - (uint64)p_found + 8L;
    p_found->len = act_size;
    p_toe->len = act_size;
    p_found->free = B_FREE;
    p_toe->free = B_FREE;
    p_found->link = NULL;
    p_toe->link = NULL;

    // coalesce with any free space at the end of the list

    p_found = b_congeal(p_found);
  }
  else   // block found--remove from list
    b_remove(p_found);

  // return the unused space to the free list

  b_fragment(p_found, size);

  // format block

  p_toe = (s_head*)((uint64)p_found + p_found->len - sizeof(s_head));
  p_found->free = B_USED;
  p_toe->free = B_USED;
  p_found->link = NULL;
  p_toe->link = NULL;

  // return pointer to the newly-allocated block

  return (void*)((uint64)p_found + sizeof(s_head));
}

/*
 * mm_free
 *
 * This function frees a block pointed to by the parameter passed in.  If the
 * parameter is not a pointer previously returned by mm_malloc, the results
 * are undefined.
 * 
 * Parameters:
 *     void* ptr - a pointer to the block to be freed
 * 
 * Returns:
 *     nothing
 */

void mm_free (void *ptr)
{
  s_head* p_head;
  s_head* p_toe;
  
  // determine true dimensions of block

  p_head = (s_head*)((uint64)ptr - sizeof(s_head));
  p_toe = (s_head*)((uint64)p_head + p_head->len - sizeof(s_head));

  // formatting as free space

  p_head->free = B_FREE;
  p_toe->free = B_FREE;

  // coalesce and add to free list

  p_head = b_congeal(p_head);
  b_add(p_head);
}

/*
 * choose_list
 *
 * This function determines which free list a block of the specified length
 * belongs in.  If it can't determine an appropriate list, it returns the
 * last one.  In this implementation, a free list with a higher index has
 * a higher maximum size for the blocks in it.
 *
 * Parameters:
 *     uint64 len - the length of the block for which a list is to be found
 *
 * Returns:
 *     uint16 - the index of the free list that best suits the size given
 */

uint16 choose_list(uint64 len)
{
  uint16 count;

  for(count = 0; count < NUM_LISTS; count++)
  {
    if (len <= p_control->list[count].max)
      return count;
  }

  return NUM_LISTS - 1;
}

/*
 * b_add
 *
 * This function adds a block of memory to the appropriate free list.
 *
 * Parameters:
 *     s_head* p_head - a pointer to the initial header of the block
 *                      to be added to the free lists.
 *
 * Returns:
 *     nothing
 */

void b_add(s_head* p_head)
{
  uint16 list_id;
  s_head* p_toe;

  // determine end of block and which list it belongs in

  list_id = choose_list(p_head->len);
  p_toe = (s_head*)((uint64)p_head + p_head->len - sizeof(s_head));

  // make sure it's classified as free

  p_head->free = B_FREE;
  p_toe->free = B_FREE;

  // add to the linked list

  p_toe->link = p_control->list[list_id].head;
  p_head->link = (s_head*)p_control;

  if (p_toe->link != NULL)
    p_toe->link->link = p_head;
  p_control->list[list_id].head = p_head;
}

/*
 * b_remove
 *
 * This function removes a block of memory from the free list that it's in,
 * in preparation for its use.
 *
 * Parameters:
 *     s_head* p_head - a pointer to the initial header of the block to
 *                      be removed from the free lists for use
 *
 * Returns:
 *     nothing
 */

void b_remove(s_head* p_head)
{
  uint16 list_id;
  s_head* p_toe;
  s_head* p_prev_toe;

  // determine block's end-pointer

  p_toe = (s_head*)((uint64)p_head + p_head->len - sizeof(s_head));
 
  // do removal from linked list

  if(p_toe->link != NULL)
    p_toe->link->link = p_head->link;

  if(p_head->link == (s_head*)p_control)
  {
    list_id = choose_list(p_head->len);
    p_control->list[list_id].head = p_toe->link;
  }
  else
  {
    p_prev_toe = (s_head*)((uint64)(p_head->link) + p_head->link->len -
			   sizeof(s_head));
    p_prev_toe->link = p_toe->link;
  }

  p_head->link = NULL;
  p_toe->link = NULL;
}

/*
 * b_fragment
 *
 * This function removes any unnecessary space from a block and puts the
 * surplus in the free lists.
 *
 * Parameters:
 *     s_head* p_head - a pointer to the initial header of the block to
 *                      fragment (the trimmed block will still start at
 *                      this address
 *     uint64 size - amount of this block (in bytes) that's actually needed
 *
 * Returns:
 *     nothing
 */

void b_fragment(s_head* p_head, uint64 size)
{
  s_head* p_toe;
  s_head* p_toe_adj;
  s_head* p_new_head;
  s_head* p_new_toe;
  uint64 old_size;
  uint64 new_size;

  // make sure desired size is 8-byte-aligned

  if ((size % 8L) != 0)
    size = (size / 8L) + 8L;

  // make sure remainder fragment would be large enough to use

  old_size = p_head->len;
  new_size = old_size - size;
  if (new_size < ((2 * sizeof(s_head)) + 8))
    return;

  // format a block of the desired size

  p_toe = (s_head*)((uint64)p_head + old_size - sizeof(s_head));
  p_toe_adj = (s_head*)((uint64)p_head + size - sizeof(s_head));
  p_head->len = size;
  p_toe_adj->len = size;
  p_head->free = B_FREE;
  p_toe_adj->free = B_FREE;
  p_head->link = NULL;
  p_toe_adj->link = NULL;

  // format remainder of space and add it back to the free list

  p_new_head = (s_head*)((uint64)p_toe_adj + sizeof(s_head));
  p_new_toe = (s_head*)((uint64)p_new_head + new_size - sizeof(s_head));
  p_new_head->len = new_size;
  p_new_toe->len = new_size;
  p_new_head->free = B_FREE;
  p_new_toe->free = B_FREE;
  p_new_head->link = NULL;
  p_new_toe->link = NULL;

  b_add(p_new_head);
}

/*
 * b_congeal
 *
 * If the given block has any free neighbors, this function will coalesce
 * them into one block.
 * 
 * Parameters:
 *     s_head* p_head - a pointer to the block to be coalesced, if possible
 * 
 * Returns:
 *     s_head* - a pointer to the head of the new, (potentially) larger block
 */

s_head* b_congeal(s_head* p_head)
{
  s_head* p_toe;
  s_head* p_new_head;
  s_head* p_new_toe;
  uint64 p_new_len;
  s_head* p_prev_head;
  s_head* p_prev_toe;
  s_head* p_next_head;
  s_head* p_next_toe;

  // determine end-pointer of block

  p_toe = (s_head*)((uint64)p_head + p_head->len - sizeof(s_head));

  // get pointers to next and previous blocks, bounds-check

  if((uint64)p_head <= (uint64)(p_control->heap_start))
    p_prev_toe = NULL;
  else
    p_prev_toe = (s_head*)((uint64)p_head - sizeof(s_head));

  p_next_head = (s_head*)((uint64)p_toe + sizeof(s_head));
  if((uint64)p_next_head > (uint64)dseg_hi)
    p_next_head = NULL;

  // if neighboring blocks are free, remove them from list and
  // expand the current block into their space

  p_new_head = p_head;
  p_new_toe = p_toe;

  if((p_prev_toe != NULL) && (p_prev_toe->free == B_FREE))
  {
    p_prev_head =
      (s_head*)((uint64)p_prev_toe - p_prev_toe->len + sizeof(s_head));
    b_remove(p_prev_head);
    p_new_head = p_prev_head;
  }

  if((p_next_head != NULL) && (p_next_head->free == B_FREE))
  {
    p_next_toe =
      (s_head*)((uint64)p_next_head + p_next_head->len - sizeof(s_head));
    b_remove(p_next_head);
    p_new_toe = p_next_toe;
  }

  // format new block, return

  p_new_len = (uint64)p_new_toe - (uint64)p_new_head + sizeof(s_head);
  p_new_head->link = NULL;
  p_new_head->len = p_new_len;
  p_new_toe->link = NULL;
  p_new_toe->len = p_new_len;

  return p_new_head;
}





