/* $Id$ */

/*
 *  Papadimitriou Spiros
 *  spapadim+@cs.cmu.edu
 *
 *  CS213 - Lab assignment 3
 *
 */

/* Philip Brighten Godfrey <pgodfrey@andrew.cmu.edu>
 *
 * DESCRIPTION OF ALGORITHM: I'm using a best-fit
 * algorithm with a doubly linked list of the free
 * nodes sorted by size, smallest to largest.  I also
 * keep a "hint" pointer around, which gives me a pointer
 * to the free block in the free-list immediately after
 * the block I just allocated.  (When I do a free, the
 * hint gets set to the freed block.)  Thus I can check
 * the size of the hint to see if it's a good place to
 * start looking for a best-sized free block.
 *
 * I'm also doing a very limited version of segregated
 * storage.  If I need to call mem_sbrk() to allocate memory
 * for what I consider to be a "small" block, then I allocate
 * enough room for two of the blocks.  Then mm_malloc() will
 * reserve (for a while) that extra block for sizes that
 * fit it exactly.  This hurt me slightly on speed, but
 * increased my average memory utilization from 92% to 95%.
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

team_t team = {
    /* Team name to be displayed on webpage */
    "BigW",
    /* First member full name */
    "Philip Brighten Godfrey",
    /* First member email address */
    "pgodfrey@andrew.cmu.edu",
    /* Second member full name (leave blank if none) */
    "",
    /* Second member email address (blank if none) */
    ""
};

/*
 * CONSTANT DEFINITIONS
 */

#define WORD_LENGTH	(8L)	/* Word length in bytes */
#define INIT_EL		(16L)	/* Start with how many words free? */
#define MIN_LENGTH	(16L)	/* Minimum block size in bytes */
#define HEADER_LENGTH   (32L)   /* Bytes that the main header takes up */
#define SMALL_SIZE_MAX	(256L)	/* Max block length in bytes considered small */

/*
 * FUNCTION DECLARATIONS
 */

inline long int *bestFit(size_t size);
inline long int *findPrevHint(size_t size);
inline long int *findPrev(size_t size);
inline void unlinkFreeBlock(long *block);
inline void linkFreeBlock(long *block);

/*
 * A WHOLE SLEW OF REALLY USEFUL #DEFINES THAT OPERATE
 * ON THE VARIOUS DATA STRUCTURES AND HEADERS AND SUCH
 */

/* Get the hint size
 */
#define hintLength	(*((long*) dseg_lo))

/* Get a pointer to the free block after the one we last tried
 */
#define hint		(*(long **)(dseg_lo + WORD_LENGTH))

/* Get a pointer to the physically last block
 */
#define lastPhysBlock	(*(long **)(dseg_lo + 2 * WORD_LENGTH))

/* Get a pointer to a pointer to the beginning of the free list.
 */
#define freeList	(*(long**)(dseg_lo + 3 * WORD_LENGTH))

/* Get the length in bytes of a block given a pointer to the block
 */
#define getLength(p)	((*(long *)(p)) & (~7L))

/* Is this block allocated?
 */
#define isAllocated(p)	((*(long *)(p)) & 1L)

/* Is this block free?
 */
#define isFree(p)	(isAllocated(p) ^ 1L)
//#define isFree(p)	(!isAllocated(p))

/* Is the physically previous block allocated?
 */
#define isPhysPrevAllocated(p)	(((*(long *)(p)) & 2L) >> 1L)

/* Is the physically previous block free?
 */
#define isPhysPrevFree(p)	(isPhysPrevAllocated(p) ^ 1L)
//#define isPhysPrevFree(p)	(!isPhysPrevAllocated(p))

/* Flip the physPrevIsAllocated bit
 */
#define flipPhysPrevAllocated(p)	(*(long *)(p) = (long*)((*(long*)(p)) ^ 2L))

/* Get a pointer to the next free block given a
 * pointer to a free block
 */
#define getNextFree(p)	(*(long **)(((long) (p)) + WORD_LENGTH))

/* Get a pointer to the previous free block in the same size class as a given
 * pointer to a free block
 */
#define getPrevFree(p)	((long *)( (*(long *)(((long) (p)) + getLength(p))) & (~7L)))

/* Get a pointer to the physically next block
 */
#define getPhysNext(p)	((long *)(((long)(p)) + getLength(p) + WORD_LENGTH))

/* Set the length, is-allocated-bit, and prev-is-allocated-bit of a block
 * p = pointer to beginning of (header of) block
 * l = 64-bit length
 * a = 0 if free, 1 if allocated
 * r = 0 if physically previous block is free, 1 if prev allocated
 */
#define setLength(p,l,a,r)	((*(long *)(p)) = ((long)(l) | (a) | ((r) << 1L)))

/* Flip the is-allocated bit of a header
 */
#define flipAllocated(p)	(*(long *)(p) = (long*)((*(long*)(p)) ^ 1L))

/* Set the address of the next free block in the same size class
 * p = pointer to beginning of (header of) block
 * n = pointer to beginning of next free block in same size class
 */
#define setNextFree(p,n)	( (*((long *)(((long) (p)) + WORD_LENGTH))) = (n))

/* Set the address of the previous free block
 * p = pointer to beginning of (header of) block
 * r = pointer to previous free block
 */
#define setPrevFree(p,r)	( (*(long *)(((long) (p)) + getLength(p))) = (long)(r))


/* Initialize stuff that needs initializing
 */
int mm_init (void)
    {
    long* firstBlock;

    if (mem_sbrk(WORD_LENGTH * INIT_EL + HEADER_LENGTH) == NULL)
        return -1;

    /* Our very first block of free memory!  What a special moment... */
    firstBlock = (long *)(dseg_lo + HEADER_LENGTH);

    /* Initialize the list */
    freeList = firstBlock;

    /* Initialize the block of memory */
    setLength(firstBlock, ((INIT_EL - 1L) * WORD_LENGTH), FALSE, TRUE);
    setNextFree(firstBlock, NULL);
    setPrevFree(firstBlock, NULL);

    /* Set our hint and last-physical pointers */
    hint = firstBlock;
    hintLength = ((INIT_EL - 1L) * WORD_LENGTH);
    lastPhysBlock = firstBlock;

    return 0;
    }

/* Find the best-fit free memory block for a size, or return
 * null if there is no block large enough
 */
inline long int *bestFit(size_t size)
    {
    long *block;

    if (size >= hintLength)
        if (hint == NULL)
            return NULL;
        else
            block = hint;
    else
        block = freeList;

    while ((block != NULL) && (getLength(block) < size))
        block = getNextFree(block);

    return block;
    }

/* Find the pointer previous to where we should insert a freed
 * block of a given size, using the Hint value as a starter if we can
 */
inline long int *findPrevHint(size_t size)
    {
    long *block, *prev;

    if (hint != NULL && size >= hintLength)
    	{
        block = hint;
        prev = hint;
        }
    else
        {
        block = freeList;
        prev = NULL;
        }

    for (;
         (block != NULL) && (getLength(block) < size);
         prev = block, block = getNextFree(block))
	;

    return prev;
    }

/* Find the pointer previous to where we should insert a freed
 * block of a given size, NOT using the Hint value
 */
inline long int *findPrev(size_t size)
    {
    long *block, *prev;

    for (block = freeList, prev = NULL;
         (block != NULL) && (getLength(block) < size);
         prev = block, block = getNextFree(block))
	;
    return prev;
    }

/* Unlink a free block from the list, presumably because it's
 * being allocated
 */
inline void unlinkFreeBlock(long *block)
    {
    long *next = getNextFree(block),
    	 *prev = getPrevFree(block);

    if (freeList == NULL)
        return;

    /* Flip the is-allocated bit */
    if (isFree(block))
        flipAllocated(block);
	
    /* If there's a next block in the list, copy in our prev-free block */
    if (next != NULL)
	setPrevFree(next, prev);
    	
    /* Set the previous free node to point to the next one */
    if (freeList == block || prev == NULL)
	freeList = next;
    else
    	setNextFree(prev, next);
    }

/*  Hmmm... what could this function do?
 */
void *mm_malloc (size_t size)
    {
    long extraSpace, *extraBlock;
    long *block = NULL,	/* Pointer to the free block we'll use */
         *old_dseg_hi = (long *)dseg_hi; /* Our old highest address */

    if (size < 1)
	return NULL;

    /* Minimum space allocated is 16 bytes (2 words) */
    if (size < 16)
	size = 16;
    /* Round size up to the nearest 8 bytes */
    else if (size & 7L)
	size = (size & (~7L)) + 8L;

    /* Look for the best block which matches our size requirement */
    block = bestFit(size);

    /* Special case: if we found one, but it's our hint, and the size
     * we need is smaller */

    /* If we didn't find a block, allocate enough space */
    if (block == NULL)
        {
        /* If the physically last block is allocated,
         * we must create a whole new block */
        if (isAllocated(lastPhysBlock) || ((size > hintLength) && (hint == lastPhysBlock) && (hint != dseg_lo + HEADER_LENGTH)))
            {
            /* Pseudo-segregated-storage-hack: for small-sized blocks, allocate
             * room for two of them right away */
            if (size <= SMALL_SIZE_MAX)
                {
        	/* Set the hint: no more room for this size or greater! */
        	hint = NULL;
        	hintLength = size;

            	/* Allocate enough memory (if we can) to store the new element */
            	if (mem_sbrk(2 * (size + WORD_LENGTH)) == NULL)
                    return NULL;

	    	block = (long *) (((long)old_dseg_hi) + 1L);

	    	/* Initialize the new block of memory as allocated */
	    	setLength(block, size, TRUE, TRUE);
	
	    	/* Initialize and free the excess */
	    	extraBlock = ((long) block) + size + WORD_LENGTH;
	    	setLength(extraBlock, size, TRUE, TRUE);
	    	linkFreeBlock(extraBlock);	
	    	lastPhysBlock = extraBlock;
	    	hint = extraBlock;
	    	}
	    else
	    	{

        	/* Set the hint: no more room for this size or greater! */
        	hint = NULL;
        	hintLength = size;
            	/* Allocate enough memory (if we can) to store the new element */
            	if (mem_sbrk(size + WORD_LENGTH) == NULL)
                    return NULL;

	    	block = (long *) (((long)old_dseg_hi) + 1L);

	    	/* Initialize the new block of memory as allocated */
	    	setLength(block, size, TRUE, TRUE);
	
	    	/* Set the new last block value */
	    	lastPhysBlock = block;
		}
	    }
	
        /* The physically last block is free: just add onto it */
        else
            {
            /* Set the hint: no more room for this size or greater! */
            hint = NULL;
            hintLength = size;

            block = lastPhysBlock;

            if (mem_sbrk(size - getLength(block)) == NULL)
                return NULL;

            unlinkFreeBlock(block);
            setLength(block, size, TRUE, TRUE);
            }

        }

    /* We we have a pointer to a block we can use */
    else /* DEALLOCATE THE FREE BLOCK */
    	{
    	/* Set the hint to the next free one in the list */
    	hint = getNextFree(block);
    	hintLength = getLength(block);
    	unlinkFreeBlock(block);

	/* If the block we allocated was longer than necessary,
     	 * free the extra space */
    	extraSpace = getLength(block) - size;
    	if (extraSpace >= WORD_LENGTH + MIN_LENGTH)
	    {
	    /* Set our new size to just what we want it to be */
	    *block -= extraSpace;
	
	    /* Initialize and free the excess */
	    extraBlock = ((long) block) + size + WORD_LENGTH;
	    setLength(extraBlock, extraSpace - WORD_LENGTH, TRUE, TRUE);
	    linkFreeBlock(extraBlock);
	
	    if (block == lastPhysBlock)
	        lastPhysBlock = extraBlock;
	    }

    	/* Otherwise, tell the next block that we're allocated */
    	if (block != lastPhysBlock)
	    {
	    /* Flip the physically next block's physPrevIsFree bit */
	    if (isPhysPrevFree(getPhysNext(block)))
	    	flipPhysPrevAllocated(getPhysNext(block));
	    }
	}

    return (char *)((long) block + WORD_LENGTH);
    }

/* Link a block of memory into the list of free memory
 */
inline void linkFreeBlock(long *block)
    {
    long length = getLength(block),
    	 *prevFree, *nextFree;

    /* Flip the is-allocated bit */
    if (isAllocated(block))
        flipAllocated(block);

    /* Get a pointer to the node in the list that should
     * come immediately before this one */
    prevFree = findPrevHint(length);

    /* If we're linking p in at the beginning of the free list... */
    if (prevFree == NULL)
	{
	if (freeList != NULL)
	    setPrevFree(freeList, block);
	setNextFree(block, freeList);
	setPrevFree(block, NULL);
	freeList = block;	
	}

    /* If we're linking p in somewhere in the middle or end of the list... */
    else
	{
	nextFree = getNextFree(prevFree);
	if (nextFree != NULL)
	    setPrevFree(nextFree, block);
	setNextFree(block, nextFree);
	setPrevFree(block, prevFree);
	setNextFree(prevFree, block);
	}
	
    if (block != lastPhysBlock)
	{
	/* Flip the physically next block's physPrevIsFree bit */
	if (isPhysPrevAllocated(getPhysNext(block)))
	    flipPhysPrevAllocated(getPhysNext(block));
	}
    }

/* Gimme that memory back!
 */
void mm_free (void *ptr)
    {
    long *p = (long *) (((long) ptr) - WORD_LENGTH),
    	 *prevFree, *nextFree, *physPrev, *physNext, *tmp0, *tmp1;
    long length;

    length = getLength(p);

    /* Error checking */
    if (p < (long *)(dseg_lo + HEADER_LENGTH) || p > (long *) dseg_hi	/* out of bounds */
        || isFree(p)				/* already freed */
        || (length < MIN_LENGTH))			/* length less than minimum */
    	return;

    /* Coalesce backward */
    if (isPhysPrevFree(p))
        {
        tmp0 = (((long)p) - WORD_LENGTH) & (~7L);
        tmp1 = *tmp0;

        if (tmp1 != NULL)
            physPrev = getNextFree(tmp1); /* Then go to next free block */
        else
            physPrev = freeList;

        unlinkFreeBlock(physPrev);
        *(long*)physPrev += (long) (length + WORD_LENGTH);
        if (p == lastPhysBlock)
            lastPhysBlock = physPrev;
        p = physPrev;
        }

    /* Coalesce forward */
    physNext = getPhysNext(p);
    if (p != lastPhysBlock && isFree(physNext))
        {
        unlinkFreeBlock(physNext);
        *p += getLength(physNext) + WORD_LENGTH;
        if (physNext == lastPhysBlock)
            lastPhysBlock = p;
        }

    length = getLength(p);

    /* Flip the is-allocated bit */
    if (isAllocated(p))
        flipAllocated(p);

    /* Get a pointer to the node in the list that should
     * come immediately before this one */
    prevFree = findPrev(length);

    /* If we're linking p in at the beginning of the free list... */
    if (prevFree == NULL)
	{
	if (freeList != NULL)
	    setPrevFree(freeList, p);
	setNextFree(p, freeList);
	setPrevFree(p, NULL);
	freeList = p;	
	}

    /* If we're linking p in somewhere in the middle of the list... */
    else
	{
	nextFree = getNextFree(prevFree);
	if (nextFree != NULL)
	    setPrevFree(nextFree, p);
	setNextFree(p, nextFree);
	setPrevFree(p, prevFree);
	setNextFree(prevFree, p);
	}

    if (p != lastPhysBlock)
	{
	/* Flip the physically next block's physPrevIsFree bit */
	if (isPhysPrevAllocated(getPhysNext(p)))
	    flipPhysPrevAllocated(getPhysNext(p));
	}

    /* Set the hint to point to the newly freed block */
    hintLength = length;
    hint = p;
    }