/*
 * mdriver.c -- driver program for the 15213-f98 Lab 2 timing server
 * 
 *              Kip Walker
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "malloc.h"
#include "mdriver.h"
#include "ftime.h"

extern team_struct team;
extern char *codename(char *, char *);

char *heap = NULL;
long heap_size;
struct range *ranges = NULL;

int coalescing = 0;
long int COUNTER;

char *filelist[] = { 
  "/afs/cs.cmu.edu/usr/kwalker/public/traces/amptjp.rep",
  "/afs/cs.cmu.edu/usr/kwalker/public/traces/cccp.rep",
  "/afs/cs.cmu.edu/usr/kwalker/public/traces/cp-decl.rep",
  "/afs/cs.cmu.edu/usr/kwalker/public/traces/expr.rep",
  NULL
};

int main(int argc, char *argv[])
{
    printf("email: %s,%s\n", team.andyid_1,
           (team.andyid_2) ? team.andyid_2 : "");
    printf("codename: %s\n", 
           codename(team.member_1, (team.member_2) ? team.member_2 : ""));
    fflush(stdout);

    srand(42);

    if (!test_correctness()) {
      fprintf(stderr, "Failed correctness test. Aborted other tests.\n");
      exit(0);
    }
    
    if (!test_efficiency()) {
      fprintf(stderr, "Doesn't coalesce. Aborted other tests.\n");
      exit(0);
    }
    
    test_performance();
    
    return 0;
}

int init_driver(long int size)
{
    void *heap_start;

    if (heap)
        free(heap);

    /* round it to a long int length */
    size = ((size+WDSZ-1) & ~WDSZ_MASK);
    
    /* get the memory for our heap (add slack to be sure we can align it)*/
    heap = malloc(size+WDSZ);
    if (!heap)
        return -1;
    
    /* align it just in case */
    heap_start = (char *)(((long)heap+WDSZ-1) & ~WDSZ_MASK);

    return init_malloc(size, heap_start);
}

int test_correctness(void)
{
    long int num_blocks;
    void **ptrs;
    void **next;
    void *newptr;
    long int size;
    int correct = 1;
    
    heap_size = 256*1024;
    if (init_driver(heap_size) == -1) {
        fprintf(stderr, "failed init_malloc in correctness test\n");
        exit(-1);
    }
    
    num_blocks = heap_size/WDSZ;
    ptrs = malloc(sizeof(char *)*num_blocks);
    next = ptrs;
    do {
        size = (int)(127.0*rand()/(RAND_MAX+1.0)) + 1;
        newptr = L2_malloc(size);
        *next++ = newptr;
        if (newptr && !add_range(newptr, size)) {
            correct = 0;
        }
    } while (newptr && correct);

    if (next-ptrs <10) {
        fprintf(stderr, "didn't get enough successful mallocs to test correctness (%ld)\n", next-ptrs);
        fflush(stderr);
        correct = 0;
    }
      
    next = ptrs;
    while (*next) {
        L2_free(*next);
        next++;
    }

    clear_ranges();

    printf("correct: %s\n", correct ? "yes" : "no");
    fflush(stdout);

    return correct;
}

int test_efficiency(void)
{
    long int num_blocks;
    char **ptrs;
    char **next;
    char *newptr, *newptr2;
    long int size;
    long int max_alloc;
    int count;
    unsigned min_alloc = 0;
    unsigned long try_size;

    size = heap_size = 64*1024;
    if (init_driver(heap_size) == -1) {
        fprintf(stderr, "failed init_malloc in overhead test\n");
        exit(-1);
    }
    num_blocks = heap_size/WDSZ;

    /* determine overhead: this gives us static heap overhead plus
     * overhead of one allocated block */
    while ((newptr = L2_malloc(size)) == NULL)
        size -= WDSZ;
    
    max_alloc = size;
    L2_free(newptr);

    count = 0;
    ptrs = (char **)malloc(sizeof(char *)*num_blocks);
    next = ptrs;
    while ((newptr = L2_malloc(65)) != NULL) {
        *next++ = newptr;
        count++;
    }
    *next = NULL;
    next = ptrs;
    while (*next) {
        L2_free(*next);
        next++;
    }
    free(ptrs);

    if ((count > 1) && (newptr = L2_malloc(max_alloc))) {
        coalescing = 1;
        L2_free(newptr);
    }

    printf("coalesce: %s\n", coalescing ? "yes" : "no");
    printf("overhead: %f\n", 1.0-((count*65)/(float)heap_size));
    fflush(stdout);

    try_size = WDSZ;

    while (min_alloc == 0) {
        if (init_driver(heap_size) == -1) {
            fprintf(stderr, "failed init_malloc in overhead test\n");
            exit(-1);
        }

        newptr = L2_malloc(max_alloc - try_size);
        newptr2 = L2_malloc(1);
        if (newptr2) {
            L2_free(newptr2);
            min_alloc = try_size;
        } else
            try_size += WDSZ;

        L2_free(newptr);
    }
    printf("minblock: %u\n", min_alloc);
    fflush(stdout);

    return coalescing;
}

void test_performance(void)
{
    double secs;
    double ratio = 0.0;

    COUNTER = 0;
    secs = ftime(simple_test, 0);
    ratio += (secs/(double)COUNTER);
    fprintf(stderr, "perf. test number 1 finished\n");
    fflush(stderr);

    COUNTER = 0;
    secs = ftime(free_test, 0);
    ratio += (secs/(double)COUNTER);
    fprintf(stderr, "perf. test number 2 finished\n");
    fflush(stderr);

    COUNTER = 0;
    secs = trace_test();
    fprintf(stderr, "perf. test number 3 finished\n");
    fflush(stderr);

    ratio += (secs/(double)COUNTER);
    ratio /= 3.0;
    printf("performance: %2.10f\n", ratio);
    fflush(stdout);
}

/*---------------------------
 * performance test engines
 ----------------------------*/

void simple_test(void)
{
    long int num_blocks;
    char **ptrs;
    char **next;
    char *newptr;

    COUNTER = 0;
    heap_size = 1024*1024;
    if (init_driver(heap_size) == -1) {
        fprintf(stderr, "failed in init_malloc in performance test\n");
        exit(-1);
    }

    num_blocks = heap_size/WDSZ;
    ptrs = (char **)malloc(sizeof(char *)*num_blocks);
    next = ptrs;
    while ((newptr = L2_malloc(WDSZ)) != NULL) {
        *next++ = newptr;
        COUNTER++;
    }
    *next = NULL;
    next = ptrs;
    while (*next) {
        L2_free(*next);
        next++;
        COUNTER++;
    }

    free(ptrs);
}

/*
 * globals here shared between trace_test and do_trace
 *   split up so I could use "ftime" without measuring file I/O
 */
long int num_blocks;
void **blocks;
unsigned int evt_index;
struct event *events;
long int trace_heap_size;

double trace_test(void)
{
    FILE *tracefile;
    char **av;
    double secs = 0.0;
    long int accum = 0;

    av = filelist;
    while (*av != NULL) {
        tracefile = fopen(*av, "r");
        if (tracefile) {
            char type[10];
            unsigned int size;
            unsigned int index, max_index;
            
            fscanf(tracefile, "%ld", &trace_heap_size);
            
            fscanf(tracefile, "%ld", &num_blocks);
            blocks = malloc(num_blocks * sizeof(void *));
            events = (struct event *)malloc(num_blocks<<1 * sizeof(struct event));
            index = 0;
            max_index = 0;
            evt_index = 0;
            while (fscanf(tracefile, "%s", type) != EOF) {
                switch(type[0]) {
                case 'a':
                    fscanf(tracefile, "%u %u", &index, &size);
                    events[evt_index].type = MALLOC;
                    events[evt_index].index = index;
                    events[evt_index].size = size;
                    
                    break;
                case 'f':
                    fscanf(tracefile, "%ud", &index);
                    events[evt_index].type = FREE;
                    events[evt_index].index = index;
                    break;
                default:
                    fprintf(stderr, "Bogus type character\n");
                    exit(-1);
                }
                evt_index++;
            }
            fclose(tracefile);
            
            secs += ftime(do_trace, 0);
            accum += COUNTER;

            free(blocks);
            free(events);
        }
        av++;
    }
    COUNTER = accum;
    return secs;
}

void do_trace(void)
{
    unsigned index;
    unsigned max_index = 0;

    COUNTER = 0;
    /* have init_driver set up the heap so we can
     * start getting blocks of memory */
    if (init_driver(trace_heap_size) == -1) {
        fprintf(stderr, "Failed in init_malloc in performance test.\n");
        exit(-1);
    }

    for (index = 0; index < evt_index; index++) {
        switch (events[index].type) {
        case MALLOC:
            blocks[events[index].index] = L2_malloc(events[index].size);
            if (!blocks[events[index].index]) {
                fprintf(stderr,
                        "Ran out of core on malloc %d\n",
                        index);
                fprintf(stderr, "email kwalker@cs if you see this message\n");
                fflush(stderr);
                exit(-1);
            }
            COUNTER++;
            if (events[index].index > max_index)
                max_index = events[index].index;
            break;
        case FREE:
            COUNTER++;
            L2_free(blocks[events[index].index]);
            blocks[events[index].index] = NULL;
            break;
        }
    }
    for (index = 0; index <= max_index; index++)
        if (blocks[index]) {
            COUNTER++;
            L2_free(blocks[index]);
        }
}

void free_test(void)
{
    int count, i;
    void **blocks;

    COUNTER = 0;
    heap_size = 256*1024;
    blocks = (void **)malloc(heap_size >> 2);

    if (init_driver(heap_size) == -1) {
        fprintf(stderr, "Failed in init_malloc in performance test.\n");
        exit(-1);
    }

    count = 0;
    while ((blocks[count] = L2_malloc(50)) != NULL) {
        COUNTER++;
        count++;
    }
    for (i=0; i<count; i+=2) {
        L2_free(blocks[i]);
        COUNTER++;
    }
    for (i-- ; i>=0; i-=2) {
        if (i>=0 && blocks[i] != NULL) {
            L2_free(blocks[i]);
            COUNTER++;
        }
    }

    free(blocks);
}

/*----------------------------
 * range routines
 -----------------------------*/
int add_range(char *start, long int size)
{
    struct range *new_range, *r;
    char *end = start+size;
    int allowed = 1;

    if (start < heap || end > heap+heap_size) {
        allowed = 0;
    } else {
        r = ranges;
        while (r) {
            if ((start >= r->start && start < r->end) ||
                (end > r->start && end <= r->end))
                break;
            else
                r = r->next;
        }
        
        if (!r) {
            new_range = (struct range *)malloc(sizeof(struct range));
            new_range->start = start;
            new_range->end = start+size;
            new_range->next = ranges;
            ranges = new_range;
        } else {
            allowed = 0;
        }
    }

    return allowed;
}

void clear_ranges(void)
{
    struct range *curr, *next;

    curr = ranges;
    while (curr) {
        next = curr->next;
        free(curr);
        curr = next;
    }
}
