#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

team_t team = {
    "ether bingers",   /* Team name to be displayed on webpage */
    "jacob richman",   /* First member full name */
    "jsr",             /* First member email address */
    "pawel opalinski", /* Second member full name */
    "pjo"              /* Second member email address  */
};


#define TEST_BIN_INDEX (0)
#define TEST_ALLOC_MAP (0)
#define TEST_SBRK      (0)
#define TEST_BS_TO_WS  (0)
#define TEST_ALIGN     (0)
#define TESTING        (TEST_BIN_INDEX | TEST_ALLOC_MAP | TEST_SBRK | TEST_BS_TO_WS | TEST_ALIGN )
#define BREAK_EARLY    (0)

#define ASSERT         (0)
#define MEM_SWEEP      (0)
#define ALLOC_DRAW_MEM (0)
#define FREE_DRAW_MEM  (0)

#define COALESE_PREV   (1)
#define COALESE_NEXT   (1)
#define COALESE_WILD   (1)
#define CHOP_BLOCKS    (0)
#define BIN_SEARCH     (1)

#define FEW_BINS       (0)

typedef int word;

#define ALIGN          (2)
#define WASTE          (3)
#define MIN_ALLOC      (4)
#define BYTES_PER_WORD (4)
#define SBRK(x)        mem_sbrk( (x) * BYTES_PER_WORD )

#define sizeAlign(x) (((x)+(ALIGN-1)) & ~(ALIGN-1))
#define byteSizeToWordSize(x)   ((x>>2) + (!!(x & (BYTES_PER_WORD - 1))))

#if FEW_BINS
	#define NBINS     (12)
#else
	#define NBINS     (128)
#endif

word *bin[NBINS];
word *wilderness;
word *heapStart;



#define NBINCLASSES (4)
#if FEW_BINS
	int binSize[ NBINCLASSES ] = { 2, 4, 6, 8 };
	int binNum[ NBINCLASSES ] = { 4, 3, 3, 0 };
#else
	int binSize[ NBINCLASSES ] = { 2,  16, 128, 512  };
	int binNum[ NBINCLASSES ]  = { 64, 32, 16,  14   };
#endif

int sizeToBinIndex(int size){
	int i,temp,binIndex = 0;	

	for ( i=0; i<NBINCLASSES ; i++ ){
		temp  = size / binSize[ i ];	
		if (temp>binNum[i]){
			temp = binNum[i];
		}
		size -= temp*binSize[i];
		binIndex += temp;
	}
	
	binIndex = binIndex + (!!size);
	#if ASSERT
		assert( binIndex >= 0 && binIndex < NBINS );
	#endif
	return binIndex;
}

#if TEST_BIN_INDEX
#define TEST_SIZE (6)
int testSize[] = { 4, 16, 128, 130 , 43000/8, 50000};
void testBinSizeCode(){
	int i,temp;
	for(i=0; i < TEST_SIZE ; i++){
		temp = testSize[i];
		printf("sizeToBinIndex(%d)=%d\n",temp,sizeToBinIndex(temp));
	}
	for(i=1; i<testSize[TEST_SIZE-1] ; i++) {
		temp = sizeToBinIndex( i );
		assert( temp >= 0 && temp < NBINS );
	}
	printf("all sizeToBinIndex(%d -> %d) are within bounds\n",1,testSize[TEST_SIZE-1]);
}
#endif




#define ALLOC_MAP (1)
#define isAllocated(x)     ((*x) & ALLOC_MAP)
#define isFree(x)          (!isAllocated(x))
#define markAsAllocated(x) ((*x) = ((*x) | ALLOC_MAP))
#define markAsFree(x)      ((*x) = ((*x) & (~ALLOC_MAP)))
#define sizeOf(x)          ((*x) & (~ALLOC_MAP))
#define setSize(x,s)       (*x) = (s); (*((x)+(s)-1)) = (s)
#define toEnd(x)           ((x)+sizeOf(x)-1)
#define toWordAddress(x)   ((word *)(((word)(x)) & (~(BYTES_PER_WORD-1))))


#define PREV_OFFSET   (1)
#define NEXT_OFFSET   (2)
#define nextFreeBlock(x)   ((word *)(*(x+NEXT_OFFSET)))
#define prevFreeBlock(x)   ((word *)(*(x+PREV_OFFSET)))

#if TEST_ALLOC_MAP
void testAllocMapCode(){
	int i;

	markAsAllocated(wilderness);

	printf("isAllocated(wilderness)=%d\n",isAllocated(wilderness));
	printf("sizeOf(wilderness)=%d\n",sizeOf(wilderness));
	for(i=0;i<sizeOf(wilderness);i++){
		printf("word %d = %d\n",i,*(wilderness+i));
	}

	markAsFree(wilderness);

	printf("isAllocated(wilderness)=%d\n",isAllocated(wilderness));
	printf("sizeOf(wilderness)=%d\n",sizeOf(wilderness));
	for(i=0;i<sizeOf(wilderness);i++){
		printf("word %d = %d\n",i,*(wilderness+i));
	}

}

#endif

#if TEST_SBRK
void testSBRKCode(){
	printf("dseg_lo=%d\n",(word)dseg_lo);
	printf("dseg_hi=%d\n",(word)dseg_hi);
	printf("wilderness=%d\n",(word)wilderness);
	printf("wilderness+sizeOf(wilderness)=%d\n",((word)(wilderness+sizeOf(wilderness))));
	printf("toEnd(wilderness)=%d\n",(word)(toEnd(wilderness)));
	printf("toWordAddress(dseg_hi)=%d\n",(word)(toWordAddress(dseg_hi)));
}
#endif
#if TEST_BS_TO_WS
void testBSToWS(){
	printf("byteSizeToWordSize(%d)=%d\n",1,byteSizeToWordSize(1));
	printf("byteSizeToWordSize(%d)=%d\n",4,byteSizeToWordSize(4));
	printf("byteSizeToWordSize(%d)=%d\n",5,byteSizeToWordSize(5));
	printf("byteSizeToWordSize(%d)=%d\n",6,byteSizeToWordSize(6));
	printf("byteSizeToWordSize(%d)=%d\n",7,byteSizeToWordSize(7));
	printf("byteSizeToWordSize(%d)=%d\n",8,byteSizeToWordSize(8));
	printf("byteSizeToWordSize(%d)=%d\n",9,byteSizeToWordSize(9));
}
#endif

#if TEST_ALIGN
void testSizeAlignCode(){
	int i;
	for (i=0;i<17;i++) {
		printf("sizeAlign(%d)=%d\n",i,sizeAlign(i));
	}
}
#endif

void drawMemory(){
	word *p;

	printf("heapStart=%d\n",(word)heapStart);
	printf("wildernes=%d\n",(word)wilderness);

	for (p=heapStart; p <= (toEnd(wilderness)) ; p++ ){
		printf("mem[%d]=%d\n",(word)(p-heapStart),*p);
	}

}

void drawBin(int i){
	word *p;
	for (p=bin[i]; p; p=nextFreeBlock(p)){
		printf("%d -> ",sizeOf(p));
	} 
}
void drawBins(){
	int i;
	for (i=0;i<NBINS;i++){
		printf("bin %d:\t",i);
		drawBin(i);
		printf("\n");
	}
}

int isInBin(int index , word *lookingFor ){
	word *p;
	for (p=bin[index];p;p=nextFreeBlock(p)){
		if (lookingFor == p){
			return 1;
		}
	}
	return 0;
}
int isInBins(word *lookingFor){
	return isInBin( sizeToBinIndex(sizeOf(lookingFor)) , lookingFor);
}

void removeFromBin(int index, word *p ){
	word *prev,*next;

	#if ASSERT
		assert( isInBin( index , p ) );
	#endif
	
	prev = prevFreeBlock(p);
	next = nextFreeBlock(p);

	if (p==bin[ index ]){
                bin[ index ] = next;
        }

        if (prev) {
                nextFreeBlock(prev) = next;
        }
        if (next) {
                prevFreeBlock(next) = prev;
        }

	#if ASSERT
		assert( !isInBin( index, p ) );
	#endif

}
void putInBin(int index, word *p ){
	
	#if ASSERT
		assert( !isInBin( index, p ) );
	#endif

	prevFreeBlock(p) = NULL;	
	nextFreeBlock(p) = bin[ index ];
	if (bin[index]){
		prevFreeBlock(bin[index]) = p;
	}
	bin[index] = p;

	#if ASSERT
		assert( isInBin( index, p ) );
	#endif
}

#if BREAK_EARLY
	int initCount=0;
#endif

int mm_init (void) {
	int i;
	#if BREAK_EARLY
		if (initCount++) exit(0);
	#endif
	
	if (!(heapStart = wilderness = SBRK( MIN_ALLOC ) )) {
		return -1;
	}
	setSize(wilderness, MIN_ALLOC);
	markAsFree(wilderness);

	for (i=0 ; i < NBINS ; i++) {
		bin[ i ] = NULL;
	}

	#if TEST_ALLOC_MAP
		testAllocMapCode();
	#endif

	#if TEST_SBRK
		testSBRKCode();
	#endif
	
	#if TEST_BS_TO_WS
		testBSToWS();
	#endif

	#if TEST_ALIGN
		testSizeAlignCode();
	#endif

	#if TEST_BIN_INDEX
		testBinSizeCode();
	#endif
	
	
	#if TESTING
		exit(0);
	#endif

	return 0;
}

word *getFromWilderness(int size){
	word* freeLocation;
	int wildSize;
	
	wildSize = sizeOf(wilderness);
	if (wildSize < (size+MIN_ALLOC) ){
		if (!(SBRK( size + MIN_ALLOC - wildSize ))){
			return NULL;
		}
		wildSize = size + MIN_ALLOC;
	}

	freeLocation = wilderness;
	wilderness = wilderness + size;

	wildSize -= size;
	setSize(wilderness, wildSize);
	markAsFree(wilderness);

	setSize(freeLocation,size);
	markAsFree(freeLocation);

	return freeLocation;	
}

void chop(word *p, int size){
	word *freeBlock;
	int totalSize = sizeOf(p);
	
	if ( totalSize >= (size+MIN_ALLOC) ){

		freeBlock = p+size;

		setSize(freeBlock,totalSize-size);
		markAsFree(freeBlock);
		putInBin(sizeToBinIndex(sizeOf(freeBlock)),freeBlock);

		setSize(p,size);
		markAsFree(p);
	}
}
word *getFromLargerBins(int startIndex,int size){
	int i;
	word *p;
	for (i=startIndex;i<NBINS;i++){
		if (bin[i]){
			p = bin[i];
			removeFromBin(i,p);
			chop(p,size);
			return p;
		}
	}
	return getFromWilderness(size);
}

word *getFreeBlock(int size){
	int binIndex = sizeToBinIndex( size );
	word *p;
	for ( p=bin[ binIndex ]; p ; p=nextFreeBlock(p) ) {
		if (size <= sizeOf(p)){
			removeFromBin( binIndex, p );
			chop(p,size);
			return p;
		}
	}

	binIndex++;
	if (binIndex<=binNum[0]) binIndex++;	
	#if BIN_SEARCH
		return getFromLargerBins( binIndex, size ); 
	#else
		return getFromWilderness(size);
	#endif
}

void memSweep(){
	word *p = heapStart;
	
	MEM_SWEEP_LOOP:
		if (p==wilderness){
			assert( ((char *)(p+sizeOf(wilderness))) == (dseg_hi+1) );
			return;
		} else {
			assert( p < wilderness );
			assert( p >= heapStart );
			if (isAllocated(p)){
				assert(!isInBins(p));
			} else {
				assert(isInBins(p));
			}
			p += sizeOf(p);
		}	
	goto MEM_SWEEP_LOOP;
}



void *mm_malloc (size_t size) {
	word *freeLocation;
	#if MEM_SWEEP
		memSweep();
	#endif

	size = byteSizeToWordSize(size);
	size += WASTE;
	size = sizeAlign(size);
	
	#if ALLOC_DRAW_MEM
		printf("\nmm_malloc(%d)\n",size);
		drawMemory();
		drawBins();
	#endif
	
	if ( (freeLocation = getFreeBlock( size )) ) {
		markAsAllocated( freeLocation );
		freeLocation += ALIGN;
	}

	#if ALLOC_DRAW_MEM
		drawMemory();
		drawBins();
	#endif
	#if MEM_SWEEP
		memSweep();
	#endif
	return freeLocation;	
}


int coaleseWithWilderness(word *p){
#if COALESE_WILD
	int totalSize;

	if ( (p + sizeOf(p)) != wilderness ){
		return 0;
	}

	totalSize = sizeOf(p) + sizeOf(wilderness);
	wilderness = p;
	setSize(wilderness,totalSize);
	
	return 1;	
#else
	return 0;
#endif
}

void merge(word *first, word *second){
	int totalSize = sizeOf(first) + sizeOf(second);
	setSize(first,totalSize);
	markAsFree(first);
}

word *coaleseWithPrev(word *p){
#if COALESE_PREV
	word *prev;
	if (p<=heapStart){
		return p;
	} else {
		prev = p - *(p-1);
		if (isAllocated(prev)){
			return p;
		} else {
			removeFromBin(sizeToBinIndex(sizeOf(prev)),prev);
			merge(prev,p);
			return prev;
		}
	}
		
#else
	return p;
#endif
}
word *coaleseWithNext(word *p){
#if COALESE_NEXT
	word *next;
	next = p + sizeOf(p);
	if ( next == wilderness ) {
		return p;
	} else {
		if (isAllocated(next)){
			return p;
		} else {
			removeFromBin(sizeToBinIndex(sizeOf(next)),next);
			merge(p,next);
			return p;
		}
	}
#else
	return p;
#endif
}

void mm_free (void *ptr) {
	word *p =  (word *) ptr;
	      p -= ALIGN;
	#if MEM_SWEEP
		memSweep();
	#endif
	
	markAsFree(p);
	#if FREE_DRAW_MEM
		printf("\nfree(%d)\n",(word)(p-heapStart));
		drawMemory();
		drawBins();
	#endif

	p = coaleseWithPrev(p);
	p = coaleseWithNext(p);	
	if (!coaleseWithWilderness(p)){
		putInBin( sizeToBinIndex( sizeOf(p) ), p );
	}

	#if FREE_DRAW_MEM
		drawMemory();
		drawBins();
	#endif
	#if MEM_SWEEP
		memSweep();
	#endif
}
