/* $Id$ */

/*
 *
 *  CS213 - Lab assignment 3
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"


/* Our method uses a single free list to keep track of which blocks are free.

   The structure of a free block is as follows:
   +------+------+------+------------+------+
   | tag  | prev | next |   filler   | tag  |
   +------+------+------+------------+------+
   The first tag keeps track of the size of the block, stored in number of
   bytes.  Since everything will be aligned by eight, the last three bytes are
   irrelevant in terms of significance.  Because of this, we use the last two
   of them to keep track of two things.  The least significant bit keeps track 
   of whether the block is empty and the second least significant bit is used
   to keep track of whether the previous block in memory is free. Areas outside
   the heap are not considered free.
   prev points to the previous element in the free list.
   next points to the next element in the free list.
   Filler does nothing. It is just free space.
   The end tag keeps track of the size.  It does not use the last two bits for
   anything.
   Since there are always four four byte regions that are used in the free
   areas, the minimum size of a free block (and thus any block) is 16 bytes.

   The structure for the allocated block is as follows:
   +------+------------+
   | tag  |   data     |
   +------+------------+
   The tag is the same as that of the free block.
   Data is the space that is returned to the application calling mm_malloc.
   Notice that the blocks are aligned by odd multiples of four. 

   Generally, when accessing size, *ref & normalize is used to eliminate the
   flags, which would otherwise change the size.  (*ref & is_empty) and
   (*ref & prev_empty) inform of whether the block is empty or the previous
   block is empty, respectively. *ref + is_empty or *ref + prev_empty set
   the empty/previous empty flags to true, respectively.


   Because we have a 4-byte tag before our allocated block and because all
   of the data has to be 8-byte aligned, the tags have to start on addresses
   that are odd multiples of four, and the number of bytes in the data portion
   of allocated blocks also has to be an odd multiple of 4. We also need a
   place in  memory to store the head of the free list. We put this in the
   first 4-bytes of the 8-byte aligned heap, and directly after that we have
   the first tag of the first block of memory (which will now start on an
   address that is an odd multiple of four). These requirements mean that the
   last 4-bytes before an 8-byte aligned equivalent of dseg_hi cannot be part
   of any blocks: we use these last bytes (which we refer to as "tail" in our
   code) to point to the beginning tag of the last block of memory we have in
   the heap (which is useful for coalescing newly requested pages of memory
   from sbrk). This is how the heap will look in general (blocks not drawn to
   size):
 */




/* These are used as tag flags.  Names are assigned for readablility sake */

#define is_empty 1
#define prev_empty 2
#define normalize (unsigned int) (-4)

team_t team = {
    /* Team name to be displayed on webpage */
    "void *",
    /* First member full name */
    "Jeffrey Ian Sarnat",
    /* First member email address */
    "jsarnat@andrew.cmu.edu",
    /* Second member full name (leave blank if none) */
    "Paul Ip",
    /* Second member email address (blank if none) */
    "pmi@andrew.cmu.edu"
};




int *coalesce(int *block1, int *block2);

int mm_init (void)
{
  int *low;
  int *high;
  if (!mem_sbrk(mem_pagesize())) return -1;/* if mem_sbrk fails, return -1 */

  /* low and high are dseg_lo and dseg_hi aligned to 8 */
  low = (int *) dseg_lo;
  high = (int *) dseg_hi;
  (int) low = ((int) low + 7) & (-8);
  (int) high = (int) high & (-8);

  *low = (int) (low + 1);  /* pointer to free list point to first free block */
  *(low+1) = ((unsigned int)high - (unsigned int)low - 8 + is_empty);/* size */
  *(low+2) = 0;             /* points to prev free block (in this case NULL) */
  *(low+3) = 0;             /* points to next free block (in this case NULL) */
  *(high-2) = ((unsigned int)high - (unsigned int)low - 8);/* end tag (size) */
  *(high-1) = *low;                             /* tail points to last block */
  return 0;
}

/* This function removes free block pointed to by ref from the free list.
   It also sets the block's next and prev pointers to null. */
void remove_from_list(int *ref, int *head)
{
  if (*(ref+1) == 0 && *(ref+2) == 0)
    *head = 0;
  if (*(ref+1) == 0 && *(ref+2) != 0)
  {
     *(((int *) *(ref+2)) + 1) = 0;
    *head = *(ref + 2);
  }
  if (*(ref+1) != 0 && *(ref+2) == 0)
    *(((int *) *(ref+1)) + 2) = 0;
  if (*(ref+1) != 0 && *(ref+2) != 0)
  {    
    *(((int *) *(ref+1)) + 2) = *(ref+2);
    *(((int *) *(ref+2)) + 1) = *(ref+1);
  }
  *(ref+1) = 0;
  *(ref+2) = 0;
}

/* This fuction inserts a free block into the head of the free list.  Only
   assumption is that the size (not bit flags) are correctly set.  Creates
   the end tag and and also verifies the start tag. */

void insert_free(int *ref, int *head)
{
  /* verifies  start and end tags.  Start should be size + is_empty.  The
     previousl block will never be empty since every new free block is
     coalesced if its previous is empty, thus there should be no adjacent
     empty blocks.  The end tag gets the size entered. */
  *ref = (*ref & normalize) + is_empty;
  *((int *) ((int) ref + (*ref & normalize) - 4)) = (*ref & normalize);
  *(ref + 1) = 0;
  *(ref + 2) = *head;
  if (*head)
    *(((int *) (*head)) + 1) = (int) ref;
  *head = (int) ref;
}

/* This fuction allocates the minimal amount of heap.  If the last block
   in the previously allocated area is free, it is coalesced with the new
   heap area.  The returned value points to a free block that is big enough
   to accomodate the next requested block and the end_tag is not set (to be
   done in insert, since it is not needed in mm_malloc.  Tail is updated
   after function returns.  */
int *more_heap(size_t size, int *tail, int *head)
{
  int *high;
  int num_new_pages;
  int page_size = mem_pagesize();
  int mem_needed = size;
  int *ref;
  int *last_block;              /* refers to the last block in old heap area */
  last_block = (int *) *tail;
  ref = tail;      /* new block will start at tail to keep blocks contiguous */
  if(*last_block & is_empty)      /* if last_block empty, need less new heap */
    mem_needed -= *last_block & normalize;
  num_new_pages = (mem_needed + page_size - 1)/page_size;
  if (!mem_sbrk(num_new_pages*page_size)) return NULL;   /* if, fails, fails */
  high = (int *) (((int) dseg_hi) & (-8));           /* reinit high and tail */
  tail = high - 1;
  *ref = ((int) tail) - ((int) ref) + is_empty;            /* calculate size */
  *(ref+1) = 0;                                             /* init pointers */
  *(ref+2) = 0;
  if(*last_block & is_empty)
  {
    remove_from_list(last_block, head);
    ref = coalesce(last_block, ref);
  }
  return ref;
}


void *mm_malloc (size_t size)
{
  int *head;
  int *high;
  int *tail;
  int *ref;
  int *end_tag;
  int *new_free;
  
  if(!size) return NULL;            /* We WILL NOT give a pointer to 0 bytes */
  if (size <= 12)        /* Since smallest free block is 16 bytes, allocated */
    size = 16;      /* blocks should be min 16 bytes to avoid fragmentations */
  else
    size = ((size + 4) + 7) & (-8);     /* else, round (up) size to mult of 8*/

  /* init high, head, tail */
  head = (int *) dseg_lo;
  high = (int *) dseg_hi;
  (int) head = ((int) head + 7) & (-8);
  (int) high = (int) high & (-8);
  tail = high - 1;

  /* traverse free list till FIRST suitable block is found, or until no more
     list to traverse, then in that case ref would be NULL */
  ref = (int *) *head;
  while(ref && ((*ref) & normalize) < size)
  {
    ref = (int *) *(ref+2);
  }
  
  /* case when no suitable free block is found */
  if(!ref)
  {
    ref = more_heap(size, tail, head);          /* gets block from more heap */
    if(!ref) return NULL;                              /* If it can't, fails */

    /* fixes tail pointer, since it wasn't done by more_heap */
    high = (int *) dseg_hi;
    (int) high = (int) high & (-8);
    tail = high - 1;
    *tail = (int) ref;                   /* the new block is inherently last */
  }
  /* if suitable block found, remove from free list, to be consisistent */
  else
  {
    remove_from_list(ref,head);
  }

  /* AT THIS POINT REF SHOULD BE A FREE BLOCK THAT IS NO LONGER IN THE
     FREE LIST, TAIL SHOULD BE UPDATED IF NECESSARY*/

  /* prevents creating of a free block of size<16 */
  if ((*ref & normalize) - size < 16)
    size = (*ref & normalize);
  
  /* establishes location of the end tag (end of area being worked on */
  end_tag = (int *) ((int) ref + (*ref & normalize) - 4);
  new_free = (int *) ((int) ref + size);          /* start of new free block */

  /* if proposed new free block location is valid */
  if (new_free < end_tag)
  {
    /* set start tag and insert into free list */
    *new_free = (*ref & normalize) - size + is_empty; 
    insert_free(new_free, head);

    /* set start tag of allocated block */
    *ref = (int) (size + (*ref & prev_empty));

    /* if this block was the last one (tail), then must update tail since
       a new (free) block was created. */
    if (*tail == (int) ref)
      *tail = (int) new_free;
  }
  /* if there is to be no new free block */
  else
  {
    *ref = *ref & normalize;                               /* sets start tag */
    /* must modify next block to say that its prev block is not empty */
    *((int *) ((int) ref + (*ref & normalize))) -= prev_empty;
  }
  return (void *) (ref + 1);
}


/* COALESCES EMPTY BLOCKS 1 AND 2 AND RETURNS ADDRESS TO COALESCED BLOCK
   ALSO SETS TAGS AT BEGINNING AND END OF BLOCK */
int *coalesce(int *block1, int *block2)
{
  *block1 = (*block2 & normalize) + (*block1 & normalize) + is_empty;
  *((int *) ((int) block1 + (*block1 & normalize) - 4)) = *block1 & normalize;
  *(block1+1) = 0;
  *(block1+2) = 0;
  
  return block1;
}




void mm_free (void *ptr)
{
  int *ref;
  int *next;
  int *prev;
  int *tail;
  int *head;

  /* sets up some variables that will be useful later on.  Next may not be in
     the heap, but we won't use it unless it is. */
  (int) tail = ((int) dseg_hi & (-8)) - 4;
  (int) head = (((int) dseg_lo) +7) & (-8);
  (int) ref  = (((int) ptr)-4);
  (int) prev = ((int) ref - (*(ref-1) & normalize));
  (int) next = ((int) ref + (*ref & normalize));

  if(!(*ref & prev_empty))
  {    
    if (next >= tail)
    { /* case when previous is allocated/nil, next outside of heap */
      *ref = (*ref & normalize) + is_empty;
      insert_free(ref, head);
      return;
    }
    else if (!(*next & is_empty))
    { /* case when previous is allocated/nil, next is allocated */
      assert (!(*next & prev_empty));
      *ref = (*ref & normalize) + is_empty;
      *next = *next + prev_empty;      /* must change next's prev_empty flag */
      insert_free(ref, head);
      return;
    }      
    else
    { /* case when previous is allocated/nil, next is free */
      assert (!(*next & prev_empty));
      remove_from_list(next, head);
      if (*tail == (int) next) *tail = (int) ref;   /* update tail if needed */
      ref = coalesce(ref, next);
      insert_free(ref, head);
      return;
    }
  }
  else
  { /* cases when previous is free, then coalesce */
    remove_from_list(prev, head);
    ref = coalesce(prev, ref);
    if ((next < tail) && (*next & is_empty))
    { /* case when next is free, coalesce more, update tail if needed */
      if (*tail == (int) next) *tail = (int) ref;
      remove_from_list(next, head);
      ref = coalesce(ref, next);
      
    }
    /* case when next is allocated, update next's prev_empty */
    else if ((next < tail) && !(*next & is_empty))
      *next = *next + prev_empty;
    else
    { /* if next is outside of heap, update tail (because coalesced) */
      *tail = (int) ref;
    }
    insert_free(ref, head);
    return;
  }
  return;
}
