/* $Id$ */

/*
 *
 *  CS213 - Lab assignment 3
 *  Filipe Fortes (ffortes)
 *  Jamie Schwartz (jas2)
 * 
 ******************************
 * About our implmentation    *
 ******************************
 *
 * We have basically chosen a segregated free list scheme with a
 * first-fit search on the free list.
 *
 * We have a lot of free lists 256, and use a large number of those
 * for the smaller (more commonly allocated) sizes.
 *
 * Each free block has a header and footer, along with previous and next
 * pointers.  However, the alloc'd blocks don't have a footer.  We got
 * around having to use a footer by instead marking the 2nd lowest bit
 * on the header of the next block (since our sizes are multiples of 4
 * this isn't a problem).
 *
 * The rest is pretty straightforward, this is a fairly quick and efficient
 * method, which works pretty well on everything but the binary.
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

/*************************************/
/* Bunch of macros for sanity's sake */
/*************************************/

#define WORDSIZE 4
#define ALIGN_SIZE 4

/* Number of seg lists we have.  These start at 1, and end at NUM_BINS */
#define NUM_BINS 256

/* The first memory available after our seg. list structure */
#define FIRST_MEM_AVAIL (int*) dseg_lo + NUM_BINS + 1

/* The minimum size block we will allocate */
#define MIN_BLOCK_SIZE 4

/* Simple max macro */
#define max(x,y) (x > y) ? x : y

/* The dereference of dseg_lo, this is what we use as our pointer to the wilderness */
#define top (int*) (*((int*) dseg_lo))

/* Acts like an array for our segregated lists ... entries 1 through NUM_BINS are valid */
#define freelist(i) (int*) (*((int*) dseg_lo + i))

/* Finds the size of a given chunk */
#define getsize(ptr) (*((int*) ptr) & -4)

/* Pointers to the previous and next items on the segregated list */
#define prev_list(ptr) (int*) (*((int*)ptr + 1))
#define next_list(ptr) (int*) (*((int*)ptr + 2))

/* Pointers to the previous and next items in physical memory */
#define prev_phys(ptr) (int*) ((int*) ptr - (*((int*)ptr - 1) & -4))
#define next_phys(ptr) (int*) ((int*) ptr + getsize(ptr))

/* Checks if the previous or next spot is taken */
#define prev_taken(ptr) ((*((int *)ptr) & 2))
#define next_taken(ptr) ((*((int *)ptr + getsize(ptr))) & 1)

/* Goes to the footer of a given chunk */
#define footer(ptr) *((int*)ptr + (getsize(ptr) - 1))

team_t team = {
  /* Team name to be displayed on webpage */
  "<IMG SRC=http://www.microsoft.com/Games/age2/img/knighthead.gif>Team Procrastination!",
  /* First member full name */
  "Filipe Fortes",
  /* First member email address */
  "ffortes",
  /* Second member full name (leave blank if none) */
  "Jamie Schwartz",
  /* Second member email address (blank if none) */
  "jas2"
};


int convert (int);
int find_index (int);

int mm_init (void)
{
  int i;

  /* Get the initial heap area, and store its pointer in top */
  top = mem_sbrk(2*mem_pagesize());

  if (top == NULL) {
    /* For some reason, there was a problem getting the memory */
    return -1;
  }

  /* Initialize all the segregated lists to NULL */
  for (i = 1; i <= (NUM_BINS); i++) {
    freelist(i) = NULL;
  }

  /* Give top the correct value (the beginning of the free memory */
  top = FIRST_MEM_AVAIL;

  /* We made it through everything all right, so go ahead and return 0 */
  return 0;
}

void *mm_malloc (size_t size)
{
  int w_size = convert(size);
  int index = find_index(w_size);
  int* new_ptr, *tmp;

 A:

  /* First, we check if we can find a match on one of the lists */
  while ((freelist(index) == NULL) && (index <= NUM_BINS)) index++;

  /* We found a seg. list for this item */
  if (index <= NUM_BINS) {
    /* Find the first free block that will fit this item */
    new_ptr = freelist(index);

    while ((new_ptr != NULL) && (getsize(new_ptr) < w_size)) {
      new_ptr = next_list(new_ptr);
    }

    /* Check if we ran out of free blocks in this bucket */
    if (!new_ptr) {
      /* Start over, trying the next index */
      index++;
      goto A;
    }

    /* We found a spot on the free list, so let's use it */

    /* First, remove the block from the free list */
    if(prev_list(new_ptr)) {
      if (next_list(new_ptr)) {
	prev_list(next_list(new_ptr)) = prev_list(new_ptr);
	next_list(prev_list(new_ptr)) = next_list(new_ptr);
      }
      else {
	next_list(prev_list(new_ptr)) = NULL;
      }
    }
    else {
      if(next_list(new_ptr)) {
	freelist(index) = next_list(new_ptr);
	prev_list(next_list(new_ptr)) = NULL;
      }
      else {
	freelist(index) = NULL;
      }
    }

    /* Now, we can use the block */
 
    /* First, see if we are going to split this block */
    if ((getsize(new_ptr) - w_size) >= MIN_BLOCK_SIZE) {
      /* It must be at least the size of our miniumum block to split up */
      
      /* Create the header for new block */
      tmp = new_ptr + w_size;

      *tmp = (getsize(new_ptr) - w_size);

      /* Make the footer for this block */
      footer(tmp) = getsize(new_ptr) - w_size;

      /* Add this new block to a free list */
      index = find_index(getsize(tmp));

      if (freelist(index)) {
	/* Place this on the front of the list */
	prev_list(freelist(index)) = tmp;
	next_list(tmp) = freelist(index);
      }
      else
	next_list(tmp) = NULL;

      freelist(index) = tmp;
      prev_list(tmp) = NULL;

    } /* Done splitting the block */
    else {
      /* This is our new working size */
      w_size = getsize(new_ptr);
    }



    /* Now set up the header for this new block */
    *new_ptr = w_size | 3;

    /* ... and now the flag on the next block */
    *next_phys(new_ptr) |= 0x2;

    /* Ready to return to user now */
    return (new_ptr + 1);

  } /* end if (index < NUM_BINS) */

  /* There was no list for our item, so we must allocate from the wild */
  else {
    /* Check if the current heap is big enough for this object */
    while (((int *)dseg_hi - top) < w_size) {
      /* Since there isn't enough room, allocate another page */
      if (mem_sbrk(max(mem_pagesize(), w_size)) == NULL) {
	/* We couldn't get more memory, so we have to exit */
	return NULL;
      }
    } /* End heap allocation loop */

    /* Create the header for the block */
    /* Logical or on the size, since we know that the previous
     * block is taken, and we are also saying that this block
     * is taken */
    *top = w_size | 0x3;

    /* Use for temp storage, add one to top to get the user's memory */
    tmp = top + 1;

    top = top + w_size;

    return tmp;
  } /* end else for wild allocation */
} /* end mm_malloc */

void mm_free (void *ptr)
{
  
  int* f_ptr, *prev_ptr, *next_ptr;
  int index;

  f_ptr = (int*) ptr - 1;

  /* Change our taken setting for our header, to free */
  *f_ptr = *f_ptr & -2;

  /* Set our footer */
  footer(f_ptr) = getsize(f_ptr);

  /* Set the flag on the next item */
  *(f_ptr + getsize(f_ptr)) &= -3;

  /* Check if the previous item is available */
  if (!(prev_taken(f_ptr))) {
    /* Remove this from its free list */

    prev_ptr = prev_phys(f_ptr);

    /* First, remove the block from the free list */
    if(prev_list(prev_ptr)) {
      if (next_list(prev_ptr)) {
	prev_list(next_list(prev_ptr)) = prev_list(prev_ptr);
	next_list(prev_list(prev_ptr)) = next_list(prev_ptr);
      }
      else {
	next_list(prev_list(prev_ptr)) = NULL;
      }
    }
    else {
      index = find_index(getsize(prev_ptr));
      if(next_list(prev_ptr)) {
	freelist(index) = next_list(prev_ptr);
	prev_list(next_list(prev_ptr)) = NULL;
      }
      else {
	freelist(index) = NULL;
      }
    }

    /* Coalesce */
    *prev_ptr = (getsize(prev_ptr) + getsize(f_ptr)) | 0x2;

    footer(prev_ptr) = getsize(prev_ptr);

    /* Update our pointer */
    f_ptr = prev_ptr;

  } /* Previous available */

  /* Check if the lower boundary is the heap */
  if (next_phys(f_ptr) == top) {
    /* We allow the wilderness to grow */
    top = f_ptr;
    return;
  }

  /* Check if the next item is available */ 

  if (!(next_taken(f_ptr))) {
    /* Remove this from its free list */
    next_ptr = next_phys(f_ptr);

    /* First, remove the block from the free list */
    if(prev_list(next_ptr)) {
      if (next_list(next_ptr)) {
	prev_list(next_list(next_ptr)) = prev_list(next_ptr);
	next_list(prev_list(next_ptr)) = next_list(next_ptr);
      }
      else {
	next_list(prev_list(next_ptr)) = NULL;
      }
    }
    else {
      index = find_index(getsize(next_ptr));
      if(next_list(next_ptr)) {
	freelist(index) = next_list(next_ptr);
	prev_list(next_list(next_ptr)) = NULL;
      }
      else {
	freelist(index) = NULL;
      }
    }

    /* Coalesce */
    *f_ptr = (getsize(next_ptr) + getsize(f_ptr)) | 0x2;

    footer(f_ptr) = getsize(f_ptr);    

  } /* Next available */

  /* Add this new (possibly coalesced) item to a free list */
  
  index = find_index(getsize(f_ptr));
  
  if (freelist(index)) {
    /* Place this on the front of the list */
    prev_list(freelist(index)) = f_ptr;

    next_list(f_ptr) = freelist(index);
  }
  else {
    next_list(f_ptr) = NULL;
  }

  prev_list(f_ptr) = NULL;

  freelist(index) = f_ptr;  

}


/* Converts a size in bytes to a size in words */
int convert (int size) {
  /* First, change the units into words */
  size /= WORDSIZE;
  
  /* Now, round up ... */
  size += (size % WORDSIZE) ? 1 : 0;

  /* Adjust for the block's header */
  size++;

  /* Adjust for our minimum block size */
  size = max(MIN_BLOCK_SIZE, size);

  /* Make sure the alignment is to 4 words (a multiple of ALIGN_SIZE) */
  size += (size % ALIGN_SIZE) ? (ALIGN_SIZE - (size % ALIGN_SIZE)) : 0;

  return size;
}

/* Basically our hash function for the free lists
 * Given a size (in words) gives the array index to use to find the
 * apporpriate free list */
int find_index (int x) {

  int m16 = ((1<<16) + ~0) << 16;                         /* groups of 16 */
  int m8 = (((1<<8)  + ~0) << 24) + (((1<<8) + ~0) << 8); /* groups of 8 */
  int m4 = (0xf0<<24) + (0xf0<<16) + (0xf0<<8) + 0xf0;    /* groups of 4 */
  int m2 = (0xcc<<24) + (0xcc<<16) + (0xcc<<8) + 0xcc;    /* groups of 2 */
  int m1 = (0xaa<<24) + (0xaa<<16) + (0xaa<<8) + 0xaa;    /* groups of 1 */
  int result = 236; /* adjust for the fact that 1-245 are for <= 2000 words */
  int upper;
  int mask;

  /* We have a bucket for each multiple of four under 2000 */
  if (x <= 976)
    return x/4;

  /* m16 */
  upper = !!(x & m16);
  result += upper << 4;

  mask = m16 ^ ~((upper<<31)>>31);

  /* m8 */
  upper = !!(x & m8 & mask);
  result += upper << 3;
  mask &= (m8 ^ ~((upper<<31)>>31)); 

  /* m4 */
  upper = !!(x & m4 & mask);
  result += upper << 2;
  mask &= (m4 ^ ~((upper<<31)>>31)); 

  /* m2 */
  upper = !!(x & m2 & mask);
  result += upper << 1;
  mask &= (m2 ^ ~((upper<<31)>>31)); 

  /* m1 */
  upper = !!(x & m1 & mask);
  result += upper;

  return (result > NUM_BINS) ? NUM_BINS : (result);

}
