/* $Id$ */

/*
 *
 *  CS213 - Lab assignment 3
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"


/*
 * pointer to the global header information
 */
#define head ((globheader*)dseg_lo)

team_t team = {
    /* Team name to be displayed on webpage */
    "QED",
    /* First member full name */
    "Andrew Marcus",
    /* First member email address */
    "amarcus",
    /* Second member full name (leave blank if none) */
    "Erik J. Sturcke",
    /* Second member email address (blank if none) */
    "esturcke"
};


/* 
 * Structure used to access the header information of a free blocks
 */
typedef struct {
  /*
   * the number of words in the previous block and this block, including the 
   * header. The sign bit of size is used to indicate if the block is free.
   * (1 used, 0 free)
   */
  int prevsize;
  int size;

  /*
   * pointers to the next and previous free blocks
   */
  void* prev,
      * next;
} freeheader;


/* 
 * Structure used to access the header information of used blocks
 */
typedef struct usedheader{
  /*
   * the number of words in the previous block and this block, including the 
   * header. The sign bit of size is used to indicate if the block is free.
   * (1 used, 0 free)
   */
  int prevsize;
  int size;
} usedheader;


/*
 * Structure to access teh global header information of the first free block
 * and the last block (free or used) in memory. The lastfreeblock stores
 * the last free block in the list of free blocks, and reverse is a flag
 * indicating if the list of free block should be traversed in the forward
 * or reverse direction (an anti-binary hack)
 */
typedef struct {
  void* lastblock;
  void* freeblock;
  void* lastfreeblock;
  int reverse;
} globheader;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    mm_init
 * init first gets a page with mem_sbrk
 * it then sets up the global header and the header for the first free block
 */
int mm_init (void)
{
  int pagesize = mem_pagesize();
  freeheader* freeblock;
  
  /*
   * get a page
   */
  mem_sbrk( pagesize );
  
  /*
   * set the global header pointer to point to the new page (with an offset 
   * of the global header size) and set the lastblock and lastfreeblcok
   * to be that same block.
   */
  head->freeblock = (freeheader*)(dseg_lo + sizeof( globheader ));
  head->lastblock = head->freeblock;
  head->lastfreeblock = head->freeblock;
  head->reverse = 0;
 
  /*
   * initialize free page
   */
  freeblock = head->freeblock;
  freeblock->prevsize = 0;
  freeblock->size = pagesize - sizeof( globheader );
  freeblock->prev = NULL;
  freeblock->next = NULL;
  
  return -1;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    split
 * This function splits up a free block at ptr so there is at least 'size'
 * bytes in the first of the two block (plus room for a used header).
 * If the second block would be too small (to hold an free header), no split
 * is made. The second block is forced to an 8 byte boundry. (If reverse
 * is true than the second block will be big enough to hold size).
 * This funtion returns true (1) if a split and false (0) otherwise
 */
int split( freeheader* ptr, size_t size, int reverse ) {
  /*
   * split only if size available is greater than the size the first block
   * has to be and the second one has to be
   * The space is rounded up to the nearest multiple of 8 to ensure 8 byte 
   * alignment and the size of the header is added
   * The 16 is a hack to improve performance on binary2.rep
   */
  size = ((size-1)/8+1)*8 + sizeof(usedheader);
  
  if( ptr->size  >= size + sizeof(freeheader) ) {
    freeheader* split;
    freeheader* next;

    /*
     * if reverse then the second block will be used to hold the new
     * information and so its size should be size, so the size of the
     * first block should be ptr->size - size
     */
    if ( reverse ) {
      size = ptr->size - size;
    }
    
    /*
     * split is the location of the second block and next is the location
     * of the next free header after ptr
     */
    split = (freeheader*)( (void*)ptr + size );
    next  = (freeheader*)ptr->next;
     
    /*
     * Set the pointer so that:  ptr --> split --> next
     * Check for next == NULL
     */
    ptr->next = split;
    split->next = next;
    split->prev = ptr;
    if( next ) next->prev = split;
   
    /*
     * Set values of the second block, split
     */
    split->prevsize = size;
    split->size = ptr->size - size;
    ptr->size = size;
    
    /*
     * If ptr is the last block, then after the split, split will be the 
     * last block. If it is not the last block, then there must be a block
     * after split that should get a new value for prevsize
     */
    if( head->lastblock == ptr ) {
      head->lastblock = split;
    } else {
      ((freeheader*)((void*)split + split->size))->prevsize = split->size;
    }

    /*
     * If the lastfreeblock was split then the new lastfreeblock should
     * be split
     */
    if( head->lastfreeblock == ptr ) {
      head->lastfreeblock = split;
    }
    return 1;
  } 
  return 0;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    findblock
 * This function goes through the list of free blocks and return the address
 * of the best fit or NULL if none could be found.
 */
void* findblock( size_t size ) {
  freeheader* list;
  freeheader* best = NULL;
  int reverse = head->reverse;
  
  /*
   * find the best fit
   */
  if( reverse ) {
      for(list = head->lastfreeblock;
	  list;
	  list = (freeheader*)list->prev ) {
	if( !best && (list->size >= size + sizeof(usedheader)) ) best = list;
	if( (list->size >= size + sizeof(usedheader)) &&
	    (list->size < best->size) ) best = list;
      }
  } else {
  
    for(list = head->freeblock;
	list;
	list = (freeheader*)list->next ) {
      if( !best && (list->size >= size + sizeof(usedheader)) ) best = list;
      if( (list->size >= size + sizeof(usedheader)) &&
	  (list->size < best->size) ) best = list;
    } 
  }
    
  /*
   * If no fit found return NULL
   */
  if( !best ) return NULL;

  /*
   * call the split function to split the block up, if reverse and a
   * split was made then the location should be best->next
   */
  if(split( best, size, reverse ) && reverse )
     return best->next;
     else return best;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    removefree
 * This function remove a block from the free list and marks that block as
 * used
 */
void removefree( freeheader* ptr, int reverse ) {
  freeheader* next = (freeheader*)ptr->next;
  freeheader* prev = (freeheader*)ptr->prev;

  /*
   * if this block was the last free block, then after it is remove
   * ptr->prev is the last free block
   */
  if( ptr == head->lastfreeblock )
    head->lastfreeblock = ptr->prev;
  
  /*
   * change prev --> ptr --> next
   * to     prev --> next
   * making sure to take care of NULL pointers
   */
  if( prev && next  ) {
    prev->next = next;
    next->prev = prev;
  } else if( prev ) {
    prev->next = NULL;
  } else if( next ) {
    ((globheader*)dseg_lo)->freeblock = next;
    next->prev = NULL;
  } else {
    ((globheader*)dseg_lo)->freeblock = NULL;
  }

  /*
   * Set the sign bit to 1 to mark as used
   */
  ptr->size |= 0x80000000;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    coalesce
 * This function merges a new free block with the list of free blocks
 */
void coalesce( freeheader* free ) {
  freeheader* next;
  freeheader* prev;
  int prevfree = 0;
  int nextfree = 0;

  /*
   * If the previous size is not 0, then it exists, so get a pointer to it and
   * check if it is free, otherwise set prev to NULL
   */
  if( free->prevsize ) {
    prev = (freeheader*)((void*)free - free->prevsize);
    prevfree = prev->size >= 0;
  } else {
    prev = NULL;
  }

  /*
   * If this block is not the last block, then the next one exists, so get
   * a pointer to it, otherwise set next to NULL
   */
  if( head->lastblock != free ) {
    next = (freeheader*)((void*)free + free->size);
    nextfree = next->size >= 0;
  } else {
    next = NULL;
  }

  /*
   * Then coalesce...
   */
  if( nextfree && prevfree ) {
    /*
     * Both previous and the next are free:
     */

    /*
     * First merge prev and free
     */
    prev->size += free->size;
    ((freeheader*)((void*)free + free->size))->prevsize = prev->size;
    
    /*
     * Then merge the new prev with next
     */
    prev->size += next->size;
    
    /*
     * If next was not the last block, then there is a block that need an 
     * updated prevsize...update it
     * otherwise we must set the last block to be this new mergerd block, prev
     */
    if( head->lastblock != next ) {
      ((freeheader*)((void*)next + next->size))->prevsize = prev->size;
    } else {
      head->lastblock = prev;
    }

    /*
     * Now change the pointers from
     *      (prev->prev) --> (prev) --> (prev->next)
     * and  (next->prev) --> (next) --> (next->next)
     * to   (prev->prev) --> (prev+free+next) --> (prev->next)
     * and  (next->prev) --> (next->next)
     * again checking for the NULLs
     */
    if( next->prev && next->next ) {
      ((freeheader*)next->prev)->next = next->next;
      ((freeheader*)next->next)->prev = next->prev;
    } else if( next->prev ) {
      ((freeheader*)next->prev)->next = NULL;
    } else if( next->next ) {
      ((freeheader*)head->freeblock) = next->next;
      ((freeheader*)next->next)->prev = NULL;
    } 
    /*
     * The last case is not needed. The must be either a next or a prev,
     * otherwise the free list is not connected
     */

    /*
     * If next was the lastfreeblock, then the block following next
     * will be the last block after the merge
     */
    if( head->lastfreeblock == next ) head->lastfreeblock = next->prev;
    
  } else if( nextfree ) {
    /*
     * Only the next is free
     */
    
    /*
     * Extend the size
     */
    free->size += next->size;
    
    /*
     * Change the pointers from
     *      (free)
     * and  (next->prev) --> (next) --> (next->next)
     * to   (next->prev) --> (free+next) --> (next->next)
     * and as always checking for those NULL pointers
     */
    free->prev = ((freeheader*)next)->prev;
    free->next = next->next;
    if( next->prev ) {
      ((freeheader*)next->prev)->next = free;
    } else {
      head->freeblock = free;
    }
    if( next->next ) {
      ((freeheader*)next->next)->prev = free;
    }
    
    /*
     * If next was not the last block, then there is a block that need an 
     * updated prevsize...update it
     * otherwise we must set the last block to be this new mergerd block, free
     */
    if( head->lastblock != next ) {
      ((freeheader*)((void*)free + free->size))->prevsize = free->size;
    } else {
      head->lastblock = free;
    }
    
    /*
     * If the next block was the last free block, then the new merged
     * block will be the last free block after the merge
     */
    if( head->lastfreeblock == next ) head->lastfreeblock = free;

  } else if( prevfree ) {
    /*
     * Only the prev is free
     */

    /*
     * Extend the size
     */ 
    prev->size += free->size;
    
    /*
     * No nodes need to be updated
     */

    /*
     * If free was not the last block, then there is a block that need an 
     * updated prevsize...update it
     * otherwise we must set the last block to be this new mergerd block, prev
     */
    if( head->lastblock != free ) {
      ((freeheader*)((void*)free + free->size))->prevsize = prev->size;
    } else {
      head->lastblock = prev;
    }
  } else {
    /*
     * No prev or next
     */
    
    /*
     * Just stick free on the beginning of the list
     */
    free->prev = NULL;
    free->next = (freeheader*)head->freeblock;
    if( (freeheader*)head->freeblock ) {
      ((freeheader*)head->freeblock)->prev = free;
    }
    (freeheader*)head->freeblock = free;
  } 
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    addpage
 * This function adds enough pages to store size
 */
void addpages( int size ) {
  /*
   * pages is rounded up, so that newpage is rounded up to the nearest
   * multiple of the page size
   */
  int pagesize = mem_pagesize();
  int pages = (size - 1) / pagesize + 1;
  int newsize = pagesize * pages;

  /*
   * Try to get a block of newsize
   */
  freeheader* newblock = (freeheader*)mem_sbrk( newsize );

  /*
   * If could not get a block big enough, return
   */
  if( !newblock ) return;

  /*
   * If there is a last block then set the prevsize of the new block to be
   * the size of the last block, otherwise the prevsize is 0
   */
  if( head->lastblock ) {
    newblock->prevsize = ((usedheader*)head->lastblock)->size & 0x7FFFFFFF;
  } else {
    newblock->prevsize = 0;
  }
  
  /*
   * set the size of the newblock and set it to be the last block
   */
  newblock->size = newsize;
  head->lastblock = newblock;

  /*
   * add the new block to the list of free blocks
   */
  coalesce( newblock );
}
  

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    mm_malloc
 * this function returns the best fit free block that is can find of size 
 * 'size", or NULL if it could not find any
 */
void *mm_malloc (size_t size)
{
  void *ptr = NULL;

  /*
   * look for a block of size 'size'
   */
  ptr = findblock( size );

  /*
   * If found, remove it from the free list and set the return pointer to
   * free block found plus the offest for the header
   * If none could be found, try adding some pages and seach again
   */
  if( ptr != NULL ) {
    removefree( (freeheader*)ptr, head->reverse );
    ptr += sizeof( usedheader );
  } else {
    head->reverse = 0;
    addpages( size+sizeof(usedheader) );
    ptr = findblock( size );
    if( ptr != NULL ) {
      removefree( (freeheader*)ptr, head->reverse );
      ptr += sizeof( usedheader );
    } else {
      return NULL;
    }
 
  }

  /*
   * next malloc should search for the free block in the reverse order
   * so that te binary traces to lead to fragmentation
   */
  head->reverse = !(head->reverse);
  
  return ptr;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    mm_free
 * This function frees the memory at ptr and adds the block back to the free
 * list
 */
void mm_free (void *ptr)
{
  freeheader* free = (freeheader*)((void*)ptr - sizeof(usedheader));
  
  /*
   * Mark it as free
   */
  free->size &= 0x7FFFFFFF;

  /*
   * Add to the free list
   */
  coalesce( free );  
}

