/* $Id$ */

/*
 *
 *  CS213 - Lab assignment 3
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

team_t team = {
    /* Team name to be displayed on webpage */
    "Why?",
    /* First member full name */
    "Jesse L Boley",
    /* First member email address */
    "boley@andrew.cmu.edu",
    /* Second member full name (leave blank if none) */
    "",
    /* Second member email address (blank if none) */
    ""
};

/* Lab 3: Implementing a Dynamic Storage Allocator (malloc)
   Jesse Boley

   The method I used for writing malloc is basically segregated
   free lists that represent bins for free blocks.  The bin
   sizes go up by powers of two, but the allocated blocks
   themselves can be any size (minumum of 8 bytes, plus 16 bytes 
   of header/footer).  

   When mm_malloc is called, it searches the
   free list of the appropriate bin for a block that is large 
   enough to hold the requested size (plus allocation data).
   If no blocks are found or the free list is NULL, it then grabs
   the next free block in the next available bin (which is guaranteed
   to be large enough, because the free lists are segregated by size).
   If a block of a larger size can be found, it splits the block, uses
   the appropriate amount, and puts the remaining portion on to the
   correct free list.  If no block can be found, it allocates a few
   chunks of the needed size (in case that size is needed again).  It
   uses one of the chunks and the rest are put on the correct free list.

   When mm_free is called, the blocks to the bottom and top of the block
   being freed are checked to see whether they are allocated or not.
   If either one or both are free, the blocks are combined to form one
   large block which is put in the correct bin.  If a block to the top
   or bottom is used, it is first removed from it's free list.

   That's it.
   */



/* The layout of a block is:
     4 b    4 b           x               4 b    4 b
   |------|------|----------------------|------|------|
   |      |      |                      |      |      | 
   |------|------|----------------------|------|------|
    *next   size       user data          size   *prev

    Block sizes are always multiples of 8, so the first bit of each size 
    part is used as an allocation flag.  Next and Prev pointers are not
    used when the block is allocated.
    */

/* These macros provide easy access to the previous and
   next pointers of a free block.  Note that the Get* macros
   dereference the pointers, so for example, you can't use
   GetBlockPrevP_ to get the previous free block pointer and
   then set the pointer manually, because you would end up
   setting the data *in* the previous free block (which may
   be NULL) */

#define GetBlockPrevP_(p)  (char*)(*(int*)(p+(GetBlockSize_(p))-4))
#define GetBlockNextP_(p)  (char*)(*(int*)(p))

#define SetBlockPrevP_(p, setp) ((*(int*)(p+(GetBlockSize_(p))-4))=(int)setp)
#define SetBlockNextP_(p, setp) ((*(int*)(p)) = (int)setp)


/* The Get/SetBlockSize macros allow access to the size area of
   a block (The set macro sets *both* size parts) 
   */
#define GetBlockSize_(p)   ((int)(*(int*)(p+4)) & -2)
#define SetBlockSize_(p, i)   ((*(int*)(p+4)) = (*(int*)(p-8+(i & -2))) = i)

/* These macros allow setting of the allocation state. */
#define SetBlockUnallocated_(p) (SetBlockSize_(p, (GetBlockSize_(p))))
#define SetBlockAllocated_(p)   (SetBlockSize_(p, (GetBlockSize_(p) | 1)))



/* Prototypes */
char* get_base_ptr(size_t size);
int logbase2(int x);


/************** MM_INIT ********************************/

int mm_init (void)
{
  char *work_chunk;
  int *base_chunk;
  int offset;

  if ( mem_usage() != -1 )
    {
      /* There's already memory on the heap....better reset
	 everything */

      /* Reinialize all base block pointers to NULL */
      for(offset=0; offset<24; offset++)
	{
	  *(int*)(dseg_lo+(offset*4)) = 0;
	}
      
      /* Use the allocated heap... */
      offset = dseg_hi - dseg_lo + 1;
      base_chunk = (int*)get_base_ptr(offset-15);
      work_chunk = dseg_lo + 96;
      *(int*)base_chunk = (int)work_chunk;
      SetBlockSize_(work_chunk, offset);
      SetBlockPrevP_(work_chunk, NULL);
      SetBlockNextP_(work_chunk, NULL);
    }
  else
    {
      /* Initialize the heap with our control data.  The control
	 data consists of 24 pointers that point to the first
	 free block of each size class.  So, there are 24 size classes.
	 */

      mem_sbrk(96);

      /* initialize all base block pointers to NULL */
      for(offset=0; offset<24; offset++)
	{
	  *(int*)(dseg_lo+(offset*4)) = 0;
	}
    }

  /* Shouldn't be any problem... */
  return 0;
}


/******************** MM_MALLOC ***************************/

void *mm_malloc (size_t size)
{
  char *work_chunk, *pos_chunk;
  char *next_chunk;
  int  *base_chunk, *free_chunk, *extra_chunk;
  int temp, i, done;

  /* Make sure size is multiple of 8 */
  if(size % 8 != 0)
    {
      size = ((size + 8) & -8);
    }

  /* Get the correct bin for the size we want to allocate */
  base_chunk = (int*)get_base_ptr(size);
  size += 16;

  if((char*)(*(int*)base_chunk) != NULL)
    {
      /* Cycle through all the blocks to see if there's one that's
	 large enough */
      work_chunk = (char*)(*(int*)base_chunk);
      while(work_chunk && (GetBlockSize_(work_chunk) < size) &&
	    (work_chunk < dseg_hi))
	{
	  work_chunk = GetBlockNextP_(work_chunk);
	}
      if(work_chunk != NULL)
	{
	  /* We found a block.  Splice it from the free list. 
	     Split it if necessary, then finish setting it up. */
	  pos_chunk = GetBlockPrevP_(work_chunk);
	  next_chunk = GetBlockNextP_(work_chunk);
	  if(pos_chunk != NULL)
	    {
	      SetBlockNextP_(pos_chunk, next_chunk);
	    }
	  else
	    {
	      *(int*)base_chunk = (int)next_chunk;
	    }
	  if(next_chunk)
	    SetBlockPrevP_(next_chunk, pos_chunk);

	  SetBlockPrevP_(work_chunk, NULL);
	  SetBlockNextP_(work_chunk, NULL);
	  
	  temp = GetBlockSize_(work_chunk);
	  /* Minimum size that the user can allocate is 8 bytes. Plus
	     16 bytes of control data, means 24 bytes is the smallest
	     chunk allocatable.  So don't allow chunks smaller than
	     24 bytes to be created */
	  if(temp - size >= 24)
	    {
	      pos_chunk = work_chunk + size;
	      SetBlockSize_(pos_chunk, (temp-size));
	      SetBlockSize_(work_chunk, size);
	      
	      extra_chunk = (int*)get_base_ptr(temp-size-15);
	      next_chunk = (char*)(*(int*)extra_chunk);
	      *(int*)extra_chunk = (int)pos_chunk;
	      SetBlockNextP_(pos_chunk, next_chunk);
	      SetBlockPrevP_(pos_chunk, NULL);
	      if(next_chunk != NULL)
		SetBlockPrevP_(next_chunk, pos_chunk);
	    }
	  
	  /* Set this to signal that we have allocated a block */
	  done = 1;
	}
      else
	/* otherwise, we haven't found a large enough block. so we
	   have to search a higher size class */
	done = 0;
    }
  else
    done = 0;

  if(!done)
    {
      /* Search all size classes (larger than the current one) 
	 until the first free chunk is found */
      free_chunk = base_chunk;
      work_chunk = NULL;
      while((char*)free_chunk < dseg_lo + 96)
	{
	  free_chunk++;
	  if((char*)(*(int*)free_chunk) != NULL)
	    {
	      work_chunk = (char*)(*(int*)free_chunk);
	      break;
	    }
	}
      if(work_chunk != NULL)
	{
	  /* Large enough chunk has been found.  Splice it out of it's
	     free list.  Split it if necessary, then finish setting
	     it up */
	  pos_chunk = GetBlockNextP_(work_chunk);
	  *(int*)free_chunk = (int)pos_chunk;
	  if(pos_chunk != NULL)
	    SetBlockPrevP_(pos_chunk, NULL);

	  SetBlockPrevP_(work_chunk, NULL);
	  SetBlockNextP_(work_chunk, NULL);

	  temp = GetBlockSize_(work_chunk);
	  if(temp - size >= 24)
	    {
	      pos_chunk = work_chunk + size;
	      SetBlockSize_(pos_chunk, (temp-size));
	      SetBlockSize_(work_chunk, size);
	      
	      extra_chunk = (int*)get_base_ptr(temp-size-15);
	      next_chunk = (char*)(*(int*)extra_chunk);
	      *(int*)extra_chunk = (int)pos_chunk;
	      SetBlockNextP_(pos_chunk, next_chunk);
	      SetBlockPrevP_(pos_chunk, NULL);
	      if(next_chunk != NULL)
		SetBlockPrevP_(next_chunk, pos_chunk);
	    }
	}
      else
	{
	  /* Couldn't find any suitable block.  Need to allocate more
	     using mem_sbrk.  If the needed size is small, allocate
	     multiple copies (in case they're needed later) */
	  if(size < 2048)
	    {
	      temp = 4*size;
	    }
	  else if(size < 8192)
	    {
	      temp = 2*size;
	    }
	  else
	    {
	      temp = size;
	    }
	  work_chunk = mem_sbrk(temp);
	  if(work_chunk != NULL)
	    {
	      *(int*)base_chunk = (int)work_chunk;
	    }
	  else
	    {
	      /* Bad: ran out of heap... */
	      return NULL;
	    }

	  SetBlockSize_(work_chunk, size);
	  SetBlockPrevP_(work_chunk, NULL);
	  next_chunk = work_chunk + size;
		 
	  for(i=0; i<(temp/size)-1; i++)
	    {
	      SetBlockSize_(next_chunk, size);
	      SetBlockNextP_(work_chunk, next_chunk);
	      SetBlockPrevP_(next_chunk, work_chunk);
	      work_chunk = next_chunk;
	      next_chunk += size;
	    }
	  SetBlockNextP_(work_chunk, NULL);
	      
	  work_chunk = (char*)(*(int*)base_chunk);
	  next_chunk = GetBlockNextP_(work_chunk);
	  *(int*)base_chunk = (int)next_chunk;
	  if(next_chunk != NULL)
	    SetBlockPrevP_(next_chunk, NULL);

	  SetBlockPrevP_(work_chunk, NULL);
	  SetBlockNextP_(work_chunk, NULL);

	}
    }

  /* Set the allocation state and return the user data portion
     of the block */
  SetBlockAllocated_(work_chunk);
  return work_chunk+8;
}


/******************* MM_FREE *****************************/

void mm_free (void *ptr)
{
  char *work_chunk, *next_chunk = NULL, *next2_chunk;
  char *prev_chunk = NULL, *prev2_chunk;
  int *baseP_chunk = NULL, *baseN_chunk = NULL, *base_chunk;
  int size, full_size;
  int prev_size = 0, next_size = 0;

  work_chunk = ((char*)ptr)-8;
  
  /* get the base pointer */
  size = GetBlockSize_(work_chunk);
  /* get the sizes of the blocks above and below the current block */
  prev_size = *(int*)(work_chunk-8);
  next_size = *(int*)(work_chunk+size+4);

  /* If the blocks above and below exist and are unallocated, then
     get pointers to them, and pointers to their bin locations */
  if(prev_size > 0 && ((prev_size & 1) == 0))
    {
      if(next_size > 0 && ((next_size & 1) == 0))
	{
	  prev_chunk = work_chunk - prev_size;
	  next_chunk = work_chunk + size;
	  
	  baseP_chunk = (int*)get_base_ptr(prev_size-15);
	  baseN_chunk = (int*)get_base_ptr(next_size-15);
	}
      else
	{
	  prev_chunk = work_chunk - prev_size;
	  next_chunk = NULL;

	  baseP_chunk = (int*)get_base_ptr(prev_size-15);
	  baseN_chunk = NULL;
	  next_size = 0;
	}
    }
  else
    {
      if(next_size > 0 && ((next_size & 1) == 0))
	{
	  prev_chunk = NULL;
	  next_chunk = work_chunk + size;

	  baseP_chunk = NULL;
	  baseN_chunk = (int*)get_base_ptr(next_size-15);
	  prev_size = 0;
	}
      else
	{
	  next_size = prev_size = 0;
	}
    }
  
  /* If the previous block is available, splice it out */
  if(prev_size)
    {
      next2_chunk = GetBlockNextP_(prev_chunk);
      prev2_chunk = GetBlockPrevP_(prev_chunk);
      if(prev2_chunk != NULL)
	SetBlockNextP_(prev2_chunk, next2_chunk);
      else
	*(int*)baseP_chunk = (int)next2_chunk;
      if(next2_chunk)
	SetBlockPrevP_(next2_chunk, prev2_chunk);
      SetBlockPrevP_(prev_chunk, NULL);
      SetBlockNextP_(prev_chunk, NULL);
      *(int*)(prev_chunk+4) = 0;
      *(int*)(prev_chunk+prev_size-8) = 0;
    }

  /* If the next block is available, splice it out */
  if(next_size)
    {
      next2_chunk = GetBlockNextP_(next_chunk);
      prev2_chunk = GetBlockPrevP_(next_chunk);
      if(prev2_chunk != NULL)
	SetBlockNextP_(prev2_chunk, next2_chunk);
      else
	*(int*)baseN_chunk = (int)next2_chunk;
      if(next2_chunk)
	SetBlockPrevP_(next2_chunk, prev2_chunk);
      SetBlockPrevP_(next_chunk, NULL);
      SetBlockNextP_(next_chunk, NULL);
      *(int*)(next_chunk+4) = 0;
      *(int*)(next_chunk+next_size-8) = 0;
    }

  /* Generate a new free block that includes the free blocks
     above and below (if available) */
  SetBlockPrevP_(work_chunk, NULL);
  SetBlockNextP_(work_chunk, NULL);
  *(int*)(work_chunk+4) = 0;
  *(int*)(work_chunk+size-8) = 0;

  full_size = size + prev_size + next_size;
  work_chunk -= prev_size;
  base_chunk = (int*)get_base_ptr(full_size-15);
  
  next2_chunk = (char*)(*(int*)base_chunk);
  *(int*)base_chunk = (int)work_chunk;
  SetBlockSize_(work_chunk, full_size);
  SetBlockPrevP_(work_chunk, NULL);
  SetBlockNextP_(work_chunk, next2_chunk);
  if(next2_chunk)
    SetBlockPrevP_(next2_chunk, work_chunk);

  /* Set allocation state */
  SetBlockUnallocated_(work_chunk);
}


/*********** UTILITY FUNCTIONS ***************************/

/* This function returns the bin pointer for a given size.  It
   assumes that this size doesn't include the header/footer size */
char* get_base_ptr(size_t size)
{
  int l = logbase2(size+15);
  return (char*)(dseg_lo+((l)-4)*4);
}

/* Calculates log2(x).  Used in determining what bin a block falls in */
int logbase2(int x)
  {
    int m16 = ((1<<16) + ~0) << 16;                         /* groups of 16 */
    int m8 = (((1<<8)  + ~0) << 24) + (((1<<8) + ~0) << 8); /* groups of 8 */
    int m4 = (0xf0<<24) + (0xf0<<16) + (0xf0<<8) + 0xf0;    /* groups of 4 */
    int m2 = (0xcc<<24) + (0xcc<<16) + (0xcc<<8) + 0xcc;    /* groups of 2 */
    int m1 = (0xaa<<24) + (0xaa<<16) + (0xaa<<8) + 0xaa;    /* groups of 1 */
    int result = 0;
    int upper;
    int mask;

    /* m16 */
    upper = !!(x & m16);
    result += upper << 4;
    mask = m16 ^ ~((upper<<31)>>31);

    /* m8 */
    upper = !!(x & m8 & mask);
    result += upper << 3;
    mask &= (m8 ^ ~((upper<<31)>>31)); 

    /* m4 */
    upper = !!(x & m4 & mask);
    result += upper << 2;
    mask &= (m4 ^ ~((upper<<31)>>31)); 

    /* m2 */
    upper = !!(x & m2 & mask);
    result += upper << 1;
    mask &= (m2 ^ ~((upper<<31)>>31)); 

    /* m1 */
    upper = !!(x & m1 & mask);
    result += upper;

    return result;
}
