/* $Id$ */

/*
 *
 *  CS213 - Lab assignment 3
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

team_t team = {
    /* Team name to be displayed on webpage */
    "Sketch",
    /* First member full name */
    "Seth Porter",
    /* First member email address */
    "symmetry@pobox.com",
    /* Second member full name (leave blank if none) */
    "",
    /* Second member email address (blank if none) */
    ""
};

/*
  Implementation notes:

  I use a single explicit, doubly-linked free list.
  The word immediately before and after a given chunk
  is the size, with the LSB 1 for allocated, and 0 for
  free.
  Coalescing occurs immediately upon freeing a chunk.
  For the sake of simplicity, malloc uses a first-fit
  algorithm.

  Various sections of code are #ifdef'd on SLOW_DEBUG,
  mostly functions that are defined as macros for
  performance. These were used strictly for testing
  and to let the compiler handle more of the type-checking.
  */


typedef struct free_block {
    struct free_block *last;
    struct free_block *next;
} free_block;

typedef unsigned int tag;

tag make_tag(int allocated, int size)
{
    return (size & ~1) | allocated;
}
/* Must be this much left over (plus control overhead) to split */
#define SPLIT_SIZE 16
#define INIT_OVERHEAD sizeof(void *)
#define BLOCK_OVERHEAD (sizeof(void *) * 2 + sizeof(unsigned int) * 2)
/* Need BLOCK_OVERHEAD bytes to make a free block */

#ifdef SLOW_DEBUG
size_t RND_8_UP(size_t x)
{
    return ((x+7)/8)*8;
}

tag *HEADER_OF(void *x)
{
    return ((unsigned int *)x) - 1;
}

void SET_HEADER(void *x, tag t)
{
    *HEADER_OF(x) = t;
}

tag GET_HEADER(void *x)
{
    return *HEADER_OF(x);
}

void SET_FOOTER(void *x, size_t size, tag t)
{
    *(tag *)((void *)(x) + (size)) = t;
}

size_t GET_SIZE(size_t x)
{
    return x & ~1;
}

unsigned int *footer_addr(void *p, size_t size)
{
    return (unsigned int *)(p + size);
}


#else

#define RND_8_UP(x)     ((((x)+7)/8)*8)
#define HEADER_OF(x)    ((unsigned int *)(x) - 1)
#define SET_HEADER(x, t) (*HEADER_OF(x) = (t))
#define GET_HEADER(x)   (*HEADER_OF(x))
#define SET_FOOTER(x, size, t) (*(tag *)((void *)(x) + (size)) = (t))
#define GET_SIZE(x)     ((x) & ~1)
#define footer_addr(x, size) ( (unsigned int *)((void *)(x)+(size)))

#endif

void splice_out(free_block *p)
{
    if(p->last != NULL)
	p->last->next = p->next;
    else
	*(free_block **)dseg_lo = p->next;
    if(p->next != NULL)
	p->next->last = p->last;
}

/*
  Performs constant-time coalescing as discussed in
  class. When the block is against the top or
  bottom of the heap, it pretends that there is an
  allocated block on the appropriate side.
  */

free_block * mm_coalesce(free_block * just_freed)
{
    unsigned int *prior_footer = (unsigned int *)just_freed-2;
    unsigned int *next_header = footer_addr(just_freed, GET_SIZE(GET_HEADER(just_freed))) + 1;
    free_block *next = (free_block *)(next_header+1);
    free_block *prior = (free_block *)(((void *)prior_footer)-GET_SIZE(*prior_footer));
    
    /* To unify handling, pretend that the block
       before the freed one is allocated, if the freed block
       is first in the heap */
    int prior_alloc =
	(just_freed == (free_block *)(RND_8_UP(
					       (unsigned int)dseg_lo+
					       sizeof(tag)+
					       INIT_OVERHEAD))) ||
	(*prior_footer & 1);
    /* Likewise, if last block, pretend next block is allocated */
    int next_alloc =
	((char *)(next_header + 1) == dseg_hi+1) ||
	(*next_header & 1);

    /* case 1 -- nothing to do */
    if(prior_alloc && next_alloc)
	return just_freed;
    /* case 2 */
    if(prior_alloc && !next_alloc)
    {
	int new_size = *next_header + GET_SIZE(GET_HEADER(just_freed))+2*sizeof(tag);
	splice_out(next);
	SET_HEADER(just_freed, new_size);
	SET_FOOTER(just_freed, new_size, new_size);
	return just_freed;
    }
    /* case 3 */
    if(!prior_alloc && next_alloc)
    {
	int new_size = *prior_footer+GET_SIZE(GET_HEADER(just_freed))+2*sizeof(tag);
	splice_out(just_freed);
	SET_HEADER(prior, new_size);
	SET_FOOTER(prior, new_size, new_size);
	return prior;
    }
    /* case 4 */
    if(!prior_alloc && !next_alloc)
    {
	int new_size = *prior_footer+GET_SIZE(GET_HEADER(just_freed))+*next_header+4*sizeof(tag);
	splice_out(just_freed);
	splice_out(next);
	SET_HEADER(prior, new_size);
	SET_FOOTER(prior, new_size, new_size);
	return prior;
    }
    return NULL;
}


/*
  Allocate more RAM and configure it as a free block.
  It makes the whole allocation a single free block, with
  room at the end for the header of the next block (when
  we resize again). The return value is filtered through
  coalesce, in case the last block in the old heap was free.
  
  increment: How much more RAM to allocate
  offset: How many bytes to allow at beginning of new allocation
    before the free_block (used to set up initial state, when
    the overhead comes first).
  reset: Ignores the previous value of the free_list (for re-init)
  return value:
    if sbrk fails, returns 0. Else returns the address of the
    constructed free block.
    */
void * mm_resize(size_t increment, size_t offset, int reset)
{
    void *ret_val;
    ret_val = mem_sbrk(increment);
    if(ret_val == NULL)
       return ret_val;
    else
    {
	free_block *p = (free_block *)(RND_8_UP(
					       (unsigned int)ret_val + offset));
	size_t block_size;

	if(reset)
	    *(free_block **)dseg_lo = NULL;

	/* 2 * sizeof(tag) leaves room for the header
	   for the next page (when allocated) to squeeze
	   onto this page.
	   It's assumed that offset is a multiple of 4
	   but not 8 -- this means that there's room after
	   offset for the header word of the first block.
	   */
	block_size = increment - (2*sizeof(tag)+RND_8_UP(offset));
	SET_HEADER(p, make_tag(0, block_size));
	SET_FOOTER(p, block_size, make_tag(0, block_size));
	
	p->last = NULL;
	p->next = *(free_block **)dseg_lo;
	*(free_block **)dseg_lo = p;
	ret_val = mm_coalesce(p);
    }
    return ret_val;
}

/*
  Delegates the work to mm_resize
  */

int mm_init (void)
{
    if(mm_resize(mem_pagesize(), INIT_OVERHEAD, 1) == NULL)
	return -1;
    else
	return 0;
}

/*
  Rounds the block size up to the nearest 8 bytes,
  and selects the first-fit block from the free list
  (calling mm_resize if needed). If the block is
  sufficiently larger than the size needed, splits
  it and adds the new free block to the list.
  */
void *mm_malloc (size_t size)
{
    size_t real_size = RND_8_UP(size);
    free_block *free_list = *(free_block **)dseg_lo;
    free_block *p = free_list;
    unsigned int page_size = mem_pagesize();
    
    if(p)
    {
	while((p->next !=  NULL) &&
	      (GET_SIZE(GET_HEADER(p)) < real_size))
	    p = p->next;
    }
    
    if((p==NULL) | (p && (GET_SIZE(GET_HEADER(p))) < real_size))
    {
	p = mm_resize(((real_size + page_size - 1) / page_size) * page_size, 0, 0);
	if(p == NULL)
	{
#ifdef SLOW_DEBUG
	    printf("malloc returning NULL!\n");
#endif
	    return NULL;
	}
    }
    
    /* either way, p is now large enough for new allocation */

    splice_out(p);

    /* SPLIT_SIZE is to ensure we don't create 0-length free
       blocks out of this. */
    if(GET_SIZE(GET_HEADER(p)) > real_size+BLOCK_OVERHEAD+SPLIT_SIZE)
	/* need to split */
    {
	int old_size = GET_SIZE(GET_HEADER(p));
	int new_block_size = old_size - (real_size + 2 * sizeof(tag));
	
	free_block *new_block = (free_block *)((void *)p + real_size + 2 * sizeof(tag));
	
	SET_HEADER(p, make_tag(1, real_size));
	SET_FOOTER(p, real_size, make_tag(1, real_size));

	SET_HEADER(new_block, make_tag(0, new_block_size));
	SET_FOOTER(new_block, new_block_size, make_tag(0, new_block_size));
	
		
	/* splice new block back into free list */
	new_block->next = p->next;
	if(new_block->next != NULL)
	    new_block->next->last = new_block;
	new_block->last = p->last;
	if(new_block->last != NULL)
	    new_block->last->next = new_block;
	else
	    *(free_block **)dseg_lo = new_block;
    }
    else
    {
#ifdef SLOW_DEBUG
	assert(!(GET_HEADER(p) & 1));
#endif
	/* just change it to allocated */
	SET_HEADER(p, GET_HEADER(p) | 1);
	SET_FOOTER(p, GET_SIZE(GET_HEADER(p)), GET_HEADER(p) | 1);
    }
#if 0
    printf("mm_malloc returning %p\n", p);
#endif
    return p;
}

/*
  Mark the block as free and insert it into the free list.
  If possible, coalesce it with adjoining free blocks.
  */
void mm_free (void *ptr)
{
    tag h;
    free_block *p;

#ifdef SLOW_DEBUG
    assert(ptr != NULL);
    assert(GET_HEADER(ptr) & 1);
    
    {
	
	free_block *temp = *(free_block **)dseg_lo;
	while(temp != NULL)
	{
	    assert(temp != ptr);
	    temp = temp->next;
	}
	
    }
    
#endif
    h = GET_HEADER(ptr);
    p = (free_block *)ptr;        

    SET_HEADER(ptr, h & ~1);
    SET_FOOTER(ptr, GET_SIZE(h), h & ~1);

    p -> last = NULL;
    p -> next = *(free_block **)dseg_lo;
    if(p->next != NULL)
    {
#ifdef SLOW_DEBUG
	assert(p->next->last == NULL);
#endif
	p->next->last = p;
    }
    
    *(free_block **)dseg_lo = p;
    mm_coalesce(p);
}


