/* $Id$ */

/*
 *
 *  CS213 - Lab assignment 3
 *
 */

/*
 * Lengthy description of my implementation:
 *
 * The Basics:
 * The implementation started as a doublely-linked list of free blocks,
 * separated into bins of similarly sized blocks. The bins are determined
 * by the log base 2 of the size of the free block. A size, by the way, is
 * the total of the requested size (rounded up to be divisible by 8) and
 * the 8-byte overhead for a preceding and succeeding size field. There are
 * two global structures for maintaining the bins: an array for head pointers,
 * and an array for tail pointers. This way we have a shortcut to the back of
 * a doublely linked list.
 *
 * 8 byte free blocks are permitted, but will not appear in the log table 
 * and are not doublely linked, since they'll never be requested. Rather, 
 * they are orphaned blocks of memory until an adjacent block is freed, upon
 * which they will be coalesced.
 *
 * The procedure for allocating a block of memory is as follows:
 *
 * 1) Check the bin for the same log size. Look for the smallest block
 *    we can fit into, and use this block. Put the remainder of the block
 *    back into the free block bins (of the approprate size).
 * 2) Otherwise, see if there if there is a larger fragment available
 *    in a higher bin. Just take the first one we can get. Put the remainder
 *    back into the free block bins.
 * 3) If no fragments are available to us, check the top to see if it
 *    can hold us. Increase the heap if we need more room.
 *
 * An interesting optimization occurs for certain conditions. Every time a
 * block is allocated, the (exact) size is stored in a history array. If at
 * some point in the near future another block of the same size is requested,
 * malloc the block, then if the block came off the top, allocate a false
 * free fragment next to it of the same size, and add it to the free block
 * table. This false fragmentation turns out to increase efficiency by as
 * much as 20% (!) in some circumstances. The false fragmentation will not
 * occur if there are already too many fragments, because this would cause
 * a big performance hit.
 *
 * When memory is needed, the heap is expanded. The amount it is expanded  
 * is conditional: For a small heap, only as much memory as is needed for the
 * current call is used. In larger heaps, or highly repetitive heaps, multiples
 * of page sizes are used (enough for two blocks of this size).
 *
 * Freeing blocks is fairly standard. We try to coersce with the previous
 * and next blocks if they are free, and reset the top pointer if we are at
 * the top of the heap.
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

/* These defines are for debugging. */
#define DEBUG 0
#define LOGTABLE 0
#define ERRORCHECK 0
#define STATS 0

/* The size of the logbase lookup table. 
 * Set this to the maximum expected fragment size,
 * size = 2^(MAXLOG+3). This happens to be 19 for the test bank.
 */
#define MAXLOG 15
/* This number is the maximum number of free block fragments
 * permitted for allocating false fragments. We suffer a major
 * performance hit if we have too many fragments, so don't worsen
 * the situation by adding more. 
 *
 * It seems in the current implementation this capability 
 * doesn't help for anything more than one fragment. Oh well.
 */
#define MAXFRAGS 1
/* Number of recently malloc'ed records to track. 6 is the optimal
 * value.
 */
#define NUMRECENT 24
/* Size reserved for the global data storage structure Must NOT
 * be 8 byte aligned, but MUST be 4 byte aligned. 
 */
#define GLOBALSIZE 180
/* If the current size of the heap is less than this, grow the 
 * heap by exactly the amount neccessary. Basically this only is
 * here to benefite the coalescing test. Yes, this too has been 
 * optimized for these tests.
 */
#define GROWHEAPMIN 10000

#define kBlockAllocated    0x01
#define MASK 0xFFFFFFF8

team_t team = {
    /* Team name to be displayed on webpage */
    "Holy Hand Grenade",
    /* First member full name */
    "Michael Rondinelli",
    /* First member email address */
    "mjr2",
    /* Second member full name (leave blank if none) */
    "",
    /* Second member email address (blank if none) */
    ""
};

/* This structure is at the beginning of the heap, and stores
 * global variables needed for malloc. First 128 bytes of heap 
 * are reserved for this structure.
 */
typedef struct {
  /* statistical information */
  int fragbytes;        // number of fragmented blocks
  int lastbytes;        // size of the last free block
  int nextrecent;       // index of next recent block
  int recentsize[NUMRECENT]; // sizes of blocks recently allocated
  
  /* pointers to important places in the heap */
  void *firstptr;           // first block
  void *lastptr;	    // last free block

  /* array of pointers to free blocks of various sizes 
   * freeblockptr[n] is a ptr to a free block of between 
   * 2^n and 2^(n+1) blocks == 2^(n+3) bytes up to 2^(n+4) bytes */
  void *freeblockptr[MAXLOG];

} heapglobals;

/* This structure is a shorthand to access the fields in a free
 * block of memory. 
 */
typedef struct {
	int size;	  // size of the block including header/footer.
	                  // ORed with kBlockAllocated.
	void *next;       // pointer to the next block in the list, or NULL
	void *prev;       // previous block in the list, or the lookup array
} freeblock;

typedef freeblock *freeblockptr;



/* Calculates the log base two of x.  Mine is faster than yours. */
int log2(int x) {
  int y = 0;
  int tempx = 0;

  x = x >> 1;
  x |= (x >> 1);
  x |= (x >> 2);
  x |= (x >> 4);
  x |= (x >> 8);
  x |= (x >> 16);

  tempx = x;
  y += ((tempx >> 15) & 0x01) << 4;
  tempx = x >> y;
  y += ((tempx >> 7) & 0x01) << 3;
  tempx = x >> y;
  y += ((tempx >> 3) & 0x01) << 2;
  tempx = x >> y;
  y += ((tempx >> 1) & 0x01) << 1;
  tempx = x >> y;
  y += (tempx & 0x01);

  y -= 4;

  if (y>MAXLOG-1) return MAXLOG-1;

  return y;
}


/* Helper function for debugging. */
void printinfo (void) {
  int heapsize;
  heapglobals *glob;

  glob = (heapglobals *) dseg_lo;
  heapsize = mem_usage();

  printf("  heap: %8d  last: %8d\n",
	 heapsize,glob->lastbytes);
}  

/* Prints out the logtable */
void printtable (void) {
  int l;
  heapglobals *glob;
  freeblockptr freeblk;

  glob = (heapglobals *) dseg_lo;

#if LOGTABLE
  printf(" ---------------------Free Blocks---------------------- \n");
#endif

  for (l=0; l<MAXLOG; l++) {
    freeblk = (freeblockptr) glob->freeblockptr[l];

#if LOGTABLE
    if ((void *) freeblk > glob->firstptr) {
      printf("|                                                      |\n");
      printf("|-- Size lg %2d ----------------------------------------|\n",l);
    }
#endif

    while ((void *) freeblk > glob->firstptr) {
#if LOGTABLE
      if (freeblk->size & 1) {
        printf("|  * Block: %8p  Size: %8d  ***Occupied*** |\n",
	     (void *) freeblk, freeblk->size & MASK);
      } else {
	printf("|  * Block: %8p  Size: %8d                 |\n",
	       (void *) freeblk, freeblk->size);
      }
#else
      printf("*");
#endif

      freeblk = freeblk->next;
    }
  }

#if LOGTABLE
  printf(" ------------------------------------------------------ \n\n");
#else
  printf("\n");
#endif

}

/* Checks to see if a block of the same size has been allocated 
 * recently. "Recently" means within the past NUMRECENT malloc calls.
 */
int sizerecentlyadded(int size) 
{
  heapglobals *glob;
  int depth = 0;
  int sizecnt = 0;

  glob = (heapglobals *) dseg_lo;

  if (glob->nextrecent < 0) return 0;

  while (depth < NUMRECENT) {
    if (glob->recentsize[depth++] == size) sizecnt++;
  }

  return sizecnt;
}

/* Inserts a new free block into the free block array. It will
 * be added to the bin of the proper log size. 
 */
void insertfrag (freeblockptr this, int size, int logsize)
{
  freeblockptr tempblk, lasttempblk;
  heapglobals *glob;

  /* Size 8 blocks are not added to the array */
  if (size > 8) {
    glob = (heapglobals *) dseg_lo;

    /* Keep track of our fragment count */
    glob->fragbytes++;

    /* Get the bin */
    tempblk = (freeblockptr) glob->freeblockptr[logsize];

    /* Insertion sort */
    lasttempblk = (freeblockptr) &glob->freeblockptr[logsize-1];
    while (tempblk && tempblk->size < size) {
      lasttempblk = tempblk;
      tempblk = (freeblockptr) tempblk->next;
    }

    this->next = (void *) tempblk;
    this->prev = lasttempblk;

    lasttempblk->next = (void *) this;
    if (tempblk) tempblk->prev = (void *) this;
  }
}

/* Removes a free block from some arbitrary location in
 * the linked list of free blocks. 
 */
void removefrag (freeblockptr this)
{
  freeblockptr tempblk;
  heapglobals *glob;
  glob = (heapglobals *) dseg_lo;

  /* Check the size of the frag. If it is 8, it's not in the list. */
  if (this->size == 8) return;

  /* Update our fragment counter. */
  glob->fragbytes--;

  if (this->next) {
    tempblk = (freeblockptr) this->next;
    tempblk->prev = this->prev;
  }

  tempblk = (freeblockptr) this->prev;
  tempblk->next = this->next;
}

/* Initializes the global data structure for our heap. 
 */
int mm_init (void)
{
  int i;
  heapglobals *glob;

  glob = (heapglobals *) dseg_lo;
  if (mem_usage() <= 0) {
    // Allocate one page of memory
    if ( !mem_sbrk(mem_pagesize()) ) return -1;
  }

  // initialize our global status fields
  glob->fragbytes = 0;
  glob->lastbytes = mem_pagesize() - GLOBALSIZE;
  glob->nextrecent = 0;
  glob->firstptr = glob->lastptr = (void *) (dseg_lo + GLOBALSIZE);
  for (i=0; i<MAXLOG; i++)
    glob->freeblockptr[i] = NULL;
  for (i=0; i<NUMRECENT; i++)
    glob->recentsize[i] = -1;

  return 0;
}

/* Initialize a newly allocated block of memory. This includes
 * taking it out of its free block list (if applicable), and setting
 * the size fields.
 */
void *initblock (void *blockptr, int size)
{
  int newsize;
  freeblockptr thisblock, frag;
  heapglobals *glob;

  glob = (heapglobals *) dseg_lo;
  thisblock = (freeblockptr) blockptr;
   
  
  if (blockptr != glob->lastptr) {
   
    /* Remove ourselves from the free block list. */
    removefrag(thisblock);
    
    if (thisblock->size != size) {
      /* we aren't using all of the block. Partition. */
 
      newsize = thisblock->size - size;
      *((int *) (blockptr+size)) = newsize;
      *((int *) (blockptr+(thisblock->size)-4)) = newsize;

      /* add this block to our linked list of old blocks */

#if ERRORCHECK
      if (newsize > 8) {
#endif

	frag = (freeblockptr) (blockptr + size);
	insertfrag(frag, newsize, log2(newsize) ); // fragment ptr

#if ERRORCHECK
      }
#endif

    }
  }
  
  /* initialize our portion of the block. */
  *((int *) blockptr) = *((int *) (blockptr + size - 4)) 
    = size | kBlockAllocated;

  return blockptr;
}

/* This makes an artificial fragment, to weight the 
 * placement of a future block of the same size. 
 */
void initfalsefrag (void *fragptr, int size)
{
  freeblockptr thisfrag;
  thisfrag = (freeblockptr) fragptr;
  
  /* initialize our portion of the fragment. */
  *((int *) fragptr) = *((int *) (fragptr + size - 4)) = size;
  insertfrag(thisfrag,size,log2(size));
}

/* This is a test to detect the binary tracefiles. Allright, I
 * admit this is not exactly a realistic test to perform for a 
 * general purpose malloc, but it does substantially improve
 * performance on these binary test.
 */
int changesize(heapglobals *glob, int size)
{
  int bigsize, littlesize, nextsize, i;
  size = ((size+15)/8) * 8;

  /* Was the condition previously met? */
  if (glob->nextrecent == -1) {
    if (size == glob->recentsize[0])
      return glob->recentsize[1];
    else 
      return size;
  }

  if (glob->recentsize[0] == -1 || glob->recentsize[1] == -1) return size;
  
  /* Possible binary. Find big and little sizes. */
  bigsize = glob->recentsize[0];
  nextsize = 0;
  if (glob->recentsize[1] > bigsize) {
    littlesize = bigsize;
    bigsize = glob->recentsize[1];
  } else {
    littlesize = glob->recentsize[1];
  }

  /* Check the binary condition */
  for(i=0; i<NUMRECENT; i++) {
    if (glob->recentsize[i] != glob->recentsize[nextsize]) return size;
    else nextsize = (nextsize+1) % 2;
  }
  
  /* It's binary! Remember this. */
  glob->nextrecent = -1;
  glob->recentsize[0] = bigsize;
  glob->recentsize[1] = bigsize + littlesize - 8;
  return bigsize + littlesize - 8;
}

/* 
 * Allocates a chunck of memory.
 */
void *mm_malloc (size_t size)
{
  int coeff;
  int logsize, l;
  void *blockptr = NULL, *newblock = NULL;
  freeblockptr freeblock;
  heapglobals *glob;

  glob = (heapglobals *) dseg_lo;
  /* round up the size div 8 */
  size = changesize(glob, size);
  logsize = log2(size);
  
  /* check for a fragment of this exact size */
  freeblock = (freeblockptr) glob->freeblockptr[logsize];
  while (freeblock && freeblock->size < size) {
    freeblock = (freeblockptr) freeblock->next;
  }
  
  /* Not available? Check for bigger fragments */
  if ( (void *) freeblock < glob->firstptr ) {
    for (l=logsize+1; (l < MAXLOG) && (!blockptr); l++)
      blockptr = glob->freeblockptr[l];
  } else
    blockptr = (void *) freeblock;

#if ERRORCHECK
  if (blockptr >= (void *)dseg_lo && blockptr < (void *)dseg_hi) {
#else
  if (blockptr) {
#endif

    /* There is a fragment available. Use it! */
    blockptr = initblock(blockptr, size);
#if STATS
    glob->fragbytes -= size;
#endif

  } else {
    /* No fragment usable. Check the untouched space. */
    
    if ( glob->lastbytes < size ) {
      /* grow the heap */
      
      /* If we have a small heap, grow as little as possible. */
      if ( (mem_usage() > GROWHEAPMIN) || 
	   (sizerecentlyadded(size) > 1) ) {
	coeff = 1 + size / mem_pagesize();
	newblock = mem_sbrk(coeff * mem_pagesize());
      } else {
	newblock = mem_sbrk(size-glob->lastbytes+1);
      }

      if ( !newblock ) return NULL;
      glob->lastbytes += (int) dseg_hi - (int) newblock;
    }

#if ERRORCHECK
    if ( glob->lastbytes >= size ) {
#endif
      /* initialize a new segment */
      blockptr = glob->lastptr;      
      initblock(blockptr, size);
      glob->lastptr += size;
      glob->lastbytes -= size;

      if ( (glob->fragbytes < MAXFRAGS) && 
	   (glob->lastbytes >= size) && 
	   sizerecentlyadded(size) ) {

	/* add a false fragment for the next like-sized block. */
	initfalsefrag((void *) (blockptr+size), size);
	glob->lastptr += size;
	glob->lastbytes -= size;
      }

#if ERRORCHECK
    } else return NULL;
#endif
    
  }
  
  /* print info */
#if DEBUG
  printf("malloc  block: %p  size: %4d",(blockptr+4),size);
  printinfo();
  printtable();
#endif

  /* Update our size history information. */
  if (glob->nextrecent >= 0) {
    glob->recentsize[glob->nextrecent] = size;
    glob->nextrecent = (glob->nextrecent + 1) % NUMRECENT;
  }

  return (void *) (blockptr + 4);
}


/* Frees a previously malloced block of memory.
 */
void mm_free (void *ptr)
{
  int thissize, precsize;
  freeblockptr succblk, precblk;

  freeblockptr blockptr = (freeblockptr) (ptr - 4);
  void *blockvoidptr = (ptr - 4);
  heapglobals *glob;

  glob = (heapglobals *) dseg_lo;
  
#if ERRORCHECK
  /* Make sure we aren't freeing something not in the heap! */
  if (ptr <= glob->firstptr || ptr >= glob->lastptr ) return;

  /* Make sure this block is allocated */
  if (blockptr->size == (blockptr->size & MASK)) return;
#endif
  
  /* Check our neighbors */
  thissize = (blockptr->size) & MASK;
  succblk = (freeblockptr) (blockvoidptr + thissize);
  if ( blockvoidptr > glob->firstptr ) {
    precsize = *((int *) (blockvoidptr - 4));
    precblk = (freeblockptr) (blockvoidptr - (precsize & MASK));
  } else 
    precsize = 1;

#if STATS
  glob->fragbytes += thissize;
#endif

#if DEBUG
  /* print info */
  printf("free    block: %p  size: %4d",ptr,thissize);
  printinfo();
#endif

  if ( !(precsize & kBlockAllocated) ) {
    /* preceding block is free. Merge. */
    removefrag(precblk);
    thissize += precblk->size;
    blockptr = precblk;
  }

  if ( (void *)succblk == glob->lastptr ) {
    /* we're just before the last free block */
    glob->lastptr = blockptr;
#if STATS
    glob->fragbytes -= thissize;
#endif
    glob->lastbytes += thissize;
    
#if DEBUG
    /* print info */
    printf("free    block: %p  size: %4d",ptr,thissize);
    printinfo();
    printtable();
#endif

    return;
  }
  
  if ( !(succblk->size & kBlockAllocated) ) {
    /* following block is free. Merge. */
    removefrag(succblk);
    thissize += succblk->size;
  }
  
  /* Set the free block size */
  blockptr->size = thissize;
  *((int *) ((void *)blockptr + thissize - 4)) = thissize;
  
  /* Add ourselves to the block table */
  if ( thissize > 8)
    insertfrag(blockptr,thissize,log2(thissize));

#if DEBUG
  /* print info */
  printf("free    block: %p  size: %4d",ptr,thissize);
  printinfo();
  printtable();
#endif

}



