/* $Id$ */

/*
 *
 *  CS213 - Lab assignment 3
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

team_t team = {
    /* Team name to be displayed on webpage */
    "The KGB",
    /* First member full name */
    "Daniel Froman",
    /* First member email address */
    "dfroman",
    /* Second member full name (leave blank if none) */
    "Margaret Delap",
    /* Second member email address (blank if none) */
    "mid"
};

#define size_align(sz)  ((((sz) & (0x7)) == 0) ? (sz) : (((sz) + 8) & ~(0x7)))
#define in_heap(x)  (((char *)(x) >= dseg_lo) && ((char *)(x) < dseg_hi))
#define smallbin_index(amt)  (((unsigned long)(amt)) >> 3)

/* Given a chunk size, find the bin it's supposed to go into. 
   smallbin_index is faster if the size is appropriate (<512). 
   The numbers are from libc, since they work well. */
#define bin_index(sz)                                                         \
(((((unsigned long)(sz)) >> 9) ==    0) ?       (((unsigned long)(sz)) >>  3):\
 ((((unsigned long)(sz)) >> 9) <=    4) ?  56 + (((unsigned long)(sz)) >>  6):\
 ((((unsigned long)(sz)) >> 9) <=   20) ?  91 + (((unsigned long)(sz)) >>  9):\
 ((((unsigned long)(sz)) >> 9) <=   84) ? 110 + (((unsigned long)(sz)) >> 12):\
 ((((unsigned long)(sz)) >> 9) <=  340) ? 119 + (((unsigned long)(sz)) >> 15):\
 ((((unsigned long)(sz)) >> 9) <= 1364) ? 124 + (((unsigned long)(sz)) >> 18):\
                                          126)
/* 127 should be top */

/* A free chunk has size fields and pointers for the doubly linked free list. 
   An allocated chunk ignores the pointers.  This is good, because we assume
   that as the free lists become smaller we want to use as much of the 
   available space as possible.
   Minimum size for a chunk is 16, 8 of which is always used. */
typedef struct malloc_chunk {
  size_t size_prev;
  size_t size;
  struct malloc_chunk *fd;
  struct malloc_chunk *bk;
} chunk;

#define NUM_BINS 128
#define MINSIZE 16

typedef struct arena {
  chunk bin[NUM_BINS];
} arena;

#define chunksize(p)  ((p)->size & ~(0x1))           /* Size of this chunk */
#define prevsize(p)   ((p)->size_prev)
#define next_chunk(p) ((char *)(p) + chunksize(p)) /* Pointer to next chunk 
						      contiguous in memory */
#define prev_chunk(p) ((char *)(p) - prevsize(p)) /* Previous chunk.  These
						     will probably do strange 
						     things if used on chunks
						     at one or the other end
						     of the usable heap 
						     space. */
#define is_allocated(p)  ((p)->size & 0x1) /* this-inuse bit (last bit of size
					      isn't necessary because of
					      alignment) */
#define allocate(p)  ((p)->size |= 0x1)    /* set the inuse bit */
#define deallocate(p) ((p)->size &= ~(0x1))
#define main_arena ((arena *)dseg_lo)


/* Take a chunk off a free list */
static void remove_from_free_list(chunk *victim){
  /* nothing to remove */
  if (!victim) 
    return;
  else {
    if (victim->bk != 0) 
      victim->bk->fd = victim->fd;
    if (victim->fd != 0) 
      victim->fd->bk = victim->bk;
  }
}


/* Add a chunk to a free list */
static void add_to_free_list(chunk *p)
{
  size_t size = chunksize(p);
  int index = (size < 512) ? smallbin_index(size) : bin_index(size);
  chunk *bin = &(main_arena->bin[index]);
  chunk *there;   /* what's already on list, if anything */
  if (index == NUM_BINS-1)                     /* overwrite top , but 
						  this should never be used */
    remove_from_free_list(bin->fd);
  there = bin->fd;
  bin->fd = p;
  p->bk = bin;
  p->fd = there;
  if (there) there->bk = p;
}

/* Try to split chunk victim into a chunk of the right size
   and the remainder.  If it's impossible, return -1.  Otherwise
   return the bin where (new) victim would go.
   If the remainder would be too small to use, just allocate the whole 
   chunk.
   If the remainder size is nonzero (implies >= 16, aligned, etc.), add it 
   to the appropriate free list.
*/
static int split_chunk(chunk *victim, size_t size, int is_from_top)
{
  size_t combined_size = chunksize(victim);
  chunk *rem;  /* remainder */
  size_t rem_size;

  if (size > combined_size)
    return -1;  /* cannot get a block of size size */
  else {
    rem_size = combined_size - size;  /* assumed to be at least 16 from context
					 where split_chunk is called */
    rem = (chunk *)((char *)victim + size);
    rem->size_prev = size;     /* shouldn't have to set victim's size_prev */
    rem->size = rem_size; /* free, so don't set bit */
    victim->size = size;
    /* victim->size_prev shouldn't have to change */
    if (is_from_top)      /* keep it there */ {
      main_arena->bin[NUM_BINS-1].fd = rem;
      rem->bk = &(main_arena->bin[NUM_BINS-1]);
      rem->fd = 0;
    } else
      add_to_free_list(rem);
    return 0;
  }
}

/* Make a table of free lists.  There are 128 bins.  Chunks are indexed into
   bins by size.  The exception is "top" (the last bin.)  top is never
   allocated;  it contains one chunk of whatever memory has been acquired
   from calls to mem_sbrk() and not yet used.  If there aren't any 
   allocatable chunks, malloc tries to get more from top.  If top isn't
   big enough, it is extended with mem_sbrk().
   Initially, lists in all bins except top are set to null, and top is set to
   whatever it gets from mem_sbrk(pagesize) minus the size of the bin table.  
   This means that the first chunk will always have to be obtained from top, 
   and the free lists will be sparse until lots of splitting and/or freeing is
   done.  
*/
int mm_init (void)
{
  int i;
  size_t top_address;

  if (!mem_sbrk(mem_pagesize())) {
    printf("Fatal: could not get heap\n");
    exit(-1);
  }
  for (i = 0; i < NUM_BINS-1; i++)
    main_arena->bin[i].fd = 0;
  /* set up top */
  top_address = size_align((int)dseg_lo + sizeof(arena));
  main_arena->bin[NUM_BINS-1].fd = (chunk*)(top_address);
  main_arena->bin[NUM_BINS-1].fd->fd = 0;
  main_arena->bin[NUM_BINS-1].fd->bk = &(main_arena->bin[NUM_BINS-1]);  
  (main_arena->bin[NUM_BINS-1].fd)->size = ((int)dseg_hi - top_address + 1);
  (main_arena->bin[NUM_BINS-1].fd)->size_prev = 0;
  return 0;
}

/* If top is too small, mem_sbrk more and add to top. */
static int more_top (size_t diff) {
  unsigned long pagesize = mem_pagesize();
  chunk *top = main_arena->bin[NUM_BINS-1].fd;
  size_t old_top_size = chunksize(top);
  size_t new_top_size;
  char* old_dseg_hi = dseg_hi;
  char* brk_ret;
  size_t mod = diff%pagesize;

  diff = (!mod) ? diff : (diff + pagesize - mod);
  brk_ret = (char*)(mem_sbrk(diff));
  if (!brk_ret) return -1; /* and pass down error in malloc */
  new_top_size = dseg_hi - old_dseg_hi + old_top_size;
  top->size = new_top_size;
  return 0;
}

/* allocate:
   If small: return the first chunk on the free list of the corresponding
             bin.  This should always work because in "small" bins, chunks 
	     are all the same size.
	     If there are no free chunks in that bin, then it will be 
	     necessary to get one from top.  It's assumed that we'll probably
	     have to split it (remainder is probably >0), so we just do that.
	     If there isn't enough space in top, then try to increase top by
	     the necessary amount.  The size of the chunk in top must always
	     be at least MINSIZE.
	     If extending top fails, then mm_malloc also fails by returning
	     a null pointer.
   Otherwise, malloc has to try everything in the free list of the 
             appropriate bin until it finds something big enough.  If it gets 
	     to the end of the list, it tries to allocate from top in the same
	     way as it does for small chunks.
   It might be more space-efficient to try some of the higher-numbered bins
   if looking in the assigned bin didn't help.  It would also be nice to keep
   the last remainder from splitting (like libc does.)  This malloc doesn't
   do either.  Without coalesce, splitting chunks in higher-numbered bins
   might actually decrease space efficency.
*/
void *mm_malloc (size_t size) {
  int index;
  chunk *victim;

  if (size <= 0) return 0;
  size = size_align(size) + 8;
  /* case small bin */
  if (size < 512) {
    index = smallbin_index(size);
    victim = main_arena->bin[index].fd;
    if (victim != 0) {
      remove_from_free_list(victim);
      allocate(victim);
      return (void *)((char *)victim + 8);
    } else {
      /* take from the last bin */
      victim = main_arena->bin[NUM_BINS-1].fd;
      if (chunksize(victim) - MINSIZE < size)
	if (more_top(size - chunksize(victim) + MINSIZE) == -1)
	  /* more_top fails, don't allocate.*/
	  return 0;
      index = split_chunk(victim, size, 1); /* assuming that works */
      if (index == -1)
	return 0; 
      /* done */  /* taken from top, therefore not in free list */
      allocate(victim);
      return (void *)((char *)victim + 8);
    }
  } else {
    /* may have to break a bin */
    index = bin_index(size);
    victim = main_arena->bin[index].fd;
    while (victim != 0) {
      if (size == chunksize(victim)) {
	remove_from_free_list(victim);
	allocate(victim);
	return (void *)((char *)victim +8);
      } else if (chunksize(victim) - MINSIZE >= size) { /* remainder at least 
						      MINSIZE, since != 0 */
	index = split_chunk(victim, size, 0);
	if (index != -1) {
	  /* done */
	  remove_from_free_list(victim);
	  allocate(victim);
	  return (void *)((char *)victim + 8);
	}
      }
      victim = victim->fd; 
    }
    index = NUM_BINS-1;
    victim = main_arena->bin[NUM_BINS-1].fd;
    if (chunksize(victim) - MINSIZE < size)
      if (more_top(size - chunksize(victim) + MINSIZE) == -1) 
	/* more_top fails */
	return 0;
    index = split_chunk(victim, size, 1);
    /* done */
    if (index == -1)
      return 0;
    allocate(victim);
    return (void *)((char *)victim + 8);
  }
}

/* This should take a pointer to a chunk and coalesce that chunk with the free
   chunks immediately before and after it, if they exist.  It should  remove 
   all defunct chunks from their free lists.  If the chunk is coalesced with 
   top, it should be returned to top.  Otherwise it should be added to a free 
   list.  At the moment it is decreasing space efficiency for some reason, so
   I assume there's something wrong and have defined it out. 
*/
static void coalesce(chunk *p){
  chunk *next;
  chunk* new_next;
  chunk *prev = (chunk*)(prev_chunk(p));
  size_t sz_temp = chunksize(p);
  chunk* start_of_usable_heap = (chunk*)((int)dseg_lo + sizeof(arena));

  if (p == main_arena->bin[NUM_BINS-1].fd)
    return;  /* You have tried to free top.  This cannot be done. */
  else {
    next = (chunk*)(next_chunk(p)); /* Assume that as long as we're not top, 
				       we aren't at the top of the heap. 
				       This should be true, since chunks are 
				       always split from the low end of top.
				    */
    if (next == main_arena->bin[NUM_BINS-1].fd) {  /* next is top */
      sz_temp += chunksize(next);
      if (p != start_of_usable_heap && !(is_allocated(prev))) {
	remove_from_free_list(prev);
	sz_temp += chunksize(prev);
	p = prev;
      } 
      p->size = sz_temp;
      main_arena->bin[NUM_BINS-1].fd = p;
      /* don't need to set prev_size */
    } else { /* next is a normal chunk */
      if (!is_allocated(next)){ /* Assuming this will work, because next is
				   not top: should be below end of heap */
	remove_from_free_list(next);
	sz_temp += chunksize(next); 
      } 
      /* something broken */
      if (p != start_of_usable_heap && !(is_allocated(prev))) {
	if (p->size_prev != prev->size) {} /* something is wrong here:
					      should not happen */
	else {
	remove_from_free_list(prev);
	sz_temp += chunksize(prev);   /* coalesce backward */
	p = prev; }
      }
      p->size = sz_temp;
      add_to_free_list(p);
      new_next = (chunk*)((char*)(p) + sz_temp); /* might have to change 
						    next's prev_size */      
      new_next->size_prev = sz_temp;
    }
  }
}

/* takes a pointer to some memory, and frees the chunk found there.  If
   given a bad pointer it will do bad things.  It does not check whether ptr
   has been allocated.
   Should coalesce too, but at the moment coalesce is killing utilization
   for some reason.
*/
void mm_free (void *ptr) {
  chunk *victim = (chunk *)((char *)ptr - 8);
#ifdef COALESCE
  coalesce(victim);
#else
  add_to_free_list(victim);
#endif
  deallocate(victim);
}
