/* $Id$ */

/******************************
 ** CS213 - Lab assignment 3 **
 *****************************/

/****************************************************************************
 ** We will be using a segregated freelist to catalog unused blocks of     **
 ** memory.                                                                **
 **                                                                        **
 ** The beginning of the heap will store a vector of freelists, with each  **
 ** index representing the log of the largest amount of data that the list **
 ** can hold.  (ie, index i contains a list of all blocks that can hold up **
 ** to 2^i bytes) Additionally, the smallest amount of data that a         **
 ** specific list at index i can hold is greater than the amount of data   **
 ** that a list at index i-1 can hold.  (ie, index i contains a list of    **
 ** all blocks that contain between 2^(i-1)+1 and 2^i bytes, inclusive)    **
 ** Since the heap is at most 20MB, there are only 26 entries (26 makes it **
 ** easier to maintain the 8-byte alignment rule).                         **
 **                                                                        **
 ** The first element of the freelist vector will contain the number of    **
 ** the highest index used in the freelist vector.                         **
 **                                                                        **
 ** Each block of data is preceeded by a 8-byte header containing the      **
 ** data-size of the block and a pointer to the next block of the same     **
 ** size class.  Each block is also proceeded by an 8-byte footer          **
 ** containing the data-size of the block and a pointer to the previous    **
 ** block of hte same size class.  Blocks at the ends of the list will     **
 ** have their appropriate pointers aimed at NULL.  Free blocks will have  **
 ** a positive size, while used blocks will have a negative size.          **
 **                                                                        **
 ** Coalescing will be done with each free() operation, and each malloc()  **
 ** operation will do firstfit search, splitting blocks and adding them to **
 ** the appropriate freelist as necessary.                                 **
 **                                                                        **
 ** It is assumed that dseg_lo is page-aligned and also 8-byte aligned.    **
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

/*********************************
 ** Miscellaneous clerical data **
 ********************************/
team_t team = {
	"Brikin'ger",                            // Team name to be displayed on webpage
	"Chanh-Duy Tran", "cnt@andrew.cmu.edu",  // First member full name and e-mail
	"Tin Lok Lam",    "tlam@andrew.cmu.edu"  // Second member full name and e-mail (blank if none)
};

/************************************************
 ** Fundamental data structures for management **
 ***********************************************/
typedef struct BlockHeader_struct {
	int data_size;                   // The amount of data (in bytes) that this block can hold
	struct BlockHeader_struct *next; // The next block in the list (in the direction away from the head)
} BlockHeader;
typedef struct BlockFooter_struct {
	int data_size;                   // The amount of data (in bytes) that this block can hold
	struct BlockFooter_struct *prev; // The previous block in the list (in the direction towards the head)
} BlockFooter;

/******************************
 ** VERY important constants **
 *****************************/
#define BLOCK_HEADER_SIZE 8 // This should be the same as sizeof(BlockHeader)
#define BLOCK_FOOTER_SIZE 8 // This should be the same as sizeof(BlockFooter)
#define BLOCK_OVERHEAD    (BLOCK_HEADER_SIZE + BLOCK_FOOTER_SIZE)

#define MIN_DATA_SIZE     8
#define MIN_ALLOCATION    (MIN_DATA_SIZE + BLOCK_OVERHEAD)

#define SLACK_SIZE        0 // Some extra space for expansion
#define SLACK_LIMIT       512 // The breaking point below which we use a power of 2 allocation

#define FREELIST_TOP      26 // Must be a multiple of 2 to satisfy the 8byte alignment rule
#define FREELIST_SIZE     (FREELIST_TOP * sizeof(BlockHeader *)) // Size of the free list vector
#define HIGHEST_USED      0

#define FREE 0
#define FULL 1

#define VERIFY 0
#define DUMP   1

#define TRUE  1
#define FALSE 0

/*********************************************************
 ** DESC: Return the log of the largest power of 2 <= n **
 ** INPT: An integer                                    **
 ** OUTP: floor(log2(n))                                **
 ********************************************************/
int BestFitDown (int x)
{
	int answer = 0;
	answer += !(!(x >> 16)) << 4;
	answer += !(!(x >> (answer + 8))) << 3;
	answer += !(!(x >> (answer + 4))) << 2;
	answer += !(!(x >> (answer + 2))) << 1;
	answer += !(!(x >> (answer + 1)));
	return answer;
}

/**********************************************************
 ** DESC: Return the log of the smallest power of 2 >= n **
 ** INPT: An integer                                     **
 ** OUTP: ceil(log2(n))                                  **
 *********************************************************/
int BestFitUp (int x)
{
	int answer = BestFitDown(x);
	return (1<<answer == x) ? answer : answer + 1;
}

/******************************************************************
 ** DESC: Round the given number up to the nearest multiple of 8 **
 ** INPT: An integer                                             **
 ** OUTP: Multiple of 8                                          **
 *****************************************************************/
int SizeAlignUp (int x)
{
	int m = ((x>>3)<<3) + ((x & 0x07) ? 8 : 0);
	if (m > MIN_DATA_SIZE) {
		return m;
	} else {
		return MIN_DATA_SIZE;
	}
}

/*********************************************
 ** Some functions to make our lives easier **
 ********************************************/
// Return the decoded block size
inline int GetBlockHeaderDataSize (BlockHeader *block) { return (block->data_size >= 0) ? block->data_size : -(block->data_size); }
inline int GetBlockFooterDataSize (BlockFooter *block) { return (block->data_size >= 0) ? block->data_size : -(block->data_size); }
// Return the decoded block usage
inline int GetBlockHeaderUsage (BlockHeader *block) { return (block->data_size > 0) ? FREE : FULL; }
inline int GetBlockFooterUsage (BlockFooter *block) { return (block->data_size > 0) ? FREE : FULL; }
// Get a block's header
inline BlockHeader *GetBlockHeader (BlockFooter *f)
{
	return (BlockHeader *)((char *)(f) - GetBlockFooterDataSize(f) - BLOCK_HEADER_SIZE);
}
// Get a block's footer
inline BlockFooter *GetBlockFooter (BlockHeader *h)
{
	return (BlockFooter *)((char *)(h) + GetBlockHeaderDataSize(h) + BLOCK_HEADER_SIZE);
}
// Find the block physically preceeding the given one
inline BlockHeader *GetPreceedingBlock (BlockHeader *h)
{
	BlockFooter *f = (BlockFooter *)h - 1;
	if (dseg_lo + FREELIST_SIZE <= (char *)f) {
		return (BlockHeader *)((char *)(h) - GetBlockFooterDataSize(f) - BLOCK_OVERHEAD);
	} else {
		return NULL;
	}
}
// Find the block physically proceeding the given one
inline BlockHeader *GetProceedingBlock (BlockHeader *h)
{
	return (BlockHeader *)((char *)(h) + GetBlockHeaderDataSize(h) + BLOCK_OVERHEAD);
}
// Remove a block from the freelist
inline void RemoveBlockFromFreelist (BlockHeader *freelist[], int bfu, BlockHeader *block)
{
	BlockFooter *block_footer = GetBlockFooter(block);
	int i;
	
	if (block->next != NULL) { (GetBlockFooter(block->next))->prev = block_footer->prev; }
	if (block_footer->prev != NULL) {
		(GetBlockHeader(block_footer->prev))->next = block->next;
	} else {
		assert(bfu > 0);
		freelist[bfu] = block->next;
	}
	if ((bfu == (int)freelist[HIGHEST_USED]) && (freelist[bfu] == NULL)) {
		for (i = bfu - 1; 0 < i; i--) { if (freelist[i] != NULL) { break;} }
		(int)freelist[HIGHEST_USED] = i;
	}
	assert((int)freelist[HIGHEST_USED] < FREELIST_TOP);
}
// Add a block to the beginning of the freelist
inline void PutBlockOnFreelist (BlockHeader *freelist[], BlockHeader *block, BlockFooter *block_footer)
{
	int bfu = BestFitUp(GetBlockHeaderDataSize(block));
	
	assert(GetBlockHeaderDataSize(block) > 0);
	assert(GetBlockHeaderDataSize(block) == GetBlockFooterDataSize(block_footer));
	assert(bfu > 0);
	if ((block->next = freelist[bfu]) != NULL) {
		(GetBlockFooter(block->next))->prev = block_footer;
	}
	block_footer->prev = NULL;
	freelist[bfu] = block;
	
	if (bfu > (int)freelist[HIGHEST_USED]) {
		(int)freelist[HIGHEST_USED] = bfu;
	}
	assert((int)freelist[HIGHEST_USED] < FREELIST_TOP);
}

/*******************************************
 ** DESC: Give a summary dump of the heap **
 ** INPT: mode=0: verify                  **
 **       mode=1: dump                    **
 ** OUTP: varies                          **
 ******************************************/
void ColorPrintBlockInfo (BlockHeader *block)
{
	switch (GetBlockHeaderUsage(block)) {
		case FREE:
			printf("%x(E\033[0;34m%d\033[0m) ", (unsigned int)block, GetBlockHeaderDataSize(block));
			break;
		case FULL:
			printf("%x(F\033[0;35m%d\033[0m) ", (unsigned int)block, GetBlockHeaderDataSize(block));
			break;
		default:
			printf("%x(!\033[0;31m%d\033[0m) ", (unsigned int)block, GetBlockHeaderDataSize(block));
			break;
	}
}
void mm_blockdiag (int mode)
{
	int total_data = 0, total_allocated = 0, s;
	BlockHeader *block = (BlockHeader *)(dseg_lo + FREELIST_SIZE);
	
	if (mode) {
		printf("BLOCK DUMP: ");
	} else {
		printf("INCORRECT BLOCKS: ");
	}
	while ((char *)block < dseg_hi) {
		s = GetBlockHeaderDataSize(block);
		if ((s & 0x07) || (s < MIN_DATA_SIZE) || mode) {
			ColorPrintBlockInfo(block);
		}
		(char *)block += s + BLOCK_OVERHEAD;
		total_data += s;
		total_allocated += s + BLOCK_OVERHEAD;
	}
	printf("[\033[0;31m%d\033[0m/\033[0;31m%d\033[0m]\n", total_data, total_allocated);
}
void mm_freediag (int mode)
{
	int i;
	BlockHeader *block, **freelist = (BlockHeader **)dseg_lo;
	
	if (mode == VERIFY) {
		printf("FREE LIST INCORRECT BLOCKS\n");
	} else {
		printf("FREE LIST DUMP\n");
	}
	printf("highest used: %d\n", (int)freelist[HIGHEST_USED]);
	for (i = 1; i < FREELIST_TOP; i++) {
		if (freelist[i] != NULL) {
			printf("freelist[\033[0;34m%d\033[0m]: ", i);
			for (block = freelist[i]; block != NULL; block = block->next) {
				if ((GetBlockHeaderDataSize(block) < MIN_DATA_SIZE) || (GetBlockHeaderUsage(block) != FREE) || (mode == DUMP)) {
					ColorPrintBlockInfo(block);
				}
			}
			printf("\n");
		}
	}
}

/*******************************
 ** DESC: Initialize the heap **
 ** INPT: none                **
 ** OUTP: Success status      **
 ******************************/
int mm_init (void)
{
	BlockHeader **freelist = (BlockHeader **)dseg_lo;
	int i, diff = (dseg_hi - dseg_lo + 1) - FREELIST_SIZE;
	
	if (diff < 0) {
		// If we don't have enough memory for at least the freelist
		//   vector, try to allocate it, otherwise abort
		// It probably isn't necessary to do this, since the heap
		//   seems to start off being zero anyway, and malloc
		//   right now doesn't take advantage of the case where
		//   the heap isn't zero
		if (mem_sbrk(-diff) == NULL) { return -1; }
	}
	
	// Initialize the freelist
	for (i = 0; i < FREELIST_TOP; i++) { freelist[i] = NULL; }
	
	return 0;
}

/************************************************************
 ** DESC: Allocate a block of the given size from the heap **
 ** INPT: Size (in bytes)                                  **
 ** OUTP: Pointer to newly allocated block                 **
 ***********************************************************/
void *mm_malloc (size_t s)
{
	BlockHeader **freelist = (BlockHeader **)dseg_lo;
	BlockHeader *block, *new_block;
	BlockFooter *block_footer, *new_block_footer;
	int new_data_size, req_data_size;
	int bfu;
	
	// If the requested size is small, rounding up may help with fragmentation/utilization
	if (s < SLACK_LIMIT) { s = 1<<BestFitUp(s); }
	// Forcing size into a multiple of 8 makes alignment maintenance *much* easier
	req_data_size = SizeAlignUp(s) + SLACK_SIZE;
	
	// Numbskull programmers should hit themselves on the head with a frying pan
	if (req_data_size <= 0) { return NULL; }
	
	// Try to find a free block and use it
	for (bfu = BestFitUp(req_data_size); bfu <= (int)freelist[HIGHEST_USED]; bfu++) {
		// Skip to the next size class if are none available at this level
		assert(bfu > 0);
		if ((block = freelist[bfu]) == NULL) { continue; }
		// Find the first block that is large enough
		while ((block->data_size < req_data_size) && (block->next != NULL)) { block = block->next; }
		// Check if we stopped looking through the list for a good reason
		if (block->data_size >= req_data_size) {
			// Found a block that's big enough
			// Take the block out of the free space
			RemoveBlockFromFreelist(freelist, bfu, block);
			// Check if we should split this block
			if (block->data_size - req_data_size >= MIN_ALLOCATION) {
				// There's leftover space, so split it
				new_data_size = block->data_size - req_data_size - BLOCK_OVERHEAD;
				assert(new_data_size >= MIN_DATA_SIZE);
				// Shrink this block
				block->data_size = -req_data_size;
				block_footer = GetBlockFooter(block);
				block_footer->data_size = -req_data_size; // Negative size means that this block is being used
				// Carve the new block out of the leftover dataspace of the original block
				new_block = GetProceedingBlock(block);
				new_block->data_size = new_data_size;
				new_block_footer = GetBlockFooter(new_block);
				new_block_footer->data_size = new_data_size;
				// Link in the new block to the front of the free list
				PutBlockOnFreelist(freelist, new_block, new_block_footer);
			} else {
				// Can't split, so just use this block as-is
				block_footer = GetBlockFooter(block);
				block_footer->data_size = block->data_size = -(block->data_size);
			}
			block->next = NULL;
			block_footer->prev = NULL;
			assert(block->data_size < 0);
			return block + 1;
		}
	}
	
	// No free blocks available, so try to allocate some more memory
	if ((block = mem_sbrk(req_data_size + BLOCK_OVERHEAD)) == NULL) {
		printf("failure trying to allocate %d bytes\n", req_data_size);
		mm_freediag(VERIFY);
		mm_freediag(DUMP);
		mm_blockdiag(VERIFY);
		mm_blockdiag(DUMP);
		return NULL;
	}
	block->data_size = -req_data_size;
	block->next = NULL;
	block_footer = GetBlockFooter(block);
	block_footer->data_size = -req_data_size;
	block_footer->prev = NULL;
	assert(block->data_size < 0);
	return block + 1;
}

/******************************************************
 ** DESC: Free a previously allocated block          **
 ** INPT: A pointer previous returned by mm_malloc() **
 ** OUTP: none                                       **
 *****************************************************/
void mm_free (void *ptr)
{
	BlockHeader **freelist = (BlockHeader **)dseg_lo;
	BlockHeader *block = (BlockHeader *)ptr - 1, *other_block;
	BlockFooter *block_footer = GetBlockFooter(block);
	int new_data_size;
	
	//mm_freediag(VERIFY);
	//mm_blockdiag(DUMP);
	
	// Mark this block as unused
	block->data_size = -(block->data_size);
	block_footer->data_size = -(block_footer->data_size);
	assert(block->data_size == block_footer->data_size);
	
	assert(block->data_size >= MIN_DATA_SIZE);
	
	// Try to join this block with the following block
	other_block = GetProceedingBlock(block);
	// Range check
	if ((char *)other_block < dseg_hi) {
		// Coalesce only empty blocks
		if (other_block->data_size > 0) {
			// Take the block out of the freelist
			RemoveBlockFromFreelist(freelist, BestFitUp(other_block->data_size), other_block);
			// One entire set of header/footer will be eaten up by coalescing, so it's important to factor that it
			block->data_size += other_block->data_size + BLOCK_OVERHEAD;
			block_footer = GetBlockFooter(block);
			block_footer->data_size = block->data_size;
		}
	}
	
	assert(block->data_size >= MIN_DATA_SIZE);
	
	// Try to join this block with the one before it
	other_block = GetPreceedingBlock(block);
	// Range check
	if ((dseg_lo + FREELIST_SIZE) <= (char *)other_block) {
		// Coalesce only empty blocks
		if (other_block->data_size > 0) {
			// One entire set of header/footer will be eaten up by coalescing, so it's important to factor that it
			new_data_size = block->data_size + other_block->data_size + BLOCK_OVERHEAD;
			// Take the block out of the freelist
			RemoveBlockFromFreelist(freelist, BestFitUp(other_block->data_size), other_block);
			block = other_block;
			block->data_size = new_data_size;
			block_footer = GetBlockFooter(block);
			block_footer->data_size = new_data_size;
		}
	}
	
	assert(block->data_size >= MIN_DATA_SIZE);
	
	PutBlockOnFreelist(freelist, block, block_footer);
	
	assert(block->data_size >= MIN_DATA_SIZE);
}

