/* $Id$ */

/*
 *
 *  CS213 - Lab assignment 3
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

team_t team = {
    /* Team name to be displayed on webpage */
    "Penelope",
    /* First member full name */
    "Panita Pongpaibool",
    /* First member email address */
    "panita",
    /* Second member full name (leave blank if none) */
    "",
    /* Second member email address (blank if none) */
    ""
};

/* Implementation 
   --------------

   Implement memory management using the simple explicit free list.
   Each free block structure contains a header, a footer, a forward
   ptr and a backward ptr. Each of these fields is 4 bytes long.
   
   Header contains block size with the lowest bit indicating if the 
   block is free or in use.
   Footer contains block size. (no inuse bit is encoded here.)
   ForwardPtr is a pointer to the next free block in the list.
   BackwardPtr is a pointer to the previous free block in the list.

   Three global variables are stored in the very first 12 bytes of the heap.
   1) freeListHead: a pointer to the first free block in the list
   2) freeListTail: a pointer to the last free block in the list
   3) firstLegalBlock: a pointer to the block at the lowest end of the heap
      
   For example, after the call to mm_init the structure of the heap is
   as follows.
    ----------------------------------------
    |LH|LT|LB|10|x |x |  |  |  |  |  |  |10|  
    ----------------------------------------
             ^  ^  ^                    ^  
             A  B  C                    D  
      
      A=Header
      B=freeListHead, freeListTail, ForwardPtr = NULL
      C=BackwardPtr = NULL
      D=Footer=NULL
   
      LH=storage for freeListHead
      LT=storage for freeListTail
      LB=storage for firstLegalBlock

  Initially heap is allocated in muliple of pages (MULT_PAGE_ALLOCATE) which
  I set to 4 pages each time the heap is expanded. 

  The mm_malloc function will return memory allocated with size >= requested
  size. The allocated memory maybe larger than the requested size b/c we
  to padding in order to make other free block aligned at 8 byte addressing.

  The mm_free function add the free block to the free list and try to coalesce
  it with its neighbor everytime mm_free is called.

  If time permits, optimization can be done at
  1) do not coalesce every mm_free call, but every certain interval-->make mm_free faster
  2) use segregated free list by initially split up the heap in to blocks
     of different sizes and keep different lists of different block sizes.-->make mm_malloc faster

 */

/* #defines */


#define ALIGNMENT 8
#define MULT_PAGE_ALLOCATE 4
#define MIN_USEFUL_FREE_BLOCK 16
 /*the free space needs to be at least 16 bytes for it to be on free list*/

/* macro */

/* getting block information */
#define Bytes(n) n*sizeof(char)
#define ForwardPtr(p) *((void **)p)
#define BackwardPtr(p) *((void **)(p + Bytes(4)))
#define Header(p) (*((int *)(p - Bytes(4))) & 0xfffffffe)
#define Footer(p) (*((int *)(p + Header(p) - Bytes(8))))
#define InUse(p) ((*((int *)(p - Bytes(4)))) & 0x00000001)

/* global variables--use heap area instead of stack area */
#define freeListHead (*((char**)dseg_lo))
#define freeListTail (*((char**)(dseg_lo+Bytes(4))))
#define firstLegalBlock (*((char**)(dseg_lo+Bytes(8))))

/*
char *freeListHead=NULL;
char *freeListTail=NULL;
char *firstLegalBlock=NULL;
*/

/* function declarations */

/* main functions */
int mm_init(void);
void *mm_malloc(size_t size);
void mm_free(void *block);

/* other functions */
void* initNewPage(void *ptr);
int aligned(void *ptr);
void *findFreeBlock(size_t size);
void removeFromFreeList(void *ptr);
void splitFreeBlock(void *ptr,size_t size);
void addToFreeList(void *ptr);
int coalesce(void *block);
int coalesceBetweenBlocks(void *block);
int coalesceNextBlock(void *block);
int coalescePrevBlock(void *block);

/* helper functions */
void PutHeader(void *ptr,int size);
void PutFooter(void *ptr,int size);
void PutInUse(void *ptr,int inuse);

void PutHeader(void *ptr,int size){
  *((int*)(ptr - Bytes(4))) = size;
}

void PutFooter(void *ptr,int size){
  *((int *)(ptr + Header(ptr) - Bytes(8))) = size;
}

void PutInUse(void *ptr,int inuse){
  /* have to put it both at header and at footer */
  /* assume that header and footer always end with bit 0--multiple of 2 bytes*/
  /* therefore, we use this last bit to indicate inUse */
  int newHeader;
  int oldHeader = Header(ptr);
  newHeader = oldHeader + inuse;

  PutHeader(ptr,newHeader);

#ifdef DB
  printf("ptr %p:inuse %d old header %x new header %x\n ",ptr,inuse,oldHeader,newHeader );
#endif

}

int mm_init (void)
{
  void * newSpace;
  int pageSize = mem_pagesize();
  void *newHeap = mem_sbrk(MULT_PAGE_ALLOCATE*pageSize);
  if (newHeap == NULL)
    return -1;

  /*allocate heap space (12bytes for 3 global variable*/

  newSpace = initNewPage(newHeap+Bytes(12));
  if (newSpace == NULL){
    return -1;
  }
  ForwardPtr(newSpace) = NULL;
  BackwardPtr(newSpace) = NULL;
  firstLegalBlock = newSpace;
  freeListHead = newSpace;
  freeListTail = newSpace;

#ifdef DB
  printf("dsegLO %d,dsegHI %d,hi-lo %d totalmem %d\n",dseg_lo,dseg_hi,dseg_hi-dseg_lo,mem_usage());
  printf("freeListHead %p:%p,freeListTail %p:%p,fwdPtr %p,bwdPtr %p,header %d,footer %d\n",freeListHead,*freeListHead,freeListTail,*freeListTail,ForwardPtr(newSpace),BackwardPtr(newSpace),Header(newSpace),Footer(newSpace));
#endif

  return 0;
}

/*return pointer to the new page that is allocated, or 
  to the new chunk of free block in case the new page is combined with
  old free space*/
void *initNewPage(void *ptr){
  int newSpaceSize,prevBlockSize,newSize;
  void *prevBlock, *start;

  newSpaceSize = (int)(dseg_hi - (int)ptr + Bytes(1));

  if (ptr != (dseg_lo+Bytes(12))) {/*this is the subsequent page allocation*/
    if (((int)(dseg_hi - (int)ptr)) < MIN_USEFUL_FREE_BLOCK)
      return NULL;
    
    /*try to combine this new space with previous free block of last page*/
    if (newSpaceSize < MIN_USEFUL_FREE_BLOCK)
      return NULL;

    prevBlockSize = *(int*)(ptr-Bytes(4));
    prevBlock = ptr - prevBlockSize + Bytes(4);
    if (!InUse(prevBlock)){ /*coalesce prevBlock with new pages*/
      newSize = prevBlockSize + newSpaceSize;
      PutHeader(prevBlock,newSize);
      PutFooter(prevBlock,newSize);
      PutInUse(prevBlock,0);
      return prevBlock;
    }else{ /*cannot coalesce, need to align ptr before*/

      /*assume page boundary is 8-byte aligned*/
      
      start = ptr + Bytes(8);
      newSize = newSpaceSize - Bytes(4);
      PutHeader(start,newSize);
      PutFooter(start,newSize);
      PutInUse(start,0);
      
      PutHeader((ptr+Bytes(4)),0);
      PutInUse((ptr+Bytes(4)),1); /*this space is useless so pretend it's used*/
 
      addToFreeList(start);
      return prevBlock;
    }
  }else{/*very first page allocation, need to align ptr*/
    start = ptr + Bytes(4);
    newSize = newSpaceSize;
    PutHeader(start,newSize);
    PutFooter(start,newSize);
    PutInUse(start,0);
    /*don't need to call addToFreeList, mm_init takes care of it*/
    return start;
  }
}

int aligned(void *ptr){
  if (((int)ptr%ALIGNMENT)==0)
    return 1;
  else return 0;
}

void *mm_malloc (size_t size)
{
  void *newHeap, *newSpace;
  /*find a free block with enough space*/
  void *newPtr = findFreeBlock(size); 
  int pageSize;

  /*cannot find enough free space, allocate new pages*/

  while (newPtr == NULL){
    if (mem_usage() >= DSEG_MAX) {/*already reach heap limit*/
      return NULL;
    }
    /*else*/
    pageSize = mem_pagesize();
    newHeap = mem_sbrk(MULT_PAGE_ALLOCATE*pageSize);
    if (newHeap == NULL)
      return NULL;
    else {/*retun 8-byte aligned ptr*/
      newSpace = initNewPage(newHeap);
      if (newSpace == NULL)
	return NULL;

#ifdef DB
      printf("add new page*****************freeListHead %p,freeListTail %p\n",freeListHead,freeListTail);
#endif

      newPtr = findFreeBlock(size);
    }
  }

  /*check if the free block needs to be splitted*/
  /*don't forget that total size is size+8*/
  if (Header(newPtr) > (size+Bytes(24))){ /*may be able to split it*/
    splitFreeBlock(newPtr,size);
  }else{/* if (Header(newPtr) <= (size + Bytes(24))) right size*/
    removeFromFreeList(newPtr);
  }

#ifdef DB
  printf("#####malloc size %x at ptr %p\n",size,newPtr);
  printf("#####freeListHead %p, freeListTail %p\n",freeListHead,freeListTail);
#endif

  return newPtr;
  
}

/*traverse the free list linearly to find a block large enough for the
  new allocation. return a ptr to this block or NULL if no block is found*/
void *findFreeBlock(size_t size){
 
  void *tempPtr
;
  if (freeListHead==NULL) /*free list is empty*/
    return NULL;

  tempPtr = freeListHead;
  while (tempPtr!=NULL){ /*tempPtr shouldn't be null??*/
    if (Header(tempPtr) >= (size+Bytes(8)))
      return tempPtr;
    
    if (tempPtr == freeListTail)
      break;
    
    tempPtr = ForwardPtr(tempPtr);
    
  }

  /*if break the while loop, that means end of list is reached without
    finding the free block*/
  return NULL;
}

/*check special cases: last block in list, first block in list*/
void removeFromFreeList(void* ptr){

  if (ptr == NULL)
    return;

  if ((freeListHead == NULL)&&(freeListTail == NULL)){
    fprintf(stderr,"Err: Remove from empty list\n");
    exit(1);
  }

  if (ptr == freeListHead) { /* head of list*/
    freeListHead = ForwardPtr(ptr);

#ifdef DB
    printf("remove freeListHead");
#endif

  }else{
    ForwardPtr(BackwardPtr(ptr)) = ForwardPtr(ptr);
  }
  if (ptr == freeListTail) { /* tail of list */
    freeListTail = BackwardPtr(ptr);
  }else{
    BackwardPtr(ForwardPtr(ptr)) = BackwardPtr(ptr);
  }
  PutInUse(ptr,1);/*this block is now in use*/
}

void splitFreeBlock(void *ptr,size_t size){
  int oldBlockSize,allocSize,splitSize;
  void *newFreeBlock, *tempPtr;

  if (ptr == NULL)
    return;
  tempPtr = ptr+size+Bytes(8); /*include footer and next header*/
  
  /*add some padding if tempPtr is not 8-byte aligned*/
  /*need to check that we don't go over the boundary of this block
    if after splitting, have less than 16 bytes, need to be careful 
    about assigning forwardPtr and backwardPtr*/
   
  while (!aligned(tempPtr)){
    tempPtr = tempPtr + Bytes(1);
  }
  
  newFreeBlock = tempPtr;
  oldBlockSize = Header(ptr);
  allocSize = (int)newFreeBlock- (int)ptr;
  splitSize = oldBlockSize-allocSize;
  
  /*what to do if size is less than 16? add in free list? */
  /*can't be in free list b/c no space for ptr*/
  /*therefore, guarantee to have a block >= 16 bytes after splitting */

  if (splitSize < Bytes(MIN_USEFUL_FREE_BLOCK)){
    removeFromFreeList(ptr);
    return;
  }

  PutHeader(newFreeBlock,splitSize);
  PutFooter(newFreeBlock,splitSize);
  PutInUse(newFreeBlock,0);
  
  ForwardPtr(newFreeBlock) = ForwardPtr(ptr);
  BackwardPtr(newFreeBlock) = BackwardPtr(ptr);

   if (ptr==freeListHead){
    /*first block in list*/

#ifdef DB
     printf("split freeListHead\n");
#endif

    freeListHead = newFreeBlock;
   }else{ /*need to update previous link*/
     ForwardPtr(BackwardPtr(ptr)) = newFreeBlock;
   }

  if (ptr==freeListTail){
    /*last block in list*/

#ifdef DB
    printf("split freeListTail\n");
#endif

    freeListTail = newFreeBlock;
  }else{
    BackwardPtr(ForwardPtr(newFreeBlock)) = newFreeBlock;
  }

  PutHeader(ptr,allocSize);
  PutFooter(ptr,allocSize);
  PutInUse(ptr,1);

}

void mm_free (void *ptr)
{
  /*need to check if ptr is a valid ptr returned from mm_malloc, how???*/
  if (ptr == NULL) 
    return;

  /*check for possibility of coalescing*/
  if (!coalesce(ptr)){
    /*cannot coalesce, add to the end of free list*/

#ifdef DB
    printf("cannot coalesce");
#endif

    addToFreeList(ptr);
  }

#ifdef DB
  printf("#####free block size %x at %p\n",Header(ptr),ptr);
  printf("#####freeListHead %p, freeListTail %p\n",freeListHead,freeListTail);
#endif

}

/*add a free block to the end of free list*/
void addToFreeList(void *ptr){
  void *lastBlock;

  if (ptr == NULL)
    return;

  /*check if add to empty list*/
  if ((freeListHead==NULL)&&(freeListTail==NULL)){
    freeListHead = ptr;
    BackwardPtr(ptr) = NULL;
  }else{ /*add to end of list*/
    lastBlock = freeListTail;
    BackwardPtr(ptr) = lastBlock;
    ForwardPtr(lastBlock) = ptr;
  }

  ForwardPtr(ptr) = NULL;
  PutInUse(ptr,0);
  freeListTail = ptr;
}

/*check if this free block can be coalesced with neighbors
  return 1 if it can, return 0 otherwise.
  this function needs to be fast!!! check total 3 cases
  1) coalesce with next free block
  2) coalesce with previous free block
  3) coalesce with both previous and next blocks
  */
int coalesce(void *block){
  int done;
  
  if (block == NULL)
    return 0;

  if (block == firstLegalBlock){
    /*lowest end of address space, check only case 1*/
    done = coalesceNextBlock(block);
    return done;
  }else if (((int)(block+Header(block))) >= (int)dseg_hi){
    /*highest end of address space, check only case 2*/
    done = coalescePrevBlock(block);
    return done;
  }else{
    /*somewhere in the middle, check case 1,2,3*/
    done = coalesceBetweenBlocks(block);
    if (!done){
      done = coalesceNextBlock(block);
      if (!done){
	done = coalescePrevBlock(block);
      }
    }
    return done;
  }
}

int coalesceBetweenBlocks(void *block){
  int prevBlockSize,totalSize;
  void *prevBlock, *nextBlock;
  
  if (block == NULL)
    return 0;

  prevBlockSize = *(int*)(block-Bytes(8)); 
  prevBlock = block-prevBlockSize;
  nextBlock = block+Header(block);
  
  if ((InUse(prevBlock)==0) && (InUse(nextBlock)==0)){
    /*both prev and next block are free to be coalesce with*/
    totalSize = prevBlockSize+Header(block)+Header(nextBlock);
    PutHeader(prevBlock,totalSize);
    PutFooter(nextBlock,totalSize);
    PutInUse(prevBlock,0);
    removeFromFreeList(nextBlock);

#ifdef DB
    printf("can coalesce with prev and next blocks");
#endif

    return 1;
  }else{
    return 0;
  }
}

int coalesceNextBlock(void *block){
  int totalSize;
  void *nextBlock = block+Header(block);
  
  if (block == NULL)
    return 0;

  if (InUse(nextBlock)==0){
    /*can coalesce with next block*/
    totalSize = Header(block)+Header(nextBlock);
    PutHeader(block,totalSize);
    PutFooter(nextBlock,totalSize);
    PutInUse(block,0);
    removeFromFreeList(nextBlock);
    addToFreeList(block);

#ifdef DB
    printf("can coalesce with next block");
#endif

    return 1;
  }else{
    return 0;
  }
}

int coalescePrevBlock(void *block){
  int totalSize,prevBlockSize;  
  void *prevBlock;

  if (block == NULL)
    return 0;

  prevBlockSize = *(int*)(block-Bytes(8)); 
  prevBlock =  block-prevBlockSize; 

  if (InUse(prevBlock)==0){
    /*can coalesce with previous block*/
    totalSize = Header(block)+prevBlockSize;
    PutHeader(prevBlock,totalSize);
    PutFooter(block,totalSize);
    PutInUse(prevBlock,0);

#ifdef DB
    printf("can coalesce with prev block");
#endif

    return 1;
  }else{
    return 0;
  }
}



