/* $Id$ */

/*  CS213 - Lab assignment 3 */

/*
  Our malloc implementation
   by Alex Burgel and David Wang

   We used a segregated list system, with a best fit policy. we divided
   our list into exact bins and sorted bins. The exact bins were sizes from
   16 bytes to and including 512 bytes. After that we had sorted bins, each
   one holding a power of 2, starting at 2^9+1 to 2^31. These bins were
   sorted by size. The blocks are linked by pointers, pointing in
   both directions.

   For our blocks, we have a 4 byte header, which is aligned to 4 bytes and not
   aligned to 8 bytes. the header contains the size, with the least significant
   bit indicating whether that block is free or not, 1 is free, 0 is allocated
   After that, we have the users area, which is aligned to 8 bytes, and must be
   a multiple of 8 bytes in size. If the block is free, then the first 8 bytes
   contain pointers to the previous and next free blocks on the particular list
   that the block is on.

   our mm_init function allocates our array of pointers for the free list right
   after dseg_lo (we assume that dseg_lo is 8byte aligned) right after our
   control structure we start the first block, which goes until dseg_hi - 3, so
   we leave 4 bytes at the end, which we initialize to 0 (so it looks allocated)
   this is for when we call mem_sbrk().

   in mm_malloc, we find the right bin for the size that the user is requesting
   if its a sorted bin, then we jump down the list to find the smallest block
   that is big enough to hold the users request. then we check if it can be split
   if it can then we do that, and put the remaining space on the free list.

   if we can't get any memory this way, then we ask for more mem from mem_sbrk

   in mm_free, we get the block, try to coalesce it with the blocks immediatly
   before and after. then add it to the appropriate free list.
*/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include "memlib.h"
#include "malloc.h"

team_t team = {
    /* Team name to be displayed on webpage */
    "Scandalous Kitty",
    /* First member full name */
    "David Wang",
    /* First member email address */
    "wang4",
    /* Second member full name (leave blank if none) */
    "Alexander Burgel",
    /* Second member email address (blank if none) */
    "aburgel"
};

/*
  We declared a structure to handle the header and two pointers of our blocks
*/
struct nodes {
  int  size;
  struct nodes*  prev;
  struct nodes*  next;
};

typedef struct nodes node;


/*
  This is the log2 function from lab1 that floors the log2 of a given number.
  We use it to calculate index.
*/
int log2(int x) {
  int m16 = ((1<<16) + ~0) << 16;                         /* groups of 16 */
  int m8 = (((1<<8)  + ~0) << 24) + (((1<<8) + ~0) << 8); /* groups of 8 */
  int m4 = (0xf0<<24) + (0xf0<<16) + (0xf0<<8) + 0xf0;    /* groups of 4 */
  int m2 = (0xcc<<24) + (0xcc<<16) + (0xcc<<8) + 0xcc;    /* groups of 2 */
  int m1 = (0xaa<<24) + (0xaa<<16) + (0xaa<<8) + 0xaa;    /* groups of 1 */
  int result = 0;
  int upper;
  int mask;
  
  /* m16 */
  upper = !!(x & m16);
  result += upper << 4;
  mask = m16 ^ ~((upper<<31)>>31);
  
  /* m8 */
  upper = !!(x & m8 & mask);
  result += upper << 3;
  mask &= (m8 ^ ~((upper<<31)>>31)); 
  
  /* m4 */
  upper = !!(x & m4 & mask);
  result += upper << 2;
  mask &= (m4 ^ ~((upper<<31)>>31)); 
  
  /* m2 */
  upper = !!(x & m2 & mask);
  result += upper << 1;
  mask &= (m2 ^ ~((upper<<31)>>31)); 
  
  /* m1 */
  upper = !!(x & m1 & mask);
  result += upper;

  return result;
}

/*
  initialize a free block, setting up the header and footer
  encode last bit of size to be allocation status:
  1 = free
  0 = used
*/
void init_block(node* the_block, int size)
{
  char* temp;
  int encoded_size = size + 1;
  
  /* create the header */
  the_block->size = encoded_size;

  /* create the footer */
  temp = (char*)the_block + size - 4;
  *((int *) temp) = encoded_size;
}

/*
  add_to_list takes a block and an index and adds the block
  to the appropriate index of the array
  for indices greater than 62, the free block list is sorted by size
*/
void add_to_list(node* the_block, int index)
{
  node* temp_ptr;
  node* curr_node;

  if (index < 63) {
    /* item is in exact bins, no sorting needed */
    temp_ptr = (node*)(((int*)dseg_lo)[index]);
    (int*)(((int*)dseg_lo)[index]) = (int*)the_block;
    the_block->prev = NULL;
    the_block->next = temp_ptr;
    if (temp_ptr != NULL)
      temp_ptr->prev = the_block;
  }
  else {
    /* item belongs to a sorted bin */
    curr_node = (node*)(((int*)dseg_lo)[index]);
    if (curr_node == NULL) {
      /* inserting to an empty list... */
      (int*)(((int*)dseg_lo)[index]) = (int*)the_block;
      the_block->next = NULL;
      the_block->prev = NULL;
    }
    else if ((curr_node->size - 1) > (the_block->size - 1)) {
      /* inserting at the head of the list... */
      curr_node->prev = the_block;
      the_block->next = curr_node;
      the_block->prev = NULL;
      (int*)(((int*)dseg_lo)[index]) = (int*)the_block;
    }
    else {
      /* standard insertion, must traverse the list... */
      
      while ((curr_node->next != NULL) && ((the_block->size - 1) > (curr_node->next->size - 1)))
	curr_node = curr_node->next;
      
      the_block->next = curr_node->next;
      the_block->prev = curr_node;
      
      /* check to make sure we dont' seg fault if we are inserting at tail */
      if (curr_node->next != NULL)
	curr_node->next->prev = the_block;
      curr_node->next = the_block;
    }
  }
}

/*
  This function finds the appropriate bin a block belongs to,
  given its size.  
*/
int find_index(int size)
{
  int  index;
  if(size <= 512)
    /* if the size is less than or equal to 512 bytes, then it belongs in an
       exact bin */
    index = size / 8;
  else
    /* otherwise it belongs in a sorted bin, use log2 function to calculate index */
    index = 64 + (log2(size - 1) + 1) - 9;
  
  return  index - 2;
}

/*
  split(block,size) will divide a block into 2 blocks
  if the requested size (size) is more than 16 bytes smaller
  than the size of the given block
  it will put the new block on the appropriate free list.
*/
void split(node* the_block, int block_size)
{
  int difference;
  node* new_block;
  
  /*  We subtract 1 because the_block->size is the encoded size, which is 1 larger  */
  difference = the_block->size - block_size - 1;

  if (difference < 16)
    /* if difference is less than 16 bytes then we don't want to split */
    return;
  
  /* set the pointer to the other half of the splitted block */
  new_block = (node*)(((char*)the_block) + block_size);
  
  /* initialize the header and footer of the two new blocks */
  init_block(new_block, difference);
  init_block(the_block, block_size);
  
  /* add the blocks to the free list individually */
  add_to_list(new_block, find_index(difference));
}

/*  
    This function takes in the user-specified size and add the space
    we need to store it and rounds it up to the nearest multiple of 8
*/
int find_block_size(int size)
{
  /* if user asked for size 0 then we should return 16 */
  if (size == 0) return 16;

  return ((size + 8 + 7) / 8) * 8;
}

/*
  This function takes in the ptr of the block we need and takes it off
  the freelist
*/
void  take_off_freelist(node*  the_block)
{
  int  index;
  int  size = the_block->size - 1;

  if(the_block->prev == NULL){
    /* block is either head of list or only one on list  */
    index = find_index(size);
    (node*)((int*)dseg_lo)[index] = the_block->next;
    /* block is head of list */
    if (the_block->next != NULL)
      the_block->next->prev = NULL;
  }
  else if(the_block->next == NULL){
    /*  if the_block is at the tail of the list...  */
    the_block->prev->next = NULL;
  }
  else {
    /*  if the_block is in the middle of the list...  */
    the_block->next->prev = the_block->prev;
    the_block->prev->next = the_block->next;
  }
}

/*
  Coalesce as we free blocks
*/
node*  coalesce(node*  the_block)
{
  char*  temp = (char*)the_block - 4;
  int    temp_size;
  
  /* first, try to coalesce with block before header */  

  /* check to see if block before header is free */
  temp_size = *((int*)temp);
  if(temp_size % 2 == 1) {
    /* if remainder is 1 then the block is free */
    temp_size -= 1; /* subtract the least significant bit to de-endcode */
    temp = (char*)the_block - temp_size; /* goto the head of the previous block */
    take_off_freelist((node*)temp);
    ((node*)temp)->size = temp_size + the_block->size;
    the_block = (node*) temp;
    temp = temp + the_block->size - 5;
    *((int*)temp) = the_block->size;
  }
  
  /* try to coalesce with block after footer */
  temp = (char*)the_block + (the_block->size - 1);
  temp_size = *((int*)temp);
  if(temp_size % 2 == 1) {
    /* if remainder is 1 then the block is free */
    temp_size -= 1;   /* subtract the least significant bit to de-endcode */  
    take_off_freelist((node*)temp);
    the_block->size += temp_size;
    temp += temp_size - 4;
    *((int*)temp) = the_block->size;
  }
  return the_block;
}

/*
  This function requests a new memory page when we don't have a match on the freelist
*/
node*  get_more_mem(int  block_size)
{
  char*  the_block;
  int  page_size = mem_pagesize();
  int  page_block = block_size;
  
  /* round requested memory up to next highest memory page and allocate */
  page_block = (((page_block + page_size - 1) / page_size) * page_size);
  the_block = mem_sbrk(page_block);

  /* in case it fails */
  if(the_block == NULL)
    return  NULL;
  
  /* go back to fix the last 4 bytes of the previous memory page */
  the_block -= 4;  
  init_block((node*)the_block, page_block);

  /* fix the last 4 bytes of the newly allocated memory page */
  *(int*)(dseg_hi - 3) = 0;
  the_block = (char*)coalesce((node*)the_block);

  /* split the new block */
  split((node*)the_block, block_size);

  return  (node*)the_block;
}

/*
  This function takes a block and changes its header and footer
  to indicate that it is occupied
*/
void*  de_encode(node*  the_block)
{
  char* temp;
  if (the_block == NULL)
    return NULL;

  /* change the header */
  the_block->size -= 1;

  /* goto the footer and change it */
  temp = (char*)the_block + the_block->size - 4;
  *((int*)temp) = the_block->size;
  
  return (void*)((char*)the_block + 4);
}

/*
   our mm_init function allocates our array of pointers for the free list right
   after dseg_lo (we assume that dseg_lo is 8byte aligned) right after our
   control structure we start the first block, which goes until dseg_hi - 3, so
   we leave 4 bytes at the end, which we initialize to 0 (so it looks allocated)
   this is for when we call mem_sbrk().
*/
int mm_init (void)
{
  int i = 0;
  int block_size;
  int index;
  
  /* get a page of memory */
  char *new_mem = mem_sbrk(mem_pagesize());
  
  if (new_mem == NULL)
    return -1;
  
  /* make space for control data, an array for pointers
     340 = 85 * 4, we use an array of length 85 */
  new_mem += (85 * 4);
  for (i = 0; i < 85; i++)
    (int*)((int*)dseg_lo)[i] = NULL;
  
  /* align data to 8-bytes and make room for header */
  /*  new_mem = ((char *)((((unsigned int)(new_mem) + 7) / 8) * 8)) + 4; */

  /* calculating block size */
  block_size = (dseg_hi - new_mem + 1) - 4;
  
  /* Create header/footer for free block */
  init_block((node*)new_mem, block_size);

  /* Initialize the last 4 bytes of memory page */
  *((int*)(dseg_hi - 3)) = 0;
	
  /* find the index */
  index = find_index(block_size);
	
  /* place free block on the free list */
  (char*)((int *)dseg_lo)[index] = new_mem;

  /* set pointers for first free block */
  ((node*)new_mem)->prev = NULL;
  ((node*)new_mem)->next = NULL;
  
  return 0;
}

/*
   in mm_malloc, we find the right bin for the size that the user is requesting
   if its a sorted bin, then we jump down the list to find the smallest block
   that is big enough to hold the users request. then we check if it can be split
   if it can then we do that, and put the remaining space on the free list.

   if we can't get any memory this way, then we ask for more mem from mem_sbrk
*/
void *mm_malloc (size_t size)
{
  int i = 0;
  int  block_size = find_block_size(size);
  int  index = find_index(block_size);
  node*  the_block = NULL;

  /* traverse the bins */
  for(i = index; i < 85; i++){
    /* for each bin we want to search the free list */
    the_block = (node*)(((int*)dseg_lo)[i]);
    
    /* traverse the freelist to find the right size */
    while((the_block != NULL) && ((the_block->size - 1) < block_size))
      the_block = the_block->next;
    
    /* take desired block off freelist and split and return */
    if(the_block != NULL) {
      take_off_freelist(the_block);
      split(the_block, block_size);
      
      /* now set the header and footer to indicate that block is not free */
      return de_encode(the_block);
    }
  }
  
  /* if cannot find space through defer coalescing, request more space */
  if(the_block == NULL)
    the_block = get_more_mem(block_size);
  
  return de_encode(the_block);
}


/*
   in mm_free, we get the block, try to coalesce it with the blocks immediatly
   before and after. then add it to the appropriate free list.
*/
void mm_free (void *ptr)
{
  node*  the_block = (node*)((char*)ptr - 4);
  int    temp_size = the_block->size;
  int    index;

  /* first, we change its header and footer to indicate that it is free  */
  init_block(the_block,temp_size);

  /* coalesce */
  the_block = coalesce(the_block);

  /* find the index */
  index = find_index(the_block->size - 1);

  /* add to free list */
  add_to_list(the_block, index);
}
