;;; -*- SYNTAX: COMMON-LISP; MODE: LISP; BASE: 10; PACKAGE: *SIM-i; MUSER: YES-*-

(in-package '*sim-i :use '(lisp))

;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+
;;;> 
;;;> The Thinking Machines *Lisp Simulator is in the public domain.
;;;> You are free to do whatever you like with it, including but
;;;> not limited to distributing, modifying, and copying.

;;;> Bugs, comments and revisions due to porting can be sent to:
;;;> bug-starlisp@think.com.  Other than to Thinking Machines'
;;;> customers, no promise of support is intended or implied.
;;;>
;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+

;;; Author:  JP Massar.


;;;;  *DEFVARS


(*proclaim '(type boolean-pvar nil!! t!!))


;;; *PROCLAIM-*DEFVAR
;;; *DEFVAR


(defmacro *proclaim-*defvar (variable pvar-type &key initial-value documentation vp-set)
  `(progn
     (*proclaim '(type ,pvar-type) ,variable)
     (*defvar ,variable ,initial-value ,documentation ,vp-set)
     ))


(defmacro *defvar (variable &optional initial-value documentation (vp-set '*default-vp-set*))
  (if (or (member variable '(t!! nil!!)) (constantp variable))
      (error "you cannot redefine ~a, it is a constant." variable))
  (let ((type (get variable 'type)))
    #+*lisp-simulator (declare (ignore type))
    (if (and #+*LISP-SIMULATOR nil
	     type
	     (not (equal type '(pvar *)))
	     (not (equal type '(pvar t)))
	     (consp type)
	     (eq (car type) 'pvar)
	     (or *compilep* (null initial-value)))
	;; *proclaim has been done, expand slightly differently.
	(let ((init (gentemp (format nil "INIT-~:@(~A~)-" variable) (symbol-package variable))))
	  `(progn
	     (proclaim '(special ,variable))
	     (setf (documentation ',variable 'variable) ',documentation)
	     #+*LISP-HARDWARE
	     (eval-when (compile eval load) (pushnew ',variable slc::*distinct-pvars*))
	     (defun ,init ()
	       ,(if (eq vp-set '*default-vp-set*)
		    `(*all (setq ,variable (allocate!! ,initial-value ',variable ',type t)) nil)
		    `(*with-vp-set ,vp-set
		       (*all (setq ,variable (allocate!! ,initial-value ',variable ',type t)) nil))))
	     (eval-when (load eval)
	       (*defvar-1 ',variable ',initial-value ',vp-set ',init ',type))
	     ',variable))
	`(progn
	   (proclaim '(special ,variable))
	   (setf (documentation ',variable 'variable) ',documentation)
	   #+*LISP-HARDWARE
	   (eval-when (compile eval load) (pushnew ',variable slc::*distinct-pvars*))
	   (eval-when (load eval) (*defvar-1 ',variable ',initial-value ',vp-set))
	   ',variable))))


(defun *defvar-1 (variable &optional initial-value (vp-set-name '*default-vp-set*) init-function type)

  ;; Add the *DEFVAR onto our list of all *DEFVARS,
  ;; and if we have *COLD-BOOT'ed, try to actually
  ;; allocate this *DEFVAR and give it its initial
  ;; value.  Don't try to allocate it if the VP SET
  ;; it belongs to has not been instantiated yet.

  (when (eq vp-set-name '*current-vp-set*) (setq vp-set-name (vp-set-name *current-vp-set*)))
  (when (or (not (symbolp vp-set-name))
	    (not (boundp vp-set-name))
	    (not (vp-set-p (symbol-value vp-set-name)))
	    (null (vp-set-name (symbol-value vp-set-name)))
	    (null (or (eq vp-set-name '*default-vp-set*) (find vp-set-name *all-def-vp-sets* :test #'eq :key #'vp-set-name))))
    (cerror "Delete the old *defvar definition if any, and return from *defvar"
	    (cond ((not (symbolp vp-set-name)) "The object ~S is not the name of a Vp Set")
		  ((not (boundp vp-set-name)) "The symbol ~S, which is supposed to have a Vp Set as its value, is unbound")
		  ((not (vp-set-p (symbol-value vp-set-name))) "The symbol ~S has value ~S, which is not a Vp Set")
		  ((null (vp-set-name (symbol-value vp-set-name)))
		   "The symbol ~S is bound to a Vp Set, ~S, which was not defined using DEF-VP-SET~@
                    but rather probably with CREATE-VP-SET.  You can only define *DEFVAR's in Vp Sets~@
                    defined with DEF-VP-SET.")
		  (t "The Vp Set named ~S is not currently defined.  You may have deleted it~@
                      from *Lisp's knowledge by using :undefine-all t in *cold-boot.  You~@
                      may wish to re-evaluate its definition."))
	    vp-set-name (if (and (symbolp vp-set-name) (boundp vp-set-name)) (symbol-value vp-set-name)))
    (delete-old-*defvar-definition variable)
    (return-from *defvar-1 nil))

  (let ((new-*defvar-specification
	  (make-*defvar-specification
	    :name variable :initial-value-form initial-value :vp-set-name vp-set-name :in-vp-set-definition-p nil
	    :initial-value-function init-function :proclaimed-type type))
	(old-*defvar-specification-index
	  (position variable *all-*defvar-specifications* :test #'eq :key #'*defvar-specification-name))
	(vp-set (symbol-value vp-set-name)))
    (*deallocate-*defvar-if-possible variable)
    (when (and (*lisp-runnable-p) (vp-set-instantiated-p (symbol-value vp-set-name)))
      (if init-function
	  (funcall (if (not (compiled-function-p (symbol-function init-function))) (compile init-function) init-function))
	  (allocate-*defvar variable initial-value vp-set)))
    (if old-*defvar-specification-index
	(setf (nth old-*defvar-specification-index *all-*defvar-specifications*) new-*defvar-specification)
	(setq *all-*defvar-specifications* (nconc *all-*defvar-specifications* (list new-*defvar-specification)))))
  (values))

(defun delete-old-*defvar-definition (name)
  (let ((specification (find name *all-*defvar-specifications* :key #'*defvar-specification-name)))
    (if (and specification (*defvar-specification-initial-value-function specification))
	(fmakunbound (*defvar-specification-initial-value-function specification))))
  (setq *all-*defvar-specifications*
	(delete name *all-*defvar-specifications* :test #'eq :key #'*defvar-specification-name))
  (*deallocate-*defvar-if-possible name)
  (makunbound name))


(defun allocate-*defvar (symbol &optional initial-value vp-set &aux return-pvar)
  ;; Allocate a pvar, and make the symbol naming the
  ;; *DEFVAR have as its value that pvar.
  (setq return-pvar (allocate-*defvar-1 symbol initial-value vp-set))
  (setf (pvar-name return-pvar) symbol)
  (setf (symbol-value symbol) return-pvar)
  return-pvar)


(defun allocate-*defvar-1 (symbol initial-value vp-set)
  ;; Turn on the VP SET to which this *defvar belongs,
  ;; and use ALLOCATE!! to create an initialize a pvar
  ;; which will be the value of this *DEFVAR.
  (let ((canonical-pvar-type (or (get symbol 'type) '(pvar *)))
	return-pvar)
    (let ((canonical-pvar-type-with-integer-sizes
	    (non-lexical-canonical-pvar-type-with-numeric-lengths-from-canonical-pvar-type canonical-pvar-type)))
      (*with-vp-set vp-set
	(*all
	  (setq return-pvar (allocate!! (eval initial-value) nil canonical-pvar-type-with-integer-sizes))
	  (setq *all-allocate!!-pvars* (delete return-pvar *all-allocate!!-pvars* :test #'eq))))
      return-pvar)))


;;; *DEALLOCATE
;;; *DEALLOCATE-*DEFVARS

;;;; DEALLOCATION OF PVARS


(defun *deallocate (pvar)

  (new-pvar-check pvar '*deallocate)

  (when (not (member pvar *all-allocate!!-pvars* :test #'eq))
    (cerror
      "Return from *deallocate without doing anything"
      "You are trying to use *deallocate on a pvar, ~S, that was not allocated using allocate!!,~@
       or has already been deallocated.  If you want to remove a pvar defined with *defvar~@
       use the *deallocate-*defvars function.
      "
      pvar
      )
    (return-from *deallocate nil)
    )

  (*deallocate-internal pvar :allocate!!)

  )



(defvar smashed-location "THIS PVAR HAS BEEN DEALLOCATED.  YOU SHOULD NOT BE REFERENCING IT")

(defun smash-deallocated-pvar (pvar)
  ;; smash the pvar so that it doesn't get reused.
  (set-pvar-length pvar 0)
  (setf (pvar-type pvar) :general)
  (setf (pvar-location pvar) smashed-location))

(defvar
  smashed-location2
  "This pvar structure was allocated with ALLOCATE!! but was reclaimed by a *COLD-BOOT.  You should not access it."
  )

(defun *defvar-*cold-boot-initialization ()

  (dolist (pvar *all-allocate!!-pvars*)
    (setf (pvar-location pvar) smashed-location2)
    (set-pvar-length pvar smashed-location2)
    (setf (pvar-type pvar) smashed-location2))

  (setq *all-allocate!!-pvars* nil)

  ;; now must recreate all *DEFVARS.  
  (allocate-*defvars-for-*cold-boot))


(defun allocate-*defvars-for-*cold-boot ()
  (let ((bad-*defvar-specifications nil))
    (mapc
      #'(lambda (*defvar-specification)
	  (let* ((pvar-symbol (*defvar-specification-name *defvar-specification))
		 (pvar-initial-value-form (*defvar-specification-initial-value-form *defvar-specification))
		 (pvar-vp-set-name (*defvar-specification-vp-set-name *defvar-specification))
		 (initial-value-function (*defvar-specification-initial-value-function *defvar-specification))
		 (proclaimed-type (*defvar-specification-proclaimed-type *defvar-specification))
		 (current-type (if initial-value-function (get pvar-symbol 'type))))
	    (when (or (not (boundp pvar-vp-set-name))
		      (not (symbolp pvar-vp-set-name))
		      (not (vp-set-p (symbol-value pvar-vp-set-name)))
		      (and (not (eq pvar-vp-set-name '*default-vp-set*))
			   (null (position pvar-vp-set-name *all-def-vp-sets* :test #'eq :key #'vp-set-name))))
	      (cerror "Blow away the *defvar definition and continue initializing other *defvars"
		      "The *defvar ~S has a Vp Set named ~S, which is not now an existing Vp Set defined with DEF-VP-SET."
		      pvar-symbol pvar-vp-set-name)
	      (push *defvar-specification bad-*defvar-specifications))
	    #+*LISP-SIMULATOR
	    (progn proclaimed-type current-type)
	    #+*LISP-HARDWARE
	    (when initial-value-function
	      (unless (equal proclaimed-type current-type)
		(warn "The proclaimed type for ~S has been changed since the *defvar ~S was evaluated.~%~@
                       The initialization function is being recompiled."
		      pvar-symbol pvar-symbol)
		(compile
		  (eval
		    `(defun ,initial-value-function () 
		       ,(if (eq pvar-vp-set-name '*default-vp-set*)
			    `(*all (setq ,pvar-symbol
					 (allocate!! ,pvar-initial-value-form ',pvar-symbol ',current-type t))
				   nil)
			    `(*with-vp-set ,pvar-vp-set-name
			       (*all (setq ,pvar-symbol
					   (allocate!! ,pvar-initial-value-form ',pvar-symbol ',current-type t))
				     nil))))))
		(setf (*defvar-specification-proclaimed-type *defvar-specification) current-type)))
	    ;; JP 5/9/89.  Make *defvars in non-instantiated Vp Sets be unbound
	    (if (vp-set-instantiated (symbol-value pvar-vp-set-name))
		(allocate-*defvar-with-errors-trapped
		  pvar-symbol pvar-initial-value-form (symbol-value pvar-vp-set-name) initial-value-function
		  )
		(makunbound pvar-symbol)
		)))
;		(when (vp-set-instantiated (symbol-value pvar-vp-set-name))
;		  (allocate-*defvar-with-errors-trapped
;		    pvar-symbol pvar-initial-value-form (symbol-value pvar-vp-set-name) initial-value-function))))
      *all-*defvar-specifications*)
    (mapc #'(lambda (*defvar-specification)
	      (delete-old-*defvar-definition (*defvar-specification-name *defvar-specification)))
	  bad-*defvar-specifications)))



(defun *deallocate-*defvars (&rest pvar-names)

  (when (or (equal pvar-names '(:all)) (equal pvar-names '(:all-noconfirm)))
    (if (equal pvar-names '(:all))
	(cerror "Do it anyway" "Deleting all *DEFVARs may cause a library package to fail.~@
                            If you are certain you want to do this, hit the Resume key."))
    (setq pvar-names (mapcar #'*defvar-specification-name *all-*defvar-specifications*))
    )

  (assert (every #'symbolp pvar-names) ()
	  "The arguments to *deallocate-*defvars must be the NAMES of pvars."
	  )

  (cond
    (pvar-names
     ;; handle the pvars specified by the user
     (dolist (pvar-name pvar-names)
       (let ((pvar-info (find pvar-name *all-*defvar-specifications* :test #'eq :key #'*defvar-specification-name)))
	 (cond
	   ((null pvar-info)
	    (warn "The symbol ~S, having value ~S, is not currently defined as a *DEFVAR."
		  pvar-name
		  (if (boundp pvar-name) (symbol-value pvar-name) "#<UNBOUND>")
		  )
	    (cond
	      ((y-or-n-p "Make it unbound anyway and continue")
	       (makunbound pvar-name)
	       )
	      ((y-or-n-p "Ignore it and continue"))
	      ))
	   (t (delete-old-*defvar-definition pvar-name))
	   ))))

    ;; do this only if there are some *defvars around to delete

    #+symbolics
    ((and *all-*defvar-specifications*)
     ;; user didn't specify any pvars, go through all of them, prompting for each.
     (let ((menu (tv:make-window
		   'tv:momentary-multiple-menu
		   :label (zl:string "Choose pvars to delete")
		   :item-list
		   (mapcar
		     #'(lambda (defvar)
			 (let* ((pvar-name (*defvar-specification-name defvar))
				(pvar-initial-value (*defvar-specification-initial-value-form defvar))
				(pvar-declaration (cadr (get pvar-name 'type)))
				)
			   `(,(let ((*print-pretty* nil))
				(format nil "  ~(~20S ~25S ~40A~)"
					pvar-name (or pvar-initial-value 'none)
					#+*LISP-HARDWARE
					(slc::pretty-pvar-element-type-declaration pvar-declaration)
					#+*LISP-SIMULATOR
					pvar-declaration
					))
			     :value ,pvar-name
			     :font fonts:cptfontb
			     )))
		     *all-*defvar-specifications*
		     ))))
       (zl:send menu :expose-near '(:mouse))
       (setq pvar-names (zl:send menu :choose))
       (zl:send menu :deactivate)
       (if pvar-names (apply #'*deallocate-*defvars pvar-names))
       ))

    (*all-*defvar-specifications*
     ;; user didn't specify any pvars, go through all of them, prompting for each.
     (let ((list ()) (*print-pretty* nil))
       (format *query-io* "~%Deallocate which pvars?~%~20A ~25A ~25A" 
	       "Pvar Name" "Initial Value" "Declaration")
       (dolist (defvar *all-*defvar-specifications*)
	 (let* ((pvar-name (*defvar-specification-name defvar))
		(pvar-initial-value (*defvar-specification-initial-value-form defvar))
		(pvar-declaration (get pvar-name 'type))
		)
	   (when (y-or-n-p
		 "~(~20S ~25S ~25A~)"
		 pvar-name (or pvar-initial-value 'none) (or pvar-declaration '(pvar *))
		 )
	     (push pvar-name list)
	     )))
       (if list (apply #'*deallocate-*defvars list))
       ))

    )

  nil

  )


(defun allocate-*defvars-for-vp-set (vp-set)
  (let ((vp-set-name (vp-set-name vp-set)))
    (mapc
      #'(lambda (*defvar-specification)
	  (when (eq vp-set-name (*defvar-specification-vp-set-name *defvar-specification))
	    (let ((in-vp-set-definition-p (*defvar-specification-in-vp-set-definition-p *defvar-specification)))
	      (*defvar-1
		(*defvar-specification-name *defvar-specification)
		(*defvar-specification-initial-value-form *defvar-specification)
		vp-set-name
		)
	      (setf (*defvar-specification-in-vp-set-definition-p
		      (find (*defvar-specification-name *defvar-specification) *all-*defvar-specifications*
			    :test #'eq :key #'*defvar-specification-name
			    ))
		    in-vp-set-definition-p
		    ))))
      *all-*defvar-specifications*
      )))


(defun deallocate-pvars-for-vp-set

       (vp-set
	&key
	(delete-*defvars t)
	(deallocate-*defvar-memory t)
	(delete-allocate!!-pvars t)
	)
  
  (when delete-allocate!!-pvars
    (mapc
      #'(lambda (pvar)
	  (when (eq (pvar-vp-set pvar) vp-set) (*deallocate-internal pvar :allocate!!))
	  )
      *all-allocate!!-pvars*
      ))

  (when deallocate-*defvar-memory
    (mapc
      #'(lambda (*defvar-specification)
	  (when (eq (*defvar-specification-vp-set-name *defvar-specification) (vp-set-name vp-set))
	    (when (and (boundp (*defvar-specification-name *defvar-specification))
		       (pvarp (eval (*defvar-specification-name *defvar-specification)))
		       )
	      (*deallocate-internal (eval (*defvar-specification-name *defvar-specification)) :*defvar)
	      (makunbound (*defvar-specification-name *defvar-specification))
	      )))
      *all-*defvar-specifications*
      ))

  (when delete-*defvars
    (mapc
      #'(lambda (*defvar-specification)
	  (when (eq (*defvar-specification-vp-set-name *defvar-specification) (vp-set-name vp-set))
	    (setq *all-*defvar-specifications*
		  (delete *defvar-specification *all-*defvar-specifications* :test #'eq)
		  )
	    (when (*defvar-specification-name *defvar-specification)
	      (makunbound (*defvar-specification-name *defvar-specification))
	      )))
      *all-*defvar-specifications*
      ))

  )


(defun *defvar-pvar-p (pvar)
  (dolist (spec *all-*defvar-specifications*)
    (let ((pvar-symbol (*defvar-specification-name spec)))
      (when (and (boundp pvar-symbol) (eq pvar (symbol-value pvar-symbol)))
	(return-from *defvar-pvar-p t)
	)))
  nil
  )


(defun allocated-pvar-p (pvar)
  (simple-pvar-argument!! pvar)
  (if (or (*defvar-pvar-p pvar)
	  (member pvar *all-allocate!!-pvars* :test #'eq)
	  )
      :heap
      (dolist (stack-pvar *temp-pvar-original-list*)
	(if (eq stack-pvar pvar)
	    (return :stack)
	    (if (eq stack-pvar (car *temp-pvar-list*)) (return nil))
	    ))))



(defun *deallocate-*defvar-if-possible (variable)
  (when (and (*lisp-runnable-p)
	     (boundp variable)
	     (pvarp (symbol-value variable))
	     (integerp (pvar-location (symbol-value variable)))
	     (plusp (pvar-location (symbol-value variable)))
	     )
    (*deallocate-internal (symbol-value variable) :*defvar)
    ))


#+*LISP-HARDWARE
(defun deallocate-for-paris (pvar)
  (cm:deallocate-heap-field (pvar-location pvar)))


#+*LISP-HARDWARE
(defun allocate-*defvar-with-errors-trapped (pvar-symbol pvar-initial-value-form pvar-vp-set init-function)
  (cmi::with-proceed-cases
    ((error ("Skip allocating the ~s *DEFVAR." pvar-symbol)
	    (makunbound pvar-symbol))
     (error
       ("Skip allocating the ~s *DEFVAR and flush it forever." pvar-symbol)
       ;; had an error allocating, flush it
       (makunbound pvar-symbol)
       (setq *all-*defvar-specifications*
	     (delete pvar-symbol *all-*defvar-specifications* :test #'eq :key #'*defvar-specification-name))))
    (if init-function 
	(funcall init-function)
	(allocate-*defvar pvar-symbol pvar-initial-value-form pvar-vp-set))))


#+*LISP-SIMULATOR
(defun allocate-*defvar-with-errors-trapped (pvar-symbol pvar-initial-value-form pvar-vp-set init-function)
  (declare (ignore init-function))
  (with-all-errors-trapped
    (allocate-*defvar pvar-symbol pvar-initial-value-form pvar-vp-set)
    (progn
      (format t "An error occurred while evaluating ~S, the initial value for for the *defvar named ~S~%"
	      pvar-initial-value-form pvar-symbol
	      )
      (format t "Your options are to skip allocating the *defvar this time and continue, ~@
                           or to skip the allocation and destroy the *defvar definition and continue, ~@
                           or to Abort and try to fix the problem"
	      )
      (cond
	((y-or-n-p "Skip allocating the *defvar and continue? ")
	 (makunbound pvar-symbol)
	 )
	((y-or-n-p "Skip allocation, destroy the *defvar and continue? ")
	 (makunbound pvar-symbol)
	 (setq *all-*defvar-specifications*
	       (delete pvar-symbol *all-*defvar-specifications* :test #'eq :key #'*defvar-specification-name)
	       ))
	(t (error "You will have to abort to top level"))
	))))


#+*LISP-HARDWARE

(defun *deallocate-internal (pvar type)

  (when (not (integerp (pvar-location pvar)))
    (cerror
      (ecase type
	(:allocate!! "Remove this pvar from *Lisp's list of allocated pvars anyway")
	(:*defvar "Return from *deallocate-internal without doing anything")
	)
      "You are trying to use *deallocate on a pvar, ~S, that presumably has already
       been deallocated or was never actually allocated.
      "
      pvar
      )
    (ecase type
      (:allocate!! (setq *all-allocate!!-pvars* (delete pvar *all-allocate!!-pvars* :test #'eq)))
      (:*defvar nil)
      )
    (smash-deallocated-pvar pvar)
    (return-from *deallocate-internal nil)
    )

  (when (*lisp-runnable-p)
    (deallocate-for-paris pvar)
    )

  (smash-deallocated-pvar pvar)

  (ecase type
    (:allocate!! (setq *all-allocate!!-pvars* (delete pvar *all-allocate!!-pvars* :test #'eq)))
    (:*defvar nil)
    )

  nil

  )


#+*LISP-SIMULATOR

(defun *deallocate-internal (pvar type)
  (ecase (pvar-class pvar)
    (:general
      (setf (vp-set-heap-pvar-arrays (pvar-vp-set pvar))
	    (cons (pvar-location pvar) (vp-set-heap-pvar-arrays (pvar-vp-set pvar)))
	    ))
    (:array (*map-array #'(lambda (element-pvar) (*deallocate-internal element-pvar :allocate!!)) pvar))
    (:structure (*map-structure #'(lambda (slot-pvar) (*deallocate-internal slot-pvar :allocate!!)) pvar))
    )
  (smash-deallocated-pvar pvar)
  (ecase type
    (:allocate!! (setq *all-allocate!!-pvars* (delete pvar *all-allocate!!-pvars* :test #'eq)))
    (:*defvar nil)
    )
  )


(defun set-pvar-length (pvar value)
  #+*LISP-HARDWARE
  (setf (pvar-length pvar) value)
  #+*LISP-SIMULATOR
  (progn pvar value nil)
  )

