;;; -*- Mode: LISP; Syntax: Common-lisp; Package: User; Base: 10 -*-


(in-package 'user)

(export '(ms msc))

(defvar *all-possible-lisp-extensions* '(".l" ".lsp" ".lisp" ""))

(defvar *alias-prefix-char-list* nil)

(defvar *alias-definition-list* nil)

;;; this is where the make-system code looks for system definition
;;; files.  Change this as appropriate to your system.

(defvar *defsystem-search-path* '(
        "./"
        "/u/massar/starlib/"
     ))

(defvar *defsystem-suffix* ".sys")

#+(OR :CCL :ALLEGRO)
(defvar *binary-lisp-extension* ".fasl")
#+SYMBOLICS
(defvar *binary-lisp-extension* ".bin")
#+KCL
(defvar *binary-lisp-extension* ".o")
#+(OR EXCL CMU)
(defvar *binary-lisp-extension* ".fasl")
#+(AND LUCID UNIX SUN SPARC)
(defvar *binary-lisp-extension* ".sbin")
#+(OR (AND LUCID SUN3) (AND LUCID MC68000))
(defvar *binary-lisp-extension* ".2bin")
#+(AND LUCID VAX)
(defvar *binary-lisp-extension* ".vbin")
#+(AND LUCID APOLLO)
(defvar *binary-lisp-extension* ".lbin")


(defmacro errmsg (format-string &rest format-args)
  `(progn
     (terpri *error-output*)
     (format *error-output* ,format-string ,@format-args)
    ))


;;; parameters for a particular make-system call

(defvar *changed-only* nil)


;;; set up mechanism so that 'logical names' (aliases) can be
;;; defined and undefined.  Aliases are recognized by beginning
;;; with a specific user-defined character.  This character is
;;; specified using define-alias-prefix-char.


;;; Actual aliases are defined using define-alias.
;;; Thus
;;; (define-alias-prefix-char #\&)
;;; (define-alias "HOST" "godot:")
;;;
;;; would cause the string "&HOST" to be interpreted as "godot:"
;;; in the context of the make-system file where appropriate.
;;;

(defun define-alias-prefix-char (char)
  (if (characterp char)
      (setq *alias-prefix-char-list*
	    (union (list char) *alias-prefix-char-list*))
      (error "DEFINE-ALIAS-PREFIX-CHAR:  Argument is not a character")
   ))


(defun undefine-alias-prefix-char (char)
  (setq *alias-prefix-char-list* (remove char *alias-prefix-char-list*))
 )


(defun define-alias (alias real-string)
  (if (or (null (stringp alias)) (null (stringp real-string)))
      (error "DEFINE-ALIAS:  Arguments must be strings.")
      (push (list alias real-string) *alias-definition-list*)
   ))


(defun undefine-alias (alias)
  (setq *alias-definition-list*
	(remove-if #'(lambda (assoc-pair) (equal (car assoc-pair) alias))
		   *alias-definition-list*
	 )))


(defun convert-alias-to-real-string (alias)
  (let ((real-string
	  (cadr (assoc alias *alias-definition-list* :test #'equal))))
    (if (null real-string)
	(error "CONVERT-ALIAS-TO-REAL-STRING: Alias not defined.")
	real-string
     )))


(defun has-alias-prefix-char (string)
  (if (zerop (length string))
      nil
      (member (char string 0) *alias-prefix-char-list*)
   ))


(defun possibly-convert-string-using-alias-info (string)
  (if (has-alias-prefix-char string)
      (convert-alias-to-real-string (subseq string 1))
      string
   ))


(defun combine-string-list-to-form-full-file-path (string-list)
  (apply
    #'concatenate
    (cons 'string
	  (mapcar
	    #'(lambda (string)
		(if (null (stringp string))
		    (error "All components must be strings")
		    (possibly-convert-string-using-alias-info string)
		 ))
	    string-list
	   ))))





(defun source-code-file (base-file-name)

  ;; Given a file name, find a file with that name and some
  ;; lispy extension, as defined by *all-possible-lisp-extensions*

  (mapc
    #'(lambda (extension)
        (let ((full-name (concatenate 'string base-file-name extension)))
          (and (probe-file full-name) (return-from source-code-file full-name))
         ))
    *all-possible-lisp-extensions*
   )
  (return-from source-code-file nil)
 )


(defun object-code-file (base-file-name)
  (concatenate 'string base-file-name *binary-lisp-extension*)
 )


(defun make (forms filelist &key changed-only)
       
  ;; consider each file in turn.  If it needs to be compiled,
  ;; compile it and then load it, otherwise just load it.
  ;; If changed-only is true, only compile and load in those
  ;; files whose source has been changed since the object
  ;; file was created.

  ;; evaluate the form paired with each file before compiling
  ;; the file.
       
 (if (or (null (listp filelist))
	 (null (every #'stringp filelist)))
     (progn
       (errmsg "~%Make expects a list of strings~%")
       (errmsg "Actual argument: ~S~%" filelist)
       (return-from make nil)
      ))
       
 (let ((*load-verbose* nil))
   (mapc
     #'(lambda (form file)
         (eval form)
	 (let ((source-file (source-code-file file))
	       (object-file (object-code-file file))
	       (changed nil)
	       )
	   (when (null (probe-file source-file))
	     (errmsg "~%File ~S does not exist~%" file)
	     (errmsg "Aborting make...~%")
	     (return-from make nil)
	     )
	   (when (or (null (probe-file object-file))
		     (> (file-write-date source-file)
			(file-write-date object-file)
			))
	     #+KCL
	     (format t "Compiling ~A~%" source-file)
	     (compile-file source-file :output-file object-file)
	     (setq changed t)
	     )
	   (when (or changed (not changed-only))
	     (format t "~%Loading ~S~%" object-file)
	     (load object-file)
	     )))
     forms
     filelist
     ))

 )



(defun find-defsystem (system-name)

  ;; given the name of a system to make, figure out where
  ;; the file describing how to make the system is located.

  (mapc
    #'(lambda (searchpath)
	(let ((pathname 
		(concatenate 'string 
		   searchpath system-name *defsystem-suffix*)))
	  (when (probe-file pathname)
	    (return-from find-defsystem pathname)
	   )
	 ))
    *defsystem-search-path*
   )
  nil
 )


(defun check-file-specification (file-spec)
  (if (or (null (consp file-spec))
	  (not (every #'stringp file-spec)))
      (errmsg "MAKE-SYSTEM-FILE-LIST: Bad file specification: ~S" file-spec)
      (combine-string-list-to-form-full-file-path file-spec)
   ))


;;; The user puts into his .sys file a call to this function,
;;; giving as arguments lists of strings which define logical
;;; pathnames for his various files.  Previously in his .sys file
;;; the user has defined aliases to be used for logical pathnames
;;; (see above)

(defun make-system-file-list (&rest compile-specifications)
  (dolist (c compile-specifications)
    (when (null (consp c))
      (error "One of the compile specifications is not a list...")
     ))
  (let* ((forms-to-evaluate
           (mapcar #'(lambda (x) (if (stringp (car x)) nil (car x)))
                   compile-specifications))
         (file-specifications
           (mapcar #'(lambda (x) (if (stringp (car x)) x (cadr x)))
                   compile-specifications))
         (defsystem-files
           (mapcar #'check-file-specification file-specifications))
        )
    (make forms-to-evaluate defsystem-files :changed-only *changed-only*)
   ))


(defun make-system (system-name &key changed-only)
       
  ;; locate and load the system definition file

  (when (not (stringp system-name))
    (errmsg "Argument to make-system is not a string: ~S~%" system-name)
    (return-from make-system nil)
   )
       
  (let ((full-system-pathname (find-defsystem system-name)))

   (when (null full-system-pathname)
     (errmsg "Cannot find defsystem ~S~%" system-name)
     (return-from make-system nil)
    )

   (setq *changed-only* changed-only)

   (let ((*load-verbose* nil))
     (format t "~%Loading system definition: ~A~%" full-system-pathname)
     (load full-system-pathname)
    )

  ))



(defun ms (name &key changed-only)
   (make-system name :changed-only changed-only)
  )


(defun msc (name)
  (make-system name :changed-only t)
 )
