;;; -*- SYNTAX: COMMON-LISP; MODE: LISP; BASE: 10; PACKAGE: *lisp-i; -*-


(IN-PACKAGE "*LISP-I")


;;;; This code is excerpts from the example code used in the *Lisp
;;;; Dictionary.


;;; ********************************************************************************

;;; ALIAS!!

(defstruct history-struct
  (date 0 :type (unsigned-byte 8)))

(*defstruct patient
  (id-no 0 :type (unsigned-byte 8))
  (doctor 0 :type (unsigned-byte 8))
  (sick-p t :type boolean)
  (case-history nil :type (pvar (array history-struct (100)))))

(defun modify-patient-slot (slot-pvar value)
  (declare (type (field-pvar *) slot-pvar value))
  nil
  (*set slot-pvar value))

(defun in-error ()
  (*let ((ellen (make-patient!!)))
    (declare (type (pvar patient) ellen))
    (modify-patient-slot (patient-sick-p!! ellen) nil!!)
    (ppp (patient-sick-p!! ellen) :end 5)))

(defun correct ()
  (*let ((ellen (make-patient!!)))
    (declare (type (pvar patient) ellen))
    (modify-patient-slot (alias!! (patient-sick-p!! ellen)) nil!!)
    (ppp (patient-sick-p!! ellen) :end 5)))
  
;;; ********************************************************************************

;; *ALL

;; example showing the use of *all in resetting the context
;; of a Vp Set to all its processors.


(def-vp-set fred '(16384))
(def-vp-set wilma '(8192))

(*with-vp-set fred
  (*when (<!! (self-address!!) (!! 100))
    (format t "~%In FRED, # active procs should be 100, is: ~d" (*sum (!! 1)))
    (*with-vp-set wilma
      (format t "~%In WILMA, # active procs should be 8192, is ~d" (*sum (!! 1)))
      (*with-vp-set fred
	(format t "~%In FRED, the # active procs should still be 100, is ~d" (*sum (!! 1)))
	(*all
	 (format t "~%In FRED, the # active procs should now be 16384, is ~D" (*sum (!! 1)))))
      (format t "~%In WILMA, # active procs should still be 8192, is: ~d" (*sum (!! 1))))
    (format t "~%In FRED, # active procs should now be 100, is: ~d" (*sum (!! 1)))))

;; Without using *all within a *with-vp-set form, the
;; user cannot in general be sure that all the
;; processors of the newly selected Vp Set are active.

;; Note the use of the *Lisp idiom (*sum (!! 1)) to determine
;; the number of active processors.

;;; ********************************************************************************

;; ALLOCATE!!

;; Allocate!! is generally not called at top level.
;; An example of how one might use allocate!! in the
;; middle of a program.

(def-vp-set fred '(8192))
(defvar list-of-pvars nil)

(defun fred-function ()
  (*with-vp-set fred
    (when (i-really-need-another-pvar)
      (push (allocate!! (!! 0) nil '(pvar (unsigned-byte 32))) list-of-pvars)
      )
    (do-something-with-my-list-of-pvars)
    ))

;; One the program is finished and another *cold-boot
;; is done, it is necessary to reset list-of-pvars
;; to NIL, because the pvar structures on the list
;; have all been deallocated and therefore are
;; invalid.  Any attempt to use those deallocated
;; pvars is an error and will generally result
;; in an error message.

;;; ********************************************************************************

;; ALLOCATE-PROCESSORS-FOR-VP-SET

;; allocate-processors-for-vp-set may be used in conjunction with
;; deallocate-processors-for-vp-set to cause a vp set to have
;; different sizes at different times in a program without
;; having to *cold-boot.  For instance, the previous example
;; could be made to process many disk files without *cold-booting
;; each time as follows:

(def-vp-set disk-data nil
  :*defvars ((disk-data-pvar nil nil (pvar single-float))))

(defun top-level-function (&rest diskfiles)
  (*cold-boot)
  (dolist (diskfile diskfiles)
    (let ((number-of-elements (read-number-of-elements-from-disk diskfile)))
      (allocate-processors-for-vp-set disk-data
				      (list (next-power-of-two->= number-of-elements)))
      (let ((array-of-data (read-data-from-disk diskfile)))
	(array-to-pvar array-of-data disk-data-pvar :cube-address-end number-of-elements)
	(process-disk-data-in-cm disk-data)
	)
      (deallocate-processors-for-vp-set disk-data)
      )))


;;; ********************************************************************************

;; AND!!

;; Example where using AND!! may cause a non-intuitive
;; result because of its deselection properties.


(ppp (and!! (evenp!! (self-address!!))
	    (<!! (scan!! (self-address!!) '+!!) (!! 3)))
     :end 8)

T NIL T NIL NIL NIL NIL NIL

(ppp (and!! (<!! (scan!! (self-address!!) '+!!) (!! 3))
	    (evenp!! (self-address!!)))
     :end 8)

T NIL NIL NIL NIL NIL NIL NIL

;; In the first case, the scan is done only in the even
;; processors.  In the second case, the scan is done
;; in all the processors, resulting in different values
;; than in the first case.  This is the result of AND!!
;; deselecting those processors which fail any clause
;; before executing the next clause.

;; One can avoid this in the following manner:

(*let ((b1 (evenp!! (self-address!!)))
       (b2 (<!! (scan!! (self-address!!) '+!!) (!! 3)))
       )
  (declare (type boolean-pvar b1 b2))
  (and!! b1 b2)
  )

;;; ********************************************************************************

;; ARRAY-TO-PVAR

;; PERFORMANCE NOTE

;; Fastest when pvars of a specific non-aggregate type are used.
;; Slower when general pvars are used.
;; Very slow if aggregate pvars are used as illustrated below.

;; Example showing how to move aggregate data efficiently into the CM.
;; Using a structure of arrays, and moving each array separately
;; is much faster than having an array of structures and moving the
;; array into the CM in one call.

(*defstruct foo
  (a 0 :type t :cm-type (pvar (unsigned-byte 32)))
  (b 0.0 :type t :cm-type (pvar single-float))
  )

(*proclaim '(type (pvar foo) a-foo-pvar))
(*defvar a-foo-pvar)

;; This is very fast, although it is more awkward.

(defvar a-foo (make-foo))

(defun init-a-foo ()
  (setf (foo-a a-foo) (make-array *number-of-processors-limit* :element-type '(unsigned-byte 32)))
  (setf (foo-b a-foo) (make-array *number-of-processors-limit* :element-type 'single-float))
  )

(defun move-a-foo-data-from-front-end-to-cm ()
  (array-to-pvar (foo-a a-foo) (alias!! (foo-a!! a-foo-pvar)))
  (array-to-pvar (foo-b a-foo) (alias!! (foo-b!! a-foo-pvar)))
  )

;; This is very slow, although it is more straightforward.

(defvar a-foo-array (make-array *number-of-processors-limit* :element-type 'foo))

(defun init-a-foo-array ()
  (dotimes (j *number-of-processors-limit*)
    (setf (aref a-foo-array j) (make-foo))
    ))
  
(defun move-a-foo-array-data-from-front-end-to-cm ()
  (array-to-pvar a-foo-array a-foo-pvar)
  )

;;; ********************************************************************************

;;; ARRAY-TO-PVAR-GRID

;;; An example illustrating the use of ARRAY-TO-PVAR-GRID
;;; to extract a subarray from a front-end array and store
;;; it into a subgrid of the CM.


(*cold-boot :initial-dimensions '(128 128))

(defparameter an-array (make-array '(5 5) :element-type 'single-float :initial-element 0.0))

(*proclaim '(type single-float-pvar xyzzy))
(*defvar xyzzy)

;;; This call transfers the 4x4 sub array of an-array whose corners are

;;;     (1 1) (4 1)
;;;     (1 4) (4 4)

;;; to the 4x4 subgrid of the pvar XYZZY whose corners are

;;;     (2 3) (6 3)
;;;     (2 7) (6 7)

(array-to-pvar-grid an-array xyzzy :array-offset '(1 1) :grid-start '(2 3))

;;; ********************************************************************************

;; AREF!!

;; AREF!! may be used with non-constant indices.

(*proclaim '(type (vector-pvar single-float 2) xyzzy))
(*defvar xyzzy)

(defun non-constant-indexing-example ()
  (*setf (aref!! xyzzy (!! 0)) (!! 1.0))
  (*setf (aref!! xyzzy (!! 1)) (!! -1.0))
  (ppp (aref!! xyzzy (if!! (evenp!! (self-address!!)) (!! 0) (!! 1))) :end 8)
  )

1.0 -1.0 1.0 -1.0 1.0 -1.0 1.0 -1.0

;; In general, especially for large arrays, the CM2
;; implementation of non-constant indexing can be
;; very slow.  See *sideways-array and sideways-aref!!
;; for a means of utilizing the CM2 architecture to
;; do fast non-constant indexing into arrays.

;; Example showing that ALIAS!! is not necessary in most cases.

;; Many people are tempted to write

(*set (the single-float-pvar x)
      (+!! (alias!! (aref!! xyzzy (!! 0))) (alias!! (aref!! xyzzy (!! 1)))))

;; to avoid having a copy of the data in elements 0 and 1 of xyzzy being
;; made.  However, as long as the *Lisp compiler is compiling
;; the code, the following code

(*set (the single-float-pvar x)
      (+!! (aref!! xyzzy (!! 0)) (aref!! xyzzy (!! 1))))

;; is equivalent and will not result in any temporary space
;; being used.  In general there is no need to use alias!!
;; when performing array accessing except in certain special
;; cases which are disucssed under the dictionary entry for
;; ALIAS!!.

;;; ********************************************************************************

;;; COERCE!!

;; Parallel type coercion on the argument pvar

;; 8 bit interger to 16 bit interger pvar

(*proclaim '(type (pvar (unsigned-byte 8)) data-8))
(*defvar data-8 (random!! (!! 20)))

(*proclaim '(type (pvar (signed-byte 16)) data-16))
(*defvar data-16)

(*set data-16 (coerce!! data-8 '(pvar (signed-byte 16))))

;; Interger to float pvar

(*proclaim '(type single-float-pvar data-sf))
(*defvar data-sf)

(*set data-sf (coerce!! data-16 '(pvar single-float)))

;; Single float to double float pvar

(*proclaim '(type double-float-pvar data-df))
(*defvar data-df)

(*set data-df (coerce!! data-sf '(pvar double-float)))

;; Single float to complex, where significand and exponent are double floats

(*proclaim '(type (pvar (complex double-float)) data-df-cmplx))
(*defvar data-df-cmplx)
				     
(*set data-df-cmplx (coerce!! data-sf '(pvar (complex double-float))))

;; Changing the size of a complex pvar, from single to double float

(*proclaim '(type single-complex-pvar data-sf-cmplx))
(*defvar data-sf-cmplx (complex!! (!! 1.0) (!! -1.0)))

(*set data-df-cmplx (coerce!! data-sf-cmplx '(pvar double-complex)))

;; Integer to character pvar, could use int-char!!, but here is the version
;; for coerce!!

(*proclaim '(type character-pvar data-char))
(*defvar data-char)

(*set data-16 (random!! (!! 65)))
(*set data-char (coerce!! data-16 'character-pvar))

;; String char array pvars of length 1 can be converted to character pvars

(*proclaim '(type (pvar (array string-char (1))) data-string-char))
(*defvar data-string-char (!! "C"))

(*set data-char (coerce!! data-string-char 'character-pvar))

;; Any pvar, except for an array or structure pvar can be coerced into a general pvar

(*proclaim '(type (pvar t) data-general))
(*defvar data-general)

(*set data-general (coerce!! data-sf-cmplx '(pvar t)))

;; It is possible to coerce the elements of an array pvar into another type.

(*proclaim '(type (pvar (array single-float (20))) data-array-sf))
(*defvar data-array-sf (make-array!! '(20)
				  :initial-element (random!! (!! 2.0))
				  :element-type 'single-float))

(*proclaim '(type (pvar (array double-float (20))) data-array-df))
(*defvar data-array-df)

(*set data-array-df (coerce!! data-array-sf '(pvar (array double-float (20)))))

;;; ********************************************************************************

;; *COLD-BOOT

;; Example showing how one might call *cold-boot inside
;; a top-level function.

(defun top-level ()
  (initialize-all-non-cm-state)
  (*cold-boot)
  (initialize-all-cm-state)
  (run-my-program)
  )

;;; ********************************************************************************

;; COND!!

;; cond!! example showing how the implicit (t!! nil!!) clause
;; causes the *Lisp compiler not to *compile certain
;; innocent looking cond!! forms

(*proclaim '(type single-float-pvar x y))
(*defvar x)
(*defvar y)

(defun does-not-compile ()
  (*set (the single-float-pvar x)
	(cond!!
	  ((minusp!! (the single-float-pvar y)) (!! -1.0))
	  ((plusp!! (the single-float-pvar y)) (!! 1.0))
	  )))

;;  Verbose: While compiling NIL!!:
;;  The expression (*SET-1 (THE SINGLE-FLOAT-PVAR X) (COND!! (# #) (# #))) is not compiled because
;;  IF!! does not understand how to compile pvars with element-type boolean, when combined with an FLOAT pvar.

(defun does-compile ()
  (*set (the single-float-pvar x)
	(cond!!
	  ((minusp!! (the single-float-pvar y)) (!! -1.0))
	  ((plusp!! (the single-float-pvar y)) (!! 1.0))
	  (t!! (!! 0.0))
	  )))


;;; ********************************************************************************

;; *COND

;; *cond example demonstrating that forms are evaluated
;; no matter whether any processors are selected.

(*cond
  ((minusp!! (self-address!!)) (do-negative-actions))
  ((plusp!! (self-address!!)) (do-positive-actions))
  ((zerop!! (self-address!!)) (do-zero-actions))
  (t!! (when (*or t!!) (error "It is impossible to execute this code")))
  )

;; The call to do-negative-actions will be executed even
;; though no processors have a negative self address.
;; Also, the T!! body clause will be executed for the
;; same reason, forcing the front-end test for no active processors
;; before issuing the error message.

;; Note that (*or t!!) is the standard *lisp idiom to
;; determine if any processor is active.

;;; ********************************************************************************

;; CREATE-VP-SET

;; Create-vp-set is normally used during program execution, not
;; at top level.  Below is an example of how Create-vp-set might
;; be used.

;; This example creates an N x N vp set, takes the first N elements of
;; PVAR and stores them into the main diagonal elements of a new pvar,
;; NEW-PVAR, created in the 2d vp set.  Then a function is called to
;; operate within this new vp set on the diagonal pvar, and finally the
;; temporarily created vp set is deallocated.

;; In this example N must be a power of two.

(defun make-2d-vp-set-with-diagonal-pvar (linear-vp-set n pvar)
  (let ((new-vp-se (create-vp-set (list n n))))
    (*with-vp-set new-vp-set
      (*let ((new-pvar (!! 0)))
	(*with-vp-set linear-vp-set
	  (*when (<!! (self-address!!) (!! n))
	    (*pset :no-collisions pvar new-pvar
		   (cube-from-vp-grid-address!! (self-address!!) (self-address!!)))
	    (*with-vp-set new-vp-set
	      (do-something-with-diagonal-pvar new-pvar)
	      )))))
    (deallocate-vp-set new-vp-set)
    ))

;;; ********************************************************************************

;; CUBE-FROM-GRID-ADDRESS!!

;; Example showing that one cannot rely on cube-from-grid-address!!
;; remaining the same across different configurations or implementations.

;; On CM hardware

  (*cold-boot :initial-dimensions '(32 16))
  (ppp (cube-from-grid-address!! (self-address-grid!! (!! 0)) (self-address-grid!! (!! 1))) :mode :grid :end '(4 4))

     DIMENSION 0 (X)  ----->

0 1 2 3 
4 5 6 7 
8 9 10 11 
12 13 14 15 


  (*cold-boot :initial-dimensions '(32 32))
  (ppp (cube-from-grid-address!! (self-address-grid!! (!! 0)) (self-address-grid!! (!! 1))) :mode :grid :end '(4 4))

     DIMENSION 0 (X)  ----->

0 1 2 3 
8 9 10 11 
16 17 18 19 
24 25 26 27 
  
;; On the *Lisp Simulator

  (*cold-boot :initial-dimensions '(32 16))
  (ppp (cube-from-grid-address!! (self-address-grid!! (!! 0)) (self-address-grid!! (!! 1))) :mode :grid :end '(4 4))

     DIMENSION 0 (X) ----->

0 16 32 48
1 17 33 49
2 18 34 50
3 19 35 51

;;; ********************************************************************************

;;; *DEFUN

;; Creates a *lisp function that will automatically reset the stack upon completion.

;; Example where *defun helps:

(defun logsumpvar (pvar)
  (log (*sum pvar)))

;; Evaluating:

(let ((total 0))
  (dotimes (i limit)
    (set total (logsumpvar (!! i)))))

;; will cause the stack to overflow, since on each iteration (!! i) is placed
;; on the stack and the stack is never reset. By changing the definition of
;; logsumpvar to a *defun, this will reset the stack automatically.
    
;; Example where *defun hurts:

(*defun pvalue (pvar)
  (expt!! pvar (random!! (!! 10))))

;; Evaluating:

(dotimes (i limit)
  (*set result-pvar (+!! result-pvar (pvalue (!! i)))))

;; will work, but be slow, since the *defun is resetting the stack. The *set in 
;; the dotimes does this, and so using the *defun is redundant. Changing the 
;; *defun to a regular defun will improve performance.

;;; ********************************************************************************

;;; DEF-VP-SET

;; Creating VP sets at the top level.

;; Note: the dimensionality of each axis must be a power of 2 and the product 
;; of the dimensions must be a power of two multiple of the physical machine size. 
;; The total of the dimensions must be at least as large as 
;; *minimum-size-for-vp-set*

;; create a vp set of 3 dimensions with no predefined *defvars. 

(def-vp-set 3-d '(1024 32 128))

;; create a vp set with predefined *defvars.

(def-vp-set anne '(65536)
  :*defvars ((x (!! 1) nil (field-pvar 2))
	     (y (self-address!!))))

;; This is as if you did:

(def-vp-set anne '(65536))
(*proclaim '(type (field-pvar 2) x))
(*defvar x (!! 1) nil anne)
(*defvar y (self-address!!) nil anne)

;; To create a flexible vp set (dimensions determined at run time):

(def-vp-set gumby nil)

;; then at run time call:

(allocate-processors-for-vp-set gumby '(128 64 32))

;; and when the code is done:

(deallocate-processors-for-vp-set gumby)

;; Instead of using these two forms you can do:

(with-processors-allocated-for-vp-set gumby
  :dimensions '(128 64 32)
  <user-code>
  )

;; Note: the above forms for instantiating a vp set do not change the
;; *current-vp-set*. You must use set-vp-set or *with-vp-set to do this.

;;; ********************************************************************************

;; ENUMERATE!!

;; (enumerate!!) <=> (1-!! (scan!! (!! 1) '+!!))

;; example

(ppp (*let ((x (!! 0)))
       (*when (oddp!! (self-address!!)) (*set x (enumerate!!)))
       x
       )
     :end 10
     )

0 0 0 1 0 2 0 3 0 4


;; ENUMERATE!! is often used to pack values in active processors
;; into the first N processors, where N is the number of active
;; processors

(ppp value-pvar :end 10)
0 1 2 3 4 5 6 7 8 9 

;; The values in the active (even) processors below processor 10 are
;; packed into the first five processors.

(*when (and!! (<!! (self-address!!) (!! 10)) (evenp!! (self-address!!)))
  (*pset :no-collisions value-pvar packed-value-pvar (enumerate!!))
  )

(ppp packed-value-pvar :end 10)

0 2 4 6 8 * * * * *

;;; ********************************************************************************

;; GRID-FROM-CUBE-ADDRESS!!

  (*cold-boot :initial-dimensions '(128 128))

  (ppp (self-address!!) :mode :grid :end '(4 4) :format "~3D ")

     DIMENSION 0 (X)  ----->

  0   1   2   3 
  8   9  10  11 
 16  17  18  19 
 24  25  26  27 

  (ppp (grid-from-cube-address!! (self-address!!) (!! 0)) :mode :grid :end '(4 4) :format "~3D ")

     DIMENSION 0 (X)  ----->

  0   1   2   3 
  0   1   2   3 
  0   1   2   3 
  0   1   2   3 

  (ppp (grid-from-cube-address!! (self-address!!) (!! 1)) :mode :grid :end '(4 4) :format "~3D ")

     DIMENSION 0 (X)  ----->

  0   0   0   0 
  1   1   1   1 
  2   2   2   2 
  3   3   3   3 

;;; ********************************************************************************
  
;;; *IF

;; takes a boolean pvar , in those processors where the boolean pvar is t,
;; the then clause is evaluated. In those processors where the boolean pvar is nil,
;; the else clause is evaluated. NOTE: both the then and else clauses are evaluated!

(defun test-*if ()
  (*let ((winners (!! 0))
	 (losers (!! 0)))
    (declare (type (pvar (unsigned-byte 1)) winners losers))
    (*if (zerop!! (random!! (!! 100)))
	 (*set winners (!! 1))
	 (*set losers (!! 1)))
    (ppp winners :end 10)
    (ppp losers :end 10)))

;; Note that both the clauses are evaluated:

(setq a 5 b 7)
(*if nil!! (setq a 1) (setq b 0))

;; *if's can be nested:

(*defvar result)

(*if (evenp!! (self-address!!))
     (*if (zerop!! (mod!! (self-address!!) (!! 4)))
	  (*set result (!! 4))
	  (*set result (!! 2)))
     (*set result (!! 1)))

;;; ********************************************************************************

;;; IF!!

(defun my-abs!! (pvar)
  (declare (type single-float-pvar pvar))
  (if!! (>!! pvar (!! 0))
	pvar
	(-!! pvar)))

;; if!!'s can be nested. This is the equivalent code to the above *if:

(*set result
      (if!! (evenp!! (self-address!!))
	    (if!! (zerop!! (mod!! (self-address!!) (!! 4)))
		  (!! 4)
		  (!! 2))
	    (!! 1)))


;;; ********************************************************************************

;; *LET

;; In general it is wise to declare the pvars allocated
;; by *LET.  This allows the *Lisp compiler to compile
;; expressions involving those pvars.  Here is the
;; previous example with DIE1 and DIE2 declared:

(*let ((die1 (1+!! (random!! (!! 6))))
       (die2 (1+!! (random!! (!! 6))))
       )
  (declare (type (field-pvar 8) die1 die2))
  (*max (+!! die1 die2))
  )

;; The length of a pvar may allocated by *let may be
;; determined at run time.  For example:

(*let ((processor-address (self-address!!)))
  (declare (type (field-pvar *current-send-address-length*) processor-address))
  ...
  )

;; This type of declaration insures that pvars are exactly
;; as big as they have to be.

;; Another example.  This function returns a floating point
;; pvar of either single or double precision depending
;; on its TYPE argument.

(defun make-me-a-float (type)
  (let ((s (if (eq type :single) 23 52))
	(e (if (eq type :single) 8 11))
	)
    (*let ((my-float (!! 0.0)))
      (declare (type (pvar (defined-float s e)) my-float))
      my-float
      )))


;; *let is able to allocate pvars without having to
;; provide initial value forms.  

(*let (x y)
  (declare (type string-char-pvar x y))
  )

;; However, the values of such pvars are undefined
;; until they are *set into, and any attempt to
;; reference a value until such time is an error.

(*let (x)
  (declare (type single-float-pvar x))
  (pref x 0)
  )

;; The above code is in error and *Lisp does not
;; define the value returned by such code.

;;; ********************************************************************************

;;; NEWS!!

;;; Examples

  (*cold-boot :initial-dimensions '(32 16))
  (*defvar source (random!! (!! 10)))

  (ppp source :mode :grid :end '(4 4) :format "~2D ")

 7  9  8  6 
 9  5  2  7 
 6  2  4  2 
 8  5  9  1 

;; This NEWS!! invocation shifts the entire grid over 1 to the left and
;; up 1.  The values on the left and upper edges of the grid are
;; wrapped around to the right and lower edges (not shown)

  (ppp (news!! source 1 1) :mode :grid :end '(4 4) :format "~2D ")

 5  2  7  4
 2  4  2  5
 5  9  1  3
 6  7  6  1

;; This NEWS invocation shows that processors which are not selected can
;; have values retrieved from them.  The processors in the even columns,
;; which are selected, retrieve data from the processors in the odd
;; columns, which are not selected.  The processors which are not
;; selected retain their original data.

  (*when (evenp!! (self-address-grid!! (!! 0)))
    (*set source (news!! source 1 0)))
  (ppp source :mode :grid :end '(4 4) :format "~2D ")

 9  9  6  6 
 5  5  7  7 
 2  2  2  2 
 5  5  1  1 

;; The SOURCE-PVAR argument to NEWS!! is evaluated only in those
;; processors from which data is being retrieved, not in the
;; processors doing the retrieving.  For instance

  (*defvar dest (!! 0))
  (*when (evenp!! (self-address-grid!! (!! 0)))
    (*set dest
	  (round!!
	    (news!! (/!! (!! 24) (self-address-grid!! (!! 0))) 1 0)
	    )))
  (ppp dest :mode :grid :end '(4 4) :format "~2D ")

     DIMENSION 0 (X)  ----->

24  0  8  0 
24  0  8  0 
24  0  8  0 
24  0  8  0 
  
;; If the division had been done in the currently selected set
;; of processors, then a division by 0 would have occurred in
;; the first column of processors, since (self-address-grid!! (!! 0))
;; is zero there.  The division was actually done in the processors
;; belonging to the odd columns, i.e., those processors having
;; data retrieved from them.

;; Performance Note

;; When NEWS!! is invoked with relative coordinates which are
;; powers of two (for example (NEWS!! PVAR 8 16)), the CM2
;; implementation of *Lisp uses special Paris instructions
;; which are able to quickly retrieve the data.  (NEWS!! PVAR 8 16)
;; is therefore signficantly faster than (NEWS!! PVAR 7 15)
;; for example.

;; Performance Note

;; Although seemingly symmetric, the CM2 *Lisp implementation of NEWS!!
;; is faster than the CM2 *Lisp implementation of *NEWS.

;;; ********************************************************************************

;;; *NEWS

;;; Examples

  (*cold-boot :initial-dimensions '(32 16))
  (*defvar source (random!! (!! 10)))
  (*defvar dest)

  (ppp source :mode :grid :end '(4 4) :format "~2D ")

 7  9  8  6 
 9  5  2  7 
 6  2  4  2 
 8  5  9  1 

;; This *NEWS invocation shifts the entire grid over 1 to the right and
;; down 1.  Values are wrapped around from the right and lower edges
;; to the left and upper edges.

  (*news source dest 1 1)
  (ppp dest :mode :grid :end '(4 4) :format "~2D ")

 8  5  8  1 
 6  7  9  8 
 8  9  5  2 
 4  6  2  4 

;; This *NEWS invocation shows that processors which are
;; not selected have their values changed.  The processors
;; in the even columns, which are selected, send data to
;; the processors in the odd columns, which are not selected.

  (*set dest (!! 0))
  (*when (evenp!! (self-address-grid!! (!! 0))) (*news source dest 1 0))
  (ppp dest :mode :grid :end '(4 4) :format "~2D ")

 0  7  0  8 
 0  9  0  2 
 0  6  0  4 
 0  8  0  9 

;; Performance Note

;; Although seemingly symmetric, the CM2 *Lisp implementation of NEWS!!
;; is faster than the CM2 *Lisp implementation of *NEWS.

;;; ********************************************************************************

;;; *NEWS, NEWS!!

;;; USAGE NOTE

;;; It is a common error to not realize that 1 dimensional NEWS
;;; coordinates are not identical to SEND (cube) addresses.
;;; Here is an illustration of the fact that the NEWS coordinates
;;; of processors in a 1 dimensional Vp Set are not everywhere the
;;; same as the processors' self-addresses.


  (*cold-boot :initial-dimensions (list cm:*physical-processors-limit*))

;;; If we print out self addresses in cube address order
;;; we get the expected result.

  (ppp (self-address!!) :end 64 :per-line 16 :format "~2D " :mode :cube)

 0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 
16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 
32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 
48 49 50 51 52 53 54 55 56 57 58 59 60 61 62 63 

;;; But if we print out processors' 1d news addresses in cube
;;; order they are not the same.

  (ppp (self-address-grid!! (!! 0)) :end 64 :per-line 16 :format "~2D " :mode :cube)

 0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 
16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 
48 49 50 51 52 53 54 55 56 57 58 59 60 61 62 63 
32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 

;;; If course of we print out processors' 1d news addresses in NEWS
;;; order, they will come out looking as we would expect.

  (ppp (self-address-grid!! (!! 0)) :end 64 :per-line 16 :format "~2D " :mode :grid)

 0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 
16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 
32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 
48 49 50 51 52 53 54 55 56 57 58 59 60 61 62 63 

;;; It is a common error to expect this operation to return a
;;; sequence of numbers in ascending order starting from 1.

  (ppp (news!! (self-address!!) 1) :end 64 :per-line 16 :format "~2D " :mode :cube)

 1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 
17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 48 
33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 96 
49 50 51 52 53 54 55 56 57 58 59 60 61 62 63 32 

;;; This operation in fact has the desired effect.

  (ppp (news!! (self-address-grid!! (!! 0)) 1) :end 64 :per-line 16 :format "~2D " :mode :grid)

 1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 
17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 
33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 48 
49 50 51 52 53 54 55 56 57 58 59 60 61 62 63 64 
  
;;; ********************************************************************************

;;; NEWS-BORDER!!

(*cold-boot :initial-dimensions '(128 128))

(ppp (news-border!! (self-address-grid!! (!! 0)) (!! -1) -1 -1)
     :mode :grid :end '(4 4) :format "~2D ")

-1 -1 -1 -1
-1  0  1  2
-1  0  1  2
-1  0  1  2

;; Diagonal copy, border of -1's

;;; ********************************************************************************

;;; *NEWS-DIRECTION

;; This function is particularly useful when writing subroutines
;; that must do NEWS operations along a particular axis but may be
;; called with VP sets of differing ranks.

(defun shift-upward-along-y-axis (dest-pvar source-pvar distance)
  (*news-direction source-pvar dest-pvar 1 (- distance)))

;;; ********************************************************************************

;;; NEWS-DIRECTION!!

;; This function is also particularly useful when writing subroutines
;; that must do NEWS operations along a particular axis but may be
;; called with VP sets of differing ranks.

(defun shift-upward-along-y-axis (pvar distance)
  (news-direction!! pvar 1 distance))

;;; ********************************************************************************

;;; OFF-VP-GRID-BORDER-P!!

;; Example.  Coordinates into TWO-D-VP-SET
;; are generated in MY-VP-SET, and OFF-VP-GRID-BORDER-P!!
;; is called to test whether those coordinates are valid.


 (def-vp-set two-d-vp-set '(4 4))
 (def-vp-set my-vp-set '(8))

 (*defvar y-coordinate (random!! (!! 5)) nil my-vp-set)
 (*defvar x-coordinate (random!! (!! 5)) nil my-vp-set)

 (ppp x-coordinate)
1 4 1 3 0 0 3 1 

 (ppp y-coordinate)
4 0 2 2 3 1 1 4 

;; The first two and the last coordinate pairs are invalid.

 (*with-vp-set my-vp-set
   (ppp (off-vp-grid-border-p!! two-d-vp-set x-coordinate y-coordinate)))
T T NIL NIL NIL NIL NIL T 

;;; ********************************************************************************

;;; OFF-GRID-BORDER-P!!

 (*cold-boot :initial-dimensions '(4 4))
 (*defvar x-coordinate (random!! (!! 6)))
 (*defvar y-coordinate (random!! (!! 6)))

 (ppp x-coordinate :mode :grid)

     DIMENSION 0 (X)  ----->

4 5 5 5 
4 2 2 2 
2 1 5 3 
5 1 2 3 

 (ppp y-coordinate :mode :grid)

     DIMENSION 0 (X)  ----->

0 1 0 5 
0 0 2 4 
1 1 4 4 
5 3 1 1 

 (ppp (off-grid-border-p!! x-coordinate y-coordinate) :mode :grid :format "~3S ")

     DIMENSION 0 (X)  ----->

T   T   T   T   
T   NIL NIL T   
NIL NIL T   T   
T   NIL NIL NIL 

;;; ********************************************************************************

;;; OFF-GRID-BORDER-RELATIVE-P!!

;;; Example

(*cold-boot :initial-dimensions '(128 128))

  (ppp (off-grid-border-relative-p!! (!! -1) (!! -1)) :mode :grid :end '(4 4) :format "~3S ")

     DIMENSION 0 (X)  ----->

T   T   T   T   
T   NIL NIL NIL 
T   NIL NIL NIL 
T   NIL NIL NIL 
  
;;; Example.  off-grid-border-relative-p!! can be
;;; used to easily determine all processors within
;;; two processors if the border.

(*set border-processor-p
      (or!! (off-grid-border-relative-p!! (!! -2) (!! -2))
	    (off-grid-border-relative-p!! (!! 2) (!! 2))
	    ))

;;; ********************************************************************************

;; OR!!

;; Simple example

(ppp (or!! (evenp!! (self-address!!)) (<!! (self-address!! (!! 3)))) :end 10)

T T T NIL T NIL T NIL T NIL


;; Example where using OR!! may cause a non-intuitive
;; result because of its deselection properties.


(ppp (or!! (evenp!! (self-address!!))
           (<!! (scan!! (self-address!!) '+!!) (!! 5)))
     :end 8)

T T T T T NIL T NIL 

(ppp (or!! (<!! (scan!! (self-address!!) '+!!) (!! 5))
	    (evenp!! (self-address!!)))
     :end 8)

T T T NIL T NIL T NIL

;; In the first case, the scan is done only in the okk
;; processors.  In the second case, the scan is done
;; in all the processors, resulting in different values
;; than in the first case.  This is the result of OR!!
;; deselecting those processors which satisfy any clause
;; before executing the next clause.

;; One can avoid this in the following manner:

(*let ((b1 (evenp!! (self-address!!)))
       (b2 (<!! (scan!! (self-address!!) '+!!) (!! 3)))
       )
  (declare (type boolean-pvar b1 b2))
  (or!! b1 b2)
  )

;;; ********************************************************************************

;; PREF!!

;; Example across Vp Sets, extracting the n diagonal elements
;; of a matrix stored as a 2 dimensional pvar into the
;; first n elements of a 1 dimensional pvar.

(*proclaim '(type (pvar (unsigned-byte 4)) matrix diagonal-elements))

(def-vp-set matrix-vp-set '(128 128)
  :*defvars ((matrix (random!! (!! 10)))))

(def-vp-set diagonal-vp-set '(8192)
  :*defvars ((diagonal-elements (!! 0))))

(defun retrieve-diagonal-elements ()
  (*with-vp-set diagonal-vp-set
    (*when (<!! (self-address!!) (!! 128))
      (*set diagonal-elements
	    (pref!! matrix
		    (cube-from-vp-grid-address!! 
		      (self-address!!) (self-address!!))
		    :vp-set matrix-vp-set)
	    ))))


;;; ********************************************************************************

;;; PREF

;; This is legal, since X was defined in all processors.
;; The value returned in 30.

(*all
  (*let ((x (self-address!!)))
    (*when (<!! (self-address!!) (!! 10))
      (pref x 30)
      )))

;; This is in error, since X was not defined in processor 30.
;; The value returned is undefined.

(*all
  (*when (<!! (self-address!!) (!! 10))
    (*let ((x (self-address!!)))
      (pref x 30)
      )))


;; Performance Note.

;; To read a single element of an array pvar out of the CM
;; there are two possibilities.

(*defvar my-array-pvar (vector!! (self-address!!) (-!! (self-address!!))))

(pref my-array-pvar 3)
#(3 -3)

;; First way

(aref (pref my-array-pvar 3) 1)
-3

;; Second way

(pref (aref!! my-array-pvar (!! 1)) 3)
-3

;; (Fewer bits move)
;; Marginal increase for small array sizes,
;; but compiler understands second method.

;; The second way is, in fact, more efficient, since
;; less data is read out of the Connection Machine.

;; The same principle applies to reading out data from
;; a single slot of a structure pvar.

;;; ********************************************************************************

;;; *PROCLAIM

;;; Examples

;; *proclaim is commonly used in four ways:

;; 1.  To provide type declarations for pvars later to be
;;     defined by *defvar.


(*proclaim '(type (pvar single-float) my-float-pvar))
(*defvar my-float-pvar)

(*proclaim '(type (vector-pvar (array (unsigned-byte 32) (4 4)) 3)
	     my-nested-array1 my-nested-array2))
(*defvar my-nested-array1)
(*defvar my-nested-array2)


;; 2.  To provide function declarations so that the *Lisp
;;     Compiler has information regarding the return type of
;;     functions it finds in *Lisp code.


;; Tell the *Lisp compiler that the hypotenuse!! function
;; takes two single float pvars as arguments and returns
;; a single float pvar as a result.

(*proclaim '(ftype (function (single-float-pvar single-float-pvar)
		    single-float-pvar) hypotenuse!!))

;; Tell the *Lisp compiler that the my-and!! function takes
;; any number of arguments of any type, and returns a boolean pvar.

(*proclaim '(ftype (function (&rest t) (pvar boolean)) my-and!!))


;; 3.  To provide the *Lisp compiler with information about
;;     Common Lisp variables.  *proclaim is used instead of proclaim
;;     to do this.

(*proclaim '(type double-float two-pi))
(defparameter two-pi (* pi 2.0))

(*proclaim '(type fixnum x-dimension y-dimension))


;; 4.  To define or change the compiler settings for the
;;     *Lisp compiler.


;; Tell the *Lisp compiler that full safety should be
;; enabled globally.

(*proclaim '(*optimize (safety 3)))

;; See the *Lisp Compiler Guide for further information.


;; Syntax Note

;; The declaration immediately following the *proclaim symbol
;; must be quoted, just as in Common Lisp the declaration
;; immediately following the proclaim symbol is quoted to
;; prevent evaluation.

;; Syntax Note

;; *proclaim forms almost always end with a double parentheses.
;; If any *proclaim form ends with a single parenthesis
;; or more than two parentheses it is almost certainly
;; an erroneous *proclaim form.

;; Note the exception in the use of *proclaim to declare
;; the *Lisp compiler safety level, example 4.

;;; ********************************************************************************

;;; *PSET

;; sends the values of source pvar to dest pvar by the addresses in address pvar

;; store self-address into dest in reverse order:

(*defvar dest)

(*pset :default (self-address!!) dest (-!! (!! (1- *number-of-processors-limit*))
					   (self-address!!))) 

(ppp dest :end 10) ; assuming a 1k machine and a vpratio of 1:
1023 1022 1021 1020 1019 1018 1017 1016 1015 1014 

;; you could implement *sum using *pset (although you shouldn't for performance reasons)

;; all processors send to processor 47 adding as they get there. A simple call to pref
;; gets the sum.
(defun my-*sum (pvar)
  (declare (type (pvar (unsigned-byte 10)) pvar))
  (pref (*let (the-sum-goes-here)
	  (declare (type (pvar (unsigned-byte 32)) the-sum-goes-here))
	  (*all (*pset :add pvar the-sum-goes-here (!! 47)))
	  the-sum-goes-here)
	47))
		
;; Using *pset to do histogram work

(*defvar data-pvar (random!! (!! 10)))

(defun histogram (pvar)
  (declare (type (pvar (unsigned-byte 4)) pvar))
  (*let ((histogram (!! 0)))
    (declare (type (pvar (unsigned-byte *current-send-address-length*)) histogram))
    (*pset :add (!! 1) histogram pvar)
    histogram))

;; inter-vp set communication:

(*proclaim '(type (pvar (unsigned-byte 16)) one-d-pvar two-d-pvar))

(def-vp-set one-d (list *number-of-processors-limit*)
  :*defvars ((one-d-pvar (1+!! (self-address!!)))))

(def-vp-set two-d (list *number-of-processors-limit* *number-of-processors-limit*)
  :*defvars ((two-d-pvar (!! 0))))

(defun send-to-diagonal ()
  (*with-vp-set one-d  ; vp set of source pvar
    (*pset :no-collisions one-d-pvar two-d-pvar
	   (cube-from-vp-grid-address!! two-d (self-address!!) (self-address!!) )
	   :vp-set two-d)))
									     
;;; ********************************************************************************

;;; PVAR-TO-ARRAY

;; Example showing how to transfer the values of a range of
;; processors out of the CM into a newly created front end array.


(pvar-to-array (self-address!!) nil :cube-address-start 10 :cube-address-end 20)

#(10 11 12 13 14 15 16 17 18 19)

(pvar-to-array (self-address!!) nil :array-offset 2
	       :cube-address-start 3 :cube-address-end 10)

#(0 0 3 4 5 6 7 8 9)


;; Performance note.  

;; The pvar-to-array functions perform most
;; efficiently when used on non-aggregate pvars of declared type, and
;; when the front end array is of corresponding type to that of the
;; pvar.  For instance, transferring data from a pvar of type
;; single-float into an array whose element type is single float
;; is very efficient.  Transferring a general pvar into an array
;; whose element type is T will not be as efficient.

;; Transferring aggregate pvars (structures and arrays) using a single
;; call to array-to-pvar, pvar-to-array, pvar-to-array-grid,
;; or array-to-pvar-grid is very slow.  See
;; the ARRAY-TO-PVAR performance note for a discussion of how to 
;; transfer aggregate data efficiently between the front end and
;; the Connection Machine.

;; Syntax Note:

;; When no array is specified to the pvar-to-array and pvar-to-array-grid
;; functions, a NIL must be provided instead if keyword arguments are to be used.

;;; ********************************************************************************

;;; PVAR-TO-ARRAY-GRID

;; An example illustrating the use of PVAR-TO-ARRAY-GRID
;; to extract a subgrid from a pvar and store it into
;; a subarray of an array.

(*cold-boot :initial-dimensions '(128 128))

(defparameter an-array
	      (make-array '(10 10)
			  :element-type 'single-float
			  :initial-element 0.0))

(*proclaim '(type single-float-pvar xyzzy))
(*defvar xyzzy (float!! (self-address!!)))

  (ppp xyzzy :mode :grid :end '(5 5) :format "~5F ")

     DIMENSION 0 (X)  ----->

  0.0   1.0   2.0   3.0   4.0 
  8.0   9.0  10.0  11.0  12.0 
 16.0  17.0  18.0  19.0  20.0 
 24.0  25.0  26.0  27.0  28.0 
128.0 129.0 130.0 131.0 132.0 

;; This call transfers the 4x4 subgrid of XYZZY whose corners are

;;     (1 1) (4 1)
;;     (1 4) (4 4)

;; to the 4x4 subarray of AN-ARRAY whose corners are

;;     (2 3) (6 3)
;;     (2 7) (6 7)

(pvar-to-array-grid xyzzy an-array
		    :array-offset '(2 3)
		    :grid-start '(1 1)
		    :grid-end '(5 5))

(aref an-array 2 3)
9.0


;; Performance note.  

;; The pvar-to-array functions perform most
;; efficiently when used on non-aggregate pvars of declared type, and
;; the when the front end array is of corresponding type to that of the
;; pvar.  For instance, transferring data from a pvar of type
;; single-float into an array whose element type is single float
;; is very efficient.  Transferring a general pvar into an array
;; whose element type is T will not be as efficient.

;; Transferring aggregate pvars (structures and arrays) using a single
;; call to array-to-pvar or array-to-pvar-grid is very slow.  See
;; the ARRAY-TO-PVAR performance note for a discussion of how to 
;; transfer aggregate data efficiently between the front end and
;; the Connection Machine.

;; Syntax Note:

;; When no array is specified to the pvar-to-array and pvar-to-array-grid
;; functions, a NIL must be provided instead if keyword arguments are to be used.

;;; ********************************************************************************

;;; RANK!!

;; returns a pvar that contains the relative ranking of the values of the argument
;; pvar.

;; Examples:

(*defvar foo (random!! (!! 22)))

(ppp foo)

18 10 14 15 9 2 3 5 15 6 9 20 7 8 11 11 

(ppp (rank!! foo '<=!!))

14 8 11 13 7 0 1 2 12 3 6 15 4 5 10 9 

(ppp (rank!! foo '<=!! :segment-pvar (evenp!! (self-address!!))))

1 0 0 1 1 0 0 1 1 0 0 1 0 1 1 0 

;; on a 2d grid, with the :dimension argument:

(ppp (self-address!!) :mode :grid :end '(4 4))

     DIMENSION 0 (X)  ----->

0 1 2 3 
4 5 6 7 
8 9 10 11 
12 13 14 15 

> (ppp (rank!! (self-address!!) '<=!! :dimension 1) :mode :grid :end '(4 4))

     DIMENSION 0 (X)  ----->

0 0 0 0 
1 1 1 1 
2 2 2 2 
3 3 3 3 
> 

;;; ********************************************************************************

;;; REDUCE-AND-SPREAD!!


 (*cold-boot :initial-dimensions '(4 4))

 (ppp (self-address!!) :mode :grid :format "~2D ")

     DIMENSION 0 (X)  ----->

 0  4  8 12 
 1  5  9 13 
 2  6 10 14 
 3  7 11 15 

 (ppp (reduce-and-spread!! (self-address!!) '+!! 1) :mode :grid :format "~2D ")

     DIMENSION 0 (X)  ----->

 6 22 38 54 
 6 22 38 54 
 6 22 38 54 
 6 22 38 54 

;;; ********************************************************************************

;;; ROW-MAJOR-SIDEWAYS-AREF!!

  ;; In each processor, the array [ 5.0 8.0 ] is stored
  ;;                              [ 3.0 0.0 ]

  (*proclaim '(type (array-pvar single-float '(2 2)) my-sideways-array))
  (*defvar my-sideways-array (!! #2A((5.0 8.0) (3.0 0.0))))

  ;; The array is turned slicewise, and is verified to be slicewise.

  (*slicewise my-sideways-array)
  (sideways-array-p my-sideways-array)
T

  ;; A different index into the array is calculated in each processor
  ;; and the array elements corresponding to the different indices
  ;; are accessed using row-major-sideways-aref!!.

  (ppp (row-major-sideways-aref!! my-sideways-array (mod!! (self-address!!) (!! 4))) :end 16)

5.0 8.0 3.0 0.0 5.0 8.0 3.0 0.0 5.0 8.0 3.0 0.0 5.0 8.0 3.0 0.0 
  
;;; ********************************************************************************

;;; SCAN!!

;; Performs a cumulatove reduction for the argument pvar either by send address
;; or along a specified dimension:

;; simple scan using +!!:

(scan!! (self-address!!) '+!!)		

0 1 3 6 10 15 21 28 36 45 55 66 78 91 105 ...

;; Examples of segment scans:

(*defvar seg-pvar (zerop!! (mod!! (self-address!!) (!! 4))))

;; note that the segment pvar starts a new reduction
(scan!! (self-address!!) '+!! :segment-pvar seg-pvar)

0 1 3 6 4 9 15 22 8 17 27 38 12 25 39 54 16 33 51 70 ...

;; copies the first value of the segment out to the rest of the segment:
(scan!! (self-address!!) 'copy!! :segment-pvar seg-pvar)

0 0 0 0 4 4 4 4 8 8 8 8 12 12 12 12 16 16 16 16 

;; use :direction to control whether to start at the high or low end of each segment
;; Note that self-address and seg-pvar are:
;; self-address:  0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 ...
;; seg-pvar:      T  F  F  F  T  F  F  F  T  F  F  F  T  F  F  F  T  F  F ...
;; so the first segment is at processor 0, and we work back summing. The first segment
;; only contains processor 0, so the result of that segment is 0. The second segment 
;; runs from processor 4 to processor 1, so the result of scan!! is that processor 4
;; has 4, processor 3 has 4+3=7, processor 2 has 4+3+2=9 and processors 1 (end of
;; the segment) has 4+3+2+1=10.
(scan!! (self-address!!) '+!! :segment-pvar seg-pvar :direction :backward)

0 10 9 7 4 26 21 15 8 42 33 23 12 58 45 31 16 74 57 39 ...

;; the :include-self argument (defaults to nil) allows you to include the first 
;; value of the segment in the returned pvar, this allows scan to put a value in the
;; first segment processor of the returned pvar:

(*defvar source (random!! (!! 10)))
  0  2  5  0  4  2  2  6  7  5  0  9  6  3  5  1  1  6  2  1

(scan!! source 'max!! :segment-pvar seg-pvar :include-self t)

  0  2  5  5  4  4  4  6  7  7  7  9  6  6  6  6  1  6  6  6

(scan!! source 'max!! :segment-pvar seg-pvar :include-self nil)

  0  0  2  5  5  4  4  4  6  7  7  7  9  6  6  6  6  1  6  6
 
;; Multidimensionsal scans are possible:

;; given a 2d grid:

(ppp (self-address!!) :mode :grid :end '(4 4) :format "~3d")

     DIMENSION 0 (X)  ----->

  0  1  2  3
  4  5  6  7
  8  9 10 11
 12 13 14 15

;; scan on dimension 0:
(ppp (scan!! (self-address!!) '+!! :dimension 0) :mode :grid :end '(4 4) :format "~3d")

     DIMENSION 0 (X)  ----->

  0  1  3  6
  4  9 15 22
  8 17 27 38
 12 25 39 54

;; now on dimension 1:
(ppp (scan!! (self-address!!) '+!! :dimension 1) :mode :grid :end '(4 4) :format "~3d")

     DIMENSION 0 (X)  ----->

  0  1  2  3
  4  6  8 10
 12 15 18 21
 24 28 32 36


;;; ********************************************************************************

;;; SELF-ADDRESS-GRID!!

(*cold-boot :initial-dimensions '(128 128))

(ppp (self-address-grid!! (!! 0)) :mode :grid :end '(4 4))

0 1 2 3
0 1 2 3
0 1 2 3
0 1 2 3


;; To select the diagonal elements of a 2 dimensional Vp Set

(*when (=!! (self-address-grid!! (!! 0))
	    (self-address-grid!! (!! 1))
	    )
  ...
  )


;;; to select the tridiagonal elements of a 2 dimensional Vp Set

(*when (or!! (=!! (self-address-grid!! (!! 0)) (self-address-grid!! (!! 1)))
	     (=!! (self-address-grid!! (!! 0)) (1+!! (self-address-grid!! (!! 1))))
	     (=!! (self-address-grid!! (!! 0)) (1-!! (self-address-grid!! (!! 1))))
	     )
  ...
  )


;; Performance Note.

;; The computation of a grid self address using self-address-grid!!
;; takes a signficant amount of time.  Rather than computing
;; a grid self address over and over again it is preferable to
;; compute it once, e.g.:

(*let ((x-addr (self-address-grid!! (!! 0)))
       (y-addr (self-address-grid!! (!! 1)))
       )
  (declare (type (field-pvar *current-send-address-length*) x-addr y-addr))
  (*when (or!! (=!! x-addr y-addr)
	       (=!! x-addr (1+!! y-addr))
	       (=!! x-addr (1-!! y-addr))
	       )
    ...
    ))

;;; ********************************************************************************

;; *SIDEWAYS-ARRAY

;; A code skeleton showing how one might use slicewise arrays.


;; We define a vector we want to fill with data and then 
;; access using indirect addressing.

(*proclaim '(type (vector-pvar single-float 20) my-sideways-vector))
(*defvar my-sideways-vector (make-array!! 20 :element-type 'single-float-pvar))

(defun top-level ()

  ;; First, call an initialization function
  ;; which fills up the data vector with values.
  
  (fill-my-sideways-vector-with-values)

  ;; Second, turn the vector sideways in our
  ;; main program so we can access and change the data values
  ;; using indirect addressing.

  (*sideways-array my-sideways-vector)

  (do-my-computations)

  ;; Finally, turn the vector back processorwise
  ;; so that we may examine the answers.

  (*sideways-array)
  (ppp my-sideways-vector :end 10)

  )

;; an example of using sideways-aref!

(*proclaim '(type (array-pvar singel-float '(2 2)) foo))

(*defvar foo (!! #2A((5.0 8.0) (3.0 0.0))))

(*sideways-array foo)

(sideways-array-p foo)

(*defvar i1 (mod!! (self-address!!) (!! 2)))

(*defvar i2 (mod!! (floor!! (self-address!!) (!! 2)) (!! 2)))

(ppp (sideways-aref!! foo i2 i1) :end 14)

5.0 8.0 3.0 0.0 5.0 8.0 3.0 0.0 5.0 8.0 3.0 0.0 5.0 8.0 

(*setf (sideways-aref!! foo (!! 0) (!! 1)) (!! 6))

(ppp (sideways-aref!! foo i2 i1) :end 14)

5.0 6.0 3.0 0.0 5.0 6.0 3.0 0.0 5.0 6.0 3.0 0.0 5.0 6.0 

;;; ********************************************************************************

;;; *WARM-BOOT

;;; Example demonstrating the necessity of doing a *warm-boot
;;; after aborting to top level.

  (*warm-boot)

  (*let (x)
    (declare (type single-float-pvar x))
    (*when (evenp!! (self-address!!)) (*set x (!! #\x))))

Error: In interpreted *SET.
The source expression in a float-general *set contains something that is not a float.
A pvar of type STRING-CHAR caused the error.

-> (*sum (!! 1))
256

;;; The currently selected set is not restored by
;;; aborting back to top level.

-> Abort
Return to Lisp Top Level in Dynamic Lisp Listener 1
Back to Lisp Top Level in Dynamic Lisp Listener 1.

  (*sum (!! 1))
256

;;; Once a *warm-boot is performed, the currently selected
;;; set is reset to all processors active.

  (*warm-boot)
NIL
  (*sum (!! 1))
512
  
;;; ********************************************************************************

;;; WITH-CSS-SAVED

(defun css-preserved (x y)
  "Returns y divided by x for y > 0.
   Returns NIL if any x is 0.  In any processors
   where y < 0 the return value is undefined."
  (block exit
    (with-css-saved
      (*when (>!! y (!! 0))
	(if (*or (zerop!! x))
	    (return-from exit nil)
	    (/!! y x)
	    )))))

(defun css-not-preserved (x y)
  (block exit
    (*when (>!! y (!! 0))
      (if (*or (zerop!! x))
	  (return-from exit nil)
	  (/!! y x)
	  ))))


(*cold-boot :initial-dimensions '(512))

(*all (progn (css-preserved (!! 0) (self-address!!)) (*sum (!! 1))))
512

;;; Because Y is zero in one processor, that processor remains
;;; unselected when the RETURN-FROM forces the CSS-NOT-PRESERVED
;;; function to exit.

(*all (progn (css-not-preserved (!! 0) (self-address!!)) (*sum (!! 1))))
511

;;; ********************************************************************************

;;; WITH-PROCESSORS-ALLOCATED-FOR-VP-SET

;;; Successive files of data are processed.  A single
;;; flexible Vp Set is used, which is instantiated
;;; and deinstatiated once for each file.  The Vp Set
;;; is made large enough to hold each file's data.

(dolist (file files-to-be-processed)
  (let ((file-size (get-file-size file)))
    (with-processors-allocated-for-vp-set
      (file-data-vp-set
	:dimensions (next-power-of-two->= file-size)
	)
      (read-file-data-into-vp-set file file-data-vp-set)
      (*with-vp-set file-data-vp-set
	(process-file-data)
	))))


;;; ********************************************************************************
