/*
    object.h  -- Data structure definitions.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


/*
	Some system constants.
*/


#define	TRUE		1	/*  boolean true value  */
#define	FALSE		0	/*  boolean false value  */

#define	LISP_PAGESIZE	2048	/*  page size in bytes  */

#define	CHCODELIM	256	/*  character code limit  */
				/*  ASCII character set  */
#define	CHFONTLIM	1	/*  character font limit  */

#define	CHBITSLIM	16	/*  character bits limit  */
#define CONTROL_BIT     1
#define META_BIT        2
#define SUPER_BIT       4
#define HYPER_BIT       8

#define	CHCODEFLEN	(sizeof (short))/*  character code field length  */
#define	CHFONTFLEN	0	/*  character font field length  */
#define	CHBITSFLEN      4	/*  character bits field length  */

#define	PHTABSIZE	512	/*  number of entries in the package hash table  */

#define	ARANKLIM	64	/*  array rank limit  */

#define	RTABSIZE	CHCODELIM	/*  read table size  */

#define	CBMINSIZE	64	/*  contiguous block minimal size  */

#define Q_SIZE    	128	/*  output character queue size (for print) */
#define IS_SIZE   	256	/*  indentation stack size (for print)	*/


typedef int bool;
typedef int fixnum;
typedef float shortfloat;
typedef double longfloat;
typedef unsigned char byte;

/*
	Definition of the type of LISP objects.
*/
typedef union lispunion *object;

/*
	OBJect NULL value.
	It should not coincide with any legal object value.
*/
#define	OBJNULL		((object)NULL)

/*
	Definition of each implementation type.
*/

#define IMMEDIATE(obje)	((int)(obje) & 3)

/* Immediate fixnums:		*/
#define MAKE_FIXNUM(n)		((object)(((int)(n) << 2) | 1))
#define	fix(obje)		(((int)(obje)) >> 2)
#ifdef LOCATIVE
#define FIXNUMP(obje)		((((int)(obje)) & 3) == 1)
#else
#define FIXNUMP(obje)		(((int)(obje)) & 1)
#endif

/* Immediate characters:	*/
#define CHARACTERP(obje)	((((int)(obje)) & 3) == 2)
#define	code_char(c)		((object)(((int)(c) << 2) | 2))
#define MAKE_CHARACTER(c,b,f)	((object)(((((b) << 16) | (int)(c)) << 2) | 2))
#define	char_code(obje)		((((int)(obje)) >> 2) & 0xffff)
#define	char_bits(obje)		(((int)(obje)) >> 18) /* CHCODEFLEN * 8 + 2 */
#define	char_font(obje)		0
#define char_int(obje)		char_code(obje)
#define int_char(i)		code_char(i)

/* Locatives:	*/
#define LOCATIVEP(obje)	((((int)(obje)) & 3) == 3)
#define MAKE_LOCATIVE(n)((object)(((int)(n) << 2) | 3))
#define DEREF(loc)	(*(object *)((unsigned int)(loc) >> 2))
#define UNBOUNDP(loc)	(DEREF(loc) == OBJNULL)

/* Immediate spice:		*/
#define MAKE_SPICE(n)	MAKE_LOCATIVE(n)
#define SPICE(n)	fix(n)

#define NUMBER_TYPE(t)	(t == t_fixnum || t >= t_bignum && t <= t_complex)
#define ARRAY_TYPE(t)	(t >= t_array && t <= t_bitvector)

#define HEADER		byte t, m

struct shortfloat_struct {
	HEADER;
	shortfloat	SFVAL;	/*  shortfloat value  */
};
#define	sf(obje)	(obje)->SF.SFVAL

struct longfloat_struct {
	HEADER;
	longfloat	LFVAL;	/*  longfloat value  */
};
#define	lf(obje)	(obje)->LF.LFVAL

struct bignum {
	HEADER;
	struct bignum   *big_cdr;	/*  bignum cdr  */
	int		big_car;	/*  bignum car  */
};

struct ratio {
	HEADER;
	object	rat_den;	/*  denominator  */
				/*  must be an integer  */
	object	rat_num;	/*  numerator  */
				/*  must be an integer  */
};

struct complex {
	HEADER;
	object	cmp_real;	/*  real part  */
				/*  must be a number  */
	object	cmp_imag;	/*  imaginary part  */
				/*  must be a number  */
};

enum stype {			/*  symbol type  */
	stp_ordinary,		/*  ordinary  */
	stp_constant,		/*  constant  */
        stp_special		/*  special  */
};

#define	Cnil			((object)&Cnil_body)
#define	Ct			((object)&Ct_body)

struct symbol {
	HEADER;
	byte    s_stype;	/*  symbol type: of enum stype  */
	byte    s_mflag;	/*  macro flag  */
	object	s_dbind;	/*  dynamic binding  */
	int	(*s_sfdef)();	/*  special form definition  */
				/*  This field coincides with c_car  */

#define	NOT_SPECIAL		((int (*)())Cnil)
#define SPECIAL(fun)		((fun)->s.s_sfdef != NOT_SPECIAL)

#define	s_fillp		st_fillp
#define	s_self		st_self

	int	s_fillp;	/*  print name length  */
	char	*s_self;	/*  print name  */
				/*  These fields coincide with  */
				/*  st_fillp and st_self.  */

	object	s_gfdef;        /*  global function definition  */
				/*  For a macro,  */
				/*  its expansion function  */
				/*  is to be stored.  */
	object	s_plist;	/*  property list  */
	object	s_hpack;	/*  home package  */
				/*  Cnil for uninterned symbols  */
};

extern struct symbol Cnil_body, Ct_body;

struct package {
	HEADER;
	object	p_name;		/*  package name  */
				/*  a string  */
	object	p_nicknames;	/*  nicknames  */
				/*  list of strings  */
	object	p_shadowings;	/*  shadowing symbol list  */
	object	p_uselist;	/*  use-list of packages  */
	object	p_usedbylist;	/*  used-by-list of packages  */
	object	*p_internal;	/*  hashtable for internal symbols  */
	object	*p_external;	/*  hashtable for external symbols  */
	struct package
		*p_link;	/*  package link  */
};

/*
	The values returned by intern and find_symbol.
	File_symbol may return 0.
*/
#define	INTERNAL	1
#define	EXTERNAL	2
#define	INHERITED	3

/*
	All the packages are linked through p_link.
*/
extern struct package *pack_pointer;	/*  package pointer  */

struct cons {
	HEADER;
	object	c_cdr;		/*  cdr  */
	object	c_car;		/*  car  */
};

enum httest {			/*  hash table key test function  */
	htt_eq,			/*  eq  */
	htt_eql,		/*  eql  */
	htt_equal		/*  equal  */
};

struct htent {			/*  hash table entry  */
	object	hte_key;	/*  key  */
	object	hte_value;	/*  value  */
};

struct hashtable {		/*  hash table header  */
	HEADER;
	short	ht_test;	/*  key test function  */
				/*  of enum httest  */
	struct  htent
		*ht_self;	/*  pointer to the hash table  */
	object	ht_rhsize;	/*  rehash size  */
	object	ht_rhthresh;	/*  rehash threshold  */
	int	ht_nent;	/*  number of entries  */
	int	ht_size;	/*  hash table size  */
};

enum aelttype {			/*  array element type  */
	aet_object,		/*  t                */
	aet_ch,			/*  string-char      */
	aet_bit,		/*  bit              */
	aet_fix,		/*  fixnum           */
	aet_sf,			/*  short-float      */
	aet_lf			/*  long-float       */
/*	,aet_char,		/*  signed char      */
/*	aet_uchar,		/*  unsigned char    */
/*	aet_short,		/*  signed short     */
/*	aet_ushort		/*  unsigned short   */
};

struct array {			/*  array header  */
	HEADER;
	byte	a_adjustable;	/*  adjustable flag  */
	byte	a_rank;		/*  array rank  */
	object	a_displaced;	/*  displaced  */
	int	a_dim;		/*  dimension  */
	int	*a_dims;	/*  table of dimensions  */
	object	*a_self;	/*  pointer to the array  */
	byte	a_elttype;	/*  element type  */
	byte	a_offset;	/*  bitvector offset  */
};

struct vector {			/*  vector header  */
	HEADER;
	byte	v_adjustable;	/*  adjustable flag  */
	byte	v_hasfillp;	/*  has-fill-pointer flag  */
	object	v_displaced;	/*  displaced  */
	int	v_dim;		/*  dimension  */
	int	v_fillp;	/*  fill pointer  */
				/*  For simple vectors,  */
				/*  v_fillp is equal to v_dim.  */
	object	*v_self;	/*  pointer to the vector  */
	byte	v_elttype;	/*  element type  */
};

struct string {			/*  string header  */
	HEADER;
	byte	st_adjustable;	/*  adjustable flag  */
	byte	st_hasfillp;	/*  has-fill-pointer flag  */
	object	st_displaced;	/*  displaced  */
	int	st_dim;		/*  dimension  */
				/*  string length  */
	int	st_fillp;	/*  fill pointer  */
				/*  For simple strings,  */
				/*  st_fillp is equal to st_dim-1.  */
	char	*st_self;	/*  pointer to the string  */
};

struct ustring {
	HEADER;
	byte	ust_adjustable;
	byte	ust_hasfillp;
	object	ust_displaced;
	int	ust_dim;
	int	ust_fillp;
	unsigned char
		*ust_self;
};

struct bitvector {		/*  bitvector header  */
	HEADER;
	byte	bv_adjustable;	/*  adjustable flag  */
	byte	bv_hasfillp;	/*  has-fill-pointer flag  */
	object	bv_displaced;	/*  displaced  */
	int	bv_dim;		/*  dimension  */
				/*  number of bits  */
	int	bv_fillp;	/*  fill pointer  */
				/*  For simple bitvectors,  */
				/*  st_fillp is equal to st_dim.  */
	char	*bv_self;	/*  pointer to the bitvector  */
	byte	bv_elttype;	/*  element type: not used (placeholder)  */
	byte	bv_offset;	/*  bitvector offset  */
				/*  the position of the first bit  */
				/*  in the first byte  */
};

struct fixarray {		/*  fixnum array header  */
	HEADER;
	byte	fixa_adjustable;/*  adjustable flag  */
	byte	fixa_rank;	/*  array rank  */
	object	fixa_displaced;	/*  displaced  */
	int	fixa_dim;	/*  dimension  */
	int	*fixa_dims;	/*  table of dimensions  */
	fixnum	*fixa_self;	/*  pointer to the array  */
	byte	fixa_elttype;	/*  element type  */
};

struct sfarray {		/*  short-float array header  */
	HEADER;
	byte	sfa_adjustable;	/*  adjustable flag  */
	byte	sfa_rank;	/*  array rank  */
	object	sfa_displaced;	/*  displaced  */
	int	sfa_dim;	/*  dimension  */
	int	*sfa_dims;	/*  table of dimensions  */
	shortfloat
		*sfa_self;	/*  pointer to the array  */
	byte	sfa_elttype;	/*  element type  */
};

struct lfarray {		/*  long-float array header  */
	HEADER;
	byte	lfa_adjustable;	/*  adjustable flag  */
	byte	lfa_rank;	/*  array rank  */
	object	lfa_displaced;	/*  displaced  */
	int	lfa_dim;		/*  dimension  */
	int	*lfa_dims;	/*  table of dimensions  */
	longfloat
		*lfa_self;	/*  pointer to the array  */
	byte	lfa_elttype;	/*  element type  */
};

struct structure {		/*  structure header  */
	HEADER;
	short	str_length;	/*  structure length  */
	object	str_name;	/*  structure name  */
	object	*str_self;	/*  structure self  */
};

enum smmode {			/*  stream mode  */
	smm_input,		/*  input  */
	smm_output,		/*  output  */
	smm_io,			/*  input-output  */
	smm_probe,		/*  probe  */
	smm_synonym,		/*  synonym  */
	smm_broadcast,		/*  broadcast  */
	smm_concatenated,	/*  concatenated  */
	smm_two_way,		/*  two way  */
	smm_echo,		/*  echo  */
	smm_string_input,	/*  string input  */
	smm_string_output	/*  string output  */
};

struct stream {
	HEADER;
	short	sm_mode;	/*  stream mode  */
				/*  of enum smmode  */
	FILE	*sm_fp;		/*  file pointer  */
	object	sm_object0;	/*  some object  */
	object	sm_object1;	/*  some object */
	int	sm_int0;	/*  some int  */
	int	sm_int1;	/*  some int  */
};


#define	BASEFF		(unsigned char *)0xffffffff

struct random {
	HEADER;
	unsigned	rnd_value;	/*  random state value  */
};

enum chattrib {			/*  character attribute  */
	cat_whitespace,		/*  whitespace  */
	cat_terminating,	/*  terminating macro  */
	cat_non_terminating,	/*  non-terminating macro  */
	cat_single_escape,	/*  single-escape  */
	cat_multiple_escape,	/*  multiple-escape  */
	cat_constituent		/*  constituent  */
};

struct rtent {				/*  read table entry  */
	enum chattrib	rte_chattrib;	/*  character attribute  */
	object		rte_macro;	/*  macro function  */
	object		*rte_dtab;	/*  pointer to the  */
					/*  dispatch table  */
					/*  NULL for  */
					/*  non-dispatching  */
					/*  macro character, or  */
					/*  non-macro character  */
};

struct readtable {			/*  read table  */
	HEADER;
	struct rtent	*rt_self;	/*  read table itself  */
};

struct pathname {
	HEADER;
	object	pn_host;	/*  host  */
	object	pn_device;	/*  device  */
	object	pn_directory;	/*  directory  */
	object	pn_name;	/*  name  */
	object	pn_type;	/*  type  */
	object	pn_version;	/*  version  */
};

struct codeblock {
	char	*cd_start;		/*  start address of the code  */
	int	cd_size;		/*  code size  */
 	object	cd_data;		/*  data vector  */
      };

struct cfun {				/*  compiled function header  */
	HEADER;
	object	cf_name;		/*  compiled function name  */
	int	(*cf_self)();		/*  entry address  */
	struct codeblock *cf_block;	/*  descriptor of C code block  */
					/*  for GC  */
};

struct cclosure {			/*  compiled closure header  */
	HEADER;
	int	(*cc_self)();		/*  entry address  */
	object	cc_env;			/*  environment  */
	struct codeblock *cc_block;	/*  descriptor of C code block  */
					/*  for GC  */
};
/*
struct spice {
	HEADER;
	int	spc_dummy;
};
*/
/*
	dummy type
*/
struct dummy {
	HEADER;
};

#ifdef MTCL
struct cont {
	HEADER;
	object	cn_thread;	/* its thread */
	bool	cn_resumed;	/* already resumed */
	bool	cn_timed_out;	/* timed out */
};

struct thread {
	HEADER;
	struct pd      *th_self;	/* the thread itself (really a *pd) */
	int	th_size;	/* its size */
	object	th_fun;		/* initial function */
	object	th_cont;	/* its cont */
};
#endif MTCL


#ifdef CLOS
struct instance {		/*  instance header  */
	HEADER;
	short	in_length;	/*  instance length  */
	object	in_class;	/*  instance class  */
	object	*in_slots;	/*  instance slots  */
};

struct gfun {			/*  generic function header  */
	HEADER;
	short	gf_arg_no;	/*  number of arguments */
	object	gf_name;	/*  generic function name  */
	object  gf_meth_ht;	/*  hashtable for cashing methods */
				/*  later on we might define a  */
				/*  specialized version */
	object  *gf_spec_how;	/*  how methods specialize on each argument: */
	                        /*  NIL  means no specialization, */
				/*  T    means specialize on type, */
				/*  list constains the list of objects used */
				/* 	 in eql specializers */
	object  gf_gfun;       /*  the generic function object */
};
#endif CLOS

/*
	Definition of lispunion.
*/
union lispunion {
	struct bignum	big;	/*  bignum  */
	struct ratio	rat;	/*  ratio  */
	struct shortfloat_struct
			SF;	/*  short floating-point number  */
	struct longfloat_struct
			LF;	/*  long floating-point number  */
	struct complex	cmp;	/*  complex number  */
	struct symbol	s;	/*  symbol  */
	struct package	p;	/*  package  */
	struct cons	c;	/*  cons  */
	struct hashtable
			ht;	/*  hash table  */
	struct array	a;	/*  array  */
	struct vector	v;	/*  vector  */
	struct string	st;	/*  string  */
	struct ustring	ust;
	struct bitvector
			bv;	/*  bit-vector  */
	struct stream	sm;	/*  stream  */
	struct random	rnd;	/*  random-states  */
	struct readtable
			rt;	/*  read table  */
	struct pathname	pn;	/*  path name  */
	struct cfun	cf;	/*  compiled function  */
	struct cclosure	cc;	/*  compiled closure  */
/*	struct spice	spc;	/*  spice  */

	struct dummy	d;	/*  dummy  */

	struct fixarray	fixa;	/*  fixnum array  */
	struct sfarray	sfa;	/*  short-float array  */
	struct lfarray	lfa;	/*  long-float array  */
#ifdef CLOS
	struct instance in;	/*  clos instance */
	struct gfun	gf;	/*  generic function */
#else
	struct structure
			str;	/*  structure  */
#endif CLOS
#ifdef MTCL
	struct cont     cn;	/*  continuation  */
	struct thread   th;	/*  thread  */
#endif MTCL
};

/* Used for initalizing array of keywords for compiled functions */
typedef union { int i; object o;} intUobject;

/*
	The struct of free lists.
*/
struct freelist {
	HEADER;
	object	f_link;
};

#define	FREE	(255)		/*  free object  */

/*
	Implementation types.
*/
enum type {
	t_cons = 0,
#ifdef APOLLO
	t_start = 0,
#else
	t_start = t_cons,
#endif APOLLO
	t_bignum,		/* 1 */
	t_ratio,		/* 2 */
	t_shortfloat,		/* 3 */
	t_longfloat,		/* 4 */
	t_complex,		/* 5 */
	t_symbol,		/* 6 */
	t_package,		/* 7 */
	t_hashtable,		/* 8 */
	t_array,		/* 9 */
	t_vector,		/* a */
	t_string,		/* b */
	t_bitvector,		/* c */
	t_stream,		/* d */
	t_random,		/* e */
	t_readtable,		/* f */
	t_pathname,		/* 10 */
	t_cfun,			/* 11 */
	t_cclosure,		/* 12 */
/*	t_spice,		/* 13 */
#ifdef CLOS
	t_instance,		/* 14 */
	t_gfun,			/* 15 */
#else
	t_structure,		/* 14 */
#endif CLOS
#ifdef MTCL
	t_cont,			/* 16	15 */
	t_thread,		/* 17	16 */
#endif
	t_end,
	t_contiguous,		/*  contiguous block  */
	t_relocatable,		/*  relocatable block  */
	t_other,		/*  other  */
	t_fixnum,		/*  immediate fixnum */
	t_character,		/*  immediate character */
	t_locative,		/*  locative (also used as spice) */
	t_spice = t_locative
};


/*
	Type map.

	enum type type_map[MAXPAGE];
*/
extern char type_map[MAXPAGE];

/*
	Type_of.
*/
#define	type_of(obje)	((enum type)(IMMEDIATE(obje) ? \
				       ((int)t_other + IMMEDIATE(obje)) \
				       : (((object)(obje)) ->d.t)))

#define	ENDP(x)	(type_of(x) == t_cons ? \
		 FALSE : x == Cnil ? TRUE : \
		 (FEwrong_type_argument(Slist, x), FALSE))

/*
	Storage manager for each type.
*/
struct typemanager {
	enum type
		tm_type;	/*  type  */
	int	tm_size;	/*  element size in bytes  */
	int	tm_nppage;	/*  number per page  */
	object	tm_free;	/*  free list  */
				/*  Note that it is of type object.  */
	int	tm_nfree;	/*  number of free elements  */
	int	tm_nused;	/*  number of elements used  */
	int	tm_npage;	/*  number of pages  */
	int	tm_maxpage;	/*  maximum number of pages  */
	char	*tm_name;	/*  type name  */
	int	tm_gccount;	/*  GC count  */
};

/*
	The table of type managers.
*/
extern struct typemanager tm_table[(int)t_end];

#define	tm_of(t)	(&(tm_table[(int)tm_table[(int)(t)].tm_type]))

/*
	Contiguous block header.
*/
struct contblock {		/*  contiguous block header  */
	int	cb_size;	/*  size in bytes  */
	struct contblock
		*cb_link;	/*  contiguous block link  */
};

/*
	The pointer to the contiguous blocks.
*/
extern struct contblock *cb_pointer;	/*  contblock pointer  */

/*
	Variables for memory management.
*/
extern int ncb;			/*  number of contblocks  */
extern int ncbpage;			/*  number of contblock pages  */
extern int maxcbpage;			/*  maximum number of contblock pages  */
extern int cbgccount;			/*  contblock gc count  */

extern int holepage;			/*  hole pages  */
extern int nrbpage;			/*  number of relblock pages  */
extern int rbgccount;			/*  relblock gc count  */

extern char *rb_start;			/*  relblock start  */
extern char *rb_end;			/*  relblock end  */
extern char *rb_limit;			/*  relblock limit  */
extern char *rb_pointer;		/*  relblock pointer  */
extern char *rb_start1;			/*  relblock start in copy space  */
extern char *rb_pointer1;		/*  relblock pointer in copy space  */

extern char *heap_end;			/*  heap end  */
extern char *data_end;			/*  core end  */

/* MTCL: If you make it bigger, the bug is less frequent */
#define	HOLEPAGE	128

#ifdef SYSV
#undef HOLEPAGE
#define	HOLEPAGE	32
#endif SYSV

#define	INIT_HOLEPAGE	150
#define	INIT_NRBPAGE	50
#define	RB_GETA		512

#define	endp(obje)	endp1(obje)

#define	TIME_ZONE	(-1)

#define	isUpper(xxx)	(((xxx)&0200) == 0 && isupper(xxx))
#define	isLower(xxx)	(((xxx)&0200) == 0 && islower(xxx))
#define	isDigit(xxx)	(((xxx)&0200) == 0 && isdigit(xxx))

#ifdef WORDS_BIGENDIAN
#define HIND 0
#define LIND 1
#else /* little endian */
#define HIND 1  /* (int) of double where the exponent and most signif is */
#define LIND 0  /* low part of a double */
#endif WORDS_BIGENDIAN
