/*
    gbc.c -- Garbage collector.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi and William F. Schelter.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#define DEBUG

#include "config.h"
#include "page.h"

/******************************* EXPORTS ******************************/

bool GC_enable;
int gc_time;			/* Beppe */

/******************************* IMPORTS ******************************/

extern int real_maxpage;
extern int new_holepage;

/******************************* ------- ******************************/

/*
   Loading a full 32 bit word and shifting is faster than byte operations.
   mark_table[m]:i represents word w = 128*m + 4*i, where m = addr-DATA_START.
   Therefore m = w >> 7, i = (w / 4) % 32 = (w >> 2) & 0x1f.
*/

#define MTbit(x)	(((((int)x) >> 2) & 0x1f))
#define MTword(x)	mark_table[((int)x - DATA_START) >> 7]
#define get_mark_bit(x) (MTword(x) >> MTbit(x) & 1)
#define set_mark_bit(x) (MTword(x) |= (1 << MTbit(x)))

extern bool saving_system;

static int gc_start = 0;	/* Beppe */

char *copy_relblock();

struct apage {
	char apage_self[LISP_PAGESIZE];
};


extern char *heap_end;
extern char *data_end;

#define	inheap(pp)	((char *)(pp) < heap_end)
/* We want to check that:
   (char *) DATA_START <= (char *) pp < heap_end
   We must consider how to handle negative pp.
   On machines (e.g. VAX) where (char *) = int, a negative pp will fail the
   first comparison.
   On machines (e.g. sun, hp) where (char *) = (unsigned int), a negative
   pp will fail the second test, provided (int) heap_end >= 0, i.e.
   heap_end < 2^31.
   */
#define VALID_DATA_ADDRESS(pp) \
   !IMMEDIATE(pp) && (char *)DATA_START <= (char *)(pp) && inheap(pp)

int maxpage;

object siVgc_verbose;

#ifdef DEBUG
bool debug;
object siVgc_message;
#endif DEBUG

#define	MARK_ORIGIN_MAX		300

#define	symbol_marked(x)	((x)->d.m)

object *mark_origin[MARK_ORIGIN_MAX];
int mark_origin_max;

int *mark_table;

enum type what_to_collect;

enter_mark_origin(object *p)
{
	if (mark_origin_max >= MARK_ORIGIN_MAX)
		error("too many mark origins");
	mark_origin[mark_origin_max++] = p;
}

mark_cons(object x)
{
#ifdef DOWN_STACK
	if ((int *)(&x) < cs_limit)
#else
	if ((int *)(&x) > cs_limit)
#endif
		error("control stack overflow in GC");

	/*  x is already marked.  */
BEGIN:
	if (CAR(x) == OBJNULL)
		;
	else if (type_of(CAR(x)) == t_cons) {
		if (CAR(x)->c.m)
			;
		else {
			CAR(x)->c.m = TRUE;
			mark_cons(CAR(x));
		}
	} else
		mark_object(CAR(x));
	x = CDR(x);
	if (x == OBJNULL)
		return;
	if (type_of(x) == t_cons) {
		if (x->c.m)
			return;
		x->c.m = TRUE;
		goto BEGIN;
	}
	if (Null(x))
		return;
	mark_object(x);
}

/* Whenever two arrays are linked together by displacement,
   if one is live, the other will be made live */
#define mark_displaced_field(ar) mark_object(ar)
/* Original version
#define mark_displaced_field(y) { y->c.m = TRUE;\
			mark_object(CAR(y));\
			for (y = CDR(y);  y != Cnil;  y = CDR(y))\
				y->c.m = TRUE;}
*/

mark_object(object x)
{
	int i, j;
	object *p;
	char *cp;
	object y;

#ifdef DOWN_STACK
	if ((int *)(&x) < cs_limit)
#else
	if ((int *)(&x) > cs_limit)
#endif
		error("control stack overflow in GC");

begin:

	if (IMMEDIATE(x)) return;	/* fixnum, character or locative */
	if (x == OBJNULL)
		return;
	if (x->d.m)
		return;
	x->d.m = TRUE;

	switch (type_of(x)) {

#ifndef IMMEDIATE
	case t_fixnum:
		break;
#endif IMMEDIATE

	case t_bignum:
	bignum:
		x = (object)(x->big.big_cdr);
		if ((struct bignum *)x == NULL)
			break;
		x->d.m = TRUE;
		goto bignum;

	case t_ratio:
		mark_object(x->rat.rat_num);
		x = x->rat.rat_den;
		goto begin;

	case t_shortfloat:
	case t_longfloat:
		break;

	case t_complex:
		mark_object(x->cmp.cmp_imag);
		x = x->cmp.cmp_real;
		goto begin;

	case t_character:
		break;

	case t_symbol:
		mark_object(x->s.s_plist);
		mark_object(x->s.s_gfdef);
		mark_object(x->s.s_dbind);
		if (x->s.s_self == NULL)
			break;
		if ((int)what_to_collect >= (int)t_contiguous) {
			if (inheap(x->s.s_self)) {
				if (what_to_collect == t_contiguous)
					mark_contblock(x->s.s_self,
						       x->s.s_fillp+1
						       );
			} else
				x->s.s_self =
				copy_relblock(x->s.s_self, x->s.s_fillp+1);
		}
		break;

	case t_package:
		mark_object(x->p.p_name);
		mark_object(x->p.p_nicknames);
		mark_object(x->p.p_shadowings);
		mark_object(x->p.p_uselist);
		mark_object(x->p.p_usedbylist);
		if (what_to_collect != t_contiguous)
			break;
		if (x->p.p_internal != NULL)
			mark_contblock((char *)(x->p.p_internal),
				       PHTABSIZE*sizeof(object));
		if (x->p.p_external != NULL)
			mark_contblock((char *)(x->p.p_external),
				       PHTABSIZE*sizeof(object));
		break;

	case t_cons:
		mark_cons(x);
		break;

	case t_hashtable:
		mark_object(x->ht.ht_rhsize);
		mark_object(x->ht.ht_rhthresh);
		if (x->ht.ht_self == NULL)
			break;
		for (i = 0, j = x->ht.ht_size;  i < j;  i++) {
			mark_object(x->ht.ht_self[i].hte_key);
			mark_object(x->ht.ht_self[i].hte_value);
		}
		if ((short)what_to_collect >= (short)t_contiguous) {
			if (inheap(x->ht.ht_self)) {
				if (what_to_collect == t_contiguous)
				    mark_contblock((char *)(x->ht.ht_self),
					           j * sizeof(struct htent));
			} else
				x->ht.ht_self = (struct htent *)
				copy_relblock((char *)(x->ht.ht_self),
					      j * sizeof(struct htent));
		}
		break;

	case t_array:
		if ((y = x->a.a_displaced) != Cnil)
		  mark_displaced_field(y);
		if ((int)what_to_collect >= (int)t_contiguous &&
		    x->a.a_dims != NULL) {
			if (inheap(x->a.a_dims)) {
				if (what_to_collect == t_contiguous)
				    mark_contblock((char *)(x->a.a_dims),
					           sizeof(int)*x->a.a_rank);
			} else
				x->a.a_dims = (int *)
				copy_relblock((char *)(x->a.a_dims),
					      sizeof(int)*x->a.a_rank);
		}
		if ((enum aelttype)x->a.a_elttype == aet_ch)
			goto CASE_STRING;
		if ((enum aelttype)x->a.a_elttype == aet_bit)
			goto CASE_BITVECTOR;
		if ((enum aelttype)x->a.a_elttype == aet_object)
			goto CASE_ARRAY_OBJECT;

	CASE_ARRAY_SPECIAL:
		cp = (char *)(x->fixa.fixa_self);
		if (cp == NULL)
			break;
		if ((enum aelttype)x->a.a_elttype == aet_lf)
			j = sizeof(longfloat)*x->lfa.lfa_dim;
		else
			j = sizeof(fixnum)*x->fixa.fixa_dim;
		goto COPY;

	CASE_ARRAY_OBJECT:
		p = x->a.a_self;
		if (p == NULL)
			break;
		if (Null(CAR(x->a.a_displaced)))
			for (i = 0, j = x->a.a_dim;  i < j;  i++)
				mark_object(p[i]);
		cp = (char *)p;
		j *= sizeof(object);
	COPY:
		if ((int)what_to_collect >= (int)t_contiguous) {
			if (inheap(cp)) {
				if (what_to_collect == t_contiguous)
					mark_contblock(cp, j);
			} else if (Null(x->a.a_displaced))
				x->a.a_self = (object *)copy_relblock(cp, j);
			else if (Null(CAR(x->a.a_displaced))) {
				i = (int)(object *)copy_relblock(cp, j)
				  - (int)(x->a.a_self);
				adjust_displaced(x, i);
			}
		}
		break;

	case t_vector:
		if ((y = x->v.v_displaced) != Cnil)
		  mark_displaced_field(y);
		if ((enum aelttype)x->v.v_elttype == aet_object)
			goto CASE_ARRAY_OBJECT;
		else
			goto CASE_ARRAY_SPECIAL;

	CASE_STRING:
	case t_string:
		if ((y = x->st.st_displaced) != Cnil)
		  mark_displaced_field(y);
		j = x->st.st_dim;
		cp = x->st.st_self;
		if (cp == NULL)
			break;
	COPY_STRING:
		if ((int)what_to_collect >= (int)t_contiguous) {
			if (inheap(cp)) {
				if (what_to_collect == t_contiguous)
					mark_contblock(cp, j);
			} else if (Null(x->st.st_displaced))
				x->st.st_self = copy_relblock(cp, j);
			else if (Null(CAR(x->st.st_displaced))) {
				i = copy_relblock(cp, j) - cp;
				adjust_displaced(x, i);
			}
		}
		break;

	CASE_BITVECTOR:
	case t_bitvector:
		if ((y = x->bv.bv_displaced) != Cnil)
		  mark_displaced_field(y);
/* We make bitvectors multiple of sizeof(int) in size allocated
 Assume 8 = number of bits in char */

#define W_SIZE (8*sizeof(int))
		j= sizeof(int) *
		   ((x->bv.bv_offset + x->bv.bv_dim + W_SIZE -1)/W_SIZE);
		cp = x->bv.bv_self;
		if (cp == NULL)
			break;
		goto COPY_STRING;

#ifndef CLOS
	case t_structure:
		mark_object(x->str.str_name);
		p = x->str.str_self;
		if (p == NULL)
			break;
		for (i = 0, j = x->str.str_length;  i < j;  i++)
			mark_object(p[i]);
		if ((int)what_to_collect >= (int)t_contiguous) {
			if (inheap(x->str.str_self)) {
				if (what_to_collect == t_contiguous)
					mark_contblock((char *)p,
						       j*sizeof(object));

			} else
				x->str.str_self = (object *)
				copy_relblock((char *)p, j*sizeof(object));
		}
		break;
#endif CLOS

	case t_stream:
		switch ((enum smmode)x->sm.sm_mode) {
		case smm_input:
		case smm_output:
		case smm_io:
		case smm_probe:
			mark_object(x->sm.sm_object0);
			mark_object(x->sm.sm_object1);
			if (saving_system) {
			  FILE *fp = x->sm.sm_fp;
			  if (fp != 0 && fp != stdin && fp !=stdout) {
			    fclose(fp);
			    x->sm.sm_fp = 0;
			  }
			}
			else if (what_to_collect == t_contiguous &&
				 x->sm.sm_fp &&
				 x->sm.sm_fp->_BASE &&
				 x->sm.sm_fp->_BASE != BASEFF)
			  mark_contblock(x->sm.sm_fp->_BASE, BUFSIZ);
			break;

		case smm_synonym:
			mark_object(x->sm.sm_object0);
			break;

		case smm_broadcast:
		case smm_concatenated:
			mark_object(x->sm.sm_object0);
			break;

		case smm_two_way:
		case smm_echo:
			mark_object(x->sm.sm_object0);
			mark_object(x->sm.sm_object1);
			break;

		case smm_string_input:
		case smm_string_output:
			mark_object(x->sm.sm_object0);
			break;

		default:
			error("mark stream botch");
		}
		break;

	case t_random:
		break;

	case t_readtable:
		if (x->rt.rt_self == NULL)
			break;
		if (what_to_collect == t_contiguous)
			mark_contblock((char *)(x->rt.rt_self),
				       RTABSIZE*sizeof(struct rtent));
		for (i = 0;  i < RTABSIZE;  i++) {
			mark_object(x->rt.rt_self[i].rte_macro);
			if (x->rt.rt_self[i].rte_dtab != NULL) {
/**/
	if (what_to_collect == t_contiguous)
		mark_contblock((char *)(x->rt.rt_self[i].rte_dtab),
			       RTABSIZE*sizeof(object));
	for (j = 0;  j < RTABSIZE;  j++)
		mark_object(x->rt.rt_self[i].rte_dtab[j]);
/**/
			}
		}
		break;

	case t_pathname:
		mark_object(x->pn.pn_host);
		mark_object(x->pn.pn_device);
		mark_object(x->pn.pn_directory);
		mark_object(x->pn.pn_name);
		mark_object(x->pn.pn_type);
		mark_object(x->pn.pn_version);
		break;

	case t_cfun: {
	  	struct codeblock *block = x->cf.cf_block;
		mark_object(x->cf.cf_name);
		if (block) {
		  mark_object(block->cd_data);
		  if (what_to_collect == t_contiguous) {
		    if (get_mark_bit((int *)(block->cd_start)))
		      break;
		    mark_contblock(block->cd_start, block->cd_size);
		  }
		}
	      }
		break;

	case t_cclosure: {
	  	struct codeblock *block = x->cc.cc_block;
		mark_object(x->cc.cc_env);
		if (block) {
		  mark_object(block->cd_data);
		  if (what_to_collect == t_contiguous) {
		    if (get_mark_bit((int *)(block->cd_start)))
		      break;
		    mark_contblock(block->cd_start, block->cd_size);
		  }
		}
	      }
		break;

/*	case t_spice:
		break;
*/

#ifdef THREADS
	case t_cont:
		mark_object(x->cn.cn_thread);
		break;

	case t_thread:
/* Already marked by malloc
 		mark_contblock(x->th.th_self, x->th.th_size);
 */
		mark_object(x->th.th_fun);
		break;
#endif THREADS
#ifdef CLOS
	case t_instance:
		mark_object(x->in.in_class);
		p = x->in.in_slots;
		if (p == NULL)
			break;
		for (i = 0, j = x->in.in_length;  i < j;  i++)
			mark_object(p[i]);
		if ((int)what_to_collect >= (int)t_contiguous) {
			if (inheap(x->in.in_slots)) {
				if (what_to_collect == t_contiguous)
					mark_contblock((char *)p,
						       j*sizeof(object));

			} else
				x->in.in_slots = (object *)
				copy_relblock((char *)p, j*sizeof(object));
		}
		break;

	case t_gfun:
		mark_object(x->gf.gf_name);
		mark_object(x->gf.gf_meth_ht);
		mark_object(x->gf.gf_gfun);
		p = x->gf.gf_spec_how;
		if (p == NULL)
			break;
		for (i = 0, j = x->gf.gf_arg_no;  i < j;  i++)
			mark_object(p[i]);
		if ((int)what_to_collect >= (int)t_contiguous) {
			if (inheap(x->gf.gf_spec_how)) {
				if (what_to_collect == t_contiguous)
					mark_contblock((char *)p,
						       j*sizeof(object));

			} else
				x->gf.gf_spec_how = (object *)
				copy_relblock((char *)p, j*sizeof(object));
		}
		break;
#endif CLOS
	default:
#ifdef DEBUG
		if (debug)
			printf("\ttype = %d\n", type_of(x));
#endif DEBUG
		error("mark botch");
	}
}

mark_stack_conservative(int *top, int *bottom)
{ int p, m, pageoffset;
  object x;
  struct typemanager *tm;
  register int *j;

  jmp_buf buf;
  /* ensure flushing of register caches */
  if (_setjmp(buf) == 0) _longjmp(buf, 1);

#ifdef DEBUG
  if (debug) { printf("Traversing C stack .."); fflush(stdout); }
#endif DEBUG

  /* On machines which align local pointers on multiple of 2 rather
     than 4 we need to mark twice

  if (offset) mark_stack_conservative(bottom, ((char *) top) + offset, 0);
     */
  for (j = top ; j >= bottom ; j--) {
    /* improved Beppe: */
    if (VALID_DATA_ADDRESS(*j)
	&& type_map[p = page(*j)] < (char)t_end
	&& !((pageoffset = (char *)*j - pagetochar(p)) %
	     (tm = tm_of((enum type)type_map[p]))->tm_size)) {
      x = (object)*j;
      if ((pageoffset < (tm->tm_size * tm->tm_nppage))
	  && (m = x->d.m) != FREE) {
	if (m == TRUE) continue;
	if (m) {
	  fprintf(stderr,
		  "** bad value %d of d.m in gc page %d skipping mark **",
		  m, p); fflush(stderr);
	} else
	  mark_object(x);
      }
    }}
#ifdef DEBUG
  if (debug) {printf(". done.\n"); fflush(stdout); }
#endif DEBUG
}

mark_phase()
{
	register object *p;
	register int i, j;
	register struct package *pp;
	register bds_ptr bdp;
	register frame_ptr frp;
	register ihs_ptr ihsp;

	mark_object(Cnil);
	mark_object(Ct);

#ifdef THREADS
	{
	  pd *pdp;
	  lpd *old_clwp = clwp;
	  clwp->lwp_vs_top = vs_top;

	  for (pdp = running_head; pdp != (pd *)NULL;
	       pdp = pdp->pd_next) {

	    clwp = pdp->pd_lpd;
	    vs_top = clwp->lwp_vs_top;
#endif THREADS
	    
	    /*  No need to mark VALUES(i). Beppe  */
	    
	    for (bdp = bds_org;  bdp <= bds_top;  bdp++) {
	      mark_object(bdp->bds_sym);
	      mark_object(bdp->bds_val);
	    }
	    
	    for (frp = frs_org;  frp <= frs_top;  frp++)
	      mark_object(frp->frs_val);
	    
	    for (ihsp = ihs_org;  ihsp <= ihs_top;  ihsp++)
	      mark_object(ihsp->ihs_function);
#ifdef THREADS	      
	    /* added to mark newly allocated objects */
	    mark_object(clwp->lwp_alloc_temporary);
	    mark_object(clwp->lwp_bind_temporary);
	    mark_object(clwp->lwp_eval_temporary);
	    mark_object(clwp->lwp_fmt_temporary_stream);
	    mark_object(clwp->lwp_PRINTstream);
	    mark_object(clwp->lwp_PRINTcase);
	    mark_object(clwp->lwp_READtable);
	    mark_object(clwp->lwp_delimiting_char);
	    mark_object(clwp->lwp_big_register_0);
	    mark_object(clwp->lwp_string_register);
	    mark_object(clwp->lwp_gensym_prefix);
	    mark_object(clwp->lwp_gentemp_prefix);
	    mark_object(clwp->lwp_token);
	    
	    /* (current-thread) can return it at any time
	     */
	    mark_object(clwp->lwp_thread);
#endif THREADS	      
	    
	    /* now collect from the c-stack of the thread ... */
	    
	    { int *where, p, m, pageoffset;
	      object x;
	      struct typemanager *tm;
	      register int *j;

#ifdef THREADS
	      if (clwp != old_clwp) /* is not the executing stack */
		where = (int *)pdp->pd_env[SP_INDEX] ;
	      else
#endif THREADS
		where = (int *)&where ;
	      
	      /* If the locals of type object in a C function could be
		 aligned other than on multiples of sizeof (char *)
		 we would have to mark twice */
	      
	      if (where > cs_org)
		mark_stack_conservative(where, cs_org);
	      else
		mark_stack_conservative(cs_org, where);
	    }
#ifdef THREADS
	  }
	  clwp = old_clwp;
	  vs_top = clwp->lwp_vs_top;
	}
#endif THREADS

	/* mark roots */
	for (i = 0;  i < mark_origin_max;  i++)
		mark_object(*mark_origin[i]);

	/* mark packages */
	for (pp = pack_pointer;  pp != NULL;  pp = pp->p_link)
		mark_object((object)pp);

#ifdef DEBUG
	if (debug) {
		printf("symbol navigation\n");
		fflush(stdout);
	}
#endif DEBUG

	for (pp = pack_pointer;  pp != NULL;  pp = pp->p_link) {
	  if (pp->p_internal != NULL)
	    for (i = 0;  i < PHTABSIZE;  i++)
	      mark_object(pp->p_internal[i]);
	  if (pp->p_external != NULL)
	    for (i = 0;  i < PHTABSIZE;  i++)
	      mark_object(pp->p_external[i]);
	}
}

sweep_phase()
{
	register int i, j, k;
	register object x;
	register char *p;
	register struct typemanager *tm;
	register object f;

	Cnil->s.m = FALSE;
	Ct->s.m = FALSE;

#ifdef DEBUG
	if (debug)
		printf("type map\n");
#endif DEBUG
	for (i = 0;  i < maxpage;  i++) {
		if (type_map[i] == (int)t_contiguous) {
			if (debug) {
				printf("-");
			/*
				fflush(stdout);
			*/
				continue;
			}
		}
		if (type_map[i] >= (int)t_end)
			continue;

		tm = tm_of((enum type)type_map[i]);

	/*
		general sweeper
	*/

#ifdef DEBUG
		if (debug) {
			printf("%c", tm->tm_name[0]);
		/*
			fflush(stdout);
		*/
		}
#endif DEBUG
		p = pagetochar(i);
		f = tm->tm_free;
		k = 0;
		for (j = tm->tm_nppage; j > 0; --j, p += tm->tm_size) {
			x = (object)p;
			if (x->d.m == FREE)
				continue;
			else if (x->d.m) {
				x->d.m = FALSE;
				continue;
			}
			switch ((enum type)x->d.t) {
			case t_array:
			case t_vector:
			case t_string:
			case t_bitvector:
				if (CAR(x->a.a_displaced) != Cnil)
					undisplace(x);
			}
			((struct freelist *)x)->f_link = f;
			x->d.m = FREE;
			f = x;
			k++;
		}
		tm->tm_free = f;
		tm->tm_nfree += k;
		tm->tm_nused -= k;

	NEXT_PAGE:
		;
	}
#ifdef DEBUG
	if (debug) {
		putchar('\n');
		fflush(stdout);
	}
#endif DEBUG
}

contblock_sweep_phase()
{
	register int i, j;
	register char *s, *e, *p, *q;
	register struct contblock *cbp;

	cb_pointer = NULL;
	ncb = 0;
	for (i = 0;  i < maxpage;) {
		if (type_map[i] != (int)t_contiguous) {
			i++;
			continue;
		}
		for (j = i+1;
		     j < maxpage && type_map[j] == (int)t_contiguous;
		     j++)
			;	
		s = pagetochar(i);
		e = pagetochar(j);
		for (p = s;  p < e;) {
			if (get_mark_bit((int *)p)) {
				p += 4;
				continue;
			}
			q = p + 4;
			while (q < e && !get_mark_bit((int *)q))
				q += 4;
			insert_contblock(p, q - p);
			p = q + 4;
		}
		i = j + 1;
	}
#ifdef DEBUG
	if (debug) {
		for (cbp = cb_pointer; cbp != NULL; cbp = cbp->cb_link)
			printf("0x%x %d\n", cbp, cbp->cb_size);
		fflush(stdout);
	}
#endif DEBUG
}


int (*GC_enter_hook)() = NULL;
int (*GC_exit_hook)() = NULL;

#ifdef THREADS
/* 
 * We execute the GC routine in the main stack.
 * The idea is to switch over the main stack that is stopped in the intha
 * and to call the GC from there on garbage_parameter. Then you can switch 
 * back after.
 * In addition the interrupt is disabled.
 */
static int i, j;
static struct apage *pp, *qq;
static jmp_buf new_env, old_env;
static int val;
static lpd *old_clwp;
static object *old_vs_top;
static enum type t;
static bool stack_switched = FALSE;

enum type garbage_parameter;

GC(enum type new_name)
{
#ifdef DEBUG
	int tm;
#endif DEBUG
	int gc_start = runtime();

	start_critical_section();
	t = new_name;
        garbage_parameter = new_name ;
#else

void GC(enum type t)
{
  int i, j;
  struct apage *pp, *qq;

#ifdef DEBUG
  int tm;
#endif DEBUG
  int gc_start = runtime();
#endif THREADS

  if (siVgc_verbose->s.s_dbind != Cnil) {
    fprintf(stdout, "\n[GC ..");
    /* To use this should add entries in tm_table for reloc and contig.
       fprintf(stdout, "\n[GC for %d %s pages ..",
       tm_of(t)->tm_npage,
       tm_table[(int)t].tm_name + 1); */
    fflush(stdout);
  }

#ifdef DEBUG
  debug = symbol_value(siVgc_message) != Cnil;
#endif DEBUG

#ifdef THREADS
  if (clwp != &main_lpd)  {
    if (debug) {
      printf("*STACK SWITCH*\n");
      fflush (stdout);
    }

    stack_switched = TRUE;
    val = setjmp(old_env);
    if (val == 0) {
	running_head->pd_env[SP_INDEX] = old_env[SP_INDEX] ; /* informations used by the garbage collector need to be updated */
	clwp->lwp_vs_top = vs_top ; /* now first_time is no more necessary */
	old_clwp = clwp;
	vs_top = main_lpd.lwp_vs_top;
	clwp = &main_lpd;
	longjmp(main_pd.pd_env , 2); /* new line */
      }
  }

  else val = 1;

  if (val == 1) {

#endif THREADS

    if (GC_enter_hook != NULL)
      (*GC_enter_hook)(0);

    if (!GC_enable)
      error("GC is not enabled");
    interrupt_enable = FALSE;

    if (saving_system)
      t = t_contiguous;

    what_to_collect = t;

    if (t == t_contiguous)
      cbgccount++;
    else if (t == t_relocatable)
      rbgccount++;
    else
      tm_table[(int)t].tm_gccount++;

#ifdef DEBUG
    if (debug) {
      if (t == t_contiguous)
	printf("GC entered for collecting contiguous blocks\n");
      else if (t == t_relocatable)
	printf("GC entered for collecting relocatable blocks\n");
      else
	printf("GC entered for collecting %s\n",
	       tm_table[(int)t].tm_name);
      fflush(stdout);
    }
#endif DEBUG

    maxpage = page(heap_end);

    if ((int)t >= (int)t_contiguous) {
      j = maxpage*16;
      /*
	1 page = 512 long word
	512 bit = 16 long word
	*/

      if (t == t_relocatable)
	j = 0;

      if (holepage < new_holepage)
	holepage = new_holepage;

      i = rb_pointer - rb_start;

      if (nrbpage > (real_maxpage-page(heap_end)
		     -holepage-real_maxpage/32)/2) {
	if (i > nrbpage*LISP_PAGESIZE)
	  error("Can't allocate.  Good-bye!.");
	else
	  nrbpage =
	    (real_maxpage-page(heap_end)
	     -holepage-real_maxpage/32)/2;
      }

      if (saving_system)
	rb_start = heap_end;
      else
	rb_start = heap_end + LISP_PAGESIZE*holepage;

      rb_end = rb_start + LISP_PAGESIZE*nrbpage;

      if (rb_start < rb_pointer)
	rb_start1 = (char *)
	  ((int)(rb_pointer + LISP_PAGESIZE-1) & -LISP_PAGESIZE);
      else
	rb_start1 = rb_start;

      rb_pointer = rb_start;
      rb_pointer1 = rb_start1;

      mark_table = (int *)(rb_start1 + i);

      if (rb_end < (char *)&mark_table[j])
	i = (char *)&mark_table[j] - heap_end;
      else
	i = rb_end - heap_end;
      alloc_page(-round_to_page(i));

      for (i = 0;  i < j; i++)
	mark_table[i] = 0;
    }

#ifdef DEBUG
    if (debug) {
      printf("mark phase\n");
      fflush(stdout);
      tm = runtime();
    }
#endif DEBUG
    mark_phase();
#ifdef DEBUG
    if (debug) {
      printf("mark ended (%d)\n", runtime() - tm);
      printf("sweep phase\n");
      fflush(stdout);
      tm = runtime();
    }
#endif DEBUG
    sweep_phase();
#ifdef DEBUG
    if (debug) {
      printf("sweep ended (%d)\n", runtime() - tm);
      fflush(stdout);
    }
#endif DEBUG

    if (t == t_contiguous) {
#ifdef DEBUG
      if (debug) {
	printf("contblock sweep phase\n");
	fflush(stdout);
	tm = runtime();
      }
#endif DEBUG
      contblock_sweep_phase();
#ifdef DEBUG
      if (debug)
	printf("contblock sweep ended (%d)\n",
	       runtime() - tm);
#endif DEBUG
    }

    if ((int)t >= (int)t_contiguous) {

      if (rb_start < rb_start1) {
	j = round_to_page(rb_pointer-rb_start);
	pp = (struct apage *)rb_start;
	qq = (struct apage *)rb_start1;
	for (i = 0;  i < j;  i++)
	  *pp++ = *qq++;
      }

      rb_limit = rb_end - 2*RB_GETA;

    }

#ifdef DEBUG
    if (debug) {
      for (i = 0, j = 0;  i < (int)t_end;  i++) {
	if (tm_table[i].tm_type == (enum type)i) {
	  printf("%13s: %8d used %8d free %4d/%d pages\n",
		 tm_table[i].tm_name,
		 tm_table[i].tm_nused,
		 tm_table[i].tm_nfree,
		 tm_table[i].tm_npage,
		 tm_table[i].tm_maxpage);
	  j += tm_table[i].tm_npage;
	} else
	  printf("%13s: linked to %s\n",
		 tm_table[i].tm_name,
		 tm_table[(int)tm_table[i].tm_type].tm_name);
      }
      printf("contblock: %d blocks %d pages\n", ncb, ncbpage);
      printf("hole: %d pages\n", holepage);
      printf("relblock: %d bytes used %d bytes free %d pages\n",
	     rb_pointer - rb_start, rb_end - rb_pointer, nrbpage);
      printf("GC ended\n");
      fflush(stdout);
    }
#endif DEBUG

    interrupt_enable = TRUE;

    if (saving_system) {
      j = round_to_page(rb_pointer-rb_start);

      heap_end += LISP_PAGESIZE*j;

      data_end = heap_end;

      for (i = 0;  i < maxpage;  i++)
	if ((enum type)type_map[i] == t_contiguous)
	  type_map[i] = (char)t_other;
      cb_pointer = NULL;
      maxcbpage -= ncbpage;
      ncbpage = 0;
      ncb = 0;

      holepage = new_holepage;

      nrbpage -= j;
      if (nrbpage <= 0)
	error("no relocatable pages left");

      rb_start = heap_end + LISP_PAGESIZE*holepage;
      rb_end = rb_start + LISP_PAGESIZE*nrbpage;
      rb_limit = rb_end - 2*RB_GETA;
      rb_pointer = rb_start;
    }

    if (GC_exit_hook != NULL)
      (*GC_exit_hook)();

#ifdef THREADS


    /* 
     * Back in the right stack
     */

    if (stack_switched) {
      if (debug) {
	printf("*STACK BACK*\n");
	fflush (stdout);
      }

      stack_switched = FALSE;

      end_critical_section();	/* you are here by the GC call in intha */
	  
      clwp = old_clwp;
      vs_top = clwp->lwp_vs_top;
      longjmp(old_env,2);
      /*	  stack_back();        */
    }
  }
#endif THREADS

  gc_time += (gc_start = runtime() - gc_start);

  if (siVgc_verbose->s.s_dbind != Cnil) {
    fprintf(stdout, ". finished in %.2f\"]", gc_start/60.0);
    fflush(stdout);
  }

#ifdef unix
  if (interrupt_flag) sigint();
#endif unix

#ifdef THREADS
  end_critical_section();
#endif THREADS
}

siLroom_report(int narg)
{
	int i, j;
	object *tl;

	check_arg(0);

	VALUES(0) = MAKE_FIXNUM(real_maxpage);
	VALUES(1) = MAKE_FIXNUM(available_pages);
	VALUES(2) = MAKE_FIXNUM(ncbpage);
	VALUES(3) = MAKE_FIXNUM(maxcbpage);
	VALUES(4) = MAKE_FIXNUM(ncb);
	VALUES(5) = MAKE_FIXNUM(cbgccount);
	VALUES(6) = MAKE_FIXNUM(holepage);
	VALUES(7) = MAKE_FIXNUM(rb_pointer - rb_start);
	VALUES(8) = MAKE_FIXNUM(rb_end - rb_pointer);
	VALUES(9) = MAKE_FIXNUM(nrbpage);
	VALUES(10) = MAKE_FIXNUM(rbgccount);
	VALUES(11) = Cnil;
	tl = &VALUES(11);
	for (i = 0;  i < (int)t_end;  i++) {
	  if (tm_table[i].tm_type == (enum type)i) {
	    tl = &CDR(*tl = CONS(MAKE_FIXNUM(tm_table[i].tm_nused), Cnil));
	    tl = &CDR(*tl = CONS(MAKE_FIXNUM(tm_table[i].tm_nfree), Cnil));
	    tl = &CDR(*tl = CONS(MAKE_FIXNUM(tm_table[i].tm_npage), Cnil));
	    tl = &CDR(*tl = CONS(MAKE_FIXNUM(tm_table[i].tm_maxpage), Cnil));
	    tl = &CDR(*tl = CONS(MAKE_FIXNUM(tm_table[i].tm_gccount), Cnil));
	  } else {
	    tl = &CDR(*tl = CONS(Cnil, Cnil));
	    tl = &CDR(*tl = CONS(MAKE_FIXNUM(tm_table[i].tm_type), Cnil));
	    tl = &CDR(*tl = CONS(Cnil, Cnil));
	    tl = &CDR(*tl = CONS(Cnil, Cnil));
	    tl = &CDR(*tl = CONS(Cnil, Cnil));
	  }
	}
	RETURN(12);
}

siLreset_gc_count(int narg)
{
	int i;

	check_arg(0);
	cbgccount = 0;
	rbgccount = 0;
	for (i = 0;  i < (int)t_end;  i++)
		tm_table[i].tm_gccount = 0;
}

/*
 *----------------------------------------------------------------------
 *
 * copy_contblock --
 *     copies s bytes from p into rb_pointer1 and then advances both
 *     rb_pointer and rb_pointer1
 *
 * Results:
 *	previous value of rb_pointer.
 *
 * Side effects:
 *	rb_pointer, rb_pointer1 and area starting from rb_pointer1.
 *
 *----------------------------------------------------------------------
 */
char *
copy_relblock(char *p, int s)
{
	register char *q, *e;

	s = round_up(s);
	memcpy(rb_pointer1, p, s);
/*
	e = p + s;
	q = rb_pointer1;
	while (p < e)
		*q++ = *p++;
*/
	q = rb_pointer;
	rb_pointer += s;
	rb_pointer1 += s;
	return(q);
}


/*
 *----------------------------------------------------------------------
 *
 * mark_contblock --
 *     sets the mark bit for words from address p to address p+s.
 *     Both p and p+s are rounded to word boundaries.
 *
 * Results:
 *	none.
 *
 * Side effects:
 *	mark_table
 *
 *----------------------------------------------------------------------
 */

mark_contblock(char *p, int s)
{
/*
	register int q, P;

	if ((enum type)type_map[page(p)] != t_contiguous)
		return;
	q = (int)p + s - 1 | 3;
	p = (char *)((int)p& ~3);
	P = ((int)p + 128) & ~127;	/* first address which goes into next word */
/*
	if (q < P)
	  /* p and q fall within same mark_table word */
	  /* Shifting by 32 is noop, so we must do it in two steps: */
/*
	  MTword(p) |= (~0 << MTbit(p)) & ~(~0 << MTbit(q) << 1);
	else {
	  MTword(p) |= (~0 << MTbit(p));
	  memset(&MTword(P), 0xff, ((q & ~127) - P) >> 5);
	  MTword(q) |= ~(~0 << MTbit(q) << 1);
	}
*/
	register char *q;
	register int *x, *y;

	if ((enum type)type_map[page(p)] != t_contiguous)
		return;
	q = p + s;
	x = (int *)(char *)((int)p&~3);
	y = (int *)(char *)(((int)q+3)&~3);
	for (;  x < y;  x++)
		set_mark_bit(x);
}

Lgc(int narg, object area)
{
	check_arg(1);

	if (area == Ct)
		GC(t_contiguous);
	else if (Null(area))
		GC(t_cons);
	else
		GC(t_relocatable);
	RETURN(0);
}

siLgc_time(int narg)
{
  VALUES(0) = MAKE_FIXNUM(gc_time);
  RETURN(1);
}

init_GC()
{
	make_si_function("ROOM-REPORT", siLroom_report);
	make_si_function("RESET-GC-COUNT", siLreset_gc_count);

	siVgc_verbose = make_si_special("*GC-VERBOSE*", Ct);

#ifdef DEBUG
	siVgc_message = make_si_special("*GC-MESSAGE*", Cnil);
#endif DEBUG

	make_function("GC", Lgc);
	make_si_function("GC-TIME", siLgc_time);

	GC_enable = TRUE;
}
