/*
    alloc.c --	Memory allocation.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


/*
  			Heap and Relocatable Area

	                 heap_end       rb_start      rb_end = data_end
    +------+--------------------+ - - - +------------------+ - - --------+
    | text |        heap        | hole  | relocatable area |      stack  |
    +------+--------------------+ - - - +------------------+ - - --------+
						rb_limit = rb_end - 2*RB_GETA

   The type_map array covers all pages of memory: those not used for objects
   are marked as type t_other.

   The tm_table array holds a struct typemanager for each type, which contains
   the first element of the free list for the type, and other bookkeeping
   information.
*/

#include "config.h"
#include "page.h"
#ifdef BSD
#include <sys/resource.h>
#endif BSD
#ifdef SYSV
#include <ulimit.h>
#endif SYSV

/******************************* EXPORTS ******************************/

int real_maxpage;
int new_holepage;
char type_map[MAXPAGE];
struct typemanager tm_table[(int)t_end];
struct contblock *cb_pointer = NULL;

int ncb;			/*  number of contblocks  */
int ncbpage;			/*  number of contblock pages  */
int maxcbpage;			/*  maximum number of contblock pages  */
int cbgccount;			/*  contblock gc count  */

int holepage;			/*  hole pages  */
int nrbpage;			/*  number of relblock pages  */
int rbgccount;			/*  relblock gc count  */

char *rb_start;			/*  relblock start  */
char *rb_end;			/*  relblock end  */
char *rb_limit;			/*  relblock limit  */
char *rb_pointer;		/*  relblock pointer  */
char *rb_start1;		/*  relblock start in copy space  */
char *rb_pointer1;		/*  relblock pointer in copy space  */

char *heap_end;			/*  heap end  */
char *data_end;			/*  end of data space  */

/******************************* ------- ******************************/

#define SIGINTENTRY 6

object Vignore_maximum_pages;

#ifdef unix
extern VOID *sbrk();
#endif unix

static object malloc_list;

/*
   If (n < 0), allocates n pages starting at heap end, without worring
   about the hole.  Basicallty just get the space from the Operating
   System.
   If (n >= 0) allocates n pages from the hole, shifting the relocatable if
   necessary to increase the hole.
*/

char *
alloc_page(int n)
{
	char *e;
	int m;

	e = heap_end;
	if (n >= 0) {
		if (n >= holepage) {
			holepage = new_holepage + n;
			GC(t_relocatable);
		}
		holepage -= n;
		heap_end += LISP_PAGESIZE*n;
		return(e);
	}
	n = -n;
	m = (data_end - heap_end)/LISP_PAGESIZE;
	if (n <= m)
		return(e);

#ifdef unix
	if (data_end != sbrk(0))
		error("Someone allocated my memory!");
	if (data_end != sbrk(LISP_PAGESIZE*(n - m)))
		error("Can't allocate.  Good-bye!");
#endif unix

	data_end += LISP_PAGESIZE*(n - m);

	return(e);
}

void
add_page_to_freelist(char *p, struct typemanager *tm)
{ enum type t;
  object x, f;
  int i;
  t = tm->tm_type;
  type_map[page(p)] = t;
  f = tm->tm_free;
  for (i = tm->tm_nppage; i > 0; --i, p += tm->tm_size) {
    x = (object)p;
    ((struct freelist *)x)->t = (short)t;
    ((struct freelist *)x)->m = FREE;
    ((struct freelist *)x)->f_link = f;
    f = x;
  }
  tm->tm_free = f;
  tm->tm_nfree += tm->tm_nppage;
  tm->tm_npage++;
}

object
alloc_object(enum type t)
{
	register object obj;
	register struct typemanager *tm;
	register int i;
	register char *p;
	register object x, f;

	switch (t) {
	case t_fixnum:
	  return MAKE_FIXNUM(0); /* Immediate fixnum */
	case t_character:
	  return MAKE_CHARACTER(' ', 0, 0); /* Immediate character */
	}
	
#ifdef THREADS
	start_critical_section(); 
#endif THREADS

ONCE_MORE:
	tm = tm_of(t);

	if (interrupt_flag) {
		interrupt_flag = FALSE;
#ifdef unix
		alarm(0);
#endif unix
		terminal_interrupt(TRUE);
	}

	obj = tm->tm_free;
	if (obj == OBJNULL) {
		if (tm->tm_npage >= tm->tm_maxpage)
			goto CALL_GC;
		if (available_pages < 1) {
			Vignore_maximum_pages->s.s_dbind = Cnil;
			goto CALL_GC;
		}
		p = alloc_page(1);
		add_page_to_freelist(p, tm);
		obj = tm->tm_free;
		/* why this? Beppe
		if (tm->tm_npage >= tm->tm_maxpage)
			goto CALL_GC; */
	}
	tm->tm_free = ((struct freelist *)obj)->f_link;
	--(tm->tm_nfree);
	(tm->tm_nused)++;
	obj->d.t = (short)t;
	obj->d.m = FALSE;
#ifdef THREADS
	/* Now initialize the object so that it can be correctly marked
	 * by the GC
	 */
	switch (t) {
	case t_bignum:
	  obj->big.big_cdr = NULL;
	  break;
	case t_ratio:
	  obj->rat.rat_num = OBJNULL;
	  obj->rat.rat_den = OBJNULL;
	  break;
	case t_shortfloat:
	case t_longfloat:
	  break;
	case t_complex:
	  obj->cmp.cmp_imag = OBJNULL;
	  obj->cmp.cmp_real = OBJNULL;
	  break;
	case t_symbol:
	  obj->s.s_plist = OBJNULL;
	  obj->s.s_gfdef = OBJNULL;
	  obj->s.s_dbind = OBJNULL;
	  obj->s.s_self = NULL;
	  break;
	case t_package:
	  obj->p.p_name = OBJNULL;
	  obj->p.p_nicknames = OBJNULL;
	  obj->p.p_shadowings = OBJNULL;
	  obj->p.p_uselist = OBJNULL;
	  obj->p.p_usedbylist = OBJNULL;
	  obj->p.p_internal = NULL;
	  obj->p.p_external = NULL;
	  break;
	case t_cons:
	  CAR(obj) = OBJNULL;
	  CDR(obj) = OBJNULL;
	  break;
	case t_hashtable:
	  obj->ht.ht_rhsize = OBJNULL;
	  obj->ht.ht_rhthresh = OBJNULL;
	  obj->ht.ht_self = NULL;
	  break;
	case t_array:
	  obj->a.a_displaced = Cnil;
	  obj->a.a_elttype = (short)aet_object;
	  obj->a.a_self = NULL;
	  break;
	case t_vector:
	  obj->a.a_displaced = Cnil;
	  obj->a.a_elttype = (short)aet_object;
	  obj->a.a_self = NULL;
	  break;
	case t_string:
	  obj->st.st_displaced = Cnil;
	  obj->st.st_self = NULL;
	  break;
	case t_bitvector:
	  obj->bv.bv_displaced = Cnil;
	  obj->bv.bv_self = NULL;
	  break;
#ifndef CLOS
	case t_structure:
	  obj->str.str_name = OBJNULL;
	  obj->str.str_self = NULL;
	  break;
#endif CLOS
	case t_stream:
	  obj->sm.sm_mode = (short)smm_synonym;
	  obj->sm.sm_object0 = OBJNULL;
	  break;
	case t_random:
	  break;
	case t_readtable:
	  obj->rt.rt_self = NULL;
	  break;
	case t_pathname:
	  obj->pn.pn_host = OBJNULL;
	  obj->pn.pn_device = OBJNULL;
	  obj->pn.pn_directory = OBJNULL;
	  obj->pn.pn_name = OBJNULL;
	  obj->pn.pn_type = OBJNULL;
	  obj->pn.pn_version = OBJNULL;
	  break;
	case t_cfun:
	  obj->cf.cf_name = OBJNULL;
	  obj->cf.cf_block = NULL;
	  break;
	case t_cclosure:
	  obj->cc.cc_env = OBJNULL;
	  obj->cc.cc_block = NULL;
	  break;
/*
	case t_spice:
	  break;
*/
	case t_cont:
	  obj->cn.cn_thread = OBJNULL;
	  break;
	case t_thread:
	  obj->th.th_fun = OBJNULL;
	  break;
#ifdef CLOS
	case t_instance:
	  obj->in.in_class = OBJNULL;
	  obj->in.in_slots = NULL;
	  break;
	case t_gfun:
	  obj->gf.gf_name = OBJNULL;
	  obj->gf.gf_meth_ht = OBJNULL;
	  obj->gf.gf_gfun = OBJNULL;
	  obj->gf.gf_spec_how = NULL;
	  break;
#endif CLOS
	default:
	  printf("\ttype = %d\n", t);
	  error("alloc botch.");
	}
	clwp->lwp_alloc_temporary = obj;
	end_critical_section();
#endif THREADS
	return(obj);

CALL_GC:
	GC(tm->tm_type);
	if (tm->tm_nfree != 0 &&
		(float)tm->tm_nfree * 10.0 >= (float)tm->tm_nused)
		goto ONCE_MORE;

/*	EXHAUSTED:	*/
	if (symbol_value(Vignore_maximum_pages) != Cnil) {
		if (tm->tm_maxpage/2 <= 0)
			tm->tm_maxpage += 1;
		else
			tm->tm_maxpage += tm->tm_maxpage/2;
		goto ONCE_MORE;
	}
	GC_enable = FALSE;
	{ object s = make_simple_string(tm_table[(int)t].tm_name+1);
	GC_enable = TRUE;
	CEerror("The storage for ~A is exhausted.~%\
Currently, ~D pages are allocated.~%\
Use ALLOCATE to expand the space.",
		"Continues execution.",
		2, s, MAKE_FIXNUM(tm->tm_npage));
	}
	goto ONCE_MORE;
}

object
make_cons(object a, object d)
{
	register object obj;
	register int i;
	register char *p;
	register object x, f;
	struct typemanager *tm=(&tm_table[(int)t_cons]);

#ifdef THREADS
	start_critical_section(); 
#endif THREADS

ONCE_MORE:
	if (interrupt_flag) {
		interrupt_flag = FALSE;
#ifdef unix
		alarm(0);
#endif unix
		terminal_interrupt(TRUE);
	}
	obj = tm->tm_free;
	if (obj == OBJNULL) {
		if (tm->tm_npage >= tm->tm_maxpage)
			goto CALL_GC;
		if (available_pages < 1) {
			Vignore_maximum_pages->s.s_dbind = Cnil;
			goto CALL_GC;
		}
		p = alloc_page(1);
		add_page_to_freelist(p,tm);
		obj = tm->tm_free;
		if (tm->tm_npage >= tm->tm_maxpage)
			goto CALL_GC;
	}
	tm->tm_free = ((struct freelist *)obj)->f_link;
	--(tm->tm_nfree);
	(tm->tm_nused)++;
	obj->c.t = (short)t_cons;
	obj->c.m = FALSE;
	CAR(obj) = a;
	CDR(obj) = d;
#ifdef THREADS
	end_critical_section();
#endif THREADS
	return(obj);

CALL_GC:
	GC(t_cons);
	if (tm->tm_nfree != 0 &
	    (float)tm->tm_nfree * 10.0 >= (float)tm->tm_nused)
		goto ONCE_MORE;

/*	EXHAUSTED:	*/
	if (symbol_value(Vignore_maximum_pages) != Cnil) {
		if (tm->tm_maxpage/2 <= 0)
			tm->tm_maxpage += 1;
		else
			tm->tm_maxpage += tm->tm_maxpage/2;
		goto ONCE_MORE;
	}
	CEerror("The storage for CONS is exhausted.~%\
Currently, ~D pages are allocated.~%\
Use ALLOCATE to expand the space.",
		"Continues execution.",
		1, MAKE_FIXNUM(tm->tm_npage));
	goto ONCE_MORE;
#undef	tm
}

char *
alloc_contblock(int n)
{
	register char *p;
	register struct contblock **cbpp;
	register int i;
	register int m;
	register bool g;
	bool gg;

	g = FALSE;
	n = round_up(n);

#ifdef THREADS
	start_critical_section(); 
#endif THREADS
ONCE_MORE:
	if (interrupt_flag) {
		interrupt_flag = FALSE;
		gg = g;
		terminal_interrupt(TRUE);
		g = gg;
	}

	/* Use extra indirection so that cb_pointer can be updated */
	for (cbpp = &cb_pointer; (*cbpp) != NULL; cbpp = &(*cbpp)->cb_link) 
		if ((*cbpp)->cb_size >= n) {
			p = (char *)(*cbpp);
			i = (*cbpp)->cb_size - n;
			*cbpp = (*cbpp)->cb_link;
			--ncb;
			insert_contblock(p+n, i);
#ifdef THREADS
			end_critical_section();
#endif THREADS
			return(p);
		}
	m = round_to_page(n);
	if (ncbpage + m > maxcbpage || available_pages < m) {
		if (available_pages < m)
			Vignore_maximum_pages->s.s_dbind = Cnil;
		if (!g) {
			GC(t_contiguous);
			g = TRUE;
			goto ONCE_MORE;
		}
		if (symbol_value(Vignore_maximum_pages) != Cnil) {
			if (maxcbpage/2 <= 0)
				maxcbpage += 1;
			else
				maxcbpage += maxcbpage/2;
			g = FALSE;
			goto ONCE_MORE;
		}
		CEerror("Contiguous blocks exhausted.~%\
Currently, ~D pages are allocated.~%\
Use ALLOCATE-CONTIGUOUS-PAGES to expand the space.",
			"Continues execution.", 1, MAKE_FIXNUM(ncbpage));
		g = FALSE;
		goto ONCE_MORE;
	}
	p = alloc_page(m);

	for (i = 0;  i < m;  i++)
		type_map[page(p) + i] = (char)t_contiguous;
	ncbpage += m;
	insert_contblock(p+n, LISP_PAGESIZE*m - n);
#ifdef THREADS
	end_critical_section();
#endif THREADS
	return(p);
}

insert_contblock(char *p, int s)
{
	struct contblock **cbpp, *cbp;

	if (s < CBMINSIZE)
		return;
	
	ncb++;
	cbp = (struct contblock *)p;
	cbp->cb_size = s;
	for (cbpp = &cb_pointer; *cbpp != NULL; cbpp = &((*cbpp)->cb_link))
		if ((*cbpp)->cb_size >= s) {
			cbp->cb_link = *cbpp;
			*cbpp = cbp;
			return;
		}
	cbp->cb_link = NULL;
	*cbpp = cbp;
}

/*
 * align must be a power of 2 representing the alignment boundary
 * required for the block.
 */
char *
alloc_relblock(int size, int align)
{
	register char *p;
	register bool g;
	bool gg;
	int i;

	g = FALSE;
	size = round_up(size);	/* ensure pointers are aligned at 4 */
	align = round_up(align)-1;

#ifdef THREADS
	start_critical_section(); 
#endif THREADS
ONCE_MORE:
	if (interrupt_flag) {
		interrupt_flag = FALSE;
		gg = g;
		terminal_interrupt(TRUE);
		g = gg;
	}
	p = (char *)((int)(rb_pointer + align) & ~align);
	if (rb_limit - p < size) {
		if (!g) {
			GC(t_relocatable);
			g = TRUE;
#ifdef hp9000s300
			{ float d1 = (float)(rb_limit - rb_pointer),
			    d2 = (float)(rb_limit - rb_start);
			    if (d1 * 10.0 > d2)
			      goto ONCE_MORE;
			  }
#else
			if ((float)(rb_limit - rb_pointer) * 10.0 >
			    (float)(rb_limit - rb_start))
			  goto ONCE_MORE;
#endif hp9000s300

		}
		if (symbol_value(Vignore_maximum_pages) != Cnil) {
			if (nrbpage/2 <= 0)
				i = 1;
			else
				i = nrbpage/2;
			nrbpage += i;
			if (available_pages < 0)
				nrbpage -= i;
			else {
				rb_end = rb_start + LISP_PAGESIZE*nrbpage;
				rb_limit = rb_end - 2*RB_GETA;
				alloc_page(-(holepage + nrbpage));
				g = FALSE;
				goto ONCE_MORE;
			}
		}
		if (rb_limit > rb_end - 2*RB_GETA)
			error("relocatable blocks exhausted");
		rb_limit += RB_GETA;
		CEerror("Relocatable blocks exhausted.~%\
Currently, ~D pages are allocated.~%\
Use ALLOCATE-RELOCATABLE-PAGES to expand the space.",
			"Continues execution.", 1, MAKE_FIXNUM(nrbpage));
		g = FALSE;
		goto ONCE_MORE;
	}
	p = (char *)((int)(rb_pointer + align) & ~align);
	rb_pointer = p + size;
#ifdef THREADS
	end_critical_section();
#endif THREADS
	return(p);
}

init_tm(enum type t, char *name, int elsize, int maxpage)
{
	int i, j;
	struct typemanager *tm = &tm_table[(int)t];

	tm->tm_name = name;
	for (i = (int)t_start, j = i-1;  i < (int)t_end;  i++)
	  if (tm_table[i].tm_size >= elsize &&
	      (j < (int)t_start || tm_table[j].tm_size > tm_table[i].tm_size))
	    j = i;
	if (j >= (int)t_start) {
		tm->tm_type = (enum type)j;
		tm_table[j].tm_maxpage += maxpage;
		return;
	}
	tm->tm_type = t;
	tm->tm_size = round_up(elsize);
	tm->tm_nppage = LISP_PAGESIZE/round_up(elsize);
	tm->tm_free = OBJNULL;
	tm->tm_nfree = 0;
	tm->tm_nused = 0;
	tm->tm_npage = 0;
	tm->tm_maxpage = maxpage;
	tm->tm_gccount = 0;
}

init_alloc()
{
	int i;
#ifdef __GNUC__
	static initialized = FALSE;
	if (initialized) return;
	initialized = TRUE;
#endif __GNUC__

	holepage = INIT_HOLEPAGE;
	new_holepage = HOLEPAGE;
	nrbpage = INIT_NRBPAGE;

#ifdef BSD
#ifdef MSDOS
	real_maxpage = MAXPAGE;
#else
	{
	  struct rlimit data_rlimit;
	  extern etext;

	  getrlimit(RLIMIT_DATA, &data_rlimit);
	  real_maxpage = ((unsigned int)&etext +
			  (unsigned)data_rlimit.rlim_cur)/LISP_PAGESIZE;
	  if (real_maxpage > MAXPAGE) real_maxpage = MAXPAGE;
	}
#endif MSDOS
#endif BSD

#ifdef SYSV
	real_maxpage= ulimit(UL_GMEMLIM)/LISP_PAGESIZE;
	if (real_maxpage > MAXPAGE) real_maxpage = MAXPAGE;
#endif SYSV

#ifdef unix
	heap_end = sbrk(0);
	if (i = ((int)heap_end & (LISP_PAGESIZE - 1)))
	  sbrk(LISP_PAGESIZE - i);
	heap_end = data_end = sbrk(0);
#endif unix

	alloc_page(-(holepage + nrbpage));
	rb_start = rb_pointer = heap_end + LISP_PAGESIZE*holepage;
	rb_end = rb_start + LISP_PAGESIZE*nrbpage;
	rb_limit = rb_end - 2*RB_GETA;

	for (i = 0;  i < MAXPAGE;  i++)
		type_map[i] = (char)t_other;

/*	Initialization must be done in increasing size order:	*/
	init_tm(t_shortfloat, "FSHORT-FLOAT", /* 8 */
		sizeof(struct shortfloat_struct), 1);
	init_tm(t_cons, ".CONS", sizeof(struct cons), 384); /* 12 */
	init_tm(t_longfloat, "LLONG-FLOAT", /* 16 */
		sizeof(struct longfloat_struct), 1);
	init_tm(t_string, "\"STRING", sizeof(struct string), 64); /* 20 */
	init_tm(t_array, "aARRAY", sizeof(struct array), 64); /* 24 */
	init_tm(t_pathname, "pPATHNAME", sizeof(struct pathname), 1); /* 28 */
	init_tm(t_symbol, "|SYMBOL", sizeof(struct symbol), 64); /* 32 */
	init_tm(t_package, ":PACKAGE", sizeof(struct package), 1); /* 36 */
	init_tm(t_bignum, "BBIGNUM", sizeof(struct bignum), 16);
	init_tm(t_ratio, "RRATIO", sizeof(struct ratio), 1);
	init_tm(t_complex, "CCOMPLEX", sizeof(struct complex), 1);
	init_tm(t_hashtable, "hHASH-TABLE", sizeof(struct hashtable), 1);
	init_tm(t_vector, "vVECTOR", sizeof(struct vector), 2);
	init_tm(t_bitvector, "bBIT-VECTOR", sizeof(struct bitvector), 1);
	init_tm(t_stream, "sSTREAM", sizeof(struct stream), 1);
	init_tm(t_random, "$RANDOM-STATE", sizeof(struct random), 1);
	init_tm(t_readtable, "rREADTABLE", sizeof(struct readtable), 1);
	init_tm(t_cfun, "fCFUN", sizeof(struct cfun), 32);
	init_tm(t_cclosure, "cCCLOSURE", sizeof(struct cclosure), 1);
/*	init_tm(t_spice, "!SPICE", sizeof(struct spice), 2); */
#ifndef CLOS
	init_tm(t_structure, "SSTRUCTURE", sizeof(struct structure), 32);
#else
	init_tm(t_instance, "IINSTANCE", sizeof(struct instance), 32);
	init_tm(t_gfun, "GGFUN", sizeof(struct gfun), 32);
#endif CLOS
#ifdef THREADS
	init_tm(t_cont, "?CONT", sizeof(struct cont), 2);
	init_tm(t_thread, "tTHREAD", sizeof(struct thread), 2);
#endif THREADS

	ncb = 0;
	ncbpage = 0;
#ifdef THREADS
	maxcbpage = 2048;
#else
	maxcbpage = 512;
#endif THREADS

#ifdef unix
	malloc_list = Cnil;
	enter_mark_origin(&malloc_list);
#endif unix
}

t_from_type(type)
object type;
{  int t;

   type = coerce_to_string(type);
   for (t = (int)t_start ; t < (int)t_end ; t++) {
     struct typemanager *tm = &tm_table[t];
     if (tm->tm_name &&
	 strncmp((tm->tm_name)+1, type->st.st_self, type->st.st_fillp) == 0)
       return(t);
   }
   FEerror("Unrecognized type", 0);
 }

siLallocate(int narg, object type, object qty, object now)
{
	struct typemanager *tm;
	int c, i, t;
	char *p, *pp;
	object f, x;

	if (narg < 2)
		FEtoo_few_arguments(&narg);
	if (narg > 3)
		FEtoo_many_arguments(&narg);
	t = t_from_type(type);
	if ( !FIXNUMP(qty) ||
	    (i = fix(qty)) < 0)
		FEerror("~A is not a non-negative fixnum.", 1, qty);
	tm = tm_of(t);
	if (tm->tm_npage > i) i = tm->tm_npage;
	tm->tm_maxpage = i;
	if (narg != 3 || now == Cnil || tm->tm_maxpage <= tm->tm_npage) {
	  VALUES(0) = Ct;
	  RETURN(1);
	}
	if (available_pages < tm->tm_maxpage - tm->tm_npage ||
	    (pp = alloc_page(tm->tm_maxpage - tm->tm_npage)) == NULL)
	  FEerror("Can't allocate ~D pages for ~A.", 2, type,
		  make_simple_string(tm->tm_name+1));
	for (;  tm->tm_npage < tm->tm_maxpage;  pp += LISP_PAGESIZE)
	  add_page_to_freelist(pp, tm);
	VALUES(0) = Ct;
	RETURN(1);
}

siLallocated_pages(int narg, object type)
{
	check_arg(1);
	VALUES(0) = MAKE_FIXNUM(tm_of(t_from_type(type))->tm_npage);
	RETURN(1);
}

siLmaxpage(int narg, object type)
{
	check_arg(1);
	VALUES(0) = MAKE_FIXNUM(tm_of(t_from_type(type))->tm_maxpage);
	RETURN(1);
}

siLalloc_contpage(int narg, object qty, object now)
{
	int i, m;
	char *p;

	if (narg < 1)
		FEtoo_few_arguments(&narg);
	if (narg > 2)
		FEtoo_many_arguments(&narg);
	if ( !FIXNUMP(qty) ||
	    (i = fix(qty)) < 0)
		FEerror("~A is not a non-negative fixnum.", 1, qty);
	if (ncbpage > i)
	  FEerror("Can't set the limit for contiguous blocks to ~D,~%\
since ~D pages are already allocated.",
			2, qty, MAKE_FIXNUM(ncbpage));
	maxcbpage = i;
	if (narg < 2 || Null(now)) {
	  VALUES(0) = Ct;
	  RETURN(1);
	}
	m = maxcbpage - ncbpage;
	if (available_pages < m || (p = alloc_page(m)) == NULL)
		FEerror("Can't allocate ~D pages for contiguous blocks.",
			1, qty);
	for (i = 0;  i < m;  i++)
		type_map[page(p + LISP_PAGESIZE*i)] = (char)t_contiguous;
	ncbpage += m;
	insert_contblock(p, LISP_PAGESIZE*m);
	VALUES(0) = Ct;
	RETURN(1);
}

siLncbpage(int narg)
{
  check_arg(0);
  VALUES(0) = MAKE_FIXNUM(ncbpage);
  RETURN(1);
}

siLmaxcbpage(int narg)
{
  check_arg(0);
  VALUES(0) = MAKE_FIXNUM(maxcbpage);
  RETURN(1);
}

siLalloc_relpage(int narg, object qty, object now)
{ int i;

  if (narg < 1)
    FEtoo_few_arguments(&narg);
  if (narg > 2)
    FEtoo_many_arguments(&narg);
  if ( !FIXNUMP(qty) ||
      (i = fix(qty)) < 0)
    FEerror("~A is not a non-negative fixnum.", 1, qty);
  if (nrbpage > i && rb_pointer >= rb_start + LISP_PAGESIZE*i - 2*RB_GETA
      || 2*i > available_pages+2*nrbpage)
    FEerror("Can't set the limit for relocatable blocks to ~D.",
	    1, qty);
  nrbpage = i;
  rb_end = rb_start + LISP_PAGESIZE*i;
  rb_limit = rb_end - 2*RB_GETA;
  alloc_page(-(holepage + nrbpage));
  VALUES(0) = Ct;
  RETURN(1);
}

siLnrbpage(int narg)
{
  check_arg(0);
  VALUES(0) = MAKE_FIXNUM(nrbpage);
  RETURN(1);
}

siLget_hole_size(int narg)
{
  check_arg(0);
  VALUES(0) = MAKE_FIXNUM(new_holepage);
  RETURN(1);
}

siLset_hole_size(int narg, object size)
{
  int i;

  check_arg(1);
  i = fixint(size);
  if (i < 1 || i > available_pages + new_holepage)
    FEerror("Illegal value for the hole size.", 0);
  new_holepage = i;
}

init_alloc_function()
{
  make_si_function("ALLOCATE", siLallocate);
  make_si_function("ALLOCATED-PAGES", siLallocated_pages);
  make_si_function("MAXIMUM-ALLOCATABLE-PAGES", siLmaxpage);
  make_si_function("ALLOCATE-CONTIGUOUS-PAGES", siLalloc_contpage);
  make_si_function("ALLOCATED-CONTIGUOUS-PAGES", siLncbpage);
  make_si_function("MAXIMUM-CONTIGUOUS-PAGES", siLmaxcbpage);
  make_si_function("ALLOCATE-RELOCATABLE-PAGES", siLalloc_relpage);
  make_si_function("ALLOCATED-RELOCATABLE-PAGES", siLnrbpage);
  make_si_function("GET-HOLE-SIZE", siLget_hole_size);
  make_si_function("SET-HOLE-SIZE", siLset_hole_size);

  Vignore_maximum_pages
    = make_special("*IGNORE-MAXIMUM-PAGES*", Ct);
}

#ifdef unix
/*
	UNIX malloc simulator.

	Used by
		getwd, popen, etc.
*/

#undef malloc
#undef calloc
#undef free
#undef cfree
#undef realloc

VOID *
malloc(size_t size)
{
  object x;

#ifdef __GNUC__
  if (!GC_enable && !initflag)
    init_alloc();
#endif __GNUC__

  x = alloc_simple_string(size-1);
  x->st.st_self = alloc_contblock(size);
  malloc_list = make_cons(x, malloc_list);
  return(x->st.st_self);
}

void
free(VOID *ptr)
{
  object *p;

  if (ptr) {
    for (p = &malloc_list;  !endp(*p);  p = &(CDR((*p))))
      if ((CAR((*p)))->st.st_self == ptr) {
	insert_contblock(CAR((*p))->st.st_self, CAR((*p))->st.st_dim);
	CAR((*p))->st.st_self = NULL;
	*p = CDR((*p));
	return;
      }
    FEerror("free(3) error.", 0);
  }
}

VOID *
realloc(VOID *ptr, size_t size)
{
  object x;
  int i, j;

  for (x = malloc_list;  !endp(x);  x = CDR(x))
    if (CAR(x)->st.st_self == ptr) {
      x = CAR(x);
      if (x->st.st_dim >= size) {
	x->st.st_fillp = size;
	return(ptr);
      } else {
	j = x->st.st_dim;
	x->st.st_self = alloc_contblock(size);
	x->st.st_fillp = x->st.st_dim = size;
	memcpy(x->st.st_self, ptr, size);
	insert_contblock(ptr, j);
	return(x->st.st_self);
      }
    }
  FEerror("realloc(3) error.", 0);
}

VOID *
calloc(size_t nelem, size_t elsize)
{
  char *ptr;
  int i;

  ptr = malloc(i = nelem*elsize);
  memset(ptr, 0 , i);
  return(ptr);
}

VOID cfree(VOID *ptr)
{
  free(ptr);
}

/* make f allocate enough extra, so that we can round
   up, the address given to an even multiple.   Special
   case of size == 0 , in which case we just want an aligned
   number in the address range
   */

#define ALLOC_ALIGNED(f, size, align) \
	((align) <= sizeof(long) ? (int)(f)(size) : \
	   ((align) * (((unsigned)(f)(size + (size ? (align) - 1 : 0)) + (align) - 1)/(align))))

VOID *
memalign(unsigned align, unsigned size)
{ object x = alloc_simple_string(size);
  x->st.st_self = (char *)ALLOC_ALIGNED(alloc_contblock, size, align);
  malloc_list = make_cons(x, malloc_list);
  return x->st.st_self;
}

#ifdef WANT_VALLOC
char *
valloc(int size)
{ return memalign(getpagesize(), size);}
#endif WANT_VALLOC
#endif unix
