/* Interactive Printer Queue Viewer -- Queue Info Retrieval */

#include "qview.h"

#include <sys/types.h>
#include <sys/stat.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#if defined(DIRENT) || defined(_POSIX_VERSION)
#include <dirent.h>
#define SDIRENT  struct dirent
#else
#if defined(SVR3) || defined(__SVR3) || defined(SVR4) || defined(__SVR4) || defined(__svr4__) || defined(HAVE_STRING_H) || defined(hpux) || defined(__hpux)
#ifdef SYSNDIR
#include <sys/ndir.h>
#else
#include <ndir.h>
#endif /* SYSNDIR */
#else
#ifdef SYSDIR
#include <sys/dir.h>
#else
#include <dir.h>
#endif /* SYSDIR */
#endif /* SYSV */
#define SDIRENT  struct direct
#endif

char *lock;                     /* Message from lock file */
char *status;                   /* Message from status file */

int nfiles;                     /* Number of queue items */
struct qitem *filelist;         /* Queue */

int compare();                  /* Forward */

/* Chdir to the spool directory.  Die complaining if we can't. */

void
gotospooldir()
{
        char spooldir[BUFSIZ];

        /* Should really consult /etc/printcap to find spool directory;
           for now I'm being lazy... */

        sprintf(spooldir, "%s/%s", spool,
                strcmp(printer, "lp") == 0 ? "lpd" : printer);
        if (chdir(spooldir) != 0)
                panic("can't chdir to spool dir");
}

/* Refresh the queue info.  Return FALSE if no files */

bool
queuerefresh()
{
        FILE *fp;
        char buf[BUFSIZ];
        struct stat sbuf;

        if (lock != NULL) {
                free(lock);
                lock= NULL;
        }
        if (status != NULL) {
                free(status);
                status= NULL;
        }

        if (!readqueue()) {
                lock= strdup("(can't read spool directory)");
                return FALSE;
        }

        if (stat("lock", &sbuf) != 0)
                lock= strdup("(no lock file)");
        else if (sbuf.st_mode & 0100) {
                lock= strdup("(printer down)");
        }
        else if (sbuf.st_mode & 0010)
                lock= strdup("(queueing turned off)");
        else {
                int n;
                int i;
                int pid;
                fp= fopen("lock", "r");
                n= fscanf(fp, "%d\n%s", &pid, buf);
                fclose(fp);
                switch (n) {
                case 2:
                        for (i= 0; i < nfiles; ++i) {
                            if (!strcmp(filelist[i].name, buf)) {
                                filelist[i].active= TRUE;
                                break;
                            }
                        }
                        if (nfiles > 0 && i >= nfiles)
                                lock= strdup("(active file not found)");
                        break;
                case 1:
                        break;
                default:
                        lock= strdup("(no daemon pid in lock file)");
                        break;
                }
        }

        if (nfiles == 0) {
                status= strdup("(queue empty)");
                return FALSE;
        }

        fp= fopen("status", "r");
        if (fp == NULL) {
                status= strdup("(no status file)");
        }
        else {
                if (fgets(buf, sizeof buf, fp) == NULL) {
                        status= strdup("(empty status file)");
                }
                else {
                        char *p= strchr(buf, EOL);
                        if (p != NULL)
                                *p= EOS;
                        if (p == buf)
                                status= strdup("(empty line in status file)");
                        else
                                status= strdup(buf);
                }
                fclose(fp);
        }
        return TRUE;
}

/* Read the queue entries */

bool
readqueue()
{
        DIR *dp;
        SDIRENT *ep;
        int i;

        /* Garbage-collect old file list first */
        for (i= 0; i < nfiles; ++i)
                free(filelist[i].name);
        L_SETSIZE(nfiles, filelist, struct qitem, 0);

        dp= opendir(".");
        if (dp == NULL)
                return FALSE;

        while ((ep= readdir(dp)) != NULL) {
                if (strncmp(ep->d_name, "cf", 2) == 0) {
                        struct qitem it;
                        struct stat sbuf;
                        if (stat(ep->d_name, &sbuf) != 0)
                                continue;
                        it.name= strdup(ep->d_name);
                        it.mtime= sbuf.st_mtime;
                        it.active= FALSE; /* Set later */
                        countdata(ep->d_name, &it);
                        L_APPEND(nfiles, filelist, struct qitem, it);
                }
        }

        closedir(dp);
        if (nfiles > 1)
                qsort(filelist, nfiles, sizeof filelist[0], compare);

        return TRUE;
}

/* Count data files, add data file sizes, set uid from data files' uid */

void
countdata(name, ip)
        char *name;
        struct qitem *ip;
{
        char buf[BUFSIZ];
        int c;

        ip->uid= -1;
        ip->ndata= 0;
        ip->size= 0;

        buf[0]= 'd';
        strcpy(buf+1, name+1);
        for (c= 'A'; c <= 'z'; (c == 'Z' ? c= 'a' : ++c)) {
                struct stat sbuf;
                buf[2]= c;
                if (stat(buf, &sbuf) != 0)
                        break;
                ip->uid= sbuf.st_uid;
                ip->ndata++;
                ip->size += sbuf.st_size;
        }
}

/* Compare routine for qsort */

int
compare(a, b)
        struct qitem *a, *b;
{
        return a->mtime - b->mtime;
}

/* Service routine for draw proc */

#include <pwd.h>
struct passwd *getpwuid();

char *
username(uid)
        int uid;
{
        struct passwd *pw= getpwuid(uid);
        if (pw != NULL)
                return pw->pw_name;
        else {
                static char buf[20];
                sprintf(buf, "#%d", uid);
                return buf;
        }
}
