;;; -*- Mode: LISP; Syntax: Common-lisp; Package: YY; Base: 10 -*-
;;; graphic-image.lisp
;;;
;;;  Copyright (C) 1989,1990,1991 Aoyama Gakuin University
;;;
;;;		All Rights Reserved
;;;
;;; This software is developed for the YY project of Aoyama Gakuin University.
;;; Permission to use, copy, modify, and distribute this software
;;; and its documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notices appear in all copies and that
;;; both that copyright notice and this permission notice appear in 
;;; supporting documentation, and that the name of Aoyama Gakuin
;;; not be used in advertising or publicity pertaining to distribution of
;;; the software without specific, written prior permission.
;;;
;;; This software is made available AS IS, and Aoyama Gakuin makes no
;;; warranty about the software, its performance or its conformity to
;;; any specification. 
;;;
;;; To make a contact: Send E-mail to ida@csrl.aoyama.ac.jp for overall
;;; issues. To ask specific questions, send to the individual authors at
;;; csrl.aoyama.ac.jp. To request a mailing list, send E-mail to 
;;; yyonx-request@csrl.aoyama.ac.jp.
;;;
;;; Authors:
;;;   version 1.3 91/01/22 by t.kosaka

;;; Drawing Method for image
;;; 
;;; Version 1.3 Add method for image handling.

(in-package :yy)

;;; $@%0%i%U%#%/%9%9%H%j!<%`$N0LCV$+$i%+%i!<$r5a$a$k(J
;;; query-color graphic-stream position
;;; ARG.
;;;      graphic-stream  $@%0%i%U%#%/%9%H%j!<%`(J
;;;      position        $@0LCV(J
;;; RET.
;;;      $@%+%i!<%$%s%9%?%s%9(J
(defmethod query-color ((stream graphic-stream) (position position))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
    (query-color-internal stream (position-x position) (position-y position))
    )

;;; $@%0%i%U%#%/%9%9%H%j!<%`$N0LCV$+$i%+%i!<$r5a$a$k4X?t%W%j%_%F%#%V(J
(defun query-color-internal (stream x y)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-translate-transform-xy ((new-x new-y) stream x  y)
     (let ((color-no (yy-protocol-53 (world-territory-no stream) new-x new-y)))
      (code-color color-no))))

;;; $@%0%i%U%#%/%9%9%H%j!<%`$N0LCV$+$i%+%i!<$r5a$a$k(J
;;; query-color-xy graphic-stream x y
;;; ARG.
;;;      graphic-stream  $@%0%i%U%#%/%9%H%j!<%`(J
;;;      x        $@0LCV#X(J
;;;      y        $@0LCV#Y(J
;;; RET.
;;;      $@%+%i!<%$%s%9%?%s%9(J
(defmethod query-color-xy ((stream graphic-stream)
			(x integer)(y integer))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
    (query-color-internal stream x y))


;;; $@%$%a!<%8$N<h$j9~$_(J
;;; get-image graphic-stream position width height 
;;; ARG.
;;;          graphic-stream   =  $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;          position         =  $@%0%i%U%#%/%9%9%H%j!<%`$G$N0LCV%$%s%9%?%s%9(J
;;;          width            =  $@%$%a!<%8$NI}(J
;;;          height           =  $@%$%a!<%8$N9b$5(J
;;; RET.
;;;          image instance
(defmethod get-image ((stream graphic-stream) (pos position) (width integer)
		      (height integer))
  (declare 
   #-CMU
   (inline - + )
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((region-width (- (region-width stream)
			 (world-x-start stream)))
	(region-height (- (region-height stream)
			  (world-y-start stream))))
    (with-translate-transform-xy ((new-x new-y) stream (position-x pos)
				  (position-y pos))
	
	 (get-image-xy-internal (world-territory-no stream)
			 (+ new-x (world-x-start stream))
			 (+ new-y (world-y-start stream))
			    region-width region-height width height))
    ))

;;; $@%$%a!<%8$N<h$j9~$_(JXY $@%9%H%j!<%`MQ(J
;;; get-image-xy graphic-stream x y width height 
;;; ARG.
;;;          graphic-stream   =  $@%0%i%U%#%/%9%9%H%j!<%`(J
;;           x y              =  $@%0%i%U%#%/%9%9%H%j!<%`Fb$G$N0LCV(J
;;;          width            =  $@%$%a!<%8$NI}(J
;;;          height           =  $@%$%a!<%8$N9b$5(J
;;; RET.
;;;          image instance
(defmethod get-image-xy ((stream graphic-stream) (x integer) (y integer)
			 (width integer) (height integer))
  (declare 
   #-CMU
   (inline - + )
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((region-width (- (region-width stream)
			 (world-x-start stream)))
        (region-height (- (region-height stream)
			  (world-y-start stream))))
	
    (with-translate-transform-xy ((new-x new-y) stream x y)
	  (get-image-xy-internal (world-territory-no stream)
                         (+ new-x (world-x-start stream))
			 (+ new-y (world-y-start stream))
			 region-width region-height width height))
    ))

;;; get-image-xy-internal
;;; $@%$%a!<%8$N<h$j9~$_$N%$%s%?!<%J%k4X?t(J
(defun get-image-xy-internal (tno x y region-w region-h width height)
  (declare 
   #-CMU
   (inline > + - length =)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((real-width (if (> region-w (+ x width))
			 width
		       (- region-w x)))
	 (real-height (if (> region-h (+ y height))
			  height
			(- region-h y)))
	 (internal-data (yy-protocol-60 tno x y real-height real-width 2))
	 (image-data (nth 3 internal-data)))

    ;;; $@%+%i!<$+%b%N%/%m$+$rJ,$1$k(J
    (cond 
     ((= (length image-data) (* real-width real-height 4)) ;;; $@%+%i!<(J
      (make-instance 'color-image :image-width real-width
		     :image-height real-height 
		     :line-bytes (* real-width 4)
		     :image-data image-data))
     (t
      (make-instance 'mono-image :image-width real-width
                     :image-height real-height
		     :line-bytes (ceiling (/ real-width 4))
		     :image-data image-data)))
    ))

;;; $@%$%a!<%8%G!<%?$r<h$j9~$`%9%H%j!<%`MQ(J
(defun get-image-xy-stream (stream x y width height)
  (declare 
   #-CMU
   (inline - + )
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((region-width (- (region-width stream)
			 (world-x-start stream)))
	 (region-height (- (region-height stream)
			  (world-y-start stream)))
	 (yy (with-translate-coordinate-stream y stream))
	 (new-yy (if (= y yy)
		     y
		   (+ y height))))

    (with-translate-transform-xy ((new-x new-y) stream x new-yy)
	  (get-image-data-xy-internal (world-territory-no stream)
                         (+ new-x (world-x-start stream))
			 (+ new-y (world-y-start stream))
			 region-width region-height width height))
    ))

;;; get-image-data-xy-internal
;;; $@%$%a!<%8%G!<%?$N<h$j9~$_$N%$%s%?!<%J%k4X?t(J
(defun get-image-data-xy-internal (tno x y region-w region-h width height)
  (declare 
   #-CMU
   (inline > + - length =)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((real-width (if (> region-w (+ x width))
			 width
		       (- region-w x)))
	 (real-height (if (> region-h (+ y height))
			  height
			(- region-h y)))
	 (internal-data (yy-protocol-60 tno x y real-height real-width 2)))
    (values (nth 3 internal-data) real-width real-height)))



;;; $@%0%i%U%#%/%9%H%j!<%`$K%$%a!<%8$rE>Aw$9$k!#(JXY$@MQ(J
;;; put-image-xy stream  image x y &optinal width height 
;;;                                &key (image-x 0) (image-y 0)
;;; ARG.
;;;       stream          = $@%0%i%U%#%/%9%H%j!<%`(J
;;;       x y             = $@%$%a!<%8$rIA2h$9$k0LCV(J
;;;       width           = $@IA2h$9$kI}(J
;;;       height          = $@IA2h$9$k9b$5(J
;;;       image-x         = $@%$%a!<%8Fb$N3+;O0LCV(J
;;;       image-y         = $@%$%a!<%8Fb$N3+;O0LCV(J
(defmethod put-image-xy ((stream graphic-stream) (image image)
			 (x integer) (y integer) 
			 &optional (width (image-width image))
			 (height (image-height image))
			 &key (image-x 0) (image-y 0))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (put-image-xy-internal stream image x y width height image-x image-y))

;;; $@%0%i%U%#%/%9%H%j!<%`$K%$%a!<%8$rE>Aw$9$k!#%]%8%7%g%sMQ(J
;;; put-image stream  image position width height &key (image-x 0) (image-y 0)
;;; ARG.
;;;       stream          = $@%0%i%U%#%/%9%H%j!<%`(J
;;;       position        = $@%$%a!<%8$rIA2h$9$k0LCV$N%$%s%9%?%s%9(J
;;;       width           = $@IA2h$9$kI}(J
;;;       height          = $@IA2h$9$k9b$5(J
;;;       image-x         = $@%$%a!<%8Fb$N3+;O0LCV(J
;;;       image-y         = $@%$%a!<%8Fb$N3+;O0LCV(J
(defmethod put-image ((stream graphic-stream) (image image)
		      (position position) &optional
		      (width (image-width image))
		      (height (image-height image)) &key (image-x 0) (image-y 0))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots 
   (x y) position
   (put-image-xy-internal stream image x y  width height image-x image-y)))

;;; $@%S%C%H%^%C%W$K%$%a!<%8$rE>Aw$9$k!#(JXY$@MQ(J
;;; put-image-xy bitmap  image x y &optinal width height 
;;;                                &key (image-x 0) (image-y 0)
;;; ARG.
;;;       bitmap          = $@%S%C%H%^%C%W(J
;;;       x y             = $@%$%a!<%8$rIA2h$9$k0LCV(J
;;;       width           = $@IA2h$9$kI}(J
;;;       height          = $@IA2h$9$k9b$5(J
;;;       image-x         = $@%$%a!<%8Fb$N3+;O0LCV(J
;;;       image-y         = $@%$%a!<%8Fb$N3+;O0LCV(J
(defmethod put-image-xy ((bitmap bitmap) (image image)
			 (x integer) (y integer) 
			 &optional (width (image-width image))
			 (height (image-height image))
			 &key (image-x 0) (image-y 0))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (put-image-xy-bitmap bitmap image x y width height image-x image-y))

;;; $@%S%C%H%^%C%W$K%$%a!<%8$rE>Aw$9$k!#(J
;;; put-image bitmap image position width height &key (image-x 0) (image-y 0)
;;; ARG.
;;;       bitmap          = $@%S%C%H%^%C%W(J
;;;       position        = $@%$%a!<%8$rIA2h$9$k0LCV$N%$%s%9%?%s%9(J
;;;       width           = $@IA2h$9$kI}(J
;;;       height          = $@IA2h$9$k9b$5(J
;;;       image-x         = $@%$%a!<%8Fb$N3+;O0LCV(J
;;;       image-y         = $@%$%a!<%8Fb$N3+;O0LCV(J
(defmethod put-image ((bitmap bitmap) (image image)
		      (position position) &optional
		      (width (image-width image))
		      (height (image-height image)) &key (image-x 0) (image-y 0))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots 
   (x y) position
   (put-image-xy-bitmap bitmap image x y width height image-x image-y)))

;;; put-image$@IA2h%W%j%_%F%#%V(J $@%S%C%H%^%C%WMQ(J
(defun put-image-xy-bitmap (bitmap image x y width height image-x image-y)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((image-data (image-data image))
         (old-width (image-width image))
         (old-line-bytes (line-bytes image))
         (old-height (image-height image))
         (new-image (clip-image-xy image image-x image-y width height
                                   :create nil))
	 (format1 (if (eq (image-format image) :yy)
		      2
		    1))
	 (format (case (image-type image)
		   (:color
		    (logior #x8000 format1))
		   (:gray
		    (logior #x4000 format1))
		   (t
		    (logior #x2000 format1)))))
    
      (yy-protocol-61 (bitmap-territory-no bitmap)
		      x y (image-width new-image) (image-height new-image)
		     format
		     (image-data new-image))

      (setf (slot-value image 'image-width) old-width
	    (line-bytes image) old-line-bytes
	    (slot-value image 'image-height) old-height
	    (slot-value image 'image-data) image-data)
      bitmap))


;;; put-image$@IA2h%W%j%_%F%#%V(J $@%9%H%j!<%`MQ(J
(defun put-image-xy-internal (stream image x y width height image-x image-y)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((image-data (image-data image))
	 (old-width (image-width image))
	 (old-line-bytes (line-bytes image))
	 (old-height (image-height image))
	 (new-image (clip-image-xy image image-x image-y width height
				   :create nil)))
    
    (put-image-internal-xy stream new-image x y width height)
    
    (setf (slot-value image 'image-width) old-width
	  (line-bytes image) old-line-bytes
	  (slot-value image 'image-height) old-height
	  (slot-value image 'image-data) image-data)
    stream))

;;; $@%$%a!<%8$NI=<(%$%s%?!<%J%k(J $@%9%H%j!<%`MQ(J
(defun put-image-internal-xy (stream image x y width height)
  (declare 
   #-CMU
   (inline + min max =)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-translate-transform-xy ((new-x new-y) stream x y)
    (with-translate-transform-xy ((x1 y1) stream (+ x width) (+ y height))
      (with-temp-region-args ((draw-image-region) (work-region1 stream)
                           :left (min new-x x1) :width width
                           :bottom (min new-y y1) :height height)
       (let* ((tno (world-territory-no stream))
	      (drawing-region (drawing-region stream))
	      (format1 (if (eq (image-format image) :yy)
			   2
			 1))
	   (yy (with-translate-coordinate-stream y stream))
	   (def-y (if (= yy y)
		       0
		      height))
	   (format (case (image-type image)
		     (:color
		      (logior #x8000 format1))
		     (:gray
		      (logior #x4000 format1))
		     (t
		      (logior #x2000 format1)))))


          ;;; $@%o!<%k%I%j!<%8%g%s$NJQ99(J
	 (setf (world-region stream) draw-image-region)

         ;;; $@0LCV$NJQ99(J
	 (setf new-x (+ new-x (world-x-start stream))
	       new-y (- (+ new-y (world-y-start stream)) def-y))
      
          ;;; $@IA2h%W%j%_%F%#%V(J
	 (yy-protocol-61 tno new-x new-y (image-width image) (image-height image)
		      format
		      (image-data image))
          ;;; drawing-region$@$K:#$N%j!<%8%g%s$rDI2C(J
	 (set-drawing-region drawing-region draw-image-region)
	 stream)))))


;;; $@%0%i%U%#%/%9%H%j!<%`$K%0%i%U%#%/%9%9%H%j!<%`$N%$%a!<%8$rIA2h$9$k!#(J
;;; draw-image distination-stream source-stream position width height
;;;            &key image-position
;;; ARG.
;;;       distination-stream   = $@%G%#%9%F%#%M!<%7%g%s%0%i%U%#%/%9%H%j!<%`(J
;;;       source-stream        = $@%=!<%9%0%i%U%#%/%9%H%j!<%`(J
;;;       position             = $@%$%a!<%8$rIA2h$9$k0LCV$N%$%s%9%?%s%9(J
;;;       width                = $@IA2h$9$kI}(J
;;;       height               = $@IA2h$9$k9b$5(J
;;;       image-position       = $@%=!<%9%0%i%U%#%/%9%H%j!<%`Fb$G$N0LCV$N(J
;;;                              $@%$%s%9%?%s%9(J
(defmethod draw-image ((d-stream graphic-stream) (s-stream graphic-stream)
		       (position position) (width integer) (height integer)
		       &key (image-position nil))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots
   (x y) position
   (if image-position
       (draw-image-stream-xy-internal d-stream s-stream
				   x y
				   width height
				   (position-x image-position)
				   (position-y image-position))
     (draw-image-stream-xy-internal d-stream s-stream
				    x y
                                   width height 0 0))))


;;; $@%0%i%U%#%/%9%H%j!<%`$K%0%i%U%#%/%9%9%H%j!<%`$N%$%a!<%8$rIA2h$9$k!#(JXY$@MQ(J
;;; draw-image-xy distination-stream source-stream dx dy width height
;;;            &key image-x image-y
;;; ARG.
;;;       distination-stream   = $@%G%#%9%F%#%M!<%7%g%s%0%i%U%#%/%9%H%j!<%`(J
;;;       source-stream        = $@%=!<%9%0%i%U%#%/%9%H%j!<%`(J
;;;       dx dy                = $@%$%a!<%8$rIA2h$9$k0LCV(J
;;;       width                = $@IA2h$9$kI}(J
;;;       height               = $@IA2h$9$k9b$5(J
;;;       image-x image-y      = $@%=!<%9%0%i%U%#%/%9%H%j!<%`Fb$G$N0LCV(J
(defmethod draw-image-xy ((d-stream graphic-stream) (s-stream graphic-stream)
			  (dx integer) (dy integer) (width integer) 
			  (height integer) &key (image-x 0) (image-y 0))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (draw-image-stream-xy-internal d-stream s-stream
				 dx dy
				 width height
				 image-x image-y))

;;; $@%0%i%U%#%/%9%H%j!<%`$K%S%C%H%^%C%W$N%$%a!<%8$rIA2h$9$k!#(J
;;; draw-image distination-stream source-bitmap position width height
;;;            &key image-position
;;; ARG.
;;;       distination-stream   = $@%G%#%9%F%#%M!<%7%g%s%0%i%U%#%/%9%H%j!<%`(J
;;;       source-bitmap        = $@%=!<%9%S%C%H%^%C%W(J
;;;       position             = $@%$%a!<%8$rIA2h$9$k0LCV$N%$%s%9%?%s%9(J
;;;       width                = $@IA2h$9$kI}(J
;;;       height               = $@IA2h$9$k9b$5(J
;;;       image-position       = $@%=!<%9%0%i%U%#%/%9%H%j!<%`Fb$G$N0LCV$N(J
;;;                              $@%$%s%9%?%s%9(J
(defmethod draw-image ((d-stream graphic-stream) (s-bitmap bitmap)
		       (position position) (width integer) (height integer)
		       &key (image-position nil))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots 
   (x y) position
   (if image-position
       (multiple-value-bind (image-data real-width real-height)
          (get-image-data-bitmap s-bitmap (position-x image-position)
				 (position-y image-position)
				 width height)
	  (let* ((line-byte (if (= (* real-width real-height 4)
				   (length image-data))
				(* real-width 4)
			      (ceiling (/ real-width 4)))))
	    (image-to-graphic-stream d-stream x y
				   real-width real-height 
				   image-data
				   line-byte
				   (if (= line-byte (* real-width 4))
				       :color
				     :mono)
				   0 0)))
     (multiple-value-bind (image-data real-width real-height)
	 (get-image-data-bitmap s-bitmap 0 0 width height)
       (let ((line-byte (if (= (* real-width real-height 4)
			       (length image-data))
			    (* real-width 4)
			  (ceiling (/ real-width 4)))))
	 (image-to-graphic-stream d-stream x y
				  real-width real-height
				  image-data
				  line-byte
				  (if (= line-byte (* real-width 4))
				      :color
				    :mono)
				  0 0)))
    )))


;;; $@%0%i%U%#%/%9%H%j!<%`$K%S%C%H%^%C%W$N%$%a!<%8$rIA2h$9$k!#(JXY$@MQ(J
;;; draw-image-xy distination-stream source-bitmap dx dy width height
;;;            &key image-x image-y
;;; ARG.
;;;       distination-stream   = $@%G%#%9%F%#%M!<%7%g%s%0%i%U%#%/%9%H%j!<%`(J
;;;       source-bitmap        = $@%=!<%9%S%C%H%^%C%W(J
;;;       dx dy                = $@%$%a!<%8$rIA2h$9$k0LCV(J
;;;       width                = $@IA2h$9$kI}(J
;;;       height               = $@IA2h$9$k9b$5(J
;;;       image-x image-y      = $@%=!<%9%0%i%U%#%/%9%H%j!<%`Fb$G$N0LCV(J
(defmethod draw-image-xy ((d-stream graphic-stream) (s-bitmap bitmap)
			  (dx integer) (dy integer) (width integer) 
			  (height integer) &key (image-x 0) (image-y 0))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (multiple-value-bind (image-data real-width real-height)
    (get-image-data-bitmap s-bitmap image-x image-y width height)
    (let ((line-byte (if (= (* real-width real-height 4)
                                   (length image-data))
                                (* real-width 4)
                              (ceiling (/ real-width 4)))))
      (image-to-graphic-stream d-stream dx dy real-width real-height image-data
			       line-byte
			       (if (= line-byte (* real-width 4))
                                       :color
                                     :mono)
			       0 0))
    ))

;;; $@%0%i%U%#%/%9%H%j!<%`$K%$%a!<%8$rIA2h$9$k!#(J
;;; draw-image distination-stream source-image position width height
;;;            &key image-position
;;; ARG.
;;;       distination-stream   = $@%G%#%9%F%#%M!<%7%g%s%0%i%U%#%/%9%H%j!<%`(J
;;;       source-image         = $@%=!<%9%$%a!<%8(J
;;;       position             = $@%$%a!<%8$rIA2h$9$k0LCV$N%$%s%9%?%s%9(J
;;;       width                = $@IA2h$9$kI}(J
;;;       height               = $@IA2h$9$k9b$5(J
;;;       image-position       = $@%=!<%9%0%i%U%#%/%9%H%j!<%`Fb$G$N0LCV$N(J
;;;                              $@%$%s%9%?%s%9(J
(defmethod draw-image ((d-stream graphic-stream) (s-image image)
		       (position position) (width integer) (height integer)
		       &key (image-position nil))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((x (if image-position
		(position-x image-position)
	      0))
	 (y (if image-position
		(position-y image-position)
	      0))
	 (real-width (if (< (+ width x) (image-width s-image))
			 width
		       (- (image-width s-image) x)))
	 (real-height (if (< (+ height y) (image-height s-image))
			  height
			(- (image-height s-image) y))))

    (image-to-graphic-stream d-stream (position-x position)
			     (position-y position) 
			     real-width real-height
			     (image-data s-image)
			     (line-bytes s-image)
			     (image-type s-image)
			     x y)
    d-stream))



;;; $@%0%i%U%#%/%9%H%j!<%`$K%$%a!<%8$rIA2h$9$k!#(JXY$@MQ(J
;;; draw-image-xy distination-stream source-image dx dy width height
;;;            &key image-x image-y
;;; ARG.
;;;       distination-stream   = $@%G%#%9%F%#%M!<%7%g%s%0%i%U%#%/%9%H%j!<%`(J
;;;       source-image         = $@%=!<%9%$%a!<%8(J
;;;       dx dy                = $@%$%a!<%8$rIA2h$9$k0LCV(J
;;;       width                = $@IA2h$9$kI}(J
;;;       height               = $@IA2h$9$k9b$5(J
;;;       image-x image-y      = $@%=!<%9%0%i%U%#%/%9%H%j!<%`Fb$G$N0LCV(J
(defmethod draw-image-xy ((d-stream graphic-stream) (s-image image)
			  (dx integer) (dy integer) (width integer) 
			  (height integer) &key (image-x 0) (image-y 0))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((real-width (if (< (+ width image-x) (image-width s-image))
                         width
                       (- (image-width s-image) image-x)))
         (real-height (if (< (+ height image-y) (image-height s-image))
                          height
                        (- (image-height s-image) image-y))))

    (image-to-graphic-stream d-stream dx dy
			     real-width real-height
                             (image-data s-image)
                             (line-bytes s-image)
                             (image-type s-image)
                             image-x image-y)

    d-stream))


;;; $@%9%H%j!<%`MQ$N%$%a!<%8$NIA2h(J
(defun draw-image-stream-xy-internal (d-stream s-stream dx dy width height
                                      sx sy)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (multiple-value-bind (image-data real-width real-height)
		       (get-image-xy-stream s-stream sx sy width height)
    (let ((line-byte (if (= (length image-data) (* real-width real-height))
			 (* real-width 4)
		       (ceiling (/ real-width 4)))))
      (image-to-graphic-stream d-stream dx dy real-width real-height image-data
			       line-byte
			       (if (= line-byte (* real-width 4))
				   :color
				 :mono)
			       0 0))
    ))

;;; $@%$%a!<%8%G!<%?$r%9%H%j!<%`$KAw$k(J
(defun image-to-graphic-stream (stream dx dy width height image-data line-byte
				       type sx sy)
  (declare 
   #-CMU
   (inline + * / incf logior ash)
   (special *black-color* *white-color* *transparent*)
	   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-translate-transform-xy ((x1 y1) stream dx dy)
    (let* ((tno (world-territory-no stream))
	   (drawing-region (drawing-region stream))
	   (count 0)
	   (x2 0)
	   (x3 0)
	   (x4 0)
	   (y2 0)
	   (y3 0)
	   (y4 0)
	   (yy (with-translate-coordinate-stream dy stream))
	   (real-y (if (= yy dy)
		       yy
		     (- yy height)))
	   (black-no (color-no *black-color*))
	   (white-no (color-no *white-color*)))
      (multiple-value-setq (x2 y2)
	   (get-new-position-xy (+ dx width) 
				(with-translate-coordinate-stream dy stream)
				stream))
	 (multiple-value-setq (x3 y3)
	   (get-new-position-xy dx (+ (with-translate-coordinate-stream dy 
						    stream)
				      height)
				stream))
	 (multiple-value-setq (x4 y4)
	   (get-new-position-xy (+ dx width)
				(+ (with-translate-coordinate-stream dy
						     stream)
				   height)
				stream))
				      
	 (with-temp-region-args ((draw-image-region) (work-region1 stream)
			     :left (min x1 x2 x3 x4)
			     :top (max y1 y2 y3 y4)
			     :right (max x1 x2 x3 x4)
			     :bottom (min y1 y2 y3 y4))
				
            ;;; $@%o!<%k%I%j!<%8%g%s$NJQ99(J
	   (setf (world-region stream) draw-image-region)

	    ;;; drawing-region$@$K:#$N%j!<%8%g%s$rDI2C(J
	   (set-drawing-region drawing-region draw-image-region))
	 
	 (case type
	   (:color 	 ;;; $@%+%i!<(J
	       ;;; $@IA2h(J
	    (setf y1 sy)
	    (do ((y real-y (incf y)))
		((= y (+ height real-y)))
	      (setf x1 sx)
	      (do ((x dx (incf x)))
		  ((= x (+ dx width)))

		(setf count (+ (* line-byte y1) (* x1 4)))

		(multiple-value-bind (xx yy) (get-new-position-xy x y stream)
		  (incf xx (world-x-start stream))
		  (incf yy (world-y-start stream))
		  (yy-protocol-20
		   tno xx yy *GCOPY* (logior (ash (elt image-data count) 24)
					   (ash (elt image-data (+ count 1)) 16)
					   (ash (elt image-data (+ count 2)) 8)
					   (elt image-data (+ count 3))))
		   )
		(incf x1))
	      (incf y1)))
	   (t ;;; $@%b%N%/%m(J
	    (setf y1 sy)
	    (do ((y real-y(incf y)))
		((= y (+ height real-y)))
	      (setf x1 sx)
	      (do ((x dx (incf x)))
		  ((= x (+ width dx)))
		
		(multiple-value-bind (xx yy) (get-new-position-xy x y stream)
		  (incf xx (world-x-start stream))
		  (incf yy (world-y-start stream))
		  (yy-protocol-20
		   tno xx yy *GSET*
		   (case (yy-image-bit image-data line-byte x1 y1)
		     (1 black-no)
		     (0 white-no)
		     (2 -1))))
		(incf x1))
	      (incf y1)))
	    )
	 stream)))


#|
;;; $@3($N=E$M9g$o$;(J $@!J%9%H%j!<%`!\%9%H%j!<%`!K(J XY$@MQ(J
;;; BITBLT-xy distination-stream distination-x distination-y
;;;        distination-width distination-height source-stream
;;;        &key (operation *GXCOPY*) (dupulicate nil) 
;;;        (source-x 0) (source-y 0)
;;; ARGS.
;;;           distination-stream            =  $@IA2h$9$k%0%i%U%#%C%/%9%9%H%j!<%`(J
;;;           distination-x distination-y   =  $@IA$-$@$70LCV(J
;;;           distination-width             =  $@I}(J
;;;           distination-height            =  $@9b$5(J
;;;           operarion                     =  $@%=!<%9$H%G%#%9%F%#%M!<%7%g%s$N(J
;;;                                            $@%*%Z%l!<%7%g%s(J
;;;           source-stream                 =  $@%=!<%9%0%i%U%#%C%/%9%9%H%j!<%`(J
;;;           dupulicate                    =  $@7+$jJV$7%U%i%0(J
;;;                                            T -> $@7+$jJV$7(J
;;;           source-x source-y             =  $@%=!<%9$N0LCV(J
(defmethod bitblt-xy ((d-stream graphic-stream) (dx integer) (dy integer)
		   (dw integer) (dh integer) 
		   (s-stream graphic-stream)
		   &key (operation *GXCOPY)
		   (dupulicate NIL) (souce-x 0) (source-y 0))
  (bitblt-xy-stream d-stream dx dy dw dh s-stream operation sx sy dupulicate))

;;; $@3($N=E$M9g$o$;(J $@!J%9%H%j!<%`!\%9%H%j!<%`!K(J $@%]%8%7%g%sMQ(J
;;; BITBLT-xy distination-stream distination-position
;;;        distination-width distination-height source-stream
;;;        &key (operation *GXCOPY*) (dupulicate nil) 
;;;        source-position
;;; ARGS.
;;;           distination-stream            =  $@IA2h$9$k%0%i%U%#%C%/%9%9%H%j!<%`(J
;;;           distination-position          =  $@IA$-$@$70LCV%$%s%9%?%s%9(J
;;;           distination-width             =  $@I}(J
;;;           distination-height            =  $@9b$5(J
;;;           operarion                     =  $@%=!<%9$H%G%#%9%F%#%M!<%7%g%s$N(J
;;;                                            $@%*%Z%l!<%7%g%s(J
;;;           source-stream                 =  $@%=!<%9%0%i%U%#%C%/%9%9%H%j!<%`(J
;;;           dupulicate                    =  $@7+$jJV$7%U%i%0(J
;;;                                            T -> $@7+$jJV$7(J
;;;           source-position               =  $@%=!<%9$N0LCV%$%s%9%?%s%9(J
(defmethod bitblt-xy ((d-stream graphic-stream) (d-posisiton position)
		   (dw integer) (dh integer) 
		   (s-stream graphic-stream)
		   &key (operation *GXCOPY)
		   (dupulicate NIL) (source-position nil))
  (if source-position
        (bitblt-xy-stream d-stream (position-x d-posisiton)
			  (position-y d-posisiton)
			  dw dh s-stream operation 
			  (position-x source-position)
			  (position-y source-position)
			  dupulicate)
    (bitblt-xy-stream d-stream (position-x d-posisiton)
		      (position-y d-posisiton)
		      dw dh s-stream operation
		      0 0 dupulicate)))


;;; $@3($N=E$M9g$o$;(J $@!J%9%H%j!<%`!\%9%H%j!<%`!K(J $@%j!<%8%g%s!"(JXY$@MQ(J
;;; BITBLT-xy distination-stream distination-region
;;;        source-stream
;;;        &key (operation *GXCOPY*) (dupulicate nil) 
;;;        (source-x 0) (source-y 0)
;;; ARGS.
;;;           distination-stream            =  $@IA2h$9$k%0%i%U%#%C%/%9%9%H%j!<%`(J
;;;           distination-region            =  $@IA$-$@$70LCV$HNN0h(J
;;;           operarion                     =  $@%=!<%9$H%G%#%9%F%#%M!<%7%g%s$N(J
;;;                                            $@%*%Z%l!<%7%g%s(J
;;;           source-stream                 =  $@%=!<%9%0%i%U%#%C%/%9%9%H%j!<%`(J
;;;           dupulicate                    =  $@7+$jJV$7%U%i%0(J
;;;                                            T -> $@7+$jJV$7(J
;;;           source-x source-y             =  $@%=!<%9$N0LCV(J
(defmethod bitblt-xy ((d-stream graphic-stream) (d-region region)
		   (s-stream graphic-stream)
		   &key (operation *GXCOPY)
		   (dupulicate NIL) (sx 0) (sy 0))
  (bitblt-xy-stream d-stream (region-left d-region)
		    (region-bottom d-region)
		    (region-width d-region)
		    (region-height d-region)
		    s-stream operation sx sy dupulicate))


  

)
|#

;;; End of file




