
(in-package "PT")

;;;***********************************************************
;;;
;;;  PATH class, methods, and support functions
;;;    A path is an ordered sequence of nodes, defining a total
;;;    ordering over some subset of the nodes in a given hyperdoc.
;;;    A path is defined by creating a new link type, named the same
;;;    as the path, and creating links of this type between each pair
;;;    of nodes in the sequence.  When the user selects a path, all
;;;    other link types are filtered out, so she is effectively
;;;    confined to the path until she either exits it or resets the
;;;    filters. 
;;;
;;;  Changed by PT:
;;;    Deleted the links field from path.
;;;    Heavily modified.
;;;    Each node can only appear once in each path.


(defdbclass path (node-set)
  ;; inherits nodes slot from node-set
  ((description :initform "A directed path through a set of nodes")
   (links :initform nil :accessor links
	  :documentation "Links connecting nodes on this path")
   (last-visited :initform nil :accessor last-visited
		 :documentation "Last node viewed last time on path"))
  (:documentation "Sequence of nodes from a given hyperdocument")
)

(defun make-path (&rest args)
  (apply #'make-instance 'path args))

(defmethod new-instance ((p path)
			 &key name (nodes (start-node #!*current-hyperdoc*))
			 &allow-other-keys
			 )
  (if (or (null name) (equal name ""))
      (setf (name p) (read-from-string
		      (get-string :prompt '("Paths must be named." 
					    "Please enter a name for the new path")))))
  (if (and (ch) (find name (paths (ch)) :key #'name))
      (setf (name p) (read-from-string
		      (get-string :prompt '("A path of that name already exists."
			    "Please enter a distinct name: ")))))
  ;; create link type and links between nodes:
  (unless (get-link-type name)
	  (make-link-type :name name :parent (get-link-type 'path) :filtered t))
;;  (if (> 1 (length nodes))
;;      (dotimes (i (1- (length nodes)))
;;	       (connect-nodes (elt nodes i) (elt nodes (1+ i)) p)))
  (add-obj p #!*current-hyperdoc*)
  )


(defun create-path ()
  (let ((new-path (call (find-po-named '("new-hip" "create-path" . "dialog")))))
    (when new-path
	  ;; add the nodes:
	  (edit new-path))))

(defmethod edit ((p path))
  (call (find-po-named '("new-hip" "edit-path" . "dialog")) :path p))

;;;
;;; Functions for manipulating paths and their nodes:
;;;

(defun connect-nodes (from to path)
  (format t "~A~%" (connected from to path))
  (let* ((link-type (get-link-type (name path)))
	 (from-marker-label (format nil "~a, node ~a" (name path) (name from)))
	 (to-marker-label (format nil "~a, node ~a" (name path) (name to)))
	 (from-marker (let ((pm (path-marker from path)))
			(if pm
			    pm
			  (make-link-marker :label from-marker-label
					    :parent from))))
	 (to-marker (let ((pm (path-marker to path)))
			(if pm
			    pm
			  (make-link-marker :label to-marker-label
					    :parent to))))
	 (the-link     (make-link :type link-type
				  :label (format nil "~a: ~a to ~a" (name path) (name from) (name to))
				  :source from-marker
				  :dest to-marker)))
#|
;;    (format t "Before : ~%the-link = ~A~%(links path) = ~A~%(link-markers from) = ~A~%(links-from from-marker)~A~%(link-markers to)~A~%(links-into from-marker)~A~%"
;;	    the-link
;;	    (mapcar #'label (links path))
;;	    (mapcar #'label (link-markers from))
;;	    (mapcar #'label (links-from from-marker))
;;	    (mapcar #'label (link-markers to))
;;	    (mapcar #'label (links-into to-marker)))
|#

    (pushnew the-link (links path))
    (pushnew from-marker (link-markers from))
    (pushnew the-link (links-from from-marker))
    (pushnew to-marker (link-markers to))
    (pushnew the-link (links-into to-marker))

#|
;;    (format t "After : ~%the-link = ~A~%(links path) = ~A~%(link-markers from) = ~A~%(links-from from-marker)~A~%(link-markers to)~A~%(links-into from-marker)~A~%"
;;	    the-link
;;	    (mapcar #'label (links path))
;;	    (mapcar #'label (link-markers from))
;;	    (mapcar #'label (links-from from-marker))
;;	    (mapcar #'label (link-markers to))
;;	    (mapcar #'label (links-into to-marker)))
|#
))

(defun connected (n1 n2 path)
  "returns t if there is a link between the path-markers for the path
   in n1 and n2"
  (path-connector n1 n2 path)) ;; might want to add a check for link
			       ;; from n2 to n1 as well

(defun path-connector (n1 n2 path)
  "returns the link, if any, connecting n1 and n2 on path"
  (let ((links (links-from (path-marker n1 path))))
    (find-if #'(lambda (l) (eq (dest l) (path-marker n2 path)))
	     links)))

(defun disconnect-nodes (from to path)
  (let ((connection (path-connector from to path)))
    (if (not connection)
	(error "Can't disconnect nodes ~a and ~a - not connected" 
	       (name from) (name to))
      (progn
	(setf (links path) (remove connection (links path)))
	(wipe-out connection)))))

(defun append-node (node path)
  (when (nodes path) (connect-nodes (end-node path) node path))
  (setf (nodes path) (append (nodes path) (list node))))

(defun insert-node (node after path)
  "inserts node after specified node on path"
  (let ((next (path-succ after path)))
    ;; adjust connections between nodes:
    (connect-nodes after node path)
    (connect-nodes node next path)
    (disconnect-nodes after next path)
    ;; insert node into the set of nodes on this path:
    (setf (nodes path) 
	  (append (subseq (nodes path) 0 (position after (nodes path)))
		  (list node)
		  (subseq (nodes path) (position next (nodes path)))))))

(defun remove-node (node path)
  (let ((before (path-prev node path))
	(after (path-succ node path)))
    (if before (disconnect-nodes before node path))
    (if after (disconnect-nodes node after path))
    (if (and before after)
	(connect-nodes before after path))
    (setf (nodes path) (remove node (nodes path)))))

;;;
;;; Useful path accessors:
;;;

(defmethod start-node ((p path))
  (car (nodes p)))

(defmethod end-node ((p path))
  (car (last (nodes p))))

(defmethod percent-complete ((p path))
  "returns percentage of path links that have been followed"
  (if (links p)
      (truncate
       (* 100
	  (/ (length (remove-if-not #'followed? (links p)))
	     (length (links p)))))
    0))

(defun path-marker (node path)
  "returns marker for path in node"
  (get-link-marker-in-node (format nil "~a, node ~a" (name path) (name node))
		      node))

(defun paths-through (node)
  (let ((paths (paths (ch))))  ;; ch == current hyperdoc
    (remove-if-not #'(lambda (p) (member node (nodes p))) paths)))

(defun goto-next (from-node &optional (path #!*current-path*))
  (unless (eq from-node (end-node path))
	  (open-node (path-succ from-node path))))

(defun my-goto-next (from-node &optional (path #!*current-path*))
  (unless (eq from-node (end-node path))
	  (let ((links (links-from (path-marker from-node path))))
	    (setf nnn from-node)
	    (setf ppp path)
	    (format t "goto-next : ~A ~A ~A ~A ~A~%"
		    path
		    from-node
		    (path-marker from-node path)
		    (links-from (path-marker from-node path))
		    links)
	    (cond ((null links)
		   (format t "Warning: No next links ~A ~A~%"
			   path from-node))
		  ((> (length links) 1)
		   (format t "Got more than 1 next links ~A ~A, ~A~%"
			   path from-node links)
		   (follow (car links)))
		  (t (follow (car links)))))))

;;	  (let ((next-link (or (find-if-not #'followed?
;;					(links-from (path-marker from-node path)))
;;			       (car (links-from (path-marker from-node path))))))
;;	    (when next-link (follow next-link)))))


(defun goto-prev (from-node &optional (path #!*current-path*))
  (unless (eq from-node (start-node path))
	  (open-node (path-prev from-node path))))

(defun my-goto-prev (from-node &optional (path #!*current-path*))
  (unless (eq from-node (start-node path))
	  (let ((links (links-into (path-marker from-node path))))
	    (format t "goto-prev : ~A ~A ~A ~A ~A~%"
		    path
		    from-node
		    (path-marker from-node path)
		    (links-into (path-marker from-node path))
		    links)
	    (cond ((null links)
		   (format t "Warning: No prev links ~A ~A~%"
			   path from-node))
		  ((> (length links) 1)
		   (format t "Got more than 1 prev links ~A ~A, ~A~%"
			   path from-node links)
		   (follow (car links)))
		  (t (follow (car links)))))))

;;	  (let ((prev-link (or (opener from-node)
;;			       (car (sort (copy-list (links-into
;;						      (path-marker from-node path)))
;;					  #'<
;;					  :key #'timestamp)))))
;;	    (when prev-link (follow prev-link)))))

;;;  These definitions of the predecessor and successor of nodes on a
;;;  path assume that a node appears only once on a given path.  We'll
;;;  have to change this if we decide to allow loops.

(defun path-prev (node path)
  (if (eq node (start-node path)) 
      nil
    (elt (nodes path) (1- (position node (nodes path))))))

(defun path-succ (node path)
  (if (eq node (end-node path))
      nil
    (elt (nodes path) (1+ (position node (nodes path))))))

;;;  
;;;  Interface support for paths:
;;;

(defun choose-path (path-set &optional (operation 'enter))
  "presents dialog box with list of paths in current hyperdoc; if node
   specified, only paths through it are shown"
  (call (find-po-named '("new-hip" "choose-path" . "dialog"))
	:paths path-set
	:op operation))

(defun choose-path-through-node (node)
  (let ((paths (paths-through node)))
    (if paths
	(choose-path paths 'enter)
      (announce-error "No paths through current node"))))

(defun enter-path (path)
  (set-link-type-selections (list (get-link-type (name path))))
  (setf #!*current-path* path)
  (setf #!*filter-state* (format nil "On Path: ~a" (name path)))
  (open-node (start-node path)))

(defun reset-path (path)
  (setf (last-visited path) nil)
  ;; mark all links as unread:
  (dolist (l (links path)) (unmark l)))  

(defun exit-path (&optional (path #!*current-path*))
  (setf (last-visited path) (cn))
  (setf #!*current-path* nil)
  (set-link-type-selections (list (get-link-type 'link))))


(defun get-all-path-info (path-set)
  (mapcar #'(lambda (p) (list (string (name p)) 
			      (description p) 
			      (if (last-visited p)
				  (string (name (last-visited p)))
				"None")
			      (format nil "~a %" (percent-complete p))))
	  path-set))

