
;;; We use the general-search package to create the list of 
;;; nodes given some roots and a child function

;;; This file contains a (hopefully impressive) set of functions 
;;; which will produce a PostScript file
;;; that, when printed, will be a reasonable representation 
;;; of an arbitrary digraph.  Enjoy...

;;; Note that most of this stuff comes from the paper 
;;; "A Browser for Directed Graphs,"
;;; by Rowe et. al. which appeared in Vol. 17, #1 of 
;;; Software - Practice and Experience

(in-package "PT")

(defun graph-node-printer (self stream depth)
  "The function which prints instances of the GRAPH-NODE structures."
  (if (and *print-level*
	   (> depth *print-level*)) ;; Don't print these infinite 
				    ;; lists of nested nodes
      (format stream "#")
      (format stream "#<GRAPH-NODE ~a Lvl: ~d, Bary: ~d (~d,~d)>"
	      (first (graph-node-text-list self))
	      (graph-node-level self)
	      (graph-node-bary-center self)
	      (graph-node-left-x self)
	      (graph-node-top-y self))))


(defun make-graph-list-from-roots (backend-object-list &key
				   (node-predicate #'identity)
				   (depth-limit nil)
				   (make-levels-from-top-p t) 
				   &aux (return-value nil))
  "This returns a list of graph-nodes representing the graph rooted by the given backend-objects
   after removing those nodes which fail NODE-PREDICATE.  Note that NODE-PREDICATE should be a predicate
   which takes a single back-end object and returns NIL if the object should not be included in the graph."

  ;; This entire function is a prime example of the "Better living through side-effects" approach to coding...
  
  ;; Walk through the graph in a a breadth-first fashion to create graph-node instances for
  ;; all of the descendants of the given roots.  At this point they just have text labels.
  (clrhash *backend-graph-node-map*)

  (breadth-first-search backend-object-list
			:depth-limit depth-limit
			:traverse-fn #'(lambda (node)
					 (remove-if-not node-predicate
							(funcall *backend-children-function* node)))
			:process-fn #'make-graph-node-from-backend-object)

  ;; Collect everything into a list so we don't have to keep spinning through the table
  (maphash #'(lambda (ignore graph-node)
	       (declare (ignore ignore))
	       (push graph-node return-value))
	   *backend-graph-node-map*)

  ;; Add the parent/children links to everybody
  (dolist (next-node return-value)

    ;; Collect the graph-nodes representing all of the subs of the kn-object for this graph-node
    (setf (graph-node-children next-node)
	  (remove-if #'null
		     (mapcar #'(lambda (obj)
				 (gethash obj *backend-graph-node-map*))
			     (funcall *backend-children-function* (graph-node-backend-object next-node)))))

    (dolist (next-child (graph-node-children next-node))
      ;; Add yourself to the list of parents for each of your children
      ;; (I'm not sure if the pushnew is necessary, but I'm a cautious kind of guy...)
      (pushnew next-node (graph-node-parents next-child))))

  ;; Determine at which level each graph node will appear on the finished diagram
  ;; The transitive-closure method works for cyclic graphs; the other method only works for acyclic graphs,
  ;; but orders things in a more top-down fashion
  (if make-levels-from-top-p 
      (dolist (next-node return-value return-value)
	(setf (graph-node-level next-node)
	      (depth-of-graph-node next-node)))
      (compute-levels-using-transitive-closure return-value))
  )

(defun compute-levels-using-transitive-closure (graph-node-list)
  ;; This function computes the levels for a given flat graph-node list
  ;; and returns a nested list of graph nodes which have their LEVEL
  ;; slot properly filled.  The levels are assigned by the method of
  ;; iteratively computing the transitive closure of the graph and
  ;; removing nodes which have the same set of predecessors and successors.
  ;; See the paper "A Browser for Directed Graphs" by Carl Meyer -- U.C.
  ;; Berkeley, Dept of EE/CS.  This is NOT the paper by Rowe et. al. of
  ;; the same name mentioned at the top of this file.
  (let ((successors-table (make-hash-table))
	(predecessors-table (make-hash-table))
	(return-value nil))
       ;; This has written to avoid a nasty #'lambda in the code
       ;; so I don't feel guilty about liberal use of special variables
       (declare (special successors-table predecessors-table)) 
       
       ;; Working on the full set of graph nodes, compute the transitive
       ;; closure of each node
       (dolist (next-graph-node graph-node-list)
	       (dolist (next-child
			(transitive-closure-of-node next-graph-node
						    (list next-graph-node)))
		       ;; Store the successors of this node
		       (push next-child
			     (gethash next-graph-node successors-table))
		       
		       ;; Mark this node as a predecessor of the child
		       (push next-graph-node
			     (gethash next-child predecessors-table))))
       
       ;; Now go through the graph and identify the bottom nodes.
       ;; They are moved to the return value and removed from both
       ;; hashtables.  This has the effect of completely removing
       ;; them from consideration.  This process is repeated until
       ;; all nodes have been removed and assigned a level.
       (do ((bottom-nodes nil nil))
	   ;; When all nodes have been processed, set the LEVEL slots for
	   ;; each graph-node and return the nested list
	   ((zerop (hash-table-count successors-table))
	    (assign-levels return-value))
	   (declare (special bottom-nodes))
	   
	   (format t "There are ~d nodes still under consideration~%"
		   (hash-table-count successors-table))
	   
	   ;; Identify the bottom nodes from those still under
	   ;; consideration and store them in BOTTOM-NODES
	   (maphash #'bottom-node-p successors-table)
	   
	   ;; Store them away on the return value list
	   (push bottom-nodes return-value)
	   
	   ;; Remove them from consideration for the next pass
	   (dolist (next-bottom-node bottom-nodes)
		   (remhash next-bottom-node successors-table)
		   (dolist (next-parent (gethash next-bottom-node
						 predecessors-table))
			   (let ((trimmed-successors
				  (delete next-bottom-node
					  (gethash next-parent
						   successors-table))))
				(if trimmed-successors
				    (setf (gethash next-parent successors-table)
					  trimmed-successors)
				    (remhash next-parent successors-table))))
		   (remhash next-bottom-node predecessors-table)))))
  
(defun transitive-closure-of-node (node visited-nodes)
  "This returns the list of nodes which are reachable from the given node
  by using the *BACKEND-CHILDREN-FUNCTION*.
  Note that this assumes that each node is reachable from itself, which is
  used for later processing.  The list passed in as VISITED-NODES is used
  to avoid infinite loops, and should start off as the list containing the
  root."
  (let* ((direct-children
	  (set-difference
	   (mapcar #'(lambda (backend-object)
			     (gethash backend-object *backend-graph-node-map*))
		   ;; Here we compute the backend-objects which descend from
		   ;; the backend object for this graph node
		   (funcall *backend-children-function*
			    (graph-node-backend-object node)))
	   visited-nodes))
	 (closure-list
	  (mapcar #'(lambda (child-node)
			    (transitive-closure-of-node
			     child-node
			     (append direct-children visited-nodes)))
		  ;; Convert the backend objects into the graph-nodes
		  ;; which represent them
		  direct-children)))
	(if closure-list
	    (cons node (flatten-list closure-list))
	    (list node))))

(defun bottom-node-p (node ignore)
  (declare (ignore ignore))
  "A function designed to be called by COMPUTE-LEVELS-USING-TRANSITIVE-CLOSURE which
   returns T if the node is on the bottom level of the graph and NIL otherwise.
   NOTE: this uses special vars which must be created by the caller."

  ;; This has mostly been written to avoid a nasty #'lambda in the code for COMPUTE-LEVELS-USING-TRANSITIVE-CLOSURE,
  ;; so I don't feel guilty about liberal use of special variables
  (declare (special bottom-nodes         ; List containing all nodes on bottom of graph
		    successors-table     ; Hashtable with nodes as keys and their successors as values
		    predecessors-table)) ; Hashtable with nodes as keys and their predecessors as values

  ;; The goal is to find nodes such that all of their successors are also their predecessors
  (when (set-equal (gethash node successors-table)
		   (intersection (gethash node successors-table)
				 (gethash node predecessors-table)))
    (push node bottom-nodes)))
  
;;;  I had to change the name of this from flatten because of a
;;;  conflict in picasso - if I ever get time, I'll do this right by
;;;  putting the graphing stuff a separate package.
(defun flatten-list (nested-list)
  "This returns a list identical to the nested list in the value and order of its atoms, but without
   any of the lower-level cons cells.  It really should be part of Common Lisp..."
  (cond ((atom nested-list)
	 nested-list)
	((atom (first nested-list))
	 (cons (first nested-list) (flatten-list (rest nested-list))))
	(t (append (flatten-list (first nested-list))
		   (flatten-list (rest nested-list))))))

(defun set-equal (set-1 set-2 &key (test #'eql) (key #'identity))
  "Returns T if the two sets are equal under the test and nil otherwise"
  (and (= (length (intersection set-1 set-2 :test test :key key))
	  (length (union set-1 set-2 :test test :key key)))))
	
(defun cycles-in-level (level)
  "Given a set of nodes on the same level this returns the subset (possibly nil) containing
   the nodes involved in cycles."
  (remove-duplicates
    (remove-if-not #'(lambda (node)
		       (intersection (graph-node-children node)
				     level))
		   level)))

(defun remove-cycles (graph-list &aux (level-counter 0))
  "This function removes cycles by combining all nodes in a cycle into a single proxy node
   which stands for all of them."
  
  (dolist (next-level graph-list graph-list)
    (let ((nodes-in-cycles (cycles-in-level next-level)))
      (when nodes-in-cycles
	;; Delete all of the nodes in cycles since we will replace them with proxies
	(setf (elt graph-list level-counter)
	      (set-difference next-level nodes-in-cycles))
	(do* ((rem-cycle-nodes nodes-in-cycles (set-difference rem-cycle-nodes nodes-in-proxy))
	      ;; Look for a node with no parents in the current level
	      (next-proxy-head (first rem-cycle-nodes) (first rem-cycle-nodes))
	      (nodes-in-proxy (when next-proxy-head
				(intersection (transitive-closure-of-node next-proxy-head (list next-proxy-head))
					      rem-cycle-nodes))
			      (when next-proxy-head
				(intersection (transitive-closure-of-node next-proxy-head (list next-proxy-head))
					      rem-cycle-nodes)))
	      (next-proxy (make-graph-node :backend-object nil
					   :level level-counter
					   :parents (set-difference (flatten-list (mapcar #'graph-node-parents nodes-in-proxy))
								    nodes-in-proxy)
					   :children (set-difference (flatten-list (mapcar #'graph-node-children nodes-in-proxy))
								    nodes-in-proxy)
					   :text-list (mapcar #'(lambda (node)
								  (first (graph-node-text-list node)))
							      nodes-in-proxy))
			  (when nodes-in-proxy
			    (make-graph-node :backend-object nil
					     :level level-counter
					     :parents (set-difference (flatten-list (mapcar #'graph-node-parents nodes-in-proxy))
								      nodes-in-proxy)
					     :children (set-difference (flatten-list (mapcar #'graph-node-children nodes-in-proxy))
								       nodes-in-proxy)
					     :text-list (mapcar #'(lambda (node)
								    (first (graph-node-text-list node)))
								nodes-in-proxy)))))
	     ((null next-proxy-head))
	  ;; Tell the user about this excitement
	  (when *graph-loudly*
	    (format t "Colapsing the cycle ~a~%" (graph-node-text-list next-proxy)))
	  
	  ;; We have to change the pointers so that nodes on different levels which pointed to one of the
	  ;; nodes in the current proxy now point to the proxy instead
	  (dolist (next-parent (graph-node-parents next-proxy))
	    (setf (graph-node-children next-parent)
		  (cons next-proxy (set-difference (graph-node-children next-parent)
						   nodes-in-proxy))))

	  (dolist (next-child (graph-node-children next-proxy))
	    (setf (graph-node-parents next-child)
		  (cons next-proxy (set-difference (graph-node-parents next-child)
						   nodes-in-proxy))))

          ;; Finally, splice this proxy node into the list for the level
          (setf (elt graph-list level-counter) (cons next-proxy (elt graph-list level-counter))))))

    (incf level-counter)))
	
(defun assign-levels (graph-nodes-by-levels)
  "A function designed to be called by COMPUTE-LEVELS-USING-TRANSITIVE-CLOSURE with a nested
   list of graph nodes.  The nodes in the first nested list are at level 0 in the graph, those
   in the next list are at level 1, etc.  AFTER removing cycles by creating <proxy nodes>, this simply fills
   the LEVEL slot in the graph-node based on its position in the list.  It returns the modified list."

  ;; First prune out the cycles in the graph by introducing proxy levels
  (setf graph-nodes-by-levels (remove-cycles graph-nodes-by-levels))

  (let ((level-counter 0))
    (dolist (next-level graph-nodes-by-levels)
      (dolist (next-node next-level)
	(setf (graph-node-level next-node) level-counter))
      (incf level-counter)))

  ;; Remember to return the list after modification
  (flatten-list graph-nodes-by-levels))

;;;************************************************************************************
;;; King Kong actions
;;;************************************************************************************

(defun text-describing-intention (intention)
  "This returns some text which describes this object."
  (list*
    (if (intention-subroot? intention)
	(format nil "#~a# (~{~a ~})"
	  (intention-name intention)
	  (mapcar #'intention-role-name (intention-roles intention)))
	(format nil "~a (~{~a ~})"
		(intention-name intention)
		(mapcar #'intention-role-name (intention-roles intention))))
    (format nil "~a" (intention-speech-act intention))

    ;; The LIST* will prune this out if a terse node is desired
    (when *verbose-node-text*
      (format nil "~a" (intention-documentation intention)))))


(defun get-intention-children (intention)
  "Returns a list of the intentions which are children of the given intention."
  (intention-supporters intention))

;;;************************************************************************************
;;; AND NOW... From the far edge of reality, we bring you the object hierarchy used
;;; by that blot on the face of ontological reason, MACPlan
;;;************************************************************************************

#|(defun text-describing-flavor (flavor)
  "This returns some text which describes this flavor."
  (list*
    (format nil "~a"
	  (flavor::flavor-name flavor))

    ;; If the user wants a full report for flavors, we are going to list the methods associate with each one
    (when *verbose-node-text*
      (mapcar #'(lambda (meth-name-list)
		  (format nil "    ~A" (first meth-name-list)))
	      (flavor::flavor-method-table flavor)))))


(defun get-flavor-children (flavor)
  "Returns a list of the intentions which are children of the given intention."
  ;; Note that the flavor defstruct which tells ALL about this flavor is stored as the sys:flavor
  ;; property on the symbol which names the flavor in question
  (mapcar #'(lambda (flavor-name)
	      (get flavor-name 'sys:flavor))
	  (remove (flavor::flavor-name flavor)
		  (flavor::flavor-local-dependents flavor))))
|#
;;;************************************************************************************
;;; We now return you to your regularly scheduled program
;;;************************************************************************************

(defun depth-of-graph-node (graph-node &aux super-list)
  "This computes the depth of given graph-node, where the roots are at level 0."
    (if (setf super-list (graph-node-parents graph-node))
	(1+ (apply #'max (mapcar #'depth-of-graph-node super-list)))
	0))

(defun make-graph-node-from-backend-object (backend-object &aux return-value)
  "Given an backend object this returns a graph-node which represents it."
  (if (setf return-value (gethash backend-object *backend-graph-node-map*))
      return-value
      (progn
	(setf return-value (make-graph-node
			     :text-list (funcall *backend-text-function* backend-object)
			     :backend-object backend-object))
	(setf (gethash backend-object *backend-graph-node-map*) return-value)
	return-value)))

;;; NOTE: I'm not sure if I should really be checking for membership in the parent/children
;;; lists at this point - the calling function may already know that, but if I don't this
;;; isn't anything more interesting than a subtraction macro.
(defun long-edge-p (high-node low-node)
  "Tests the given nodes for the presence of a long-edge separating them.  Returns NIL
   if no such edge exists, or the length of the edge if there is one."
  (let ((distance (- (graph-node-level low-node) (graph-node-level high-node))))
    (if (and (member low-node (graph-node-children high-node))
	     (member high-node (graph-node-parents low-node))
	     (> distance 1))
	distance)))
	
(defun make-nested-list-from-sorted-list (sorted-list key &aux (return-list nil))
  "This takes a list of the form (1 2 2 2 4 4 4 4 6), where the numbers are the result of
   applying the KEY to each item in the sorted list, and returns a list of the form
   ((1) (2 2 2) (4 4 4 4) (6)).  A list of this form is used by the graph-manipulation
   routines, although there should never be a gap in the numbering sequence."

  ;; The basic story here is to cut off the portion at the head of the list with a uniform
  ;; key value, push it as a list onto the return list, and repeat the process for the
  ;; remaining items on the sorted list.  I reverse the list to undo the effect of pushing things
  ;; onto the result list. (Well, if you can keep straight which list is which you probably
  ;; don't need the comment in the first place...)
  
  (if *graph-loudly* (format t "Converting ~d graph nodes to the GRAPH-LIST data structure~%" (length sorted-list)))
  
  (do* ((rev-list (reverse sorted-list) (subseq rev-list next-key-position))
	(current-key (funcall key (first rev-list)) (funcall key (first rev-list)))
	(next-key-position (position current-key rev-list :test-not #'= :key key)
			   (position current-key rev-list :test-not #'= :key key)))
       ((null next-key-position) (push rev-list return-list))
    (push (subseq rev-list 0 next-key-position) return-list)))

(defun generate-replacement-edge (high-node low-node)
  "This takes two nodes which are connected by an edge which spans more than 1 level and
   returns a list of dummy nodes which are each 1 level apart.  The first dummy in the
   return list will need to be substituted as a child of the high node, and the last dummy
   will have to be substituted as a parent of the low node BY THE CALLING FUNCTION."

  (let ((dummy-count (1- (- (graph-node-level low-node) (graph-node-level high-node)))) ; How many dummy nodes are needed?
	(return-list nil))
    
    ;; Make sure the calling function isn't dazed and confused...
    (assert (long-edge-p high-node low-node) ()
	    "The graph-nodes ~s and ~%~s are not connected by a long edge"
	    high-node low-node)

    ;; Create as many dummy nodes as needed to span the gap
    (do ((level (1+ (graph-node-level high-node)) (1+ level))
	 (remaining dummy-count (1- remaining)))
	((zerop remaining))

      ;; While the names aren't printed, and the size will probably be ignored by the eventual drawing
      ;; routine, giving them distinct identies and correct sizes makes the intermediate debugging easier
      (push (make-graph-node :text-list (list (string (gensym "D"))) :dummy-p t :x-size 0 :y-size 0 :level level)
	    return-list))

    ;; The returned value is supposed to have the highest node at the front of the list
    (setf return-list (reverse return-list))
    (push (first return-list) *heads-of-replacement-edges*)
    
    ;; Link the dummy nodes together internally and return the list of dummys to the calling routine
    (do* ((dummy-list return-list (cdr dummy-list))
	  (first-dummy (first dummy-list) (first dummy-list))
	  (second-dummy (second dummy-list) (second dummy-list)))
	 ((null second-dummy) return-list)

      
      (setf (graph-node-children first-dummy) (list second-dummy)
	    (graph-node-parents second-dummy) (list first-dummy)))))
    
(defun remove-long-edges-from-graph-list (graph-list &aux (return-value graph-list))
  "This function takes the outp<ut from MAKE-GRAPH-LIST-FROM-ROOTS and removes all long edges
   by inserting dummy nodes that are only 1 level apart.  This returns an augmented list
   of graph nodes that have been sorted by depth."

  (setf *heads-of-replacement-edges* nil)
  
  (if *graph-loudly* (format t "~&Removing long edges:"))
  
  ;; Note that return value starts off with the original list, so we just need to add any
  ;; newly created dummy nodes to it before returning.
  (dolist (next-node graph-list)
    (dolist (next-child (graph-node-children next-node))
      (if (long-edge-p next-node next-child)

	  ;; New feature added to support the transitive-closure method of computing levels
	  ;; That approach puts ALL leaves on the bottom level, even if there is no compelling reason to
	  ;; do so.  We need to move those up under their lowest-level parents
	  (if (and (leaf-node-p next-child)
		   (every #'(lambda (next-parent)
			      (long-edge-p next-parent next-child))
			  (graph-node-parents next-child)))
	      (block then
		(setf (graph-node-level next-child)
		      (1+ (apply #'max (mapcar #'graph-node-level (graph-node-parents next-child)))))
		(when *graph-loudly*
		  (format t "Decided to move ~a closer to its parents~%"
			  (first (graph-node-text-list next-child)))))

	      ;;The rest of this is a big ELSE for the normal long-edge situations
	      ;; Get the list of dummy nodes which will span the multiple levels
	      (let* ((replacement-edge (generate-replacement-edge next-node next-child))
		     (dummy-head (first replacement-edge))
		     (dummy-tail (first (last replacement-edge))))
		
		(if *graph-loudly* (princ #\.))
	    
		;; POINTER ALERT: the following looks suspiciously like C code...
		;; Change the high node to point to the front of the replacement chain...
		(setf (graph-node-children next-node) (substitute dummy-head
								  next-child
								  (graph-node-children next-node))

		      ;; ... link the chain into the parent ...
		      (graph-node-parents dummy-head) (list next-node)
		  
		      ;; ... make the low node to point to the end of the replacement chain ...
		      (graph-node-parents next-child) (substitute dummy-tail
								  next-node
								  (graph-node-parents next-child))
		      ;; ... and link the chain to the child.
		      (graph-node-children dummy-tail) (list next-child))

		;; Add these new dummy nodes to the graph list itself.
		(dolist (next-dummy replacement-edge)
		  (push next-dummy return-value)))))))

  (if *graph-loudly* (terpri))

  ;; Now sort this by node level so that MAKE-NESTED-LIST-FROM-SORTED-LIST can do the right thing with it
  (stable-sort return-value #'< :key #'graph-node-level))

(defun up-bary-center (node)
  "This returns the NORMALIZED up-bary-center of a node, which is defined to be the average of the
   existing bary-centers of each node above this one in the graph, expressed as a percentage of the
   position divided by the size of the level. If there are no nodes
   above this one return the current value. A second T/NIL value is returned to indicate whether or
   not this node had any parents in the given direction."
  (let ((parent-bary-centers (mapcar #'graph-node-bary-center (graph-node-parents node))))

    (values
      ;; Compute the average of the existing bary-centers.  If there aren't any, just return the current value.
      (if parent-bary-centers
	  (/ (apply #'+ parent-bary-centers) (length parent-bary-centers))
	  (graph-node-bary-center node))
      parent-bary-centers)))

(defun down-bary-center (node)
  "This returns the NORMALIZED down-bary-center of a node, which is defined to be the average of the
   existing bary-centers of each node below this one in the graph, expressed as a percentage of the
   position divided by the size of the level. If there are no nodes
   below this one return the current value. A second T/NIL value is returned to indicate whether or
   not this node had any parents in the given direction."
  (let ((children-bary-centers (mapcar #'graph-node-bary-center (graph-node-children node))))

    (values
      ;; Compute the average of the existing bary-centers
      (if children-bary-centers
	  (/ (apply #'+ children-bary-centers) (length children-bary-centers))
	  (graph-node-bary-center node))
      children-bary-centers)))

(defun average-bary-center (node)
  "This returns the NORMALIZED up- and down-bary-centers for the given node, used to compute a position
   during later passes through the graph."
  (let ((up-value-list (multiple-value-list (up-bary-center node)))
	(down-value-list (multiple-value-list (down-bary-center node))))

    ;; If both returned values are based on positions relative to parents, average them
    (cond ((and (second up-value-list)
		(second down-value-list))
	   (/ (+ (up-bary-center node)
		 (down-bary-center node ))
	      2))

	  ;; If only one set of parents exists, just use those values and ignore current value for other direction
	  ((second up-value-list) (first up-value-list))
	  ((second down-value-list) (first down-value-list))

	  ;; If this just floats in outer space, leave it where it is...
	  (t (graph-node-bary-center node)))))

(defun make-bary-center-pass (graph-list down-p pass-type)
  "This makes a single pass through the graph, recalculating the bary-center value for each node,
   where the DOWN-P flag is T iff the pass should proceed down the graph, and nil if it should go up.
   NOTE: the first level processed must already have some bary-value, so that subsequent values can
   be computed from them.
   The PASS-TYPE flag is one of 'UP, 'DOWN, 'AVERAGE, indicating the type of bary-center value to be used.
   Note that this function ONLY COMPUTES the new values, and does not re-order the list.  I don't
   see any need to actually reorder the list after every pass - I think you can wait until all
   passes have been made and then do one reshuffle. It returns the internally modified graph-list."

  ;; If we're supposed to go through the list bottom-up, just reverse it and proceed normally
  (let ((mod-graph-list (if down-p
			    graph-list
			    (reverse graph-list)))
	;; Get the correct function for determining the bary-value for this pass
	(bary-value (ecase pass-type
		      (up #'up-bary-center)
		      (down #'down-bary-center)
		      (average #'average-bary-center))))

    ;; Now just loop through everything and compute the new values.
    (do* ((rest-of-graph mod-graph-list (cdr rest-of-graph))
	  (parent-level nil this-level)
	  (this-level (first rest-of-graph) (first rest-of-graph))
	  (level-num 0 (1+ level-num))
	  (child-level (second rest-of-graph) (second rest-of-graph)))
	 ((null this-level) mod-graph-list)
	(dolist (next-node this-level)
	  (setf (graph-node-bary-center next-node)
		(funcall bary-value next-node)))
	
	;; Sort each level and assign new normalized values so that they will propogate to the next level
	;; Note that I have to splice the result into MOD-GRAPH-LIST since THIS-LEVEL is just a local var.
	(setf (nth level-num mod-graph-list)
	      (stable-sort this-level #'< :key #'graph-node-bary-center))

	;; Replace the newly computed bary-center values with counting numbers to keep the value from getting
	;; strange when we make lots of passes.  NOTE: These values have been normalized so that they are really a
	;; percentage of the relative position within the level.  This way nodes near the end of long levels don't have
	;; much larger values than the nodes near the end of short nodes, causing all sorts of havoc.

	(dotimes (ordinal-number (length this-level))
	  ;; The call to NTH just gets the node at position ORDINAL-NUMBER in the list.
	  (setf (graph-node-bary-center (nth ordinal-number this-level))
		(/ ordinal-number (float (length this-level))))))))


(defun sort-levels-by-bary-center (graph-list num-passes)
  "This runs NUM-PASSES over the graph-list, calculating new bary-centers for each
   one as indicated in the paper.  The first pass goes downwards using up-bary-centers,
   the second one goes upwards using down-bary-centers, and all subsequent passes alternate
   directions using average-bary-centers."

  (if *graph-loudly* (format t "Ordering nodes within levels:"))
  
  ;; Prime the pump by assigning normalized sequential values as bary-centers on EACH row (needed for the new level algorithm)
  (dolist (next-level graph-list)
    (do* ((counter 1.0 (1+ counter))
	  (node-list next-level (cdr node-list))
	  (level-length (length node-list))
	  (node (first node-list) (first node-list)))
	 ((null node-list))
      (setf (graph-node-bary-center node) (/ counter level-length))))
  
  ;; Now conduct the passes, modifying the parameters as indicated above
  (let ((pass-type 'up)
	(down-p t))
    (dotimes (pass-num num-passes)
      (make-bary-center-pass graph-list down-p pass-type)

      ;; New heuristic #2: if a parent has both leaf and non-leaf children, assign the  bary-center
      ;; value of one of the non-leaf nodes to all leaves
      (dolist (next-level graph-list)
	(dolist (next-node next-level)
	  (let* ((children (graph-node-children next-node))
		 (non-leaf (find nil children :test 
				 #'(lambda (ignore node) 
					   (declare (ignore ignore))
					   (graph-node-children node)))))
	    (when (and (some #'graph-node-children children)
		       (some #'leaf-node-p children))

	      ; Don't move those nodes which are not leaves, OR leaves with multiple parents
	      (dolist (next-leaf (remove-if #'(lambda (node) (cdr (graph-node-parents node)))
					    (remove-if-not #'leaf-node-p children)))
		(setf (graph-node-bary-center next-leaf)
		      (graph-node-bary-center non-leaf)))))))
      
      (if *graph-loudly* (princ #\.))

      ;; The direction switches after each pass, and the pass-type is 'average for all but the 2nd pass.
      (setf down-p (not down-p)
	    pass-type (if (evenp pass-num)
			  'down
			  'up))))

  ;; A NEW heuristic to try to straighten out the long edges: Starting at each of the dummy nodes in
  ;; *heads-of-replacement-edges*, set their children to have the same bary-center as the head.
  (dolist (next-head *heads-of-replacement-edges*)
    (do* ((current-head next-head current-child)

	  ;; Note that the child-list of a dummy is ALWAYS 1 long
	  (current-child (first (graph-node-children current-head))
			 (first (graph-node-children current-head))))
	 ((not (graph-node-dummy-p current-child)));(null current-child))
      (setf (graph-node-bary-center current-child)
	    (graph-node-bary-center current-head))))

  ;; After the clever heuristics above, we need to resort the list to take the modified values into account
  ;; Sort the list in place after each adjustment
  (dotimes (level-num (length graph-list))
    (setf (nth level-num graph-list)
	  (stable-sort (nth level-num graph-list) #'< :key #'graph-node-bary-center)))

  ;; Undo the normalization so that the bary-center value of a given node is the index into where
  ;; it will be displayed on the chart.  Several things, including the SPLIT-NEAR-CHILDREN-P feature
  ;; depend on these values.
  (dolist (next-level graph-list)
	(dotimes (ordinal-number (length next-level))
	  (setf (graph-node-bary-center (nth ordinal-number next-level))
		ordinal-number)))
  (if *graph-loudly* (terpri))
  graph-list)

(defun leaf-node-p (node)
  "Predicate which returns T if the node has no children."
  (not (graph-node-children node)))

(defun determine-size-of-nodes (graph-list)
  "This takes a graph and adds values to the X-SIZE and Y-SIZE slots of each node.  I'm currently using
   the PostScript Courier 12pt. font, with 16 pt. high rectangles and 5 points of space before and after
   the text in the rectangle."

  (if *graph-loudly* (format t "Assigning output dimensions to graph nodes~%"))
  
  (dolist (next-level graph-list graph-list)
    (dolist (next-node next-level)

      ;; Set the height of each box, or leave as 0 for dummy nodes since they size to the rest of the level
      (setf (graph-node-y-size next-node)
	    (if (graph-node-dummy-p next-node)
		0
		;; Each line of text is 16 points high
		(* 16 (length (graph-node-text-list next-node)))))

      ;; Now figure out how wide the thing is, with 7.2 pt wide chars and 10 pts total padding,
      ;; using the length of the longest line to be printed.
      (setf (graph-node-x-size next-node)
	    (if (graph-node-dummy-p next-node)
		0
		(+ 10 (* (apply #'max 
				(mapcar #'length
					(graph-node-text-list next-node)))
			 7.2)))))))

(defun compute-level-sizes (graph-list min-node-gap orientation)
  "This takes a graph and a minimum gap between nodes on a given level and returns a list
   of numbers, where each number is the size in points of the corresponding level in the graph.
   The orientation flag is :horizontal for graphs with vertical levels that go to the right, and
   :vertical for graphs with horizontal levels that go down the page."

  (let ((return-value nil)
	(size-accessor (ccase orientation
			 (:horizontal #'graph-node-y-size)
			 (:vertical #'graph-node-x-size))))

    (dolist (next-level graph-list (reverse return-value))
      (push (+ (* min-node-gap (1+ (length next-level)))    ; Compute the total space needed for the gaps...
	       (apply #'+ (mapcar size-accessor next-level))) ; and the total size required for all nodes on this level
	    return-value))))

(defun place-nodes-within-level (graph-list min-node-gap orientation)
  "This takes a graph, the minimum gap between nodes on a level, and the orientation of the graph
   and sets either the LEFT-X slot (for vertical graphs) or the TOP-Y slot (for horizontal graphs).
   It returns the graph with modifications."


  (if *graph-loudly* (format t "Assigning output coordinates to nodes in levels~%"))
  
  (let ((max-level-size (apply #'max (compute-level-sizes graph-list min-node-gap orientation))))

    ;; Store away the largest level size so that we can pan over the graph automatically later
    (ccase orientation
	  (:horizontal (setf *graph-y-size* max-level-size))
	  (:vertical (setf *graph-x-size* max-level-size)))
    
    (dolist (next-level graph-list)

      ;; The approach here is to determine how much white space exists for the entire level, and then parcel
      ;; that out evenly to each node. We (conceptually) place a block of whitespace as computed by
      ;; ACTUAL-NODE-PADDING followed by the node itself, repeating for each node in the level
      (let* ((size-accessor (ccase orientation
			 (:horizontal #'graph-node-y-size)
			 (:vertical #'graph-node-x-size)))
	     (this-level-size (apply #'+ (mapcar size-accessor next-level)))
	     (actual-node-padding (/ (- max-level-size this-level-size)
				     (float (1+ (length next-level))))))

	;; Make sure that if we change the algorithm and screw up the calculations we don't skip over the problem
	(assert (not (< (ceiling actual-node-padding) min-node-gap)) ()
		"Internal error: ACTUAL-NODE-PADDING smaller than user-supplied value")
	
	(do* ((rest-of-level next-level (cdr rest-of-level))

	      ;; NOTE: This is slightly tricky.  Updating this value uses the PREVIOUS node, since NEXT-NODE isn't
	      ;; recomputed until after next-position.  That way we can determine how much space the last node needed
	      ;; to draw itself, and then add the node gap.  I start at 72 pts (1 inch) in to give a border)
	      (next-position 72 (+ next-position actual-node-padding (funcall size-accessor next-node)))
	      (next-node (first rest-of-level) (first rest-of-level)))
	     ((null next-node))

	  ;; Now for the anti-climax - just set the position slot for this node as computed by the stuff above
	  (ccase orientation
	    (:horizontal (setf (graph-node-top-y next-node) (+ next-position
							       actual-node-padding
							       (graph-node-y-size next-node))))
	    (:vertical (setf (graph-node-left-x next-node) (+ next-position
							      actual-node-padding))))
	  ))))

  graph-list)

(defun place-levels-in-graph (graph-list min-level-gap orientation)
  "This takes a graph, a minimum separation distance between levels, and an orientation, and
   sets the appropriate slot on each nod so that it is properly placed.  It returns the modified graph."

  (if *graph-loudly* (format t "Assigning output coordinates levels in graph~%"))
  
  ;; At some point we might want to increase the spacing between levels so that we don't get a graph which is
  ;; very flat, but that is a tweak for the future.
  ;; NOTE: for :vertical graphs, start by placing the bottom row, and moving up so that position is always
  ;; the top y position of the next level.
  (do* ((size-accessor (ccase orientation
			 (:vertical #'graph-node-y-size)
			 (:horizontal #'graph-node-x-size)))
	(rest-of-graph (if (eq orientation :vertical)
			   (reverse graph-list)
			   graph-list)
		       (cdr rest-of-graph))
	(next-level (first rest-of-graph) (first rest-of-graph))

	;; As above, I start at 72 to give a 1 inch border to the page.
	(position (+ 72 (if (eq orientation :vertical)
			    (+ (apply #'max (mapcar size-accessor next-level)) min-level-gap)
			    0))

		  ;; If this is vertical, go up based on the tallest node in the level about to be drawn.
		  ;; If horizontal, go right based on the tallest node in the level just drawn
		  (+ position (if (eq orientation :vertical)
				  (if next-level
				      (+ (apply #'max (mapcar size-accessor next-level)) min-level-gap)
				      0)
				  size-of-level)))
	
	;; Find the largest node in the direction that inter-level gaps are oriented and go beyond that
	;; Note that this is slightly too clever for itself, and needs to make sure that there IS another level.
	(size-of-level (+ (apply #'max (mapcar size-accessor next-level)) min-level-gap)
		       (if next-level
			   (+ (apply #'max (mapcar size-accessor next-level)) min-level-gap)
			   0)))
       ((null next-level)
	;; Save up the right edge or bottom  of the graph as apropriate so that we can determine how to
	;; pan through it.  Note that this position includes the unnecessary gap to the next level, so
	;; take that out to determine edge of actual drawing.
	(ccase orientation
	  (:vertical (setf *graph-y-size* (- position size-of-level)))
	  (:horizontal (setf *graph-x-size* (- position min-level-gap))))
	graph-list)

    ;; Spin through this level and set all nodes to start at the apropriate place.  Note that dummy
    ;; nodes should be a very small rectangle going through the level, thus allowing long edges to
    ;; thread their way into the gaps between the real nodes on this level.
    (dolist (next-node next-level)
      (ccase orientation
	(:horizontal (setf (graph-node-left-x next-node) position)
	 (if (graph-node-dummy-p next-node)
	     (progn
	       (setf (graph-node-x-size next-node) (- size-of-level min-level-gap))
	       (setf (graph-node-y-size next-node) 0.5))))
	(:vertical (setf (graph-node-top-y next-node) position)
	 (if (graph-node-dummy-p next-node)
	     (progn
	       (setf (graph-node-y-size next-node) (- size-of-level min-level-gap))
	       (setf (graph-node-x-size next-node) 0.5))))))))


(defun write-postscript-file-header (graph-list stream)
  "This writes out the constant PostScript information to the given file."

  (format stream "%!Automatically generated digraph~%")

  ;; Create a new dictionary which can contain the definitions of /oldlinewidth, /rectangle (and 2 local vars),
  ;; /drawGraph, and all level and node drawing functions, since the default value of 200 is usually too small.
  (let ((num-levels (length graph-list))
	(num-nodes (apply #'+ (mapcar #'length graph-list))))
    (format stream "/drawingDict ~d dict def drawingDict begin~%" (+ num-levels num-nodes 5))
    
    ;; Definition of the /rectangle operator, which takes x y x-offset y-offset and draws the box
    (format stream "%Define the RECTANGLE procedure to draw boxes given a corner and the length of both edges~%")
    (format stream "/rectangle {~%/yOff exch def~%/xOff exch def~%")
    (format stream "moveto xOff 0 rlineto~%0 yOff rlineto~%-1 xOff mul 0 rlineto~%0 -1 yOff mul rlineto } def~2%")))

            
(defun position-sequential (sequence &key (key #'identity) (increment 1) (start 0) (end (length sequence)) (test #'eql))
  "This returns the POSITION in the sequence where it stops being sequential, as indicated by the
   key, increment and test.  For example, the call (position-sequential '(1 2 4) :test #'= ) ==> 2, since
   the 4 does not follow the sequence of the beginning of the list.  As a special case, if the sequence is
   empty this returns nil"

  ;; While not terribly interesting, we don't want to break if the user passes in empty sequences
  (when (zerop (length sequence))
    (return-from position-sequential nil))
  
  (do* ((index start (1+ index))
	(value (funcall key (elt sequence index))
	       (incf value increment)))
       ((= index end) nil)

    (unless (funcall test value (funcall key (elt sequence index)))
      (return-from position-sequential index))))

(defun find-groups-of-children (parent-node)
  "This returns a list of lists, where each sub-list contains those children of this parent which
   have sequential bary-centers and thus can be drawn with a single split point near them. This is
   mostly a helping function for WRITE-POSTSCRIPT-EDGES-TO-CHILDREN."

  ;; If the user doesn't want us to split the lines near the children, just return a list
  ;; indicating that each child is indeed an island, and no split lines will be drawn
  (unless *split-near-children-p*
    (return-from find-groups-of-children
      (mapcar #'list (graph-node-children parent-node))))
  
  (let ((child-list (sort (copy-list (graph-node-children parent-node))
			  #'< :key #'graph-node-bary-center))
	(return-value nil))

    ;; Loop through the list of children, picking out subsequences with sequential bary-centers
    ;; and push them as a sublist onto the return value
    (do* ((rem-children child-list (when next-break (subseq rem-children next-break)))
	  (next-break (position-sequential rem-children :key #'graph-node-bary-center :test #'=)
		      (position-sequential rem-children :key #'graph-node-bary-center :test #'=)))
	 ((null next-break) (push rem-children return-value))
      (push (subseq rem-children 0 next-break) return-value))))

;;; The user doesn't want to know about the dummy nodes, so we just want to present the real
;;; nodes at the appropriate end of the chain for comparison.  Note that we know that dummy
;;; nodes have exactly one child and one parent, so the call to FIRST isn't a hack...
(defun find-real-parent (graph-node)
  (if (eq (graph-node-backend-object graph-node) '$UNKNOWN$)
      (find-real-parent (first (graph-node-parents graph-node)))
      graph-node))

(defun find-real-child (graph-node)
  (if (eq (graph-node-backend-object graph-node) '$UNKNOWN$)
      (find-real-child (first (graph-node-children graph-node)))
      graph-node))

(defun draw-edge-to-single-child (x1 y1 parent child orientation stream)
  "Given the starting position for the edge (either the end of the line from the parent node
   or the split point for clusters of children) this produces PostScript code on the stream
   which will draw a line to the appropriate point on the child."

  ;; A New! Improved!! Whiter!!! feature that allows you to specify the type of edges to draw between the
  ;; given parent and child.  The function stored in *BACKEND-EDGE-WEIGHT-FUNCTION* should return a small
  ;; integer in the approximate range 1-5 for specifying the thickness of the connecting edge.
  (let ((edge-weight (funcall *backend-edge-weight-function*
			      (graph-node-backend-object (find-real-parent parent))
			      (graph-node-backend-object (find-real-child child))))
	x2 y2)
    (ccase orientation
      (:vertical (setf
		   ;; This line connects to the center of the top edge of the child box
		   x2 (+ (/ (graph-node-x-size child) 2.0)
			     (graph-node-left-x child))
		   y2 (graph-node-top-y child)))
      (:horizontal (setf
		     ;; This line connects to the center of the left edge of the child box
		     x2 (graph-node-left-x child)
		     y2 (- (graph-node-top-y child)
			   (/ (graph-node-y-size child) 2.0)))))

    ;; We assume that a path has already been opened for drawing this line
    (emit-code-for-typed-edge stream x1 y1 x2 y2 edge-weight)))

(defun emit-code-for-typed-edge (stream x1 y1 x2 y2 edge-weight)
  (assert (typep edge-weight '(integer 1 5)) (edge-weight) "The edge type must be an integer 1-5")

  ;; The default case is a solid line, so only bother with the dash operator for 2-5
  (unless (= edge-weight 1)
    (format stream "newpath [~d] 0 setdash~%" (* 2 (1- edge-weight))))
  (format stream "~d ~d moveto ~d ~d lineto stroke~%" x1 y1 x2 y2)

  ;; I don't know if this is necessary, but it probably doesn't hurt.
  (format stream "[] 0 setdash~%"))


(defun draw-edges-to-child-group (x1 y1 parent child-group inter-level-gap orientation stream)
  "Given a point near the parent an a list of adjacent children this function produces
   PostScript code on the stream which will draw a line from the parent to a split-point
   near the children, and then individual lines from the split point to each child."

  (if (= (length child-group) 1)
      (draw-edge-to-single-child x1 y1 parent (first child-group) orientation stream)

      ;; There are at least two children in this group so we need a split point
      (let* (x2 y2
	     (last-child (first (last child-group)))

	     ;; Calculate the distance into the gap for the split point
	     (offset-1 (* 0.8 inter-level-gap))
	     ;; And the center of the children contained by this group, which is the average of the BOTTOM
	     ;; of the first child and the TOP of the last child
	     (offset-2 (ccase orientation
			 (:vertical (/ (+ (graph-node-left-x (first child-group))
					  (+ (graph-node-left-x last-child) (graph-node-x-size last-child)))
				       2.0))

			 ;; REMEMBER THIS WELL, MY SON... We start at postion 0 for the first node in the
			 ;; level, adding as we go along the level.  PostScript thinks Y=0 is the BOTTOM of
			 ;; the page, so the first node in the list gets drawn at the lowest position in the graph.
			 ;; HENCE, the bottom of the group is the top of the first child MINUS its height, and the
			 ;; top of the group is the top of the last child
			 (:horizontal (/ (+ (- (graph-node-top-y (first child-group)) (graph-node-y-size (first child-group)))
					    (graph-node-top-y last-child))
				       2.0)))))

;	(offset-2 (/ (+ (/ (funcall size-accessor (first (last child-group))) 2.0)
;			     (apply #'+ (mapcar position-accessor child-group)))
;			  (float (length child-group)))))

	(ccase orientation
	  (:horizontal (setf x2 (+ x1 offset-1)
			   y2 offset-2))
	  (:vertical (setf x2 offset-2
			     y2 (- y1 offset-1))))

	;;Now draw a line from the parent to the split, and then connect the split to each child
	(format stream "~d ~d moveto ~d ~d lineto~%" x1 y1 x2 y2)
	(dolist (next-child child-group)
	  (draw-edge-to-single-child x2 y2 parent next-child orientation stream)))))

(defun extend-parent-to-end-of-level (parent orientation size-of-level stream)
  "This draws a line from the end of the parent box to the end of the level, and returns a list containing
   the x and y coordinates of the end of that line."
  (let ((parent-x (+ (graph-node-left-x parent)
		     (ccase orientation
		       (:horizontal (graph-node-x-size parent))
		       (:vertical (/ (graph-node-x-size parent)
				     2.0)))))
	(parent-y (- (graph-node-top-y parent)
		     (ccase orientation
		       (:vertical (graph-node-y-size parent))
			(:horizontal (/ (graph-node-y-size parent)
					2.0)))))
	(level-x (+ (graph-node-left-x parent)
		    (ccase orientation
		      (:horizontal size-of-level)
		      (:vertical (/ (graph-node-x-size parent)
				    2.0)))))
	(level-y (- (graph-node-top-y parent)
		    (ccase orientation
		      (:vertical size-of-level)
		      (:horizontal (/ (graph-node-y-size parent)
				      2.0))))))

    ;; Write a line from the edge of the box to the end of the level, but not for the longest box(es) on the level
    (unless (and (= parent-x level-x) (= parent-y level-y))
      (format stream "newpath ~d ~d moveto ~d ~d lineto stroke~%" parent-x parent-y level-x level-y))

    ;; Return the end of the level so that the other functions can use it
    (list level-x level-y)))

(defun write-postscript-edges-to-children (parent-node orientation size-of-level stream)
  "This function takes a parent node and adds PostScript commands to create edges connecting the
   parent to each of it's children.  The edges are drawn as a straight line which extends to the
   end of the current level and then branches out to each of the children.  For groups of 2 or more
   adjacent children, a single line is drawn to a split-point near the children and then lines fan
   out to each child in the group.  Each parent can have multiple such groups."

  ;; Start a new PostScript path which will include all of the edges connecting this parent to its children
  (let* ((inter-level-gap (ccase orientation
			    (:horizontal (- (graph-node-left-x (first (graph-node-children parent-node)))
					    (graph-node-left-x parent-node)
					    size-of-level))
			    (:vertical (- (graph-node-top-y parent-node)
					  (graph-node-top-y (first (graph-node-children parent-node)))
					  size-of-level))))
	 (parent-end-list (extend-parent-to-end-of-level parent-node orientation size-of-level stream))
	 (parent-x (first parent-end-list))
	 (parent-y (second parent-end-list))
	 (grouped-children (find-groups-of-children parent-node)))

    ;; Tell the user HOW we have mung'ed the list of adjacent children
    (when (and *graph-loudly* (/= (length (graph-node-children parent-node))
				  (length grouped-children)))
      (format t "Decided to draw ~d children of ~a as ~d group~:P~%"
	      (length (graph-node-children parent-node))
	      (first (graph-node-text-list parent-node))
	      (length grouped-children)))
    
    (dolist (next-group grouped-children)
      (draw-edges-to-child-group parent-x parent-y parent-node next-group inter-level-gap orientation stream))))
    
(defun write-postscript-panning-code (x-scale y-scale stream)
  "This repeatedly translates and redraws sections of the graph, one page at a time."

  ;; Figure out how many 8"x10.5" pages are needed to cover the graph
  (let* ((vertical-points-per-page (/ 756.0 y-scale))
	 (horizontal-points-per-page (/ 576.0 x-scale))
	 (vertical-pages (ceiling (/ *graph-y-size* vertical-points-per-page)))
	 (horizontal-pages (ceiling (/ *graph-x-size* horizontal-points-per-page))))

    ;; Tell the user how big this thing is, in case they are surprised
    (when *graph-loudly*
      (format t "~%The graph is ~d page~:P wide, ~d page~:P high, and was saved as ~A~%"
	      (/ *graph-x-size* horizontal-points-per-page)
	      (/ *graph-y-size* vertical-points-per-page)
	      (pathname stream)))
    
    ;; We have to redo the drawing for each page, after translating the coordinate system first so that
    ;; the current pane to be displayed starts at the 0,0 corner of the printer
    (do* ((horiz-page-count 0 (1+ horiz-page-count))
	  (x-translation 0 (- x-translation horizontal-points-per-page)))
	 ((= horiz-page-count horizontal-pages))
      (do* ((vert-page-count 0 (1+ vert-page-count))
	    (y-translation 0 (- y-translation vertical-points-per-page)))
	   ((= vert-page-count vertical-pages))

	;; The rest of the file written by WRITE-POSTSCRIPT-FILE-FROM-GRAPH defines a function called
	;; drawGraph which contains all of the commands needed to produce the graph.  Just keep calling
	;; it to create the graph, which will be clipped to fit the pane currently visible under these
	;; translations, until all portions of it have been drawn.

	(format stream "~d ~d scale~%" x-scale y-scale)
	(format stream "~d ~d translate~%drawGraph~%showpage~2%"
		x-translation y-translation)))))
	
(defun write-postscript-file-from-graph (graph-list orientation x-scale y-scale output-file)
  "This is the point of this whole project.  This takes a fully specified graph and produces a
   PostScript file which is the graphical representation. In'shalla..."

  ;; We are going to define a function named nodeXXXX to draw each node and the links to its
  ;; children.  These then all get wrapped up in a function called drawGraph which draws the entire
  ;; thing.  This is to keep the size of any single function definition down to a manageable limit.
  (let ((node-counter 0)
	(last-node-counter 0)
	(level-counter 0))
    (with-open-file (s output-file :direction :output :if-exists :new-version)

      ;; There are a number of interesting definitions that are independent of the particular graph
      (write-postscript-file-header graph-list s)

      ;; Now put the boxes and edges for each node
      (dolist (next-level graph-list)

	(let* ((size-accessor (ccase orientation
				(:vertical #'graph-node-y-size)
				(:horizontal #'graph-node-x-size)))
	       (size-of-level (apply #'max (mapcar size-accessor next-level))))

	  (dolist (next-node next-level)
	
	    ;; Start defining a new function which will draw this node and connections to all children
	    (format s "%Function to draw ~a~%" (first (graph-node-text-list next-node)))
	    (format s "/n~d {~%" node-counter)
	  
	    ;; Draw the box surrounding this node
	    (format s "newpath ~d ~d ~d ~d rectangle stroke~%"
		    (graph-node-left-x next-node)
		    (graph-node-top-y next-node)
		    (graph-node-x-size next-node)
		    (* -1 (graph-node-y-size next-node)))

	    ;; Place the text 5 points to the left of the box and 4 points up from the bottom,
	    ;; with any subsequent lines spaced 16 points apart vertically
	    (unless (graph-node-dummy-p next-node)
	      (do* ((x-position (+ (graph-node-left-x next-node) 5))
		    (y-position (- (graph-node-top-y next-node) 12) (- y-position 16))
		    (remaining-strings (graph-node-text-list next-node) (cdr remaining-strings))
		    (next-string (first remaining-strings) (first remaining-strings)))
		   ((null next-string))
	      (format s "~d ~d moveto (~a) show~%" x-position y-position next-string)))

	    ;; Connect this node to all of it's children
	    (when (graph-node-children next-node)
	      (write-postscript-edges-to-children next-node orientation size-of-level s))

	    ;; Close the definition of the function to draw this node
	    (format s " } def~2%")
	    (incf node-counter)))

	;; Define lXXX as a function which draws all nodes on this level
	(if *graph-loudly* (format t "Creating PostScript to draw Level ~d~%" level-counter))
	
	(format s "%Function to draw all nodes in level ~d~%" level-counter)
	(format s "/l~d {~%" level-counter)
	(do ((n last-node-counter (1+ n)))
	    ((= n node-counter))
	  (format s "n~d~%" n))
	(format s " } def~2%")

	;; Adjust the counters to go on to the next level
	(incf level-counter)
	(setf last-node-counter node-counter))

      ;; Define drawGraph to call each of the level drawing functions in turn
      (format s "/drawGraph {~%/Courier findfont 12 scalefont setfont~%")

      (dotimes (n level-counter)
	(format s "l~d~%" n))
      (format s " } def~2%")

      ;; Since a graph may be more than one page, add some great stuff at the bottom of the
      ;; file to automatically print it with the given scale factors on as many pages as are needed.
      (write-postscript-panning-code x-scale y-scale s)

      ;; As I discovered when I tried to incorporate one of these files into a LaTeX document,
      ;; it isn't nice to create a dictionary (as we do) and then not remove it when finished.
      (format s "end % This closes the environment of /drawingDict at the top of the file ~%"))))

;;;************************************************************************************
;;; Some functions to call the usual graph-routines with all of the correct parameters
;;;************************************************************************************
(defun make-kn-rep-postscript-digraphs ()
  "A wrapper which calls MAKE-POSTSCRIPT-DIGRAPH with the kn-objects of interest."
  (make-postscript-digraph
    (mapcar #'getpred
	      '(*thing*))
    :backend-text-function #'text-describing-kn-object
    :backend-children-function #'get-kn-object-children
    :make-levels-from-top-p t
    :min-level-gap 150
    :min-node-gap 15
    :x-scale 0.6 :y-scale 0.6
    :output-filename (picasso-path "lib/hip/core-graph.ps"))
  )

(defun make-krs-digraphs (list-of-kn-object-names)
  (make-postscript-digraph
    (mapcar #'getpred
	      list-of-kn-object-names)
    :backend-text-function #'text-describing-kn-object
    :backend-children-function #'get-kn-object-children
    :min-level-gap 150
    :min-node-gap 15
    :x-scale 0.55 :y-scale 0.55
    :output-filename (picasso-path "lib/hip/krs-graph.ps")))

(defun make-intention-digraphs ()
  (make-postscript-digraph
    *intentions*
    :backend-text-function #'text-describing-intention
    :backend-children-function #'get-intention-children
    :min-level-gap 200
    :min-node-gap 35
    :orientation :horizontal
    :x-scale 0.5 :y-scale 0.5
    :output-filename (picasso-path "lib/hip/intention-graph.ps")))

;(defun make-flavor-digraphs ()
;  (make-postscript-digraph
;    (list (get 'mac::dynamic 'sys:flavor))
;    :backend-text-function #'text-describing-flavor
;    :backend-children-function #'get-flavor-children
;    :min-level-gap 300
;    :min-node-gap 35
;    :orientation :horizontal
;    :x-scale 0.8 :y-scale 0.8
;    :bary-center-passes 20
;    :verbose-nodes-p nil
;    :split-near-children-p nil
;    :output-filename (picasso-path "lib/hip/flavor-graph-dynamic-19.ps")))

(defun make-call-graph ()
  "This is the sample graph given for GRAB in one of the papers"
  (let* ((savestr (make-temp :name 'savestr :children nil))
	 (lookup1 (make-temp :name 'lookup1 :children ( list savestr)))
	 (addarc1 (make-temp :name 'addarc1 :children (list lookup1)))
	 (setin (make-temp :name 'setin :children nil))
	 (addarc (make-temp :name 'addarc :children (list addarc1)))
	 (getstr (make-temp :name 'getstr :children nil))
	 (putcol (make-temp :name 'putcol :children (list setin)))
	 (find1 (make-temp :name 'find1 :children nil))
	 (visit (make-temp :name 'visit :children nil))
	 (print (make-temp :name 'print :children (list putcol)))
	 (readin (make-temp :name 'readin :children (list getstr addarc)))
	 (alphabetize (make-temp :name 'alphabetize :children nil))
	 (find (make-temp :name 'find :children (list find1)))
	 (getitem (make-temp :name 'getitem :children nil))
	 (sclose (make-temp :name 'sclose :children (list visit putcol)))
	 (call (make-temp :name 'call :children (list putcol)))
	 (execute (make-temp :name 'execute :children (list print)))
	 (report (make-temp :name 'report :children (list setin putcol alphabetize find1)))
	 (edit (make-temp :name 'edit :children (list readin find)))
	 (down (make-temp :name 'down :children (list find)))
	 (home (make-temp :name 'home :children (list find)))
	 (quit (make-temp :name 'quit :children nil))
	 (toggle (make-temp :name 'toggle :children nil))
	 (readindex (make-temp :name 'readindex :children (list getitem savestr lookup1)))
	 (help (make-temp :name 'help :children nil))
	 (up (make-temp :name 'up :children nil))
	 (main (make-temp :name 'main :children (list up help readindex toggle quit home down
						     edit report execute call sclose print))))
    (make-postscript-digraph (list main)
			     :backend-text-function #'text-describing-temp
			     :backend-children-function #'temp-children
			     :split-near-children-p nil
			     :orientation :vertical
			     :x-scale 0.6 :y-scale 0.6
			     :output-filename (picasso-path "lib/hip/call-graph.ps"))))
  
(defun default-edge-weight-function (parent child)
  "By default, all edges are drawn in the same 1-unit line size."
  (declare (ignore parent child))
  1)
	  
(defun make-postscript-digraph (graph-list &key
				(backend-text-function '$no-function$ text-fn-p)
				(backend-children-function '$no-function$ children-fn-p)
				(backend-edge-weight-function #'default-edge-weight-function)
				(make-levels-from-top-p t)
				(generate-all-children-p nil)
				(depth-limit nil)
				(bary-center-passes 5)
				(verbose-execution-p t)
				(verbose-nodes-p nil)
				(split-near-children-p t)
				(orientation :horizontal)
				(min-level-gap 150)
				(min-node-gap 25)
				(x-scale 1.0)
				(y-scale 1.0)
				(output-filename (picasso-path "lib/hip/graph.ps"))
				)
  "The user-callable function which does all of the above.  The GRAPH-LIST argument contains the
   directed graph information to be drawn.  Each node in the graph should be represented by an
   instance of the structure GRAPH-NODE and should have the following fields filled in:
       BACKEND-OBJECT should be the backend object being represented by this node.
       LEVEL should be the level of the node in the graph, where each node is below all parents.
       PARENTS and CHILDREN should be lists of the GRAPH-NODE structures which are linked to this one.
       TEXT should be a single line of text to be printed as the label for this node." 

  ;; Make sure that the caller of this function provided all of the interesting information before proceeding
   (assert text-fn-p ()
	  "You did not provide a function to describe your backend objects")
  (assert children-fn-p ()
	  "You did not provide a function to generate the children of your backend objects")
  (setf *backend-text-function* backend-text-function
	*backend-children-function* backend-children-function
	*backend-edge-weight-function* backend-edge-weight-function)
  
  ;; Control the printing of status messages as we proceed
  (setf *graph-loudly* verbose-execution-p)

  ;; Control the amount of text displayed at each node in the final graph
  (setf *verbose-node-text* verbose-nodes-p)

  ;; Indicate whether or not to include individuals when selecting nodes for the graph
  (setf *generate-all-children-p* generate-all-children-p)

  ;; Tell the PostScript output routine whether or not to try splitting lines to adjacent sets of children
  (setf *split-near-children-p* split-near-children-p)
  
  (setf *graph-list*
	(place-levels-in-graph
	  (place-nodes-within-level
	    (determine-size-of-nodes
	      (sort-levels-by-bary-center
		(make-nested-list-from-sorted-list
		  (remove-long-edges-from-graph-list
		    (make-graph-list-from-roots graph-list
						:depth-limit depth-limit
						:make-levels-from-top-p make-levels-from-top-p))
		  #'graph-node-level)
		bary-center-passes))
	    min-node-gap orientation)
	  min-level-gap orientation))

    (write-postscript-file-from-graph *graph-list* orientation x-scale y-scale output-filename))

(defun find-kn-object-in-graph-list (kn-object graph-list)
  (dolist (next-level graph-list)
    (dolist (next-node next-level)
      (when (eq kn-object (graph-node-backend-object next-node))
	(return-from find-kn-object-in-graph-list next-node)))))

(defun show-children (next-node)
  (format t "~&The children of ~s are ~s~%"
		      next-node (graph-node-children next-node)))


(defun print-temp (self stream depth)
  (if (and *print-level*
	   (> depth *print-level*)) ;; Don't print these infinite lists of nested nodes
      (format stream "#")
      (format stream "#<TEMP ~a ~a>"
	      (temp-name self) (temp-children self))))

(defstruct (temp (:print-function print-temp))
  name children)

(defun text-describing-temp (temp)
  (list (symbol-name (temp-name temp))))

(defun hyper-node-edge-weights (parent child)
  (declare (ignore parent))
  (if (hyper-node-children child)
      3
      1))

(defun make-sample-digraph ()
  (let* ((class-defs (make-hyper-node :name "Text: Class Defs." :children nil))
	 (cim-video (make-hyper-node :name "Video: Cim Video" :children nil))
	 (hip-tool (make-hyper-node :name "Text: Hip Tool" :children nil))
	 (progress-report (make-hyper-node :name "Text: Progress Report" :children (list class-defs)))
	 (node-types-table (make-hyper-node :name "Table: Node Types Table" :children nil))
	 (catfud (make-hyper-node :name "Image: Catfud" :children nil))
	 (extending-hip (make-hyper-node :name "Text: Extending Hip" :children (list class-defs)))
	 (hip-overview (make-hyper-node :name "Text: Hip Overview"
					:children (list class-defs extending-hip catfud node-types-table
							progress-report hip-tool cim-video))))
  (make-postscript-digraph
    (list hip-overview)
    :split-near-children-p nil
    :backend-text-function #'text-describing-hyper-node
    :backend-children-function #'hyper-node-children
    :backend-edge-weight-function #'hyper-node-edge-weights
    :min-level-gap 100
    :min-node-gap 50
    :orientation :horizontal
    :x-scale 0.8 :y-scale 0.8
    :output-filename (picasso-path "~lib/hip/ralph-graph.ps"))))

(defun hyper-node-printer (self stream depth)
  (if (and *print-level*
	   (> depth *print-level*))
      (format stream "#")
      (format stream "#<RAIL-NODE ~a ~a>" (hyper-node-name self) (hyper-node-children self))))

(defun text-describing-hyper-node (node)
  (list (hyper-node-name node)))


