;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: chungl $
;;; $Source: /pic2/picasso/src/toolkit/resource/RCS/font.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 1992/04/06 01:07:37 $
;;;

(in-package "PT")

;;; font is a dirty class, i.e. fonts that are created and attached
;;; are not always detached.  Thus we have to keep track of all
;;; fonts created, so we can clean them up.

;; fonts are hashed by name in a hash table stored in the 'font-table
;; slot of the display in which the font exists.
;; get-font looks up a font by name from this hash table.

;; fonts must be associated with a display.  if no display is specified,
;; (current-display) is used.
;; normally the last display attached is the (current-display).


;;;
;;; Definition of the font class
;;;
(defclass font (pmc)
  ((instances
    :initform nil
    :type list
    :accessor instances
    :allocation :class)
   (res
    :initform nil
    :type vector
    :reader res)
   (name
    :initarg :name
    :initform *default-font-name*
    :type string
    :reader name)
   (display
    :initarg :display
    :initform nil
    :type display
    :reader display)
   (ref-count
    :initform 0
    :type integer
    :reader ref-count)))

;;;
;;; 	Dimension accessors
;;;

(defmethod width ((self font)
		  &key 
		  &allow-other-keys)
  (if (attached-p self)
      (xlib:max-char-width (res self))
      0))

(defmethod font-width (self)
  (if (stringp self)
      (font-width (get-font self))
      (if (attached-p self)
	  (xlib:max-char-width (res self))
	  0)))

(defun font-ascent (font &optional gc)
  (if (stringp font)
      (font-height (get-font font) gc)
      (if font 
	  (cond ((font-p font)
		 (if (attached-p font)
		     (xlib:max-char-ascent (res font))
		     0))
		((xlib:font-p font)
		 (xlib:max-char-ascent font))
		(t (error "font-ascent: bad font \'~s\'" font)))
	  (if (and (xlib:gcontext-p gc) (setq font (xlib:gcontext-font gc))) 
	      (xlib:max-char-ascent font)
	      0))))

(defun font-descent (font &optional gc)
  (if (stringp font)
      (font-height (get-font font) gc)
      (if font 
	  (cond ((font-p font)
		 (if (attached-p font)
		     (xlib:max-char-descent (res font))
		     0))
		((xlib:font-p font)
		 (xlib:max-char-descent font))
		(t (error "font-height: bad font \'~s\'" font)))
	  (if (and (xlib:gcontext-p gc) (setq font (xlib:gcontext-font gc))) 
	      (xlib:max-char-descent font)
	      0))))

(defun font-height (font &optional gc)
  (if (stringp font)
      (font-height (get-font font) gc)
      (if font 
	  (cond ((font-p font)
		 (if (attached-p font) 
		     (+ (xlib:max-char-ascent (res font))  
			(xlib:max-char-descent (res font))) 
		     0))
		((xlib:font-p font)
		 (+ (xlib:max-char-ascent font)  
		    (xlib:max-char-descent font))) 
		(t (error "font-height: bad font \'~s\'" font)))
	  (if (and (xlib:gcontext-p gc) (setq font (xlib:gcontext-font gc))) 
	      (+ (xlib:max-char-ascent font)  
		 (xlib:max-char-descent font))
	      0))))

(defmethod height ((font font))
  (if (stringp font)
      (font-height (get-font font))
      (cond ((font-p font)
	     (if (attached-p font) 
		 (+ (xlib:max-char-ascent (res font))  
		    (xlib:max-char-descent (res font))) 
		 0))
	    ((xlib:font-p font)
	     (+ (xlib:max-char-ascent font)  
		(xlib:max-char-descent font))) 
	    (t (error "font-height: bad font \'~s\'" font)))))

;;;
;;;	Path accessors
;;;

(defun font-path (&optional (display (current-display)))
  (xlib:font-path (res display)))

(defun set-font-path (val &optional (display (current-display)))
  (setf (xlib:font-path (res display)) val))

(defun setf-font-path (val)
  (setf (xlib:font-path (res (current-display))) val))

;;;
;;;	Lookup font in table
;;;

(defun get-font (&optional (name *default-font-name*) 
			   spec &aux display)
  (setq display
	(cond ((window-p spec)
	       (display spec))
	      ((display-p spec)
	       spec) 
	      (t (current-display))))
  (gethash name (font-table display)))

;;;
;;;	Create a new font instance or retrieve an old one. . .
;;;
;; if name is a font already created, return the font
;; if name is a valid font name, not yet created, create the font and
;; return it.

(defun make-font (&key
		  (name *default-font-name*)
		  (display (current-display))
		  (attach-p nil)
		  &allow-other-keys
		  &aux font) 
  ;;  set display 
  (if (not (display-p display))
      (setq display (current-display))) 

  ;;  test font, defaulting if invalid
  (setq font 
	(cond ((setq font (get-font name display))
	       font)
	      ((stringp name)
	       (if (font-name-p name display)
		   (make-instance 'font :name name :display display
				  :allow-other-keys t)
		   (progn
		    (warn "make-font: Invalid font-name: \'~s\'." name)
		    (get-font))))
	      (t (get-font))))

  ;;  attach if needed
  (when attach-p (do-attach font))
  font)

(defmethod new-instance ((self font)
			 &key 
			 (display (current-display))
			 &allow-other-keys)
  ;; add self to class instance list.
  (setf (instances self) (cons self (instances self)))
  ;; put in hashtable
  (setf (gethash (name self) (font-table display)) self))

;;;
;;;	Attach & Detach methods
;;;

(defmethod do-attach  ((self font)) 
  (incf (slot-value self 'ref-count))
  (unless (res self)
	  (setf (slot-value self 'res)
		(xlib:open-font (res (display self)) (name self))))
  self)

(defmethod do-detach ((self font) &aux res)
  (when (zerop (decf (slot-value self 'ref-count)))
	(when (setq res (res self))
	      ;; (xlib:close-font res)
	      (setf (slot-value self 'res) nil))))

