(in-package "VIDEO")

(defclass video-disk-player ()
  ((tty :initform "/dev/ttya" :type string :accessor tty)
   (disk :initform nil :accessor disk)
   (stream :initform nil)))

(defun make-video-disk-player (&rest args)
  (apply #'make-instance 'video-disk-player :allow-other-keys t args))

(eval-when (load)
  (setf *current-player* (make-video-disk-player)))

(defun check-reply (player)
  (let ((reply (input-from-player player)))
       (unless (string= reply "R")
	       (format t "Video disk error!~%")
	       (format t "Last command was ~s~%" *last-arg*)
	       (cond ((string= reply "E00")
		      (error "Communication error~%"))
		     ((string= reply "E04")
		      (error "Internal Error -- feature not available~%"))
		     ((string= reply "E06")
		      (error "Internal Error -- missing argument~%"))
		     ((string= reply "E11")
		      (error "Disc not loaded~%"))
		     ((string= reply "E12")
		      (error "Search address not found~%"))
		     ((string= reply "E13")
		      (error "Cannot focus on disc~%"))
		     ((string= reply "E15")
		      (error "Picture stopped~%"))
		     ((string= reply "E99")
		      (error "Unrecoverable video disc error~%"))
		     (t 
		      (cerror "Ignore" "Unknown error code ~s~%" reply)))))
  t)

(defun disconnect-player (&optional (player *current-player*))
  (when (slot-value player 'stream)
	(stop-disc)
        (close (slot-value player 'stream))
	(setf (slot-value player 'stream) nil)))

(defun mode (&optional (player *current-player*))
  (output-to-player player "?P")
  (let ((reply (input-from-player player)))
       (cond ((string= reply "P00") :door-open) 
	     ((string= reply "P01") :park) 
	     ((string= reply "P04") :play)
	     ((string= reply "P05") :still)
	     ((string= reply "P06") :pause) 
	     ((string= reply "P09") :multi-speed) 
	     (t :unknown))))

(defun door-open (&optional (player *current-player*))
  "Open video disc door"
  (output-to-player player "OP")
  (check-reply player))

(defun stop-disc (&optional (player *current-player*))
  "Stop the video disc"
  (output-to-player player "RJ")
  (check-reply player))

(defun start-disc (&optional (player *current-player*))
  "Start the disc rotating"
  (output-to-player player "SA")
  (check-reply player))

(defun search-to-frame (frame &optional (player *current-player*))
  "Goto frame passed"
  (case (mode player)
	(:door-open 
	 (cerror "Close door and retry" "Door open on player")
	 (search-to-frame  frame player))
	((:still :pause :play :multi-speed)
	 (output-to-player player "~dSE" frame)
	 (check-reply player))
	(:park
	 (start-disc player)
	 (search-to-frame  frame player))))

(defun play (&key (player *current-player*)
	     (end nil) (speed 60) &aux (forward t))
  "Play to the specified address"
  (setq speed (max -255 (min speed 255)))
  (if (< speed 0)
      (setq speed (- speed)
	    forward nil))
  (case (mode player)
	(:door-open 
	 (cerror "Close door and retry" "Door open on player")
	 (play :player player :end end :speed speed))
	((:still :pause)
	 (if (and (= speed 60) forward)
	     (if (numberp end)
		 (output-to-player player "FR~dPL" end)
		 (output-to-player player "PL"))
	     (if forward
		 (if (numberp end)
		     (output-to-player player "FR~dSP~dMF" speed end)
		     (output-to-player player "FR~dSPMF" speed))
		 (if (numberp end)
		     (output-to-player player "FR~dSP~dMR" speed end)
		     (output-to-player player "FR~dSPMR" speed))))
	 (check-reply player))
	((:play :multi-speed)
	 (still player)
	 (play :player player :end end :speed speed))
	(:park
	 (start-disc player)
	 (play :player player :end end :speed speed))))

(defun display-text (text &key (line 6) (player *current-player*))
  (output-to-player player "4RA~dPR" line)
  (check-reply player)
  (output-to-player player "~a" text)
  (check-reply player))

(defun clear-text (&optional (player *current-player*))
  (output-to-player player "CS")
  (check-reply player))

(defun raw-output (string &optional (player *current-player*))
  (output-to-player player "~a" string)
  (input-from-player player))

(defun key-lock (&optional (player *current-player*))
  (output-to-player player "1KL")
  (check-reply player))

(defun key-unlock (&optional (player *current-player*))
  (output-to-player player "0KL")
  (check-reply player))

(defun frame-number (&optional (player *current-player*))
  (output-to-player player "?F")
  (input-from-player player))

(defun display-on (&optional (player *current-player*))
  (output-to-player player "1DS")
  (check-reply player))

(defun display-off (&optional (player *current-player*))
  (output-to-player player "0DS")
  (check-reply player))

(defun video-on (&optional (player *current-player*))
  (output-to-player player "1VD")
  (check-reply player))

(defun video-off (&optional (player *current-player*))
  (output-to-player player "0VD")
  (check-reply player))

(defun audio-control (&optional (mode :stereo) (player *current-player*))
  (case mode
	(:off (output-to-player player "0AD"))
	(:ch1 (output-to-player player "1AD"))
	(:ch2 (output-to-player player "2AD"))
	(:stereo (output-to-player player "3AD")))
  (check-reply player))

(defun audio-on (&optional (player *current-player*))
  (output-to-player player "3AD")
  (check-reply player))

(defun audio-off (&optional (player *current-player*))
  (output-to-player player "0AD")
  (check-reply player))

(defun pause (&optional (player *current-player*))
  "Pause the disc"
  (output-to-player player "PA")
  (check-reply player))

(defun still (&optional (player *current-player*))
  "Pause at picture displayed"
  (output-to-player player "ST")
  (check-reply player))

(defun step-forward (&optional (player *current-player*))
  "Step forward one frame"
  (output-to-player player "SF")
  (check-reply player))

(defun step-backward (&optional (player *current-player*))
  "Step forward one frame"
  (output-to-player player "SR")
  (check-reply player))

(defun scan-forward (&optional (count 1) (player *current-player*))
  "Step forward one frame"
  (dotimes (i count)
	   (output-to-player player "NF")
	   (check-reply player)))

(defun scan-backward (&optional (count 1) (player *current-player*))
  "Step forward one frame"
  (dotimes (i count)
	   (output-to-player player "NR")
	   (check-reply player)))

(defun play-range (start end video ch1 ch2 &optional (player *current-player*))
  "Play the specified address range"
  (case (mode player)
	(:door-open 
	 (cerror "Close door and retry" "Door open on player")
	 (play-range start end video ch1 ch2 player))
	((:still :pause)
	 (search-to-frame start player)
	 (if video (video-on player) (video-off player))
	 (cond ((and ch1 ch2) (audio-control :stereo player))
	       (ch1 (audio-control :ch1 player))
	       (ch2 (audio-control :ch2 player))
	       (t (audio-control :off player)))
	 (play :end end :player player))
	((:play :multi-speed)
	 (still player)
	 (play-range start end video ch1 ch2 player))
	(:park
	 (start-disc player)
	 (play-range start end video ch1 ch2 player))))

(defun check-player (player)
  (if (not (typep player 'video-disk-player))
      (error "Attempt to use invalid player ~s~%" player))
  (if (and *real-player-p* (null (slot-value player 'stream)))
      (error "Attempt to used unconnected player ~s~%" player)))

(defun output-to-player (player str &rest args)
  (check-player player)
  (let ((stream (slot-value player 'stream)))
       (setq *last-arg* (apply #'format nil str args))
       (if *real-player-p*
	   (format (slot-value player 'stream) "~a" *last-arg*)
	   (format t "~a" *last-arg*))
       (if *real-player-p*
	   (format (slot-value player 'stream) "~%")
	   (format t "~%"))
       (force-output stream)))

(defun input-from-player (player)
  (check-player player)
  (if *real-player-p*
      (read (slot-value player 'stream))
      "R"))

(defun connect-player (&optional (player *current-player*))
  (when (slot-value player 'stream)
        (warn "connect-player: Player ~s already connected~%" player)
        (return-from connect-player (slot-value player 'stream)))
  (if *real-player-p*
      (setf (slot-value player 'stream)
	    (open (tty player) :direction :io :if-exists :append)))
  (if (null (slot-value player 'stream))
      (progn
          (setf *real-player-p* nil)
	  (warn "connect-player: Cannot connect on ttya ~s~%" (tty player)))
      ;; set up the (open) tty...
      (let* ((tty (tty player))
	     #+allegro
             (cmd (format nil "stty 4800 -parity cs8 -ixon -echo > ~s" tty))
	     #+lucid
	     (cmd (list "4800" "-parity" "cs8" "-ixon" "-echo")))
   	#+allegro 
	(excl:run-shell-command cmd :wait t)
	#+lucid
	(user::run-program "stty" :if-output-exists :append
	:output tty :arguments cmd :wait t)))
  (slot-value player 'stream))
