(defclass video-disk-player ()
  ((tty :initform "/dev/ttya" :type string :accessor tty)
   (disk :initform nil :accessor disk)
   (stream :initform nil)))

(defun make-video-disk-player (&rest args)
  (apply #'make-instance 'video-disk-player :allow-other-keys t args))

(defvar *current-player* nil)

(eval-when (load)
  (setf *current-player* (make-video-disk-player)))

(defun start-disc (&optional (player *current-player*))
  "Start the disc rotating"
  (output-to-player player "SA")
  (check-reply player))

(defun stop-disc (&optional (player *current-player*))
  "Stop the video disc"
  (output-to-player player "RJ")
  (check-reply player))

(defun check-reply (player)
  (let ((reply (input-from-player player)))
       (unless (string= reply "R")
               (format t "Video disk error!~%")
               (format t "Last command was ~s~%" *last-arg*)
               (cond ((string= reply "E00")
                      (error "Communication error~%"))
                     ((string= reply "E04")
                      (error "Internal Error -- feature not available~%"))
                     ((string= reply "E06")
                      (error "Internal Error -- missing argument~%"))
                     ((string= reply "E11")
                      (error "Disc not loaded~%"))
                     ((string= reply "E12")
                      (error "Search address not found~%"))
                     ((string= reply "E13")
                      (error "Cannot focus on disc~%"))
                     ((string= reply "E15")
                      (error "Picture stopped~%"))
                     ((string= reply "E99")
                      (error "Unrecoverable video disc error~%"))
                     (t
                      (cerror "Ignore" "Unknown error code ~s~%" reply)))))
  t)

(defun disconnect-player (&optional (player *current-player*))
  (when (slot-value player 'stream)
        (stop-disc)
        (close (slot-value player 'stream))
        (setf (slot-value player 'stream) nil)))

(defun check-player (player)
  (if (not (typep player 'video-disk-player))
      (error "Attempt to use invalid player ~s~%" player))
  (if (and *real-player-p* (null (slot-value player 'stream)))
      (error "Attempt to used unconnected player ~s~%" player)))

(defun output-to-player (player str &rest args)
  (check-player player)
  (let ((stream (slot-value player 'stream)))
       (setq *last-arg* (apply #'format nil str args))
       (if *real-player-p*
           (format (slot-value player 'stream) "~a" *last-arg*)
           (format t "~a" *last-arg*))
       (if *real-player-p*
           (format (slot-value player 'stream) "~%")
           (format t "~%"))
       (force-output stream)))

(defun input-from-player (player)
  (check-player player)
  (if *real-player-p*
      (read (slot-value player 'stream))
      "R"))

(defun connect-player (&optional (player *current-player*))
  (when (slot-value player 'stream)
        (warn "connect-player: Player ~s already connected~%" player)
        (return-from connect-player (slot-value player 'stream)))
  (if *real-player-p*
      (setf (slot-value player 'stream)
            (open (tty player) :direction :io :if-exists :append)))
  (if (null (slot-value player 'stream))
      (progn
          (setf *real-player-p* nil)
          (warn "connect-player: Cannot connect on ttya ~s~%" (tty player)))
      ;; set up the (open) tty...
      (let* ((tty (tty player))
             #+allegro
             (cmd (format nil "stty 4800 -parity cs8 -ixon -echo > ~s" tty))
             #+lucid
             (cmd (list "4800" "-parity" "cs8" "-ixon" "-echo")))
        #+allegro
        (excl:run-shell-command cmd :wait t)
        #+lucid
        (user::run-program "stty" :if-output-exists :append
        :output tty :arguments cmd :wait t)))
  (slot-value player 'stream))
