;;;
;;; Abstraction for Time Coordinate Systems...
;;;
;;; A timevalue is an ADT that specifies a point in time relative to
;;; a coordinate system (TCS).  For example, a timevalue for a laserdisc player
;;; is an integer and the the TCS specifies this is to be interpreted as a
;;; frame number.  As a second example, the timevalue for the VCR is a
;;; record with four integers, and the TCS specified that this is a timecode
;;; (hour:minute:second:frame).  Various function are available for
;;; converting between TCS's.
;;; 

(defclass tcs () nil
  (:documentation "An abstract class for time coordinate systems"))

(defgeneric calculate-seconds (tcs)
  (:documentation "Return the number of seconds in a tcs"))

;;; ========================================================================

(defclass timecode (tcs)
  ((hours :initarg :hours :initform 0 :type fixnum :accessor hours)
   (minutes :initarg :minutes :initform 0 :type fixnum :accessor minutes)
   (seconds :initarg :seconds :initform 0 :type fixnum :accessor seconds)
   (frame :initarg :frame :initform 0 :type fixnum :accessor frame)))

(defun make-timecode (&key (hours 0) (minutes 0) (seconds 0) (frame 0))
  (make-instance 'timecode
		 :hours (coerce hours 'fixnum)
		 :minutes (coerce minutes 'fixnum)
		 :seconds (coerce seconds 'fixnum)
		 :frame (coerce frame 'fixnum)))

(defmethod calculate-seconds ((self timecode))
  (+ (seconds self)
     (* 60 (minutes self))
     (* 3600 (hours self))
     (/ (frame self) 30)))

(defmethod update-instance-for-different-class :after
  ((previous tcs) (target timecode) &rest initargs &aux (frame 0) (time 0))
  (declare (fixnum frame time))
  (multiple-value-setq (time frame) (truncate (calculate-seconds previous)))
  (setf (frame target) (round (* 30 frame))
	(seconds target) (mod time 60)
	time (truncate (/ time 60))
	(minutes target) (mod time 60)
	time (truncate (/ time 60))
	(hours target) time))

;;; ========================================================================

(defclass frame-number (tcs)
  ((frame :initarg :frame :initform 0 :type fixnum :accessor frame)))

(defun make-frame-number (&key (frame 0))
  (make-instance 'frame-number :frame (coerce frame 'fixnum)))

(defmethod calculate-seconds ((self frame-number))
  (/ (frame self) 30))

(defmethod update-instance-for-different-class :after
  ((previous tcs) (target frame-number) &rest initargs)
  (setf (frame target) (round (* 30 (calculate-seconds previous)))))
