;;;
;;; Shared Object Hierarchy
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/widgets/soh/RCS/store.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:08:49 $
;;;

#+libpq
(in-package 'pcl :use '(lisp libpq))

#+cling
(in-package 'pcl :use '(lisp cling))

(defvar access-table (make-hash-table))
(defvar regexp-table (make-hash-table))

(defvar oid-table (make-hash-table))
(defvar object-table (make-hash-table))

;;; local-table will cache the make-functions in the "local" relation
;;;  key == Loid, val == make-function.
(defvar local-table (make-hash-table))

(proclaim '(special access-table 
		    regexp-table 
		    local-table 
		    oid-table 
		    object-table))

(defun reset-soh ()
  (clrhash access-table)
  (clrhash regexp-table)
  (clrhash oid-table)
  (clrhash local-table)
  (clrhash object-table))

#+libpq
(defun cache-locals ()
  (clrhash local-table)
  (libpq::pqexec "retrieve (local.all)")
  (let* ((larray (libpq::pqparray "blank"))
         (n (libpq::pqntuples larray)))
    (dotimes (i n)
             (setf (gethash (read-from-string
			 (libpq::pqgetvalue larray i 0)) local-table)
		     (libpq::pqgetvalue larray i 1))))
    t)

#+cling
(defun cache-locals ()
  (clrhash local-table)
  (let* ((larray (db-select '(l.loid l.make) :from '((local l)))))
    (mapcar #'(lambda (tuple)
		(setf (gethash (read-from-string (first tuple)) local-table)
		      (second tuple)))
	    larray))
  t)

(defmethod store-object ((self t))
  ;; return self
  self)

(defmethod object-id ((self t))
  ;; get object information
  (let* ((oid (gethash self object-table :failure)))
    ;; test if object has been stored
    (if (eq oid :failure)
	;; get class name and universal time
	(let ((cn (class-name (class-of self)))
	      (ut (get-universal-time)))
	  ;; generate object-id
	  ;;(setq oid (read-from-string 
	  ;;	     (symbol-name (gensym (format nil "~s.~x" cn ut)))))))
	  (setq oid (read-from-string 
	  	     (symbol-name (gensym (format nil "~s" cn)))))))
    ;; return oid
    oid))

(defmethod store-object ((self cons))
  ;; get cons information
  (let* ((oid (object-id self))
	 (regop (format nil "(let ((~s (cons" oid))
	 (setops nil)
	 (sval nil)
	 (soid nil))
    ;; insert self into access-table
    (setf (gethash self access-table) oid)
    ;; get car of cons
    (setq sval (car self))
    ;; get oid of car
    (setq soid (gethash soid access-table))
    ;; test if car has been seen
    (when (null soid)
	  ;; store sval
	  (store-object sval)
	  ;; get objectid of sval
	  (setq soid (gethash sval access-table)))
    ;; test if sval was accessed (i.e. a composite object)
    (when (not (null soid))
	  ;; push setf & load setop onto setops list
	  (push `(setf (car ,oid) (find-object ',soid)) setops)
	  ;;(push `(load-object ',soid) setops)
	  (push `(lo ',soid) setops)
	  ;; set value to nil
	  (setq sval nil))
    ;; add car to regop
    (setq regop (format nil "~a ~s" regop sval))
    ;; get cdr of cons
    (setq sval (cdr self))
    ;; get oid of cdr
    (setq soid (gethash soid access-table))
    ;; test if cdr has been seen
    (when (null soid)
	  ;; store sval
	  (store-object sval)
	  ;; get objectid of sval
	  (setq soid (gethash sval access-table)))
    ;; test if sval was accessed (i.e. a composite object)
    (when (not (null soid))
	  ;; push setf & load setop onto setops list
	  (push `(setf (cdr ,oid) (find-object ',soid)) setops)
	  ;;(push `(load-object ',soid) setops)
	  (push `(lo ',soid) setops)
	  ;; set value to nil
	  (setq sval nil))
    ;; add cdr to regop
    (setq regop (format nil "~a ~s" regop sval))
    ;; close let and append register expression
    (setq regop (format nil "~a))) (register-object '~s ~s)" regop oid oid))
    ;; test if setops is non-nil
    (if (not (null setops))
	;; append setops to regop
	(dolist (setop setops)
		;; append setop
		(setq regop (format nil "~a ~s" regop setop))))
    ;; close regop sexp
    (setq regop (format nil "~a ~s)" regop oid))
    ;; store regop
    (store-local oid regop)
    (setf (gethash oid regexp-table) regop)
    ;; register object in memory
    (register-object oid self)
    ;; return oid
    oid))

#+libpq
(defun store-local (oid regop)
  (let ((loid (symbol-name oid))
	(pqop nil))
    (setq pqop (format
		nil 
		"delete local where local.loid = text[~s]" 
		loid))
    (libpq::pqexec pqop)
    (setq pqop (format
		nil 
		"append local (loid = text[~s], make = text[~s])"
		loid regop))
    (libpq::pqexec pqop)))

#+cling
(defun store-local (oid regop)
  (let ((loid (symbol-name oid)))
    (db-delete 'local :where `(= local.loid ,loid))
    (db-insert 'local '(loid make) 
	       (list loid (of-value-to-db regop)))))

(defmethod store-slots ((self object)
			&optional
			(slots nil slotsp))
  ;; test if slots are specified
  (if (null slotsp)
      ;; get slots from class
      (setq slots (class-slots (class-of self)))
    ;; test slot list
    (if (not (listp slots))
	;; signal error
	(error "store-slots: invalid slot list ~s" slots)))
  ;; coerce list to all slot descriptors
  (do* ((i 0 (incf i))
	(s (nth i slots) (nth i slots))
	(d nil))
       ((= i (length slots)))
       ;; test if slot descriptor or slot name
       (when (symbolp s)
	     ;; get slot descriptor of this name
	     (setq d (find s (class-slots (class-of self))
			   :key #'slotd-name))
	     ;; test if descriptor was found
	     (if (null d)
		 ;; signal error
		 (error "store-slots: can't find slotd named ~s" s))
	     ;; change name to slot descriptor
	     (setf (nth i slots) d)))
  ;; remove slots exempt from db
  ;; (setq slots (remove-exempt self slots))
  ;; allocate locals for slot information
  (let ((sname nil)
	(svalue nil)
	(sclass nil)
	(soid nil))
    ;; store slot values
    (dolist (s slots)
	    ;; get slot name and value
	    (setq sname (slot-value s 'name))
	    (setq svalue (slot-value self sname))
	    ;; determine the class of the slot
	    (let ((stype (slot-value s 'type)))
	      ;; test if the slot's value is non-nil
	      (if (null svalue)
		  ;; attempt to determine the slot class
		  (if (listp stype)
		      ;; get least specific type
		      (setq sclass (find-class (car (last stype))))
		    ;; get class from type
		    (setq sclass (find-class (slot-value s 'type))))
		;; get class from value
		(setq sclass (class-of svalue))))
	    ;; test if svalue has been seen
	    (setq soid (gethash svalue access-table))
	    ;; test if svalue was accessed (i.e. not yet stored)
	    (when (null soid)
		  ;; store-object svalue
		  (store-object svalue)
		  ;; get objectid of svalue
		  (setq soid (gethash svalue access-table)))
	    ;; test if svalue was accessed (i.e. a composite object)
	    (when (not (null soid))
		  ;; push load setop onto setops list
		  (push `(setf (slot-value ,oid ',sname) 
			       (find-object ',soid)) setops)
		  ;;(push `(load-object ',soid) setops)
		  (push `(lo ',soid) setops)
		  ;; set svalue to nil
		  (setq svalue nil))
	    ;; add slot -name and -value to regop
	    (setq regop (format nil "~a :~s ~s" regop sname svalue)))))

(defmethod store-object ((self object))
  ;; get object information
  (let* ((class (class-of self))
	 (cname (class-name class))
	 (oid (object-id self))
	 (regop (format nil "(let ((~s (make-instance '~s" oid cname))
	 (setops nil))
    ;; make oid, regop and setops special
    (declare (special oid regop setops))
    ;; insert self into access-table
    (setf (gethash self access-table) oid)
    ;; store the slots
    (store-slots self)
    ;; close let and append register expression
    (setq regop (format nil "~a))) (register-object '~s ~s)" regop oid oid))
    ;; test if setops is non-nil
    (if (not (null setops))
	;; append setops to regop
	(dolist (setop setops)
		;; append setop
		(setq regop (format nil "~a ~s" regop setop))))
    ;; close regop sexp
    (setq regop (format nil "~a ~s)" regop oid))
    ;; store regop
    (store-local oid regop)
    (setf (gethash oid regexp-table) regop)
    ;; register object in memory
    (register-object oid self)
    ;; return oid
    oid))

(defun load-object (oid)
  ;; search for it in the local-table first
  (let ((make-fun (gethash oid local-table nil)))
    ;; if present, evaling it will insert it in the oid and object tables
    (when make-fun
      (eval (read-from-string make-fun))
      ;; now remove it since it is already in the other tables
      (remhash oid local-table)))
  ;; allocate object
  (let ((object (gethash oid oid-table :failure)))
    ;; search for object in oid-table
    (if (eq object :failure)
	;; need to fetch from the backend
	(let ((regexp (fetch-make oid)))
	  (if (null regexp) 
	      ;; signal error
	      (error "load-object: can't find object ~s" oid)
	    ;; evaluate regexp
	    (setq object (eval (read-from-string regexp))))))
    ;; return object
    object))

#+libpq
(defun fetch-make (oid)
  "Return make string or nil on failure"
  (let* ((loid (symbol-name oid))
	 (pqop (format
		nil
		"retrieve (local.all) where local.loid = text[~s]"
		loid))
	 (parray nil))
    ;; exec pqop
    (libpq::pqexec pqop)
    ;; get portal array
    (setq parray (libpq::pqparray "blank"))
    ;; test for retrieve failure
    (if (zerop (libpq::pqntuples parray))
	;; no entry found
	nil
      (let* ((fnumber (libpq::pqfnumber parray 0 "make"))
	     (regexp (libpq::pqgetvalue parray 0 fnumber)))
	regexp))))

#+cling
(defun fetch-make (oid)
  "Return make string or nil on failure"
  (let ((tuples (db-select '(l.make) :from '((local l)) 
			   :where `(= l.loid ,(symbol-name oid)))))
    (if (null tuples)
	nil
      (first (first tuples)))))

;; lo is an alias for load-object, so the number of chars
;; sent to backend is minimized
(setf (symbol-function 'lo) (symbol-function 'load-object))

(defun find-object (oid &optional (default :failure))
  ;; allocate object
  (let ((object (gethash oid oid-table default)))
    ;; test for failure
    (if (eq object default)
	;; warn that object was not found
	(warn "find-object: object-id ~s not found" oid))
    ;; return object
    object))

(defun register-object (oid object)
  ;; test if object is already registered
  (when (eq (gethash oid oid-table :failure) :failure)
	;; put object in oid-table
	(setf (gethash oid oid-table) object)
	;; put oid in object-table
	(setf (gethash object object-table) oid))
  ;; return object
  object)

(defun find-regexp (oid &optional (default :failure))
  ;; allocate regexp
  (let ((regexp (gethash oid regexp-table default)))
    ;; test for failure
    (if (eq regexp :failure)
	;; warn that regexp was not found
	(warn "find-regexp: object-id ~s not found" oid))
    ;; return regexp
    regexp))

;;;
;;; Exports from PCL package
;;;
(export '(store-object
	  load-object
	  lo
	  find-object
	  object-id
	  find-regexp
	  cache-locals
	  local-table
	  register-object)
	(find-package 'pcl))
