;;;
;;; Shared Object Hierarchy
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/widgets/soh/RCS/map.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:08:49 $
;;;

#+libpq
(in-package 'pcl :use '(lisp libpq))

#+cling
(in-package 'pcl :use '(lisp cling))

;;;
;;; This is a hack to get around the postgres limit on the length
;;; of class and attribute names.
;;;

#+longsymbol
(defvar *symbol-to-db* (make-hash-table :test #'eq))
#+longsymbol
(defvar *symbol-from-db* (make-hash-table :test #'eq))

#+longsymbol
(defun long-symbol-hack (symbol-list)
  (clrhash *symbol-to-db*)
  (clrhash *symbol-from-db*)
  (dolist (s symbol-list)
          (setf (gethash (car s) *symbol-to-db*) (cadr s))
	  (setf (gethash (cadr s) *symbol-from-db*) (car s))))

;;;
;;; POSTGRES does not allow the character '-' within a symbol, so we have to
;;; convert #\- in Commom Lisp symbol names to #\_ when storing to database
;;; and convert it back when fetching from database. 
;;; If you happen to use both '-' and '_', you LOSE.
;;;

#+longsymbol
(defun symbol-to-db (symbol)
  "Convert a Common Lisp \fBsymbol\fP to a database string"
  (let ((dbsymbol (gethash symbol *symbol-to-db*)))
    (if (null dbsymbol)
	(setq dbsymbol symbol))
    (substitute #\_ #\- (string-downcase (symbol-name dbsymbol)))))
	 
#-longsymbol
(defun symbol-to-db (symbol)
  "Convert a Common Lisp \fBsymbol\fP to a database string"
  (substitute #\_ #\- (string-downcase (symbol-name symbol))))
	 
#+longsymbol
(defun symbol-from-db (string)
  "Convert a database \fBstring\fP back to a Common Lisp symbol"
  (if (symbolp string) (setq string (prin1-to-string string)))
  (let* ((dbsymbol (read-from-string (substitute #\- #\_ string)))
	 (symbol (gethash dbsymbol *symbol-from-db*)))
    (if symbol
	symbol
      dbsymbol)))

#-longsymbol
(defun symbol-from-db (string)
  "Convert a database \fBstring\fP back to a Common Lisp symbol"
  (if (symbolp string) (setq string (prin1-to-string string)))
  (read-from-string (substitute #\- #\_ string)))

;;;
;;; The way SOH catalogs are stored depend on the database catalog.
;;; For example, class named are stored in char16 at current time.
;;; We need functions to convert class names, etc. to the appropriate
;;; form to be stored in the database.  Consult the initialization
;;; database functions for the description of SOH catalogs.
;;;
(defun class-name-to-db (class-name)
  "Convert a class name to the format to be stored in the database"
  (symbol-to-db class-name))

;;;
;;; *OFTODB* is a hashtable mapping CLOS types to database types.
;;; Add an entry to this table for each type mapped to other than TEXT.
;;; In the future this table will be fetched from the database when the
;;; system is initialized.
;;;

(defvar *oftodb* (make-hash-table :test #'eq))
(defmacro maptype (lisptype dbtype)
  `(setf (gethash ,lisptype *oftodb*) ,dbtype))

#+libpq
(progn
  ;(maptype 'integer "int4")
  )


#+cling
(progn
 (maptype 'integer 'integer4)
 (maptype 'number 'integer4)
 (maptype 'single-float 'float4)
 (maptype 'string '(varchar 64))
 (maptype 'symbol '(varchar 32))
)

;;;
;;; All the slot types must be of either BUILT-IN-CLASS, CLASS, or DBCLASS.
;;; Otherwise it will be mapped to TEXT type.
;;; In PCL 5/22/87 release, only the following Common Lisp types have classes:
;;;
;;;  ARRAY BIT-VECTOR CHARACTER COMPLEX CONS FLOAT INTEGER LIST
;;;  NULL NUMBER RATIO RATIONAL SEQUENCE STRING SYMBOL T VECTOR
;;;

;;;
;;; OF-TYPE-TO-DB maps an OBJFADS type (a class) to a database type.
;;; Default type is text.
;;;

#+libpq
(progn
  (defvar *db-text-type* "text")
  (defmacro of-value-to-text (value)
    `(format nil "text[~S]" ,value))
  )

#+cling
(progn
  (defvar *db-text-type* '(varchar 128))
  (defmacro of-value-to-text (value)
    ;; Values mapped by DB interface
    value)
  )

(defmethod of-type-to-db ((self built-in-class))
  "By default TEXT data type is used"
  (gethash (class-name self) *oftodb* *db-text-type*))

(defmethod of-type-to-db ((self class))
  "For CLASS objects, a make function is stored in the TEXT form"
  *db-text-type*)

(defmethod of-type-to-db ((self dbclass))
  "For DBCLASS objects (dbobjects), its objid is stored in the TEXT form"
  *db-text-type*)

(defmethod of-type-to-db (self)
  "Default types are stored in the TEXT form"
  (gethash self *oftodb* *db-text-type*))

;;;
;;; OF-VALUE-TO-DB mapps an OBJFADS value to a database value
;;;

;;;
;;; By default, the printed format is stored for plain values.
;;; For compiled functions, a mapping function is stored.
;;;

#+libpq
(defmacro of-value-to-db-type (type self)
  ;; Try to cast into a given type.  Ignored in postgres.
  `(of-value-to-db ,self))

#+cling
(defun of-value-to-db-type (type self)
  ;; Try to cast into desired type.
  (if (and (member type '(integer number float))
	   (eq (class-of self) (find-class type)))
      self
    (of-value-to-db self)))

(defmethod of-value-to-db (self)
  "Store compiled-function via function mapping.  
   By default store an OBJFADS value in its print form"
  (cond ((eq (type-of self) 'compiled-function)
	 (of-value-to-text 
	  (prin1-to-string 
	   (list 'fn-fetch-instance
		 (list 'quote (fn-fetch-name self))))))
	(t
	 (of-value-to-text (prin1-to-string self)))))

#|
#+libpq
(progn
  (defmethod of-value-to-db ((self string))
    "Store string in database as a string without the double quotes"
    (format nil "text[~S]" self))
  (defmethod of-value-to-db ((self integer))
    "Integer is stored in the type int4 (4 byte integer)"
    (prin1-to-string self))
  )
|#

(defmethod of-value-to-db ((self object))
  "Store a make function for an object"
  (of-value-to-text (build-make-function self)))

(defmethod of-value-to-db ((self dbobject))
  "Store the objid for a dbobject"
  ;(store-dbobject self)
  ;; HACK
  (when (minusp (cadr (handle-id self)))
        (handle-set-id self 
		       (make-objid (class-of self) 'deferred-update))
	(handle-set-indb self t)
	(handle-set-modified self t)
	(handle-set-mode self 'deferred-update)
	(handle-set-pinned self nil)
	(put-into-object-cache self))
  (of-value-to-text
	  (prin1-to-string  (handle-id self))))

(defmethod of-value-to-db ((self cons))
  "Lists need special treatment"
  (of-value-to-text (prin1-to-string (of-value-to-db-internal self))))

(defmethod of-value-to-db-internal ((self cons))
  (if (null self)
      nil
    (cons (of-value-to-db-internal (car self))
	  (of-value-to-db-internal (cdr self)))))

(defmethod of-value-to-db-internal (self)
  self)

(defmethod of-value-to-db-internal ((self object))
  "Store a make function for an object"
  (read-from-string (build-make-function self)))

(defmethod of-value-to-db-internal ((self dbobject))
  (when (minusp (cadr (handle-id self)))
        (handle-set-id self
		       (make-objid (class-of self) 'deferred-update))
	(handle-set-indb self t)
	(handle-set-modified self t)
	(handle-set-mode self 'deferred-update)
	(handle-set-pinned self nil)
	(put-into-object-cache self))
  (handle-id self))

(defmethod of-restriction-to-db ((self list))
  (mapcar #'of-restriction-to-db self))

(defmethod of-restriction-to-db ((self symbol))
  (read-from-string (substitute #\_ #\- (prin1-to-string self))))

(defmethod of-restriction-to-db (self)
  (of-value-to-db self))
;;;
;;; work to be done for of-value-to-db:
;;; 1) we may need to check that the dbobject referenced is in DB
;;; this will be added later on
;;; 2) There is also a consistency problem here, what will happen
;;; if the dbobject is updated after its referee is stored to
;;; the database. This will be considered together with other 
;;; consistency problems.
;;; 3) for the time being, we assume objects do not referrence to
;;; dbobjects.
;;; 4) It might be a good idea to store a function 
;;; (dbobject-cache objid) instead of the objid
;;; Then if an object referrences to a dbobject, the make function
;;; will automatically get the dbobject.
;;;
#|
(defmethod build-make-function ((self object))
  "Build a make function for a CLOS object"
  (let* ((class (class-of self))
	 (instance-slots (class-slots class))
	 (make-function ""))
    (when (not (null instance-slots))
	  (dolist (c instance-slots)
		  (setq make-function (format nil "~A :~A ~A"
					      make-function
					      (symbol-name (slotd-name c)) 
					      (build-make-function (slot-value self (slotd-name c)))))))
    (format nil "(make '~A ~A)" 
	    (symbol-name (class-name class))
	    make-function)))
|#

;;;
;;; the methods 'store is defined by D.C.Martin in store-object.cl
;;;
;;(defmethod build-make-function ((self object))
;;  (format nil "(load-object '~s)" (store-object self)))
 
(defmethod build-make-function ((self object))
  (format nil "(lo '~s)" (store-object self)))
 
(defmethod build-make-function (self)
  "By default build-make-function returns the value itself as a string"
  (format nil "~S" self))

(defmethod build-make-function ((self dbobject))
  "An Object is not allowed to reference dbobject for the current time"
  (error "An object which references dbobject(s) cannot be stored into the database"))

;;;
;;; DB-VALUE-TO-OF converts a database value to a certain type.
;;; If the type is not implemented as a class by PCL,
;;; (find-class slot-type) will signal an error.
;;;
;;; should write methods for this function
;;;
(defun db-value-to-of (slot-type db-value)
  (if (equal db-value "") (setq db-value "NIL"))
  (cond
   ((and (eq slot-type 'string) (stringp db-value)) (read-from-string db-value))
   ((eq slot-type (class-name (class-of db-value))) (hack-value db-value))
   (t
    (case (class-name (class-of (find-class slot-type)))
	  ((built-in-class) (hack-value (read-from-string db-value)))
	  ((class) (hack-value (read-from-string db-value)))
	  ((standard-class) (hack-value (read-from-string db-value)))
	  ((dbclass) (if (string-equal "NIL" db-value) nil
		       (let ((objid-list (read-from-string db-value)))
			 (make-handle (find-class slot-type) objid-list))))))))
;;;
;;; Some forms need to be evaluated
;;;
(defun hack-value (value)
  (if (listp value)
      (if (or (equal (car value) '*make-instance)
	      (equal (car value) 'make-instance)  ;; we changed this before
	      (equal (car value) 'load-object)
	      (equal (car value) 'lo) ;; alias for load-object
	      (equal (car value) 'fn-fetch-instance))
	  (eval value)
	value)
    value))


;;;
;;; exports
;;;

#+longsymbol
(export '(of-type-to-db
	  of-value-to-db
	  db-value-to-of
	  build-make-function
	  *long-symbol-mappings*
	  long-symbol-hack)
	(find-package 'pcl))

#-longsymbol
(export '(of-type-to-db
	  of-value-to-db
	  db-value-to-of
	  build-make-function)
	(find-package 'pcl))

