;;;
;;; Shared Object Hierarchy
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/widgets/soh/RCS/dbobject.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:08:49 $
;;;

(in-package clos)
(use-package 'libpq 'cling)

;;;
;;; DBOBJECT
;;;	The class dbobject should be inherited by all shared classes.
;;;

(defclass dbobject (object) 
  ;; Access to objid is actually trapped to handle-id.
  ((sohid :type integer))
  (:metaclass dbclass))

(defmethod dbobject-p ((self dbobject))
  t)

(defmethod dbobject-p (self)
  (declare (ignore self))
  nil)

;;;
;;; MAKE-INSTANCE.
;;; If we need to change the behavior of make-instance, we can 
;;; change the following method.
;;;

#|
(defmethod *make-instance ((class dbclass) &rest initargs)
  (setq initargs (*default-initargs class initargs))
  (let ((instance (apply #'allocate-instance class initargs)))
    (apply #'*initialize-instance instance initargs)
    instance))
|#

;;;
;;; INITIALIZE-DBOBJECT.
;;; Besides the normal initialization, we also need to assign
;;; an objid to the object, and put it in the object cache.
;;; The reason that I choose to change initialize-instance
;;; instead of shared-initialize is that shared-initialize
;;; is also called by other initialization methods such as
;;; reinitialize-instance.  It is not clear whether we always
;;; want to call initialize-dbobject.
;;;	
;;; An instance is usually initialized before it is put into the 
;;; object cache (except when a handle already exists). 
;;; 
;;; Besides initial values for ordinary slots, the initargs can
;;; also contain two other keyword values :dbmode and :objid.
;;; The :dbmode determines the update mode for the object created.
;;; The :objid is only provided for objects fetched from the database.
;;; These objects already have an objid.
;;;

;;; this is the method for the new initialization protocol, but
;;; it doesn't quite work, neither initform nor initargs are in
;;; effect.
(defmethod *initialize-instance ((instance dbobject) &rest initargs)
  (apply #'shared-initialize instance t initargs)
  (initialize-dbobject instance initargs))


(defun initialize-dbobject (instance initargs)
  (let ((dbmode (cadr (member ':dbmode initargs))))
    (if (illegal-mode dbmode) (setq dbmode *default-mode*))
    ;; If an object is fetched from the database, put it in the cache.
    (if (handle-id instance)
	;; In some cases, such as when a handle already exists, we
	;; may end up putting it twice, but it doesn't really matter.
	(progn
	  (handle-set-indb instance t)
	  (handle-set-modified instance nil)
	  (handle-set-mode instance dbmode)
	  (handle-set-pinned instance nil)
	  (put-into-object-cache instance))
      ;; If an object is first created, assign an objid, 
      ;;; and put it in the cache.
      (progn
	(handle-set-id instance (make-objid (class-of instance) dbmode))
	(if (eq dbmode 'local-copy)
	    (handle-set-indb instance nil)
	  (handle-set-indb instance t))
	(handle-set-modified instance t)
	(handle-set-mode instance dbmode)
	(handle-set-pinned instance nil)
	;; Local objects are not stored in the cache.
	(if (not (eq dbmode 'local-copy))
	    (put-into-object-cache instance))
	;; Store it to the database if direct update.
	(if (eq dbmode 'direct-update)
	    (store-dbobject instance))))))

;;;
;;; MAKE-OBJID
;;; Create an objid for a dbclass.
;;; This is a hack before we can preallocate a block of oid's.
;;;

(defun make-objid (class dbmode)
  (let ((class-name (class-name class))
	(relid (dbclass-relid class))
	;; generate a unique oid
	(fake-oid (prin1-to-string (mod (get-internal-run-time) *max-objid*))))
    (if (eq dbmode 'local-copy)
	;; local-copy dbobjects will not be stored in the database
	;; their oid is negative
	(list relid (- 0 (read-from-string fake-oid)))
      (do-make-objid class-name fake-oid))))


(defun do-make-objid (class-name fake-oid)
  (if *libpq-available*
      (let ((query1 (format nil "append ~A (sohid = text[~S])"
			    (symbol-to-db class-name)
			    fake-oid))
	    (query2 (format nil "retrieve (~A.oid) where ~A.sohid = text[~S]"
			    (symbol-to-db class-name)
			    (symbol-to-db class-name)
			    fake-oid)))
	   (pqexec query1)
	   (pqexec query2)
	   (list (symbol-to-db class-name)
		 (read-from-string (pqgetvalue (pqparray "blank") 0 0))))
      ;; else
  (if *cling-available*
      (let* ((class-rel (symbol-to-db class-name))
	     (sohid (of-value-to-db-type 'integer (read-from-string fake-oid))))
	    (db-insert class-rel '(sohid) (list sohid))
	    ;; Assume success
	    (list class-rel sohid)))))


;;;
;;; STORE-DBOBJECT 
;;; Stores a dbobject into database.  If it is already in the database,
;;; replace it, otherwise append it.
;;;

(defmethod store-dbobject ((self dbobject) &optional store-components-p)
  ;; Unless the dbobject is not fetched in, or already in the database 
  ;; and not modified.
  (unless (or (null (handle-instance self))
	      (and (handle-indb self)
		   (not (handle-modified self))))
	  ;; The flags are set before flushing to DB
	  ;; to solve the circular reference problem.
	  (handle-set-indb self t)
	  (handle-set-modified self nil))
	  ;; flush the dbobject to the database
	  (flush-dbobject self)
  self)

(defun flush-dbobject (handle)
  (let* ((class (class-of handle))
	 (instance-slots (class-instance-slots class))
	 (local-slots (soh-local-slots class))
	 ;; for now, indb-p is always t
	 (indb-p (handle-indb handle))
	 (slots nil))
    ;; HACK
    ;; if the dbobject to be stored has mode 'local-copy, and it is not
    ;; in the database yet, we have to get the real objid for it first.
    (when (minusp (cadr (handle-id handle)))
	  (handle-set-id handle 
			 (make-objid (class-of handle) 'deferred-update))
	  (handle-set-mode handle 'deferred-update)
	  ;; a local object is not stored in the object cache until now
	  (put-into-object-cache handle))
    ;; build up the database query 
    ;; instance-slots will never be NIL because each dbobject must 
    ;; have an objid
    (dolist (c instance-slots)
	    ;; Do not store slots in  non-stored-slots
	    (if (null (member (slotd-name c) local-slots))
		(setq slots
		      (extend-instance-slots 
		       slots
		       (symbol-to-db (slotd-name c))
		       ;; convert OBJFADS values to POSTGRES
		       (of-value-to-db-type
			(slotd-type c)
			(slot-value handle (slotd-name c)))))))
    (do-db-flush indb-p (symbol-to-db (class-name class)) slots (sohid handle))))

(defun do-db-flush (replace reln slots sohid)
  (if *libpq-available*
      (let ((query 
	     (format nil "~A ~A (~A) ~A"
		     ;; REPLACE if in database, otherwise append
		     (if replace "replace" "append") 
		     reln
		     ;; get rid of the extra ", " 
		     (subseq slots 0 (- (length slots) 2))
		     ;; add a WHERE clause if REPLACE 
		     (if replace
			 (format nil "where ~A.oid = oid[~S]"
				 reln
				 (prin1-to-string sohid))
			 ;; nothing more is needed for APPEND
			 ""))))
	   (PQexec query))
      ;; else
  (if *cling-available*
      (if replace
	  (db-update reln (cling-slot-update-list slots)
		     :where `(= sohid ,sohid))
	  (db-insert reln (cling-slot-name-list slots) 
		     (cling-slot-value-list slots))))))


;;; DELETE-DBOBJECT
;;;    Remove a dbobject from the database.
;;;
;;; Delete an object from the database.  Convert the in-core instance
;;; to a local-copy.
;;;

(defmethod delete-dbobject ((self dbobject)
			    &optional no-error-p)
  (declare (ignore no-error-p))
  (when (handle-indb self)
	(do-db-delete self)
	(handle-set-indb self nil)
	(handle-set-modified self nil)
	(handle-set-mode self 'local-copy)
	(handle-set-pinned self nil)))

(defmethod do-db-delete ((self dbobject))
  (if *libpq-available*
      (error "POSTGRES can't delete objects yet")
      ;;else
  (if *cling-available*
      (db-delete (symbol-to-db (class-name (class-of self)))
		 :where `(= sohid ,(sohid self))))))


;;;
;;; FETCH-DBOBJECT 
;;;	Fetch a dbobject from the database.
;;;
;;; Since we may fetch dbobjects recursively (for a composite object),
;;; we need to generate unique portal names.
;;;

(defun make-portal ()
  (if *libpq-available*
      (string (gensym "portal_"))
      ;;else
  (if *cling-available*
      nil)))

(defmethod fetch-dbobject ((handle dbobject) &rest the-rest)
  "fetch a dbobject given a handle"
  (let* ((dbmode (first the-rest))
	 (no-error-p (second the-rest))
	 (class (class-of handle))
	 (portal (make-portal))
	 (relid (symbol-to-db (class-name class)))
	 (query
	  (if *libpq-available*
	      (format nil "retrieve portal ~A (~A.all) where ~A.oid = oid[~S]"
		      portal relid relid (prin1-to-string (sohid handle)))
	      ;;else
	  (if *cling-available*
	      `(db-select 
		'(*)     
		:from '(,relid)
		:where '(= sohid ,(sohid handle)))))))
    (make-dbobject-from-database class query portal dbmode no-error-p)))

(defmethod fetch-dbobject ((objid list) &rest the-rest)
  ;; if the dbclass is not already defined in core, bring it in
  (let* ((dbmode (first the-rest))
	 (no-error-p (second the-rest))
	 (class (or (cached-dbclass (car objid) t)
		    (fetch-dbclass (car objid) t)))
	 (portal (make-portal))
	 (query
	  (if *libpq-available*
	      (format nil "retrieve portal ~A (x.all) from x in <~A> 
						      where x.oid = oid[~S]"
		  portal
		  (car objid)
		  (prin1-to-string (cadr objid)))
	      ;;else
	  (if *cling-available*
	      `(db-select 
		'(*)
		:from '(,(car objid))
		:where '(= sohid ,(cadr objid)))))))
    (make-dbobject-from-database class query portal dbmode no-error-p)))

(defmethod fetch-dbobject ((self dbclass) &rest the-rest)
  "fetch a dbobject by slot value"
  (let* ((slot-name (first the-rest))
	 (slot-value (second the-rest))
	 (dbmode (third the-rest))
	 (no-error-p (fourth the-rest))
	 (class-name (class-name self))
	 (portal (make-portal))
	 (relid (symbol-to-db class-name))
	 (query
	  (if *libpq-available*
	      (format nil "retrieve portal ~A (~A.all) where ~A.~A=~A"
		      portal
		      relid
		      relid
		      (symbol-to-db slot-name)
		      (of-value-to-db slot-value))
	      ;; else
	  (if *cling-available*
	      `(db-select
		'(*)
		:from '(,relid)
		:where '(= ,slot-name
			   ,(of-value-to-db slot-value)))))))
    (make-dbobject-from-database self query portal dbmode no-error-p)))

;;; 
;;; MAKE-DBOBJECT-FROM-DATABASE makes a dbobject from the tuple fetched
;;; from the database.  The objid of this dbobject is returned.
;;;

(defun do-fetch-tuples (query portal)
  (if *libpq-available*
      (progn
       (pqexec query)
       (pqexec (format nil "fetch all in ~A" portal))
       ;; can leave them open for when debugging
       (pqexec (format nil "close ~A" portal))
       (list-of-tuples (PQntuples (PQparray portal))))

      ;; else
  (if *cling-available*
      (eval query))))

(defun do-describe-tuples (query portal)
  (declare (ignore query))
  (if *libpq-available*
      (PQparray portal)
      ;;else
  (if *cling-available*
      (db-describe nil))))

(defun db-value (parray/cols index/tuple slot)
  (if *libpq-available*
      (db-value-to-of
       (slotd-type slot)
       (pqgetvalue parray/cols index/tuple
		   (pqfnumber parray/cols index/tuple 
			      (symbol-to-db (slotd-name slot)))))
      ;;else
  (if *cling-available*
      (db-value-to-of (slotd-type slot)
		      (nth (value-pos parray/cols 
				      (symbol-to-db (slotd-name slot)))
			   index/tuple)))))

(defun db-column-value (parray/cols index/tuple name)
  (if *libpq-available*
      (pqgetvalue parray/cols index/tuple
		  (pqfnumber parray/cols index/tuple (symbol-to-db name)))
      ;;else
  (if *cling-available*
      (nth (value-pos parray/cols (symbol-to-db name)) index/tuple))))

(defun list-of-tuples (n)
  (if *libpq-available*
      (if (> n 0) (append (list-of-tuples (1- n) (list (1- n)))))))

(defun value-pos (cols name)
  (if *cling-available*
      (position-if #'(lambda (col-desc) (equal (second col-desc) name))
		   cols)))

(defun read-sohid (parray/cols index/tuple)
  (if *libpq-available*
      (read-from-string (db-column-value parray/cols index/tuple 'sohid))
      ;; else
  (if *cling-available*
      (db-column-value parray/cols index/tuple 'sohid))))

(defun make-dbobject-from-database (class query portal dbmode
					  &optional no-error-p)
  ;; query the instance relation for the dbobject
  (let* ((tuples (do-fetch-tuples query portal))
	 (tuple-desc (do-describe-tuples query portal)))
  ;; if no qualified tuple returned
  (if (null tuples)
      ;; signal an error unless no-error-p is true
      (if no-error-p () (error "The dbobject is not in the database."))
    ;; otherwise build and evaluate the make-function to make the instance
    (let* ((tuple (first tuples))
	   (instance-slots (class-instance-slots class))
	   (local-slots (soh-local-slots class))
           (class-name (class-name class))
	   (make-function `(make-instance ',class-name :dbmode ',dbmode)))
      (dolist (c instance-slots)
	      (let ((slot-name (slotd-name c)))
		(if (not (member slot-name local-slots))
		    (setq make-function
			  (append 
			   make-function
			   (make-slot-function 
			    slot-name 
			    (db-value tuple-desc tuple c)))))))
      ;; debug
      (write make-function)
      (let ((new-object (eval make-function)))
	(handle-set-indb new-object t)
	(reinitialize-dbobject new-object))))))

(defun is-handle (spec)
  (and (stringp (first spec)) (numberp (second spec))))

(defun make-slot-value (slot-value)
  (cond
   ((atom slot-value) `(quote ,slot-value))
   ((stringp slot-value) slot-value)
   ((is-handle slot-value) `(make-handle (find-class ',(symbol-from-db (first slot-value))) ',slot-value))
   ((not (listp (rest slot-value))) `(quote ,slot-value)) ;; Dotted pair
   (t (cons 'list (mapcar #'make-slot-value slot-value)))))

    
(defun make-slot-function (slot-name slot-value)
  (list 
    (read-from-string (format nil ":~A" slot-name))
    (make-slot-value slot-value)))

;;;
;;; REINITIALIZE-DBOBJECT
;;; Called when an object is fetched from the database.
;;;
(defmethod reinitialize-dbobject ((self dbobject))
  self)

;;;
;;;
;;; MFETCH-DBOBJECT fetches a set of objects from the database
;;; all of which have the given value for a given slot-name.
;;;

(defmethod mfetch-dbobject ((self symbol) &rest the-rest)
  (let ((slot-name (first the-rest))
	(slot-value (second the-rest))
	(dbmode (third the-rest))
	(no-error-p (fourth the-rest)))
    (mfetch-dbobject (find-class self) slot-name slot-value dbmode no-error-p)))

(defmethod mfetch-dbobject ((self dbclass) &rest the-rest)
  "fetch dbobject list by slot value"
  (let* ((slot-name (first the-rest))
	 (slot-value (second the-rest))
	 (dbmode (third the-rest))
	 (no-error-p (fourth the-rest))
	 (where 
	  (if *libpq-available*
	      (format nil "~A = ~A" (symbol-to-db slot-name)
		      (of-value-to-db slot-value))
	      ;; else
	  (if *cling-available*
	      `(= ,slot-name ,slot-value)))))
    (mfetch-dbobject-where self where dbmode no-error-p)))

(defmethod mfetch-dbobject-where ((self symbol) &rest the-rest)
  (let* ((where (first the-rest))
	 (dbmode (second the-rest))
	 (no-error-p (third the-rest)))
     (mfetch-dbobject-where (find-class self) where dbmode no-error-p)))

(defmethod mfetch-dbobject-where ((self dbclass) &rest the-rest)
  "fetch dbobject with restrictions"
  (let*  ((where (first the-rest))
	  (dbmode (second the-rest))
	  (no-error-p (third the-rest))
	  (class-name (class-name self))
          (portal (make-portal))
	  (relid (symbol-to-db class-name))
	  (query
	   (if *libpq-available*
	       (format nil "retrieve portal ~A (~A.all) ~A" portal relid
		       (if where (format nil "where ~A" where) ""))
	       ;; else
	   (if *cling-available*
	       (if where
		   `(db-select '(*) :from '(,relid) 
			       :where (quote ,(of-restriction-to-db where)))
		   `(db-select '(*) :from '(,relid)))))))

    (mmake-dbobject-from-database self query portal dbmode no-error-p)))


;;;
;;; MMAKE-DBOBJECT-FROM-DATABASE makes dboobjects from the tuples
;;; fetched from the database. A list of the objid's of these 
;;; dbobjects is returned.

(defun mmake-dbobject-from-database (class query
                                         portal dbmode
                                         &optional no-error-p)
  ;; query the instance relation for the dbobjects
  (let* ((tuples (do-fetch-tuples query portal))
	 (tuple-desc (do-describe-tuples query portal)))
    ;; if no qualified tuple returned
    (if (null tuples)
	;; signal an error unless no-error-p is true
	(if no-error-p ()(error "No qualified dbobjects found"))
      ;; otherwise build and evaluate the make-function to make the
      ;; instance
      (let* ((instance-slots (class-instance-slots class))
	     (local-slots (soh-local-slots class))
	     (class-name (class-name class))
	     (result-list nil))
	(dolist (tuple tuples)
		(let ((make-function `(make-instance ',class-name :dbmode ',dbmode)))
		  (dolist (c instance-slots)
			  (let ((slot-name (slotd-name c)))
			    (if (not (member slot-name local-slots))
				(setq make-function
				      (append 
				       make-function
				       (make-slot-function 
					slot-name 
					(db-value tuple-desc tuple c)))))))
		  (write make-function)
		  (format t "~%")
		  (let ((new-object nil)
			(new-handle (get-handle 
				     (list 
				      class-name
				      (read-sohid tuple-desc tuple)))))
		    (if (and new-handle (handle-instance new-handle))
			(setq new-object new-handle)
		      (setq new-object (reinitialize-dbobject (eval make-function)))
		      ;;(setq new-object  make-function)
		      )
		    (handle-set-indb new-object t)
		    (push new-object result-list))))
	result-list))))

;; returns t if object is in the cache
(defun object-in-cache (objectid)
  (let ((handle (get-handle objectid)))
    (if handle (handle-instance handle) nil)))

