;;;
;;; Shared Object Hierarchy
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/widgets/soh/RCS/cache.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:08:49 $
;;;

(in-package clos)

;;;
;;; Two caches are defined in this file:
;;; (1) a dbclass-cache which maps a relid to a dbclass.
;;; (2) an object-cache which maps an objid to a handle of dbobject.
;;;
;;; QUESTION: class objects and instance objects are not treated
;;; the same way in this implementation. If they are treated uniformly,
;;; then the two caches can be merged into one.
;;;

;;;
;;; DBCLASS-CACHE
;;; A handle table which maps a relid, the POSTGRES oid's of the 
;;; corresponding class instance relation, to dbclass objects.
;;;

(defun dbclass-cache-size ()
  *dbclass-cache-size*)

;;;
;;; OBJECT-CACHE
;;; Define a hashtable which maps an objid to a handle.
;;; 

(defun object-cache-size ()
  *object-cache-size*)

;;;
;;; Put a dbobject in the object-cache.
;;;
(defun put-into-object-cache (handle)
  ;; if cache is full, clean it by storing dbobjects to the database and
  ;; remove the entries from the hashtable.
  (when (> (incf *cache-count*) *cache-full*) 
	(clean-cache))
  (setf (gethash (handle-id handle) *object-cache*) handle))

(defun list-cache ()
  (maphash #'(lambda (key val)
	       (format t "Key ~A, val ~A~%" key val))
	   *object-cache*))
;;;
;;; This function needs to be rewritten to reflect the change 
;;; of cache structure.
;;;
(defun clean-cache ()
  ;; removes all unmodified and unpinned entries
  (maphash #'(lambda (key val)
	       (when (and (null (handle-modified val))
			  (null (handle-pinned val)))
		     (handle-set-instance val nil)
		     (remhash key *object-cache*)
		     (decf *cache-count*)))
	   *object-cache*)
  ;; if the hashtable is still full
  (when (> *cache-count* *cache-full*)
	;; store all dbobjects with the mode 'deferred-update and remove them
	;; from the hashtable
	(maphash #'(lambda (key val)
		     (when (and (eq (handle-mode val) 'deferred-update)
				(not (handle-pinned val)))
			   (store-dbobject val)
			   (handle-set-instance val nil)
			   (remhash key *object-cache*)
			   (decf *cache-count*)))
		 *object-cache*))
  ;; if the hashtable is still full (means the hashtable is too small)
  (when (> *cache-count* *cache-full*)
	;; force the hashtable size to be increased.
	(setq *cache-full* (* *cache-full* 2))))

;;;
;;; These are the functions Larry wants
;;;
(defun touch (handle)
  (handle-set-modified handle t))
